import fs from "node:fs";
import path from "node:path";
import { fileURLToPath } from "node:url";
function getJavaScriptFile() {
  const dirname = path.dirname(fileURLToPath(import.meta.url));
  const filePath = [
    path.resolve(`${dirname}/js/standalone.js`),
    path.resolve(`${dirname}/../../dist/js/standalone.js`)
  ].find((file) => fs.existsSync(file));
  if (filePath === void 0) {
    throw new Error(
      `JavaScript file not found: ${path.resolve(
        `${dirname}/js/standalone.js`
      )}`
    );
  }
  return fs.readFileSync(filePath, "utf8");
}
const schemaToHideRoute = {
  hide: true
};
const getJavaScriptUrl = (routePrefix) => `${routePrefix ?? ""}/@scalar/fastify-api-reference/js/browser.js`.replace(
  /\/\//g,
  "/"
);
const defaultCss = `
:root {
  --theme-font: 'Inter', var(--system-fonts);
}

.light-mode {
  color-scheme: light;
  --theme-color-1: #1c1e21;
  --theme-color-2: #757575;
  --theme-color-3: #8e8e8e;
  --theme-color-disabled: #b4b1b1;
  --theme-color-ghost: #a7a7a7;
  --theme-color-accent: #2f8555;
  --theme-background-1: #fff;
  --theme-background-2: #f5f5f5;
  --theme-background-3: #ededed;
  --theme-background-4: rgba(0, 0, 0, 0.06);
  --theme-background-accent: #2f85551f;

  --theme-border-color: rgba(0, 0, 0, 0.1);
  --theme-scrollbar-color: rgba(0, 0, 0, 0.18);
  --theme-scrollbar-color-active: rgba(0, 0, 0, 0.36);
  --theme-lifted-brightness: 1;
  --theme-backdrop-brightness: 1;

  --theme-shadow-1: 0 1px 3px 0 rgba(0, 0, 0, 0.11);
  --theme-shadow-2: rgba(0, 0, 0, 0.08) 0px 13px 20px 0px,
    rgba(0, 0, 0, 0.08) 0px 3px 8px 0px, #eeeeed 0px 0 0 1px;

  --theme-button-1: rgb(49 53 56);
  --theme-button-1-color: #fff;
  --theme-button-1-hover: rgb(28 31 33);

  --theme-color-green: #007300;
  --theme-color-red: #af272b;
  --theme-color-yellow: #b38200;
  --theme-color-blue: #3b8ba5;
  --theme-color-orange: #fb892c;
  --theme-color-purple: #5203d1;
}

.dark-mode {
  color-scheme: dark;
  --theme-color-1: rgba(255, 255, 255, 0.9);
  --theme-color-2: rgba(255, 255, 255, 0.62);
  --theme-color-3: rgba(255, 255, 255, 0.44);
  --theme-color-disabled: rgba(255, 255, 255, 0.34);
  --theme-color-ghost: rgba(255, 255, 255, 0.26);
  --theme-color-accent: #27c2a0;
  --theme-background-1: #1b1b1d;
  --theme-background-2: #242526;
  --theme-background-3: #3b3b3b;
  --theme-background-4: rgba(255, 255, 255, 0.06);
  --theme-background-accent: #27c2a01f;

  --theme-border-color: rgba(255, 255, 255, 0.1);
  --theme-scrollbar-color: rgba(255, 255, 255, 0.24);
  --theme-scrollbar-color-active: rgba(255, 255, 255, 0.48);
  --theme-lifted-brightness: 1.45;
  --theme-backdrop-brightness: 0.5;

  --theme-shadow-1: 0 1px 3px 0 rgb(0, 0, 0, 0.1);
  --theme-shadow-2: rgba(15, 15, 15, 0.2) 0px 3px 6px,
    rgba(15, 15, 15, 0.4) 0px 9px 24px, 0 0 0 1px rgba(255, 255, 255, 0.1);

  --theme-button-1: #f6f6f6;
  --theme-button-1-color: #000;
  --theme-button-1-hover: #e7e7e7;

  --theme-color-green: #26b226;
  --theme-color-red: #fb565b;
  --theme-color-yellow: #ffc426;
  --theme-color-blue: #6ecfef;
  --theme-color-orange: #ff8d4d;
  --theme-color-purple: #b191f9;
}
`;
const javascript = (options) => {
  var _a, _b, _c, _d;
  const { configuration } = options;
  return `
    <script
      id="api-reference"
      type="application/json"
      data-configuration="${JSON.stringify(configuration ?? {}).split('"').join("&quot;")}">${((_a = configuration == null ? void 0 : configuration.spec) == null ? void 0 : _a.content) ? typeof ((_b = configuration == null ? void 0 : configuration.spec) == null ? void 0 : _b.content) === "function" ? JSON.stringify((_c = configuration == null ? void 0 : configuration.spec) == null ? void 0 : _c.content()) : JSON.stringify((_d = configuration == null ? void 0 : configuration.spec) == null ? void 0 : _d.content) : ""}<\/script>
      <script src="${getJavaScriptUrl(options.routePrefix)}"><\/script>
  `;
};
function htmlDocument(options) {
  return `
<!DOCTYPE html>
<html>
  <head>
    <title>API Reference</title>
    <meta charset="utf-8" />
    <meta
      name="viewport"
      content="width=device-width, initial-scale=1" />
    <style>
      body {
        margin: 0;
      }
    </style>
  </head>
  <body>
    ${javascript(options)}
  </body>
</html>
`;
}
const fastifyApiReference = async (fastify, options) => {
  var _a, _b;
  let { configuration } = options;
  const hasSwaggerPlugin = fastify.hasPlugin("@fastify/swagger");
  if (!fastify.hasPlugin("fastify-html")) {
    await fastify.register(import("fastify-html"));
  }
  if (!((_a = configuration == null ? void 0 : configuration.spec) == null ? void 0 : _a.content) && !((_b = configuration == null ? void 0 : configuration.spec) == null ? void 0 : _b.url) && !hasSwaggerPlugin) {
    fastify.log.warn(
      "[@scalar/fastify-api-reference] You didn’t provide a spec.content or spec.url, and @fastify/swagger could not be found. Please provide one of these options."
    );
    return;
  }
  const fileContent = getJavaScriptFile();
  fastify.route({
    method: "GET",
    url: options.routePrefix ?? "/",
    // We don’t know whether @fastify/swagger is registered, but it doesn’t hurt to add a schema anyway.
    // @ts-ignore
    schema: schemaToHideRoute,
    async handler(_, reply) {
      var _a2, _b2;
      reply.header("Content-Type", "text/html; charset=utf-8");
      if (!((_a2 = configuration == null ? void 0 : configuration.spec) == null ? void 0 : _a2.content) && !((_b2 = configuration == null ? void 0 : configuration.spec) == null ? void 0 : _b2.url) && hasSwaggerPlugin) {
        configuration = {
          ...configuration,
          spec: {
            content: () => {
              return fastify.swagger();
            }
          }
        };
      }
      if (!(configuration == null ? void 0 : configuration.customCss) && !(configuration == null ? void 0 : configuration.theme)) {
        configuration = {
          ...configuration,
          customCss: defaultCss
        };
      }
      return reply.html`!${htmlDocument({
        ...options,
        configuration
      })}`;
    }
  });
  fastify.route({
    method: "GET",
    url: getJavaScriptUrl(options.routePrefix),
    // We don’t know whether @fastify/swagger is registered, but it doesn’t hurt to add a schema anyway.
    // @ts-ignore
    schema: schemaToHideRoute,
    async handler(_, reply) {
      reply.header("Content-Type", "application/javascript; charset=utf-8");
      reply.send(fileContent);
    }
  });
};
export {
  fastifyApiReference as default
};
