import type { AnyResult, ResErr, ResultUnion } from '../fn/types';
export type OrElse = {
    <Res extends AnyResult, NextRes extends AnyResult>(mapFn: (err: ResErr<Res>) => NextRes): {
        (result: Res): ResultUnion<Res, NextRes>;
    };
    <Res extends AnyResult, NextRes extends AnyResult>(mapFn: (err: ResErr<Res>) => NextRes, result: Res): ResultUnion<Res, NextRes>;
};
/**
 * Like the `||` operator, but applied to `Result` types with an `() => Ok`.
 *
 * You should use `orElse` when you want to transform the `Err` case before returning the fallback `Result`. For
 * example, if you want to return a default `Ok` value that depends on the error message of a `Result` object
 *
 * The `orElse` function is used to provide a fallback value in case the result is an `Err` variant. It takes a function
 * that maps the error value to a new `Result` object, and returns a new `Result` object that contains either the
 * original `Ok` value or the new value generated by the mapping function.
 *
 * :::tip
 *
 * Both `or` and `orElse` can be used to provide fallback behavior for `Err` cases in a `Result` object. However, `or`
 * returns the provided fallback `Result` object directly, while `orElse` applies a function to transform the `Err` case
 * before returning the fallback `Result`.
 *
 * :::
 *
 * ## Example
 *
 * In the first example, since the `Result` is `Ok('Hello')`, the fallback function provided to `orElse` is **not**
 * executed, and the `Result` remains the same.
 *
 * In the second example, since the `Result` is `Err('Oops!')`, the fallback function provided to `orElse` **is**
 * executed with the error value, which generates a new `Ok` value containing the fallback message.
 *
 * ```ts
 * import { pipe } from 'tightrope/fn/pipe';
 * import { Ok, Err } from 'tightrope/result';
 * import { orElse } from 'tightrope/result/or-else';
 * import { unwrap } from 'tightrope/result/unwrap';
 *
 * const result1 = pipe(
 *   Ok.create('Hello'),
 *   orElse((err) => new Ok(`Fallback value: ${err}`)),
 *   unwrap,
 * );
 *
 * console.log(result1);
 * // result1 is 'Hello'
 *
 * const result2 = pipe(
 *   Err.create('Oops!'),
 *   orElse((err) => new Ok(`Fallback value: ${err}`)),
 *   unwrap,
 * );
 *
 * console.log(result2);
 * // result2 is 'Fallback value: Oops!'
 * ```
 *
 * ## Use Cases
 *
 * - **Error handling:** provide a default value or action when an error occurs. For example, if a network request fails,
 *   you might want to retry the request with a different endpoint or fallback to cached data.
 * - **Chaining operations:** chain operations that may fail. For example, if you have a series of functions that return
 *   `Result` objects, you can use `orElse` to chain them together and handle any errors that may occur.
 * - **Data validation:** validate data and provide a fallback value or action when the data fails validation. For
 *   example, if you are parsing data from a file or API response, you can use `orElse` to provide a default value or
 *   action when the data is invalid.
 *
 * @tags result, transform, transform-result, recover, errors, left-biased
 * @see https://jamiemason.github.io/tightrope/api/result/or
 * @see https://jamiemason.github.io/tightrope/api/result/and
 * @see https://doc.rust-lang.org/core/result/enum.Result.html#method.or_else
 */
export declare const orElse: OrElse;
