import type { Gen, GenYield } from '../fn/types';
export type Take = {
    <T extends Gen<any>>(amount: number): {
        (gen: T): Gen<GenYield<T>>;
    };
    <T extends Gen<any>>(amount: number, gen: T): Gen<GenYield<T>>;
};
/**
 * Do not `yield` values after `count` values have been yielded.
 *
 * A higher-order function that takes a number amount as its argument and returns a new function that can be used to
 * take the first amount values generated by a generator.
 *
 * The returned function takes a generator as its argument and returns a new generator that yields at most the first
 * amount values generated by the input generator.
 *
 * ## Example
 *
 * Here's an example of how to use the `take` function to take the first three values generated by a range generator:
 *
 * ```ts
 * import { pipe } from 'tightrope/fn/pipe';
 * import { range } from 'tightrope/gen/range';
 * import { take } from 'tightrope/gen/take';
 *
 * const gen = pipe(range(1, 10), take(3));
 *
 * for (const value of gen) {
 *   console.log(value); // 1, 2, 3
 * }
 * ```
 *
 * In this example, we use the `pipe` function to create a new generator that takes the first three values generated by
 * the `range` generator.
 *
 * We create a new generator `gen` that is the result of applying the `pipe` function to the `range` generator.
 *
 * We use a `for...of` loop to iterate over the values generated by the `gen` generator and log them to the console.
 *
 * The output of the code should be `1, 2, 3`.
 *
 * In summary, the `take` function is a useful tool for limiting the number of values generated by a generator. By using
 * it in combination with other functions from tightrope, we can create powerful and flexible pipelines that help us
 * write more efficient and focused code.
 *
 * ## Use Cases
 *
 * Here are some common use cases for the `take` function:
 *
 * 1. Limiting the size of data: You may need to limit the size of a dataset to improve performance or prevent memory
 *    issues. The `take` function can be used to limit the number of items processed at any one time.
 * 2. Extracting a subset of data: If you are working with a large dataset, you may need to extract a subset of the data
 *    that meets certain criteria. The `take` function can be used to extract the first n items that meet your
 *    criteria.
 * 3. Creating a sample: If you are working with a large dataset and want to create a smaller sample for testing or
 *    exploration, the `take` function can be used to extract a random or fixed number of items from the dataset.
 *
 * Overall, the `take` function is a versatile tool that can be used in a variety of contexts to limit the size of data
 * or extract subsets of data from a larger dataset.
 *
 * @tags generator
 */
export declare const take: Take;
