import { curry } from '../fn/curry';
/**
 * Reduce and `yield` each value produced by a generator.
 *
 * A higher-order function that takes a `reducer` function as its argument and returns a new function that can be used
 * to reduce the values generated by a generator.
 *
 * The returned function takes a generator as its argument and returns a new generator that yields the partial results
 * of the reduction operation. It works by applying the `reducer` function to each pair of values generated by the input
 * generator, and yielding the partial results of the reduction operation.
 *
 * The `reducer` function takes two arguments: an `accumulator` value that represents the current state of the reduction
 * operation, and a `value` generated by the input generator. The reducer function applies some operation to the
 * accumulator value and the input value, and returns a new accumulator value that represents the updated state of the
 * reduction operation.
 *
 * In summary, `reduceEach` is a powerful tool for working with generators in JavaScript and TypeScript. It can be used
 * to perform complex reduction operations on large or complex data sets, and can help us write more efficient and
 * focused code.
 *
 * ## Example
 *
 * In this example, we use the `range` function to generate a sequence of numbers from `1` to `10`, the `map` function
 * to double each value in the sequence, and the `take` function to select the first `5` values in the sequence.
 * Finally, we apply the `reduceEach` function with the `sum` function as its argument to generate the partial sums of
 * the remaining values in the sequence.
 *
 * We then use a `for...of` loop to iterate over the partial sums generated by the `partialSums` generator, and log them
 * to the console.
 *
 * ```ts
 * import { pipe } from 'tightrope/fn/pipe';
 * import { map } from 'tightrope/gen/map';
 * import { range } from 'tightrope/gen/range';
 * import { reduceEach } from 'tightrope/gen/reduce-each';
 * import { take } from 'tightrope/gen/take';
 * import { multiply } from 'tightrope/number/multiply';
 *
 * function sum(acc: number, curr: number): number {
 *   return acc + curr;
 * }
 *
 * const partialSums = pipe(range(1, 10), map(multiply(2)), take(5), reduceEach(sum));
 *
 * for (const value of partialSums) {
 *   console.log(value);
 *   //  6
 *   // 12
 *   // 20
 *   // 30
 * }
 * ```
 *
 * ## Use Cases
 *
 * Common use cases for `reduceEach`:
 *
 * 1. **Summing values**: calculate the sum of a series of numbers generated by a generator.
 * 2. **Counting values**: count the number of values generated by a generator that satisfy a given condition.
 * 3. **Finding minimum or maximum values**: find the minimum or maximum value generated by a generator.
 * 4. **Concatenating strings**: concatenate a series of strings generated by a generator.
 * 5. **Grouping values**: group a series of values generated by a generator into a nested structure, such as an object or
 *    an array.
 * 6. **Filtering values**: filter the values generated by a generator based on a given condition.
 * 7. **Computing averages**: compute the average of a series of numbers generated by a generator.
 * 8. **Building up an object**: build up an object by accumulating properties and values generated by a generator.
 *
 * In general, `reduceEach` can be used in any situation where we need to perform some kind of reduction operation on a
 * series of values generated by a generator. Its flexibility and power make it a versatile tool for working with
 * generators in JavaScript and TypeScript.
 *
 * ## `reduce` or `reduceEach`?
 *
 * `reduce` and `reduceEach` are very similar functions, and it can be difficult to decide which one to use in a given
 * situation. Here are some guidelines to help you decide:
 *
 * 1. Use `reduce` if you want to perform a reduction operation on the entire sequence of values generated by a generator,
 *    and you only need to return a single value.
 * 2. Use `reduceEach` if you want to perform a reduction operation on each individual value generated by a generator, and
 *    you need to return a sequence of partial results.
 *
 * In general, the choice between `reduce` and `reduceEach` will depend on the specific requirements of your problem, as
 * well as the nature of the data you are working with.
 *
 * - If you need to perform a simple reduction operation on a single sequence of values, `reduce` may be the better
 *   choice.
 * - If you need to perform a more complex reduction operation that involves multiple sequences of values, or if you need
 *   to return a sequence of partial results, `reduceEach` may be the better choice.
 *
 * @tags transform, transform-value, generator
 * @see https://jamiemason.github.io/tightrope/api/gen/reduce
 */
export const reduceEach = curry(function* reduceEach(fn, initialValue, gen) {
    let acc = initialValue;
    for (const value of gen) {
        acc = fn(acc, value);
        yield acc;
    }
}, 3);
