import type { Gen, GenYield, UnaryGuard } from '../fn/types';
export type Filter = {
    <F extends UnaryGuard<GenYield<T>>, T extends Gen<any>>(guard: F): {
        (gen: T): Gen<GenYield<T>>;
    };
    <F extends UnaryGuard<GenYield<T>>, T extends Gen<any>>(guard: F, gen: T): Gen<GenYield<T>>;
};
/**
 * Only `yield` each value which passes the predicate.
 *
 * A higher-order function that takes a function as its argument and returns a new function that can be used to filter
 * values from a generator.
 *
 * The returned function takes a generator as its argument and returns a new generator that only yields the values that
 * pass the filter function.
 *
 * The filter function applies the filter function to each value generated by the input generator, and only yields the
 * values for which the filter function returns `true`. Any values for which the filter function returns `false` are
 * discarded.
 *
 * In summary, the filter function is a powerful tool for working with generators in JavaScript and TypeScript. By using
 * it to filter values from a generator, we can easily create new generators that yield only the values that meet our
 * specific criteria. This can help us write more efficient and focused code, and can make it easier to work with large
 * or complex data sets.
 *
 * ## Example
 *
 * In this example, we then use the `pipe` function to create a pipeline of generators and filters.
 *
 * We start with the `range` function, which generates a sequence of numbers starting from `0` and ending at `100`
 * (inclusive). We then apply the `filter` function to the pipeline to filter out even numbers using the `isEvenNumber`
 * guard function, and then apply the `take` function to limit the number of values generated to `5`.
 *
 * Finally, we use a `for...of` loop to iterate over the 5 even numbers generated by the pipeline, and log them to the
 * console.
 *
 * ```ts
 * import { pipe } from 'tightrope/fn/pipe';
 * import { filter } from 'tightrope/gen/filter';
 * import { range } from 'tightrope/gen/range';
 * import { take } from 'tightrope/gen/take';
 * import { isEvenNumber } from 'tightrope/guard/is-even-number';
 *
 * const evenNumbers = pipe(range(0, 100), filter(isEvenNumber), take(5));
 *
 * for (const value of evenNumbers) {
 *   console.log(value);
 *   // 0
 *   // 2
 *   // 4
 *   // 6
 *   // 8
 * }
 * ```
 *
 * @tags generator
 */
export declare const filter: Filter;
