import { isFunction } from '../guard/is-function';
import { isPrimitive } from '../guard/is-primitive';
import { isReadable } from '../guard/is-readable';
import { Err, Ok } from '../result';
import { isErr } from '../result/is-err';
import { isOk } from '../result/is-ok';
export function get(origin, ...props) {
    return isReadable(origin)
        ? props.reduce(getChild, origin)
        : ERR_UNREADABLE_ORIGIN(props, origin);
}
/** Used internally by get to retrieve a single child property from a parent object. */
function getChild(parent, prop) {
    // quit if any ancestor was already not found
    if (isErr(parent))
        return parent;
    // ensure we have a plain value and not an Ok
    const value = isOk(parent) ? parent.value : parent;
    // quit if we can't read properties of value (eg value.likeThis)
    if (!isReadable(value))
        return ERR_UNREADABLE_CHILD(prop, value);
    // quit if value is object/array/function etc but the child is not found
    if (!isPrimitive(value) && prop in value === false)
        return ERR_NOT_FOUND(prop, value);
    // quit if eg true.toFixed, 12.toUpperCase
    if (isPrimitive(value) && value[prop] === undefined)
        return ERR_NOT_FOUND(prop, value);
    // the value is present, return it
    const child = value[prop];
    return new Ok(isFunction(child) ? child.bind(value) : child);
}
function ERR_UNREADABLE_CHILD(child, value) {
    return Err.fromMessage(`Cannot read "${child}" from unreadable value: ${value}`);
}
function ERR_NOT_FOUND(child, value) {
    return Err.fromMessage(`Property "${child}" not found on value: ${value}`);
}
function ERR_UNREADABLE_ORIGIN(props, origin) {
    return Err.fromMessage(`Cannot read "${props.join('.')}" from unreadable value: ${origin}`);
}
