"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mapOrElse = void 0;
const curry_1 = require("../fn/curry");
const is_some_1 = require("./is-some");
/**
 * Applies the provided function to the contained value of `Some`, or returns the result of evaluating the provided
 * fallback function if the `Option` is `None`
 *
 * ## Example
 *
 * ```ts
 * import { pipe } from 'tightrope/fn/pipe';
 * import { mapOrElse } from 'tightrope/option/map-or-else';
 * import { Some } from 'tightrope/option';
 * import { none } from 'tightrope/option';
 *
 * pipe(
 *   new Some('hello'),
 *   mapOrElse(
 *     (value) => value.toUpperCase(),
 *     () => 'default',
 *   ),
 * ); // 'HELLO'
 *
 * pipe(
 *   none,
 *   mapOrElse(
 *     (value) => value.toUpperCase(),
 *     () => 'default',
 *   ),
 * ); // 'default'
 * ```
 *
 * :::info
 *
 * This function is right-biased, meaning that it returns the result of evaluating the fallback function without
 * applying the provided function if the input `Option` is `None`.
 *
 * :::
 *
 * @param mapFn The function to apply to the contained value of `Some`.
 * @param defaultFn The fallback function to evaluate if the `Option` is `None`.
 * @param opt The `Option` to apply the provided function to.
 * @returns The result of applying the provided function to the contained value of `Some`, or the result of evaluating
 *   the fallback function if the `Option` is `None`.
 * @tags option, transform, transform-option, right-biased, unwrap, result
 */
exports.mapOrElse = (0, curry_1.curry)((mapFn, defaultFn, opt) => {
    return (0, is_some_1.isSome)(opt) ? mapFn(opt.value) : defaultFn();
}, 3);
