"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.filter = void 0;
const _1 = require(".");
const curry_1 = require("../fn/curry");
const is_some_1 = require("./is-some");
/**
 * Calls the provided predicate function on the contained value T if the `Option` is `Some(T)`, and returns `Some(T)` if
 * the function returns `true`; otherwise, returns `None`
 *
 * ## Example
 *
 * ```ts
 * import { pipe } from 'tightrope/fn/pipe';
 * import { filter } from 'tightrope/option/filter';
 * import { none } from 'tightrope/option';
 * import { Some } from 'tightrope/option';
 *
 * const result = pipe(
 *   new Some(2),
 *   filter((value) => value % 2 === 0),
 * );
 * // result is a new Some(2)
 *
 * const result2 = pipe(
 *   none,
 *   filter((value) => value % 2 === 0),
 * );
 * // result2 is none
 * ```
 *
 * :::info
 *
 * This function is right-biased, meaning that if the input `Option` is `None`, it immediately returns `None` without
 * evaluating the predicate function.
 *
 * :::
 *
 * @param predicate The function to call on the contained value of the `Option`.
 * @param opt The `Option` to apply the filter to.
 * @returns An `Option` with the same type as the input that contains the filtered value, or `None`.
 * @tags option, filter, right-biased
 */
exports.filter = (0, curry_1.curry)((predicate, opt) => {
    if ((0, is_some_1.isSome)(opt) && predicate(opt.value))
        return opt;
    return _1.none;
}, 2);
