"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.reduce = void 0;
const curry_1 = require("../fn/curry");
const pipe_1 = require("../fn/pipe");
const reduce_each_1 = require("./reduce-each");
const take_last_1 = require("./take-last");
/**
 * Reduce every value produced by a generator and `yield` the final result.
 *
 * A higher-order function that takes a reducer function as its argument and returns a new function that can be used to
 * reduce the values generated by a generator to a single value.
 *
 * The returned function takes a generator as its argument and returns a new generator that yields the final result of
 * the reduction operation.
 *
 * The `reduce` function works by using the `reduceEach` function to generate a sequence of partial results for the
 * reduction operation, and then returning the final result of the reduction operation.
 *
 * The `reducer` function takes two arguments: an `accumulator` value that represents the current state of the reduction
 * operation, and a `value` generated by the input generator. The reducer function applies some operation to the
 * accumulator value and the input value, and returns a new accumulator value that represents the updated state of the
 * reduction operation.
 *
 * If an initial value is provided, it will be used as the initial state of the reduction operation. If no initial value
 * is provided, the first value generated by the input generator will be used as the initial state.
 *
 * ## Example
 *
 * In this example, we use the pipe function to create a pipeline of functions that generate a series of numbers, double
 * each number using `multiply(2)`, and then calculate the sum of the resulting values.
 *
 * 1. We define a `sum` function that takes two numbers as input and returns their sum.
 * 2. We use the `range` function to generate a series of numbers from `1` to `10`.
 * 3. We use the `map` function with `multiply(2)` to double each number in the series.
 * 4. We use the `reduce` function to calculate the sum of the resulting values.
 * 5. We use a `for...of` loop to iterate over the values generated by `doubledSum`, which will be one value, the final
 *    total, which should be `110`.
 *
 * ```ts
 * import { pipe } from 'tightrope/fn/pipe';
 * import { map } from 'tightrope/gen/map';
 * import { range } from 'tightrope/gen/range';
 * import { reduce } from 'tightrope/gen/reduce';
 * import { multiply } from 'tightrope/number/multiply';
 *
 * function sum(acc: number, curr: number): number {
 *   return acc + curr;
 * }
 *
 * const doubledSum = pipe(range(1, 10), map(multiply(2)), reduce(sum));
 *
 * for (const value of doubledSum) {
 *   console.log(value); // 110
 * }
 * ```
 *
 * ## `reduce` or `reduceEach`?
 *
 * `reduce` and `reduceEach` are very similar functions, and it can be difficult to decide which one to use in a given
 * situation. Here are some guidelines to help you decide:
 *
 * 1. Use `reduce` if you want to perform a reduction operation on the entire sequence of values generated by a generator,
 *    and you only need to return a single value.
 * 2. Use `reduceEach` if you want to perform a reduction operation on each individual value generated by a generator, and
 *    you need to return a sequence of partial results.
 *
 * In general, the choice between `reduce` and `reduceEach` will depend on the specific requirements of your problem, as
 * well as the nature of the data you are working with.
 *
 * - If you need to perform a simple reduction operation on a single sequence of values, `reduce` may be the better
 *   choice.
 * - If you need to perform a more complex reduction operation that involves multiple sequences of values, or if you need
 *   to return a sequence of partial results, `reduceEach` may be the better choice.
 *
 * @tags transform, transform-value, generator
 * @see https://jamiemason.github.io/tightrope/api/gen/reduce-each
 */
exports.reduce = (0, curry_1.curry)(function* reduce(fn, initialValue, gen) {
    yield* (0, pipe_1.pipe)(gen, (0, reduce_each_1.reduceEach)(fn, initialValue), take_last_1.takeLast);
}, 3);
