"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.map = void 0;
const curry_1 = require("../fn/curry");
/**
 * Transform and `yield` each value produced by a generator.
 *
 * A higher-order function that takes a function as its argument and returns a new function that can be used to
 * transform values from a generator.
 *
 * The returned function takes a generator as its argument and returns a new generator that yields the transformed
 * values.
 *
 * The `map` function applies the transformation function to each value generated by the input generator, and yields the
 * transformed values in the output generator. The transformation function can be any function that takes a single input
 * value and returns a single output value of any type.
 *
 * In summary, the `map` function is a powerful tool for working with generators in JavaScript and TypeScript. By using
 * it to transform values from a generator, we can easily create new generators that yield the transformed values
 * according to our specific criteria. This can help us write more efficient and focused code, and can make it easier to
 * work with large or complex data sets.
 *
 * ## Example
 *
 * In this example, we use the `pipe` function to create a pipeline of generators and maps that double the values
 * generated by the range function.
 *
 * ```ts
 * import { pipe } from 'tightrope/fn/pipe';
 * import { map } from 'tightrope/gen/map';
 * import { multiply } from 'tightrope/number/multiply';
 * import { range } from 'tightrope/gen/range';
 * import { take } from 'tightrope/gen/take';
 *
 * const doubledNumbers = pipe(range(0, 100), map(multiply(2)), take(5));
 *
 * for (const value of doubledNumbers) {
 *   console.log(value);
 * }
 * ```
 *
 * We start with the `range` function, which generates a sequence of numbers starting from `0`. We then apply the `map`
 * function to the pipeline to double each value generated by the `range` function, and then apply the `take` function
 * to limit the number of values generated to `5`.
 *
 * Finally, we use a `for...of` loop to iterate over the 5 doubled numbers generated by the pipeline, and log them to
 * the console.
 *
 * When we run this example, we should see the following output:
 *
 *     // 2
 *     // 4
 *     // 6
 *     // 8
 *     // 10
 *
 * ## Use Cases
 *
 * Common use cases for `map`:
 *
 * 1. **Transforming data**: transform data from one form to another. For example, we can use map to convert an array of
 *    numbers into an array of strings, or to extract a specific property from an array of objects.
 * 2. **Computing values**: compute new values based on existing values. For example, we can use map to calculate the
 *    square of each number in an array, or to compute the distance between each pair of points in a set of
 *    coordinates.
 * 3. **Reorganizing data**: reorganize data into a different format by extracting or manipulating specific values. For
 *    example, we can use map to extract only the names of a set of users from an array of user objects, or to extract
 *    only the titles of a set of books from an array of book objects.
 * 4. **Simplifying data**: simplify complex data structures by extracting or manipulating specific values. For example, we
 *    can use map to extract only the numeric values from an array of mixed data types.
 * 5. **Generating new sequences**: generate new sequences of values based on some input data. For example, we can use map
 *    to generate a new array of values based on some mathematical function, or to generate a new array of values based
 *    on some conditional logic.
 *
 * @tags transform, generator
 */
exports.map = (0, curry_1.curry)(function* map(fn, gen) {
    for (const value of gen) {
        yield fn(value);
    }
}, 2);
