import * as Context from "../Context.js";
import * as Duration from "../Duration.js";
import * as Equal from "../Equal.js";
import { dual, pipe } from "../Function.js";
import * as Hash from "../Hash.js";
import * as HashSet from "../HashSet.js";
import { pipeArguments } from "../Pipeable.js";
import { hasProperty } from "../Predicate.js";
import * as effect from "./core-effect.js";
import * as core from "./core.js";
import * as fiberRuntime from "./fiberRuntime.js";
import * as queue from "./queue.js";
import * as ref from "./ref.js";
/** @internal */
const PoolSymbolKey = "effect/Pool";
/** @internal */
export const PoolTypeId = /*#__PURE__*/Symbol.for(PoolSymbolKey);
const poolVariance = {
  /* c8 ignore next */
  _E: _ => _,
  /* c8 ignore next */
  _A: _ => _
};
/**
 * A strategy that does nothing to shrink excess items. This is useful when
 * the minimum size of the pool is equal to its maximum size and so there is
 * nothing to do.
 */
class NoneStrategy {
  initial() {
    return core.unit;
  }
  track() {
    return core.unit;
  }
  run() {
    return core.unit;
  }
}
/**
 * A strategy that shrinks the pool down to its minimum size if items in the
 * pool have not been used for the specified duration.
 */
class TimeToLiveStrategy {
  timeToLive;
  constructor(timeToLive) {
    this.timeToLive = timeToLive;
  }
  initial() {
    return core.flatMap(effect.clock, clock => core.flatMap(clock.currentTimeMillis, now => core.map(ref.make(now), ref => [clock, ref])));
  }
  track(state) {
    return core.asUnit(core.flatMap(state[0].currentTimeMillis, now => ref.set(state[1], now)));
  }
  run(state, getExcess, shrink) {
    return core.flatMap(getExcess, excess => excess <= 0 ? core.zipRight(state[0].sleep(this.timeToLive), this.run(state, getExcess, shrink)) : pipe(core.zipWith(ref.get(state[1]), state[0].currentTimeMillis, (start, end) => end - start), core.flatMap(duration => {
      if (duration >= Duration.toMillis(this.timeToLive)) {
        return core.zipRight(shrink, this.run(state, getExcess, shrink));
      } else {
        return core.zipRight(state[0].sleep(this.timeToLive), this.run(state, getExcess, shrink));
      }
    })));
  }
}
class PoolImpl {
  creator;
  min;
  max;
  isShuttingDown;
  state;
  items;
  invalidated;
  track;
  [PoolTypeId] = poolVariance;
  constructor(creator, min, max, isShuttingDown, state, items, invalidated, track) {
    this.creator = creator;
    this.min = min;
    this.max = max;
    this.isShuttingDown = isShuttingDown;
    this.state = state;
    this.items = items;
    this.invalidated = invalidated;
    this.track = track;
  }
  [Hash.symbol]() {
    return pipe(Hash.hash(this.creator), Hash.combine(Hash.number(this.min)), Hash.combine(Hash.number(this.max)), Hash.combine(Hash.hash(this.isShuttingDown)), Hash.combine(Hash.hash(this.state)), Hash.combine(Hash.hash(this.items)), Hash.combine(Hash.hash(this.invalidated)), Hash.combine(Hash.hash(this.track)));
  }
  [Equal.symbol](that) {
    return isPool(that) && Equal.equals(this.creator, that.creator) && this.min === that.min && this.max === that.max && Equal.equals(this.isShuttingDown, that.isShuttingDown) && Equal.equals(this.state, that.state) && Equal.equals(this.items, that.items) && Equal.equals(this.invalidated, that.invalidated) && Equal.equals(this.track, that.track);
  }
  pipe() {
    return pipeArguments(this, arguments);
  }
  get get() {
    const acquire = restore => core.flatMap(ref.get(this.isShuttingDown), down => down ? core.interrupt : core.flatten(ref.modify(this.state, state => {
      if (state.free > 0 || state.size >= this.max) {
        return [core.flatMap(queue.take(this.items), attempted => core.exitMatch(attempted.result, {
          onFailure: () => core.succeed(attempted),
          onSuccess: item => core.flatMap(ref.get(this.invalidated), set => {
            if (pipe(set, HashSet.has(item))) {
              return core.zipRight(finalizeInvalid(this, attempted), acquire(restore));
            }
            return core.succeed(attempted);
          })
        })), {
          ...state,
          free: state.free - 1
        }];
      }
      if (state.size >= 0) {
        return [core.zipRight(allocate(this, restore), acquire(restore)), {
          size: state.size + 1,
          free: state.free + 1
        }];
      }
      return [core.interrupt, state];
    })));
    const release = attempted => core.exitMatch(attempted.result, {
      onFailure: () => core.flatten(ref.modify(this.state, state => {
        if (state.size <= this.min) {
          return [allocateUinterruptible(this), {
            ...state,
            free: state.free + 1
          }];
        }
        return [core.unit, {
          ...state,
          size: state.size - 1
        }];
      })),
      onSuccess: item => core.flatMap(ref.get(this.invalidated), set => {
        if (pipe(set, HashSet.has(item))) {
          return finalizeInvalid(this, attempted);
        }
        return pipe(ref.update(this.state, state => ({
          ...state,
          free: state.free + 1
        })), core.zipRight(queue.offer(this.items, attempted)), core.zipRight(this.track(attempted.result)), core.zipRight(core.whenEffect(getAndShutdown(this), ref.get(this.isShuttingDown))));
      })
    });
    return pipe(core.uninterruptibleMask(restore => core.tap(acquire(restore), a => fiberRuntime.addFinalizer(_exit => release(a)))), fiberRuntime.withEarlyRelease, fiberRuntime.disconnect, core.flatMap(([release, attempted]) => pipe(effect.when(release, () => isFailure(attempted)), core.zipRight(toEffect(attempted)))));
  }
  invalidate(item) {
    return ref.update(this.invalidated, HashSet.add(item));
  }
}
const allocate = (self, restore) => core.flatMap(fiberRuntime.scopeMake(), scope => core.flatMap(core.exit(restore(fiberRuntime.scopeExtend(self.creator, scope))), exit => core.flatMap(core.succeed({
  result: exit,
  finalizer: core.scopeClose(scope, core.exitSucceed(void 0))
}), attempted => pipe(queue.offer(self.items, attempted), core.zipRight(self.track(attempted.result)), core.zipRight(core.whenEffect(getAndShutdown(self), ref.get(self.isShuttingDown))), core.as(attempted)))));
const allocateUinterruptible = self => core.uninterruptibleMask(restore => allocate(self, restore));
/**
 * Returns the number of items in the pool in excess of the minimum size.
 */
const excess = self => core.map(ref.get(self.state), state => state.size - Math.min(self.min, state.free));
const finalizeInvalid = (self, attempted) => pipe(forEach(attempted, a => ref.update(self.invalidated, HashSet.remove(a))), core.zipRight(attempted.finalizer), core.zipRight(core.flatten(ref.modify(self.state, state => {
  if (state.size <= self.min) {
    return [allocateUinterruptible(self), {
      ...state,
      free: state.free + 1
    }];
  }
  return [core.unit, {
    ...state,
    size: state.size - 1
  }];
}))));
/**
 * Gets items from the pool and shuts them down as long as there are items
 * free, signalling shutdown of the pool if the pool is empty.
 */
const getAndShutdown = self => core.flatten(ref.modify(self.state, state => {
  if (state.free > 0) {
    return [core.matchCauseEffect(queue.take(self.items), {
      onFailure: () => core.unit,
      onSuccess: attempted => pipe(forEach(attempted, a => ref.update(self.invalidated, HashSet.remove(a))), core.zipRight(attempted.finalizer), core.zipRight(ref.update(self.state, state => ({
        ...state,
        size: state.size - 1
      }))), core.flatMap(() => getAndShutdown(self)))
    }), {
      ...state,
      free: state.free - 1
    }];
  }
  if (state.size > 0) {
    return [core.unit, state];
  }
  return [queue.shutdown(self.items), {
    ...state,
    size: state.size - 1
  }];
}));
/**
 * Begins pre-allocating pool entries based on minimum pool size.
 */
const initialize = self => fiberRuntime.replicateEffect(core.uninterruptibleMask(restore => core.flatten(ref.modify(self.state, state => {
  if (state.size < self.min && state.size >= 0) {
    return [allocate(self, restore), {
      size: state.size + 1,
      free: state.free + 1
    }];
  }
  return [core.unit, state];
}))), self.min, {
  discard: true
});
/**
 * Shrinks the pool down, but never to less than the minimum size.
 */
const shrink = self => core.uninterruptible(core.flatten(ref.modify(self.state, state => {
  if (state.size > self.min && state.free > 0) {
    return [pipe(queue.take(self.items), core.flatMap(attempted => pipe(forEach(attempted, a => ref.update(self.invalidated, HashSet.remove(a))), core.zipRight(attempted.finalizer), core.zipRight(ref.update(self.state, state => ({
      ...state,
      size: state.size - 1
    })))))), {
      ...state,
      free: state.free - 1
    }];
  }
  return [core.unit, state];
})));
const shutdown = self => core.flatten(ref.modify(self.isShuttingDown, down => down ? [queue.awaitShutdown(self.items), true] : [core.zipRight(getAndShutdown(self), queue.awaitShutdown(self.items)), true]));
const isFailure = self => core.exitIsFailure(self.result);
const forEach = (self, f) => core.exitMatch(self.result, {
  onFailure: () => core.unit,
  onSuccess: f
});
const toEffect = self => self.result;
/**
 * A more powerful variant of `make` that allows specifying a `Strategy` that
 * describes how a pool whose excess items are not being used will be shrunk
 * down to the minimum size.
 */
const makeWith = options => core.uninterruptibleMask(restore => pipe(fiberRuntime.all([core.context(), ref.make(false), ref.make({
  size: 0,
  free: 0
}), queue.bounded(options.max), ref.make(HashSet.empty()), options.strategy.initial()]), core.flatMap(([context, down, state, items, inv, initial]) => {
  const pool = new PoolImpl(core.mapInputContext(options.acquire, old => Context.merge(old)(context)), options.min, options.max, down, state, items, inv, exit => options.strategy.track(initial, exit));
  return pipe(fiberRuntime.forkDaemon(restore(initialize(pool))), core.flatMap(fiber => core.flatMap(fiberRuntime.forkDaemon(restore(options.strategy.run(initial, excess(pool), shrink(pool)))), shrink => fiberRuntime.addFinalizer(() => pipe(shutdown(pool), core.zipRight(core.interruptFiber(fiber)), core.zipRight(core.interruptFiber(shrink)))))), core.as(pool));
})));
/** @internal */
export const isPool = u => hasProperty(u, PoolTypeId);
/** @internal */
export const make = options => makeWith({
  acquire: options.acquire,
  min: options.size,
  max: options.size,
  strategy: new NoneStrategy()
});
/** @internal */
export const makeWithTTL = options => makeWith({
  acquire: options.acquire,
  min: options.min,
  max: options.max,
  strategy: new TimeToLiveStrategy(Duration.decode(options.timeToLive))
});
/** @internal */
export const get = self => self.get;
/** @internal */
export const invalidate = /*#__PURE__*/dual(2, (self, value) => self.invalidate(value));
//# sourceMappingURL=pool.js.map