/**
 * This module provides utility functions and type class instances for working with the `string` type in TypeScript.
 * It includes functions for basic string manipulation, as well as type class instances for
 * `Equivalence` and `Order`.
 *
 * @since 2.0.0
 */
import * as equivalence from "./Equivalence.js";
import * as Option from "./Option.js";
import * as order from "./Order.js";
import type * as Ordering from "./Ordering.js";
import type { Refinement } from "./Predicate.js";
import type { NonEmptyArray } from "./ReadonlyArray.js";
/**
 * Tests if a value is a `string`.
 *
 * @param input - The value to test.
 *
 * @example
 * import { isString } from 'effect/String'
 *
 * assert.deepStrictEqual(isString("a"), true)
 * assert.deepStrictEqual(isString(1), false)
 *
 * @category guards
 * @since 2.0.0
 */
export declare const isString: Refinement<unknown, string>;
/**
 * @category instances
 * @since 2.0.0
 */
export declare const Equivalence: equivalence.Equivalence<string>;
/**
 * @category instances
 * @since 2.0.0
 */
export declare const Order: order.Order<string>;
/**
 * The empty string `""`.
 *
 * @since 2.0.0
 */
export declare const empty: "";
/**
 * Concatenates two strings at the type level.
 *
 * @since 2.0.0
 */
export type Concat<A extends string, B extends string> = `${A}${B}`;
/**
 * Concatenates two strings at runtime.
 *
 * @since 2.0.0
 */
export declare const concat: {
    <B extends string>(that: B): <A extends string>(self: A) => Concat<A, B>;
    <A extends string, B extends string>(self: A, that: B): Concat<A, B>;
};
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe('a', S.toUpperCase), 'A')
 *
 * @since 2.0.0
 */
export declare const toUpperCase: <S extends string>(self: S) => Uppercase<S>;
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe('A', S.toLowerCase), 'a')
 *
 * @since 2.0.0
 */
export declare const toLowerCase: <T extends string>(self: T) => Lowercase<T>;
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe('abc', S.capitalize), 'Abc')
 *
 * @since 2.0.0
 */
export declare const capitalize: <T extends string>(self: T) => Capitalize<T>;
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe('ABC', S.uncapitalize), 'aBC')
 *
 * @since 2.0.0
 */
export declare const uncapitalize: <T extends string>(self: T) => Uncapitalize<T>;
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe('abc', S.replace('b', 'd')), 'adc')
 *
 * @since 2.0.0
 */
export declare const replace: (searchValue: string | RegExp, replaceValue: string) => (self: string) => string;
/**
 * @since 2.0.0
 */
export type Trim<A extends string> = TrimEnd<TrimStart<A>>;
/**
 * @example
 * import * as S from 'effect/String'
 *
 * assert.deepStrictEqual(S.trim(' a '), 'a')
 *
 * @since 2.0.0
 */
export declare const trim: <A extends string>(self: A) => TrimEnd<TrimStart<A>>;
/**
 * @since 2.0.0
 */
export type TrimStart<A extends string> = A extends ` ${infer B}` ? TrimStart<B> : A extends `\n${infer B}` ? TrimStart<B> : A extends `\t${infer B}` ? TrimStart<B> : A extends `\r${infer B}` ? TrimStart<B> : A;
/**
 * @example
 * import * as S from 'effect/String'
 *
 * assert.deepStrictEqual(S.trimStart(' a '), 'a ')
 *
 * @since 2.0.0
 */
export declare const trimStart: <A extends string>(self: A) => TrimStart<A>;
/**
 * @since 2.0.0
 */
export type TrimEnd<A extends string> = A extends `${infer B} ` ? TrimEnd<B> : A extends `${infer B}\n` ? TrimEnd<B> : A extends `${infer B}\t` ? TrimEnd<B> : A extends `${infer B}\r` ? TrimEnd<B> : A;
/**
 * @example
 * import * as S from 'effect/String'
 *
 * assert.deepStrictEqual(S.trimEnd(' a '), ' a')
 *
 * @since 2.0.0
 */
export declare const trimEnd: <A extends string>(self: A) => TrimEnd<A>;
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe('abcd', S.slice(1, 3)), 'bc')
 *
 * @since 2.0.0
 */
export declare const slice: (start?: number, end?: number) => (self: string) => string;
/**
 * Test whether a `string` is empty.
 *
 * @example
 * import * as S from 'effect/String'
 *
 * assert.deepStrictEqual(S.isEmpty(''), true)
 * assert.deepStrictEqual(S.isEmpty('a'), false)
 *
 * @since 2.0.0
 */
export declare const isEmpty: (self: string) => self is "";
/**
 * Test whether a `string` is non empty.
 *
 * @since 2.0.0
 */
export declare const isNonEmpty: (self: string) => boolean;
/**
 * Calculate the number of characters in a `string`.
 *
 * @example
 * import * as S from 'effect/String'
 *
 * assert.deepStrictEqual(S.length('abc'), 3)
 *
 * @since 2.0.0
 */
export declare const length: (self: string) => number;
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe('abc', S.split('')), ['a', 'b', 'c'])
 * assert.deepStrictEqual(pipe('', S.split('')), [''])
 *
 * @since 2.0.0
 */
export declare const split: {
    (separator: string | RegExp): (self: string) => NonEmptyArray<string>;
    (self: string, separator: string | RegExp): NonEmptyArray<string>;
};
/**
 * Returns `true` if `searchString` appears as a substring of `self`, at one or more positions that are
 * greater than or equal to `position`; otherwise, returns `false`.
 *
 * @since 2.0.0
 */
export declare const includes: (searchString: string, position?: number) => (self: string) => boolean;
/**
 * @since 2.0.0
 */
export declare const startsWith: (searchString: string, position?: number) => (self: string) => boolean;
/**
 * @since 2.0.0
 */
export declare const endsWith: (searchString: string, position?: number) => (self: string) => boolean;
/**
 * @example
 * import * as S from 'effect/String'
 * import * as Option from 'effect/Option'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe("abc", S.charCodeAt(1)), Option.some(98))
 * assert.deepStrictEqual(pipe("abc", S.charCodeAt(4)), Option.none())
 *
 * @since 2.0.0
 */
export declare const charCodeAt: {
    (index: number): (self: string) => Option.Option<number>;
    (self: string, index: number): Option.Option<number>;
};
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe("abcd", S.substring(1)), "bcd")
 * assert.deepStrictEqual(pipe("abcd", S.substring(1, 3)), "bc")
 *
 * @since 2.0.0
 */
export declare const substring: (start: number, end?: number) => (self: string) => string;
/**
 * @example
 * import * as S from 'effect/String'
 * import * as Option from 'effect/Option'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe("abc", S.at(1)), Option.some("b"))
 * assert.deepStrictEqual(pipe("abc", S.at(4)), Option.none())
 *
 * @since 2.0.0
 */
export declare const at: {
    (index: number): (self: string) => Option.Option<string>;
    (self: string, index: number): Option.Option<string>;
};
/**
 * @example
 * import * as S from 'effect/String'
 * import * as Option from 'effect/Option'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe("abc", S.charAt(1)), Option.some("b"))
 * assert.deepStrictEqual(pipe("abc", S.charAt(4)), Option.none())
 *
 * @since 2.0.0
 */
export declare const charAt: {
    (index: number): (self: string) => Option.Option<string>;
    (self: string, index: number): Option.Option<string>;
};
/**
 * @example
 * import * as S from 'effect/String'
 * import * as Option from 'effect/Option'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe("abc", S.codePointAt(1)), Option.some(98))
 *
 * @since 2.0.0
 */
export declare const codePointAt: {
    (index: number): (self: string) => Option.Option<number>;
    (self: string, index: number): Option.Option<number>;
};
/**
 * @example
 * import * as S from 'effect/String'
 * import * as Option from 'effect/Option'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe("abbbc", S.indexOf("b")), Option.some(1))
 *
 * @since 2.0.0
 */
export declare const indexOf: (searchString: string) => (self: string) => Option.Option<number>;
/**
 * @example
 * import * as S from 'effect/String'
 * import * as Option from 'effect/Option'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe("abbbc", S.lastIndexOf("b")), Option.some(3))
 * assert.deepStrictEqual(pipe("abbbc", S.lastIndexOf("d")), Option.none())
 *
 * @since 2.0.0
 */
export declare const lastIndexOf: (searchString: string) => (self: string) => Option.Option<number>;
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe("a", S.localeCompare("b")), -1)
 * assert.deepStrictEqual(pipe("b", S.localeCompare("a")), 1)
 * assert.deepStrictEqual(pipe("a", S.localeCompare("a")), 0)
 *
 * @since 2.0.0
 */
export declare const localeCompare: (that: string, locales?: Array<string>, options?: Intl.CollatorOptions) => (self: string) => Ordering.Ordering;
/**
 * It is the `pipe`-able version of the native `match` method.
 *
 * @since 2.0.0
 */
export declare const match: (regexp: RegExp | string) => (self: string) => Option.Option<RegExpMatchArray>;
/**
 * It is the `pipe`-able version of the native `matchAll` method.
 *
 * @since 2.0.0
 */
export declare const matchAll: (regexp: RegExp) => (self: string) => IterableIterator<RegExpMatchArray>;
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * const str = "\u1E9B\u0323";
 * assert.deepStrictEqual(pipe(str, S.normalize()), "\u1E9B\u0323")
 * assert.deepStrictEqual(pipe(str, S.normalize("NFC")), "\u1E9B\u0323")
 * assert.deepStrictEqual(pipe(str, S.normalize("NFD")), "\u017F\u0323\u0307")
 * assert.deepStrictEqual(pipe(str, S.normalize("NFKC")), "\u1E69")
 * assert.deepStrictEqual(pipe(str, S.normalize("NFKD")), "\u0073\u0323\u0307")
 *
 * @since 2.0.0
 */
export declare const normalize: (form?: "NFC" | "NFD" | "NFKC" | "NFKD") => (self: string) => string;
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe("a", S.padEnd(5)), "a    ")
 * assert.deepStrictEqual(pipe("a", S.padEnd(5, "_")), "a____")
 *
 * @since 2.0.0
 */
export declare const padEnd: (maxLength: number, fillString?: string) => (self: string) => string;
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe("a", S.padStart(5)), "    a")
 * assert.deepStrictEqual(pipe("a", S.padStart(5, "_")), "____a")
 *
 * @since 2.0.0
 */
export declare const padStart: (maxLength: number, fillString?: string) => (self: string) => string;
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe("a", S.repeat(5)), "aaaaa")
 *
 * @since 2.0.0
 */
export declare const repeat: (count: number) => (self: string) => string;
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe("ababb", S.replaceAll("b", "c")), "acacc")
 * assert.deepStrictEqual(pipe("ababb", S.replaceAll(/ba/g, "cc")), "accbb")
 *
 * @since 2.0.0
 */
export declare const replaceAll: (searchValue: string | RegExp, replaceValue: string) => (self: string) => string;
/**
 * @example
 * import * as S from 'effect/String'
 * import * as Option from 'effect/Option'
 * import { pipe } from 'effect/Function'
 *
 * assert.deepStrictEqual(pipe("ababb", S.search("b")), Option.some(1))
 * assert.deepStrictEqual(pipe("ababb", S.search(/abb/)), Option.some(2))
 * assert.deepStrictEqual(pipe("ababb", S.search("d")), Option.none())
 *
 * @since 2.0.0
 */
export declare const search: {
    (regexp: RegExp | string): (self: string) => Option.Option<number>;
    (self: string, regexp: RegExp | string): Option.Option<number>;
};
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * const str = "\u0130"
 * assert.deepStrictEqual(pipe(str, S.toLocaleLowerCase("tr")), "i")
 *
 * @since 2.0.0
 */
export declare const toLocaleLowerCase: (locale?: string | Array<string>) => (self: string) => string;
/**
 * @example
 * import * as S from 'effect/String'
 * import { pipe } from 'effect/Function'
 *
 * const str = "i\u0307"
 * assert.deepStrictEqual(pipe(str, S.toLocaleUpperCase("lt-LT")), "I")
 *
 * @since 2.0.0
 */
export declare const toLocaleUpperCase: (locale?: string | Array<string>) => (self: string) => string;
/**
 * Keep the specified number of characters from the start of a string.
 *
 * If `n` is larger than the available number of characters, the string will
 * be returned whole.
 *
 * If `n` is not a positive number, an empty string will be returned.
 *
 * If `n` is a float, it will be rounded down to the nearest integer.
 *
 * @example
 * import * as S from 'effect/String'
 *
 * assert.deepStrictEqual(S.takeLeft("Hello World", 5), "Hello")
 *
 * @since 2.0.0
 */
export declare const takeLeft: {
    (n: number): (self: string) => string;
    (self: string, n: number): string;
};
/**
 * Keep the specified number of characters from the end of a string.
 *
 * If `n` is larger than the available number of characters, the string will
 * be returned whole.
 *
 * If `n` is not a positive number, an empty string will be returned.
 *
 * If `n` is a float, it will be rounded down to the nearest integer.
 *
 * @example
 * import * as S from 'effect/String'
 *
 * assert.deepStrictEqual(S.takeRight("Hello World", 5), "World")
 *
 * @since 2.0.0
 */
export declare const takeRight: {
    (n: number): (self: string) => string;
    (self: string, n: number): string;
};
/**
 * Returns an `IterableIterator` which yields each line contained within the
 * string, trimming off the trailing newline character.
 *
 * @since 2.0.0
 */
/**
 * Returns an `IterableIterator` which yields each line contained within the
 * string as well as the trailing newline character.
 *
 * @since 2.0.0
 */
export declare const linesWithSeparators: (s: string) => LinesIterator;
/**
 * For every line in this string, strip a leading prefix consisting of blanks
 * or control characters followed by the character specified by `marginChar`
 * from the line.
 *
 * @since 2.0.0
 */
export declare const stripMarginWith: {
    (marginChar: string): (self: string) => string;
    (self: string, marginChar: string): string;
};
/**
 * For every line in this string, strip a leading prefix consisting of blanks
 * or control characters followed by the `"|"` character from the line.
 *
 * @since 2.0.0
 */
export declare const stripMargin: (self: string) => string;
declare class LinesIterator implements IterableIterator<string> {
    readonly s: string;
    readonly stripped: boolean;
    private index;
    private readonly length;
    constructor(s: string, stripped?: boolean);
    next(): IteratorResult<string>;
    [Symbol.iterator](): IterableIterator<string>;
    private get done();
}
export {};
//# sourceMappingURL=String.d.ts.map