/**
 * This module provides utility functions and type class instances for working with the `bigint` type in TypeScript.
 * It includes functions for basic arithmetic operations, as well as type class instances for
 * `Equivalence` and `Order`.
 *
 * @since 2.0.0
 */
import * as equivalence from "./Equivalence.js";
import * as Option from "./Option.js";
import * as order from "./Order.js";
import type { Ordering } from "./Ordering.js";
/**
 * Tests if a value is a `bigint`.
 *
 * @param input - The value to test.
 *
 * @example
 * import { isBigInt } from "effect/BigInt"
 *
 * assert.deepStrictEqual(isBigInt(1n), true)
 * assert.deepStrictEqual(isBigInt(1), false)
 *
 * @category guards
 * @since 2.0.0
 */
export declare const isBigInt: (u: unknown) => u is bigint;
/**
 * Provides an addition operation on `bigint`s.
 *
 * @param self - The first operand.
 * @param that - The second operand.
 *
 * @example
 * import { sum } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(sum(2n, 3n), 5n)
 *
 * @category math
 * @since 2.0.0
 */
export declare const sum: {
    (that: bigint): (self: bigint) => bigint;
    (self: bigint, that: bigint): bigint;
};
/**
 * Provides a multiplication operation on `bigint`s.
 *
 * @param self - The first operand.
 * @param that - The second operand.
 *
 * @example
 * import { multiply } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(multiply(2n, 3n), 6n)
 *
 * @category math
 * @since 2.0.0
 */
export declare const multiply: {
    (that: bigint): (self: bigint) => bigint;
    (self: bigint, that: bigint): bigint;
};
/**
 * Provides a subtraction operation on `bigint`s.
 *
 * @param self - The first operand.
 * @param that - The second operand.
 *
 * @example
 * import { subtract } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(subtract(2n, 3n), -1n)
 *
 * @category math
 * @since 2.0.0
 */
export declare const subtract: {
    (that: bigint): (self: bigint) => bigint;
    (self: bigint, that: bigint): bigint;
};
/**
 * Provides a division operation on `bigint`s.
 *
 * If the dividend is not a multiple of the divisor the result will be a `bigint` value
 * which represents the integer division rounded down to the nearest integer.
 *
 * Returns `None` if the divisor is `0n`.
 *
 * @param self - The dividend operand.
 * @param that - The divisor operand.
 *
 * @example
 * import { divide } from 'effect/BigInt'
 * import { some, none } from 'effect/Option'
 *
 * assert.deepStrictEqual(divide(6n, 3n), some(2n))
 * assert.deepStrictEqual(divide(6n, 0n), none())
 *
 * @category math
 * @since 2.0.0
 */
export declare const divide: {
    (that: bigint): (self: bigint) => Option.Option<bigint>;
    (self: bigint, that: bigint): Option.Option<bigint>;
};
/**
 * Provides a division operation on `bigint`s.
 *
 * If the dividend is not a multiple of the divisor the result will be a `bigint` value
 * which represents the integer division rounded down to the nearest integer.
 *
 * Throws a `RangeError` if the divisor is `0n`.
 *
 * @param self - The dividend operand.
 * @param that - The divisor operand.
 *
 * @example
 * import { unsafeDivide } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(unsafeDivide(6n, 3n), 2n)
 * assert.deepStrictEqual(unsafeDivide(6n, 4n), 1n)
 *
 * @category math
 * @since 2.0.0
 */
export declare const unsafeDivide: {
    (that: bigint): (self: bigint) => bigint;
    (self: bigint, that: bigint): bigint;
};
/**
 * Returns the result of adding `1n` to a given number.
 *
 * @param n - A `bigint` to be incremented.
 *
 * @example
 * import { increment } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(increment(2n), 3n)
 *
 * @category math
 * @since 2.0.0
 */
export declare const increment: (n: bigint) => bigint;
/**
 * Decrements a number by `1n`.
 *
 * @param n - A `bigint` to be decremented.
 *
 * @example
 * import { decrement } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(decrement(3n), 2n)
 *
 * @category math
 * @since 2.0.0
 */
export declare const decrement: (n: bigint) => bigint;
/**
 * @category instances
 * @since 2.0.0
 */
export declare const Equivalence: equivalence.Equivalence<bigint>;
/**
 * @category instances
 * @since 2.0.0
 */
export declare const Order: order.Order<bigint>;
/**
 * Returns `true` if the first argument is less than the second, otherwise `false`.
 *
 * @param self - The first argument.
 * @param that - The second argument.
 *
 * @example
 * import { lessThan } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(lessThan(2n, 3n), true)
 * assert.deepStrictEqual(lessThan(3n, 3n), false)
 * assert.deepStrictEqual(lessThan(4n, 3n), false)
 *
 * @category predicates
 * @since 2.0.0
 */
export declare const lessThan: {
    (that: bigint): (self: bigint) => boolean;
    (self: bigint, that: bigint): boolean;
};
/**
 * Returns a function that checks if a given `bigint` is less than or equal to the provided one.
 *
 * @param self - The first `bigint` to compare with.
 * @param that - The second `bigint` to compare with.
 *
 * @example
 * import { lessThanOrEqualTo } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(lessThanOrEqualTo(2n, 3n), true)
 * assert.deepStrictEqual(lessThanOrEqualTo(3n, 3n), true)
 * assert.deepStrictEqual(lessThanOrEqualTo(4n, 3n), false)
 *
 * @category predicates
 * @since 2.0.0
 */
export declare const lessThanOrEqualTo: {
    (that: bigint): (self: bigint) => boolean;
    (self: bigint, that: bigint): boolean;
};
/**
 * Returns `true` if the first argument is greater than the second, otherwise `false`.
 *
 * @param self - The first argument.
 * @param that - The second argument.
 *
 * @example
 * import { greaterThan } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(greaterThan(2n, 3n), false)
 * assert.deepStrictEqual(greaterThan(3n, 3n), false)
 * assert.deepStrictEqual(greaterThan(4n, 3n), true)
 *
 * @category predicates
 * @since 2.0.0
 */
export declare const greaterThan: {
    (that: bigint): (self: bigint) => boolean;
    (self: bigint, that: bigint): boolean;
};
/**
 * Returns a function that checks if a given `bigint` is greater than or equal to the provided one.
 *
 * @param self - The first `bigint` to compare with.
 * @param that - The second `bigint` to compare with.
 *
 * @example
 * import { greaterThanOrEqualTo } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(greaterThanOrEqualTo(2n, 3n), false)
 * assert.deepStrictEqual(greaterThanOrEqualTo(3n, 3n), true)
 * assert.deepStrictEqual(greaterThanOrEqualTo(4n, 3n), true)
 *
 * @category predicates
 * @since 2.0.0
 */
export declare const greaterThanOrEqualTo: {
    (that: bigint): (self: bigint) => boolean;
    (self: bigint, that: bigint): boolean;
};
/**
 * Checks if a `bigint` is between a `minimum` and `maximum` value (inclusive).
 *
 * @param self - The `number` to check.
 * @param minimum - The `minimum` value to check.
 * @param maximum - The `maximum` value to check.
 *
 * @example
 * import * as BigInt from 'effect/BigInt'
 *
 * const between = BigInt.between({ minimum: 0n, maximum: 5n })
 *
 * assert.deepStrictEqual(between(3n), true)
 * assert.deepStrictEqual(between(-1n), false)
 * assert.deepStrictEqual(between(6n), false)
 *
 * @category predicates
 * @since 2.0.0
 */
export declare const between: {
    (options: {
        minimum: bigint;
        maximum: bigint;
    }): (self: bigint) => boolean;
    (self: bigint, options: {
        minimum: bigint;
        maximum: bigint;
    }): boolean;
};
/**
 * Restricts the given `bigint` to be within the range specified by the `minimum` and `maximum` values.
 *
 * - If the `bigint` is less than the `minimum` value, the function returns the `minimum` value.
 * - If the `bigint` is greater than the `maximum` value, the function returns the `maximum` value.
 * - Otherwise, it returns the original `bigint`.
 *
 * @param self - The `bigint` to be clamped.
 * @param minimum - The lower end of the range.
 * @param maximum - The upper end of the range.
 *
 * @example
 * import * as BigInt from 'effect/BigInt'
 *
 * const clamp = BigInt.clamp({ minimum: 1n, maximum: 5n })
 *
 * assert.equal(clamp(3n), 3n)
 * assert.equal(clamp(0n), 1n)
 * assert.equal(clamp(6n), 5n)
 *
 * @since 2.0.0
 */
export declare const clamp: {
    (options: {
        minimum: bigint;
        maximum: bigint;
    }): (self: bigint) => bigint;
    (self: bigint, options: {
        minimum: bigint;
        maximum: bigint;
    }): bigint;
};
/**
 * Returns the minimum between two `bigint`s.
 *
 * @param self - The first `bigint`.
 * @param that - The second `bigint`.
 *
 * @example
 * import { min } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(min(2n, 3n), 2n)
 *
 * @since 2.0.0
 */
export declare const min: {
    (that: bigint): (self: bigint) => bigint;
    (self: bigint, that: bigint): bigint;
};
/**
 * Returns the maximum between two `bigint`s.
 *
 * @param self - The first `bigint`.
 * @param that - The second `bigint`.
 *
 * @example
 * import { max } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(max(2n, 3n), 3n)
 *
 * @since 2.0.0
 */
export declare const max: {
    (that: bigint): (self: bigint) => bigint;
    (self: bigint, that: bigint): bigint;
};
/**
 * Determines the sign of a given `bigint`.
 *
 * @param n - The `bigint` to determine the sign of.
 *
 * @example
 * import { sign } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(sign(-5n), -1)
 * assert.deepStrictEqual(sign(0n), 0)
 * assert.deepStrictEqual(sign(5n), 1)
 *
 * @category math
 * @since 2.0.0
 */
export declare const sign: (n: bigint) => Ordering;
/**
 * Determines the absolute value of a given `bigint`.
 *
 * @param n - The `bigint` to determine the absolute value of.
 *
 * @example
 * import { abs } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(abs(-5n), 5n)
 * assert.deepStrictEqual(abs(0n), 0n)
 * assert.deepStrictEqual(abs(5n), 5n)
 *
 * @category math
 * @since 2.0.0
 */
export declare const abs: (n: bigint) => bigint;
/**
 * Determines the greatest common divisor of two `bigint`s.
 *
 * @param a - The first `bigint`.
 * @param b - The second `bigint`.
 *
 * @example
 * import { gcd } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(gcd(2n, 3n), 1n)
 * assert.deepStrictEqual(gcd(2n, 4n), 2n)
 * assert.deepStrictEqual(gcd(16n, 24n), 8n)
 *
 * @category math
 * @since 2.0.0
 */
export declare const gcd: {
    (that: bigint): (self: bigint) => bigint;
    (self: bigint, that: bigint): bigint;
};
/**
 * Determines the least common multiple of two `bigint`s.
 *
 * @param a - The first `bigint`.
 * @param b - The second `bigint`.
 *
 * @example
 * import { lcm } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(lcm(2n, 3n), 6n)
 * assert.deepStrictEqual(lcm(2n, 4n), 4n)
 * assert.deepStrictEqual(lcm(16n, 24n), 48n)
 *
 * @category math
 * @since 2.0.0
 */
export declare const lcm: {
    (that: bigint): (self: bigint) => bigint;
    (self: bigint, that: bigint): bigint;
};
/**
 * Determines the square root of a given `bigint` unsafely. Throws if the given `bigint` is negative.
 *
 * @param n - The `bigint` to determine the square root of.
 *
 * @example
 * import { unsafeSqrt } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(unsafeSqrt(4n), 2n)
 * assert.deepStrictEqual(unsafeSqrt(9n), 3n)
 * assert.deepStrictEqual(unsafeSqrt(16n), 4n)
 *
 * @category math
 * @since 2.0.0
 */
export declare const unsafeSqrt: (n: bigint) => bigint;
/**
 * Determines the square root of a given `bigint` safely. Returns `none` if the given `bigint` is negative.
 *
 * @param n - The `bigint` to determine the square root of.
 *
 * @example
 * import { sqrt } from 'effect/BigInt'
 * import * as Option from 'effect/Option'
 *
 * assert.deepStrictEqual(sqrt(4n), Option.some(2n))
 * assert.deepStrictEqual(sqrt(9n), Option.some(3n))
 * assert.deepStrictEqual(sqrt(16n), Option.some(4n))
 * assert.deepStrictEqual(sqrt(-1n), Option.none())
 *
 * @category math
 * @since 2.0.0
 */
export declare const sqrt: (n: bigint) => Option.Option<bigint>;
/**
 * Takes an `Iterable` of `bigint`s and returns their sum as a single `bigint
 *
 * @param collection - The collection of `bigint`s to sum.
 *
 * @example
 * import { sumAll } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(sumAll([2n, 3n, 4n]), 9n)
 *
 * @category math
 * @since 2.0.0
 */
export declare const sumAll: (collection: Iterable<bigint>) => bigint;
/**
 * Takes an `Iterable` of `bigint`s and returns their multiplication as a single `number`.
 *
 * @param collection - The collection of `bigint`s to multiply.
 *
 * @example
 * import { multiplyAll } from 'effect/BigInt'
 *
 * assert.deepStrictEqual(multiplyAll([2n, 3n, 4n]), 24n)
 *
 * @category math
 * @since 2.0.0
 */
export declare const multiplyAll: (collection: Iterable<bigint>) => bigint;
//# sourceMappingURL=BigInt.d.ts.map