/**
 * This module provides utility functions and type class instances for working with the `BigDecimal` type in TypeScript.
 * It includes functions for basic arithmetic operations, as well as type class instances for `Equivalence` and `Order`.
 *
 * A `BigDecimal` allows storing any real number to arbitrary precision; which avoids common floating point errors
 * (such as 0.1 + 0.2 ≠ 0.3) at the cost of complexity.
 *
 * Internally, `BigDecimal` uses a `BigInt` object, paired with a 64-bit integer which determines the position of the
 * decimal point. Therefore, the precision *is not* actually arbitrary, but limited to 2<sup>63</sup> decimal places.
 *
 * It is not recommended to convert a floating point number to a decimal directly, as the floating point representation
 * may be unexpected.
 *
 * @since 2.0.0
 */
import * as Equal from "./Equal.js";
import * as equivalence from "./Equivalence.js";
import { type Inspectable } from "./Inspectable.js";
import * as Option from "./Option.js";
import * as order from "./Order.js";
import type { Ordering } from "./Ordering.js";
import { type Pipeable } from "./Pipeable.js";
/**
 * @since 2.0.0
 * @category symbols
 */
export declare const TypeId: unique symbol;
/**
 * @since 2.0.0
 * @category symbol
 */
export type TypeId = typeof TypeId;
/**
 * @since 2.0.0
 * @category models
 */
export interface BigDecimal extends Equal.Equal, Pipeable, Inspectable {
    readonly [TypeId]: TypeId;
    readonly value: bigint;
    readonly scale: number;
}
/**
 * Checks if a given value is a `BigDecimal`.
 *
 * @param u - The value to check.
 *
 * @since 2.0.0
 * @category guards
 */
export declare const isBigDecimal: (u: unknown) => u is BigDecimal;
/**
 * Creates a `BigDecimal` from a `bigint` value and a scale.
 *
 * @param value - The `bigint` value to create a `BigDecimal` from.
 * @param scale - The scale of the `BigDecimal`.
 *
 * @since 2.0.0
 * @category constructors
 */
export declare const make: (value: bigint, scale: number) => BigDecimal;
/**
 * Normalizes a given `BigDecimal` by removing trailing zeros.
 *
 * @param self - The `BigDecimal` to normalize.
 *
 * @example
 * import { normalize, make, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(normalize(unsafeFromString("123.00000")), normalize(make(123n, 0)))
 * assert.deepStrictEqual(normalize(unsafeFromString("12300000")), normalize(make(123n, -5)))
 *
 * @since 2.0.0
 * @category scaling
 */
export declare const normalize: (self: BigDecimal) => BigDecimal;
/**
 * Scales a given `BigDecimal` to the specified scale.
 *
 * If the given scale is smaller than the current scale, the value will be rounded down to
 * the nearest integer.
 *
 * @param self - The `BigDecimal` to scale.
 * @param scale - The scale to scale to.
 *
 * @since 2.0.0
 * @category scaling
 */
export declare const scale: (self: BigDecimal, scale: number) => BigDecimal;
/**
 * Provides an addition operation on `BigDecimal`s.
 *
 * @param self - The first operand.
 * @param that - The second operand.
 *
 * @example
 * import { sum, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(sum(unsafeFromString("2"), unsafeFromString("3")), unsafeFromString("5"))
 *
 * @since 2.0.0
 * @category math
 */
export declare const sum: {
    (that: BigDecimal): (self: BigDecimal) => BigDecimal;
    (self: BigDecimal, that: BigDecimal): BigDecimal;
};
/**
 * Provides a multiplication operation on `BigDecimal`s.
 *
 * @param self - The first operand.
 * @param that - The second operand.
 *
 * @example
 * import { multiply, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(multiply(unsafeFromString("2"), unsafeFromString("3")), unsafeFromString("6"))
 *
 * @since 2.0.0
 * @category math
 */
export declare const multiply: {
    (that: BigDecimal): (self: BigDecimal) => BigDecimal;
    (self: BigDecimal, that: BigDecimal): BigDecimal;
};
/**
 * Provides a subtraction operation on `BigDecimal`s.
 *
 * @param self - The first operand.
 * @param that - The second operand.
 *
 * @example
 * import { subtract, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(subtract(unsafeFromString("2"), unsafeFromString("3")), unsafeFromString("-1"))
 *
 * @since 2.0.0
 * @category math
 */
export declare const subtract: {
    (that: BigDecimal): (self: BigDecimal) => BigDecimal;
    (self: BigDecimal, that: BigDecimal): BigDecimal;
};
/**
 * Provides a division operation on `BigDecimal`s.
 *
 * If the dividend is not a multiple of the divisor the result will be a `BigDecimal` value
 * which represents the integer division rounded down to the nearest integer.
 *
 * If the divisor is `0`, the result will be `None`.
 *
 * @param self - The dividend operand.
 * @param that - The divisor operand.
 *
 * @example
 * import { divide, unsafeFromString } from "effect/BigDecimal"
 * import { some, none } from "effect/Option"
 *
 * assert.deepStrictEqual(divide(unsafeFromString("6"), unsafeFromString("3")), some(unsafeFromString("2")))
 * assert.deepStrictEqual(divide(unsafeFromString("6"), unsafeFromString("4")), some(unsafeFromString("1.5")))
 * assert.deepStrictEqual(divide(unsafeFromString("6"), unsafeFromString("0")), none())
 *
 * @since 2.0.0
 * @category math
 */
export declare const divide: {
    (that: BigDecimal): (self: BigDecimal) => Option.Option<BigDecimal>;
    (self: BigDecimal, that: BigDecimal): Option.Option<BigDecimal>;
};
/**
 * Provides an unsafe division operation on `BigDecimal`s.
 *
 * If the dividend is not a multiple of the divisor the result will be a `BigDecimal` value
 * which represents the integer division rounded down to the nearest integer.
 *
 * Throws a `RangeError` if the divisor is `0`.
 *
 * @param self - The dividend operand.
 * @param that - The divisor operand.as
 *
 * @example
 * import { unsafeDivide, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(unsafeDivide(unsafeFromString("6"), unsafeFromString("3")), unsafeFromString("2"))
 * assert.deepStrictEqual(unsafeDivide(unsafeFromString("6"), unsafeFromString("4")), unsafeFromString("1.5"))
 *
 * @since 2.0.0
 * @category math
 */
export declare const unsafeDivide: {
    (that: BigDecimal): (self: BigDecimal) => BigDecimal;
    (self: BigDecimal, that: BigDecimal): BigDecimal;
};
/**
 * @since 2.0.0
 * @category instances
 */
export declare const Order: order.Order<BigDecimal>;
/**
 * Returns `true` if the first argument is less than the second, otherwise `false`.
 *
 * @param self - The first argument.
 * @param that - The second argument.
 *
 * @example
 * import { lessThan, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(lessThan(unsafeFromString("2"), unsafeFromString("3")), true)
 * assert.deepStrictEqual(lessThan(unsafeFromString("3"), unsafeFromString("3")), false)
 * assert.deepStrictEqual(lessThan(unsafeFromString("4"), unsafeFromString("3")), false)
 *
 * @since 2.0.0
 * @category predicates
 */
export declare const lessThan: {
    (that: BigDecimal): (self: BigDecimal) => boolean;
    (self: BigDecimal, that: BigDecimal): boolean;
};
/**
 * Checks if a given `BigDecimal` is less than or equal to the provided one.
 *
 * @param self - The first `BigDecimal` to compare with.
 * @param that - The second `BigDecimal` to compare with.
 *
 * @example
 * import { lessThanOrEqualTo, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(lessThanOrEqualTo(unsafeFromString("2"), unsafeFromString("3")), true)
 * assert.deepStrictEqual(lessThanOrEqualTo(unsafeFromString("3"), unsafeFromString("3")), true)
 * assert.deepStrictEqual(lessThanOrEqualTo(unsafeFromString("4"), unsafeFromString("3")), false)
 *
 * @since 2.0.0
 * @category predicates
 */
export declare const lessThanOrEqualTo: {
    (that: BigDecimal): (self: BigDecimal) => boolean;
    (self: BigDecimal, that: BigDecimal): boolean;
};
/**
 * Returns `true` if the first argument is greater than the second, otherwise `false`.
 *
 * @param self - The first argument.
 * @param that - The second argument.
 *
 * @example
 * import { greaterThan, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(greaterThan(unsafeFromString("2"), unsafeFromString("3")), false)
 * assert.deepStrictEqual(greaterThan(unsafeFromString("3"), unsafeFromString("3")), false)
 * assert.deepStrictEqual(greaterThan(unsafeFromString("4"), unsafeFromString("3")), true)
 *
 * @since 2.0.0
 * @category predicates
 */
export declare const greaterThan: {
    (that: BigDecimal): (self: BigDecimal) => boolean;
    (self: BigDecimal, that: BigDecimal): boolean;
};
/**
 * Checks if a given `BigDecimal` is greater than or equal to the provided one.
 *
 * @param self - The first `BigDecimal` to compare with.
 * @param that - The second `BigDecimal` to compare with.
 *
 * @example
 * import { greaterThanOrEqualTo, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(greaterThanOrEqualTo(unsafeFromString("2"), unsafeFromString("3")), false)
 * assert.deepStrictEqual(greaterThanOrEqualTo(unsafeFromString("3"), unsafeFromString("3")), true)
 * assert.deepStrictEqual(greaterThanOrEqualTo(unsafeFromString("4"), unsafeFromString("3")), true)
 *
 * @since 2.0.0
 * @category predicates
 */
export declare const greaterThanOrEqualTo: {
    (that: BigDecimal): (self: BigDecimal) => boolean;
    (self: BigDecimal, that: BigDecimal): boolean;
};
/**
 * Checks if a `BigDecimal` is between a `minimum` and `maximum` value (inclusive).
 *
 * @param self - The `number` to check.
 * @param minimum - The `minimum` value to check.
 * @param maximum - The `maximum` value to check.
 *
 * @example
 * import * as BigDecimal from "effect/BigDecimal"
 *
 * const between = BigDecimal.between({
 *   minimum: BigDecimal.unsafeFromString("1"),
 *   maximum: BigDecimal.unsafeFromString("5") }
 * )
 *
 * assert.deepStrictEqual(between(BigDecimal.unsafeFromString("3")), true)
 * assert.deepStrictEqual(between(BigDecimal.unsafeFromString("0")), false)
 * assert.deepStrictEqual(between(BigDecimal.unsafeFromString("6")), false)
 *
 * @since 2.0.0
 * @category predicates
 */
export declare const between: {
    (options: {
        minimum: BigDecimal;
        maximum: BigDecimal;
    }): (self: BigDecimal) => boolean;
    (self: BigDecimal, options: {
        minimum: BigDecimal;
        maximum: BigDecimal;
    }): boolean;
};
/**
 * Restricts the given `BigDecimal` to be within the range specified by the `minimum` and `maximum` values.
 *
 * - If the `BigDecimal` is less than the `minimum` value, the function returns the `minimum` value.
 * - If the `BigDecimal` is greater than the `maximum` value, the function returns the `maximum` value.
 * - Otherwise, it returns the original `BigDecimal`.
 *
 * @param self - The `BigDecimal` to be clamped.
 * @param minimum - The lower end of the range.
 * @param maximum - The upper end of the range.
 *
 * @example
 * import * as BigDecimal from "effect/BigDecimal"
 *
 * const clamp = BigDecimal.clamp({
 *   minimum: BigDecimal.unsafeFromString("1"),
 *   maximum: BigDecimal.unsafeFromString("5") }
 * )
 *
 * assert.deepStrictEqual(clamp(BigDecimal.unsafeFromString("3")), BigDecimal.unsafeFromString("3"))
 * assert.deepStrictEqual(clamp(BigDecimal.unsafeFromString("0")), BigDecimal.unsafeFromString("1"))
 * assert.deepStrictEqual(clamp(BigDecimal.unsafeFromString("6")), BigDecimal.unsafeFromString("5"))
 *
 * @since 2.0.0
 * @category math
 */
export declare const clamp: {
    (options: {
        minimum: BigDecimal;
        maximum: BigDecimal;
    }): (self: BigDecimal) => BigDecimal;
    (self: BigDecimal, options: {
        minimum: BigDecimal;
        maximum: BigDecimal;
    }): BigDecimal;
};
/**
 * Returns the minimum between two `BigDecimal`s.
 *
 * @param self - The first `BigDecimal`.
 * @param that - The second `BigDecimal`.
 *
 * @example
 * import { min, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(min(unsafeFromString("2"), unsafeFromString("3")), unsafeFromString("2"))
 *
 * @since 2.0.0
 * @category math
 */
export declare const min: {
    (that: BigDecimal): (self: BigDecimal) => BigDecimal;
    (self: BigDecimal, that: BigDecimal): BigDecimal;
};
/**
 * Returns the maximum between two `BigDecimal`s.
 *
 * @param self - The first `BigDecimal`.
 * @param that - The second `BigDecimal`.
 *
 * @example
 * import { max, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(max(unsafeFromString("2"), unsafeFromString("3")), unsafeFromString("3"))
 *
 * @since 2.0.0
 * @category math
 */
export declare const max: {
    (that: BigDecimal): (self: BigDecimal) => BigDecimal;
    (self: BigDecimal, that: BigDecimal): BigDecimal;
};
/**
 * Determines the sign of a given `BigDecimal`.
 *
 * @param n - The `BigDecimal` to determine the sign of.
 *
 * @example
 * import { sign, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(sign(unsafeFromString("-5")), -1)
 * assert.deepStrictEqual(sign(unsafeFromString("0")), 0)
 * assert.deepStrictEqual(sign(unsafeFromString("5")), 1)
 *
 * @since 2.0.0
 * @category math
 */
export declare const sign: (n: BigDecimal) => Ordering;
/**
 * Determines the absolute value of a given `BigDecimal`.
 *
 * @param n - The `BigDecimal` to determine the absolute value of.
 *
 * @example
 * import { abs, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(abs(unsafeFromString("-5")), unsafeFromString("5"))
 * assert.deepStrictEqual(abs(unsafeFromString("0")), unsafeFromString("0"))
 * assert.deepStrictEqual(abs(unsafeFromString("5")), unsafeFromString("5"))
 *
 * @since 2.0.0
 * @category math
 */
export declare const abs: (n: BigDecimal) => BigDecimal;
/**
 * Provides a negate operation on `BigDecimal`s.
 *
 * @param n - The `BigDecimal` to negate.
 *
 * @example
 * import { negate, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(negate(unsafeFromString("3")), unsafeFromString("-3"))
 * assert.deepStrictEqual(negate(unsafeFromString("-6")), unsafeFromString("6"))
 *
 * @since 2.0.0
 * @category math
 */
export declare const negate: (n: BigDecimal) => BigDecimal;
/**
 * Returns the remainder left over when one operand is divided by a second operand.
 *
 * If the divisor is `0`, the result will be `None`.
 *
 * @param self - The dividend.
 * @param divisor - The divisor.
 *
 * @example
 * import { remainder, unsafeFromString } from "effect/BigDecimal"
 * import { some } from "effect/Option"
 *
 * assert.deepStrictEqual(remainder(unsafeFromString("2"), unsafeFromString("2")), some(unsafeFromString("0")))
 * assert.deepStrictEqual(remainder(unsafeFromString("3"), unsafeFromString("2")), some(unsafeFromString("1")))
 * assert.deepStrictEqual(remainder(unsafeFromString("-4"), unsafeFromString("2")), some(unsafeFromString("0")))
 *
 * @since 2.0.0
 * @category math
 */
export declare const remainder: {
    (divisor: BigDecimal): (self: BigDecimal) => Option.Option<BigDecimal>;
    (self: BigDecimal, divisor: BigDecimal): Option.Option<BigDecimal>;
};
/**
 * Returns the remainder left over when one operand is divided by a second operand.
 *
 * Throws a `RangeError` if the divisor is `0`.
 *
 * @param self - The dividend.
 * @param divisor - The divisor.
 *
 * @example
 * import { unsafeRemainder, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(unsafeRemainder(unsafeFromString("2"), unsafeFromString("2")), unsafeFromString("0"))
 * assert.deepStrictEqual(unsafeRemainder(unsafeFromString("3"), unsafeFromString("2")), unsafeFromString("1"))
 * assert.deepStrictEqual(unsafeRemainder(unsafeFromString("-4"), unsafeFromString("2")), unsafeFromString("0"))
 *
 * @since 2.0.0
 * @category math
 */
export declare const unsafeRemainder: {
    (divisor: BigDecimal): (self: BigDecimal) => BigDecimal;
    (self: BigDecimal, divisor: BigDecimal): BigDecimal;
};
/**
 * @category instances
 * @since 2.0.0
 */
export declare const Equivalence: equivalence.Equivalence<BigDecimal>;
/**
 * Checks if two `BigDecimal`s are equal.
 *
 * @since 2.0.0
 * @category predicates
 */
export declare const equals: {
    (that: BigDecimal): (self: BigDecimal) => boolean;
    (self: BigDecimal, that: BigDecimal): boolean;
};
/**
 * Creates a `BigDecimal` from a `bigint` value.
 *
 * @param value - The `bigint` value to create a `BigDecimal` from.
 *
 * @since 2.0.0
 * @category constructors
 */
export declare const fromBigInt: (n: bigint) => BigDecimal;
/**
 * Creates a `BigDecimal` from a `number` value.
 *
 * It is not recommended to convert a floating point number to a decimal directly,
 * as the floating point representation may be unexpected.
 *
 * @param value - The `number` value to create a `BigDecimal` from.
 *
 * @example
 * import { fromNumber, make } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(fromNumber(123), make(123n, 0))
 * assert.deepStrictEqual(fromNumber(123.456), make(123456n, 3))
 *
 * @since 2.0.0
 * @category constructors
 */
export declare const fromNumber: (n: number) => BigDecimal;
/**
 * Parses a numerical `string` into a `BigDecimal`.
 *
 * @param s - The `string` to parse.
 *
 * @example
 * import { fromString, make } from "effect/BigDecimal"
 * import { some, none } from "effect/Option"
 *
 * assert.deepStrictEqual(fromString("123"), some(make(123n, 0)))
 * assert.deepStrictEqual(fromString("123.456"), some(make(123456n, 3)))
 * assert.deepStrictEqual(fromString("123.abc"), none())
 *
 * @since 2.0.0
 * @category constructors
 */
export declare const fromString: (s: string) => Option.Option<BigDecimal>;
/**
 * Parses a numerical `string` into a `BigDecimal`.
 *
 * @param s - The `string` to parse.
 *
 * @example
 * import { unsafeFromString, make } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(unsafeFromString("123"), make(123n, 0))
 * assert.deepStrictEqual(unsafeFromString("123.456"), make(123456n, 3))
 * assert.throws(() => unsafeFromString("123.abc"))
 *
 * @since 2.0.0
 * @category constructors
 */
export declare const unsafeFromString: (s: string) => BigDecimal;
/**
 * Formats a given `BigDecimal` as a `string`.
 *
 * @param normalized - The `BigDecimal` to format.
 *
 * @example
 * import { format, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(format(unsafeFromString("-5")), "-5")
 * assert.deepStrictEqual(format(unsafeFromString("123.456")), "123.456")
 * assert.deepStrictEqual(format(unsafeFromString("-0.00000123")), "-0.00000123")
 *
 * @since 2.0.0
 * @category conversions
 */
export declare const format: (n: BigDecimal) => string;
/**
 * Converts a `BigDecimal` to a `number`.
 *
 * This function will produce incorrect results if the `BigDecimal` exceeds the 64-bit range of a `number`.
 *
 * @param n - The `BigDecimal` to convert.
 *
 * @example
 * import { unsafeToNumber, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(unsafeToNumber(unsafeFromString("123.456")), 123.456)
 *
 * @since 2.0.0
 * @category conversions
 */
export declare const unsafeToNumber: (n: BigDecimal) => number;
/**
 * Checks if a given `BigDecimal` is an integer.
 *
 * @param n - The `BigDecimal` to check.
 *
 * @example
 * import { isInteger, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(isInteger(unsafeFromString("0")), true)
 * assert.deepStrictEqual(isInteger(unsafeFromString("1")), true)
 * assert.deepStrictEqual(isInteger(unsafeFromString("1.1")), false)
 *
 * @since 2.0.0
 * @category predicates
 */
export declare const isInteger: (n: BigDecimal) => boolean;
/**
 * Checks if a given `BigDecimal` is `0`.
 *
 * @param n - The `BigDecimal` to check.
 *
 * @example
 * import { isZero, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(isZero(unsafeFromString("0")), true)
 * assert.deepStrictEqual(isZero(unsafeFromString("1")), false)
 *
 * @since 2.0.0
 * @category predicates
 */
export declare const isZero: (n: BigDecimal) => boolean;
/**
 * Checks if a given `BigDecimal` is negative.
 *
 * @param n - The `BigDecimal` to check.
 *
 * @example
 * import { isNegative, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(isNegative(unsafeFromString("-1")), true)
 * assert.deepStrictEqual(isNegative(unsafeFromString("0")), false)
 * assert.deepStrictEqual(isNegative(unsafeFromString("1")), false)
 *
 * @since 2.0.0
 * @category predicates
 */
export declare const isNegative: (n: BigDecimal) => boolean;
/**
 * Checks if a given `BigDecimal` is positive.
 *
 * @param n - The `BigDecimal` to check.
 *
 * @example
 * import { isPositive, unsafeFromString } from "effect/BigDecimal"
 *
 * assert.deepStrictEqual(isPositive(unsafeFromString("-1")), false)
 * assert.deepStrictEqual(isPositive(unsafeFromString("0")), false)
 * assert.deepStrictEqual(isPositive(unsafeFromString("1")), true)
 *
 * @since 2.0.0
 * @category predicates
 */
export declare const isPositive: (n: BigDecimal) => boolean;
//# sourceMappingURL=BigDecimal.d.ts.map