"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.zero = exports.weeks = exports.unsafeToNanos = exports.toSeconds = exports.toNanos = exports.toMillis = exports.toHrTime = exports.times = exports.sum = exports.seconds = exports.nanos = exports.minutes = exports.min = exports.millis = exports.micros = exports.max = exports.matchWith = exports.match = exports.lessThanOrEqualTo = exports.lessThan = exports.isDuration = exports.infinity = exports.hours = exports.greaterThanOrEqualTo = exports.greaterThan = exports.format = exports.equals = exports.decode = exports.days = exports.clamp = exports.between = exports.Order = exports.Equivalence = void 0;
var Equal = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("./Equal.js"));
var _Function = /*#__PURE__*/require("./Function.js");
var Hash = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("./Hash.js"));
var _Inspectable = /*#__PURE__*/require("./Inspectable.js");
var Option = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("./Option.js"));
var order = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("./Order.js"));
var _Pipeable = /*#__PURE__*/require("./Pipeable.js");
var _Predicate = /*#__PURE__*/require("./Predicate.js");
function _getRequireWildcardCache(e) {
  if ("function" != typeof WeakMap) return null;
  var r = new WeakMap(),
    t = new WeakMap();
  return (_getRequireWildcardCache = function (e) {
    return e ? t : r;
  })(e);
}
function _interopRequireWildcard(e, r) {
  if (!r && e && e.__esModule) return e;
  if (null === e || "object" != typeof e && "function" != typeof e) return {
    default: e
  };
  var t = _getRequireWildcardCache(r);
  if (t && t.has(e)) return t.get(e);
  var n = {
      __proto__: null
    },
    a = Object.defineProperty && Object.getOwnPropertyDescriptor;
  for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) {
    var i = a ? Object.getOwnPropertyDescriptor(e, u) : null;
    i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u];
  }
  return n.default = e, t && t.set(e, n), n;
}
/**
 * @since 2.0.0
 */

const TypeId = /*#__PURE__*/Symbol.for("effect/Duration");
const bigint1e3 = /*#__PURE__*/BigInt(1000);
const bigint1e9 = /*#__PURE__*/BigInt(1000000000);
const DURATION_REGEX = /^(-?\d+(?:\.\d+)?)\s+(nanos|micros|millis|seconds|minutes|hours|days|weeks)$/;
/**
 * @since 2.0.0
 */
const decode = input => {
  if (isDuration(input)) {
    return input;
  } else if ((0, _Predicate.isNumber)(input)) {
    return millis(input);
  } else if ((0, _Predicate.isBigInt)(input)) {
    return nanos(input);
  } else if (Array.isArray(input)) {
    if (input.length === 2 && (0, _Predicate.isNumber)(input[0]) && (0, _Predicate.isNumber)(input[1])) {
      return nanos(BigInt(input[0]) * bigint1e9 + BigInt(input[1]));
    }
  } else {
    DURATION_REGEX.lastIndex = 0; // Reset the lastIndex before each use
    const match = DURATION_REGEX.exec(input);
    if (match) {
      const [_, valueStr, unit] = match;
      const value = Number(valueStr);
      switch (unit) {
        case "nanos":
          return nanos(BigInt(valueStr));
        case "micros":
          return micros(BigInt(valueStr));
        case "millis":
          return millis(value);
        case "seconds":
          return seconds(value);
        case "minutes":
          return minutes(value);
        case "hours":
          return hours(value);
        case "days":
          return days(value);
        case "weeks":
          return weeks(value);
      }
    }
  }
  throw new Error("Invalid duration input");
};
exports.decode = decode;
const zeroValue = {
  _tag: "Millis",
  millis: 0
};
const infinityValue = {
  _tag: "Infinity"
};
const DurationProto = {
  [TypeId]: TypeId,
  [Hash.symbol]() {
    return Hash.structure(this.value);
  },
  [Equal.symbol](that) {
    return isDuration(that) && equals(this, that);
  },
  toString() {
    return `Duration(${format(this)})`;
  },
  toJSON() {
    switch (this.value._tag) {
      case "Millis":
        return {
          _id: "Duration",
          _tag: "Millis",
          millis: this.value.millis
        };
      case "Nanos":
        return {
          _id: "Duration",
          _tag: "Nanos",
          hrtime: toHrTime(this)
        };
      case "Infinity":
        return {
          _id: "Duration",
          _tag: "Infinity"
        };
    }
  },
  [_Inspectable.NodeInspectSymbol]() {
    return this.toJSON();
  },
  pipe() {
    return (0, _Pipeable.pipeArguments)(this, arguments);
  }
};
const make = input => {
  const duration = Object.create(DurationProto);
  if ((0, _Predicate.isNumber)(input)) {
    if (isNaN(input) || input < 0) {
      duration.value = zeroValue;
    } else if (!Number.isFinite(input)) {
      duration.value = infinityValue;
    } else if (!Number.isInteger(input)) {
      duration.value = {
        _tag: "Nanos",
        nanos: BigInt(Math.round(input * 1000000))
      };
    } else {
      duration.value = {
        _tag: "Millis",
        millis: input
      };
    }
  } else if (input < BigInt(0)) {
    duration.value = zeroValue;
  } else {
    duration.value = {
      _tag: "Nanos",
      nanos: input
    };
  }
  return duration;
};
/**
 * @since 2.0.0
 * @category guards
 */
const isDuration = u => (0, _Predicate.hasProperty)(u, TypeId);
/**
 * @since 2.0.0
 * @category constructors
 */
exports.isDuration = isDuration;
const zero = exports.zero = /*#__PURE__*/make(0);
/**
 * @since 2.0.0
 * @category constructors
 */
const infinity = exports.infinity = /*#__PURE__*/make(Infinity);
/**
 * @since 2.0.0
 * @category constructors
 */
const nanos = nanos => make(nanos);
/**
 * @since 2.0.0
 * @category constructors
 */
exports.nanos = nanos;
const micros = micros => make(micros * bigint1e3);
/**
 * @since 2.0.0
 * @category constructors
 */
exports.micros = micros;
const millis = millis => make(millis);
/**
 * @since 2.0.0
 * @category constructors
 */
exports.millis = millis;
const seconds = seconds => make(seconds * 1000);
/**
 * @since 2.0.0
 * @category constructors
 */
exports.seconds = seconds;
const minutes = minutes => make(minutes * 60000);
/**
 * @since 2.0.0
 * @category constructors
 */
exports.minutes = minutes;
const hours = hours => make(hours * 3600000);
/**
 * @since 2.0.0
 * @category constructors
 */
exports.hours = hours;
const days = days => make(days * 86400000);
/**
 * @since 2.0.0
 * @category constructors
 */
exports.days = days;
const weeks = weeks => make(weeks * 604800000);
/**
 * @since 2.0.0
 * @category getters
 */
exports.weeks = weeks;
const toMillis = self => {
  const _self = decode(self);
  switch (_self.value._tag) {
    case "Infinity":
      return Infinity;
    case "Nanos":
      return Number(_self.value.nanos) / 1000000;
    case "Millis":
      return _self.value.millis;
  }
};
/**
 * @since 2.0.0
 * @category getters
 */
exports.toMillis = toMillis;
const toSeconds = self => toMillis(self) / 1000;
/**
 * Get the duration in nanoseconds as a bigint.
 *
 * If the duration is infinite, returns `Option.none()`
 *
 * @since 2.0.0
 * @category getters
 */
exports.toSeconds = toSeconds;
const toNanos = self => {
  const _self = decode(self);
  switch (_self.value._tag) {
    case "Infinity":
      return Option.none();
    case "Nanos":
      return Option.some(_self.value.nanos);
    case "Millis":
      return Option.some(BigInt(Math.round(_self.value.millis * 1000000)));
  }
};
/**
 * Get the duration in nanoseconds as a bigint.
 *
 * If the duration is infinite, it throws an error.
 *
 * @since 2.0.0
 * @category getters
 */
exports.toNanos = toNanos;
const unsafeToNanos = self => {
  const _self = decode(self);
  switch (_self.value._tag) {
    case "Infinity":
      throw new Error("Cannot convert infinite duration to nanos");
    case "Nanos":
      return _self.value.nanos;
    case "Millis":
      return BigInt(Math.round(_self.value.millis * 1000000));
  }
};
/**
 * @since 2.0.0
 * @category getters
 */
exports.unsafeToNanos = unsafeToNanos;
const toHrTime = self => {
  const _self = decode(self);
  switch (_self.value._tag) {
    case "Infinity":
      return [Infinity, 0];
    case "Nanos":
      return [Number(_self.value.nanos / bigint1e9), Number(_self.value.nanos % bigint1e9)];
    case "Millis":
      return [Math.floor(_self.value.millis / 1000), Math.round(_self.value.millis % 1000 * 1000000)];
  }
};
/**
 * @since 2.0.0
 * @category pattern matching
 */
exports.toHrTime = toHrTime;
const match = exports.match = /*#__PURE__*/(0, _Function.dual)(2, (self, options) => {
  const _self = decode(self);
  switch (_self.value._tag) {
    case "Nanos":
      return options.onNanos(_self.value.nanos);
    case "Infinity":
      return options.onMillis(Infinity);
    case "Millis":
      return options.onMillis(_self.value.millis);
  }
});
/**
 * @since 2.0.0
 * @category pattern matching
 */
const matchWith = exports.matchWith = /*#__PURE__*/(0, _Function.dual)(3, (self, that, options) => {
  const _self = decode(self);
  const _that = decode(that);
  if (_self.value._tag === "Infinity" || _that.value._tag === "Infinity") {
    return options.onMillis(toMillis(_self), toMillis(_that));
  } else if (_self.value._tag === "Nanos" || _that.value._tag === "Nanos") {
    const selfNanos = _self.value._tag === "Nanos" ? _self.value.nanos : BigInt(Math.round(_self.value.millis * 1000000));
    const thatNanos = _that.value._tag === "Nanos" ? _that.value.nanos : BigInt(Math.round(_that.value.millis * 1000000));
    return options.onNanos(selfNanos, thatNanos);
  }
  return options.onMillis(_self.value.millis, _that.value.millis);
});
/**
 * @category instances
 * @since 2.0.0
 */
const Order = exports.Order = /*#__PURE__*/order.make((self, that) => matchWith(self, that, {
  onMillis: (self, that) => self < that ? -1 : self > that ? 1 : 0,
  onNanos: (self, that) => self < that ? -1 : self > that ? 1 : 0
}));
/**
 * Checks if a `Duration` is between a `minimum` and `maximum` value.
 *
 * @category predicates
 * @since 2.0.0
 */
const between = exports.between = /*#__PURE__*/order.between( /*#__PURE__*/order.mapInput(Order, decode));
/**
 * @category instances
 * @since 2.0.0
 */
const Equivalence = (self, that) => matchWith(self, that, {
  onMillis: (self, that) => self === that,
  onNanos: (self, that) => self === that
});
exports.Equivalence = Equivalence;
const _min = /*#__PURE__*/order.min(Order);
/**
 * @since 2.0.0
 */
const min = exports.min = /*#__PURE__*/(0, _Function.dual)(2, (self, that) => _min(decode(self), decode(that)));
const _max = /*#__PURE__*/order.max(Order);
/**
 * @since 2.0.0
 */
const max = exports.max = /*#__PURE__*/(0, _Function.dual)(2, (self, that) => _max(decode(self), decode(that)));
const _clamp = /*#__PURE__*/order.clamp(Order);
/**
 * @since 2.0.0
 */
const clamp = exports.clamp = /*#__PURE__*/(0, _Function.dual)(2, (self, options) => _clamp(decode(self), {
  minimum: decode(options.minimum),
  maximum: decode(options.maximum)
}));
/**
 * @since 2.0.0
 * @category math
 */
const times = exports.times = /*#__PURE__*/(0, _Function.dual)(2, (self, times) => match(self, {
  onMillis: millis => make(millis * times),
  onNanos: nanos => make(nanos * BigInt(times))
}));
/**
 * @since 2.0.0
 * @category math
 */
const sum = exports.sum = /*#__PURE__*/(0, _Function.dual)(2, (self, that) => matchWith(self, that, {
  onMillis: (self, that) => make(self + that),
  onNanos: (self, that) => make(self + that)
}));
/**
 * @since 2.0.0
 * @category predicates
 */
const lessThan = exports.lessThan = /*#__PURE__*/(0, _Function.dual)(2, (self, that) => matchWith(self, that, {
  onMillis: (self, that) => self < that,
  onNanos: (self, that) => self < that
}));
/**
 * @since 2.0.0
 * @category predicates
 */
const lessThanOrEqualTo = exports.lessThanOrEqualTo = /*#__PURE__*/(0, _Function.dual)(2, (self, that) => matchWith(self, that, {
  onMillis: (self, that) => self <= that,
  onNanos: (self, that) => self <= that
}));
/**
 * @since 2.0.0
 * @category predicates
 */
const greaterThan = exports.greaterThan = /*#__PURE__*/(0, _Function.dual)(2, (self, that) => matchWith(self, that, {
  onMillis: (self, that) => self > that,
  onNanos: (self, that) => self > that
}));
/**
 * @since 2.0.0
 * @category predicates
 */
const greaterThanOrEqualTo = exports.greaterThanOrEqualTo = /*#__PURE__*/(0, _Function.dual)(2, (self, that) => matchWith(self, that, {
  onMillis: (self, that) => self >= that,
  onNanos: (self, that) => self >= that
}));
/**
 * @since 2.0.0
 * @category predicates
 */
const equals = exports.equals = /*#__PURE__*/(0, _Function.dual)(2, (self, that) => Equivalence(decode(self), decode(that)));
/**
 * Converts a `Duration` to a human readable string.
 * @since 2.0.0
 *
 * @example
 * import * as Duration from "effect/Duration"
 *
 * Duration.format(Duration.millis(1000)) // "1s"
 * Duration.format(Duration.millis(1001)) // "1s 1ms"
 */
const format = self => {
  const duration = decode(self);
  const parts = [];
  if (duration.value._tag === "Infinity") {
    return "Infinity";
  }
  const nanos = unsafeToNanos(duration);
  if (nanos % 1000000n) {
    parts.push(`${nanos % 1000000n}ns`);
  }
  const ms = nanos / 1000000n;
  if (ms % 1000n !== 0n) {
    parts.push(`${ms % 1000n}ms`);
  }
  const sec = ms / 1000n;
  if (sec % 60n !== 0n) {
    parts.push(`${sec % 60n}s`);
  }
  const min = sec / 60n;
  if (min % 60n !== 0n) {
    parts.push(`${min % 60n}m`);
  }
  const hr = min / 60n;
  if (hr % 24n !== 0n) {
    parts.push(`${hr % 24n}h`);
  }
  const days = hr / 24n;
  if (days !== 0n) {
    parts.push(`${days}d`);
  }
  return parts.reverse().join(" ");
};
exports.format = format;
//# sourceMappingURL=Duration.js.map