/**
 * @since 1.0.0
 */
import * as Effect from "effect/Effect";
import * as Either from "effect/Either";
import * as Inspectable from "effect/Inspectable";
import * as Option from "effect/Option";
import * as TreeFormatter from "./TreeFormatter.js";
class ParseErrorImpl {
  errors;
  _tag = "ParseError";
  constructor(errors) {
    this.errors = errors;
  }
  toString() {
    return TreeFormatter.formatErrors(this.errors);
  }
  toJSON() {
    return {
      _id: "ParseError",
      message: this.toString()
    };
  }
  [Inspectable.NodeInspectSymbol]() {
    return this.toJSON();
  }
}
/**
 * @since 1.0.0
 */
export const parseError = errors => new ParseErrorImpl(errors);
/**
 * @category constructors
 * @since 1.0.0
 */
export const type = (expected, actual, message) => ({
  _tag: "Type",
  expected,
  actual,
  message: Option.fromNullable(message)
});
/**
 * @category constructors
 * @since 1.0.0
 */
export const forbidden = {
  _tag: "Forbidden"
};
/**
 * @category constructors
 * @since 1.0.0
 */
export const index = (index, errors) => ({
  _tag: "Index",
  index,
  errors
});
/**
 * @category constructors
 * @since 1.0.0
 */
export const key = (key, errors) => ({
  _tag: "Key",
  key,
  errors
});
/**
 * @category constructors
 * @since 1.0.0
 */
export const missing = {
  _tag: "Missing"
};
/**
 * @category constructors
 * @since 1.0.0
 */
export const unexpected = ast => ({
  _tag: "Unexpected",
  ast
});
/**
 * @category constructors
 * @since 1.0.0
 */
export const unionMember = errors => ({
  _tag: "UnionMember",
  errors
});
/**
 * @category constructors
 * @since 1.0.0
 */
export const succeed = Either.right;
const _try = Either.try;
export {
/**
 * @category constructors
 * @since 1.0.0
 */
_try as try };
/**
 * @category constructors
 * @since 1.0.0
 */
export const fail = error => {
  const e = error;
  if ("_tag" in e) {
    return e._tag === "ParseError" ? Either.left(e) : Either.left(parseError([e]));
  }
  return Either.left(parseError(e));
};
/**
 * @category optimisation
 * @since 1.0.0
 */
export const eitherOrUndefined = self => {
  const s = self;
  if (s["_tag"] === "Left" || s["_tag"] === "Right") {
    return s;
  }
};
/**
 * @category optimisation
 * @since 1.0.0
 */
export const flatMap = (self, f) => {
  const s = self;
  if (s["_tag"] === "Left") {
    return s;
  }
  if (s["_tag"] === "Right") {
    return f(s.right);
  }
  return Effect.flatMap(self, f);
};
/**
 * @category optimisation
 * @since 1.0.0
 */
export const map = (self, f) => {
  const s = self;
  if (s["_tag"] === "Left") {
    return s;
  }
  if (s["_tag"] === "Right") {
    return Either.right(f(s.right));
  }
  return Effect.map(self, f);
};
/**
 * @category optimisation
 * @since 1.0.0
 */
export const mapLeft = (self, f) => {
  const s = self;
  if (s["_tag"] === "Left") {
    return Either.left(f(s.left));
  }
  if (s["_tag"] === "Right") {
    return s;
  }
  return Effect.mapError(self, f);
};
/**
 * @category optimisation
 * @since 1.0.0
 */
export const bimap = (self, f, g) => {
  const s = self;
  if (s["_tag"] === "Left") {
    return Either.left(f(s.left));
  }
  if (s["_tag"] === "Right") {
    return Either.right(g(s.right));
  }
  return Effect.mapBoth(self, {
    onFailure: f,
    onSuccess: g
  });
};
/**
 * @category optimisation
 * @since 1.0.0
 */
export const orElse = (self, f) => {
  const s = self;
  if (s["_tag"] === "Left") {
    return f(s.left);
  }
  if (s["_tag"] === "Right") {
    return s;
  }
  return Effect.catchAll(self, f);
};
//# sourceMappingURL=ParseResult.js.map