/**
 * @since 1.0.0
 */
import * as BigDecimal from "effect/BigDecimal";
import * as Brand from "effect/Brand";
import * as Cause from "effect/Cause";
import * as Chunk from "effect/Chunk";
import * as Data from "effect/Data";
import * as Duration from "effect/Duration";
import type * as Effect from "effect/Effect";
import * as Either from "effect/Either";
import * as Equivalence from "effect/Equivalence";
import * as Exit from "effect/Exit";
import * as FiberId from "effect/FiberId";
import * as Option from "effect/Option";
import type { Pipeable } from "effect/Pipeable";
import * as Predicate from "effect/Predicate";
import * as ReadonlyArray from "effect/ReadonlyArray";
import * as Request from "effect/Request";
import * as Secret from "effect/Secret";
import type { Equals, Mutable, Simplify } from "effect/Types";
import type { Arbitrary } from "./Arbitrary.js";
import type { ParseOptions } from "./AST.js";
import * as AST from "./AST.js";
import * as ParseResult from "./ParseResult.js";
import * as Pretty from "./Pretty.js";
import type * as Serializable from "./Serializable.js";
/**
 * @since 1.0.0
 * @category symbol
 */
export declare const TypeId: unique symbol;
/**
 * @since 1.0.0
 * @category symbol
 */
export type TypeId = typeof TypeId;
/**
 * @category model
 * @since 1.0.0
 */
export interface Schema<From, To = From> extends Schema.Variance<From, To>, Pipeable {
    readonly ast: AST.AST;
}
/**
 * @since 1.0.0
 */
export declare namespace Schema {
    /**
     * @since 1.0.0
     */
    interface Variance<From, To> {
        readonly [TypeId]: {
            readonly From: (_: From) => From;
            readonly To: (_: To) => To;
        };
    }
    /**
     * @since 1.0.0
     */
    type From<S extends {
        readonly [TypeId]: {
            readonly From: (..._: any) => any;
        };
    }> = Parameters<S[TypeId]["From"]>[0];
    /**
     * @since 1.0.0
     */
    type To<S extends {
        readonly [TypeId]: {
            readonly To: (..._: any) => any;
        };
    }> = Parameters<S[TypeId]["To"]>[0];
    /**
     * @since 1.0.0
     */
    type ToAsserts<S extends Schema<any>> = (input: unknown, options?: AST.ParseOptions) => asserts input is Schema.To<S>;
}
/**
 * @since 1.0.0
 */
export declare const from: <I, A>(schema: Schema<I, A>) => Schema<I, I>;
/**
 * @since 1.0.0
 */
export declare const to: <I, A>(schema: Schema<I, A>) => Schema<A, A>;
export { 
/**
 * @category validation
 * @since 1.0.0
 */
asserts, 
/**
 * @category decoding
 * @since 1.0.0
 */
decode, 
/**
 * @category decoding
 * @since 1.0.0
 */
decodeEither, 
/**
 * @category decoding
 * @since 1.0.0
 */
decodeOption, 
/**
 * @category decoding
 * @since 1.0.0
 */
decodePromise, 
/**
 * @category decoding
 * @since 1.0.0
 */
decodeSync, 
/**
 * @category encoding
 * @since 1.0.0
 */
encode, 
/**
 * @category encoding
 * @since 1.0.0
 */
encodeEither, 
/**
 * @category encoding
 * @since 1.0.0
 */
encodeOption, 
/**
 * @category encoding
 * @since 1.0.0
 */
encodePromise, 
/**
 * @category encoding
 * @since 1.0.0
 */
encodeSync, 
/**
 * @category validation
 * @since 1.0.0
 */
is, 
/**
 * @category parsing
 * @since 1.0.0
 */
parse, 
/**
 * @category parsing
 * @since 1.0.0
 */
parseEither, 
/**
 * @category parsing
 * @since 1.0.0
 */
parseOption, 
/**
 * @category parsing
 * @since 1.0.0
 */
parsePromise, 
/**
 * @category parsing
 * @since 1.0.0
 */
parseSync, 
/**
 * @category validation
 * @since 1.0.0
 */
validate, 
/**
 * @category validation
 * @since 1.0.0
 */
validateEither, 
/**
 * @category validation
 * @since 1.0.0
 */
validateOption, 
/**
 * @category validation
 * @since 1.0.0
 */
validatePromise, 
/**
 * @category validation
 * @since 1.0.0
 */
validateSync } from "./Parser.js";
/**
 * Tests if a value is a `Schema`.
 *
 * @category guards
 * @since 1.0.0
 */
export declare const isSchema: (u: unknown) => u is Schema<unknown, unknown>;
/**
 * @category constructors
 * @since 1.0.0
 */
export declare const make: <I, A>(ast: AST.AST) => Schema<I, A>;
/**
 * @category constructors
 * @since 1.0.0
 */
export declare const literal: <Literals extends readonly AST.LiteralValue[]>(...literals: Literals) => Schema<Literals[number], Literals[number]>;
/**
 * @category constructors
 * @since 1.0.0
 */
export declare const uniqueSymbol: <S extends symbol>(symbol: S, annotations?: AST.Annotations) => Schema<S, S>;
/**
 * @category constructors
 * @since 1.0.0
 */
export declare const enums: <A extends {
    [x: string]: string | number;
}>(enums: A) => Schema<A[keyof A], A[keyof A]>;
/**
 * @since 1.0.0
 */
export type Join<T> = T extends [infer Head, ...infer Tail] ? `${Head & (string | number | bigint | boolean | null | undefined)}${Tail extends [] ? "" : Join<Tail>}` : never;
/**
 * @category constructors
 * @since 1.0.0
 */
export declare const templateLiteral: <T extends [Schema<any, any>, ...Schema<any, any>[]]>(...[head, ...tail]: T) => Schema<Join<{ [K in keyof T]: Schema.To<T[K]>; }>, Join<{ [K in keyof T]: Schema.To<T[K]>; }>>;
/**
  @category constructors
  @since 1.0.0
*/
export declare const declare: (typeParameters: ReadonlyArray<Schema<any>>, type: Schema<any>, decode: (isDecoding: boolean, ...typeParameters: ReadonlyArray<Schema<any>>) => (input: any, options: ParseOptions, ast: AST.AST) => ParseResult.ParseResult<any>, annotations?: AST.Annotations) => Schema<any>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const BrandTypeId: unique symbol;
/**
 * @category constructors
 * @since 1.0.0
 */
export declare const fromBrand: <C extends Brand.Brand<string | symbol>>(constructor: Brand.Brand.Constructor<C>, options?: FilterAnnotations<Brand.Brand.Unbranded<C>> | undefined) => <I, A extends Brand.Brand.Unbranded<C>>(self: Schema<I, A>) => Schema<I, A & C>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const InstanceOfTypeId: unique symbol;
/**
 * @category constructors
 * @since 1.0.0
 */
export declare const instanceOf: <A extends abstract new (...args: any) => any>(constructor: A, options?: FilterAnnotations<InstanceType<A>> | undefined) => Schema<InstanceType<A>, InstanceType<A>>;
declare const _undefined: Schema<undefined>;
declare const _void: Schema<void>;
declare const _null: Schema<null>;
export { 
/**
 * @category primitives
 * @since 1.0.0
 */
_null as null, 
/**
 * @category primitives
 * @since 1.0.0
 */
_undefined as undefined, 
/**
 * @category primitives
 * @since 1.0.0
 */
_void as void };
/**
 * @category primitives
 * @since 1.0.0
 */
export declare const never: Schema<never>;
/**
 * @category primitives
 * @since 1.0.0
 */
export declare const unknown: Schema<unknown>;
/**
 * @category primitives
 * @since 1.0.0
 */
export declare const any: Schema<any>;
/**
 * @category primitives
 * @since 1.0.0
 */
export declare const string: Schema<string>;
/**
 * @category primitives
 * @since 1.0.0
 */
export declare const number: Schema<number>;
/**
 * @category primitives
 * @since 1.0.0
 */
export declare const boolean: Schema<boolean>;
/**
 * @category primitives
 * @since 1.0.0
 */
export declare const bigintFromSelf: Schema<bigint>;
/**
 * @category primitives
 * @since 1.0.0
 */
export declare const symbolFromSelf: Schema<symbol>;
/**
 * @category primitives
 * @since 1.0.0
 */
export declare const object: Schema<object>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const union: <Members extends readonly Schema<any, any>[]>(...members: Members) => Schema<Schema.From<Members[number]>, Schema.To<Members[number]>>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const nullable: <From, To>(self: Schema<From, To>) => Schema<From | null, To | null>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const orUndefined: <From, To>(self: Schema<From, To>) => Schema<From | undefined, To | undefined>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const nullish: <From, To>(self: Schema<From, To>) => Schema<From | null | undefined, To | null | undefined>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const keyof: <I, A>(schema: Schema<I, A>) => Schema<keyof A, keyof A>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const tuple: <Elements extends readonly Schema<any, any>[]>(...elements: Elements) => Schema<{ readonly [K in keyof Elements]: Schema.From<Elements[K]>; }, { readonly [K_1 in keyof Elements]: Schema.To<Elements[K_1]>; }>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const rest: <IR, R>(rest: Schema<IR, R>) => <I extends readonly any[], A extends readonly any[]>(self: Schema<I, A>) => Schema<readonly [...I, ...IR[]], readonly [...A, ...R[]]>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const element: <IE, E>(element: Schema<IE, E>) => <I extends readonly any[], A extends readonly any[]>(self: Schema<I, A>) => Schema<readonly [...I, IE], readonly [...A, E]>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const optionalElement: <IE, E>(element: Schema<IE, E>) => <I extends readonly any[], A extends readonly any[]>(self: Schema<I, A>) => Schema<readonly [...I, IE?], readonly [...A, E?]>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const array: <I, A>(item: Schema<I, A>) => Schema<readonly I[], readonly A[]>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const nonEmptyArray: <I, A>(item: Schema<I, A>) => Schema<readonly [I, ...I[]], readonly [A, ...A[]]>;
/**
 * @since 1.0.0
 */
export interface PropertySignature<From, FromIsOptional, To, ToIsOptional> extends Schema.Variance<From, To>, Pipeable {
    readonly FromIsOptional: FromIsOptional;
    readonly ToIsOptional: ToIsOptional;
}
/**
 * @since 1.0.0
 */
export declare const propertySignatureAnnotations: (annotations: DocAnnotations) => <S extends Schema<any, any> | Schema<never, never> | PropertySignature<any, boolean, any, boolean> | PropertySignature<never, boolean, never, boolean>>(self: S) => S extends Schema<infer I, infer A> ? PropertySignature<I, false, A, false> : S;
/**
 * @category optional
 * @since 1.0.0
 */
export declare const optionalToRequired: <I, A, B>(from: Schema<I, A>, to: Schema<B, B>, decode: (o: Option.Option<A>) => B, encode: (b: B) => Option.Option<A>, options?: DocAnnotations) => PropertySignature<I, true, B, false>;
/**
 * @since 1.0.0
 */
export declare const optional: {
    <I, A>(schema: Schema<I, A>, options: {
        readonly exact: true;
        readonly default: () => A;
        readonly nullable: true;
    }): PropertySignature<I | null, true, A, false>;
    <I, A>(schema: Schema<I, A>, options: {
        readonly exact: true;
        readonly default: () => A;
    }): PropertySignature<I, true, A, false>;
    <I, A>(schema: Schema<I, A>, options: {
        readonly exact: true;
        readonly nullable: true;
        readonly as: "Option";
    }): PropertySignature<I | null, true, Option.Option<A>, false>;
    <I, A>(schema: Schema<I, A>, options: {
        readonly exact: true;
        readonly as: "Option";
    }): PropertySignature<I, true, Option.Option<A>, false>;
    <I, A>(schema: Schema<I, A>, options: {
        readonly exact: true;
    }): PropertySignature<I, true, A, true>;
    <I, A>(schema: Schema<I, A>, options: {
        readonly default: () => A;
        readonly nullable: true;
    }): PropertySignature<I | null | undefined, true, A, false>;
    <I, A>(schema: Schema<I, A>, options: {
        readonly default: () => A;
    }): PropertySignature<I | undefined, true, A, false>;
    <I, A>(schema: Schema<I, A>, options: {
        readonly nullable: true;
        readonly as: "Option";
    }): PropertySignature<I | undefined | null, true, Option.Option<A>, false>;
    <I, A>(schema: Schema<I, A>, options: {
        readonly as: "Option";
    }): PropertySignature<I | undefined, true, Option.Option<A>, false>;
    <I, A>(schema: Schema<I, A>): PropertySignature<I | undefined, true, A | undefined, true>;
};
/**
 * @since 1.0.0
 */
export type FromOptionalKeys<Fields> = {
    [K in keyof Fields]: Fields[K] extends PropertySignature<any, true, any, boolean> | PropertySignature<never, true, never, boolean> ? K : never;
}[keyof Fields];
/**
 * @since 1.0.0
 */
export type ToOptionalKeys<Fields> = {
    [K in keyof Fields]: Fields[K] extends PropertySignature<any, boolean, any, true> | PropertySignature<never, boolean, never, true> ? K : never;
}[keyof Fields];
/**
 * @since 1.0.0
 */
export type StructFields = Record<PropertyKey, Schema<any, any> | Schema<never, never> | PropertySignature<any, boolean, any, boolean> | PropertySignature<never, boolean, never, boolean>>;
/**
 * @since 1.0.0
 */
export type FromStruct<Fields extends StructFields> = {
    readonly [K in Exclude<keyof Fields, FromOptionalKeys<Fields>>]: Schema.From<Fields[K]>;
} & {
    readonly [K in FromOptionalKeys<Fields>]?: Schema.From<Fields[K]>;
};
/**
 * @since 1.0.0
 */
export type ToStruct<Fields extends StructFields> = {
    readonly [K in Exclude<keyof Fields, ToOptionalKeys<Fields>>]: Schema.To<Fields[K]>;
} & {
    readonly [K in ToOptionalKeys<Fields>]?: Schema.To<Fields[K]>;
};
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const struct: <Fields extends StructFields>(fields: Fields) => Schema<Simplify<FromStruct<Fields>>, Simplify<ToStruct<Fields>>>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const pick: <A, Keys extends readonly (keyof A)[]>(...keys: Keys) => <I extends { [K in keyof A]?: any; }>(self: Schema<I, A>) => Schema<Simplify<Pick<I, Keys[number]>>, Simplify<Pick<A, Keys[number]>>>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const omit: <A, Keys extends readonly (keyof A)[]>(...keys: Keys) => <I extends { [K in keyof A]?: any; }>(self: Schema<I, A>) => Schema<Simplify<Omit<I, Keys[number]>>, Simplify<Omit<A, Keys[number]>>>;
/**
 * @category model
 * @since 1.0.0
 */
export interface BrandSchema<From, To extends Brand.Brand<any>> extends Schema<From, To>, Brand.Brand.Constructor<To> {
}
/**
 * Returns a nominal branded schema by applying a brand to a given schema.
 *
 * ```
 * Schema<A> + B -> Schema<A & Brand<B>>
 * ```
 *
 * @param self - The input schema to be combined with the brand.
 * @param brand - The brand to apply.
 *
 * @example
 * import * as Schema from "@effect/schema/Schema"
 *
 * const Int = Schema.number.pipe(Schema.int(), Schema.brand("Int"))
 * type Int = Schema.Schema.To<typeof Int> // number & Brand<"Int">
 *
 * @category combinators
 * @since 1.0.0
 */
export declare const brand: <B extends string | symbol, A>(brand: B, options?: DocAnnotations) => <I>(self: Schema<I, A>) => BrandSchema<I, A & Brand.Brand<B>>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const partial: <I, A>(self: Schema<I, A>) => Schema<Simplify<Partial<I>>, Simplify<Partial<A>>>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const required: <I, A>(self: Schema<I, A>) => Schema<Simplify<Required<I>>, Simplify<Required<A>>>;
/**
 * Creates a new schema with shallow mutability applied to its properties.
 *
 * @param schema - The original schema to make properties mutable (shallowly).
 *
 * @category combinators
 * @since 1.0.0
 */
export declare const mutable: <I, A>(schema: Schema<I, A>) => Schema<Simplify<Mutable<I>>, Simplify<Mutable<A>>>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const record: <IK extends string | symbol, AK extends IK, IV, AV>(key: Schema<IK, AK>, value: Schema<IV, AV>) => Schema<{ readonly [k in IK]: IV; }, { readonly [k_1 in AK]: AV; }>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const extend: {
    <IB, B>(that: Schema<IB, B>): <I, A>(self: Schema<I, A>) => Schema<Simplify<I & IB>, Simplify<A & B>>;
    <I, A, IB, B>(self: Schema<I, A>, that: Schema<IB, B>): Schema<Simplify<I & IB>, Simplify<A & B>>;
};
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const compose: {
    <C, D>(bc: Schema<C, D>): <A, B>(ab: Schema<A, B>) => Schema<A, D>;
    <A, B, C, D>(ab: Schema<A, B>, cd: Schema<C, D>): Schema<A, D>;
};
/**
 * @category combinators
 * @since 1.0.0
 */
export declare const suspend: <I, A = I>(f: () => Schema<I, A>, annotations?: AST.Annotations) => Schema<I, A>;
/**
 * @category combinators
 * @since 1.0.0
 */
export declare function filter<A>(f: (a: A, options: ParseOptions, self: AST.AST) => Option.Option<ParseResult.ParseError>, options?: FilterAnnotations<A>): <I>(self: Schema<I, A>) => Schema<I, A>;
export declare function filter<C extends A, B extends A, A = C>(refinement: Predicate.Refinement<A, B>, options?: FilterAnnotations<A>): <I>(self: Schema<I, C>) => Schema<I, C & B>;
export declare function filter<B extends A, A = B>(predicate: Predicate.Predicate<A>, options?: FilterAnnotations<A>): <I>(self: Schema<I, B>) => Schema<I, B>;
/**
 * Create a new `Schema` by transforming the input and output of an existing `Schema`
 * using the provided decoding functions.
 *
 * @category combinators
 * @since 1.0.0
 */
export declare const transformOrFail: {
    <C, D, B>(to: Schema<C, D>, decode: (b: B, options: ParseOptions, ast: AST.AST) => ParseResult.ParseResult<C>, encode: (c: C, options: ParseOptions, ast: AST.AST) => ParseResult.ParseResult<B>): <A>(self: Schema<A, B>) => Schema<A, D>;
    <C, D, B>(to: Schema<C, D>, decode: (b: B, options: ParseOptions, ast: AST.AST) => ParseResult.ParseResult<unknown>, encode: (c: C, options: ParseOptions, ast: AST.AST) => ParseResult.ParseResult<unknown>, options: {
        strict: false;
    }): <A>(self: Schema<A, B>) => Schema<A, D>;
    <A, B, C, D>(from: Schema<A, B>, to: Schema<C, D>, decode: (b: B, options: ParseOptions, ast: AST.AST) => ParseResult.ParseResult<C>, encode: (c: C, options: ParseOptions, ast: AST.AST) => ParseResult.ParseResult<B>): Schema<A, D>;
    <A, B, C, D>(from: Schema<A, B>, to: Schema<C, D>, decode: (b: B, options: ParseOptions, ast: AST.AST) => ParseResult.ParseResult<unknown>, encode: (c: C, options: ParseOptions, ast: AST.AST) => ParseResult.ParseResult<unknown>, options: {
        strict: false;
    }): Schema<A, D>;
};
/**
 * Create a new `Schema` by transforming the input and output of an existing `Schema`
 * using the provided mapping functions.
 *
 * @category combinators
 * @since 1.0.0
 */
export declare const transform: {
    <C, D, B>(to: Schema<C, D>, decode: (b: B, options: ParseOptions, ast: AST.AST) => C, encode: (c: C, options: ParseOptions, ast: AST.AST) => B): <A>(self: Schema<A, B>) => Schema<A, D>;
    <C, D, B>(to: Schema<C, D>, decode: (b: B, options: ParseOptions, ast: AST.AST) => unknown, encode: (c: C, options: ParseOptions, ast: AST.AST) => unknown, options: {
        strict: false;
    }): <A>(self: Schema<A, B>) => Schema<A, D>;
    <A, B, C, D>(from: Schema<A, B>, to: Schema<C, D>, decode: (b: B, options: ParseOptions, ast: AST.AST) => C, encode: (c: C, options: ParseOptions, ast: AST.AST) => B): Schema<A, D>;
    <A, B, C, D>(from: Schema<A, B>, to: Schema<C, D>, decode: (b: B, options: ParseOptions, ast: AST.AST) => unknown, encode: (c: C, options: ParseOptions, ast: AST.AST) => unknown, options: {
        strict: false;
    }): Schema<A, D>;
};
/**
 * Creates a new `Schema` which transforms literal values.
 *
 * @example
 * import * as S from "@effect/schema/Schema"
 *
 * const schema = S.transformLiteral(0, "a")
 *
 * assert.deepStrictEqual(S.decodeSync(schema)(0), "a")
 *
 * @category constructors
 * @since 1.0.0
 */
export declare const transformLiteral: <From extends AST.LiteralValue, To extends AST.LiteralValue>(from: From, to: To) => Schema<From, To>;
/**
 * Creates a new `Schema` which maps between corresponding literal values.
 *
 * @example
 * import * as S from "@effect/schema/Schema"
 *
 * const Animal = S.transformLiterals(
 *   [0, "cat"],
 *   [1, "dog"],
 *   [2, "cow"]
 * )
 *
 * assert.deepStrictEqual(S.decodeSync(Animal)(1), "dog")
 *
 * @category constructors
 * @since 1.0.0
 */
export declare const transformLiterals: <const A extends readonly (readonly [from: AST.LiteralValue, to: AST.LiteralValue])[]>(...pairs: A) => Schema<A[number][0], A[number][1]>;
/**
 * Attaches a property signature with the specified key and value to the schema.
 * This API is useful when you want to add a property to your schema which doesn't describe the shape of the input,
 * but rather maps to another schema, for example when you want to add a discriminant to a simple union.
 *
 * @param self - The input schema.
 * @param key - The name of the property to add to the schema.
 * @param value - The value of the property to add to the schema.
 *
 * @example
 * import * as S from "@effect/schema/Schema"
 * import { pipe } from "effect/Function"
 *
 * const Circle = S.struct({ radius: S.number })
 * const Square = S.struct({ sideLength: S.number })
 * const Shape = S.union(
 *   Circle.pipe(S.attachPropertySignature("kind", "circle")),
 *   Square.pipe(S.attachPropertySignature("kind", "square"))
 * )
 *
 * assert.deepStrictEqual(S.decodeSync(Shape)({ radius: 10 }), {
 *   kind: "circle",
 *   radius: 10
 * })
 *
 * @category combinators
 * @since 1.0.0
 */
export declare const attachPropertySignature: {
    <K extends PropertyKey, V extends AST.LiteralValue | symbol>(key: K, value: V, options?: DocAnnotations): <I, A extends object>(schema: Schema<I, A>) => Schema<I, Simplify<A & {
        readonly [k in K]: V;
    }>>;
    <I, A, K extends PropertyKey, V extends AST.LiteralValue | symbol>(schema: Schema<I, A>, key: K, value: V, options?: DocAnnotations): Schema<I, Simplify<A & {
        readonly [k in K]: V;
    }>>;
};
/**
 * @since 1.0.0
 */
export interface DocAnnotations extends AST.Annotations {
    readonly identifier?: AST.IdentifierAnnotation;
    readonly title?: AST.TitleAnnotation;
    readonly description?: AST.DescriptionAnnotation;
    readonly examples?: AST.ExamplesAnnotation;
    readonly default?: AST.DefaultAnnotation;
    readonly documentation?: AST.DocumentationAnnotation;
}
/**
 * @since 1.0.0
 */
export interface FilterAnnotations<A> extends DocAnnotations {
    readonly message?: AST.MessageAnnotation<A>;
    readonly typeId?: AST.TypeAnnotation | {
        id: AST.TypeAnnotation;
        params: unknown;
    };
    /**
     * Attaches a JSON Schema annotation to this refinement.
     *
     * If the schema is composed of more than one refinement, the corresponding annotations will be merged.
     */
    readonly jsonSchema?: AST.JSONSchemaAnnotation;
    readonly arbitrary?: (...args: ReadonlyArray<Arbitrary<any>>) => Arbitrary<any>;
    readonly pretty?: (...args: ReadonlyArray<Pretty.Pretty<any>>) => Pretty.Pretty<any>;
    readonly equivalence?: () => Equivalence.Equivalence<A>;
}
/**
 * @category annotations
 * @since 1.0.0
 */
export declare const annotations: (annotations: AST.Annotations) => <I, A>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category annotations
 * @since 1.0.0
 */
export declare const message: (message: AST.MessageAnnotation<unknown>) => <I, A>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category annotations
 * @since 1.0.0
 */
export declare const identifier: (identifier: AST.IdentifierAnnotation) => <I, A>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category annotations
 * @since 1.0.0
 */
export declare const title: (title: AST.TitleAnnotation) => <I, A>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category annotations
 * @since 1.0.0
 */
export declare const description: (description: AST.DescriptionAnnotation) => <I, A>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category annotations
 * @since 1.0.0
 */
export declare const examples: (examples: AST.ExamplesAnnotation) => <I, A>(self: Schema<I, A>) => Schema<I, A>;
declare const _default: <A>(value: A) => <I>(self: Schema<I, A>) => Schema<I, A>;
export { 
/**
 * @category annotations
 * @since 1.0.0
 */
_default as default };
/**
 * @category annotations
 * @since 1.0.0
 */
export declare const documentation: (documentation: AST.DocumentationAnnotation) => <I, A>(self: Schema<I, A>) => Schema<I, A>;
/**
 * Attaches a JSON Schema annotation to a schema that represents a refinement.
 *
 * If the schema is composed of more than one refinement, the corresponding annotations will be merged.
 *
 * @category annotations
 * @since 1.0.0
 */
export declare const jsonSchema: (jsonSchema: AST.JSONSchemaAnnotation) => <I, A>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category annotations
 * @since 1.0.0
 */
export declare const equivalence: <A>(equivalence: Equivalence.Equivalence<A>) => <I>(self: Schema<I, A>) => Schema<I, A>;
type Rename<A, M> = {
    [K in keyof A as K extends keyof M ? M[K] extends PropertyKey ? M[K] : never : K]: A[K];
};
/**
 * @category renaming
 * @since 1.0.0
 */
export declare const rename: {
    <A, const M extends {
        readonly [K in keyof A]?: PropertyKey;
    } & {
        readonly [K in Exclude<keyof M, keyof A>]: never;
    }>(mapping: M): <I>(self: Schema<I, A>) => Schema<I, Simplify<Rename<A, M>>>;
    <I, A, const M extends {
        readonly [K in keyof A]?: PropertyKey;
    } & {
        readonly [K in Exclude<keyof M, keyof A>]: never;
    }>(self: Schema<I, A>, mapping: M): Schema<I, Simplify<Rename<A, M>>>;
};
/**
 * @category type id
 * @since 1.0.0
 */
export declare const TrimmedTypeId: unique symbol;
/**
 * Verifies that a string contains no leading or trailing whitespaces.
 *
 * Note. This combinator does not make any transformations, it only validates.
 * If what you were looking for was a combinator to trim strings, then check out the `trim` combinator.
 *
 * @category string filters
 * @since 1.0.0
 */
export declare const trimmed: <A extends string>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const MaxLengthTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type MaxLengthTypeId = typeof MaxLengthTypeId;
/**
 * @category string filters
 * @since 1.0.0
 */
export declare const maxLength: <A extends string>(maxLength: number, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const MinLengthTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type MinLengthTypeId = typeof MinLengthTypeId;
/**
 * @category string filters
 * @since 1.0.0
 */
export declare const minLength: <A extends string>(minLength: number, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const PatternTypeId: unique symbol;
/**
 * @category string filters
 * @since 1.0.0
 */
export declare const pattern: <A extends string>(regex: RegExp, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const StartsWithTypeId: unique symbol;
/**
 * @category string filters
 * @since 1.0.0
 */
export declare const startsWith: <A extends string>(startsWith: string, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const EndsWithTypeId: unique symbol;
/**
 * @category string filters
 * @since 1.0.0
 */
export declare const endsWith: <A extends string>(endsWith: string, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const IncludesTypeId: unique symbol;
/**
 * @category string filters
 * @since 1.0.0
 */
export declare const includes: <A extends string>(searchString: string, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const LowercasedTypeId: unique symbol;
/**
 * Verifies that a string is lowercased.
 *
 * @category string filters
 * @since 1.0.0
 */
export declare const lowercased: <A extends string>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const UppercasedTypeId: unique symbol;
/**
 * Verifies that a string is uppercased.
 *
 * @category string filters
 * @since 1.0.0
 */
export declare const uppercased: <A extends string>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const LengthTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type LengthTypeId = typeof LengthTypeId;
/**
 * @category string filters
 * @since 1.0.0
 */
export declare const length: <A extends string>(length: number, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category string filters
 * @since 1.0.0
 */
export declare const nonEmpty: <A extends string>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * This schema converts a string to lowercase.
 *
 * @category string transformations
 * @since 1.0.0
 */
export declare const Lowercase: Schema<string>;
/**
 * This schema converts a string to uppercase.
 *
 * @category string transformations
 * @since 1.0.0
 */
export declare const Uppercase: Schema<string>;
/**
 * This schema allows removing whitespaces from the beginning and end of a string.
 *
 * @category string transformations
 * @since 1.0.0
 */
export declare const Trim: Schema<string>;
/**
 * Returns a achema that allows splitting a string into an array of strings.
 *
 * @category string transformations
 * @since 1.0.0
 */
export declare const split: (separator: string) => Schema<string, ReadonlyArray<string>>;
/**
 * @since 1.0.0
 */
export type ParseJsonOptions = {
    readonly reviver?: Parameters<typeof JSON.parse>[1];
    readonly replacer?: Parameters<typeof JSON.stringify>[1];
    readonly space?: Parameters<typeof JSON.stringify>[2];
};
/**
 * The `parseJson` combinator provides a method to convert JSON strings into the `unknown` type using the underlying
 * functionality of `JSON.parse`. It also utilizes `JSON.stringify` for encoding.
 *
 * You can optionally provide a `ParseJsonOptions` to configure both `JSON.parse` and `JSON.stringify` executions.
 *
 * Optionally, you can pass a schema `Schema<I, A>` to obtain an `A` type instead of `unknown`.
 *
 * @example
 * import * as S from "@effect/schema/Schema"
 *
 * assert.deepStrictEqual(S.parseSync(S.parseJson())(`{"a":"1"}`), { a: "1" })
 * assert.deepStrictEqual(S.parseSync(S.parseJson(S.struct({ a: S.NumberFromString })))(`{"a":"1"}`), { a: 1 })
 *
 * @category string transformations
 * @since 1.0.0
 */
export declare const parseJson: {
    <I, A>(schema: Schema<I, A>, options?: ParseJsonOptions): Schema<string, A>;
    (options?: ParseJsonOptions): Schema<string, unknown>;
};
/**
 * @category string constructors
 * @since 1.0.0
 */
export declare const NonEmpty: Schema<string>;
/**
 * @category string constructors
 * @since 1.0.0
 */
export declare const Trimmed: Schema<string>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const UUIDTypeId: unique symbol;
/**
 * @category string constructors
 * @since 1.0.0
 */
export declare const UUID: Schema<string>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const ULIDTypeId: unique symbol;
/**
 * @category string constructors
 * @since 1.0.0
 */
export declare const ULID: Schema<string>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const FiniteTypeId: unique symbol;
/**
 * @category number filters
 * @since 1.0.0
 */
export declare const finite: <A extends number>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const GreaterThanTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type GreaterThanTypeId = typeof GreaterThanTypeId;
/**
 * @category number filters
 * @since 1.0.0
 */
export declare const greaterThan: <A extends number>(min: number, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const GreaterThanOrEqualToTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type GreaterThanOrEqualToTypeId = typeof GreaterThanOrEqualToTypeId;
/**
 * @category number filters
 * @since 1.0.0
 */
export declare const greaterThanOrEqualTo: <A extends number>(min: number, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const MultipleOfTypeId: unique symbol;
/**
 * @category number filters
 * @since 1.0.0
 */
export declare const multipleOf: <A extends number>(divisor: number, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const IntTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type IntTypeId = typeof IntTypeId;
/**
 * @category number filters
 * @since 1.0.0
 */
export declare const int: <A extends number>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const LessThanTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type LessThanTypeId = typeof LessThanTypeId;
/**
 * @category number filters
 * @since 1.0.0
 */
export declare const lessThan: <A extends number>(max: number, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const LessThanOrEqualToTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type LessThanOrEqualToTypeId = typeof LessThanOrEqualToTypeId;
/**
 * @category number filters
 * @since 1.0.0
 */
export declare const lessThanOrEqualTo: <A extends number>(max: number, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const BetweenTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type BetweenTypeId = typeof BetweenTypeId;
/**
 * @category number filters
 * @since 1.0.0
 */
export declare const between: <A extends number>(min: number, max: number, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const NonNaNTypeId: unique symbol;
/**
 * @category number filters
 * @since 1.0.0
 */
export declare const nonNaN: <A extends number>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category number filters
 * @since 1.0.0
 */
export declare const positive: <A extends number>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category number filters
 * @since 1.0.0
 */
export declare const negative: <A extends number>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category number filters
 * @since 1.0.0
 */
export declare const nonPositive: <A extends number>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category number filters
 * @since 1.0.0
 */
export declare const nonNegative: <A extends number>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * Clamps a number between a minimum and a maximum value.
 *
 * @category number transformations
 * @since 1.0.0
 */
export declare const clamp: (minimum: number, maximum: number) => <I, A extends number>(self: Schema<I, A>) => Schema<I, A>;
/**
 * This schema transforms a `string` into a `number` by parsing the string using the `Number` function.
 *
 * It returns an error if the value can't be converted (for example when non-numeric characters are provided).
 *
 * The following special string values are supported: "NaN", "Infinity", "-Infinity".
 *
 * @category number constructors
 * @since 1.0.0
 */
export declare const NumberFromString: Schema<string, number>;
/**
 * @category number constructors
 * @since 1.0.0
 */
export declare const Finite: Schema<number>;
/**
 * @category number constructors
 * @since 1.0.0
 */
export declare const Int: Schema<number>;
/**
 * @category number constructors
 * @since 1.0.0
 */
export declare const NonNaN: Schema<number>;
/**
 * @category number constructors
 * @since 1.0.0
 */
export declare const Positive: Schema<number>;
/**
 * @category number constructors
 * @since 1.0.0
 */
export declare const Negative: Schema<number>;
/**
 * @category number constructors
 * @since 1.0.0
 */
export declare const NonPositive: Schema<number>;
/**
 * @category number constructors
 * @since 1.0.0
 */
export declare const NonNegative: Schema<number>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const JsonNumberTypeId: unique symbol;
/**
 * The `JsonNumber` is a schema for representing JSON numbers. It ensures that the provided value is a valid
 * number by filtering out `NaN` and `(+/-) Infinity`. This is useful when you want to validate and represent numbers in JSON
 * format.
 *
 * @example
 * import * as S from "@effect/schema/Schema"
 *
 * const is = S.is(S.JsonNumber)
 *
 * assert.deepStrictEqual(is(42), true)
 * assert.deepStrictEqual(is(Number.NaN), false)
 * assert.deepStrictEqual(is(Number.POSITIVE_INFINITY), false)
 * assert.deepStrictEqual(is(Number.NEGATIVE_INFINITY), false)
 *
 * @category number constructors
 * @since 1.0.0
 */
export declare const JsonNumber: Schema<number>;
/**
 * @category boolean transformations
 * @since 1.0.0
 */
export declare const Not: Schema<boolean>;
/**
 * This schema transforms a `string` into a `symbol`.
 *
 * @category symbol transformations
 * @since 1.0.0
 */
export declare const symbol: Schema<string, symbol>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const GreaterThanBigintTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type GreaterThanBigintTypeId = typeof GreaterThanBigintTypeId;
/**
 * @category bigint filters
 * @since 1.0.0
 */
export declare const greaterThanBigint: <A extends bigint>(min: bigint, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const GreaterThanOrEqualToBigintTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type GreaterThanOrEqualToBigintTypeId = typeof GreaterThanOrEqualToBigintTypeId;
/**
 * @category bigint filters
 * @since 1.0.0
 */
export declare const greaterThanOrEqualToBigint: <A extends bigint>(min: bigint, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const LessThanBigintTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type LessThanBigintTypeId = typeof LessThanBigintTypeId;
/**
 * @category bigint filters
 * @since 1.0.0
 */
export declare const lessThanBigint: <A extends bigint>(max: bigint, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const LessThanOrEqualToBigintTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type LessThanOrEqualToBigintTypeId = typeof LessThanOrEqualToBigintTypeId;
/**
 * @category bigint filters
 * @since 1.0.0
 */
export declare const lessThanOrEqualToBigint: <A extends bigint>(max: bigint, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const BetweenBigintTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type BetweenBigintTypeId = typeof BetweenBigintTypeId;
/**
 * @category bigint filters
 * @since 1.0.0
 */
export declare const betweenBigint: <A extends bigint>(min: bigint, max: bigint, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category bigint filters
 * @since 1.0.0
 */
export declare const positiveBigint: <A extends bigint>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category bigint filters
 * @since 1.0.0
 */
export declare const negativeBigint: <A extends bigint>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category bigint filters
 * @since 1.0.0
 */
export declare const nonNegativeBigint: <A extends bigint>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category bigint filters
 * @since 1.0.0
 */
export declare const nonPositiveBigint: <A extends bigint>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * Clamps a bigint between a minimum and a maximum value.
 *
 * @category bigint transformations
 * @since 1.0.0
 */
export declare const clampBigint: (minimum: bigint, maximum: bigint) => <I, A extends bigint>(self: Schema<I, A>) => Schema<I, A>;
/**
 * This schema transforms a `string` into a `bigint` by parsing the string using the `BigInt` function.
 *
 * It returns an error if the value can't be converted (for example when non-numeric characters are provided).
 *
 * @category bigint transformations
 * @since 1.0.0
 */
export declare const bigint: Schema<string, bigint>;
/**
 * @category bigint constructors
 * @since 1.0.0
 */
export declare const PositiveBigintFromSelf: Schema<bigint>;
/**
 * @category bigint constructors
 * @since 1.0.0
 */
export declare const PositiveBigint: Schema<string, bigint>;
/**
 * @category bigint constructors
 * @since 1.0.0
 */
export declare const NegativeBigintFromSelf: Schema<bigint>;
/**
 * @category bigint constructors
 * @since 1.0.0
 */
export declare const NegativeBigint: Schema<string, bigint>;
/**
 * @category bigint constructors
 * @since 1.0.0
 */
export declare const NonPositiveBigintFromSelf: Schema<bigint>;
/**
 * @category bigint constructors
 * @since 1.0.0
 */
export declare const NonPositiveBigint: Schema<string, bigint>;
/**
 * @category bigint constructors
 * @since 1.0.0
 */
export declare const NonNegativeBigintFromSelf: Schema<bigint>;
/**
 * @category bigint constructors
 * @since 1.0.0
 */
export declare const NonNegativeBigint: Schema<string, bigint>;
/**
 * This schema transforms a `number` into a `bigint` by parsing the number using the `BigInt` function.
 *
 * It returns an error if the value can't be safely encoded as a `number` due to being out of range.
 *
 * @category bigint transformations
 * @since 1.0.0
 */
export declare const BigintFromNumber: Schema<number, bigint>;
/**
 * @category Secret constructors
 * @since 1.0.0
 */
export declare const SecretFromSelf: Schema<Secret.Secret>;
declare const _Secret: Schema<string, Secret.Secret>;
export { 
/**
 * A schema that transforms a `string` into a `Secret`.
 *
 * @category Secret transformations
 * @since 1.0.0
 */
_Secret as Secret };
/**
 * @category Duration constructors
 * @since 1.0.0
 */
export declare const DurationFromSelf: Schema<Duration.Duration>;
/**
 * A schema that transforms a `bigint` tuple into a `Duration`.
 * Treats the value as the number of nanoseconds.
 *
 * @category Duration transformations
 * @since 1.0.0
 */
export declare const DurationFromNanos: Schema<bigint, Duration.Duration>;
/**
 * A schema that transforms a `number` tuple into a `Duration`.
 * Treats the value as the number of milliseconds.
 *
 * @category Duration transformations
 * @since 1.0.0
 */
export declare const DurationFromMillis: Schema<number, Duration.Duration>;
declare const _Duration: Schema<readonly [seconds: number, nanos: number], Duration.Duration>;
export { 
/**
 * A schema that transforms a `[number, number]` tuple into a `Duration`.
 *
 * @category Duration transformations
 * @since 1.0.0
 */
_Duration as Duration };
/**
 * Clamps a `Duration` between a minimum and a maximum value.
 *
 * @category Duration transformations
 * @since 1.0.0
 */
export declare const clampDuration: (minimum: Duration.DurationInput, maximum: Duration.DurationInput) => <I, A extends Duration.Duration>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const LessThanDurationTypeId: unique symbol;
/**
 * @category Duration filters
 * @since 1.0.0
 */
export declare const lessThanDuration: <A extends Duration.Duration>(max: Duration.DurationInput, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const LessThanOrEqualToDurationTypeId: unique symbol;
/**
 * @category Duration filters
 * @since 1.0.0
 */
export declare const lessThanOrEqualToDuration: <A extends Duration.Duration>(max: Duration.DurationInput, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const GreaterThanDurationTypeId: unique symbol;
/**
 * @category Duration filters
 * @since 1.0.0
 */
export declare const greaterThanDuration: <A extends Duration.Duration>(min: Duration.DurationInput, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const GreaterThanOrEqualToDurationTypeId: unique symbol;
/**
 * @category Duration filters
 * @since 1.0.0
 */
export declare const greaterThanOrEqualToDuration: <A extends Duration.Duration>(min: Duration.DurationInput, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const BetweenDurationTypeId: unique symbol;
/**
 * @category Duration filters
 * @since 1.0.0
 */
export declare const betweenDuration: <A extends Duration.Duration>(minimum: Duration.DurationInput, maximum: Duration.DurationInput, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category Uint8Array constructors
 * @since 1.0.0
 */
export declare const Uint8ArrayFromSelf: Schema<Uint8Array>;
declare const _Uint8Array: Schema<ReadonlyArray<number>, Uint8Array>;
export { 
/**
 * A schema that transforms a `number` array into a `Uint8Array`.
 *
 * @category Uint8Array transformations
 * @since 1.0.0
 */
_Uint8Array as Uint8Array };
/**
 * @category Encoding transformations
 * @since 1.0.0
 */
export declare const Base64: Schema<string, Uint8Array>;
/**
 * @category Encoding transformations
 * @since 1.0.0
 */
export declare const Base64Url: Schema<string, Uint8Array>;
/**
 * @category Encoding transformations
 * @since 1.0.0
 */
export declare const Hex: Schema<string, Uint8Array>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const MinItemsTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type MinItemsTypeId = typeof MinItemsTypeId;
/**
 * @category ReadonlyArray filters
 * @since 1.0.0
 */
export declare const minItems: <A>(n: number, options?: FilterAnnotations<readonly A[]> | undefined) => <I>(self: Schema<I, readonly A[]>) => Schema<I, readonly A[]>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const MaxItemsTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type MaxItemsTypeId = typeof MaxItemsTypeId;
/**
 * @category ReadonlyArray filters
 * @since 1.0.0
 */
export declare const maxItems: <A>(n: number, options?: FilterAnnotations<readonly A[]> | undefined) => <I>(self: Schema<I, readonly A[]>) => Schema<I, readonly A[]>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const ItemsCountTypeId: unique symbol;
/**
 * @category type id
 * @since 1.0.0
 */
export type ItemsCountTypeId = typeof ItemsCountTypeId;
/**
 * @category ReadonlyArray filters
 * @since 1.0.0
 */
export declare const itemsCount: <A>(n: number, options?: FilterAnnotations<readonly A[]> | undefined) => <I>(self: Schema<I, readonly A[]>) => Schema<I, readonly A[]>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const ValidDateTypeId: unique symbol;
/**
 * A filter that **excludes invalid** dates (e.g., `new Date("Invalid Date")` is rejected).
 *
 * @category Date filters
 * @since 1.0.0
 */
export declare const validDate: (options?: FilterAnnotations<Date>) => <I>(self: Schema<I, Date>) => Schema<I, Date>;
/**
 * Represents a schema for handling potentially **invalid** `Date` instances (e.g., `new Date("Invalid Date")` is not rejected).
 *
 * @category Date constructors
 * @since 1.0.0
 */
export declare const DateFromSelf: Schema<Date>;
/**
 * Represents a schema for handling only **valid** dates. For example, `new Date("Invalid Date")` is rejected, even though it is an instance of `Date`.
 *
 * @category Date constructors
 * @since 1.0.0
 */
export declare const ValidDateFromSelf: Schema<Date>;
/**
 * Represents a schema that converts a `string` into a (potentially invalid) `Date` (e.g., `new Date("Invalid Date")` is not rejected).
 *
 * @category Date transformations
 * @since 1.0.0
 */
export declare const DateFromString: Schema<string, Date>;
declare const _Date: Schema<string, Date>;
export { 
/**
 * A schema that transforms a `string` into a **valid** `Date`, ensuring that invalid dates, such as `new Date("Invalid Date")`, are rejected.
 *
 * @category Date transformations
 * @since 1.0.0
 */
_Date as Date };
/**
 * @category Option utils
 * @since 1.0.0
 */
export type OptionFrom<I> = {
    readonly _tag: "None";
} | {
    readonly _tag: "Some";
    readonly value: I;
};
/**
 * @category Option transformations
 * @since 1.0.0
 */
export declare const optionFromSelf: <I, A>(value: Schema<I, A>) => Schema<Option.Option<I>, Option.Option<A>>;
/**
 * @category Option transformations
 * @since 1.0.0
 */
export declare const option: <I, A>(value: Schema<I, A>) => Schema<OptionFrom<I>, Option.Option<A>>;
/**
 * @category Option transformations
 * @since 1.0.0
 */
export declare const optionFromNullable: <I, A>(value: Schema<I, A>) => Schema<I | null, Option.Option<A>>;
/**
 * @category Option transformations
 * @since 1.0.0
 */
export declare const optionFromNullish: <I, A>(value: Schema<I, A>, onNoneEncoding: null | undefined) => Schema<I | null | undefined, Option.Option<A>>;
/**
 * @category Either utils
 * @since 1.0.0
 */
export type EitherFrom<IE, IA> = {
    readonly _tag: "Left";
    readonly left: IE;
} | {
    readonly _tag: "Right";
    readonly right: IA;
};
/**
 * @category Either transformations
 * @since 1.0.0
 */
export declare const eitherFromSelf: <IE, E, IA, A>(left: Schema<IE, E>, right: Schema<IA, A>) => Schema<Either.Either<IE, IA>, Either.Either<E, A>>;
/**
 * @category Either transformations
 * @since 1.0.0
 */
export declare const either: <IE, E, IA, A>(left: Schema<IE, E>, right: Schema<IA, A>) => Schema<EitherFrom<IE, IA>, Either.Either<E, A>>;
/**
 * @category ReadonlyMap transformations
 * @since 1.0.0
 */
export declare const readonlyMapFromSelf: <IK, K, IV, V>(key: Schema<IK, K>, value: Schema<IV, V>) => Schema<ReadonlyMap<IK, IV>, ReadonlyMap<K, V>>;
/**
 * @category ReadonlyMap transformations
 * @since 1.0.0
 */
export declare const readonlyMap: <IK, K, IV, V>(key: Schema<IK, K>, value: Schema<IV, V>) => Schema<readonly (readonly [IK, IV])[], ReadonlyMap<K, V>>;
/**
 * @category ReadonlySet transformations
 * @since 1.0.0
 */
export declare const readonlySetFromSelf: <I, A>(item: Schema<I, A>) => Schema<ReadonlySet<I>, ReadonlySet<A>>;
/**
 * @category ReadonlySet transformations
 * @since 1.0.0
 */
export declare const readonlySet: <I, A>(item: Schema<I, A>) => Schema<readonly I[], ReadonlySet<A>>;
/**
 * @category BigDecimal constructors
 * @since 1.0.0
 */
export declare const BigDecimalFromSelf: Schema<BigDecimal.BigDecimal>;
declare const _BigDecimal: Schema<string, BigDecimal.BigDecimal>;
export { 
/**
 * @category BigDecimal transformations
 * @since 1.0.0
 */
_BigDecimal as BigDecimal };
/**
 * A schema that transforms a `number` into a `BigDecimal`.
 * When encoding, this Schema will produce incorrect results if the BigDecimal exceeds the 64-bit range of a number.
 *
 * @category BigDecimal transformations
 * @since 1.0.0
 */
export declare const BigDecimalFromNumber: Schema<number, BigDecimal.BigDecimal>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const GreaterThanBigDecimalTypeId: unique symbol;
/**
 * @category BigDecimal filters
 * @since 1.0.0
 */
export declare const greaterThanBigDecimal: <A extends BigDecimal.BigDecimal>(min: BigDecimal.BigDecimal, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const GreaterThanOrEqualToBigDecimalTypeId: unique symbol;
/**
 * @category BigDecimal filters
 * @since 1.0.0
 */
export declare const greaterThanOrEqualToBigDecimal: <A extends BigDecimal.BigDecimal>(min: BigDecimal.BigDecimal, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const LessThanBigDecimalTypeId: unique symbol;
/**
 * @category BigDecimal filters
 * @since 1.0.0
 */
export declare const lessThanBigDecimal: <A extends BigDecimal.BigDecimal>(max: BigDecimal.BigDecimal, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const LessThanOrEqualToBigDecimalTypeId: unique symbol;
/**
 * @category BigDecimal filters
 * @since 1.0.0
 */
export declare const lessThanOrEqualToBigDecimal: <A extends BigDecimal.BigDecimal>(max: BigDecimal.BigDecimal, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const PositiveBigDecimalTypeId: unique symbol;
/**
 * @category BigDecimal filters
 * @since 1.0.0
 */
export declare const positiveBigDecimal: <A extends BigDecimal.BigDecimal>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const NonNegativeBigDecimalTypeId: unique symbol;
/**
 * @category BigDecimal filters
 * @since 1.0.0
 */
export declare const nonNegativeBigDecimal: <A extends BigDecimal.BigDecimal>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const NegativeBigDecimalTypeId: unique symbol;
/**
 * @category BigDecimal filters
 * @since 1.0.0
 */
export declare const negativeBigDecimal: <A extends BigDecimal.BigDecimal>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const NonPositiveBigDecimalTypeId: unique symbol;
/**
 * @category BigDecimal filters
 * @since 1.0.0
 */
export declare const nonPositiveBigDecimal: <A extends BigDecimal.BigDecimal>(options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category type id
 * @since 1.0.0
 */
export declare const BetweenBigDecimalTypeId: unique symbol;
/**
 * @category BigDecimal filters
 * @since 1.0.0
 */
export declare const betweenBigDecimal: <A extends BigDecimal.BigDecimal>(minimum: BigDecimal.BigDecimal, maximum: BigDecimal.BigDecimal, options?: FilterAnnotations<A> | undefined) => <I>(self: Schema<I, A>) => Schema<I, A>;
/**
 * Clamps a `BigDecimal` between a minimum and a maximum value.
 *
 * @category BigDecimal transformations
 * @since 1.0.0
 */
export declare const clampBigDecimal: (minimum: BigDecimal.BigDecimal, maximum: BigDecimal.BigDecimal) => <I, A extends BigDecimal.BigDecimal>(self: Schema<I, A>) => Schema<I, A>;
/**
 * Negates a `BigDecimal`.
 *
 * @category BigDecimal transformations
 * @since 1.0.0
 */
export declare const negateBigDecimal: <I, A extends BigDecimal.BigDecimal>(self: Schema<I, A>) => Schema<I, A>;
/**
 * @category Chunk transformations
 * @since 1.0.0
 */
export declare const chunkFromSelf: <I, A>(item: Schema<I, A>) => Schema<Chunk.Chunk<I>, Chunk.Chunk<A>>;
/**
 * @category Chunk transformations
 * @since 1.0.0
 */
export declare const chunk: <I, A>(item: Schema<I, A>) => Schema<readonly I[], Chunk.Chunk<A>>;
/**
 * @category Data transformations
 * @since 1.0.0
 */
export declare const dataFromSelf: <I extends readonly any[] | Readonly<Record<string, any>>, A extends readonly any[] | Readonly<Record<string, any>>>(item: Schema<I, A>) => Schema<Data.Data<I>, Data.Data<A>>;
/**
 * @category Data transformations
 * @since 1.0.0
 */
export declare const data: <I extends readonly any[] | Readonly<Record<string, any>>, A extends readonly any[] | Readonly<Record<string, any>>>(item: Schema<I, A>) => Schema<I, Data.Data<A>>;
type MissingSelfGeneric<Usage extends string, Params extends string = ""> = `Missing \`Self\` generic - use \`class Self extends ${Usage}<Self>()(${Params}{ ... })\``;
/**
 * @category classes
 * @since 1.0.0
 */
export interface Class<I, A, C, Self, Inherited = Data.Case> extends Schema<I, Self> {
    new (props: Equals<C, {}> extends true ? void | {} : C, disableValidation?: boolean): A & Omit<Inherited, keyof A>;
    readonly struct: Schema<I, A>;
    readonly extend: <Extended>() => <FieldsB extends StructFields>(fields: FieldsB) => [unknown] extends [Extended] ? MissingSelfGeneric<"Base.extend"> : Class<Simplify<Omit<I, keyof FieldsB> & FromStruct<FieldsB>>, Simplify<Omit<A, keyof FieldsB> & ToStruct<FieldsB>>, Simplify<Omit<C, keyof FieldsB> & ToStruct<FieldsB>>, Extended, Self>;
    readonly transform: <Transformed>() => <FieldsB extends StructFields>(fields: FieldsB, decode: (input: A) => ParseResult.ParseResult<Omit<A, keyof FieldsB> & ToStruct<FieldsB>>, encode: (input: Simplify<Omit<A, keyof FieldsB> & ToStruct<FieldsB>>) => ParseResult.ParseResult<A>) => [unknown] extends [Transformed] ? MissingSelfGeneric<"Base.transform"> : Class<I, Simplify<Omit<A, keyof FieldsB> & ToStruct<FieldsB>>, Simplify<Omit<C, keyof FieldsB> & ToStruct<FieldsB>>, Transformed, Self>;
    readonly transformFrom: <Transformed>() => <FieldsB extends StructFields>(fields: FieldsB, decode: (input: I) => ParseResult.ParseResult<Omit<I, keyof FieldsB> & FromStruct<FieldsB>>, encode: (input: Simplify<Omit<I, keyof FieldsB> & FromStruct<FieldsB>>) => ParseResult.ParseResult<I>) => [unknown] extends [Transformed] ? MissingSelfGeneric<"Base.transformFrom"> : Class<I, Simplify<Omit<A, keyof FieldsB> & ToStruct<FieldsB>>, Simplify<Omit<C, keyof FieldsB> & ToStruct<FieldsB>>, Transformed, Self>;
}
/**
 * @category classes
 * @since 1.0.0
 */
export declare const Class: <Self>() => <Fields extends StructFields>(fields: Fields) => [unknown] extends [Self] ? "Missing `Self` generic - use `class Self extends Class<Self>()({ ... })`" : Class<Simplify<FromStruct<Fields>>, Simplify<ToStruct<Fields>>, Simplify<ToStruct<Fields>>, Self, Data.Case>;
/**
 * @category classes
 * @since 1.0.0
 */
export declare const TaggedClass: <Self>() => <Tag extends string, Fields extends StructFields>(tag: Tag, fields: Fields) => [unknown] extends [Self] ? "Missing `Self` generic - use `class Self extends TaggedClass<Self>()(\"Tag\", { ... })`" : Class<Simplify<{
    readonly _tag: Tag;
} & { readonly [K in Exclude<keyof Fields, FromOptionalKeys<Fields>>]: Schema.From<Fields[K]>; } & { readonly [K_1 in FromOptionalKeys<Fields>]?: Schema.From<Fields[K_1]>; }>, Simplify<{
    readonly _tag: Tag;
} & { readonly [K_2 in Exclude<keyof Fields, ToOptionalKeys<Fields>>]: Schema.To<Fields[K_2]>; } & { readonly [K_3 in ToOptionalKeys<Fields>]?: Schema.To<Fields[K_3]>; }>, Simplify<ToStruct<Fields>>, Self, Data.Case>;
/**
 * @category classes
 * @since 1.0.0
 */
export declare const TaggedError: <Self>() => <Tag extends string, Fields extends StructFields>(tag: Tag, fields: Fields) => [unknown] extends [Self] ? "Missing `Self` generic - use `class Self extends TaggedError<Self>()(\"Tag\", { ... })`" : Class<Simplify<{
    readonly _tag: Tag;
} & { readonly [K in Exclude<keyof Fields, FromOptionalKeys<Fields>>]: Schema.From<Fields[K]>; } & { readonly [K_1 in FromOptionalKeys<Fields>]?: Schema.From<Fields[K_1]>; }>, Simplify<{
    readonly _tag: Tag;
} & { readonly [K_2 in Exclude<keyof Fields, ToOptionalKeys<Fields>>]: Schema.To<Fields[K_2]>; } & { readonly [K_3 in ToOptionalKeys<Fields>]?: Schema.To<Fields[K_3]>; }>, Simplify<ToStruct<Fields>>, Self, Effect.Effect<never, Self, never> & Error>;
/**
 * @category classes
 * @since 1.0.0
 */
export interface TaggedRequest<Tag extends string, IS, S, IE, E, IA, A> extends Request.Request<E, A>, Serializable.SerializableWithResult<IS, S, IE, E, IA, A> {
    readonly _tag: Tag;
}
/**
 * @category classes
 * @since 1.0.0
 */
export declare namespace TaggedRequest {
    /**
     * @category classes
     * @since 1.0.0
     */
    type Any = TaggedRequest<string, any, any, any, any, any, any> | TaggedRequest<string, any, any, never, never, any, any>;
}
/**
 * @category classes
 * @since 1.0.0
 */
export declare const TaggedRequest: <Self>() => <Tag extends string, Fields extends StructFields, EI, EA, AI, AA>(tag: Tag, Failure: Schema<EI, EA>, Success: Schema<AI, AA>, fields: Fields) => [unknown] extends [Self] ? "Missing `Self` generic - use `class Self extends TaggedRequest<Self>()(\"Tag\", SuccessSchema, FailureSchema, { ... })`" : Class<Simplify<{
    readonly _tag: Tag;
} & { readonly [K in Exclude<keyof Fields, FromOptionalKeys<Fields>>]: Schema.From<Fields[K]>; } & { readonly [K_1 in FromOptionalKeys<Fields>]?: Schema.From<Fields[K_1]>; }>, Simplify<{
    readonly _tag: Tag;
} & { readonly [K_2 in Exclude<keyof Fields, ToOptionalKeys<Fields>>]: Schema.To<Fields[K_2]>; } & { readonly [K_3 in ToOptionalKeys<Fields>]?: Schema.To<Fields[K_3]>; }>, Simplify<ToStruct<Fields>>, Self, TaggedRequest<Tag, Simplify<{
    readonly _tag: Tag;
} & { readonly [K in Exclude<keyof Fields, FromOptionalKeys<Fields>>]: Schema.From<Fields[K]>; } & { readonly [K_1 in FromOptionalKeys<Fields>]?: Schema.From<Fields[K_1]>; }>, Self, EI, EA, AI, AA>>;
/**
 * @category FiberId
 * @since 1.0.0
 */
export type FiberIdFrom = {
    readonly _tag: "Composite";
    readonly left: FiberIdFrom;
    readonly right: FiberIdFrom;
} | {
    readonly _tag: "None";
} | {
    readonly _tag: "Runtime";
    readonly id: number;
    readonly startTimeMillis: number;
};
/**
 * @category FiberId constructors
 * @since 1.0.0
 */
export declare const FiberIdFromSelf: Schema<FiberId.FiberId, FiberId.FiberId>;
declare const _FiberId: Schema<FiberIdFrom, FiberId.FiberId>;
export { 
/**
 * @category FiberId transformations
 * @since 1.0.0
 */
_FiberId as FiberId };
/**
 * @category Cause utils
 * @since 1.0.0
 */
export type CauseFrom<E> = {
    readonly _tag: "Die";
    readonly defect: unknown;
} | {
    readonly _tag: "Empty";
} | {
    readonly _tag: "Fail";
    readonly error: E;
} | {
    readonly _tag: "Interrupt";
    readonly fiberId: FiberIdFrom;
} | {
    readonly _tag: "Parallel";
    readonly left: CauseFrom<E>;
    readonly right: CauseFrom<E>;
} | {
    readonly _tag: "Sequential";
    readonly left: CauseFrom<E>;
    readonly right: CauseFrom<E>;
};
/**
 * @category Cause transformations
 * @since 1.0.0
 */
export declare const causeFromSelf: <IE, E>(error: Schema<IE, E>, defect?: Schema<unknown, unknown>) => Schema<Cause.Cause<IE>, Cause.Cause<E>>;
/**
 * @category Cause transformations
 * @since 1.0.0
 */
export declare const cause: <EI, E>(error: Schema<EI, E>, defect?: Schema<unknown, unknown>) => Schema<CauseFrom<EI>, Cause.Cause<E>>;
/**
 * @category Exit utils
 * @since 1.0.0
 */
export type ExitFrom<E, A> = {
    readonly _tag: "Failure";
    readonly cause: CauseFrom<E>;
} | {
    readonly _tag: "Success";
    readonly value: A;
};
/**
 * @category Exit transformations
 * @since 1.0.0
 */
export declare const exitFromSelf: <IE, E, IA, A>(error: Schema<IE, E>, value: Schema<IA, A>, defect?: Schema<unknown, unknown>) => Schema<Exit.Exit<IE, IA>, Exit.Exit<E, A>>;
/**
 * @category Exit transformations
 * @since 1.0.0
 */
export declare const exit: <IE, E, IA, A>(error: Schema<IE, E>, value: Schema<IA, A>, defect?: Schema<unknown, unknown>) => Schema<ExitFrom<IE, IA>, Exit.Exit<E, A>>;
//# sourceMappingURL=Schema.d.ts.map