"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMessage = exports.formatExpected = exports.formatErrors = exports.formatActual = void 0;
var Option = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("effect/Option"));
var Predicate = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("effect/Predicate"));
var AST = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("./AST.js"));
function _getRequireWildcardCache(e) {
  if ("function" != typeof WeakMap) return null;
  var r = new WeakMap(),
    t = new WeakMap();
  return (_getRequireWildcardCache = function (e) {
    return e ? t : r;
  })(e);
}
function _interopRequireWildcard(e, r) {
  if (!r && e && e.__esModule) return e;
  if (null === e || "object" != typeof e && "function" != typeof e) return {
    default: e
  };
  var t = _getRequireWildcardCache(r);
  if (t && t.has(e)) return t.get(e);
  var n = {
      __proto__: null
    },
    a = Object.defineProperty && Object.getOwnPropertyDescriptor;
  for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) {
    var i = a ? Object.getOwnPropertyDescriptor(e, u) : null;
    i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u];
  }
  return n.default = e, t && t.set(e, n), n;
}
/**
 * @since 1.0.0
 */

const make = (value, forest = []) => ({
  value,
  forest
});
/**
 * @category formatting
 * @since 1.0.0
 */
const formatErrors = errors => drawTree(make(`error(s) found`, errors.map(go)));
exports.formatErrors = formatErrors;
const drawTree = tree => tree.value + draw("\n", tree.forest);
const draw = (indentation, forest) => {
  let r = "";
  const len = forest.length;
  let tree;
  for (let i = 0; i < len; i++) {
    tree = forest[i];
    const isLast = i === len - 1;
    r += indentation + (isLast ? "└" : "├") + "─ " + tree.value;
    r += draw(indentation + (len > 1 && !isLast ? "│  " : "   "), tree.forest);
  }
  return r;
};
/** @internal */
const formatActual = actual => {
  if (Predicate.isString(actual)) {
    return JSON.stringify(actual);
  } else if (Predicate.isNumber(actual) || actual == null || Predicate.isBoolean(actual) || Predicate.isSymbol(actual) || Predicate.isDate(actual)) {
    return String(actual);
  } else if (Predicate.isBigInt(actual)) {
    return String(actual) + "n";
  } else if (!Array.isArray(actual) && Predicate.hasProperty(actual, "toString") && Predicate.isFunction(actual["toString"]) && actual["toString"] !== Object.prototype.toString) {
    return actual["toString"]();
  }
  try {
    return JSON.stringify(actual);
  } catch (e) {
    return String(actual);
  }
};
exports.formatActual = formatActual;
const formatTemplateLiteralSpan = span => {
  switch (span.type._tag) {
    case "StringKeyword":
      return "${string}";
    case "NumberKeyword":
      return "${number}";
  }
};
const formatTemplateLiteral = ast => ast.head + ast.spans.map(span => formatTemplateLiteralSpan(span) + span.literal).join("");
const getExpected = ast => AST.getIdentifierAnnotation(ast).pipe(Option.orElse(() => AST.getTitleAnnotation(ast)), Option.orElse(() => AST.getDescriptionAnnotation(ast)));
/** @internal */
const formatExpected = ast => {
  switch (ast._tag) {
    case "StringKeyword":
    case "NumberKeyword":
    case "BooleanKeyword":
    case "BigIntKeyword":
    case "UndefinedKeyword":
    case "SymbolKeyword":
    case "ObjectKeyword":
    case "AnyKeyword":
    case "UnknownKeyword":
    case "VoidKeyword":
    case "NeverKeyword":
      return Option.getOrElse(getExpected(ast), () => ast._tag);
    case "Literal":
      return Option.getOrElse(getExpected(ast), () => formatActual(ast.literal));
    case "UniqueSymbol":
      return Option.getOrElse(getExpected(ast), () => formatActual(ast.symbol));
    case "Union":
      return [...new Set(ast.types.map(formatExpected))].join(" or ");
    case "TemplateLiteral":
      return Option.getOrElse(getExpected(ast), () => formatTemplateLiteral(ast));
    case "Tuple":
      return Option.getOrElse(getExpected(ast), () => "<anonymous tuple or array schema>");
    case "TypeLiteral":
      return Option.getOrElse(getExpected(ast), () => "<anonymous type literal schema>");
    case "Enums":
      return Option.getOrElse(getExpected(ast), () => ast.enums.map((_, value) => JSON.stringify(value)).join(" | "));
    case "Suspend":
      return Option.getOrElse(getExpected(ast), () => "<anonymous suspended schema>");
    case "Declaration":
      return Option.getOrElse(getExpected(ast), () => "<anonymous declaration schema>");
    case "Refinement":
      return Option.getOrElse(getExpected(ast), () => "<anonymous refinement schema>");
    case "Transform":
      return Option.getOrElse(getExpected(ast), () => `${formatExpected(ast.from)} <-> ${formatExpected(ast.to)}`);
  }
};
exports.formatExpected = formatExpected;
const isCollapsible = (es, errors) => es.length === 1 && es[0].forest.length !== 0 && errors[0]._tag !== "UnionMember";
/** @internal */
const getMessage = e => AST.getMessageAnnotation(e.expected).pipe(Option.map(annotation => annotation(e.actual)), Option.orElse(() => e.message), Option.getOrElse(() => `Expected ${formatExpected(e.expected)}, actual ${formatActual(e.actual)}`));
exports.getMessage = getMessage;
const go = e => {
  switch (e._tag) {
    case "Type":
      return make(getMessage(e));
    case "Forbidden":
      return make("is forbidden");
    case "Index":
      {
        const es = e.errors.map(go);
        if (isCollapsible(es, e.errors)) {
          return make(`[${e.index}]${es[0].value}`, es[0].forest);
        }
        return make(`[${e.index}]`, es);
      }
    case "Unexpected":
      return make("is unexpected" + (Option.isSome(e.ast) ? `, expected ${formatExpected(e.ast.value)}` : ""));
    case "Key":
      {
        const es = e.errors.map(go);
        if (isCollapsible(es, e.errors)) {
          return make(`[${formatActual(e.key)}]${es[0].value}`, es[0].forest);
        }
        return make(`[${formatActual(e.key)}]`, es);
      }
    case "Missing":
      return make("is missing");
    case "UnionMember":
      return make("union member", e.errors.map(go));
  }
};
//# sourceMappingURL=TreeFormatter.js.map