"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateSync = exports.validatePromise = exports.validateOption = exports.validateEither = exports.validate = exports.parseSync = exports.parsePromise = exports.parseOption = exports.parseEither = exports.parse = exports.is = exports.getTemplateLiteralRegex = exports.getSearchTree = exports.getLiterals = exports.getFinalTransformation = exports.getEither = exports.encodeSync = exports.encodePromise = exports.encodeOption = exports.encodeEither = exports.encode = exports.defaultParseOption = exports.decodeSync = exports.decodePromise = exports.decodeOption = exports.decodeEither = exports.decode = exports.asserts = void 0;
var Effect = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("effect/Effect"));
var Either = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("effect/Either"));
var _GlobalValue = /*#__PURE__*/require("effect/GlobalValue");
var Option = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("effect/Option"));
var Predicate = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("effect/Predicate"));
var ReadonlyArray = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("effect/ReadonlyArray"));
var AST = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("./AST.js"));
var Internal = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("./internal/ast.js"));
var ParseResult = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("./ParseResult.js"));
var TreeFormatter = /*#__PURE__*/_interopRequireWildcard( /*#__PURE__*/require("./TreeFormatter.js"));
function _getRequireWildcardCache(e) {
  if ("function" != typeof WeakMap) return null;
  var r = new WeakMap(),
    t = new WeakMap();
  return (_getRequireWildcardCache = function (e) {
    return e ? t : r;
  })(e);
}
function _interopRequireWildcard(e, r) {
  if (!r && e && e.__esModule) return e;
  if (null === e || "object" != typeof e && "function" != typeof e) return {
    default: e
  };
  var t = _getRequireWildcardCache(r);
  if (t && t.has(e)) return t.get(e);
  var n = {
      __proto__: null
    },
    a = Object.defineProperty && Object.getOwnPropertyDescriptor;
  for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) {
    var i = a ? Object.getOwnPropertyDescriptor(e, u) : null;
    i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u];
  }
  return n.default = e, t && t.set(e, n), n;
}
/**
 * @since 1.0.0
 */

/** @internal */
const getEither = (ast, isDecoding) => goMemo(ast, isDecoding);
exports.getEither = getEither;
const getSync = (ast, isDecoding) => {
  const parser = getEither(ast, isDecoding);
  return (input, options) => {
    const result = parser(input, options);
    if (Either.isLeft(result)) {
      throw new Error(TreeFormatter.formatErrors(result.left.errors));
    }
    return result.right;
  };
};
const getOption = (ast, isDecoding) => {
  const parser = getEither(ast, isDecoding);
  return (input, options) => Option.getRight(parser(input, options));
};
const getEffect = (ast, isDecoding) => {
  const parser = goMemo(ast, isDecoding);
  return (input, options) => parser(input, {
    ...options,
    isEffectAllowed: true
  });
};
const getPromise = (ast, isDecoding) => {
  const parser = getEffect(ast, isDecoding);
  return (input, options) => Effect.runPromise(parser(input, options));
};
/**
 * @category parsing
 * @since 1.0.0
 */
const parseSync = schema => getSync(schema.ast, true);
/**
 * @category parsing
 * @since 1.0.0
 */
exports.parseSync = parseSync;
const parseOption = schema => getOption(schema.ast, true);
/**
 * @category parsing
 * @since 1.0.0
 */
exports.parseOption = parseOption;
const parseEither = schema => getEither(schema.ast, true);
/**
 * @category parsing
 * @since 1.0.0
 */
exports.parseEither = parseEither;
const parsePromise = schema => getPromise(schema.ast, true);
/**
 * @category parsing
 * @since 1.0.0
 */
exports.parsePromise = parsePromise;
const parse = schema => getEffect(schema.ast, true);
/**
 * @category decoding
 * @since 1.0.0
 */
exports.parse = parse;
const decodeSync = exports.decodeSync = parseSync;
/**
 * @category decoding
 * @since 1.0.0
 */
const decodeOption = exports.decodeOption = parseOption;
/**
 * @category decoding
 * @since 1.0.0
 */
const decodeEither = exports.decodeEither = parseEither;
/**
 * @category decoding
 * @since 1.0.0
 */
const decodePromise = exports.decodePromise = parsePromise;
/**
 * @category decoding
 * @since 1.0.0
 */
const decode = exports.decode = parse;
/**
 * @category validation
 * @since 1.0.0
 */
const validateSync = schema => getSync(AST.to(schema.ast), true);
/**
 * @category validation
 * @since 1.0.0
 */
exports.validateSync = validateSync;
const validateOption = schema => getOption(AST.to(schema.ast), true);
/**
 * @category validation
 * @since 1.0.0
 */
exports.validateOption = validateOption;
const validateEither = schema => getEither(AST.to(schema.ast), true);
/**
 * @category validation
 * @since 1.0.0
 */
exports.validateEither = validateEither;
const validatePromise = schema => getPromise(AST.to(schema.ast), true);
/**
 * @category validation
 * @since 1.0.0
 */
exports.validatePromise = validatePromise;
const validate = schema => getEffect(AST.to(schema.ast), true);
/**
 * @category validation
 * @since 1.0.0
 */
exports.validate = validate;
const is = schema => {
  const getEither = validateEither(schema);
  return a => Either.isRight(getEither(a));
};
/**
 * @category validation
 * @since 1.0.0
 */
exports.is = is;
const asserts = schema => {
  const get = validateSync(schema);
  return (a, options) => {
    get(a, options);
  };
};
/**
 * @category encoding
 * @since 1.0.0
 */
exports.asserts = asserts;
const encodeSync = schema => getSync(schema.ast, false);
/**
 * @category encoding
 * @since 1.0.0
 */
exports.encodeSync = encodeSync;
const encodeOption = schema => getOption(schema.ast, false);
/**
 * @category encoding
 * @since 1.0.0
 */
exports.encodeOption = encodeOption;
const encodeEither = schema => getEither(schema.ast, false);
/**
 * @category encoding
 * @since 1.0.0
 */
exports.encodeEither = encodeEither;
const encodePromise = schema => getPromise(schema.ast, false);
/**
 * @category encoding
 * @since 1.0.0
 */
exports.encodePromise = encodePromise;
const encode = schema => getEffect(schema.ast, false);
/**
 * @since 1.0.0"
 */
exports.encode = encode;
const defaultParseOption = exports.defaultParseOption = {};
const decodeMemoMap = /*#__PURE__*/(0, _GlobalValue.globalValue)( /*#__PURE__*/Symbol.for("@effect/schema/Parser/decodeMemoMap"), () => new WeakMap());
const encodeMemoMap = /*#__PURE__*/(0, _GlobalValue.globalValue)( /*#__PURE__*/Symbol.for("@effect/schema/Parser/encodeMemoMap"), () => new WeakMap());
const goMemo = (ast, isDecoding) => {
  const memoMap = isDecoding ? decodeMemoMap : encodeMemoMap;
  const memo = memoMap.get(ast);
  if (memo) {
    return memo;
  }
  const parser = go(ast, isDecoding);
  memoMap.set(ast, parser);
  return parser;
};
const go = (ast, isDecoding) => {
  switch (ast._tag) {
    case "Refinement":
      {
        if (isDecoding) {
          const from = goMemo(ast.from, true);
          return (i, options) => handleForbidden(ParseResult.flatMap(from(i, options), a => Option.match(ast.filter(a, options ?? defaultParseOption, ast), {
            onNone: () => ParseResult.succeed(a),
            onSome: ParseResult.fail
          })), options);
        } else {
          const from = goMemo(AST.to(ast), true);
          const to = goMemo(dropRightRefinement(ast.from), false);
          return (i, options) => handleForbidden(ParseResult.flatMap(from(i, options), a => to(a, options)), options);
        }
      }
    case "Transform":
      {
        const transform = getFinalTransformation(ast.transformation, isDecoding);
        const from = isDecoding ? goMemo(ast.from, true) : goMemo(ast.to, false);
        const to = isDecoding ? goMemo(ast.to, true) : goMemo(ast.from, false);
        return (i1, options) => handleForbidden(ParseResult.flatMap(from(i1, options), a => ParseResult.flatMap(transform(a, options ?? defaultParseOption, ast), i2 => to(i2, options))), options);
      }
    case "Declaration":
      {
        const parse = ast.decode(isDecoding, ...ast.typeParameters);
        return (i, options) => handleForbidden(parse(i, options ?? defaultParseOption, ast), options);
      }
    case "Literal":
      return fromRefinement(ast, u => u === ast.literal);
    case "UniqueSymbol":
      return fromRefinement(ast, u => u === ast.symbol);
    case "UndefinedKeyword":
      return fromRefinement(ast, Predicate.isUndefined);
    case "VoidKeyword":
      return fromRefinement(ast, Predicate.isUndefined);
    case "NeverKeyword":
      return fromRefinement(ast, Predicate.isNever);
    case "UnknownKeyword":
    case "AnyKeyword":
      return ParseResult.succeed;
    case "StringKeyword":
      return fromRefinement(ast, Predicate.isString);
    case "NumberKeyword":
      return fromRefinement(ast, Predicate.isNumber);
    case "BooleanKeyword":
      return fromRefinement(ast, Predicate.isBoolean);
    case "BigIntKeyword":
      return fromRefinement(ast, Predicate.isBigInt);
    case "SymbolKeyword":
      return fromRefinement(ast, Predicate.isSymbol);
    case "ObjectKeyword":
      return fromRefinement(ast, Predicate.isObject);
    case "Enums":
      return fromRefinement(ast, u => ast.enums.some(([_, value]) => value === u));
    case "TemplateLiteral":
      {
        const regex = getTemplateLiteralRegex(ast);
        return fromRefinement(ast, u => Predicate.isString(u) && regex.test(u));
      }
    case "Tuple":
      {
        const elements = ast.elements.map(e => goMemo(e.type, isDecoding));
        const rest = Option.map(ast.rest, ReadonlyArray.map(ast => goMemo(ast, isDecoding)));
        let requiredLen = ast.elements.filter(e => !e.isOptional).length;
        if (Option.isSome(ast.rest)) {
          requiredLen += ast.rest.value.length - 1;
        }
        return (input, options) => {
          if (!Array.isArray(input)) {
            return ParseResult.fail(ParseResult.type(ast, input));
          }
          const allErrors = options?.errors === "all";
          const es = [];
          let stepKey = 0;
          // ---------------------------------------------
          // handle missing indexes
          // ---------------------------------------------
          const len = input.length;
          for (let i = len; i <= requiredLen - 1; i++) {
            const e = ParseResult.index(i, [ParseResult.missing]);
            if (allErrors) {
              es.push([stepKey++, e]);
              continue;
            } else {
              return ParseResult.fail(e);
            }
          }
          // ---------------------------------------------
          // handle excess indexes
          // ---------------------------------------------
          if (Option.isNone(ast.rest)) {
            for (let i = ast.elements.length; i <= len - 1; i++) {
              const e = ParseResult.index(i, [ParseResult.unexpected(Option.none())]);
              if (allErrors) {
                es.push([stepKey++, e]);
                continue;
              } else {
                return ParseResult.fail(mutableAppend(sortByIndex(es), e));
              }
            }
          }
          const output = [];
          let i = 0;
          let queue = undefined;
          // ---------------------------------------------
          // handle elements
          // ---------------------------------------------
          for (; i < elements.length; i++) {
            if (len < i + 1) {
              if (ast.elements[i].isOptional) {
                // the input element is missing
                continue;
              }
            } else {
              const parser = elements[i];
              const te = parser(input[i], options);
              const eu = ParseResult.eitherOrUndefined(te);
              if (eu) {
                if (Either.isLeft(eu)) {
                  // the input element is present but is not valid
                  const e = ParseResult.index(i, eu.left.errors);
                  if (allErrors) {
                    es.push([stepKey++, e]);
                    continue;
                  } else {
                    return ParseResult.fail(mutableAppend(sortByIndex(es), e));
                  }
                }
                output.push([stepKey++, eu.right]);
              } else {
                const nk = stepKey++;
                const index = i;
                if (!queue) {
                  queue = [];
                }
                queue.push(({
                  es,
                  output
                }) => Effect.flatMap(Effect.either(te), t => {
                  if (Either.isLeft(t)) {
                    // the input element is present but is not valid
                    const e = ParseResult.index(index, t.left.errors);
                    if (allErrors) {
                      es.push([nk, e]);
                      return Effect.unit;
                    } else {
                      return ParseResult.fail(mutableAppend(sortByIndex(es), e));
                    }
                  }
                  output.push([nk, t.right]);
                  return Effect.unit;
                }));
              }
            }
          }
          // ---------------------------------------------
          // handle rest element
          // ---------------------------------------------
          if (Option.isSome(rest)) {
            const [head, ...tail] = rest.value;
            for (; i < len - tail.length; i++) {
              const te = head(input[i], options);
              const eu = ParseResult.eitherOrUndefined(te);
              if (eu) {
                if (Either.isLeft(eu)) {
                  const e = ParseResult.index(i, eu.left.errors);
                  if (allErrors) {
                    es.push([stepKey++, e]);
                    continue;
                  } else {
                    return ParseResult.fail(mutableAppend(sortByIndex(es), e));
                  }
                } else {
                  output.push([stepKey++, eu.right]);
                }
              } else {
                const nk = stepKey++;
                const index = i;
                if (!queue) {
                  queue = [];
                }
                queue.push(({
                  es,
                  output
                }) => Effect.flatMap(Effect.either(te), t => {
                  if (Either.isLeft(t)) {
                    const e = ParseResult.index(index, t.left.errors);
                    if (allErrors) {
                      es.push([nk, e]);
                      return Effect.unit;
                    } else {
                      return ParseResult.fail(mutableAppend(sortByIndex(es), e));
                    }
                  } else {
                    output.push([nk, t.right]);
                    return Effect.unit;
                  }
                }));
              }
            }
            // ---------------------------------------------
            // handle post rest elements
            // ---------------------------------------------
            for (let j = 0; j < tail.length; j++) {
              i += j;
              if (len < i + 1) {
                continue;
              } else {
                const te = tail[j](input[i], options);
                const eu = ParseResult.eitherOrUndefined(te);
                if (eu) {
                  if (Either.isLeft(eu)) {
                    // the input element is present but is not valid
                    const e = ParseResult.index(i, eu.left.errors);
                    if (allErrors) {
                      es.push([stepKey++, e]);
                      continue;
                    } else {
                      return ParseResult.fail(mutableAppend(sortByIndex(es), e));
                    }
                  }
                  output.push([stepKey++, eu.right]);
                } else {
                  const nk = stepKey++;
                  const index = i;
                  if (!queue) {
                    queue = [];
                  }
                  queue.push(({
                    es,
                    output
                  }) => Effect.flatMap(Effect.either(te), t => {
                    if (Either.isLeft(t)) {
                      // the input element is present but is not valid
                      const e = ParseResult.index(index, t.left.errors);
                      if (allErrors) {
                        es.push([nk, e]);
                        return Effect.unit;
                      } else {
                        return ParseResult.fail(mutableAppend(sortByIndex(es), e));
                      }
                    }
                    output.push([nk, t.right]);
                    return Effect.unit;
                  }));
                }
              }
            }
          }
          // ---------------------------------------------
          // compute output
          // ---------------------------------------------
          const computeResult = ({
            es,
            output
          }) => ReadonlyArray.isNonEmptyArray(es) ? ParseResult.fail(sortByIndex(es)) : ParseResult.succeed(sortByIndex(output));
          if (queue && queue.length > 0) {
            const cqueue = queue;
            return Effect.suspend(() => {
              const state = {
                es: Array.from(es),
                output: Array.from(output)
              };
              return Effect.flatMap(Effect.forEach(cqueue, f => f(state), {
                concurrency: "unbounded",
                discard: true
              }), () => computeResult(state));
            });
          }
          return computeResult({
            output,
            es
          });
        };
      }
    case "TypeLiteral":
      {
        if (ast.propertySignatures.length === 0 && ast.indexSignatures.length === 0) {
          return fromRefinement(ast, Predicate.isNotNullable);
        }
        const propertySignatures = [];
        const expectedKeys = {};
        for (const ps of ast.propertySignatures) {
          propertySignatures.push(goMemo(ps.type, isDecoding));
          expectedKeys[ps.name] = null;
        }
        const indexSignatures = ast.indexSignatures.map(is => [goMemo(is.parameter, isDecoding), goMemo(is.type, isDecoding)]);
        const expectedAST = AST.createUnion(ast.indexSignatures.map(is => is.parameter).concat(Internal.ownKeys(expectedKeys).map(key => Predicate.isSymbol(key) ? AST.createUniqueSymbol(key) : AST.createLiteral(key))));
        const expected = goMemo(expectedAST, isDecoding);
        return (input, options) => {
          if (!Predicate.isRecord(input)) {
            return ParseResult.fail(ParseResult.type(ast, input));
          }
          const allErrors = options?.errors === "all";
          const es = [];
          let stepKey = 0;
          // ---------------------------------------------
          // handle excess properties
          // ---------------------------------------------
          const onExcessPropertyError = options?.onExcessProperty === "error";
          if (onExcessPropertyError) {
            for (const key of Internal.ownKeys(input)) {
              const eu = ParseResult.eitherOrUndefined(expected(key, options));
              if (eu && Either.isLeft(eu)) {
                const e = ParseResult.key(key, [ParseResult.unexpected(Option.some(expectedAST))]);
                if (allErrors) {
                  es.push([stepKey++, e]);
                  continue;
                } else {
                  return ParseResult.fail(mutableAppend(sortByIndex(es), e));
                }
              }
            }
          }
          // ---------------------------------------------
          // handle property signatures
          // ---------------------------------------------
          const output = {};
          let queue = undefined;
          for (let i = 0; i < propertySignatures.length; i++) {
            const ps = ast.propertySignatures[i];
            const parser = propertySignatures[i];
            const name = ps.name;
            if (Object.prototype.hasOwnProperty.call(input, name)) {
              const te = parser(input[name], options);
              const eu = ParseResult.eitherOrUndefined(te);
              if (eu) {
                if (Either.isLeft(eu)) {
                  // the input key is present but is not valid
                  const e = ParseResult.key(name, eu.left.errors);
                  if (allErrors) {
                    es.push([stepKey++, e]);
                    continue;
                  } else {
                    return ParseResult.fail(mutableAppend(sortByIndex(es), e));
                  }
                }
                output[name] = eu.right;
              } else {
                const nk = stepKey++;
                const index = name;
                if (!queue) {
                  queue = [];
                }
                queue.push(({
                  es,
                  output
                }) => Effect.flatMap(Effect.either(te), t => {
                  if (Either.isLeft(t)) {
                    // the input key is present but is not valid
                    const e = ParseResult.key(index, t.left.errors);
                    if (allErrors) {
                      es.push([nk, e]);
                      return Effect.unit;
                    } else {
                      return ParseResult.fail(mutableAppend(sortByIndex(es), e));
                    }
                  }
                  output[index] = t.right;
                  return Effect.unit;
                }));
              }
            } else {
              // ---------------------------------------------
              // handle missing keys
              // ---------------------------------------------
              if (!ps.isOptional) {
                const e = ParseResult.key(name, [ParseResult.missing]);
                if (allErrors) {
                  es.push([stepKey++, e]);
                  continue;
                } else {
                  return ParseResult.fail(e);
                }
              }
            }
          }
          // ---------------------------------------------
          // handle index signatures
          // ---------------------------------------------
          for (let i = 0; i < indexSignatures.length; i++) {
            const indexSignature = indexSignatures[i];
            const parameter = indexSignature[0];
            const type = indexSignature[1];
            const keys = Internal.getKeysForIndexSignature(input, ast.indexSignatures[i].parameter);
            for (const key of keys) {
              // ---------------------------------------------
              // handle keys
              // ---------------------------------------------
              const keu = ParseResult.eitherOrUndefined(parameter(key, options));
              if (keu && Either.isRight(keu)) {
                // ---------------------------------------------
                // handle values
                // ---------------------------------------------
                const vpr = type(input[key], options);
                const veu = ParseResult.eitherOrUndefined(vpr);
                if (veu) {
                  if (Either.isLeft(veu)) {
                    const e = ParseResult.key(key, veu.left.errors);
                    if (allErrors) {
                      es.push([stepKey++, e]);
                      continue;
                    } else {
                      return ParseResult.fail(mutableAppend(sortByIndex(es), e));
                    }
                  } else {
                    if (!Object.prototype.hasOwnProperty.call(expectedKeys, key)) {
                      output[key] = veu.right;
                    }
                  }
                } else {
                  const nk = stepKey++;
                  const index = key;
                  if (!queue) {
                    queue = [];
                  }
                  queue.push(({
                    es,
                    output
                  }) => Effect.flatMap(Effect.either(vpr), tv => {
                    if (Either.isLeft(tv)) {
                      const e = ParseResult.key(index, tv.left.errors);
                      if (allErrors) {
                        es.push([nk, e]);
                        return Effect.unit;
                      } else {
                        return ParseResult.fail(mutableAppend(sortByIndex(es), e));
                      }
                    } else {
                      if (!Object.prototype.hasOwnProperty.call(expectedKeys, key)) {
                        output[key] = tv.right;
                      }
                      return Effect.unit;
                    }
                  }));
                }
              }
            }
          }
          // ---------------------------------------------
          // compute output
          // ---------------------------------------------
          const computeResult = ({
            es,
            output
          }) => ReadonlyArray.isNonEmptyArray(es) ? ParseResult.fail(sortByIndex(es)) : ParseResult.succeed(output);
          if (queue && queue.length > 0) {
            const cqueue = queue;
            return Effect.suspend(() => {
              const state = {
                es: Array.from(es),
                output: Object.assign({}, output)
              };
              return Effect.flatMap(Effect.forEach(cqueue, f => f(state), {
                concurrency: "unbounded",
                discard: true
              }), () => computeResult(state));
            });
          }
          return computeResult({
            es,
            output
          });
        };
      }
    case "Union":
      {
        const searchTree = getSearchTree(ast.types, isDecoding);
        const ownKeys = Internal.ownKeys(searchTree.keys);
        const len = ownKeys.length;
        const map = new Map();
        for (let i = 0; i < ast.types.length; i++) {
          map.set(ast.types[i], goMemo(ast.types[i], isDecoding));
        }
        return (input, options) => {
          const es = [];
          let stepKey = 0;
          let candidates = [];
          if (len > 0) {
            // if there is at least one key then input must be an object
            if (Predicate.isRecord(input)) {
              for (let i = 0; i < len; i++) {
                const name = ownKeys[i];
                const buckets = searchTree.keys[name].buckets;
                // for each property that should contain a literal, check if the input contains that property
                if (Object.prototype.hasOwnProperty.call(input, name)) {
                  const literal = String(input[name]);
                  // check that the value obtained from the input for the property corresponds to an existing bucket
                  if (Object.prototype.hasOwnProperty.call(buckets, literal)) {
                    // retrive the minimal set of candidates for decoding
                    candidates = candidates.concat(buckets[literal]);
                  } else {
                    es.push([stepKey++, ParseResult.key(name, [ParseResult.type(searchTree.keys[name].ast, input[name])])]);
                  }
                } else {
                  es.push([stepKey++, ParseResult.key(name, [ParseResult.missing])]);
                }
              }
            } else {
              es.push([stepKey++, ParseResult.type(ast, input)]);
            }
          }
          if (searchTree.otherwise.length > 0) {
            candidates = candidates.concat(searchTree.otherwise);
          }
          let queue = undefined;
          for (let i = 0; i < candidates.length; i++) {
            const pr = map.get(candidates[i])(input, options);
            // the members of a union are ordered based on which one should be decoded first,
            // therefore if one member has added a task, all subsequent members must
            // also add a task to the queue even if they are synchronous
            const eu = !queue || queue.length === 0 ? ParseResult.eitherOrUndefined(pr) : undefined;
            if (eu) {
              if (Either.isRight(eu)) {
                return ParseResult.succeed(eu.right);
              } else {
                es.push([stepKey++, ParseResult.unionMember(eu.left.errors)]);
              }
            } else {
              const nk = stepKey++;
              if (!queue) {
                queue = [];
              }
              queue.push(state => Effect.suspend(() => {
                if ("finalResult" in state) {
                  return Effect.unit;
                } else {
                  return Effect.flatMap(Effect.either(pr), t => {
                    if (Either.isRight(t)) {
                      state.finalResult = ParseResult.succeed(t.right);
                    } else {
                      state.es.push([nk, ParseResult.unionMember(t.left.errors)]);
                    }
                    return Effect.unit;
                  });
                }
              }));
            }
          }
          // ---------------------------------------------
          // compute output
          // ---------------------------------------------
          const computeResult = es => ReadonlyArray.isNonEmptyArray(es) ? ParseResult.fail(sortByIndex(es)) :
          // this should never happen
          ParseResult.fail(ParseResult.type(AST.neverKeyword, input));
          if (queue && queue.length > 0) {
            const cqueue = queue;
            return Effect.suspend(() => {
              const state = {
                es: Array.from(es)
              };
              return Effect.flatMap(Effect.forEach(cqueue, f => f(state), {
                concurrency: 1,
                discard: true
              }), () => {
                if ("finalResult" in state) {
                  return state.finalResult;
                }
                return computeResult(state.es);
              });
            });
          }
          return computeResult(es);
        };
      }
    case "Suspend":
      {
        const get = Internal.memoizeThunk(() => goMemo(ast.f(), isDecoding));
        return (a, options) => get()(a, options);
      }
  }
};
const fromRefinement = (ast, refinement) => u => refinement(u) ? ParseResult.succeed(u) : ParseResult.fail(ParseResult.type(ast, u));
/** @internal */
const getLiterals = (ast, isDecoding) => {
  switch (ast._tag) {
    case "Declaration":
      return getLiterals(ast.type, isDecoding);
    case "TypeLiteral":
      {
        const out = [];
        for (let i = 0; i < ast.propertySignatures.length; i++) {
          const propertySignature = ast.propertySignatures[i];
          const type = isDecoding ? AST.from(propertySignature.type) : AST.to(propertySignature.type);
          if (AST.isLiteral(type) && !propertySignature.isOptional) {
            out.push([propertySignature.name, type]);
          }
        }
        return out;
      }
    case "Refinement":
      return getLiterals(ast.from, isDecoding);
    case "Transform":
      return getLiterals(isDecoding ? ast.from : ast.to, isDecoding);
  }
  return [];
};
/**
 * The purpose of the algorithm is to narrow down the pool of possible candidates for decoding as much as possible.
 *
 * This function separates the schemas into two groups, `keys` and `otherwise`:
 *
 * - `keys`: the schema has at least one property with a literal value
 * - `otherwise`: the schema has no properties with a literal value
 *
 * If a schema has at least one property with a literal value, so it ends up in `keys`, first a namespace is created for
 * the name of the property containing the literal, and then within this namespace a "bucket" is created for the literal
 * value in which to store all the schemas that have the same property and literal value.
 *
 * @internal
 */
exports.getLiterals = getLiterals;
const getSearchTree = (members, isDecoding) => {
  const keys = {};
  const otherwise = [];
  for (let i = 0; i < members.length; i++) {
    const member = members[i];
    const tags = getLiterals(member, isDecoding);
    if (tags.length > 0) {
      for (let j = 0; j < tags.length; j++) {
        const [key, literal] = tags[j];
        const hash = String(literal.literal);
        keys[key] = keys[key] || {
          buckets: {},
          ast: AST.neverKeyword
        };
        const buckets = keys[key].buckets;
        if (Object.prototype.hasOwnProperty.call(buckets, hash)) {
          if (j < tags.length - 1) {
            continue;
          }
          buckets[hash].push(member);
          keys[key].ast = AST.createUnion([keys[key].ast, literal]);
        } else {
          buckets[hash] = [member];
          keys[key].ast = AST.createUnion([keys[key].ast, literal]);
          break;
        }
      }
    } else {
      otherwise.push(member);
    }
  }
  return {
    keys,
    otherwise
  };
};
exports.getSearchTree = getSearchTree;
const dropRightRefinement = ast => AST.isRefinement(ast) ? dropRightRefinement(ast.from) : ast;
const handleForbidden = (conditional, options) => {
  const eu = ParseResult.eitherOrUndefined(conditional);
  return eu ? eu : options?.isEffectAllowed === true ? conditional : ParseResult.fail(ParseResult.forbidden);
};
const mutableAppend = (self, a) => {
  self.push(a);
  return self;
};
/** @internal */
const getTemplateLiteralRegex = ast => {
  let pattern = `^${ast.head}`;
  for (const span of ast.spans) {
    if (AST.isStringKeyword(span.type)) {
      pattern += ".*";
    } else if (AST.isNumberKeyword(span.type)) {
      pattern += "[+-]?\\d*\\.?\\d+(?:[Ee][+-]?\\d+)?";
    }
    pattern += span.literal;
  }
  pattern += "$";
  return new RegExp(pattern);
};
exports.getTemplateLiteralRegex = getTemplateLiteralRegex;
function sortByIndex(es) {
  return es.sort(([a], [b]) => a > b ? 1 : a < b ? -1 : 0).map(([_, a]) => a);
}
// -------------------------------------------------------------------------------------
// transformations interpreter
// -------------------------------------------------------------------------------------
const getFinalPropertySignatureTransformation = (transformation, isDecoding) => {
  switch (transformation._tag) {
    case "FinalPropertySignatureTransformation":
      return isDecoding ? transformation.decode : transformation.encode;
  }
};
/** @internal */
const getFinalTransformation = (transformation, isDecoding) => {
  switch (transformation._tag) {
    case "FinalTransformation":
      return isDecoding ? transformation.decode : transformation.encode;
    case "ComposeTransformation":
      return ParseResult.succeed;
    case "TypeLiteralTransformation":
      return input => {
        let out = Either.right(input);
        // ---------------------------------------------
        // handle property signature transformations
        // ---------------------------------------------
        for (const pst of transformation.propertySignatureTransformations) {
          const [from, to] = isDecoding ? [pst.from, pst.to] : [pst.to, pst.from];
          const transform = getFinalPropertySignatureTransformation(pst.propertySignatureTransformation, isDecoding);
          const f = input => {
            const o = transform(Object.prototype.hasOwnProperty.call(input, from) ? Option.some(input[from]) : Option.none());
            delete input[from];
            if (Option.isSome(o)) {
              input[to] = o.value;
            }
            return input;
          };
          out = ParseResult.map(out, f);
        }
        return out;
      };
  }
};
exports.getFinalTransformation = getFinalTransformation;
//# sourceMappingURL=Parser.js.map