var test = (function (exports) {
  'use strict';

  /**
   * Utility module to work with key-value stores.
   *
   * @module map
   */

  /**
   * Creates a new Map instance.
   *
   * @function
   * @return {Map<any, any>}
   *
   * @function
   */
  const create$5 = () => new Map();

  /**
   * Copy a Map object into a fresh Map object.
   *
   * @function
   * @template X,Y
   * @param {Map<X,Y>} m
   * @return {Map<X,Y>}
   */
  const copy = m => {
    const r = create$5();
    m.forEach((v, k) => { r.set(k, v); });
    return r
  };

  /**
   * Get map property. Create T if property is undefined and set T on map.
   *
   * ```js
   * const listeners = map.setIfUndefined(events, 'eventName', set.create)
   * listeners.add(listener)
   * ```
   *
   * @function
   * @template T,K
   * @param {Map<K, T>} map
   * @param {K} key
   * @param {function():T} createT
   * @return {T}
   */
  const setIfUndefined = (map, key, createT) => {
    let set = map.get(key);
    if (set === undefined) {
      map.set(key, set = createT());
    }
    return set
  };

  /**
   * Creates an Array and populates it with the content of all key-value pairs using the `f(value, key)` function.
   *
   * @function
   * @template K
   * @template V
   * @template R
   * @param {Map<K,V>} m
   * @param {function(V,K):R} f
   * @return {Array<R>}
   */
  const map = (m, f) => {
    const res = [];
    for (const [key, value] of m) {
      res.push(f(value, key));
    }
    return res
  };

  /**
   * Tests whether any key-value pairs pass the test implemented by `f(value, key)`.
   *
   * @todo should rename to some - similarly to Array.some
   *
   * @function
   * @template K
   * @template V
   * @param {Map<K,V>} m
   * @param {function(V,K):boolean} f
   * @return {boolean}
   */
  const any = (m, f) => {
    for (const [key, value] of m) {
      if (f(value, key)) {
        return true
      }
    }
    return false
  };

  /**
   * Utility module to work with sets.
   *
   * @module set
   */

  const create$4 = () => new Set();

  /**
   * Utility module to work with Arrays.
   *
   * @module array
   */

  /**
   * Return the last element of an array. The element must exist
   *
   * @template L
   * @param {Array<L>} arr
   * @return {L}
   */
  const last = arr => arr[arr.length - 1];

  /**
   * Append elements from src to dest
   *
   * @template M
   * @param {Array<M>} dest
   * @param {Array<M>} src
   */
  const appendTo = (dest, src) => {
    for (let i = 0; i < src.length; i++) {
      dest.push(src[i]);
    }
  };

  /**
   * Transforms something array-like to an actual Array.
   *
   * @function
   * @template T
   * @param {ArrayLike<T>|Iterable<T>} arraylike
   * @return {T}
   */
  const from = Array.from;

  const isArray = Array.isArray;

  /**
   * Observable class prototype.
   *
   * @module observable
   */

  /**
   * Handles named events.
   *
   * @template N
   */
  class Observable {
    constructor () {
      /**
       * Some desc.
       * @type {Map<N, any>}
       */
      this._observers = create$5();
    }

    /**
     * @param {N} name
     * @param {function} f
     */
    on (name, f) {
      setIfUndefined(this._observers, name, create$4).add(f);
    }

    /**
     * @param {N} name
     * @param {function} f
     */
    once (name, f) {
      /**
       * @param  {...any} args
       */
      const _f = (...args) => {
        this.off(name, _f);
        f(...args);
      };
      this.on(name, _f);
    }

    /**
     * @param {N} name
     * @param {function} f
     */
    off (name, f) {
      const observers = this._observers.get(name);
      if (observers !== undefined) {
        observers.delete(f);
        if (observers.size === 0) {
          this._observers.delete(name);
        }
      }
    }

    /**
     * Emit a named event. All registered event listeners that listen to the
     * specified name will receive the event.
     *
     * @todo This should catch exceptions
     *
     * @param {N} name The event name.
     * @param {Array<any>} args The arguments that are applied to the event listener.
     */
    emit (name, args) {
      // copy all listeners to an array first to make sure that no event is emitted to listeners that are subscribed while the event handler is called.
      return from((this._observers.get(name) || create$5()).values()).forEach(f => f(...args))
    }

    destroy () {
      this._observers = create$5();
    }
  }

  /**
   * Common Math expressions.
   *
   * @module math
   */

  const floor = Math.floor;
  const abs = Math.abs;
  const log10 = Math.log10;

  /**
   * @function
   * @param {number} a
   * @param {number} b
   * @return {number} The smaller element of a and b
   */
  const min = (a, b) => a < b ? a : b;

  /**
   * @function
   * @param {number} a
   * @param {number} b
   * @return {number} The bigger element of a and b
   */
  const max = (a, b) => a > b ? a : b;

  /**
   * @param {number} n
   * @return {boolean} Wether n is negative. This function also differentiates between -0 and +0
   */
  const isNegativeZero = n => n !== 0 ? n < 0 : 1 / n < 0;

  /**
   * Utility module to work with strings.
   *
   * @module string
   */

  const fromCharCode = String.fromCharCode;

  /**
   * @param {string} s
   * @return {string}
   */
  const toLowerCase = s => s.toLowerCase();

  const trimLeftRegex = /^\s*/g;

  /**
   * @param {string} s
   * @return {string}
   */
  const trimLeft = s => s.replace(trimLeftRegex, '');

  const fromCamelCaseRegex = /([A-Z])/g;

  /**
   * @param {string} s
   * @param {string} separator
   * @return {string}
   */
  const fromCamelCase = (s, separator) => trimLeft(s.replace(fromCamelCaseRegex, match => `${separator}${toLowerCase(match)}`));

  /**
   * @param {string} str
   * @return {Uint8Array}
   */
  const _encodeUtf8Polyfill = str => {
    const encodedString = unescape(encodeURIComponent(str));
    const len = encodedString.length;
    const buf = new Uint8Array(len);
    for (let i = 0; i < len; i++) {
      buf[i] = /** @type {number} */ (encodedString.codePointAt(i));
    }
    return buf
  };

  /* istanbul ignore next */
  const utf8TextEncoder = /** @type {TextEncoder} */ (typeof TextEncoder !== 'undefined' ? new TextEncoder() : null);

  /**
   * @param {string} str
   * @return {Uint8Array}
   */
  const _encodeUtf8Native = str => utf8TextEncoder.encode(str);

  /**
   * @param {string} str
   * @return {Uint8Array}
   */
  /* istanbul ignore next */
  const encodeUtf8 = utf8TextEncoder ? _encodeUtf8Native : _encodeUtf8Polyfill;

  /* istanbul ignore next */
  let utf8TextDecoder = typeof TextDecoder === 'undefined' ? null : new TextDecoder('utf-8', { fatal: true, ignoreBOM: true });

  /* istanbul ignore next */
  if (utf8TextDecoder && utf8TextDecoder.decode(new Uint8Array()).length === 1) {
    // Safari doesn't handle BOM correctly.
    // This fixes a bug in Safari 13.0.5 where it produces a BOM the first time it is called.
    // utf8TextDecoder.decode(new Uint8Array()).length === 1 on the first call and
    // utf8TextDecoder.decode(new Uint8Array()).length === 1 on the second call
    // Another issue is that from then on no BOM chars are recognized anymore
    /* istanbul ignore next */
    utf8TextDecoder = null;
  }

  /**
   * Often used conditions.
   *
   * @module conditions
   */

  /**
   * @template T
   * @param {T|null|undefined} v
   * @return {T|null}
   */
  /* istanbul ignore next */
  const undefinedToNull = v => v === undefined ? null : v;

  /* global localStorage, addEventListener */

  /**
   * Isomorphic variable storage.
   *
   * Uses LocalStorage in the browser and falls back to in-memory storage.
   *
   * @module storage
   */

  /* istanbul ignore next */
  class VarStoragePolyfill {
    constructor () {
      this.map = new Map();
    }

    /**
     * @param {string} key
     * @param {any} newValue
     */
    setItem (key, newValue) {
      this.map.set(key, newValue);
    }

    /**
     * @param {string} key
     */
    getItem (key) {
      return this.map.get(key)
    }
  }

  /* istanbul ignore next */
  /**
   * @type {any}
   */
  let _localStorage = new VarStoragePolyfill();
  let usePolyfill = true;

  try {
    // if the same-origin rule is violated, accessing localStorage might thrown an error
    /* istanbul ignore next */
    if (typeof localStorage !== 'undefined') {
      _localStorage = localStorage;
      usePolyfill = false;
    }
  } catch (e) { }

  /* istanbul ignore next */
  /**
   * This is basically localStorage in browser, or a polyfill in nodejs
   */
  const varStorage = _localStorage;

  /* istanbul ignore next */
  /**
   * A polyfill for `addEventListener('storage', event => {..})` that does nothing if the polyfill is being used.
   *
   * @param {function({ key: string, newValue: string, oldValue: string }): void} eventHandler
   * @function
   */
  const onChange = eventHandler => usePolyfill || addEventListener('storage', /** @type {any} */ (eventHandler));

  /**
   * Isomorphic module to work access the environment (query params, env variables).
   *
   * @module map
   */

  /* istanbul ignore next */
  // @ts-ignore
  const isNode = typeof process !== 'undefined' && process.release && /node|io\.js/.test(process.release.name);
  /* istanbul ignore next */
  const isBrowser = typeof window !== 'undefined' && !isNode;
  /* istanbul ignore next */
  typeof navigator !== 'undefined' ? /Mac/.test(navigator.platform) : false;

  /**
   * @type {Map<string,string>}
   */
  let params;

  /* istanbul ignore next */
  const computeParams = () => {
    if (params === undefined) {
      if (isNode) {
        params = create$5();
        const pargs = process.argv;
        let currParamName = null;
        /* istanbul ignore next */
        for (let i = 0; i < pargs.length; i++) {
          const parg = pargs[i];
          if (parg[0] === '-') {
            if (currParamName !== null) {
              params.set(currParamName, '');
            }
            currParamName = parg;
          } else {
            if (currParamName !== null) {
              params.set(currParamName, parg);
              currParamName = null;
            }
          }
        }
        if (currParamName !== null) {
          params.set(currParamName, '');
        }
      // in ReactNative for example this would not be true (unless connected to the Remote Debugger)
      } else if (typeof location === 'object') {
        params = create$5()
        // eslint-disable-next-line no-undef
        ;(location.search || '?').slice(1).split('&').forEach(kv => {
          if (kv.length !== 0) {
            const [key, value] = kv.split('=');
            params.set(`--${fromCamelCase(key, '-')}`, value);
            params.set(`-${fromCamelCase(key, '-')}`, value);
          }
        });
      } else {
        params = create$5();
      }
    }
    return params
  };

  /**
   * @param {string} name
   * @return {boolean}
   */
  /* istanbul ignore next */
  const hasParam = name => computeParams().has(name);
  // export const getArgs = name => computeParams() && args

  /**
   * @param {string} name
   * @return {string|null}
   */
  /* istanbul ignore next */
  const getVariable = name => isNode ? undefinedToNull(process.env[name.toUpperCase()]) : undefinedToNull(varStorage.getItem(name));

  /**
   * @param {string} name
   * @return {boolean}
   */
  /* istanbul ignore next */
  const hasConf = name => hasParam('--' + name) || getVariable(name) !== null;

  /* istanbul ignore next */
  hasConf('production');

  /* eslint-env browser */

  /**
   * Binary data constants.
   *
   * @module binary
   */

  /**
   * n-th bit activated.
   *
   * @type {number}
   */
  const BIT1 = 1;
  const BIT2 = 2;
  const BIT3 = 4;
  const BIT4 = 8;
  const BIT6 = 32;
  const BIT7 = 64;
  const BIT8 = 128;
  const BITS5 = 31;
  const BITS6 = 63;
  const BITS7 = 127;
  /**
   * @type {number}
   */
  const BITS31 = 0x7FFFFFFF;

  /**
   * Efficient schema-less binary decoding with support for variable length encoding.
   *
   * Use [lib0/decoding] with [lib0/encoding]. Every encoding function has a corresponding decoding function.
   *
   * Encodes numbers in little-endian order (least to most significant byte order)
   * and is compatible with Golang's binary encoding (https://golang.org/pkg/encoding/binary/)
   * which is also used in Protocol Buffers.
   *
   * ```js
   * // encoding step
   * const encoder = new encoding.createEncoder()
   * encoding.writeVarUint(encoder, 256)
   * encoding.writeVarString(encoder, 'Hello world!')
   * const buf = encoding.toUint8Array(encoder)
   * ```
   *
   * ```js
   * // decoding step
   * const decoder = new decoding.createDecoder(buf)
   * decoding.readVarUint(decoder) // => 256
   * decoding.readVarString(decoder) // => 'Hello world!'
   * decoding.hasContent(decoder) // => false - all data is read
   * ```
   *
   * @module decoding
   */

  /**
   * A Decoder handles the decoding of an Uint8Array.
   */
  class Decoder {
    /**
     * @param {Uint8Array} uint8Array Binary data to decode
     */
    constructor (uint8Array) {
      /**
       * Decoding target.
       *
       * @type {Uint8Array}
       */
      this.arr = uint8Array;
      /**
       * Current decoding position.
       *
       * @type {number}
       */
      this.pos = 0;
    }
  }

  /**
   * @function
   * @param {Uint8Array} uint8Array
   * @return {Decoder}
   */
  const createDecoder = uint8Array => new Decoder(uint8Array);

  /**
   * @function
   * @param {Decoder} decoder
   * @return {boolean}
   */
  const hasContent = decoder => decoder.pos !== decoder.arr.length;

  /**
   * Create an Uint8Array view of the next `len` bytes and advance the position by `len`.
   *
   * Important: The Uint8Array still points to the underlying ArrayBuffer. Make sure to discard the result as soon as possible to prevent any memory leaks.
   *            Use `buffer.copyUint8Array` to copy the result into a new Uint8Array.
   *
   * @function
   * @param {Decoder} decoder The decoder instance
   * @param {number} len The length of bytes to read
   * @return {Uint8Array}
   */
  const readUint8Array = (decoder, len) => {
    const view = createUint8ArrayViewFromArrayBuffer(decoder.arr.buffer, decoder.pos + decoder.arr.byteOffset, len);
    decoder.pos += len;
    return view
  };

  /**
   * Read variable length Uint8Array.
   *
   * Important: The Uint8Array still points to the underlying ArrayBuffer. Make sure to discard the result as soon as possible to prevent any memory leaks.
   *            Use `buffer.copyUint8Array` to copy the result into a new Uint8Array.
   *
   * @function
   * @param {Decoder} decoder
   * @return {Uint8Array}
   */
  const readVarUint8Array = decoder => readUint8Array(decoder, readVarUint(decoder));

  /**
   * Read one byte as unsigned integer.
   * @function
   * @param {Decoder} decoder The decoder instance
   * @return {number} Unsigned 8-bit integer
   */
  const readUint8 = decoder => decoder.arr[decoder.pos++];

  /**
   * Read unsigned integer (32bit) with variable length.
   * 1/8th of the storage is used as encoding overhead.
   *  * numbers < 2^7 is stored in one bytlength
   *  * numbers < 2^14 is stored in two bylength
   *
   * @function
   * @param {Decoder} decoder
   * @return {number} An unsigned integer.length
   */
  const readVarUint = decoder => {
    let num = 0;
    let len = 0;
    while (true) {
      const r = decoder.arr[decoder.pos++];
      num = num | ((r & BITS7) << len);
      len += 7;
      if (r < BIT8) {
        return num >>> 0 // return unsigned number!
      }
      /* istanbul ignore if */
      if (len > 53) {
        throw new Error('Integer out of range!')
      }
    }
  };

  /**
   * Read signed integer (32bit) with variable length.
   * 1/8th of the storage is used as encoding overhead.
   *  * numbers < 2^7 is stored in one bytlength
   *  * numbers < 2^14 is stored in two bylength
   * @todo This should probably create the inverse ~num if number is negative - but this would be a breaking change.
   *
   * @function
   * @param {Decoder} decoder
   * @return {number} An unsigned integer.length
   */
  const readVarInt = decoder => {
    let r = decoder.arr[decoder.pos++];
    let num = r & BITS6;
    let len = 6;
    const sign = (r & BIT7) > 0 ? -1 : 1;
    if ((r & BIT8) === 0) {
      // don't continue reading
      return sign * num
    }
    while (true) {
      r = decoder.arr[decoder.pos++];
      num = num | ((r & BITS7) << len);
      len += 7;
      if (r < BIT8) {
        return sign * (num >>> 0)
      }
      /* istanbul ignore if */
      if (len > 53) {
        throw new Error('Integer out of range!')
      }
    }
  };

  /**
   * Read string of variable length
   * * varUint is used to store the length of the string
   *
   * Transforming utf8 to a string is pretty expensive. The code performs 10x better
   * when String.fromCodePoint is fed with all characters as arguments.
   * But most environments have a maximum number of arguments per functions.
   * For effiency reasons we apply a maximum of 10000 characters at once.
   *
   * @function
   * @param {Decoder} decoder
   * @return {String} The read String.
   */
  const readVarString = decoder => {
    let remainingLen = readVarUint(decoder);
    if (remainingLen === 0) {
      return ''
    } else {
      let encodedString = String.fromCodePoint(readUint8(decoder)); // remember to decrease remainingLen
      if (--remainingLen < 100) { // do not create a Uint8Array for small strings
        while (remainingLen--) {
          encodedString += String.fromCodePoint(readUint8(decoder));
        }
      } else {
        while (remainingLen > 0) {
          const nextLen = remainingLen < 10000 ? remainingLen : 10000;
          // this is dangerous, we create a fresh array view from the existing buffer
          const bytes = decoder.arr.subarray(decoder.pos, decoder.pos + nextLen);
          decoder.pos += nextLen;
          // Starting with ES5.1 we can supply a generic array-like object as arguments
          encodedString += String.fromCodePoint.apply(null, /** @type {any} */ (bytes));
          remainingLen -= nextLen;
        }
      }
      return decodeURIComponent(escape(encodedString))
    }
  };

  /**
   * @param {Decoder} decoder
   * @param {number} len
   * @return {DataView}
   */
  const readFromDataView = (decoder, len) => {
    const dv = new DataView(decoder.arr.buffer, decoder.arr.byteOffset + decoder.pos, len);
    decoder.pos += len;
    return dv
  };

  /**
   * @param {Decoder} decoder
   */
  const readFloat32 = decoder => readFromDataView(decoder, 4).getFloat32(0, false);

  /**
   * @param {Decoder} decoder
   */
  const readFloat64 = decoder => readFromDataView(decoder, 8).getFloat64(0, false);

  /**
   * @param {Decoder} decoder
   */
  const readBigInt64 = decoder => /** @type {any} */ (readFromDataView(decoder, 8)).getBigInt64(0, false);

  /**
   * @type {Array<function(Decoder):any>}
   */
  const readAnyLookupTable = [
    decoder => undefined, // CASE 127: undefined
    decoder => null, // CASE 126: null
    readVarInt, // CASE 125: integer
    readFloat32, // CASE 124: float32
    readFloat64, // CASE 123: float64
    readBigInt64, // CASE 122: bigint
    decoder => false, // CASE 121: boolean (false)
    decoder => true, // CASE 120: boolean (true)
    readVarString, // CASE 119: string
    decoder => { // CASE 118: object<string,any>
      const len = readVarUint(decoder);
      /**
       * @type {Object<string,any>}
       */
      const obj = {};
      for (let i = 0; i < len; i++) {
        const key = readVarString(decoder);
        obj[key] = readAny(decoder);
      }
      return obj
    },
    decoder => { // CASE 117: array<any>
      const len = readVarUint(decoder);
      const arr = [];
      for (let i = 0; i < len; i++) {
        arr.push(readAny(decoder));
      }
      return arr
    },
    readVarUint8Array // CASE 116: Uint8Array
  ];

  /**
   * @param {Decoder} decoder
   */
  const readAny = decoder => readAnyLookupTable[127 - readUint8(decoder)](decoder);

  /**
   * T must not be null.
   *
   * @template T
   */
  class RleDecoder extends Decoder {
    /**
     * @param {Uint8Array} uint8Array
     * @param {function(Decoder):T} reader
     */
    constructor (uint8Array, reader) {
      super(uint8Array);
      /**
       * The reader
       */
      this.reader = reader;
      /**
       * Current state
       * @type {T|null}
       */
      this.s = null;
      this.count = 0;
    }

    read () {
      if (this.count === 0) {
        this.s = this.reader(this);
        if (hasContent(this)) {
          this.count = readVarUint(this) + 1; // see encoder implementation for the reason why this is incremented
        } else {
          this.count = -1; // read the current value forever
        }
      }
      this.count--;
      return /** @type {T} */ (this.s)
    }
  }

  class UintOptRleDecoder extends Decoder {
    /**
     * @param {Uint8Array} uint8Array
     */
    constructor (uint8Array) {
      super(uint8Array);
      /**
       * @type {number}
       */
      this.s = 0;
      this.count = 0;
    }

    read () {
      if (this.count === 0) {
        this.s = readVarInt(this);
        // if the sign is negative, we read the count too, otherwise count is 1
        const isNegative = isNegativeZero(this.s);
        this.count = 1;
        if (isNegative) {
          this.s = -this.s;
          this.count = readVarUint(this) + 2;
        }
      }
      this.count--;
      return /** @type {number} */ (this.s)
    }
  }

  class IntDiffOptRleDecoder extends Decoder {
    /**
     * @param {Uint8Array} uint8Array
     */
    constructor (uint8Array) {
      super(uint8Array);
      /**
       * @type {number}
       */
      this.s = 0;
      this.count = 0;
      this.diff = 0;
    }

    /**
     * @return {number}
     */
    read () {
      if (this.count === 0) {
        const diff = readVarInt(this);
        // if the first bit is set, we read more data
        const hasCount = diff & 1;
        this.diff = diff >> 1;
        this.count = 1;
        if (hasCount) {
          this.count = readVarUint(this) + 2;
        }
      }
      this.s += this.diff;
      this.count--;
      return this.s
    }
  }

  class StringDecoder {
    /**
     * @param {Uint8Array} uint8Array
     */
    constructor (uint8Array) {
      this.decoder = new UintOptRleDecoder(uint8Array);
      this.str = readVarString(this.decoder);
      /**
       * @type {number}
       */
      this.spos = 0;
    }

    /**
     * @return {string}
     */
    read () {
      const end = this.spos + this.decoder.read();
      const res = this.str.slice(this.spos, end);
      this.spos = end;
      return res
    }
  }

  /**
   * Utility functions to work with buffers (Uint8Array).
   *
   * @module buffer
   */

  /**
   * @param {number} len
   */
  const createUint8ArrayFromLen = len => new Uint8Array(len);

  /**
   * Create Uint8Array with initial content from buffer
   *
   * @param {ArrayBuffer} buffer
   * @param {number} byteOffset
   * @param {number} length
   */
  const createUint8ArrayViewFromArrayBuffer = (buffer, byteOffset, length) => new Uint8Array(buffer, byteOffset, length);

  /**
   * Create Uint8Array with initial content from buffer
   *
   * @param {ArrayBuffer} buffer
   */
  const createUint8ArrayFromArrayBuffer = buffer => new Uint8Array(buffer);

  /* istanbul ignore next */
  /**
   * @param {Uint8Array} bytes
   * @return {string}
   */
  const toBase64Browser = bytes => {
    let s = '';
    for (let i = 0; i < bytes.byteLength; i++) {
      s += fromCharCode(bytes[i]);
    }
    // eslint-disable-next-line no-undef
    return btoa(s)
  };

  /**
   * @param {Uint8Array} bytes
   * @return {string}
   */
  const toBase64Node = bytes => Buffer.from(bytes.buffer, bytes.byteOffset, bytes.byteLength).toString('base64');

  /* istanbul ignore next */
  /**
   * @param {string} s
   * @return {Uint8Array}
   */
  const fromBase64Browser = s => {
    // eslint-disable-next-line no-undef
    const a = atob(s);
    const bytes = createUint8ArrayFromLen(a.length);
    for (let i = 0; i < a.length; i++) {
      bytes[i] = a.charCodeAt(i);
    }
    return bytes
  };

  /**
   * @param {string} s
   */
  const fromBase64Node = s => {
    const buf = Buffer.from(s, 'base64');
    return new Uint8Array(buf.buffer, buf.byteOffset, buf.byteLength)
  };

  /* istanbul ignore next */
  const toBase64 = isBrowser ? toBase64Browser : toBase64Node;

  /* istanbul ignore next */
  const fromBase64 = isBrowser ? fromBase64Browser : fromBase64Node;

  /**
   * Copy the content of an Uint8Array view to a new ArrayBuffer.
   *
   * @param {Uint8Array} uint8Array
   * @return {Uint8Array}
   */
  const copyUint8Array = uint8Array => {
    const newBuf = createUint8ArrayFromLen(uint8Array.byteLength);
    newBuf.set(uint8Array);
    return newBuf
  };

  /**
   * Utility helpers for working with numbers.
   *
   * @module number
   */

  /**
   * @module number
   */

  /* istanbul ignore next */
  const isInteger = Number.isInteger || (num => typeof num === 'number' && isFinite(num) && floor(num) === num);

  /**
   * Efficient schema-less binary encoding with support for variable length encoding.
   *
   * Use [lib0/encoding] with [lib0/decoding]. Every encoding function has a corresponding decoding function.
   *
   * Encodes numbers in little-endian order (least to most significant byte order)
   * and is compatible with Golang's binary encoding (https://golang.org/pkg/encoding/binary/)
   * which is also used in Protocol Buffers.
   *
   * ```js
   * // encoding step
   * const encoder = new encoding.createEncoder()
   * encoding.writeVarUint(encoder, 256)
   * encoding.writeVarString(encoder, 'Hello world!')
   * const buf = encoding.toUint8Array(encoder)
   * ```
   *
   * ```js
   * // decoding step
   * const decoder = new decoding.createDecoder(buf)
   * decoding.readVarUint(decoder) // => 256
   * decoding.readVarString(decoder) // => 'Hello world!'
   * decoding.hasContent(decoder) // => false - all data is read
   * ```
   *
   * @module encoding
   */

  /**
   * A BinaryEncoder handles the encoding to an Uint8Array.
   */
  class Encoder {
    constructor () {
      this.cpos = 0;
      this.cbuf = new Uint8Array(100);
      /**
       * @type {Array<Uint8Array>}
       */
      this.bufs = [];
    }
  }

  /**
   * @function
   * @return {Encoder}
   */
  const createEncoder = () => new Encoder();

  /**
   * The current length of the encoded data.
   *
   * @function
   * @param {Encoder} encoder
   * @return {number}
   */
  const length$1 = encoder => {
    let len = encoder.cpos;
    for (let i = 0; i < encoder.bufs.length; i++) {
      len += encoder.bufs[i].length;
    }
    return len
  };

  /**
   * Transform to Uint8Array.
   *
   * @function
   * @param {Encoder} encoder
   * @return {Uint8Array} The created ArrayBuffer.
   */
  const toUint8Array = encoder => {
    const uint8arr = new Uint8Array(length$1(encoder));
    let curPos = 0;
    for (let i = 0; i < encoder.bufs.length; i++) {
      const d = encoder.bufs[i];
      uint8arr.set(d, curPos);
      curPos += d.length;
    }
    uint8arr.set(createUint8ArrayViewFromArrayBuffer(encoder.cbuf.buffer, 0, encoder.cpos), curPos);
    return uint8arr
  };

  /**
   * Verify that it is possible to write `len` bytes wtihout checking. If
   * necessary, a new Buffer with the required length is attached.
   *
   * @param {Encoder} encoder
   * @param {number} len
   */
  const verifyLen = (encoder, len) => {
    const bufferLen = encoder.cbuf.length;
    if (bufferLen - encoder.cpos < len) {
      encoder.bufs.push(createUint8ArrayViewFromArrayBuffer(encoder.cbuf.buffer, 0, encoder.cpos));
      encoder.cbuf = new Uint8Array(max(bufferLen, len) * 2);
      encoder.cpos = 0;
    }
  };

  /**
   * Write one byte to the encoder.
   *
   * @function
   * @param {Encoder} encoder
   * @param {number} num The byte that is to be encoded.
   */
  const write = (encoder, num) => {
    const bufferLen = encoder.cbuf.length;
    if (encoder.cpos === bufferLen) {
      encoder.bufs.push(encoder.cbuf);
      encoder.cbuf = new Uint8Array(bufferLen * 2);
      encoder.cpos = 0;
    }
    encoder.cbuf[encoder.cpos++] = num;
  };

  /**
   * Write one byte as an unsigned integer.
   *
   * @function
   * @param {Encoder} encoder
   * @param {number} num The number that is to be encoded.
   */
  const writeUint8 = write;

  /**
   * Write a variable length unsigned integer.
   *
   * Encodes integers in the range from [0, 4294967295] / [0, 0xffffffff]. (max 32 bit unsigned integer)
   *
   * @function
   * @param {Encoder} encoder
   * @param {number} num The number that is to be encoded.
   */
  const writeVarUint = (encoder, num) => {
    while (num > BITS7) {
      write(encoder, BIT8 | (BITS7 & num));
      num >>>= 7;
    }
    write(encoder, BITS7 & num);
  };

  /**
   * Write a variable length integer.
   *
   * Encodes integers in the range from [-2147483648, -2147483647].
   *
   * We don't use zig-zag encoding because we want to keep the option open
   * to use the same function for BigInt and 53bit integers (doubles).
   *
   * We use the 7th bit instead for signaling that this is a negative number.
   *
   * @function
   * @param {Encoder} encoder
   * @param {number} num The number that is to be encoded.
   */
  const writeVarInt = (encoder, num) => {
    const isNegative = isNegativeZero(num);
    if (isNegative) {
      num = -num;
    }
    //             |- whether to continue reading         |- whether is negative     |- number
    write(encoder, (num > BITS6 ? BIT8 : 0) | (isNegative ? BIT7 : 0) | (BITS6 & num));
    num >>>= 6;
    // We don't need to consider the case of num === 0 so we can use a different
    // pattern here than above.
    while (num > 0) {
      write(encoder, (num > BITS7 ? BIT8 : 0) | (BITS7 & num));
      num >>>= 7;
    }
  };

  /**
   * Write a variable length string.
   *
   * @function
   * @param {Encoder} encoder
   * @param {String} str The string that is to be encoded.
   */
  const writeVarString = (encoder, str) => {
    const encodedString = unescape(encodeURIComponent(str));
    const len = encodedString.length;
    writeVarUint(encoder, len);
    for (let i = 0; i < len; i++) {
      write(encoder, /** @type {number} */ (encodedString.codePointAt(i)));
    }
  };

  /**
   * Append fixed-length Uint8Array to the encoder.
   *
   * @function
   * @param {Encoder} encoder
   * @param {Uint8Array} uint8Array
   */
  const writeUint8Array = (encoder, uint8Array) => {
    const bufferLen = encoder.cbuf.length;
    const cpos = encoder.cpos;
    const leftCopyLen = min(bufferLen - cpos, uint8Array.length);
    const rightCopyLen = uint8Array.length - leftCopyLen;
    encoder.cbuf.set(uint8Array.subarray(0, leftCopyLen), cpos);
    encoder.cpos += leftCopyLen;
    if (rightCopyLen > 0) {
      // Still something to write, write right half..
      // Append new buffer
      encoder.bufs.push(encoder.cbuf);
      // must have at least size of remaining buffer
      encoder.cbuf = new Uint8Array(max(bufferLen * 2, rightCopyLen));
      // copy array
      encoder.cbuf.set(uint8Array.subarray(leftCopyLen));
      encoder.cpos = rightCopyLen;
    }
  };

  /**
   * Append an Uint8Array to Encoder.
   *
   * @function
   * @param {Encoder} encoder
   * @param {Uint8Array} uint8Array
   */
  const writeVarUint8Array = (encoder, uint8Array) => {
    writeVarUint(encoder, uint8Array.byteLength);
    writeUint8Array(encoder, uint8Array);
  };

  /**
   * Create an DataView of the next `len` bytes. Use it to write data after
   * calling this function.
   *
   * ```js
   * // write float32 using DataView
   * const dv = writeOnDataView(encoder, 4)
   * dv.setFloat32(0, 1.1)
   * // read float32 using DataView
   * const dv = readFromDataView(encoder, 4)
   * dv.getFloat32(0) // => 1.100000023841858 (leaving it to the reader to find out why this is the correct result)
   * ```
   *
   * @param {Encoder} encoder
   * @param {number} len
   * @return {DataView}
   */
  const writeOnDataView = (encoder, len) => {
    verifyLen(encoder, len);
    const dview = new DataView(encoder.cbuf.buffer, encoder.cpos, len);
    encoder.cpos += len;
    return dview
  };

  /**
   * @param {Encoder} encoder
   * @param {number} num
   */
  const writeFloat32 = (encoder, num) => writeOnDataView(encoder, 4).setFloat32(0, num, false);

  /**
   * @param {Encoder} encoder
   * @param {number} num
   */
  const writeFloat64 = (encoder, num) => writeOnDataView(encoder, 8).setFloat64(0, num, false);

  /**
   * @param {Encoder} encoder
   * @param {bigint} num
   */
  const writeBigInt64 = (encoder, num) => /** @type {any} */ (writeOnDataView(encoder, 8)).setBigInt64(0, num, false);

  const floatTestBed = new DataView(new ArrayBuffer(4));
  /**
   * Check if a number can be encoded as a 32 bit float.
   *
   * @param {number} num
   * @return {boolean}
   */
  const isFloat32 = num => {
    floatTestBed.setFloat32(0, num);
    return floatTestBed.getFloat32(0) === num
  };

  /**
   * Encode data with efficient binary format.
   *
   * Differences to JSON:
   * • Transforms data to a binary format (not to a string)
   * • Encodes undefined, NaN, and ArrayBuffer (these can't be represented in JSON)
   * • Numbers are efficiently encoded either as a variable length integer, as a
   *   32 bit float, as a 64 bit float, or as a 64 bit bigint.
   *
   * Encoding table:
   *
   * | Data Type           | Prefix   | Encoding Method    | Comment |
   * | ------------------- | -------- | ------------------ | ------- |
   * | undefined           | 127      |                    | Functions, symbol, and everything that cannot be identified is encoded as undefined |
   * | null                | 126      |                    | |
   * | integer             | 125      | writeVarInt        | Only encodes 32 bit signed integers |
   * | float32             | 124      | writeFloat32       | |
   * | float64             | 123      | writeFloat64       | |
   * | bigint              | 122      | writeBigInt64      | |
   * | boolean (false)     | 121      |                    | True and false are different data types so we save the following byte |
   * | boolean (true)      | 120      |                    | - 0b01111000 so the last bit determines whether true or false |
   * | string              | 119      | writeVarString     | |
   * | object<string,any>  | 118      | custom             | Writes {length} then {length} key-value pairs |
   * | array<any>          | 117      | custom             | Writes {length} then {length} json values |
   * | Uint8Array          | 116      | writeVarUint8Array | We use Uint8Array for any kind of binary data |
   *
   * Reasons for the decreasing prefix:
   * We need the first bit for extendability (later we may want to encode the
   * prefix with writeVarUint). The remaining 7 bits are divided as follows:
   * [0-30]   the beginning of the data range is used for custom purposes
   *          (defined by the function that uses this library)
   * [31-127] the end of the data range is used for data encoding by
   *          lib0/encoding.js
   *
   * @param {Encoder} encoder
   * @param {undefined|null|number|bigint|boolean|string|Object<string,any>|Array<any>|Uint8Array} data
   */
  const writeAny = (encoder, data) => {
    switch (typeof data) {
      case 'string':
        // TYPE 119: STRING
        write(encoder, 119);
        writeVarString(encoder, data);
        break
      case 'number':
        if (isInteger(data) && abs(data) <= BITS31) {
          // TYPE 125: INTEGER
          write(encoder, 125);
          writeVarInt(encoder, data);
        } else if (isFloat32(data)) {
          // TYPE 124: FLOAT32
          write(encoder, 124);
          writeFloat32(encoder, data);
        } else {
          // TYPE 123: FLOAT64
          write(encoder, 123);
          writeFloat64(encoder, data);
        }
        break
      case 'bigint':
        // TYPE 122: BigInt
        write(encoder, 122);
        writeBigInt64(encoder, data);
        break
      case 'object':
        if (data === null) {
          // TYPE 126: null
          write(encoder, 126);
        } else if (data instanceof Array) {
          // TYPE 117: Array
          write(encoder, 117);
          writeVarUint(encoder, data.length);
          for (let i = 0; i < data.length; i++) {
            writeAny(encoder, data[i]);
          }
        } else if (data instanceof Uint8Array) {
          // TYPE 116: ArrayBuffer
          write(encoder, 116);
          writeVarUint8Array(encoder, data);
        } else {
          // TYPE 118: Object
          write(encoder, 118);
          const keys = Object.keys(data);
          writeVarUint(encoder, keys.length);
          for (let i = 0; i < keys.length; i++) {
            const key = keys[i];
            writeVarString(encoder, key);
            writeAny(encoder, data[key]);
          }
        }
        break
      case 'boolean':
        // TYPE 120/121: boolean (true/false)
        write(encoder, data ? 120 : 121);
        break
      default:
        // TYPE 127: undefined
        write(encoder, 127);
    }
  };

  /**
   * Now come a few stateful encoder that have their own classes.
   */

  /**
   * Basic Run Length Encoder - a basic compression implementation.
   *
   * Encodes [1,1,1,7] to [1,3,7,1] (3 times 1, 1 time 7). This encoder might do more harm than good if there are a lot of values that are not repeated.
   *
   * It was originally used for image compression. Cool .. article http://csbruce.com/cbm/transactor/pdfs/trans_v7_i06.pdf
   *
   * @note T must not be null!
   *
   * @template T
   */
  class RleEncoder extends Encoder {
    /**
     * @param {function(Encoder, T):void} writer
     */
    constructor (writer) {
      super();
      /**
       * The writer
       */
      this.w = writer;
      /**
       * Current state
       * @type {T|null}
       */
      this.s = null;
      this.count = 0;
    }

    /**
     * @param {T} v
     */
    write (v) {
      if (this.s === v) {
        this.count++;
      } else {
        if (this.count > 0) {
          // flush counter, unless this is the first value (count = 0)
          writeVarUint(this, this.count - 1); // since count is always > 0, we can decrement by one. non-standard encoding ftw
        }
        this.count = 1;
        // write first value
        this.w(this, v);
        this.s = v;
      }
    }
  }

  /**
   * @param {UintOptRleEncoder} encoder
   */
  const flushUintOptRleEncoder = encoder => {
    /* istanbul ignore else */
    if (encoder.count > 0) {
      // flush counter, unless this is the first value (count = 0)
      // case 1: just a single value. set sign to positive
      // case 2: write several values. set sign to negative to indicate that there is a length coming
      writeVarInt(encoder.encoder, encoder.count === 1 ? encoder.s : -encoder.s);
      if (encoder.count > 1) {
        writeVarUint(encoder.encoder, encoder.count - 2); // since count is always > 1, we can decrement by one. non-standard encoding ftw
      }
    }
  };

  /**
   * Optimized Rle encoder that does not suffer from the mentioned problem of the basic Rle encoder.
   *
   * Internally uses VarInt encoder to write unsigned integers. If the input occurs multiple times, we write
   * write it as a negative number. The UintOptRleDecoder then understands that it needs to read a count.
   *
   * Encodes [1,2,3,3,3] as [1,2,-3,3] (once 1, once 2, three times 3)
   */
  class UintOptRleEncoder {
    constructor () {
      this.encoder = new Encoder();
      /**
       * @type {number}
       */
      this.s = 0;
      this.count = 0;
    }

    /**
     * @param {number} v
     */
    write (v) {
      if (this.s === v) {
        this.count++;
      } else {
        flushUintOptRleEncoder(this);
        this.count = 1;
        this.s = v;
      }
    }

    toUint8Array () {
      flushUintOptRleEncoder(this);
      return toUint8Array(this.encoder)
    }
  }

  /**
   * @param {IntDiffOptRleEncoder} encoder
   */
  const flushIntDiffOptRleEncoder = encoder => {
    if (encoder.count > 0) {
      //          31 bit making up the diff | wether to write the counter
      const encodedDiff = encoder.diff << 1 | (encoder.count === 1 ? 0 : 1);
      // flush counter, unless this is the first value (count = 0)
      // case 1: just a single value. set first bit to positive
      // case 2: write several values. set first bit to negative to indicate that there is a length coming
      writeVarInt(encoder.encoder, encodedDiff);
      if (encoder.count > 1) {
        writeVarUint(encoder.encoder, encoder.count - 2); // since count is always > 1, we can decrement by one. non-standard encoding ftw
      }
    }
  };

  /**
   * A combination of the IntDiffEncoder and the UintOptRleEncoder.
   *
   * The count approach is similar to the UintDiffOptRleEncoder, but instead of using the negative bitflag, it encodes
   * in the LSB whether a count is to be read. Therefore this Encoder only supports 31 bit integers!
   *
   * Encodes [1, 2, 3, 2] as [3, 1, 6, -1] (more specifically [(1 << 1) | 1, (3 << 0) | 0, -1])
   *
   * Internally uses variable length encoding. Contrary to normal UintVar encoding, the first byte contains:
   * * 1 bit that denotes whether the next value is a count (LSB)
   * * 1 bit that denotes whether this value is negative (MSB - 1)
   * * 1 bit that denotes whether to continue reading the variable length integer (MSB)
   *
   * Therefore, only five bits remain to encode diff ranges.
   *
   * Use this Encoder only when appropriate. In most cases, this is probably a bad idea.
   */
  class IntDiffOptRleEncoder {
    constructor () {
      this.encoder = new Encoder();
      /**
       * @type {number}
       */
      this.s = 0;
      this.count = 0;
      this.diff = 0;
    }

    /**
     * @param {number} v
     */
    write (v) {
      if (this.diff === v - this.s) {
        this.s = v;
        this.count++;
      } else {
        flushIntDiffOptRleEncoder(this);
        this.count = 1;
        this.diff = v - this.s;
        this.s = v;
      }
    }

    toUint8Array () {
      flushIntDiffOptRleEncoder(this);
      return toUint8Array(this.encoder)
    }
  }

  /**
   * Optimized String Encoder.
   *
   * Encoding many small strings in a simple Encoder is not very efficient. The function call to decode a string takes some time and creates references that must be eventually deleted.
   * In practice, when decoding several million small strings, the GC will kick in more and more often to collect orphaned string objects (or maybe there is another reason?).
   *
   * This string encoder solves the above problem. All strings are concatenated and written as a single string using a single encoding call.
   *
   * The lengths are encoded using a UintOptRleEncoder.
   */
  class StringEncoder {
    constructor () {
      /**
       * @type {Array<string>}
       */
      this.sarr = [];
      this.s = '';
      this.lensE = new UintOptRleEncoder();
    }

    /**
     * @param {string} string
     */
    write (string) {
      this.s += string;
      if (this.s.length > 19) {
        this.sarr.push(this.s);
        this.s = '';
      }
      this.lensE.write(string.length);
    }

    toUint8Array () {
      const encoder = new Encoder();
      this.sarr.push(this.s);
      this.s = '';
      writeVarString(encoder, this.sarr.join(''));
      writeUint8Array(encoder, this.lensE.toUint8Array());
      return toUint8Array(encoder)
    }
  }

  class DeleteItem {
    /**
     * @param {number} clock
     * @param {number} len
     */
    constructor (clock, len) {
      /**
       * @type {number}
       */
      this.clock = clock;
      /**
       * @type {number}
       */
      this.len = len;
    }
  }

  /**
   * We no longer maintain a DeleteStore. DeleteSet is a temporary object that is created when needed.
   * - When created in a transaction, it must only be accessed after sorting, and merging
   *   - This DeleteSet is send to other clients
   * - We do not create a DeleteSet when we send a sync message. The DeleteSet message is created directly from StructStore
   * - We read a DeleteSet as part of a sync/update message. In this case the DeleteSet is already sorted and merged.
   */
  class DeleteSet {
    constructor () {
      /**
       * @type {Map<number,Array<DeleteItem>>}
       */
      this.clients = new Map();
    }
  }

  /**
   * Iterate over all structs that the DeleteSet gc's.
   *
   * @param {Transaction} transaction
   * @param {DeleteSet} ds
   * @param {function(GC|Item):void} f
   *
   * @function
   */
  const iterateDeletedStructs = (transaction, ds, f) =>
    ds.clients.forEach((deletes, clientid) => {
      const structs = /** @type {Array<GC|Item>} */ (transaction.doc.store.clients.get(clientid));
      for (let i = 0; i < deletes.length; i++) {
        const del = deletes[i];
        iterateStructs(transaction, structs, del.clock, del.len, f);
      }
    });

  /**
   * @param {Array<DeleteItem>} dis
   * @param {number} clock
   * @return {number|null}
   *
   * @private
   * @function
   */
  const findIndexDS = (dis, clock) => {
    let left = 0;
    let right = dis.length - 1;
    while (left <= right) {
      const midindex = floor((left + right) / 2);
      const mid = dis[midindex];
      const midclock = mid.clock;
      if (midclock <= clock) {
        if (clock < midclock + mid.len) {
          return midindex
        }
        left = midindex + 1;
      } else {
        right = midindex - 1;
      }
    }
    return null
  };

  /**
   * @param {DeleteSet} ds
   * @param {ID} id
   * @return {boolean}
   *
   * @private
   * @function
   */
  const isDeleted = (ds, id) => {
    const dis = ds.clients.get(id.client);
    return dis !== undefined && findIndexDS(dis, id.clock) !== null
  };

  /**
   * @param {DeleteSet} ds
   *
   * @private
   * @function
   */
  const sortAndMergeDeleteSet = ds => {
    ds.clients.forEach(dels => {
      dels.sort((a, b) => a.clock - b.clock);
      // merge items without filtering or splicing the array
      // i is the current pointer
      // j refers to the current insert position for the pointed item
      // try to merge dels[i] into dels[j-1] or set dels[j]=dels[i]
      let i, j;
      for (i = 1, j = 1; i < dels.length; i++) {
        const left = dels[j - 1];
        const right = dels[i];
        if (left.clock + left.len >= right.clock) {
          left.len = max(left.len, right.clock + right.len - left.clock);
        } else {
          if (j < i) {
            dels[j] = right;
          }
          j++;
        }
      }
      dels.length = j;
    });
  };

  /**
   * @param {Array<DeleteSet>} dss
   * @return {DeleteSet} A fresh DeleteSet
   */
  const mergeDeleteSets = dss => {
    const merged = new DeleteSet();
    for (let dssI = 0; dssI < dss.length; dssI++) {
      dss[dssI].clients.forEach((delsLeft, client) => {
        if (!merged.clients.has(client)) {
          // Write all missing keys from current ds and all following.
          // If merged already contains `client` current ds has already been added.
          /**
           * @type {Array<DeleteItem>}
           */
          const dels = delsLeft.slice();
          for (let i = dssI + 1; i < dss.length; i++) {
            appendTo(dels, dss[i].clients.get(client) || []);
          }
          merged.clients.set(client, dels);
        }
      });
    }
    sortAndMergeDeleteSet(merged);
    return merged
  };

  /**
   * @param {DeleteSet} ds
   * @param {number} client
   * @param {number} clock
   * @param {number} length
   *
   * @private
   * @function
   */
  const addToDeleteSet = (ds, client, clock, length) => {
    setIfUndefined(ds.clients, client, () => []).push(new DeleteItem(clock, length));
  };

  const createDeleteSet = () => new DeleteSet();

  /**
   * @param {StructStore} ss
   * @return {DeleteSet} Merged and sorted DeleteSet
   *
   * @private
   * @function
   */
  const createDeleteSetFromStructStore = ss => {
    const ds = createDeleteSet();
    ss.clients.forEach((structs, client) => {
      /**
       * @type {Array<DeleteItem>}
       */
      const dsitems = [];
      for (let i = 0; i < structs.length; i++) {
        const struct = structs[i];
        if (struct.deleted) {
          const clock = struct.id.clock;
          let len = struct.length;
          if (i + 1 < structs.length) {
            for (let next = structs[i + 1]; i + 1 < structs.length && next.deleted; next = structs[++i + 1]) {
              len += next.length;
            }
          }
          dsitems.push(new DeleteItem(clock, len));
        }
      }
      if (dsitems.length > 0) {
        ds.clients.set(client, dsitems);
      }
    });
    return ds
  };

  /**
   * @param {DSEncoderV1 | DSEncoderV2} encoder
   * @param {DeleteSet} ds
   *
   * @private
   * @function
   */
  const writeDeleteSet = (encoder, ds) => {
    writeVarUint(encoder.restEncoder, ds.clients.size);
    ds.clients.forEach((dsitems, client) => {
      encoder.resetDsCurVal();
      writeVarUint(encoder.restEncoder, client);
      const len = dsitems.length;
      writeVarUint(encoder.restEncoder, len);
      for (let i = 0; i < len; i++) {
        const item = dsitems[i];
        encoder.writeDsClock(item.clock);
        encoder.writeDsLen(item.len);
      }
    });
  };

  /**
   * @param {DSDecoderV1 | DSDecoderV2} decoder
   * @return {DeleteSet}
   *
   * @private
   * @function
   */
  const readDeleteSet = decoder => {
    const ds = new DeleteSet();
    const numClients = readVarUint(decoder.restDecoder);
    for (let i = 0; i < numClients; i++) {
      decoder.resetDsCurVal();
      const client = readVarUint(decoder.restDecoder);
      const numberOfDeletes = readVarUint(decoder.restDecoder);
      if (numberOfDeletes > 0) {
        const dsField = setIfUndefined(ds.clients, client, () => []);
        for (let i = 0; i < numberOfDeletes; i++) {
          dsField.push(new DeleteItem(decoder.readDsClock(), decoder.readDsLen()));
        }
      }
    }
    return ds
  };

  /**
   * @todo YDecoder also contains references to String and other Decoders. Would make sense to exchange YDecoder.toUint8Array for YDecoder.DsToUint8Array()..
   */

  /**
   * @param {DSDecoderV1 | DSDecoderV2} decoder
   * @param {Transaction} transaction
   * @param {StructStore} store
   * @return {Uint8Array|null} Returns a v2 update containing all deletes that couldn't be applied yet; or null if all deletes were applied successfully.
   *
   * @private
   * @function
   */
  const readAndApplyDeleteSet = (decoder, transaction, store) => {
    const unappliedDS = new DeleteSet();
    const numClients = readVarUint(decoder.restDecoder);
    for (let i = 0; i < numClients; i++) {
      decoder.resetDsCurVal();
      const client = readVarUint(decoder.restDecoder);
      const numberOfDeletes = readVarUint(decoder.restDecoder);
      const structs = store.clients.get(client) || [];
      const state = getState(store, client);
      for (let i = 0; i < numberOfDeletes; i++) {
        const clock = decoder.readDsClock();
        const clockEnd = clock + decoder.readDsLen();
        if (clock < state) {
          if (state < clockEnd) {
            addToDeleteSet(unappliedDS, client, state, clockEnd - state);
          }
          let index = findIndexSS(structs, clock);
          /**
           * We can ignore the case of GC and Delete structs, because we are going to skip them
           * @type {Item}
           */
          // @ts-ignore
          let struct = structs[index];
          // split the first item if necessary
          if (!struct.deleted && struct.id.clock < clock) {
            structs.splice(index + 1, 0, splitItem(transaction, struct, clock - struct.id.clock));
            index++; // increase we now want to use the next struct
          }
          while (index < structs.length) {
            // @ts-ignore
            struct = structs[index++];
            if (struct.id.clock < clockEnd) {
              if (!struct.deleted) {
                if (clockEnd < struct.id.clock + struct.length) {
                  structs.splice(index, 0, splitItem(transaction, struct, clockEnd - struct.id.clock));
                }
                struct.delete(transaction);
              }
            } else {
              break
            }
          }
        } else {
          addToDeleteSet(unappliedDS, client, clock, clockEnd - clock);
        }
      }
    }
    if (unappliedDS.clients.size > 0) {
      const ds = new UpdateEncoderV2();
      writeVarUint(ds.restEncoder, 0); // encode 0 structs
      writeDeleteSet(ds, unappliedDS);
      return ds.toUint8Array()
    }
    return null
  };

  /* eslint-env browser */

  const isoCrypto = typeof crypto === 'undefined' ? null : crypto;

  /**
   * @type {function(number):ArrayBuffer}
   */
  const cryptoRandomBuffer = isoCrypto !== null
    ? len => {
      // browser
      const buf = new ArrayBuffer(len);
      const arr = new Uint8Array(buf);
      isoCrypto.getRandomValues(arr);
      return buf
    }
    : len => {
      // polyfill
      const buf = new ArrayBuffer(len);
      const arr = new Uint8Array(buf);
      for (let i = 0; i < len; i++) {
        arr[i] = Math.ceil((Math.random() * 0xFFFFFFFF) >>> 0);
      }
      return buf
    };

  const rand = Math.random;

  const uint32 = () => new Uint32Array(cryptoRandomBuffer(4))[0];

  // @ts-ignore
  const uuidv4Template = [1e7] + -1e3 + -4e3 + -8e3 + -1e11;
  const uuidv4 = () => uuidv4Template.replace(/[018]/g, /** @param {number} c */ c =>
    (c ^ uint32() & 15 >> c / 4).toString(16)
  );

  /**
   * Utility module to work with time.
   *
   * @module time
   */

  /**
   * Return current unix time.
   *
   * @return {number}
   */
  const getUnixTime = Date.now;

  /**
   * Utility helpers to work with promises.
   *
   * @module promise
   */

  /**
   * @template T
   * @callback PromiseResolve
   * @param {T|PromiseLike<T>} [result]
   */

  /**
   * @template T
   * @param {function(PromiseResolve<T>,function(Error):void):any} f
   * @return {Promise<T>}
   */
  const create$3 = f => /** @type {Promise<T>} */ (new Promise(f));

  /**
   * @param {Error} [reason]
   * @return {Promise<never>}
   */
  const reject = reason => Promise.reject(reason);

  /**
   * @template T
   * @param {T|void} res
   * @return {Promise<T|void>}
   */
  const resolve = res => Promise.resolve(res);

  /**
   * @module Y
   */

  const generateNewClientId = uint32;

  /**
   * @typedef {Object} DocOpts
   * @property {boolean} [DocOpts.gc=true] Disable garbage collection (default: gc=true)
   * @property {function(Item):boolean} [DocOpts.gcFilter] Will be called before an Item is garbage collected. Return false to keep the Item.
   * @property {string} [DocOpts.guid] Define a globally unique identifier for this document
   * @property {string | null} [DocOpts.collectionid] Associate this document with a collection. This only plays a role if your provider has a concept of collection.
   * @property {any} [DocOpts.meta] Any kind of meta information you want to associate with this document. If this is a subdocument, remote peers will store the meta information as well.
   * @property {boolean} [DocOpts.autoLoad] If a subdocument, automatically load document. If this is a subdocument, remote peers will load the document as well automatically.
   * @property {boolean} [DocOpts.shouldLoad] Whether the document should be synced by the provider now. This is toggled to true when you call ydoc.load()
   */

  /**
   * A Yjs instance handles the state of shared data.
   * @extends Observable<string>
   */
  class Doc extends Observable {
    /**
     * @param {DocOpts} [opts] configuration
     */
    constructor ({ guid = uuidv4(), collectionid = null, gc = true, gcFilter = () => true, meta = null, autoLoad = false, shouldLoad = true } = {}) {
      super();
      this.gc = gc;
      this.gcFilter = gcFilter;
      this.clientID = generateNewClientId();
      this.guid = guid;
      this.collectionid = collectionid;
      /**
       * @type {Map<string, AbstractType<YEvent<any>>>}
       */
      this.share = new Map();
      this.store = new StructStore();
      /**
       * @type {Transaction | null}
       */
      this._transaction = null;
      /**
       * @type {Array<Transaction>}
       */
      this._transactionCleanups = [];
      /**
       * @type {Set<Doc>}
       */
      this.subdocs = new Set();
      /**
       * If this document is a subdocument - a document integrated into another document - then _item is defined.
       * @type {Item?}
       */
      this._item = null;
      this.shouldLoad = shouldLoad;
      this.autoLoad = autoLoad;
      this.meta = meta;
      this.isLoaded = false;
      this.whenLoaded = create$3(resolve => {
        this.on('load', () => {
          this.isLoaded = true;
          resolve(this);
        });
      });
    }

    /**
     * Notify the parent document that you request to load data into this subdocument (if it is a subdocument).
     *
     * `load()` might be used in the future to request any provider to load the most current data.
     *
     * It is safe to call `load()` multiple times.
     */
    load () {
      const item = this._item;
      if (item !== null && !this.shouldLoad) {
        transact(/** @type {any} */ (item.parent).doc, transaction => {
          transaction.subdocsLoaded.add(this);
        }, null, true);
      }
      this.shouldLoad = true;
    }

    getSubdocs () {
      return this.subdocs
    }

    getSubdocGuids () {
      return new Set(Array.from(this.subdocs).map(doc => doc.guid))
    }

    /**
     * Changes that happen inside of a transaction are bundled. This means that
     * the observer fires _after_ the transaction is finished and that all changes
     * that happened inside of the transaction are sent as one message to the
     * other peers.
     *
     * @param {function(Transaction):void} f The function that should be executed as a transaction
     * @param {any} [origin] Origin of who started the transaction. Will be stored on transaction.origin
     *
     * @public
     */
    transact (f, origin = null) {
      transact(this, f, origin);
    }

    /**
     * Define a shared data type.
     *
     * Multiple calls of `y.get(name, TypeConstructor)` yield the same result
     * and do not overwrite each other. I.e.
     * `y.define(name, Y.Array) === y.define(name, Y.Array)`
     *
     * After this method is called, the type is also available on `y.share.get(name)`.
     *
     * *Best Practices:*
     * Define all types right after the Yjs instance is created and store them in a separate object.
     * Also use the typed methods `getText(name)`, `getArray(name)`, ..
     *
     * @example
     *   const y = new Y(..)
     *   const appState = {
     *     document: y.getText('document')
     *     comments: y.getArray('comments')
     *   }
     *
     * @param {string} name
     * @param {Function} TypeConstructor The constructor of the type definition. E.g. Y.Text, Y.Array, Y.Map, ...
     * @return {AbstractType<any>} The created type. Constructed with TypeConstructor
     *
     * @public
     */
    get (name, TypeConstructor = AbstractType) {
      const type = setIfUndefined(this.share, name, () => {
        // @ts-ignore
        const t = new TypeConstructor();
        t._integrate(this, null);
        return t
      });
      const Constr = type.constructor;
      if (TypeConstructor !== AbstractType && Constr !== TypeConstructor) {
        if (Constr === AbstractType) {
          // @ts-ignore
          const t = new TypeConstructor();
          t._map = type._map;
          type._map.forEach(/** @param {Item?} n */ n => {
            for (; n !== null; n = n.left) {
              // @ts-ignore
              n.parent = t;
            }
          });
          t._start = type._start;
          for (let n = t._start; n !== null; n = n.right) {
            n.parent = t;
          }
          t._length = type._length;
          this.share.set(name, t);
          t._integrate(this, null);
          return t
        } else {
          throw new Error(`Type with the name ${name} has already been defined with a different constructor`)
        }
      }
      return type
    }

    /**
     * @template T
     * @param {string} [name]
     * @return {YArray<T>}
     *
     * @public
     */
    getArray (name = '') {
      // @ts-ignore
      return this.get(name, YArray)
    }

    /**
     * @param {string} [name]
     * @return {YText}
     *
     * @public
     */
    getText (name = '') {
      // @ts-ignore
      return this.get(name, YText)
    }

    /**
     * @template T
     * @param {string} [name]
     * @return {YMap<T>}
     *
     * @public
     */
    getMap (name = '') {
      // @ts-ignore
      return this.get(name, YMap)
    }

    /**
     * @param {string} [name]
     * @return {YXmlFragment}
     *
     * @public
     */
    getXmlFragment (name = '') {
      // @ts-ignore
      return this.get(name, YXmlFragment)
    }

    /**
     * Converts the entire document into a js object, recursively traversing each yjs type
     * Doesn't log types that have not been defined (using ydoc.getType(..)).
     *
     * @deprecated Do not use this method and rather call toJSON directly on the shared types.
     *
     * @return {Object<string, any>}
     */
    toJSON () {
      /**
       * @type {Object<string, any>}
       */
      const doc = {};

      this.share.forEach((value, key) => {
        doc[key] = value.toJSON();
      });

      return doc
    }

    /**
     * Emit `destroy` event and unregister all event handlers.
     */
    destroy () {
      from(this.subdocs).forEach(subdoc => subdoc.destroy());
      const item = this._item;
      if (item !== null) {
        this._item = null;
        const content = /** @type {ContentDoc} */ (item.content);
        content.doc = new Doc({ guid: this.guid, ...content.opts, shouldLoad: false });
        content.doc._item = item;
        transact(/** @type {any} */ (item).parent.doc, transaction => {
          const doc = content.doc;
          if (!item.deleted) {
            transaction.subdocsAdded.add(doc);
          }
          transaction.subdocsRemoved.add(this);
        }, null, true);
      }
      this.emit('destroyed', [true]);
      this.emit('destroy', [this]);
      super.destroy();
    }

    /**
     * @param {string} eventName
     * @param {function(...any):any} f
     */
    on (eventName, f) {
      super.on(eventName, f);
    }

    /**
     * @param {string} eventName
     * @param {function} f
     */
    off (eventName, f) {
      super.off(eventName, f);
    }
  }

  class DSDecoderV1 {
    /**
     * @param {decoding.Decoder} decoder
     */
    constructor (decoder) {
      this.restDecoder = decoder;
    }

    resetDsCurVal () {
      // nop
    }

    /**
     * @return {number}
     */
    readDsClock () {
      return readVarUint(this.restDecoder)
    }

    /**
     * @return {number}
     */
    readDsLen () {
      return readVarUint(this.restDecoder)
    }
  }

  class UpdateDecoderV1 extends DSDecoderV1 {
    /**
     * @return {ID}
     */
    readLeftID () {
      return createID(readVarUint(this.restDecoder), readVarUint(this.restDecoder))
    }

    /**
     * @return {ID}
     */
    readRightID () {
      return createID(readVarUint(this.restDecoder), readVarUint(this.restDecoder))
    }

    /**
     * Read the next client id.
     * Use this in favor of readID whenever possible to reduce the number of objects created.
     */
    readClient () {
      return readVarUint(this.restDecoder)
    }

    /**
     * @return {number} info An unsigned 8-bit integer
     */
    readInfo () {
      return readUint8(this.restDecoder)
    }

    /**
     * @return {string}
     */
    readString () {
      return readVarString(this.restDecoder)
    }

    /**
     * @return {boolean} isKey
     */
    readParentInfo () {
      return readVarUint(this.restDecoder) === 1
    }

    /**
     * @return {number} info An unsigned 8-bit integer
     */
    readTypeRef () {
      return readVarUint(this.restDecoder)
    }

    /**
     * Write len of a struct - well suited for Opt RLE encoder.
     *
     * @return {number} len
     */
    readLen () {
      return readVarUint(this.restDecoder)
    }

    /**
     * @return {any}
     */
    readAny () {
      return readAny(this.restDecoder)
    }

    /**
     * @return {Uint8Array}
     */
    readBuf () {
      return copyUint8Array(readVarUint8Array(this.restDecoder))
    }

    /**
     * Legacy implementation uses JSON parse. We use any-decoding in v2.
     *
     * @return {any}
     */
    readJSON () {
      return JSON.parse(readVarString(this.restDecoder))
    }

    /**
     * @return {string}
     */
    readKey () {
      return readVarString(this.restDecoder)
    }
  }

  class DSDecoderV2 {
    /**
     * @param {decoding.Decoder} decoder
     */
    constructor (decoder) {
      /**
       * @private
       */
      this.dsCurrVal = 0;
      this.restDecoder = decoder;
    }

    resetDsCurVal () {
      this.dsCurrVal = 0;
    }

    /**
     * @return {number}
     */
    readDsClock () {
      this.dsCurrVal += readVarUint(this.restDecoder);
      return this.dsCurrVal
    }

    /**
     * @return {number}
     */
    readDsLen () {
      const diff = readVarUint(this.restDecoder) + 1;
      this.dsCurrVal += diff;
      return diff
    }
  }

  class UpdateDecoderV2 extends DSDecoderV2 {
    /**
     * @param {decoding.Decoder} decoder
     */
    constructor (decoder) {
      super(decoder);
      /**
       * List of cached keys. If the keys[id] does not exist, we read a new key
       * from stringEncoder and push it to keys.
       *
       * @type {Array<string>}
       */
      this.keys = [];
      readVarUint(decoder); // read feature flag - currently unused
      this.keyClockDecoder = new IntDiffOptRleDecoder(readVarUint8Array(decoder));
      this.clientDecoder = new UintOptRleDecoder(readVarUint8Array(decoder));
      this.leftClockDecoder = new IntDiffOptRleDecoder(readVarUint8Array(decoder));
      this.rightClockDecoder = new IntDiffOptRleDecoder(readVarUint8Array(decoder));
      this.infoDecoder = new RleDecoder(readVarUint8Array(decoder), readUint8);
      this.stringDecoder = new StringDecoder(readVarUint8Array(decoder));
      this.parentInfoDecoder = new RleDecoder(readVarUint8Array(decoder), readUint8);
      this.typeRefDecoder = new UintOptRleDecoder(readVarUint8Array(decoder));
      this.lenDecoder = new UintOptRleDecoder(readVarUint8Array(decoder));
    }

    /**
     * @return {ID}
     */
    readLeftID () {
      return new ID(this.clientDecoder.read(), this.leftClockDecoder.read())
    }

    /**
     * @return {ID}
     */
    readRightID () {
      return new ID(this.clientDecoder.read(), this.rightClockDecoder.read())
    }

    /**
     * Read the next client id.
     * Use this in favor of readID whenever possible to reduce the number of objects created.
     */
    readClient () {
      return this.clientDecoder.read()
    }

    /**
     * @return {number} info An unsigned 8-bit integer
     */
    readInfo () {
      return /** @type {number} */ (this.infoDecoder.read())
    }

    /**
     * @return {string}
     */
    readString () {
      return this.stringDecoder.read()
    }

    /**
     * @return {boolean}
     */
    readParentInfo () {
      return this.parentInfoDecoder.read() === 1
    }

    /**
     * @return {number} An unsigned 8-bit integer
     */
    readTypeRef () {
      return this.typeRefDecoder.read()
    }

    /**
     * Write len of a struct - well suited for Opt RLE encoder.
     *
     * @return {number}
     */
    readLen () {
      return this.lenDecoder.read()
    }

    /**
     * @return {any}
     */
    readAny () {
      return readAny(this.restDecoder)
    }

    /**
     * @return {Uint8Array}
     */
    readBuf () {
      return readVarUint8Array(this.restDecoder)
    }

    /**
     * This is mainly here for legacy purposes.
     *
     * Initial we incoded objects using JSON. Now we use the much faster lib0/any-encoder. This method mainly exists for legacy purposes for the v1 encoder.
     *
     * @return {any}
     */
    readJSON () {
      return readAny(this.restDecoder)
    }

    /**
     * @return {string}
     */
    readKey () {
      const keyClock = this.keyClockDecoder.read();
      if (keyClock < this.keys.length) {
        return this.keys[keyClock]
      } else {
        const key = this.stringDecoder.read();
        this.keys.push(key);
        return key
      }
    }
  }

  /**
   * Error helpers.
   *
   * @module error
   */

  /* istanbul ignore next */
  /**
   * @param {string} s
   * @return {Error}
   */
  const create$2 = s => new Error(s);

  /* istanbul ignore next */
  /**
   * @throws {Error}
   * @return {never}
   */
  const methodUnimplemented = () => {
    throw create$2('Method unimplemented')
  };

  /* istanbul ignore next */
  /**
   * @throws {Error}
   * @return {never}
   */
  const unexpectedCase = () => {
    throw create$2('Unexpected case')
  };

  class DSEncoderV1 {
    constructor () {
      this.restEncoder = createEncoder();
    }

    toUint8Array () {
      return toUint8Array(this.restEncoder)
    }

    resetDsCurVal () {
      // nop
    }

    /**
     * @param {number} clock
     */
    writeDsClock (clock) {
      writeVarUint(this.restEncoder, clock);
    }

    /**
     * @param {number} len
     */
    writeDsLen (len) {
      writeVarUint(this.restEncoder, len);
    }
  }

  class UpdateEncoderV1 extends DSEncoderV1 {
    /**
     * @param {ID} id
     */
    writeLeftID (id) {
      writeVarUint(this.restEncoder, id.client);
      writeVarUint(this.restEncoder, id.clock);
    }

    /**
     * @param {ID} id
     */
    writeRightID (id) {
      writeVarUint(this.restEncoder, id.client);
      writeVarUint(this.restEncoder, id.clock);
    }

    /**
     * Use writeClient and writeClock instead of writeID if possible.
     * @param {number} client
     */
    writeClient (client) {
      writeVarUint(this.restEncoder, client);
    }

    /**
     * @param {number} info An unsigned 8-bit integer
     */
    writeInfo (info) {
      writeUint8(this.restEncoder, info);
    }

    /**
     * @param {string} s
     */
    writeString (s) {
      writeVarString(this.restEncoder, s);
    }

    /**
     * @param {boolean} isYKey
     */
    writeParentInfo (isYKey) {
      writeVarUint(this.restEncoder, isYKey ? 1 : 0);
    }

    /**
     * @param {number} info An unsigned 8-bit integer
     */
    writeTypeRef (info) {
      writeVarUint(this.restEncoder, info);
    }

    /**
     * Write len of a struct - well suited for Opt RLE encoder.
     *
     * @param {number} len
     */
    writeLen (len) {
      writeVarUint(this.restEncoder, len);
    }

    /**
     * @param {any} any
     */
    writeAny (any) {
      writeAny(this.restEncoder, any);
    }

    /**
     * @param {Uint8Array} buf
     */
    writeBuf (buf) {
      writeVarUint8Array(this.restEncoder, buf);
    }

    /**
     * @param {any} embed
     */
    writeJSON (embed) {
      writeVarString(this.restEncoder, JSON.stringify(embed));
    }

    /**
     * @param {string} key
     */
    writeKey (key) {
      writeVarString(this.restEncoder, key);
    }
  }

  class DSEncoderV2 {
    constructor () {
      this.restEncoder = createEncoder(); // encodes all the rest / non-optimized
      this.dsCurrVal = 0;
    }

    toUint8Array () {
      return toUint8Array(this.restEncoder)
    }

    resetDsCurVal () {
      this.dsCurrVal = 0;
    }

    /**
     * @param {number} clock
     */
    writeDsClock (clock) {
      const diff = clock - this.dsCurrVal;
      this.dsCurrVal = clock;
      writeVarUint(this.restEncoder, diff);
    }

    /**
     * @param {number} len
     */
    writeDsLen (len) {
      if (len === 0) {
        unexpectedCase();
      }
      writeVarUint(this.restEncoder, len - 1);
      this.dsCurrVal += len;
    }
  }

  class UpdateEncoderV2 extends DSEncoderV2 {
    constructor () {
      super();
      /**
       * @type {Map<string,number>}
       */
      this.keyMap = new Map();
      /**
       * Refers to the next uniqe key-identifier to me used.
       * See writeKey method for more information.
       *
       * @type {number}
       */
      this.keyClock = 0;
      this.keyClockEncoder = new IntDiffOptRleEncoder();
      this.clientEncoder = new UintOptRleEncoder();
      this.leftClockEncoder = new IntDiffOptRleEncoder();
      this.rightClockEncoder = new IntDiffOptRleEncoder();
      this.infoEncoder = new RleEncoder(writeUint8);
      this.stringEncoder = new StringEncoder();
      this.parentInfoEncoder = new RleEncoder(writeUint8);
      this.typeRefEncoder = new UintOptRleEncoder();
      this.lenEncoder = new UintOptRleEncoder();
    }

    toUint8Array () {
      const encoder = createEncoder();
      writeVarUint(encoder, 0); // this is a feature flag that we might use in the future
      writeVarUint8Array(encoder, this.keyClockEncoder.toUint8Array());
      writeVarUint8Array(encoder, this.clientEncoder.toUint8Array());
      writeVarUint8Array(encoder, this.leftClockEncoder.toUint8Array());
      writeVarUint8Array(encoder, this.rightClockEncoder.toUint8Array());
      writeVarUint8Array(encoder, toUint8Array(this.infoEncoder));
      writeVarUint8Array(encoder, this.stringEncoder.toUint8Array());
      writeVarUint8Array(encoder, toUint8Array(this.parentInfoEncoder));
      writeVarUint8Array(encoder, this.typeRefEncoder.toUint8Array());
      writeVarUint8Array(encoder, this.lenEncoder.toUint8Array());
      // @note The rest encoder is appended! (note the missing var)
      writeUint8Array(encoder, toUint8Array(this.restEncoder));
      return toUint8Array(encoder)
    }

    /**
     * @param {ID} id
     */
    writeLeftID (id) {
      this.clientEncoder.write(id.client);
      this.leftClockEncoder.write(id.clock);
    }

    /**
     * @param {ID} id
     */
    writeRightID (id) {
      this.clientEncoder.write(id.client);
      this.rightClockEncoder.write(id.clock);
    }

    /**
     * @param {number} client
     */
    writeClient (client) {
      this.clientEncoder.write(client);
    }

    /**
     * @param {number} info An unsigned 8-bit integer
     */
    writeInfo (info) {
      this.infoEncoder.write(info);
    }

    /**
     * @param {string} s
     */
    writeString (s) {
      this.stringEncoder.write(s);
    }

    /**
     * @param {boolean} isYKey
     */
    writeParentInfo (isYKey) {
      this.parentInfoEncoder.write(isYKey ? 1 : 0);
    }

    /**
     * @param {number} info An unsigned 8-bit integer
     */
    writeTypeRef (info) {
      this.typeRefEncoder.write(info);
    }

    /**
     * Write len of a struct - well suited for Opt RLE encoder.
     *
     * @param {number} len
     */
    writeLen (len) {
      this.lenEncoder.write(len);
    }

    /**
     * @param {any} any
     */
    writeAny (any) {
      writeAny(this.restEncoder, any);
    }

    /**
     * @param {Uint8Array} buf
     */
    writeBuf (buf) {
      writeVarUint8Array(this.restEncoder, buf);
    }

    /**
     * This is mainly here for legacy purposes.
     *
     * Initial we incoded objects using JSON. Now we use the much faster lib0/any-encoder. This method mainly exists for legacy purposes for the v1 encoder.
     *
     * @param {any} embed
     */
    writeJSON (embed) {
      writeAny(this.restEncoder, embed);
    }

    /**
     * Property keys are often reused. For example, in y-prosemirror the key `bold` might
     * occur very often. For a 3d application, the key `position` might occur very often.
     *
     * We cache these keys in a Map and refer to them via a unique number.
     *
     * @param {string} key
     */
    writeKey (key) {
      const clock = this.keyMap.get(key);
      if (clock === undefined) {
        /**
         * @todo uncomment to introduce this feature finally
         *
         * Background. The ContentFormat object was always encoded using writeKey, but the decoder used to use readString.
         * Furthermore, I forgot to set the keyclock. So everything was working fine.
         *
         * However, this feature here is basically useless as it is not being used (it actually only consumes extra memory).
         *
         * I don't know yet how to reintroduce this feature..
         *
         * Older clients won't be able to read updates when we reintroduce this feature. So this should probably be done using a flag.
         *
         */
        // this.keyMap.set(key, this.keyClock)
        this.keyClockEncoder.write(this.keyClock++);
        this.stringEncoder.write(key);
      } else {
        this.keyClockEncoder.write(clock);
      }
    }
  }

  /**
   * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
   * @param {Array<GC|Item>} structs All structs by `client`
   * @param {number} client
   * @param {number} clock write structs starting with `ID(client,clock)`
   *
   * @function
   */
  const writeStructs = (encoder, structs, client, clock) => {
    // write first id
    clock = max(clock, structs[0].id.clock); // make sure the first id exists
    const startNewStructs = findIndexSS(structs, clock);
    // write # encoded structs
    writeVarUint(encoder.restEncoder, structs.length - startNewStructs);
    encoder.writeClient(client);
    writeVarUint(encoder.restEncoder, clock);
    const firstStruct = structs[startNewStructs];
    // write first struct with an offset
    firstStruct.write(encoder, clock - firstStruct.id.clock);
    for (let i = startNewStructs + 1; i < structs.length; i++) {
      structs[i].write(encoder, 0);
    }
  };

  /**
   * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
   * @param {StructStore} store
   * @param {Map<number,number>} _sm
   *
   * @private
   * @function
   */
  const writeClientsStructs = (encoder, store, _sm) => {
    // we filter all valid _sm entries into sm
    const sm = new Map();
    _sm.forEach((clock, client) => {
      // only write if new structs are available
      if (getState(store, client) > clock) {
        sm.set(client, clock);
      }
    });
    getStateVector(store).forEach((clock, client) => {
      if (!_sm.has(client)) {
        sm.set(client, 0);
      }
    });
    // write # states that were updated
    writeVarUint(encoder.restEncoder, sm.size);
    // Write items with higher client ids first
    // This heavily improves the conflict algorithm.
    Array.from(sm.entries()).sort((a, b) => b[0] - a[0]).forEach(([client, clock]) => {
      // @ts-ignore
      writeStructs(encoder, store.clients.get(client), client, clock);
    });
  };

  /**
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder The decoder object to read data from.
   * @param {Doc} doc
   * @return {Map<number, { i: number, refs: Array<Item | GC> }>}
   *
   * @private
   * @function
   */
  const readClientsStructRefs = (decoder, doc) => {
    /**
     * @type {Map<number, { i: number, refs: Array<Item | GC> }>}
     */
    const clientRefs = create$5();
    const numOfStateUpdates = readVarUint(decoder.restDecoder);
    for (let i = 0; i < numOfStateUpdates; i++) {
      const numberOfStructs = readVarUint(decoder.restDecoder);
      /**
       * @type {Array<GC|Item>}
       */
      const refs = new Array(numberOfStructs);
      const client = decoder.readClient();
      let clock = readVarUint(decoder.restDecoder);
      // const start = performance.now()
      clientRefs.set(client, { i: 0, refs });
      for (let i = 0; i < numberOfStructs; i++) {
        const info = decoder.readInfo();
        switch (BITS5 & info) {
          case 0: { // GC
            const len = decoder.readLen();
            refs[i] = new GC(createID(client, clock), len);
            clock += len;
            break
          }
          case 10: { // Skip Struct (nothing to apply)
            // @todo we could reduce the amount of checks by adding Skip struct to clientRefs so we know that something is missing.
            const len = readVarUint(decoder.restDecoder);
            refs[i] = new Skip(createID(client, clock), len);
            clock += len;
            break
          }
          default: { // Item with content
            /**
             * The optimized implementation doesn't use any variables because inlining variables is faster.
             * Below a non-optimized version is shown that implements the basic algorithm with
             * a few comments
             */
            const cantCopyParentInfo = (info & (BIT7 | BIT8)) === 0;
            // If parent = null and neither left nor right are defined, then we know that `parent` is child of `y`
            // and we read the next string as parentYKey.
            // It indicates how we store/retrieve parent from `y.share`
            // @type {string|null}
            const struct = new Item(
              createID(client, clock),
              null, // leftd
              (info & BIT8) === BIT8 ? decoder.readLeftID() : null, // origin
              null, // right
              (info & BIT7) === BIT7 ? decoder.readRightID() : null, // right origin
              cantCopyParentInfo ? (decoder.readParentInfo() ? doc.get(decoder.readString()) : decoder.readLeftID()) : null, // parent
              cantCopyParentInfo && (info & BIT6) === BIT6 ? decoder.readString() : null, // parentSub
              readItemContent(decoder, info) // item content
            );
            /* A non-optimized implementation of the above algorithm:

            // The item that was originally to the left of this item.
            const origin = (info & binary.BIT8) === binary.BIT8 ? decoder.readLeftID() : null
            // The item that was originally to the right of this item.
            const rightOrigin = (info & binary.BIT7) === binary.BIT7 ? decoder.readRightID() : null
            const cantCopyParentInfo = (info & (binary.BIT7 | binary.BIT8)) === 0
            const hasParentYKey = cantCopyParentInfo ? decoder.readParentInfo() : false
            // If parent = null and neither left nor right are defined, then we know that `parent` is child of `y`
            // and we read the next string as parentYKey.
            // It indicates how we store/retrieve parent from `y.share`
            // @type {string|null}
            const parentYKey = cantCopyParentInfo && hasParentYKey ? decoder.readString() : null

            const struct = new Item(
              createID(client, clock),
              null, // leftd
              origin, // origin
              null, // right
              rightOrigin, // right origin
              cantCopyParentInfo && !hasParentYKey ? decoder.readLeftID() : (parentYKey !== null ? doc.get(parentYKey) : null), // parent
              cantCopyParentInfo && (info & binary.BIT6) === binary.BIT6 ? decoder.readString() : null, // parentSub
              readItemContent(decoder, info) // item content
            )
            */
            refs[i] = struct;
            clock += struct.length;
          }
        }
      }
      // console.log('time to read: ', performance.now() - start) // @todo remove
    }
    return clientRefs
  };

  /**
   * Resume computing structs generated by struct readers.
   *
   * While there is something to do, we integrate structs in this order
   * 1. top element on stack, if stack is not empty
   * 2. next element from current struct reader (if empty, use next struct reader)
   *
   * If struct causally depends on another struct (ref.missing), we put next reader of
   * `ref.id.client` on top of stack.
   *
   * At some point we find a struct that has no causal dependencies,
   * then we start emptying the stack.
   *
   * It is not possible to have circles: i.e. struct1 (from client1) depends on struct2 (from client2)
   * depends on struct3 (from client1). Therefore the max stack size is eqaul to `structReaders.length`.
   *
   * This method is implemented in a way so that we can resume computation if this update
   * causally depends on another update.
   *
   * @param {Transaction} transaction
   * @param {StructStore} store
   * @param {Map<number, { i: number, refs: (GC | Item)[] }>} clientsStructRefs
   * @return { null | { update: Uint8Array, missing: Map<number,number> } }
   *
   * @private
   * @function
   */
  const integrateStructs = (transaction, store, clientsStructRefs) => {
    /**
     * @type {Array<Item | GC>}
     */
    const stack = [];
    // sort them so that we take the higher id first, in case of conflicts the lower id will probably not conflict with the id from the higher user.
    let clientsStructRefsIds = Array.from(clientsStructRefs.keys()).sort((a, b) => a - b);
    if (clientsStructRefsIds.length === 0) {
      return null
    }
    const getNextStructTarget = () => {
      if (clientsStructRefsIds.length === 0) {
        return null
      }
      let nextStructsTarget = /** @type {{i:number,refs:Array<GC|Item>}} */ (clientsStructRefs.get(clientsStructRefsIds[clientsStructRefsIds.length - 1]));
      while (nextStructsTarget.refs.length === nextStructsTarget.i) {
        clientsStructRefsIds.pop();
        if (clientsStructRefsIds.length > 0) {
          nextStructsTarget = /** @type {{i:number,refs:Array<GC|Item>}} */ (clientsStructRefs.get(clientsStructRefsIds[clientsStructRefsIds.length - 1]));
        } else {
          return null
        }
      }
      return nextStructsTarget
    };
    let curStructsTarget = getNextStructTarget();
    if (curStructsTarget === null && stack.length === 0) {
      return null
    }

    /**
     * @type {StructStore}
     */
    const restStructs = new StructStore();
    const missingSV = new Map();
    /**
     * @param {number} client
     * @param {number} clock
     */
    const updateMissingSv = (client, clock) => {
      const mclock = missingSV.get(client);
      if (mclock == null || mclock > clock) {
        missingSV.set(client, clock);
      }
    };
    /**
     * @type {GC|Item}
     */
    let stackHead = /** @type {any} */ (curStructsTarget).refs[/** @type {any} */ (curStructsTarget).i++];
    // caching the state because it is used very often
    const state = new Map();

    const addStackToRestSS = () => {
      for (const item of stack) {
        const client = item.id.client;
        const unapplicableItems = clientsStructRefs.get(client);
        if (unapplicableItems) {
          // decrement because we weren't able to apply previous operation
          unapplicableItems.i--;
          restStructs.clients.set(client, unapplicableItems.refs.slice(unapplicableItems.i));
          clientsStructRefs.delete(client);
          unapplicableItems.i = 0;
          unapplicableItems.refs = [];
        } else {
          // item was the last item on clientsStructRefs and the field was already cleared. Add item to restStructs and continue
          restStructs.clients.set(client, [item]);
        }
        // remove client from clientsStructRefsIds to prevent users from applying the same update again
        clientsStructRefsIds = clientsStructRefsIds.filter(c => c !== client);
      }
      stack.length = 0;
    };

    // iterate over all struct readers until we are done
    while (true) {
      if (stackHead.constructor !== Skip) {
        const localClock = setIfUndefined(state, stackHead.id.client, () => getState(store, stackHead.id.client));
        const offset = localClock - stackHead.id.clock;
        if (offset < 0) {
          // update from the same client is missing
          stack.push(stackHead);
          updateMissingSv(stackHead.id.client, stackHead.id.clock - 1);
          // hid a dead wall, add all items from stack to restSS
          addStackToRestSS();
        } else {
          const missing = stackHead.getMissing(transaction, store);
          if (missing !== null) {
            stack.push(stackHead);
            // get the struct reader that has the missing struct
            /**
             * @type {{ refs: Array<GC|Item>, i: number }}
             */
            const structRefs = clientsStructRefs.get(/** @type {number} */ (missing)) || { refs: [], i: 0 };
            if (structRefs.refs.length === structRefs.i) {
              // This update message causally depends on another update message that doesn't exist yet
              updateMissingSv(/** @type {number} */ (missing), getState(store, missing));
              addStackToRestSS();
            } else {
              stackHead = structRefs.refs[structRefs.i++];
              continue
            }
          } else if (offset === 0 || offset < stackHead.length) {
            // all fine, apply the stackhead
            stackHead.integrate(transaction, offset);
            state.set(stackHead.id.client, stackHead.id.clock + stackHead.length);
          }
        }
      }
      // iterate to next stackHead
      if (stack.length > 0) {
        stackHead = /** @type {GC|Item} */ (stack.pop());
      } else if (curStructsTarget !== null && curStructsTarget.i < curStructsTarget.refs.length) {
        stackHead = /** @type {GC|Item} */ (curStructsTarget.refs[curStructsTarget.i++]);
      } else {
        curStructsTarget = getNextStructTarget();
        if (curStructsTarget === null) {
          // we are done!
          break
        } else {
          stackHead = /** @type {GC|Item} */ (curStructsTarget.refs[curStructsTarget.i++]);
        }
      }
    }
    if (restStructs.clients.size > 0) {
      const encoder = new UpdateEncoderV2();
      writeClientsStructs(encoder, restStructs, new Map());
      // write empty deleteset
      // writeDeleteSet(encoder, new DeleteSet())
      writeVarUint(encoder.restEncoder, 0); // => no need for an extra function call, just write 0 deletes
      return { missing: missingSV, update: encoder.toUint8Array() }
    }
    return null
  };

  /**
   * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
   * @param {Transaction} transaction
   *
   * @private
   * @function
   */
  const writeStructsFromTransaction = (encoder, transaction) => writeClientsStructs(encoder, transaction.doc.store, transaction.beforeState);

  /**
   * Read and apply a document update.
   *
   * This function has the same effect as `applyUpdate` but accepts an decoder.
   *
   * @param {decoding.Decoder} decoder
   * @param {Doc} ydoc
   * @param {any} [transactionOrigin] This will be stored on `transaction.origin` and `.on('update', (update, origin))`
   * @param {UpdateDecoderV1 | UpdateDecoderV2} [structDecoder]
   *
   * @function
   */
  const readUpdateV2 = (decoder, ydoc, transactionOrigin, structDecoder = new UpdateDecoderV2(decoder)) =>
    transact(ydoc, transaction => {
      // force that transaction.local is set to non-local
      transaction.local = false;
      let retry = false;
      const doc = transaction.doc;
      const store = doc.store;
      // let start = performance.now()
      const ss = readClientsStructRefs(structDecoder, doc);
      // console.log('time to read structs: ', performance.now() - start) // @todo remove
      // start = performance.now()
      // console.log('time to merge: ', performance.now() - start) // @todo remove
      // start = performance.now()
      const restStructs = integrateStructs(transaction, store, ss);
      const pending = store.pendingStructs;
      if (pending) {
        // check if we can apply something
        for (const [client, clock] of pending.missing) {
          if (clock < getState(store, client)) {
            retry = true;
            break
          }
        }
        if (restStructs) {
          // merge restStructs into store.pending
          for (const [client, clock] of restStructs.missing) {
            const mclock = pending.missing.get(client);
            if (mclock == null || mclock > clock) {
              pending.missing.set(client, clock);
            }
          }
          pending.update = mergeUpdatesV2([pending.update, restStructs.update]);
        }
      } else {
        store.pendingStructs = restStructs;
      }
      // console.log('time to integrate: ', performance.now() - start) // @todo remove
      // start = performance.now()
      const dsRest = readAndApplyDeleteSet(structDecoder, transaction, store);
      if (store.pendingDs) {
        // @todo we could make a lower-bound state-vector check as we do above
        const pendingDSUpdate = new UpdateDecoderV2(createDecoder(store.pendingDs));
        readVarUint(pendingDSUpdate.restDecoder); // read 0 structs, because we only encode deletes in pendingdsupdate
        const dsRest2 = readAndApplyDeleteSet(pendingDSUpdate, transaction, store);
        if (dsRest && dsRest2) {
          // case 1: ds1 != null && ds2 != null
          store.pendingDs = mergeUpdatesV2([dsRest, dsRest2]);
        } else {
          // case 2: ds1 != null
          // case 3: ds2 != null
          // case 4: ds1 == null && ds2 == null
          store.pendingDs = dsRest || dsRest2;
        }
      } else {
        // Either dsRest == null && pendingDs == null OR dsRest != null
        store.pendingDs = dsRest;
      }
      // console.log('time to cleanup: ', performance.now() - start) // @todo remove
      // start = performance.now()

      // console.log('time to resume delete readers: ', performance.now() - start) // @todo remove
      // start = performance.now()
      if (retry) {
        const update = /** @type {{update: Uint8Array}} */ (store.pendingStructs).update;
        store.pendingStructs = null;
        applyUpdateV2(transaction.doc, update);
      }
    }, transactionOrigin, false);

  /**
   * Apply a document update created by, for example, `y.on('update', update => ..)` or `update = encodeStateAsUpdate()`.
   *
   * This function has the same effect as `readUpdate` but accepts an Uint8Array instead of a Decoder.
   *
   * @param {Doc} ydoc
   * @param {Uint8Array} update
   * @param {any} [transactionOrigin] This will be stored on `transaction.origin` and `.on('update', (update, origin))`
   * @param {typeof UpdateDecoderV1 | typeof UpdateDecoderV2} [YDecoder]
   *
   * @function
   */
  const applyUpdateV2 = (ydoc, update, transactionOrigin, YDecoder = UpdateDecoderV2) => {
    const decoder = createDecoder(update);
    readUpdateV2(decoder, ydoc, transactionOrigin, new YDecoder(decoder));
  };

  /**
   * Apply a document update created by, for example, `y.on('update', update => ..)` or `update = encodeStateAsUpdate()`.
   *
   * This function has the same effect as `readUpdate` but accepts an Uint8Array instead of a Decoder.
   *
   * @param {Doc} ydoc
   * @param {Uint8Array} update
   * @param {any} [transactionOrigin] This will be stored on `transaction.origin` and `.on('update', (update, origin))`
   *
   * @function
   */
  const applyUpdate = (ydoc, update, transactionOrigin) => applyUpdateV2(ydoc, update, transactionOrigin, UpdateDecoderV1);

  /**
   * Write all the document as a single update message. If you specify the state of the remote client (`targetStateVector`) it will
   * only write the operations that are missing.
   *
   * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
   * @param {Doc} doc
   * @param {Map<number,number>} [targetStateVector] The state of the target that receives the update. Leave empty to write all known structs
   *
   * @function
   */
  const writeStateAsUpdate = (encoder, doc, targetStateVector = new Map()) => {
    writeClientsStructs(encoder, doc.store, targetStateVector);
    writeDeleteSet(encoder, createDeleteSetFromStructStore(doc.store));
  };

  /**
   * Write all the document as a single update message that can be applied on the remote document. If you specify the state of the remote client (`targetState`) it will
   * only write the operations that are missing.
   *
   * Use `writeStateAsUpdate` instead if you are working with lib0/encoding.js#Encoder
   *
   * @param {Doc} doc
   * @param {Uint8Array} [encodedTargetStateVector] The state of the target that receives the update. Leave empty to write all known structs
   * @param {UpdateEncoderV1 | UpdateEncoderV2} [encoder]
   * @return {Uint8Array}
   *
   * @function
   */
  const encodeStateAsUpdateV2 = (doc, encodedTargetStateVector = new Uint8Array([0]), encoder = new UpdateEncoderV2()) => {
    const targetStateVector = decodeStateVector(encodedTargetStateVector);
    writeStateAsUpdate(encoder, doc, targetStateVector);
    const updates = [encoder.toUint8Array()];
    // also add the pending updates (if there are any)
    if (doc.store.pendingDs) {
      updates.push(doc.store.pendingDs);
    }
    if (doc.store.pendingStructs) {
      updates.push(diffUpdateV2(doc.store.pendingStructs.update, encodedTargetStateVector));
    }
    if (updates.length > 1) {
      if (encoder.constructor === UpdateEncoderV1) {
        return mergeUpdates(updates.map((update, i) => i === 0 ? update : convertUpdateFormatV2ToV1(update)))
      } else if (encoder.constructor === UpdateEncoderV2) {
        return mergeUpdatesV2(updates)
      }
    }
    return updates[0]
  };

  /**
   * Write all the document as a single update message that can be applied on the remote document. If you specify the state of the remote client (`targetState`) it will
   * only write the operations that are missing.
   *
   * Use `writeStateAsUpdate` instead if you are working with lib0/encoding.js#Encoder
   *
   * @param {Doc} doc
   * @param {Uint8Array} [encodedTargetStateVector] The state of the target that receives the update. Leave empty to write all known structs
   * @return {Uint8Array}
   *
   * @function
   */
  const encodeStateAsUpdate = (doc, encodedTargetStateVector) => encodeStateAsUpdateV2(doc, encodedTargetStateVector, new UpdateEncoderV1());

  /**
   * Read state vector from Decoder and return as Map
   *
   * @param {DSDecoderV1 | DSDecoderV2} decoder
   * @return {Map<number,number>} Maps `client` to the number next expected `clock` from that client.
   *
   * @function
   */
  const readStateVector = decoder => {
    const ss = new Map();
    const ssLength = readVarUint(decoder.restDecoder);
    for (let i = 0; i < ssLength; i++) {
      const client = readVarUint(decoder.restDecoder);
      const clock = readVarUint(decoder.restDecoder);
      ss.set(client, clock);
    }
    return ss
  };

  /**
   * Read decodedState and return State as Map.
   *
   * @param {Uint8Array} decodedState
   * @return {Map<number,number>} Maps `client` to the number next expected `clock` from that client.
   *
   * @function
   */
  // export const decodeStateVectorV2 = decodedState => readStateVector(new DSDecoderV2(decoding.createDecoder(decodedState)))

  /**
   * Read decodedState and return State as Map.
   *
   * @param {Uint8Array} decodedState
   * @return {Map<number,number>} Maps `client` to the number next expected `clock` from that client.
   *
   * @function
   */
  const decodeStateVector = decodedState => readStateVector(new DSDecoderV1(createDecoder(decodedState)));

  /**
   * @param {DSEncoderV1 | DSEncoderV2} encoder
   * @param {Map<number,number>} sv
   * @function
   */
  const writeStateVector = (encoder, sv) => {
    writeVarUint(encoder.restEncoder, sv.size);
    Array.from(sv.entries()).sort((a, b) => b[0] - a[0]).forEach(([client, clock]) => {
      writeVarUint(encoder.restEncoder, client); // @todo use a special client decoder that is based on mapping
      writeVarUint(encoder.restEncoder, clock);
    });
    return encoder
  };

  /**
   * @param {DSEncoderV1 | DSEncoderV2} encoder
   * @param {Doc} doc
   *
   * @function
   */
  const writeDocumentStateVector = (encoder, doc) => writeStateVector(encoder, getStateVector(doc.store));

  /**
   * Encode State as Uint8Array.
   *
   * @param {Doc|Map<number,number>} doc
   * @param {DSEncoderV1 | DSEncoderV2} [encoder]
   * @return {Uint8Array}
   *
   * @function
   */
  const encodeStateVectorV2 = (doc, encoder = new DSEncoderV2()) => {
    if (doc instanceof Map) {
      writeStateVector(encoder, doc);
    } else {
      writeDocumentStateVector(encoder, doc);
    }
    return encoder.toUint8Array()
  };

  /**
   * Encode State as Uint8Array.
   *
   * @param {Doc|Map<number,number>} doc
   * @return {Uint8Array}
   *
   * @function
   */
  const encodeStateVector = doc => encodeStateVectorV2(doc, new DSEncoderV1());

  /**
   * Utility functions for working with EcmaScript objects.
   *
   * @module object
   */

  /**
   * @param {Object<string,any>} obj
   */
  const keys$1 = Object.keys;

  /**
   * @param {Object<string,any>} obj
   * @return {number}
   */
  const length = obj => keys$1(obj).length;

  /**
   * @param {Object<string,any>} obj
   * @param {function(any,string):boolean} f
   * @return {boolean}
   */
  const every = (obj, f) => {
    for (const key in obj) {
      if (!f(obj[key], key)) {
        return false
      }
    }
    return true
  };

  /**
   * Calls `Object.prototype.hasOwnProperty`.
   *
   * @param {any} obj
   * @param {string|symbol} key
   * @return {boolean}
   */
  const hasProperty = (obj, key) => Object.prototype.hasOwnProperty.call(obj, key);

  /**
   * @param {Object<string,any>} a
   * @param {Object<string,any>} b
   * @return {boolean}
   */
  const equalFlat = (a, b) => a === b || (length(a) === length(b) && every(a, (val, key) => (val !== undefined || hasProperty(b, key)) && b[key] === val));

  /**
   * Common functions and function call helpers.
   *
   * @module function
   */

  /**
   * Calls all functions in `fs` with args. Only throws after all functions were called.
   *
   * @param {Array<function>} fs
   * @param {Array<any>} args
   */
  const callAll = (fs, args, i = 0) => {
    try {
      for (; i < fs.length; i++) {
        fs[i](...args);
      }
    } finally {
      if (i < fs.length) {
        callAll(fs, args, i + 1);
      }
    }
  };

  const nop = () => {};

  /**
   * @template T
   *
   * @param {T} a
   * @param {T} b
   * @return {boolean}
   */
  const equalityStrict = (a, b) => a === b;

  /**
   * @param {any} a
   * @param {any} b
   * @return {boolean}
   */
  const equalityDeep = (a, b) => {
    if (a == null || b == null) {
      return equalityStrict(a, b)
    }
    if (a.constructor !== b.constructor) {
      return false
    }
    if (a === b) {
      return true
    }
    switch (a.constructor) {
      case ArrayBuffer:
        a = new Uint8Array(a);
        b = new Uint8Array(b);
      // eslint-disable-next-line no-fallthrough
      case Uint8Array: {
        if (a.byteLength !== b.byteLength) {
          return false
        }
        for (let i = 0; i < a.length; i++) {
          if (a[i] !== b[i]) {
            return false
          }
        }
        break
      }
      case Set: {
        if (a.size !== b.size) {
          return false
        }
        for (const value of a) {
          if (!b.has(value)) {
            return false
          }
        }
        break
      }
      case Map: {
        if (a.size !== b.size) {
          return false
        }
        for (const key of a.keys()) {
          if (!b.has(key) || !equalityDeep(a.get(key), b.get(key))) {
            return false
          }
        }
        break
      }
      case Object:
        if (length(a) !== length(b)) {
          return false
        }
        for (const key in a) {
          if (!hasProperty(a, key) || !equalityDeep(a[key], b[key])) {
            return false
          }
        }
        break
      case Array:
        if (a.length !== b.length) {
          return false
        }
        for (let i = 0; i < a.length; i++) {
          if (!equalityDeep(a[i], b[i])) {
            return false
          }
        }
        break
      default:
        return false
    }
    return true
  };

  /**
   * General event handler implementation.
   *
   * @template ARG0, ARG1
   *
   * @private
   */
  class EventHandler {
    constructor () {
      /**
       * @type {Array<function(ARG0, ARG1):void>}
       */
      this.l = [];
    }
  }

  /**
   * @template ARG0,ARG1
   * @returns {EventHandler<ARG0,ARG1>}
   *
   * @private
   * @function
   */
  const createEventHandler = () => new EventHandler();

  /**
   * Adds an event listener that is called when
   * {@link EventHandler#callEventListeners} is called.
   *
   * @template ARG0,ARG1
   * @param {EventHandler<ARG0,ARG1>} eventHandler
   * @param {function(ARG0,ARG1):void} f The event handler.
   *
   * @private
   * @function
   */
  const addEventHandlerListener = (eventHandler, f) =>
    eventHandler.l.push(f);

  /**
   * Removes an event listener.
   *
   * @template ARG0,ARG1
   * @param {EventHandler<ARG0,ARG1>} eventHandler
   * @param {function(ARG0,ARG1):void} f The event handler that was added with
   *                     {@link EventHandler#addEventListener}
   *
   * @private
   * @function
   */
  const removeEventHandlerListener = (eventHandler, f) => {
    const l = eventHandler.l;
    const len = l.length;
    eventHandler.l = l.filter(g => f !== g);
    if (len === eventHandler.l.length) {
      console.error('[yjs] Tried to remove event handler that doesn\'t exist.');
    }
  };

  /**
   * Call all event listeners that were added via
   * {@link EventHandler#addEventListener}.
   *
   * @template ARG0,ARG1
   * @param {EventHandler<ARG0,ARG1>} eventHandler
   * @param {ARG0} arg0
   * @param {ARG1} arg1
   *
   * @private
   * @function
   */
  const callEventHandlerListeners = (eventHandler, arg0, arg1) =>
    callAll(eventHandler.l, [arg0, arg1]);

  class ID {
    /**
     * @param {number} client client id
     * @param {number} clock unique per client id, continuous number
     */
    constructor (client, clock) {
      /**
       * Client id
       * @type {number}
       */
      this.client = client;
      /**
       * unique per client id, continuous number
       * @type {number}
       */
      this.clock = clock;
    }
  }

  /**
   * @param {ID | null} a
   * @param {ID | null} b
   * @return {boolean}
   *
   * @function
   */
  const compareIDs = (a, b) => a === b || (a !== null && b !== null && a.client === b.client && a.clock === b.clock);

  /**
   * @param {number} client
   * @param {number} clock
   *
   * @private
   * @function
   */
  const createID = (client, clock) => new ID(client, clock);

  /**
   * The top types are mapped from y.share.get(keyname) => type.
   * `type` does not store any information about the `keyname`.
   * This function finds the correct `keyname` for `type` and throws otherwise.
   *
   * @param {AbstractType<any>} type
   * @return {string}
   *
   * @private
   * @function
   */
  const findRootTypeKey = type => {
    // @ts-ignore _y must be defined, otherwise unexpected case
    for (const [key, value] of type.doc.share.entries()) {
      if (value === type) {
        return key
      }
    }
    throw unexpectedCase()
  };

  /**
   * Check if `parent` is a parent of `child`.
   *
   * @param {AbstractType<any>} parent
   * @param {Item|null} child
   * @return {Boolean} Whether `parent` is a parent of `child`.
   *
   * @private
   * @function
   */
  const isParentOf = (parent, child) => {
    while (child !== null) {
      if (child.parent === parent) {
        return true
      }
      child = /** @type {AbstractType<any>} */ (child.parent)._item;
    }
    return false
  };

  /**
   * A relative position is based on the Yjs model and is not affected by document changes.
   * E.g. If you place a relative position before a certain character, it will always point to this character.
   * If you place a relative position at the end of a type, it will always point to the end of the type.
   *
   * A numeric position is often unsuited for user selections, because it does not change when content is inserted
   * before or after.
   *
   * ```Insert(0, 'x')('a|bc') = 'xa|bc'``` Where | is the relative position.
   *
   * One of the properties must be defined.
   *
   * @example
   *   // Current cursor position is at position 10
   *   const relativePosition = createRelativePositionFromIndex(yText, 10)
   *   // modify yText
   *   yText.insert(0, 'abc')
   *   yText.delete(3, 10)
   *   // Compute the cursor position
   *   const absolutePosition = createAbsolutePositionFromRelativePosition(y, relativePosition)
   *   absolutePosition.type === yText // => true
   *   console.log('cursor location is ' + absolutePosition.index) // => cursor location is 3
   *
   */
  class RelativePosition {
    /**
     * @param {ID|null} type
     * @param {string|null} tname
     * @param {ID|null} item
     * @param {number} assoc
     */
    constructor (type, tname, item, assoc = 0) {
      /**
       * @type {ID|null}
       */
      this.type = type;
      /**
       * @type {string|null}
       */
      this.tname = tname;
      /**
       * @type {ID | null}
       */
      this.item = item;
      /**
       * A relative position is associated to a specific character. By default
       * assoc >= 0, the relative position is associated to the character
       * after the meant position.
       * I.e. position 1 in 'ab' is associated to character 'b'.
       *
       * If assoc < 0, then the relative position is associated to the caharacter
       * before the meant position.
       *
       * @type {number}
       */
      this.assoc = assoc;
    }
  }

  /**
   * @param {RelativePosition} rpos
   * @return {any}
   */
  const relativePositionToJSON = rpos => {
    const json = {};
    if (rpos.type) {
      json.type = rpos.type;
    }
    if (rpos.tname) {
      json.tname = rpos.tname;
    }
    if (rpos.item) {
      json.item = rpos.item;
    }
    if (rpos.assoc != null) {
      json.assoc = rpos.assoc;
    }
    return json
  };

  /**
   * @param {any} json
   * @return {RelativePosition}
   *
   * @function
   */
  const createRelativePositionFromJSON = json => new RelativePosition(json.type == null ? null : createID(json.type.client, json.type.clock), json.tname || null, json.item == null ? null : createID(json.item.client, json.item.clock), json.assoc == null ? 0 : json.assoc);

  class AbsolutePosition {
    /**
     * @param {AbstractType<any>} type
     * @param {number} index
     * @param {number} [assoc]
     */
    constructor (type, index, assoc = 0) {
      /**
       * @type {AbstractType<any>}
       */
      this.type = type;
      /**
       * @type {number}
       */
      this.index = index;
      this.assoc = assoc;
    }
  }

  /**
   * @param {AbstractType<any>} type
   * @param {number} index
   * @param {number} [assoc]
   *
   * @function
   */
  const createAbsolutePosition = (type, index, assoc = 0) => new AbsolutePosition(type, index, assoc);

  /**
   * @param {AbstractType<any>} type
   * @param {ID|null} item
   * @param {number} [assoc]
   *
   * @function
   */
  const createRelativePosition = (type, item, assoc) => {
    let typeid = null;
    let tname = null;
    if (type._item === null) {
      tname = findRootTypeKey(type);
    } else {
      typeid = createID(type._item.id.client, type._item.id.clock);
    }
    return new RelativePosition(typeid, tname, item, assoc)
  };

  /**
   * Create a relativePosition based on a absolute position.
   *
   * @param {AbstractType<any>} type The base type (e.g. YText or YArray).
   * @param {number} index The absolute position.
   * @param {number} [assoc]
   * @return {RelativePosition}
   *
   * @function
   */
  const createRelativePositionFromTypeIndex = (type, index, assoc = 0) => {
    let t = type._start;
    if (assoc < 0) {
      // associated to the left character or the beginning of a type, increment index if possible.
      if (index === 0) {
        return createRelativePosition(type, null, assoc)
      }
      index--;
    }
    while (t !== null) {
      if (!t.deleted && t.countable) {
        if (t.length > index) {
          // case 1: found position somewhere in the linked list
          return createRelativePosition(type, createID(t.id.client, t.id.clock + index), assoc)
        }
        index -= t.length;
      }
      if (t.right === null && assoc < 0) {
        // left-associated position, return last available id
        return createRelativePosition(type, t.lastId, assoc)
      }
      t = t.right;
    }
    return createRelativePosition(type, null, assoc)
  };

  /**
   * @param {RelativePosition} rpos
   * @param {Doc} doc
   * @return {AbsolutePosition|null}
   *
   * @function
   */
  const createAbsolutePositionFromRelativePosition = (rpos, doc) => {
    const store = doc.store;
    const rightID = rpos.item;
    const typeID = rpos.type;
    const tname = rpos.tname;
    const assoc = rpos.assoc;
    let type = null;
    let index = 0;
    if (rightID !== null) {
      if (getState(store, rightID.client) <= rightID.clock) {
        return null
      }
      const res = followRedone(store, rightID);
      const right = res.item;
      if (!(right instanceof Item)) {
        return null
      }
      type = /** @type {AbstractType<any>} */ (right.parent);
      if (type._item === null || !type._item.deleted) {
        index = (right.deleted || !right.countable) ? 0 : (res.diff + (assoc >= 0 ? 0 : 1)); // adjust position based on left association if necessary
        let n = right.left;
        while (n !== null) {
          if (!n.deleted && n.countable) {
            index += n.length;
          }
          n = n.left;
        }
      }
    } else {
      if (tname !== null) {
        type = doc.get(tname);
      } else if (typeID !== null) {
        if (getState(store, typeID.client) <= typeID.clock) {
          // type does not exist yet
          return null
        }
        const { item } = followRedone(store, typeID);
        if (item instanceof Item && item.content instanceof ContentType) {
          type = item.content.type;
        } else {
          // struct is garbage collected
          return null
        }
      } else {
        throw unexpectedCase()
      }
      if (assoc >= 0) {
        index = type._length;
      } else {
        index = 0;
      }
    }
    return createAbsolutePosition(type, index, rpos.assoc)
  };

  /**
   * @param {RelativePosition|null} a
   * @param {RelativePosition|null} b
   * @return {boolean}
   *
   * @function
   */
  const compareRelativePositions = (a, b) => a === b || (
    a !== null && b !== null && a.tname === b.tname && compareIDs(a.item, b.item) && compareIDs(a.type, b.type) && a.assoc === b.assoc
  );

  class Snapshot {
    /**
     * @param {DeleteSet} ds
     * @param {Map<number,number>} sv state map
     */
    constructor (ds, sv) {
      /**
       * @type {DeleteSet}
       */
      this.ds = ds;
      /**
       * State Map
       * @type {Map<number,number>}
       */
      this.sv = sv;
    }
  }

  /**
   * @param {DeleteSet} ds
   * @param {Map<number,number>} sm
   * @return {Snapshot}
   */
  const createSnapshot = (ds, sm) => new Snapshot(ds, sm);

  createSnapshot(createDeleteSet(), new Map());

  /**
   * @param {Item} item
   * @param {Snapshot|undefined} snapshot
   *
   * @protected
   * @function
   */
  const isVisible = (item, snapshot) => snapshot === undefined
    ? !item.deleted
    : snapshot.sv.has(item.id.client) && (snapshot.sv.get(item.id.client) || 0) > item.id.clock && !isDeleted(snapshot.ds, item.id);

  /**
   * @param {Transaction} transaction
   * @param {Snapshot} snapshot
   */
  const splitSnapshotAffectedStructs = (transaction, snapshot) => {
    const meta = setIfUndefined(transaction.meta, splitSnapshotAffectedStructs, create$4);
    const store = transaction.doc.store;
    // check if we already split for this snapshot
    if (!meta.has(snapshot)) {
      snapshot.sv.forEach((clock, client) => {
        if (clock < getState(store, client)) {
          getItemCleanStart(transaction, createID(client, clock));
        }
      });
      iterateDeletedStructs(transaction, snapshot.ds, item => {});
      meta.add(snapshot);
    }
  };

  class StructStore {
    constructor () {
      /**
       * @type {Map<number,Array<GC|Item>>}
       */
      this.clients = new Map();
      /**
       * @type {null | { missing: Map<number, number>, update: Uint8Array }}
       */
      this.pendingStructs = null;
      /**
       * @type {null | Uint8Array}
       */
      this.pendingDs = null;
    }
  }

  /**
   * Return the states as a Map<client,clock>.
   * Note that clock refers to the next expected clock id.
   *
   * @param {StructStore} store
   * @return {Map<number,number>}
   *
   * @public
   * @function
   */
  const getStateVector = store => {
    const sm = new Map();
    store.clients.forEach((structs, client) => {
      const struct = structs[structs.length - 1];
      sm.set(client, struct.id.clock + struct.length);
    });
    return sm
  };

  /**
   * @param {StructStore} store
   * @param {number} client
   * @return {number}
   *
   * @public
   * @function
   */
  const getState = (store, client) => {
    const structs = store.clients.get(client);
    if (structs === undefined) {
      return 0
    }
    const lastStruct = structs[structs.length - 1];
    return lastStruct.id.clock + lastStruct.length
  };

  /**
   * @param {StructStore} store
   * @param {GC|Item} struct
   *
   * @private
   * @function
   */
  const addStruct = (store, struct) => {
    let structs = store.clients.get(struct.id.client);
    if (structs === undefined) {
      structs = [];
      store.clients.set(struct.id.client, structs);
    } else {
      const lastStruct = structs[structs.length - 1];
      if (lastStruct.id.clock + lastStruct.length !== struct.id.clock) {
        throw unexpectedCase()
      }
    }
    structs.push(struct);
  };

  /**
   * Perform a binary search on a sorted array
   * @param {Array<Item|GC>} structs
   * @param {number} clock
   * @return {number}
   *
   * @private
   * @function
   */
  const findIndexSS = (structs, clock) => {
    let left = 0;
    let right = structs.length - 1;
    let mid = structs[right];
    let midclock = mid.id.clock;
    if (midclock === clock) {
      return right
    }
    // @todo does it even make sense to pivot the search?
    // If a good split misses, it might actually increase the time to find the correct item.
    // Currently, the only advantage is that search with pivoting might find the item on the first try.
    let midindex = floor((clock / (midclock + mid.length - 1)) * right); // pivoting the search
    while (left <= right) {
      mid = structs[midindex];
      midclock = mid.id.clock;
      if (midclock <= clock) {
        if (clock < midclock + mid.length) {
          return midindex
        }
        left = midindex + 1;
      } else {
        right = midindex - 1;
      }
      midindex = floor((left + right) / 2);
    }
    // Always check state before looking for a struct in StructStore
    // Therefore the case of not finding a struct is unexpected
    throw unexpectedCase()
  };

  /**
   * Expects that id is actually in store. This function throws or is an infinite loop otherwise.
   *
   * @param {StructStore} store
   * @param {ID} id
   * @return {GC|Item}
   *
   * @private
   * @function
   */
  const find$1 = (store, id) => {
    /**
     * @type {Array<GC|Item>}
     */
    // @ts-ignore
    const structs = store.clients.get(id.client);
    return structs[findIndexSS(structs, id.clock)]
  };

  /**
   * Expects that id is actually in store. This function throws or is an infinite loop otherwise.
   * @private
   * @function
   */
  const getItem = /** @type {function(StructStore,ID):Item} */ (find$1);

  /**
   * @param {Transaction} transaction
   * @param {Array<Item|GC>} structs
   * @param {number} clock
   */
  const findIndexCleanStart = (transaction, structs, clock) => {
    const index = findIndexSS(structs, clock);
    const struct = structs[index];
    if (struct.id.clock < clock && struct instanceof Item) {
      structs.splice(index + 1, 0, splitItem(transaction, struct, clock - struct.id.clock));
      return index + 1
    }
    return index
  };

  /**
   * Expects that id is actually in store. This function throws or is an infinite loop otherwise.
   *
   * @param {Transaction} transaction
   * @param {ID} id
   * @return {Item}
   *
   * @private
   * @function
   */
  const getItemCleanStart = (transaction, id) => {
    const structs = /** @type {Array<Item>} */ (transaction.doc.store.clients.get(id.client));
    return structs[findIndexCleanStart(transaction, structs, id.clock)]
  };

  /**
   * Expects that id is actually in store. This function throws or is an infinite loop otherwise.
   *
   * @param {Transaction} transaction
   * @param {StructStore} store
   * @param {ID} id
   * @return {Item}
   *
   * @private
   * @function
   */
  const getItemCleanEnd = (transaction, store, id) => {
    /**
     * @type {Array<Item>}
     */
    // @ts-ignore
    const structs = store.clients.get(id.client);
    const index = findIndexSS(structs, id.clock);
    const struct = structs[index];
    if (id.clock !== struct.id.clock + struct.length - 1 && struct.constructor !== GC) {
      structs.splice(index + 1, 0, splitItem(transaction, struct, id.clock - struct.id.clock + 1));
    }
    return struct
  };

  /**
   * Replace `item` with `newitem` in store
   * @param {StructStore} store
   * @param {GC|Item} struct
   * @param {GC|Item} newStruct
   *
   * @private
   * @function
   */
  const replaceStruct = (store, struct, newStruct) => {
    const structs = /** @type {Array<GC|Item>} */ (store.clients.get(struct.id.client));
    structs[findIndexSS(structs, struct.id.clock)] = newStruct;
  };

  /**
   * Iterate over a range of structs
   *
   * @param {Transaction} transaction
   * @param {Array<Item|GC>} structs
   * @param {number} clockStart Inclusive start
   * @param {number} len
   * @param {function(GC|Item):void} f
   *
   * @function
   */
  const iterateStructs = (transaction, structs, clockStart, len, f) => {
    if (len === 0) {
      return
    }
    const clockEnd = clockStart + len;
    let index = findIndexCleanStart(transaction, structs, clockStart);
    let struct;
    do {
      struct = structs[index++];
      if (clockEnd < struct.id.clock + struct.length) {
        findIndexCleanStart(transaction, structs, clockEnd);
      }
      f(struct);
    } while (index < structs.length && structs[index].id.clock < clockEnd)
  };

  /**
   * Utility module to work with EcmaScript Symbols.
   *
   * @module symbol
   */

  /**
   * Return fresh symbol.
   *
   * @return {Symbol}
   */
  const create$1 = Symbol;

  /**
   * Working with value pairs.
   *
   * @module pair
   */

  /**
   * @template L,R
   */
  class Pair {
    /**
     * @param {L} left
     * @param {R} right
     */
    constructor (left, right) {
      this.left = left;
      this.right = right;
    }
  }

  /**
   * @template L,R
   * @param {L} left
   * @param {R} right
   * @return {Pair<L,R>}
   */
  const create = (left, right) => new Pair(left, right);

  /**
   * @template L,R
   * @param {Array<Pair<L,R>>} arr
   * @param {function(L, R):any} f
   */
  const forEach = (arr, f) => arr.forEach(p => f(p.left, p.right));

  /* eslint-env browser */

  /* istanbul ignore next */
  /**
   * @type {Document}
   */
  const doc$1 = /** @type {Document} */ (typeof document !== 'undefined' ? document : {});

  /**
   * @param {string} name
   * @return {HTMLElement}
   */
  /* istanbul ignore next */
  const createElement = name => doc$1.createElement(name);

  /**
   * @return {DocumentFragment}
   */
  /* istanbul ignore next */
  const createDocumentFragment = () => doc$1.createDocumentFragment();

  /**
   * @param {string} text
   * @return {Text}
   */
  /* istanbul ignore next */
  const createTextNode = text => doc$1.createTextNode(text);

  /* istanbul ignore next */
  /** @type {DOMParser} */ (typeof DOMParser !== 'undefined' ? new DOMParser() : null);

  /**
   * @param {Element} el
   * @param {Array<pair.Pair<string,string|boolean>>} attrs Array of key-value pairs
   * @return {Element}
   */
  /* istanbul ignore next */
  const setAttributes = (el, attrs) => {
    forEach(attrs, (key, value) => {
      if (value === false) {
        el.removeAttribute(key);
      } else if (value === true) {
        el.setAttribute(key, '');
      } else {
        // @ts-ignore
        el.setAttribute(key, value);
      }
    });
    return el
  };

  /**
   * @param {Array<Node>|HTMLCollection} children
   * @return {DocumentFragment}
   */
  /* istanbul ignore next */
  const fragment = children => {
    const fragment = createDocumentFragment();
    for (let i = 0; i < children.length; i++) {
      appendChild(fragment, children[i]);
    }
    return fragment
  };

  /**
   * @param {Element} parent
   * @param {Array<Node>} nodes
   * @return {Element}
   */
  /* istanbul ignore next */
  const append = (parent, nodes) => {
    appendChild(parent, fragment(nodes));
    return parent
  };

  /**
   * @param {string} name
   * @param {Array<pair.Pair<string,string>|pair.Pair<string,boolean>>} attrs Array of key-value pairs
   * @param {Array<Node>} children
   * @return {Element}
   */
  /* istanbul ignore next */
  const element = (name, attrs = [], children = []) =>
    append(setAttributes(createElement(name), attrs), children);

  /**
   * @param {string} t
   * @return {Text}
   */
  /* istanbul ignore next */
  const text = createTextNode;

  /**
   * @param {Map<string,string>} m
   * @return {string}
   */
  /* istanbul ignore next */
  const mapToStyleString = m => map(m, (value, key) => `${key}:${value};`).join('');

  /**
   * @param {Node} parent
   * @param {Node} child
   * @return {Node}
   */
  /* istanbul ignore next */
  const appendChild = (parent, child) => parent.appendChild(child);

  doc$1.ELEMENT_NODE;
  doc$1.TEXT_NODE;
  doc$1.CDATA_SECTION_NODE;
  doc$1.COMMENT_NODE;
  doc$1.DOCUMENT_NODE;
  doc$1.DOCUMENT_TYPE_NODE;
  doc$1.DOCUMENT_FRAGMENT_NODE;

  /**
   * Isomorphic logging module with support for colors!
   *
   * @module logging
   */

  const BOLD = create$1();
  const UNBOLD = create$1();
  const BLUE = create$1();
  const GREY = create$1();
  const GREEN = create$1();
  const RED = create$1();
  const PURPLE = create$1();
  const ORANGE = create$1();
  const UNCOLOR = create$1();

  /**
   * @type {Object<Symbol,pair.Pair<string,string>>}
   */
  const _browserStyleMap = {
    [BOLD]: create('font-weight', 'bold'),
    [UNBOLD]: create('font-weight', 'normal'),
    [BLUE]: create('color', 'blue'),
    [GREEN]: create('color', 'green'),
    [GREY]: create('color', 'grey'),
    [RED]: create('color', 'red'),
    [PURPLE]: create('color', 'purple'),
    [ORANGE]: create('color', 'orange'), // not well supported in chrome when debugging node with inspector - TODO: deprecate
    [UNCOLOR]: create('color', 'black')
  };

  const _nodeStyleMap = {
    [BOLD]: '\u001b[1m',
    [UNBOLD]: '\u001b[2m',
    [BLUE]: '\x1b[34m',
    [GREEN]: '\x1b[32m',
    [GREY]: '\u001b[37m',
    [RED]: '\x1b[31m',
    [PURPLE]: '\x1b[35m',
    [ORANGE]: '\x1b[38;5;208m',
    [UNCOLOR]: '\x1b[0m'
  };

  /* istanbul ignore next */
  /**
   * @param {Array<string|Symbol|Object|number>} args
   * @return {Array<string|object|number>}
   */
  const computeBrowserLoggingArgs = args => {
    const strBuilder = [];
    const styles = [];
    const currentStyle = create$5();
    /**
     * @type {Array<string|Object|number>}
     */
    let logArgs = [];
    // try with formatting until we find something unsupported
    let i = 0;

    for (; i < args.length; i++) {
      const arg = args[i];
      // @ts-ignore
      const style = _browserStyleMap[arg];
      if (style !== undefined) {
        currentStyle.set(style.left, style.right);
      } else {
        if (arg.constructor === String || arg.constructor === Number) {
          const style = mapToStyleString(currentStyle);
          if (i > 0 || style.length > 0) {
            strBuilder.push('%c' + arg);
            styles.push(style);
          } else {
            strBuilder.push(arg);
          }
        } else {
          break
        }
      }
    }

    if (i > 0) {
      // create logArgs with what we have so far
      logArgs = styles;
      logArgs.unshift(strBuilder.join(''));
    }
    // append the rest
    for (; i < args.length; i++) {
      const arg = args[i];
      if (!(arg instanceof Symbol)) {
        logArgs.push(arg);
      }
    }
    return logArgs
  };

  /**
   * @param {Array<string|Symbol|Object|number>} args
   * @return {Array<string|object|number>}
   */
  const computeNodeLoggingArgs = args => {
    const strBuilder = [];
    const logArgs = [];

    // try with formatting until we find something unsupported
    let i = 0;

    for (; i < args.length; i++) {
      const arg = args[i];
      // @ts-ignore
      const style = _nodeStyleMap[arg];
      if (style !== undefined) {
        strBuilder.push(style);
      } else {
        if (arg.constructor === String || arg.constructor === Number) {
          strBuilder.push(arg);
        } else {
          break
        }
      }
    }
    if (i > 0) {
      // create logArgs with what we have so far
      strBuilder.push('\x1b[0m');
      logArgs.push(strBuilder.join(''));
    }
    // append the rest
    for (; i < args.length; i++) {
      const arg = args[i];
      /* istanbul ignore else */
      if (!(arg instanceof Symbol)) {
        logArgs.push(arg);
      }
    }
    return logArgs
  };

  /* istanbul ignore next */
  const computeLoggingArgs = isNode ? computeNodeLoggingArgs : computeBrowserLoggingArgs;

  /**
   * @param {Array<string|Symbol|Object|number>} args
   */
  const print = (...args) => {
    console.log(...computeLoggingArgs(args));
    /* istanbul ignore next */
    vconsoles.forEach(vc => vc.print(args));
  };

  const vconsoles = new Set();

  const loggingColors = [GREEN, PURPLE, ORANGE, BLUE];
  let nextColor = 0;
  let lastLoggingTime = getUnixTime();

  /**
   * @param {string} moduleName
   * @return {function(...any):void}
   */
  const createModuleLogger = moduleName => {
    const color = loggingColors[nextColor];
    const debugRegexVar = getVariable('log');
    const doLogging = debugRegexVar !== null && (debugRegexVar === '*' || debugRegexVar === 'true' || new RegExp(debugRegexVar, 'gi').test(moduleName));
    nextColor = (nextColor + 1) % loggingColors.length;
    moduleName += ': ';

    return !doLogging ? nop : (...args) => {
      const timeNow = getUnixTime();
      const timeDiff = timeNow - lastLoggingTime;
      lastLoggingTime = timeNow;
      print(color, moduleName, UNCOLOR, ...args.map(arg => (typeof arg === 'string' || typeof arg === 'symbol') ? arg : JSON.stringify(arg)), color, ' +' + timeDiff + 'ms');
    }
  };

  /**
   * A transaction is created for every change on the Yjs model. It is possible
   * to bundle changes on the Yjs model in a single transaction to
   * minimize the number on messages sent and the number of observer calls.
   * If possible the user of this library should bundle as many changes as
   * possible. Here is an example to illustrate the advantages of bundling:
   *
   * @example
   * const map = y.define('map', YMap)
   * // Log content when change is triggered
   * map.observe(() => {
   *   console.log('change triggered')
   * })
   * // Each change on the map type triggers a log message:
   * map.set('a', 0) // => "change triggered"
   * map.set('b', 0) // => "change triggered"
   * // When put in a transaction, it will trigger the log after the transaction:
   * y.transact(() => {
   *   map.set('a', 1)
   *   map.set('b', 1)
   * }) // => "change triggered"
   *
   * @public
   */
  class Transaction$1 {
    /**
     * @param {Doc} doc
     * @param {any} origin
     * @param {boolean} local
     */
    constructor (doc, origin, local) {
      /**
       * The Yjs instance.
       * @type {Doc}
       */
      this.doc = doc;
      /**
       * Describes the set of deleted items by ids
       * @type {DeleteSet}
       */
      this.deleteSet = new DeleteSet();
      /**
       * Holds the state before the transaction started.
       * @type {Map<Number,Number>}
       */
      this.beforeState = getStateVector(doc.store);
      /**
       * Holds the state after the transaction.
       * @type {Map<Number,Number>}
       */
      this.afterState = new Map();
      /**
       * All types that were directly modified (property added or child
       * inserted/deleted). New types are not included in this Set.
       * Maps from type to parentSubs (`item.parentSub = null` for YArray)
       * @type {Map<AbstractType<YEvent<any>>,Set<String|null>>}
       */
      this.changed = new Map();
      /**
       * Stores the events for the types that observe also child elements.
       * It is mainly used by `observeDeep`.
       * @type {Map<AbstractType<YEvent<any>>,Array<YEvent<any>>>}
       */
      this.changedParentTypes = new Map();
      /**
       * @type {Array<AbstractStruct>}
       */
      this._mergeStructs = [];
      /**
       * @type {any}
       */
      this.origin = origin;
      /**
       * Stores meta information on the transaction
       * @type {Map<any,any>}
       */
      this.meta = new Map();
      /**
       * Whether this change originates from this doc.
       * @type {boolean}
       */
      this.local = local;
      /**
       * @type {Set<Doc>}
       */
      this.subdocsAdded = new Set();
      /**
       * @type {Set<Doc>}
       */
      this.subdocsRemoved = new Set();
      /**
       * @type {Set<Doc>}
       */
      this.subdocsLoaded = new Set();
    }
  }

  /**
   * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
   * @param {Transaction} transaction
   * @return {boolean} Whether data was written.
   */
  const writeUpdateMessageFromTransaction = (encoder, transaction) => {
    if (transaction.deleteSet.clients.size === 0 && !any(transaction.afterState, (clock, client) => transaction.beforeState.get(client) !== clock)) {
      return false
    }
    sortAndMergeDeleteSet(transaction.deleteSet);
    writeStructsFromTransaction(encoder, transaction);
    writeDeleteSet(encoder, transaction.deleteSet);
    return true
  };

  /**
   * If `type.parent` was added in current transaction, `type` technically
   * did not change, it was just added and we should not fire events for `type`.
   *
   * @param {Transaction} transaction
   * @param {AbstractType<YEvent<any>>} type
   * @param {string|null} parentSub
   */
  const addChangedTypeToTransaction = (transaction, type, parentSub) => {
    const item = type._item;
    if (item === null || (item.id.clock < (transaction.beforeState.get(item.id.client) || 0) && !item.deleted)) {
      setIfUndefined(transaction.changed, type, create$4).add(parentSub);
    }
  };

  /**
   * @param {Array<AbstractStruct>} structs
   * @param {number} pos
   */
  const tryToMergeWithLeft = (structs, pos) => {
    const left = structs[pos - 1];
    const right = structs[pos];
    if (left.deleted === right.deleted && left.constructor === right.constructor) {
      if (left.mergeWith(right)) {
        structs.splice(pos, 1);
        if (right instanceof Item && right.parentSub !== null && /** @type {AbstractType<any>} */ (right.parent)._map.get(right.parentSub) === right) {
          /** @type {AbstractType<any>} */ (right.parent)._map.set(right.parentSub, /** @type {Item} */ (left));
        }
      }
    }
  };

  /**
   * @param {DeleteSet} ds
   * @param {StructStore} store
   * @param {function(Item):boolean} gcFilter
   */
  const tryGcDeleteSet = (ds, store, gcFilter) => {
    for (const [client, deleteItems] of ds.clients.entries()) {
      const structs = /** @type {Array<GC|Item>} */ (store.clients.get(client));
      for (let di = deleteItems.length - 1; di >= 0; di--) {
        const deleteItem = deleteItems[di];
        const endDeleteItemClock = deleteItem.clock + deleteItem.len;
        for (
          let si = findIndexSS(structs, deleteItem.clock), struct = structs[si];
          si < structs.length && struct.id.clock < endDeleteItemClock;
          struct = structs[++si]
        ) {
          const struct = structs[si];
          if (deleteItem.clock + deleteItem.len <= struct.id.clock) {
            break
          }
          if (struct instanceof Item && struct.deleted && !struct.keep && gcFilter(struct)) {
            struct.gc(store, false);
          }
        }
      }
    }
  };

  /**
   * @param {DeleteSet} ds
   * @param {StructStore} store
   */
  const tryMergeDeleteSet = (ds, store) => {
    // try to merge deleted / gc'd items
    // merge from right to left for better efficiecy and so we don't miss any merge targets
    ds.clients.forEach((deleteItems, client) => {
      const structs = /** @type {Array<GC|Item>} */ (store.clients.get(client));
      for (let di = deleteItems.length - 1; di >= 0; di--) {
        const deleteItem = deleteItems[di];
        // start with merging the item next to the last deleted item
        const mostRightIndexToCheck = min(structs.length - 1, 1 + findIndexSS(structs, deleteItem.clock + deleteItem.len - 1));
        for (
          let si = mostRightIndexToCheck, struct = structs[si];
          si > 0 && struct.id.clock >= deleteItem.clock;
          struct = structs[--si]
        ) {
          tryToMergeWithLeft(structs, si);
        }
      }
    });
  };

  /**
   * @param {Array<Transaction>} transactionCleanups
   * @param {number} i
   */
  const cleanupTransactions = (transactionCleanups, i) => {
    if (i < transactionCleanups.length) {
      const transaction = transactionCleanups[i];
      const doc = transaction.doc;
      const store = doc.store;
      const ds = transaction.deleteSet;
      const mergeStructs = transaction._mergeStructs;
      try {
        sortAndMergeDeleteSet(ds);
        transaction.afterState = getStateVector(transaction.doc.store);
        doc._transaction = null;
        doc.emit('beforeObserverCalls', [transaction, doc]);
        /**
         * An array of event callbacks.
         *
         * Each callback is called even if the other ones throw errors.
         *
         * @type {Array<function():void>}
         */
        const fs = [];
        // observe events on changed types
        transaction.changed.forEach((subs, itemtype) =>
          fs.push(() => {
            if (itemtype._item === null || !itemtype._item.deleted) {
              itemtype._callObserver(transaction, subs);
            }
          })
        );
        fs.push(() => {
          // deep observe events
          transaction.changedParentTypes.forEach((events, type) =>
            fs.push(() => {
              // We need to think about the possibility that the user transforms the
              // Y.Doc in the event.
              if (type._item === null || !type._item.deleted) {
                events = events
                  .filter(event =>
                    event.target._item === null || !event.target._item.deleted
                  );
                events
                  .forEach(event => {
                    event.currentTarget = type;
                  });
                // sort events by path length so that top-level events are fired first.
                events
                  .sort((event1, event2) => event1.path.length - event2.path.length);
                // We don't need to check for events.length
                // because we know it has at least one element
                callEventHandlerListeners(type._dEH, events, transaction);
              }
            })
          );
          fs.push(() => doc.emit('afterTransaction', [transaction, doc]));
        });
        callAll(fs, []);
      } finally {
        // Replace deleted items with ItemDeleted / GC.
        // This is where content is actually remove from the Yjs Doc.
        if (doc.gc) {
          tryGcDeleteSet(ds, store, doc.gcFilter);
        }
        tryMergeDeleteSet(ds, store);

        // on all affected store.clients props, try to merge
        transaction.afterState.forEach((clock, client) => {
          const beforeClock = transaction.beforeState.get(client) || 0;
          if (beforeClock !== clock) {
            const structs = /** @type {Array<GC|Item>} */ (store.clients.get(client));
            // we iterate from right to left so we can safely remove entries
            const firstChangePos = max(findIndexSS(structs, beforeClock), 1);
            for (let i = structs.length - 1; i >= firstChangePos; i--) {
              tryToMergeWithLeft(structs, i);
            }
          }
        });
        // try to merge mergeStructs
        // @todo: it makes more sense to transform mergeStructs to a DS, sort it, and merge from right to left
        //        but at the moment DS does not handle duplicates
        for (let i = 0; i < mergeStructs.length; i++) {
          const { client, clock } = mergeStructs[i].id;
          const structs = /** @type {Array<GC|Item>} */ (store.clients.get(client));
          const replacedStructPos = findIndexSS(structs, clock);
          if (replacedStructPos + 1 < structs.length) {
            tryToMergeWithLeft(structs, replacedStructPos + 1);
          }
          if (replacedStructPos > 0) {
            tryToMergeWithLeft(structs, replacedStructPos);
          }
        }
        if (!transaction.local && transaction.afterState.get(doc.clientID) !== transaction.beforeState.get(doc.clientID)) {
          print(ORANGE, BOLD, '[yjs] ', UNBOLD, RED, 'Changed the client-id because another client seems to be using it.');
          doc.clientID = generateNewClientId();
        }
        // @todo Merge all the transactions into one and provide send the data as a single update message
        doc.emit('afterTransactionCleanup', [transaction, doc]);
        if (doc._observers.has('update')) {
          const encoder = new UpdateEncoderV1();
          const hasContent = writeUpdateMessageFromTransaction(encoder, transaction);
          if (hasContent) {
            doc.emit('update', [encoder.toUint8Array(), transaction.origin, doc, transaction]);
          }
        }
        if (doc._observers.has('updateV2')) {
          const encoder = new UpdateEncoderV2();
          const hasContent = writeUpdateMessageFromTransaction(encoder, transaction);
          if (hasContent) {
            doc.emit('updateV2', [encoder.toUint8Array(), transaction.origin, doc, transaction]);
          }
        }
        const { subdocsAdded, subdocsLoaded, subdocsRemoved } = transaction;
        if (subdocsAdded.size > 0 || subdocsRemoved.size > 0 || subdocsLoaded.size > 0) {
          subdocsAdded.forEach(subdoc => {
            subdoc.clientID = doc.clientID;
            if (subdoc.collectionid == null) {
              subdoc.collectionid = doc.collectionid;
            }
            doc.subdocs.add(subdoc);
          });
          subdocsRemoved.forEach(subdoc => doc.subdocs.delete(subdoc));
          doc.emit('subdocs', [{ loaded: subdocsLoaded, added: subdocsAdded, removed: subdocsRemoved }, doc, transaction]);
          subdocsRemoved.forEach(subdoc => subdoc.destroy());
        }

        if (transactionCleanups.length <= i + 1) {
          doc._transactionCleanups = [];
          doc.emit('afterAllTransactions', [doc, transactionCleanups]);
        } else {
          cleanupTransactions(transactionCleanups, i + 1);
        }
      }
    }
  };

  /**
   * Implements the functionality of `y.transact(()=>{..})`
   *
   * @param {Doc} doc
   * @param {function(Transaction):void} f
   * @param {any} [origin=true]
   *
   * @function
   */
  const transact = (doc, f, origin = null, local = true) => {
    const transactionCleanups = doc._transactionCleanups;
    let initialCall = false;
    if (doc._transaction === null) {
      initialCall = true;
      doc._transaction = new Transaction$1(doc, origin, local);
      transactionCleanups.push(doc._transaction);
      if (transactionCleanups.length === 1) {
        doc.emit('beforeAllTransactions', [doc]);
      }
      doc.emit('beforeTransaction', [doc._transaction, doc]);
    }
    try {
      f(doc._transaction);
    } finally {
      if (initialCall && transactionCleanups[0] === doc._transaction) {
        // The first transaction ended, now process observer calls.
        // Observer call may create new transactions for which we need to call the observers and do cleanup.
        // We don't want to nest these calls, so we execute these calls one after
        // another.
        // Also we need to ensure that all cleanups are called, even if the
        // observes throw errors.
        // This file is full of hacky try {} finally {} blocks to ensure that an
        // event can throw errors and also that the cleanup is called.
        cleanupTransactions(transactionCleanups, 0);
      }
    }
  };

  class StackItem {
    /**
     * @param {DeleteSet} deletions
     * @param {DeleteSet} insertions
     */
    constructor (deletions, insertions) {
      this.insertions = insertions;
      this.deletions = deletions;
      /**
       * Use this to save and restore metadata like selection range
       */
      this.meta = new Map();
    }
  }
  /**
   * @param {Transaction} tr
   * @param {UndoManager} um
   * @param {StackItem} stackItem
   */
  const clearUndoManagerStackItem = (tr, um, stackItem) => {
    iterateDeletedStructs(tr, stackItem.deletions, item => {
      if (item instanceof Item && um.scope.some(type => isParentOf(type, item))) {
        keepItem(item, false);
      }
    });
  };

  /**
   * @param {UndoManager} undoManager
   * @param {Array<StackItem>} stack
   * @param {string} eventType
   * @return {StackItem?}
   */
  const popStackItem = (undoManager, stack, eventType) => {
    /**
     * Whether a change happened
     * @type {StackItem?}
     */
    let result = null;
    /**
     * Keep a reference to the transaction so we can fire the event with the changedParentTypes
     * @type {any}
     */
    let _tr = null;
    const doc = undoManager.doc;
    const scope = undoManager.scope;
    transact(doc, transaction => {
      while (stack.length > 0 && result === null) {
        const store = doc.store;
        const stackItem = /** @type {StackItem} */ (stack.pop());
        /**
         * @type {Set<Item>}
         */
        const itemsToRedo = new Set();
        /**
         * @type {Array<Item>}
         */
        const itemsToDelete = [];
        let performedChange = false;
        iterateDeletedStructs(transaction, stackItem.insertions, struct => {
          if (struct instanceof Item) {
            if (struct.redone !== null) {
              let { item, diff } = followRedone(store, struct.id);
              if (diff > 0) {
                item = getItemCleanStart(transaction, createID(item.id.client, item.id.clock + diff));
              }
              struct = item;
            }
            if (!struct.deleted && scope.some(type => isParentOf(type, /** @type {Item} */ (struct)))) {
              itemsToDelete.push(struct);
            }
          }
        });
        iterateDeletedStructs(transaction, stackItem.deletions, struct => {
          if (
            struct instanceof Item &&
            scope.some(type => isParentOf(type, struct)) &&
            // Never redo structs in stackItem.insertions because they were created and deleted in the same capture interval.
            !isDeleted(stackItem.insertions, struct.id)
          ) {
            itemsToRedo.add(struct);
          }
        });
        itemsToRedo.forEach(struct => {
          performedChange = redoItem(transaction, struct, itemsToRedo, stackItem.insertions, undoManager.ignoreRemoteMapChanges) !== null || performedChange;
        });
        // We want to delete in reverse order so that children are deleted before
        // parents, so we have more information available when items are filtered.
        for (let i = itemsToDelete.length - 1; i >= 0; i--) {
          const item = itemsToDelete[i];
          if (undoManager.deleteFilter(item)) {
            item.delete(transaction);
            performedChange = true;
          }
        }
        result = performedChange ? stackItem : null;
      }
      transaction.changed.forEach((subProps, type) => {
        // destroy search marker if necessary
        if (subProps.has(null) && type._searchMarker) {
          type._searchMarker.length = 0;
        }
      });
      _tr = transaction;
    }, undoManager);
    if (result != null) {
      const changedParentTypes = _tr.changedParentTypes;
      undoManager.emit('stack-item-popped', [{ stackItem: result, type: eventType, changedParentTypes }, undoManager]);
    }
    return result
  };

  /**
   * @typedef {Object} UndoManagerOptions
   * @property {number} [UndoManagerOptions.captureTimeout=500]
   * @property {function(Transaction):boolean} [UndoManagerOptions.captureTransaction] Do not capture changes of a Transaction if result false.
   * @property {function(Item):boolean} [UndoManagerOptions.deleteFilter=()=>true] Sometimes
   * it is necessary to filter whan an Undo/Redo operation can delete. If this
   * filter returns false, the type/item won't be deleted even it is in the
   * undo/redo scope.
   * @property {Set<any>} [UndoManagerOptions.trackedOrigins=new Set([null])]
   * @property {boolean} [ignoreRemoteMapChanges] Experimental. By default, the UndoManager will never overwrite remote changes. Enable this property to enable overwriting remote changes on key-value changes (Y.Map, properties on Y.Xml, etc..).
   * @property {Doc} [doc] The document that this UndoManager operates on. Only needed if typeScope is empty.
   */

  /**
   * Fires 'stack-item-added' event when a stack item was added to either the undo- or
   * the redo-stack. You may store additional stack information via the
   * metadata property on `event.stackItem.meta` (it is a `Map` of metadata properties).
   * Fires 'stack-item-popped' event when a stack item was popped from either the
   * undo- or the redo-stack. You may restore the saved stack information from `event.stackItem.meta`.
   *
   * @extends {Observable<'stack-item-added'|'stack-item-popped'|'stack-cleared'|'stack-item-updated'>}
   */
  class UndoManager extends Observable {
    /**
     * @param {AbstractType<any>|Array<AbstractType<any>>} typeScope Accepts either a single type, or an array of types
     * @param {UndoManagerOptions} options
     */
    constructor (typeScope, {
      captureTimeout = 500,
      captureTransaction = tr => true,
      deleteFilter = () => true,
      trackedOrigins = new Set([null]),
      ignoreRemoteMapChanges = false,
      doc = /** @type {Doc} */ (isArray(typeScope) ? typeScope[0].doc : typeScope.doc)
    } = {}) {
      super();
      /**
       * @type {Array<AbstractType<any>>}
       */
      this.scope = [];
      this.addToScope(typeScope);
      this.deleteFilter = deleteFilter;
      trackedOrigins.add(this);
      this.trackedOrigins = trackedOrigins;
      this.captureTransaction = captureTransaction;
      /**
       * @type {Array<StackItem>}
       */
      this.undoStack = [];
      /**
       * @type {Array<StackItem>}
       */
      this.redoStack = [];
      /**
       * Whether the client is currently undoing (calling UndoManager.undo)
       *
       * @type {boolean}
       */
      this.undoing = false;
      this.redoing = false;
      this.doc = doc;
      this.lastChange = 0;
      this.ignoreRemoteMapChanges = ignoreRemoteMapChanges;
      /**
       * @param {Transaction} transaction
       */
      this.afterTransactionHandler = transaction => {
        // Only track certain transactions
        if (
          !this.captureTransaction(transaction) ||
          !this.scope.some(type => transaction.changedParentTypes.has(type)) ||
          (!this.trackedOrigins.has(transaction.origin) && (!transaction.origin || !this.trackedOrigins.has(transaction.origin.constructor)))
        ) {
          return
        }
        const undoing = this.undoing;
        const redoing = this.redoing;
        const stack = undoing ? this.redoStack : this.undoStack;
        if (undoing) {
          this.stopCapturing(); // next undo should not be appended to last stack item
        } else if (!redoing) {
          // neither undoing nor redoing: delete redoStack
          this.clear(false, true);
        }
        const insertions = new DeleteSet();
        transaction.afterState.forEach((endClock, client) => {
          const startClock = transaction.beforeState.get(client) || 0;
          const len = endClock - startClock;
          if (len > 0) {
            addToDeleteSet(insertions, client, startClock, len);
          }
        });
        const now = getUnixTime();
        let didAdd = false;
        if (now - this.lastChange < captureTimeout && stack.length > 0 && !undoing && !redoing) {
          // append change to last stack op
          const lastOp = stack[stack.length - 1];
          lastOp.deletions = mergeDeleteSets([lastOp.deletions, transaction.deleteSet]);
          lastOp.insertions = mergeDeleteSets([lastOp.insertions, insertions]);
        } else {
          // create a new stack op
          stack.push(new StackItem(transaction.deleteSet, insertions));
          didAdd = true;
        }
        if (!undoing && !redoing) {
          this.lastChange = now;
        }
        // make sure that deleted structs are not gc'd
        iterateDeletedStructs(transaction, transaction.deleteSet, /** @param {Item|GC} item */ item => {
          if (item instanceof Item && this.scope.some(type => isParentOf(type, item))) {
            keepItem(item, true);
          }
        });
        const changeEvent = [{ stackItem: stack[stack.length - 1], origin: transaction.origin, type: undoing ? 'redo' : 'undo', changedParentTypes: transaction.changedParentTypes }, this];
        if (didAdd) {
          this.emit('stack-item-added', changeEvent);
        } else {
          this.emit('stack-item-updated', changeEvent);
        }
      };
      this.doc.on('afterTransaction', this.afterTransactionHandler);
      this.doc.on('destroy', () => {
        this.destroy();
      });
    }

    /**
     * @param {Array<AbstractType<any>> | AbstractType<any>} ytypes
     */
    addToScope (ytypes) {
      ytypes = isArray(ytypes) ? ytypes : [ytypes];
      ytypes.forEach(ytype => {
        if (this.scope.every(yt => yt !== ytype)) {
          this.scope.push(ytype);
        }
      });
    }

    /**
     * @param {any} origin
     */
    addTrackedOrigin (origin) {
      this.trackedOrigins.add(origin);
    }

    /**
     * @param {any} origin
     */
    removeTrackedOrigin (origin) {
      this.trackedOrigins.delete(origin);
    }

    clear (clearUndoStack = true, clearRedoStack = true) {
      if ((clearUndoStack && this.canUndo()) || (clearRedoStack && this.canRedo())) {
        this.doc.transact(tr => {
          if (clearUndoStack) {
            this.undoStack.forEach(item => clearUndoManagerStackItem(tr, this, item));
            this.undoStack = [];
          }
          if (clearRedoStack) {
            this.redoStack.forEach(item => clearUndoManagerStackItem(tr, this, item));
            this.redoStack = [];
          }
          this.emit('stack-cleared', [{ undoStackCleared: clearUndoStack, redoStackCleared: clearRedoStack }]);
        });
      }
    }

    /**
     * UndoManager merges Undo-StackItem if they are created within time-gap
     * smaller than `options.captureTimeout`. Call `um.stopCapturing()` so that the next
     * StackItem won't be merged.
     *
     *
     * @example
     *     // without stopCapturing
     *     ytext.insert(0, 'a')
     *     ytext.insert(1, 'b')
     *     um.undo()
     *     ytext.toString() // => '' (note that 'ab' was removed)
     *     // with stopCapturing
     *     ytext.insert(0, 'a')
     *     um.stopCapturing()
     *     ytext.insert(0, 'b')
     *     um.undo()
     *     ytext.toString() // => 'a' (note that only 'b' was removed)
     *
     */
    stopCapturing () {
      this.lastChange = 0;
    }

    /**
     * Undo last changes on type.
     *
     * @return {StackItem?} Returns StackItem if a change was applied
     */
    undo () {
      this.undoing = true;
      let res;
      try {
        res = popStackItem(this, this.undoStack, 'undo');
      } finally {
        this.undoing = false;
      }
      return res
    }

    /**
     * Redo last undo operation.
     *
     * @return {StackItem?} Returns StackItem if a change was applied
     */
    redo () {
      this.redoing = true;
      let res;
      try {
        res = popStackItem(this, this.redoStack, 'redo');
      } finally {
        this.redoing = false;
      }
      return res
    }

    /**
     * Are undo steps available?
     *
     * @return {boolean} `true` if undo is possible
     */
    canUndo () {
      return this.undoStack.length > 0
    }

    /**
     * Are redo steps available?
     *
     * @return {boolean} `true` if redo is possible
     */
    canRedo () {
      return this.redoStack.length > 0
    }

    destroy () {
      this.trackedOrigins.delete(this);
      this.doc.off('afterTransaction', this.afterTransactionHandler);
      super.destroy();
    }
  }

  /**
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   */
  function * lazyStructReaderGenerator (decoder) {
    const numOfStateUpdates = readVarUint(decoder.restDecoder);
    for (let i = 0; i < numOfStateUpdates; i++) {
      const numberOfStructs = readVarUint(decoder.restDecoder);
      const client = decoder.readClient();
      let clock = readVarUint(decoder.restDecoder);
      for (let i = 0; i < numberOfStructs; i++) {
        const info = decoder.readInfo();
        // @todo use switch instead of ifs
        if (info === 10) {
          const len = readVarUint(decoder.restDecoder);
          yield new Skip(createID(client, clock), len);
          clock += len;
        } else if ((BITS5 & info) !== 0) {
          const cantCopyParentInfo = (info & (BIT7 | BIT8)) === 0;
          // If parent = null and neither left nor right are defined, then we know that `parent` is child of `y`
          // and we read the next string as parentYKey.
          // It indicates how we store/retrieve parent from `y.share`
          // @type {string|null}
          const struct = new Item(
            createID(client, clock),
            null, // left
            (info & BIT8) === BIT8 ? decoder.readLeftID() : null, // origin
            null, // right
            (info & BIT7) === BIT7 ? decoder.readRightID() : null, // right origin
            // @ts-ignore Force writing a string here.
            cantCopyParentInfo ? (decoder.readParentInfo() ? decoder.readString() : decoder.readLeftID()) : null, // parent
            cantCopyParentInfo && (info & BIT6) === BIT6 ? decoder.readString() : null, // parentSub
            readItemContent(decoder, info) // item content
          );
          yield struct;
          clock += struct.length;
        } else {
          const len = decoder.readLen();
          yield new GC(createID(client, clock), len);
          clock += len;
        }
      }
    }
  }

  class LazyStructReader {
    /**
     * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
     * @param {boolean} filterSkips
     */
    constructor (decoder, filterSkips) {
      this.gen = lazyStructReaderGenerator(decoder);
      /**
       * @type {null | Item | Skip | GC}
       */
      this.curr = null;
      this.done = false;
      this.filterSkips = filterSkips;
      this.next();
    }

    /**
     * @return {Item | GC | Skip |null}
     */
    next () {
      // ignore "Skip" structs
      do {
        this.curr = this.gen.next().value || null;
      } while (this.filterSkips && this.curr !== null && this.curr.constructor === Skip)
      return this.curr
    }
  }

  class LazyStructWriter {
    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     */
    constructor (encoder) {
      this.currClient = 0;
      this.startClock = 0;
      this.written = 0;
      this.encoder = encoder;
      /**
       * We want to write operations lazily, but also we need to know beforehand how many operations we want to write for each client.
       *
       * This kind of meta-information (#clients, #structs-per-client-written) is written to the restEncoder.
       *
       * We fragment the restEncoder and store a slice of it per-client until we know how many clients there are.
       * When we flush (toUint8Array) we write the restEncoder using the fragments and the meta-information.
       *
       * @type {Array<{ written: number, restEncoder: Uint8Array }>}
       */
      this.clientStructs = [];
    }
  }

  /**
   * @param {Array<Uint8Array>} updates
   * @return {Uint8Array}
   */
  const mergeUpdates = updates => mergeUpdatesV2(updates, UpdateDecoderV1, UpdateEncoderV1);

  /**
   * This method is intended to slice any kind of struct and retrieve the right part.
   * It does not handle side-effects, so it should only be used by the lazy-encoder.
   *
   * @param {Item | GC | Skip} left
   * @param {number} diff
   * @return {Item | GC}
   */
  const sliceStruct = (left, diff) => {
    if (left.constructor === GC) {
      const { client, clock } = left.id;
      return new GC(createID(client, clock + diff), left.length - diff)
    } else if (left.constructor === Skip) {
      const { client, clock } = left.id;
      return new Skip(createID(client, clock + diff), left.length - diff)
    } else {
      const leftItem = /** @type {Item} */ (left);
      const { client, clock } = leftItem.id;
      return new Item(
        createID(client, clock + diff),
        null,
        createID(client, clock + diff - 1),
        null,
        leftItem.rightOrigin,
        leftItem.parent,
        leftItem.parentSub,
        leftItem.content.splice(diff)
      )
    }
  };

  /**
   *
   * This function works similarly to `readUpdateV2`.
   *
   * @param {Array<Uint8Array>} updates
   * @param {typeof UpdateDecoderV1 | typeof UpdateDecoderV2} [YDecoder]
   * @param {typeof UpdateEncoderV1 | typeof UpdateEncoderV2} [YEncoder]
   * @return {Uint8Array}
   */
  const mergeUpdatesV2 = (updates, YDecoder = UpdateDecoderV2, YEncoder = UpdateEncoderV2) => {
    if (updates.length === 1) {
      return updates[0]
    }
    const updateDecoders = updates.map(update => new YDecoder(createDecoder(update)));
    let lazyStructDecoders = updateDecoders.map(decoder => new LazyStructReader(decoder, true));

    /**
     * @todo we don't need offset because we always slice before
     * @type {null | { struct: Item | GC | Skip, offset: number }}
     */
    let currWrite = null;

    const updateEncoder = new YEncoder();
    // write structs lazily
    const lazyStructEncoder = new LazyStructWriter(updateEncoder);

    // Note: We need to ensure that all lazyStructDecoders are fully consumed
    // Note: Should merge document updates whenever possible - even from different updates
    // Note: Should handle that some operations cannot be applied yet ()

    while (true) {
      // Write higher clients first ⇒ sort by clientID & clock and remove decoders without content
      lazyStructDecoders = lazyStructDecoders.filter(dec => dec.curr !== null);
      lazyStructDecoders.sort(
        /** @type {function(any,any):number} */ (dec1, dec2) => {
          if (dec1.curr.id.client === dec2.curr.id.client) {
            const clockDiff = dec1.curr.id.clock - dec2.curr.id.clock;
            if (clockDiff === 0) {
              // @todo remove references to skip since the structDecoders must filter Skips.
              return dec1.curr.constructor === dec2.curr.constructor
                ? 0
                : dec1.curr.constructor === Skip ? 1 : -1 // we are filtering skips anyway.
            } else {
              return clockDiff
            }
          } else {
            return dec2.curr.id.client - dec1.curr.id.client
          }
        }
      );
      if (lazyStructDecoders.length === 0) {
        break
      }
      const currDecoder = lazyStructDecoders[0];
      // write from currDecoder until the next operation is from another client or if filler-struct
      // then we need to reorder the decoders and find the next operation to write
      const firstClient = /** @type {Item | GC} */ (currDecoder.curr).id.client;

      if (currWrite !== null) {
        let curr = /** @type {Item | GC | null} */ (currDecoder.curr);
        let iterated = false;

        // iterate until we find something that we haven't written already
        // remember: first the high client-ids are written
        while (curr !== null && curr.id.clock + curr.length <= currWrite.struct.id.clock + currWrite.struct.length && curr.id.client >= currWrite.struct.id.client) {
          curr = currDecoder.next();
          iterated = true;
        }
        if (
          curr === null || // current decoder is empty
          curr.id.client !== firstClient || // check whether there is another decoder that has has updates from `firstClient`
          (iterated && curr.id.clock > currWrite.struct.id.clock + currWrite.struct.length) // the above while loop was used and we are potentially missing updates
        ) {
          continue
        }

        if (firstClient !== currWrite.struct.id.client) {
          writeStructToLazyStructWriter(lazyStructEncoder, currWrite.struct, currWrite.offset);
          currWrite = { struct: curr, offset: 0 };
          currDecoder.next();
        } else {
          if (currWrite.struct.id.clock + currWrite.struct.length < curr.id.clock) {
            // @todo write currStruct & set currStruct = Skip(clock = currStruct.id.clock + currStruct.length, length = curr.id.clock - self.clock)
            if (currWrite.struct.constructor === Skip) {
              // extend existing skip
              currWrite.struct.length = curr.id.clock + curr.length - currWrite.struct.id.clock;
            } else {
              writeStructToLazyStructWriter(lazyStructEncoder, currWrite.struct, currWrite.offset);
              const diff = curr.id.clock - currWrite.struct.id.clock - currWrite.struct.length;
              /**
               * @type {Skip}
               */
              const struct = new Skip(createID(firstClient, currWrite.struct.id.clock + currWrite.struct.length), diff);
              currWrite = { struct, offset: 0 };
            }
          } else { // if (currWrite.struct.id.clock + currWrite.struct.length >= curr.id.clock) {
            const diff = currWrite.struct.id.clock + currWrite.struct.length - curr.id.clock;
            if (diff > 0) {
              if (currWrite.struct.constructor === Skip) {
                // prefer to slice Skip because the other struct might contain more information
                currWrite.struct.length -= diff;
              } else {
                curr = sliceStruct(curr, diff);
              }
            }
            if (!currWrite.struct.mergeWith(/** @type {any} */ (curr))) {
              writeStructToLazyStructWriter(lazyStructEncoder, currWrite.struct, currWrite.offset);
              currWrite = { struct: curr, offset: 0 };
              currDecoder.next();
            }
          }
        }
      } else {
        currWrite = { struct: /** @type {Item | GC} */ (currDecoder.curr), offset: 0 };
        currDecoder.next();
      }
      for (
        let next = currDecoder.curr;
        next !== null && next.id.client === firstClient && next.id.clock === currWrite.struct.id.clock + currWrite.struct.length && next.constructor !== Skip;
        next = currDecoder.next()
      ) {
        writeStructToLazyStructWriter(lazyStructEncoder, currWrite.struct, currWrite.offset);
        currWrite = { struct: next, offset: 0 };
      }
    }
    if (currWrite !== null) {
      writeStructToLazyStructWriter(lazyStructEncoder, currWrite.struct, currWrite.offset);
      currWrite = null;
    }
    finishLazyStructWriting(lazyStructEncoder);

    const dss = updateDecoders.map(decoder => readDeleteSet(decoder));
    const ds = mergeDeleteSets(dss);
    writeDeleteSet(updateEncoder, ds);
    return updateEncoder.toUint8Array()
  };

  /**
   * @param {Uint8Array} update
   * @param {Uint8Array} sv
   * @param {typeof UpdateDecoderV1 | typeof UpdateDecoderV2} [YDecoder]
   * @param {typeof UpdateEncoderV1 | typeof UpdateEncoderV2} [YEncoder]
   */
  const diffUpdateV2 = (update, sv, YDecoder = UpdateDecoderV2, YEncoder = UpdateEncoderV2) => {
    const state = decodeStateVector(sv);
    const encoder = new YEncoder();
    const lazyStructWriter = new LazyStructWriter(encoder);
    const decoder = new YDecoder(createDecoder(update));
    const reader = new LazyStructReader(decoder, false);
    while (reader.curr) {
      const curr = reader.curr;
      const currClient = curr.id.client;
      const svClock = state.get(currClient) || 0;
      if (reader.curr.constructor === Skip) {
        // the first written struct shouldn't be a skip
        reader.next();
        continue
      }
      if (curr.id.clock + curr.length > svClock) {
        writeStructToLazyStructWriter(lazyStructWriter, curr, max(svClock - curr.id.clock, 0));
        reader.next();
        while (reader.curr && reader.curr.id.client === currClient) {
          writeStructToLazyStructWriter(lazyStructWriter, reader.curr, 0);
          reader.next();
        }
      } else {
        // read until something new comes up
        while (reader.curr && reader.curr.id.client === currClient && reader.curr.id.clock + reader.curr.length <= svClock) {
          reader.next();
        }
      }
    }
    finishLazyStructWriting(lazyStructWriter);
    // write ds
    const ds = readDeleteSet(decoder);
    writeDeleteSet(encoder, ds);
    return encoder.toUint8Array()
  };

  /**
   * @param {LazyStructWriter} lazyWriter
   */
  const flushLazyStructWriter = lazyWriter => {
    if (lazyWriter.written > 0) {
      lazyWriter.clientStructs.push({ written: lazyWriter.written, restEncoder: toUint8Array(lazyWriter.encoder.restEncoder) });
      lazyWriter.encoder.restEncoder = createEncoder();
      lazyWriter.written = 0;
    }
  };

  /**
   * @param {LazyStructWriter} lazyWriter
   * @param {Item | GC} struct
   * @param {number} offset
   */
  const writeStructToLazyStructWriter = (lazyWriter, struct, offset) => {
    // flush curr if we start another client
    if (lazyWriter.written > 0 && lazyWriter.currClient !== struct.id.client) {
      flushLazyStructWriter(lazyWriter);
    }
    if (lazyWriter.written === 0) {
      lazyWriter.currClient = struct.id.client;
      // write next client
      lazyWriter.encoder.writeClient(struct.id.client);
      // write startClock
      writeVarUint(lazyWriter.encoder.restEncoder, struct.id.clock + offset);
    }
    struct.write(lazyWriter.encoder, offset);
    lazyWriter.written++;
  };
  /**
   * Call this function when we collected all parts and want to
   * put all the parts together. After calling this method,
   * you can continue using the UpdateEncoder.
   *
   * @param {LazyStructWriter} lazyWriter
   */
  const finishLazyStructWriting = (lazyWriter) => {
    flushLazyStructWriter(lazyWriter);

    // this is a fresh encoder because we called flushCurr
    const restEncoder = lazyWriter.encoder.restEncoder;

    /**
     * Now we put all the fragments together.
     * This works similarly to `writeClientsStructs`
     */

    // write # states that were updated - i.e. the clients
    writeVarUint(restEncoder, lazyWriter.clientStructs.length);

    for (let i = 0; i < lazyWriter.clientStructs.length; i++) {
      const partStructs = lazyWriter.clientStructs[i];
      /**
       * Works similarly to `writeStructs`
       */
      // write # encoded structs
      writeVarUint(restEncoder, partStructs.written);
      // write the rest of the fragment
      writeUint8Array(restEncoder, partStructs.restEncoder);
    }
  };

  /**
   * @param {Uint8Array} update
   * @param {typeof UpdateDecoderV2 | typeof UpdateDecoderV1} YDecoder
   * @param {typeof UpdateEncoderV2 | typeof UpdateEncoderV1 } YEncoder
   */
  const convertUpdateFormat = (update, YDecoder, YEncoder) => {
    const updateDecoder = new YDecoder(createDecoder(update));
    const lazyDecoder = new LazyStructReader(updateDecoder, false);
    const updateEncoder = new YEncoder();
    const lazyWriter = new LazyStructWriter(updateEncoder);

    for (let curr = lazyDecoder.curr; curr !== null; curr = lazyDecoder.next()) {
      writeStructToLazyStructWriter(lazyWriter, curr, 0);
    }
    finishLazyStructWriting(lazyWriter);
    const ds = readDeleteSet(updateDecoder);
    writeDeleteSet(updateEncoder, ds);
    return updateEncoder.toUint8Array()
  };

  /**
   * @param {Uint8Array} update
   */
  const convertUpdateFormatV2ToV1 = update => convertUpdateFormat(update, UpdateDecoderV2, UpdateEncoderV1);

  /**
   * @template {AbstractType<any>} T
   * YEvent describes the changes on a YType.
   */
  class YEvent {
    /**
     * @param {T} target The changed type.
     * @param {Transaction} transaction
     */
    constructor (target, transaction) {
      /**
       * The type on which this event was created on.
       * @type {T}
       */
      this.target = target;
      /**
       * The current target on which the observe callback is called.
       * @type {AbstractType<any>}
       */
      this.currentTarget = target;
      /**
       * The transaction that triggered this event.
       * @type {Transaction}
       */
      this.transaction = transaction;
      /**
       * @type {Object|null}
       */
      this._changes = null;
      /**
       * @type {null | Map<string, { action: 'add' | 'update' | 'delete', oldValue: any, newValue: any }>}
       */
      this._keys = null;
      /**
       * @type {null | Array<{ insert?: string | Array<any> | object | AbstractType<any>, retain?: number, delete?: number, attributes?: Object<string, any> }>}
       */
      this._delta = null;
    }

    /**
     * Computes the path from `y` to the changed type.
     *
     * @todo v14 should standardize on path: Array<{parent, index}> because that is easier to work with.
     *
     * The following property holds:
     * @example
     *   let type = y
     *   event.path.forEach(dir => {
     *     type = type.get(dir)
     *   })
     *   type === event.target // => true
     */
    get path () {
      // @ts-ignore _item is defined because target is integrated
      return getPathTo(this.currentTarget, this.target)
    }

    /**
     * Check if a struct is deleted by this event.
     *
     * In contrast to change.deleted, this method also returns true if the struct was added and then deleted.
     *
     * @param {AbstractStruct} struct
     * @return {boolean}
     */
    deletes (struct) {
      return isDeleted(this.transaction.deleteSet, struct.id)
    }

    /**
     * @type {Map<string, { action: 'add' | 'update' | 'delete', oldValue: any, newValue: any }>}
     */
    get keys () {
      if (this._keys === null) {
        const keys = new Map();
        const target = this.target;
        const changed = /** @type Set<string|null> */ (this.transaction.changed.get(target));
        changed.forEach(key => {
          if (key !== null) {
            const item = /** @type {Item} */ (target._map.get(key));
            /**
             * @type {'delete' | 'add' | 'update'}
             */
            let action;
            let oldValue;
            if (this.adds(item)) {
              let prev = item.left;
              while (prev !== null && this.adds(prev)) {
                prev = prev.left;
              }
              if (this.deletes(item)) {
                if (prev !== null && this.deletes(prev)) {
                  action = 'delete';
                  oldValue = last(prev.content.getContent());
                } else {
                  return
                }
              } else {
                if (prev !== null && this.deletes(prev)) {
                  action = 'update';
                  oldValue = last(prev.content.getContent());
                } else {
                  action = 'add';
                  oldValue = undefined;
                }
              }
            } else {
              if (this.deletes(item)) {
                action = 'delete';
                oldValue = last(/** @type {Item} */ item.content.getContent());
              } else {
                return // nop
              }
            }
            keys.set(key, { action, oldValue });
          }
        });
        this._keys = keys;
      }
      return this._keys
    }

    /**
     * @type {Array<{insert?: string | Array<any> | object | AbstractType<any>, retain?: number, delete?: number, attributes?: Object<string, any>}>}
     */
    get delta () {
      return this.changes.delta
    }

    /**
     * Check if a struct is added by this event.
     *
     * In contrast to change.deleted, this method also returns true if the struct was added and then deleted.
     *
     * @param {AbstractStruct} struct
     * @return {boolean}
     */
    adds (struct) {
      return struct.id.clock >= (this.transaction.beforeState.get(struct.id.client) || 0)
    }

    /**
     * @type {{added:Set<Item>,deleted:Set<Item>,keys:Map<string,{action:'add'|'update'|'delete',oldValue:any}>,delta:Array<{insert?:Array<any>|string, delete?:number, retain?:number}>}}
     */
    get changes () {
      let changes = this._changes;
      if (changes === null) {
        const target = this.target;
        const added = create$4();
        const deleted = create$4();
        /**
         * @type {Array<{insert:Array<any>}|{delete:number}|{retain:number}>}
         */
        const delta = [];
        changes = {
          added,
          deleted,
          delta,
          keys: this.keys
        };
        const changed = /** @type Set<string|null> */ (this.transaction.changed.get(target));
        if (changed.has(null)) {
          /**
           * @type {any}
           */
          let lastOp = null;
          const packOp = () => {
            if (lastOp) {
              delta.push(lastOp);
            }
          };
          for (let item = target._start; item !== null; item = item.right) {
            if (item.deleted) {
              if (this.deletes(item) && !this.adds(item)) {
                if (lastOp === null || lastOp.delete === undefined) {
                  packOp();
                  lastOp = { delete: 0 };
                }
                lastOp.delete += item.length;
                deleted.add(item);
              } // else nop
            } else {
              if (this.adds(item)) {
                if (lastOp === null || lastOp.insert === undefined) {
                  packOp();
                  lastOp = { insert: [] };
                }
                lastOp.insert = lastOp.insert.concat(item.content.getContent());
                added.add(item);
              } else {
                if (lastOp === null || lastOp.retain === undefined) {
                  packOp();
                  lastOp = { retain: 0 };
                }
                lastOp.retain += item.length;
              }
            }
          }
          if (lastOp !== null && lastOp.retain === undefined) {
            packOp();
          }
        }
        this._changes = changes;
      }
      return /** @type {any} */ (changes)
    }
  }

  /**
   * Compute the path from this type to the specified target.
   *
   * @example
   *   // `child` should be accessible via `type.get(path[0]).get(path[1])..`
   *   const path = type.getPathTo(child)
   *   // assuming `type instanceof YArray`
   *   console.log(path) // might look like => [2, 'key1']
   *   child === type.get(path[0]).get(path[1])
   *
   * @param {AbstractType<any>} parent
   * @param {AbstractType<any>} child target
   * @return {Array<string|number>} Path to the target
   *
   * @private
   * @function
   */
  const getPathTo = (parent, child) => {
    const path = [];
    while (child._item !== null && child !== parent) {
      if (child._item.parentSub !== null) {
        // parent is map-ish
        path.unshift(child._item.parentSub);
      } else {
        // parent is array-ish
        let i = 0;
        let c = /** @type {AbstractType<any>} */ (child._item.parent)._start;
        while (c !== child._item && c !== null) {
          if (!c.deleted) {
            i++;
          }
          c = c.right;
        }
        path.unshift(i);
      }
      child = /** @type {AbstractType<any>} */ (child._item.parent);
    }
    return path
  };

  /**
   * Utility module to create and manipulate Iterators.
   *
   * @module iterator
   */

  /**
   * @template T
   * @param {function():IteratorResult<T>} next
   * @return {IterableIterator<T>}
   */
  const createIterator = next => ({
    /**
     * @return {IterableIterator<T>}
     */
    [Symbol.iterator] () {
      return this
    },
    // @ts-ignore
    next
  });

  /**
   * @template T
   * @param {Iterator<T>} iterator
   * @param {function(T):boolean} filter
   */
  const iteratorFilter = (iterator, filter) => createIterator(() => {
    let res;
    do {
      res = iterator.next();
    } while (!res.done && !filter(res.value))
    return res
  });

  /**
   * @template T,M
   * @param {Iterator<T>} iterator
   * @param {function(T):M} fmap
   */
  const iteratorMap = (iterator, fmap) => createIterator(() => {
    const { done, value } = iterator.next();
    return { done, value: done ? undefined : fmap(value) }
  });

  const maxSearchMarker = 80;

  /**
   * A unique timestamp that identifies each marker.
   *
   * Time is relative,.. this is more like an ever-increasing clock.
   *
   * @type {number}
   */
  let globalSearchMarkerTimestamp = 0;

  class ArraySearchMarker {
    /**
     * @param {Item} p
     * @param {number} index
     */
    constructor (p, index) {
      p.marker = true;
      this.p = p;
      this.index = index;
      this.timestamp = globalSearchMarkerTimestamp++;
    }
  }

  /**
   * @param {ArraySearchMarker} marker
   */
  const refreshMarkerTimestamp = marker => { marker.timestamp = globalSearchMarkerTimestamp++; };

  /**
   * This is rather complex so this function is the only thing that should overwrite a marker
   *
   * @param {ArraySearchMarker} marker
   * @param {Item} p
   * @param {number} index
   */
  const overwriteMarker = (marker, p, index) => {
    marker.p.marker = false;
    marker.p = p;
    p.marker = true;
    marker.index = index;
    marker.timestamp = globalSearchMarkerTimestamp++;
  };

  /**
   * @param {Array<ArraySearchMarker>} searchMarker
   * @param {Item} p
   * @param {number} index
   */
  const markPosition = (searchMarker, p, index) => {
    if (searchMarker.length >= maxSearchMarker) {
      // override oldest marker (we don't want to create more objects)
      const marker = searchMarker.reduce((a, b) => a.timestamp < b.timestamp ? a : b);
      overwriteMarker(marker, p, index);
      return marker
    } else {
      // create new marker
      const pm = new ArraySearchMarker(p, index);
      searchMarker.push(pm);
      return pm
    }
  };

  /**
   * Search marker help us to find positions in the associative array faster.
   *
   * They speed up the process of finding a position without much bookkeeping.
   *
   * A maximum of `maxSearchMarker` objects are created.
   *
   * This function always returns a refreshed marker (updated timestamp)
   *
   * @param {AbstractType<any>} yarray
   * @param {number} index
   */
  const findMarker = (yarray, index) => {
    if (yarray._start === null || index === 0 || yarray._searchMarker === null) {
      return null
    }
    const marker = yarray._searchMarker.length === 0 ? null : yarray._searchMarker.reduce((a, b) => abs(index - a.index) < abs(index - b.index) ? a : b);
    let p = yarray._start;
    let pindex = 0;
    if (marker !== null) {
      p = marker.p;
      pindex = marker.index;
      refreshMarkerTimestamp(marker); // we used it, we might need to use it again
    }
    // iterate to right if possible
    while (p.right !== null && pindex < index) {
      if (!p.deleted && p.countable) {
        if (index < pindex + p.length) {
          break
        }
        pindex += p.length;
      }
      p = p.right;
    }
    // iterate to left if necessary (might be that pindex > index)
    while (p.left !== null && pindex > index) {
      p = p.left;
      if (!p.deleted && p.countable) {
        pindex -= p.length;
      }
    }
    // we want to make sure that p can't be merged with left, because that would screw up everything
    // in that cas just return what we have (it is most likely the best marker anyway)
    // iterate to left until p can't be merged with left
    while (p.left !== null && p.left.id.client === p.id.client && p.left.id.clock + p.left.length === p.id.clock) {
      p = p.left;
      if (!p.deleted && p.countable) {
        pindex -= p.length;
      }
    }

    // @todo remove!
    // assure position
    // {
    //   let start = yarray._start
    //   let pos = 0
    //   while (start !== p) {
    //     if (!start.deleted && start.countable) {
    //       pos += start.length
    //     }
    //     start = /** @type {Item} */ (start.right)
    //   }
    //   if (pos !== pindex) {
    //     debugger
    //     throw new Error('Gotcha position fail!')
    //   }
    // }
    // if (marker) {
    //   if (window.lengthes == null) {
    //     window.lengthes = []
    //     window.getLengthes = () => window.lengthes.sort((a, b) => a - b)
    //   }
    //   window.lengthes.push(marker.index - pindex)
    //   console.log('distance', marker.index - pindex, 'len', p && p.parent.length)
    // }
    if (marker !== null && abs(marker.index - pindex) < /** @type {YText|YArray<any>} */ (p.parent).length / maxSearchMarker) {
      // adjust existing marker
      overwriteMarker(marker, p, pindex);
      return marker
    } else {
      // create new marker
      return markPosition(yarray._searchMarker, p, pindex)
    }
  };

  /**
   * Update markers when a change happened.
   *
   * This should be called before doing a deletion!
   *
   * @param {Array<ArraySearchMarker>} searchMarker
   * @param {number} index
   * @param {number} len If insertion, len is positive. If deletion, len is negative.
   */
  const updateMarkerChanges = (searchMarker, index, len) => {
    for (let i = searchMarker.length - 1; i >= 0; i--) {
      const m = searchMarker[i];
      if (len > 0) {
        /**
         * @type {Item|null}
         */
        let p = m.p;
        p.marker = false;
        // Ideally we just want to do a simple position comparison, but this will only work if
        // search markers don't point to deleted items for formats.
        // Iterate marker to prev undeleted countable position so we know what to do when updating a position
        while (p && (p.deleted || !p.countable)) {
          p = p.left;
          if (p && !p.deleted && p.countable) {
            // adjust position. the loop should break now
            m.index -= p.length;
          }
        }
        if (p === null || p.marker === true) {
          // remove search marker if updated position is null or if position is already marked
          searchMarker.splice(i, 1);
          continue
        }
        m.p = p;
        p.marker = true;
      }
      if (index < m.index || (len > 0 && index === m.index)) { // a simple index <= m.index check would actually suffice
        m.index = max(index, m.index + len);
      }
    }
  };

  /**
   * Call event listeners with an event. This will also add an event to all
   * parents (for `.observeDeep` handlers).
   *
   * @template EventType
   * @param {AbstractType<EventType>} type
   * @param {Transaction} transaction
   * @param {EventType} event
   */
  const callTypeObservers = (type, transaction, event) => {
    const changedType = type;
    const changedParentTypes = transaction.changedParentTypes;
    while (true) {
      // @ts-ignore
      setIfUndefined(changedParentTypes, type, () => []).push(event);
      if (type._item === null) {
        break
      }
      type = /** @type {AbstractType<any>} */ (type._item.parent);
    }
    callEventHandlerListeners(changedType._eH, event, transaction);
  };

  /**
   * @template EventType
   * Abstract Yjs Type class
   */
  class AbstractType {
    constructor () {
      /**
       * @type {Item|null}
       */
      this._item = null;
      /**
       * @type {Map<string,Item>}
       */
      this._map = new Map();
      /**
       * @type {Item|null}
       */
      this._start = null;
      /**
       * @type {Doc|null}
       */
      this.doc = null;
      this._length = 0;
      /**
       * Event handlers
       * @type {EventHandler<EventType,Transaction>}
       */
      this._eH = createEventHandler();
      /**
       * Deep event handlers
       * @type {EventHandler<Array<YEvent<any>>,Transaction>}
       */
      this._dEH = createEventHandler();
      /**
       * @type {null | Array<ArraySearchMarker>}
       */
      this._searchMarker = null;
    }

    /**
     * @return {AbstractType<any>|null}
     */
    get parent () {
      return this._item ? /** @type {AbstractType<any>} */ (this._item.parent) : null
    }

    /**
     * Integrate this type into the Yjs instance.
     *
     * * Save this struct in the os
     * * This type is sent to other client
     * * Observer functions are fired
     *
     * @param {Doc} y The Yjs instance
     * @param {Item|null} item
     */
    _integrate (y, item) {
      this.doc = y;
      this._item = item;
    }

    /**
     * @return {AbstractType<EventType>}
     */
    _copy () {
      throw methodUnimplemented()
    }

    /**
     * @return {AbstractType<EventType>}
     */
    clone () {
      throw methodUnimplemented()
    }

    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     */
    _write (encoder) { }

    /**
     * The first non-deleted item
     */
    get _first () {
      let n = this._start;
      while (n !== null && n.deleted) {
        n = n.right;
      }
      return n
    }

    /**
     * Creates YEvent and calls all type observers.
     * Must be implemented by each type.
     *
     * @param {Transaction} transaction
     * @param {Set<null|string>} parentSubs Keys changed on this type. `null` if list was modified.
     */
    _callObserver (transaction, parentSubs) {
      if (!transaction.local && this._searchMarker) {
        this._searchMarker.length = 0;
      }
    }

    /**
     * Observe all events that are created on this type.
     *
     * @param {function(EventType, Transaction):void} f Observer function
     */
    observe (f) {
      addEventHandlerListener(this._eH, f);
    }

    /**
     * Observe all events that are created by this type and its children.
     *
     * @param {function(Array<YEvent<any>>,Transaction):void} f Observer function
     */
    observeDeep (f) {
      addEventHandlerListener(this._dEH, f);
    }

    /**
     * Unregister an observer function.
     *
     * @param {function(EventType,Transaction):void} f Observer function
     */
    unobserve (f) {
      removeEventHandlerListener(this._eH, f);
    }

    /**
     * Unregister an observer function.
     *
     * @param {function(Array<YEvent<any>>,Transaction):void} f Observer function
     */
    unobserveDeep (f) {
      removeEventHandlerListener(this._dEH, f);
    }

    /**
     * @abstract
     * @return {any}
     */
    toJSON () {}
  }

  /**
   * @param {AbstractType<any>} type
   * @param {number} start
   * @param {number} end
   * @return {Array<any>}
   *
   * @private
   * @function
   */
  const typeListSlice = (type, start, end) => {
    if (start < 0) {
      start = type._length + start;
    }
    if (end < 0) {
      end = type._length + end;
    }
    let len = end - start;
    const cs = [];
    let n = type._start;
    while (n !== null && len > 0) {
      if (n.countable && !n.deleted) {
        const c = n.content.getContent();
        if (c.length <= start) {
          start -= c.length;
        } else {
          for (let i = start; i < c.length && len > 0; i++) {
            cs.push(c[i]);
            len--;
          }
          start = 0;
        }
      }
      n = n.right;
    }
    return cs
  };

  /**
   * @param {AbstractType<any>} type
   * @return {Array<any>}
   *
   * @private
   * @function
   */
  const typeListToArray = type => {
    const cs = [];
    let n = type._start;
    while (n !== null) {
      if (n.countable && !n.deleted) {
        const c = n.content.getContent();
        for (let i = 0; i < c.length; i++) {
          cs.push(c[i]);
        }
      }
      n = n.right;
    }
    return cs
  };

  /**
   * Executes a provided function on once on overy element of this YArray.
   *
   * @param {AbstractType<any>} type
   * @param {function(any,number,any):void} f A function to execute on every element of this YArray.
   *
   * @private
   * @function
   */
  const typeListForEach = (type, f) => {
    let index = 0;
    let n = type._start;
    while (n !== null) {
      if (n.countable && !n.deleted) {
        const c = n.content.getContent();
        for (let i = 0; i < c.length; i++) {
          f(c[i], index++, type);
        }
      }
      n = n.right;
    }
  };

  /**
   * @template C,R
   * @param {AbstractType<any>} type
   * @param {function(C,number,AbstractType<any>):R} f
   * @return {Array<R>}
   *
   * @private
   * @function
   */
  const typeListMap = (type, f) => {
    /**
     * @type {Array<any>}
     */
    const result = [];
    typeListForEach(type, (c, i) => {
      result.push(f(c, i, type));
    });
    return result
  };

  /**
   * @param {AbstractType<any>} type
   * @return {IterableIterator<any>}
   *
   * @private
   * @function
   */
  const typeListCreateIterator = type => {
    let n = type._start;
    /**
     * @type {Array<any>|null}
     */
    let currentContent = null;
    let currentContentIndex = 0;
    return {
      [Symbol.iterator] () {
        return this
      },
      next: () => {
        // find some content
        if (currentContent === null) {
          while (n !== null && n.deleted) {
            n = n.right;
          }
          // check if we reached the end, no need to check currentContent, because it does not exist
          if (n === null) {
            return {
              done: true,
              value: undefined
            }
          }
          // we found n, so we can set currentContent
          currentContent = n.content.getContent();
          currentContentIndex = 0;
          n = n.right; // we used the content of n, now iterate to next
        }
        const value = currentContent[currentContentIndex++];
        // check if we need to empty currentContent
        if (currentContent.length <= currentContentIndex) {
          currentContent = null;
        }
        return {
          done: false,
          value
        }
      }
    }
  };

  /**
   * @param {AbstractType<any>} type
   * @param {number} index
   * @return {any}
   *
   * @private
   * @function
   */
  const typeListGet = (type, index) => {
    const marker = findMarker(type, index);
    let n = type._start;
    if (marker !== null) {
      n = marker.p;
      index -= marker.index;
    }
    for (; n !== null; n = n.right) {
      if (!n.deleted && n.countable) {
        if (index < n.length) {
          return n.content.getContent()[index]
        }
        index -= n.length;
      }
    }
  };

  /**
   * @param {Transaction} transaction
   * @param {AbstractType<any>} parent
   * @param {Item?} referenceItem
   * @param {Array<Object<string,any>|Array<any>|boolean|number|null|string|Uint8Array>} content
   *
   * @private
   * @function
   */
  const typeListInsertGenericsAfter = (transaction, parent, referenceItem, content) => {
    let left = referenceItem;
    const doc = transaction.doc;
    const ownClientId = doc.clientID;
    const store = doc.store;
    const right = referenceItem === null ? parent._start : referenceItem.right;
    /**
     * @type {Array<Object|Array<any>|number|null>}
     */
    let jsonContent = [];
    const packJsonContent = () => {
      if (jsonContent.length > 0) {
        left = new Item(createID(ownClientId, getState(store, ownClientId)), left, left && left.lastId, right, right && right.id, parent, null, new ContentAny(jsonContent));
        left.integrate(transaction, 0);
        jsonContent = [];
      }
    };
    content.forEach(c => {
      if (c === null) {
        jsonContent.push(c);
      } else {
        switch (c.constructor) {
          case Number:
          case Object:
          case Boolean:
          case Array:
          case String:
            jsonContent.push(c);
            break
          default:
            packJsonContent();
            switch (c.constructor) {
              case Uint8Array:
              case ArrayBuffer:
                left = new Item(createID(ownClientId, getState(store, ownClientId)), left, left && left.lastId, right, right && right.id, parent, null, new ContentBinary(new Uint8Array(/** @type {Uint8Array} */ (c))));
                left.integrate(transaction, 0);
                break
              case Doc:
                left = new Item(createID(ownClientId, getState(store, ownClientId)), left, left && left.lastId, right, right && right.id, parent, null, new ContentDoc(/** @type {Doc} */ (c)));
                left.integrate(transaction, 0);
                break
              default:
                if (c instanceof AbstractType) {
                  left = new Item(createID(ownClientId, getState(store, ownClientId)), left, left && left.lastId, right, right && right.id, parent, null, new ContentType(c));
                  left.integrate(transaction, 0);
                } else {
                  throw new Error('Unexpected content type in insert operation')
                }
            }
        }
      }
    });
    packJsonContent();
  };

  const lengthExceeded = create$2('Length exceeded!');

  /**
   * @param {Transaction} transaction
   * @param {AbstractType<any>} parent
   * @param {number} index
   * @param {Array<Object<string,any>|Array<any>|number|null|string|Uint8Array>} content
   *
   * @private
   * @function
   */
  const typeListInsertGenerics = (transaction, parent, index, content) => {
    if (index > parent._length) {
      throw lengthExceeded
    }
    if (index === 0) {
      if (parent._searchMarker) {
        updateMarkerChanges(parent._searchMarker, index, content.length);
      }
      return typeListInsertGenericsAfter(transaction, parent, null, content)
    }
    const startIndex = index;
    const marker = findMarker(parent, index);
    let n = parent._start;
    if (marker !== null) {
      n = marker.p;
      index -= marker.index;
      // we need to iterate one to the left so that the algorithm works
      if (index === 0) {
        // @todo refactor this as it actually doesn't consider formats
        n = n.prev; // important! get the left undeleted item so that we can actually decrease index
        index += (n && n.countable && !n.deleted) ? n.length : 0;
      }
    }
    for (; n !== null; n = n.right) {
      if (!n.deleted && n.countable) {
        if (index <= n.length) {
          if (index < n.length) {
            // insert in-between
            getItemCleanStart(transaction, createID(n.id.client, n.id.clock + index));
          }
          break
        }
        index -= n.length;
      }
    }
    if (parent._searchMarker) {
      updateMarkerChanges(parent._searchMarker, startIndex, content.length);
    }
    return typeListInsertGenericsAfter(transaction, parent, n, content)
  };

  /**
   * Pushing content is special as we generally want to push after the last item. So we don't have to update
   * the serach marker.
   *
   * @param {Transaction} transaction
   * @param {AbstractType<any>} parent
   * @param {Array<Object<string,any>|Array<any>|number|null|string|Uint8Array>} content
   *
   * @private
   * @function
   */
  const typeListPushGenerics = (transaction, parent, content) => {
    // Use the marker with the highest index and iterate to the right.
    const marker = (parent._searchMarker || []).reduce((maxMarker, currMarker) => currMarker.index > maxMarker.index ? currMarker : maxMarker, { index: 0, p: parent._start });
    let n = marker.p;
    if (n) {
      while (n.right) {
        n = n.right;
      }
    }
    return typeListInsertGenericsAfter(transaction, parent, n, content)
  };

  /**
   * @param {Transaction} transaction
   * @param {AbstractType<any>} parent
   * @param {number} index
   * @param {number} length
   *
   * @private
   * @function
   */
  const typeListDelete = (transaction, parent, index, length) => {
    if (length === 0) { return }
    const startIndex = index;
    const startLength = length;
    const marker = findMarker(parent, index);
    let n = parent._start;
    if (marker !== null) {
      n = marker.p;
      index -= marker.index;
    }
    // compute the first item to be deleted
    for (; n !== null && index > 0; n = n.right) {
      if (!n.deleted && n.countable) {
        if (index < n.length) {
          getItemCleanStart(transaction, createID(n.id.client, n.id.clock + index));
        }
        index -= n.length;
      }
    }
    // delete all items until done
    while (length > 0 && n !== null) {
      if (!n.deleted) {
        if (length < n.length) {
          getItemCleanStart(transaction, createID(n.id.client, n.id.clock + length));
        }
        n.delete(transaction);
        length -= n.length;
      }
      n = n.right;
    }
    if (length > 0) {
      throw lengthExceeded
    }
    if (parent._searchMarker) {
      updateMarkerChanges(parent._searchMarker, startIndex, -startLength + length /* in case we remove the above exception */);
    }
  };

  /**
   * @param {Transaction} transaction
   * @param {AbstractType<any>} parent
   * @param {string} key
   *
   * @private
   * @function
   */
  const typeMapDelete = (transaction, parent, key) => {
    const c = parent._map.get(key);
    if (c !== undefined) {
      c.delete(transaction);
    }
  };

  /**
   * @param {Transaction} transaction
   * @param {AbstractType<any>} parent
   * @param {string} key
   * @param {Object|number|null|Array<any>|string|Uint8Array|AbstractType<any>} value
   *
   * @private
   * @function
   */
  const typeMapSet = (transaction, parent, key, value) => {
    const left = parent._map.get(key) || null;
    const doc = transaction.doc;
    const ownClientId = doc.clientID;
    let content;
    if (value == null) {
      content = new ContentAny([value]);
    } else {
      switch (value.constructor) {
        case Number:
        case Object:
        case Boolean:
        case Array:
        case String:
          content = new ContentAny([value]);
          break
        case Uint8Array:
          content = new ContentBinary(/** @type {Uint8Array} */ (value));
          break
        case Doc:
          content = new ContentDoc(/** @type {Doc} */ (value));
          break
        default:
          if (value instanceof AbstractType) {
            content = new ContentType(value);
          } else {
            throw new Error('Unexpected content type')
          }
      }
    }
    new Item(createID(ownClientId, getState(doc.store, ownClientId)), left, left && left.lastId, null, null, parent, key, content).integrate(transaction, 0);
  };

  /**
   * @param {AbstractType<any>} parent
   * @param {string} key
   * @return {Object<string,any>|number|null|Array<any>|string|Uint8Array|AbstractType<any>|undefined}
   *
   * @private
   * @function
   */
  const typeMapGet = (parent, key) => {
    const val = parent._map.get(key);
    return val !== undefined && !val.deleted ? val.content.getContent()[val.length - 1] : undefined
  };

  /**
   * @param {AbstractType<any>} parent
   * @return {Object<string,Object<string,any>|number|null|Array<any>|string|Uint8Array|AbstractType<any>|undefined>}
   *
   * @private
   * @function
   */
  const typeMapGetAll = (parent) => {
    /**
     * @type {Object<string,any>}
     */
    const res = {};
    parent._map.forEach((value, key) => {
      if (!value.deleted) {
        res[key] = value.content.getContent()[value.length - 1];
      }
    });
    return res
  };

  /**
   * @param {AbstractType<any>} parent
   * @param {string} key
   * @return {boolean}
   *
   * @private
   * @function
   */
  const typeMapHas = (parent, key) => {
    const val = parent._map.get(key);
    return val !== undefined && !val.deleted
  };

  /**
   * @param {Map<string,Item>} map
   * @return {IterableIterator<Array<any>>}
   *
   * @private
   * @function
   */
  const createMapIterator = map => iteratorFilter(map.entries(), /** @param {any} entry */ entry => !entry[1].deleted);

  /**
   * @module YArray
   */

  /**
   * Event that describes the changes on a YArray
   * @template T
   * @extends YEvent<YArray<T>>
   */
  class YArrayEvent extends YEvent {
    /**
     * @param {YArray<T>} yarray The changed type
     * @param {Transaction} transaction The transaction object
     */
    constructor (yarray, transaction) {
      super(yarray, transaction);
      this._transaction = transaction;
    }
  }

  /**
   * A shared Array implementation.
   * @template T
   * @extends AbstractType<YArrayEvent<T>>
   * @implements {Iterable<T>}
   */
  class YArray extends AbstractType {
    constructor () {
      super();
      /**
       * @type {Array<any>?}
       * @private
       */
      this._prelimContent = [];
      /**
       * @type {Array<ArraySearchMarker>}
       */
      this._searchMarker = [];
    }

    /**
     * Construct a new YArray containing the specified items.
     * @template T
     * @param {Array<T>} items
     * @return {YArray<T>}
     */
    static from (items) {
      const a = new YArray();
      a.push(items);
      return a
    }

    /**
     * Integrate this type into the Yjs instance.
     *
     * * Save this struct in the os
     * * This type is sent to other client
     * * Observer functions are fired
     *
     * @param {Doc} y The Yjs instance
     * @param {Item} item
     */
    _integrate (y, item) {
      super._integrate(y, item);
      this.insert(0, /** @type {Array<any>} */ (this._prelimContent));
      this._prelimContent = null;
    }

    _copy () {
      return new YArray()
    }

    /**
     * @return {YArray<T>}
     */
    clone () {
      const arr = new YArray();
      arr.insert(0, this.toArray().map(el =>
        el instanceof AbstractType ? el.clone() : el
      ));
      return arr
    }

    get length () {
      return this._prelimContent === null ? this._length : this._prelimContent.length
    }

    /**
     * Creates YArrayEvent and calls observers.
     *
     * @param {Transaction} transaction
     * @param {Set<null|string>} parentSubs Keys changed on this type. `null` if list was modified.
     */
    _callObserver (transaction, parentSubs) {
      super._callObserver(transaction, parentSubs);
      callTypeObservers(this, transaction, new YArrayEvent(this, transaction));
    }

    /**
     * Inserts new content at an index.
     *
     * Important: This function expects an array of content. Not just a content
     * object. The reason for this "weirdness" is that inserting several elements
     * is very efficient when it is done as a single operation.
     *
     * @example
     *  // Insert character 'a' at position 0
     *  yarray.insert(0, ['a'])
     *  // Insert numbers 1, 2 at position 1
     *  yarray.insert(1, [1, 2])
     *
     * @param {number} index The index to insert content at.
     * @param {Array<T>} content The array of content
     */
    insert (index, content) {
      if (this.doc !== null) {
        transact(this.doc, transaction => {
          typeListInsertGenerics(transaction, this, index, content);
        });
      } else {
        /** @type {Array<any>} */ (this._prelimContent).splice(index, 0, ...content);
      }
    }

    /**
     * Appends content to this YArray.
     *
     * @param {Array<T>} content Array of content to append.
     *
     * @todo Use the following implementation in all types.
     */
    push (content) {
      if (this.doc !== null) {
        transact(this.doc, transaction => {
          typeListPushGenerics(transaction, this, content);
        });
      } else {
        /** @type {Array<any>} */ (this._prelimContent).push(...content);
      }
    }

    /**
     * Preppends content to this YArray.
     *
     * @param {Array<T>} content Array of content to preppend.
     */
    unshift (content) {
      this.insert(0, content);
    }

    /**
     * Deletes elements starting from an index.
     *
     * @param {number} index Index at which to start deleting elements
     * @param {number} length The number of elements to remove. Defaults to 1.
     */
    delete (index, length = 1) {
      if (this.doc !== null) {
        transact(this.doc, transaction => {
          typeListDelete(transaction, this, index, length);
        });
      } else {
        /** @type {Array<any>} */ (this._prelimContent).splice(index, length);
      }
    }

    /**
     * Returns the i-th element from a YArray.
     *
     * @param {number} index The index of the element to return from the YArray
     * @return {T}
     */
    get (index) {
      return typeListGet(this, index)
    }

    /**
     * Transforms this YArray to a JavaScript Array.
     *
     * @return {Array<T>}
     */
    toArray () {
      return typeListToArray(this)
    }

    /**
     * Transforms this YArray to a JavaScript Array.
     *
     * @param {number} [start]
     * @param {number} [end]
     * @return {Array<T>}
     */
    slice (start = 0, end = this.length) {
      return typeListSlice(this, start, end)
    }

    /**
     * Transforms this Shared Type to a JSON object.
     *
     * @return {Array<any>}
     */
    toJSON () {
      return this.map(c => c instanceof AbstractType ? c.toJSON() : c)
    }

    /**
     * Returns an Array with the result of calling a provided function on every
     * element of this YArray.
     *
     * @template M
     * @param {function(T,number,YArray<T>):M} f Function that produces an element of the new Array
     * @return {Array<M>} A new array with each element being the result of the
     *                 callback function
     */
    map (f) {
      return typeListMap(this, /** @type {any} */ (f))
    }

    /**
     * Executes a provided function on once on overy element of this YArray.
     *
     * @param {function(T,number,YArray<T>):void} f A function to execute on every element of this YArray.
     */
    forEach (f) {
      typeListForEach(this, f);
    }

    /**
     * @return {IterableIterator<T>}
     */
    [Symbol.iterator] () {
      return typeListCreateIterator(this)
    }

    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     */
    _write (encoder) {
      encoder.writeTypeRef(YArrayRefID);
    }
  }

  /**
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   *
   * @private
   * @function
   */
  const readYArray = decoder => new YArray();

  /**
   * @template T
   * @extends YEvent<YMap<T>>
   * Event that describes the changes on a YMap.
   */
  class YMapEvent extends YEvent {
    /**
     * @param {YMap<T>} ymap The YArray that changed.
     * @param {Transaction} transaction
     * @param {Set<any>} subs The keys that changed.
     */
    constructor (ymap, transaction, subs) {
      super(ymap, transaction);
      this.keysChanged = subs;
    }
  }

  /**
   * @template MapType
   * A shared Map implementation.
   *
   * @extends AbstractType<YMapEvent<MapType>>
   * @implements {Iterable<MapType>}
   */
  class YMap extends AbstractType {
    /**
     *
     * @param {Iterable<readonly [string, any]>=} entries - an optional iterable to initialize the YMap
     */
    constructor (entries) {
      super();
      /**
       * @type {Map<string,any>?}
       * @private
       */
      this._prelimContent = null;

      if (entries === undefined) {
        this._prelimContent = new Map();
      } else {
        this._prelimContent = new Map(entries);
      }
    }

    /**
     * Integrate this type into the Yjs instance.
     *
     * * Save this struct in the os
     * * This type is sent to other client
     * * Observer functions are fired
     *
     * @param {Doc} y The Yjs instance
     * @param {Item} item
     */
    _integrate (y, item) {
      super._integrate(y, item)
      ;/** @type {Map<string, any>} */ (this._prelimContent).forEach((value, key) => {
        this.set(key, value);
      });
      this._prelimContent = null;
    }

    _copy () {
      return new YMap()
    }

    /**
     * @return {YMap<MapType>}
     */
    clone () {
      const map = new YMap();
      this.forEach((value, key) => {
        map.set(key, value instanceof AbstractType ? value.clone() : value);
      });
      return map
    }

    /**
     * Creates YMapEvent and calls observers.
     *
     * @param {Transaction} transaction
     * @param {Set<null|string>} parentSubs Keys changed on this type. `null` if list was modified.
     */
    _callObserver (transaction, parentSubs) {
      callTypeObservers(this, transaction, new YMapEvent(this, transaction, parentSubs));
    }

    /**
     * Transforms this Shared Type to a JSON object.
     *
     * @return {Object<string,any>}
     */
    toJSON () {
      /**
       * @type {Object<string,MapType>}
       */
      const map = {};
      this._map.forEach((item, key) => {
        if (!item.deleted) {
          const v = item.content.getContent()[item.length - 1];
          map[key] = v instanceof AbstractType ? v.toJSON() : v;
        }
      });
      return map
    }

    /**
     * Returns the size of the YMap (count of key/value pairs)
     *
     * @return {number}
     */
    get size () {
      return [...createMapIterator(this._map)].length
    }

    /**
     * Returns the keys for each element in the YMap Type.
     *
     * @return {IterableIterator<string>}
     */
    keys () {
      return iteratorMap(createMapIterator(this._map), /** @param {any} v */ v => v[0])
    }

    /**
     * Returns the values for each element in the YMap Type.
     *
     * @return {IterableIterator<any>}
     */
    values () {
      return iteratorMap(createMapIterator(this._map), /** @param {any} v */ v => v[1].content.getContent()[v[1].length - 1])
    }

    /**
     * Returns an Iterator of [key, value] pairs
     *
     * @return {IterableIterator<any>}
     */
    entries () {
      return iteratorMap(createMapIterator(this._map), /** @param {any} v */ v => [v[0], v[1].content.getContent()[v[1].length - 1]])
    }

    /**
     * Executes a provided function on once on every key-value pair.
     *
     * @param {function(MapType,string,YMap<MapType>):void} f A function to execute on every element of this YArray.
     */
    forEach (f) {
      /**
       * @type {Object<string,MapType>}
       */
      const map = {};
      this._map.forEach((item, key) => {
        if (!item.deleted) {
          f(item.content.getContent()[item.length - 1], key, this);
        }
      });
      return map
    }

    /**
     * Returns an Iterator of [key, value] pairs
     *
     * @return {IterableIterator<any>}
     */
    [Symbol.iterator] () {
      return this.entries()
    }

    /**
     * Remove a specified element from this YMap.
     *
     * @param {string} key The key of the element to remove.
     */
    delete (key) {
      if (this.doc !== null) {
        transact(this.doc, transaction => {
          typeMapDelete(transaction, this, key);
        });
      } else {
        /** @type {Map<string, any>} */ (this._prelimContent).delete(key);
      }
    }

    /**
     * Adds or updates an element with a specified key and value.
     *
     * @param {string} key The key of the element to add to this YMap
     * @param {MapType} value The value of the element to add
     */
    set (key, value) {
      if (this.doc !== null) {
        transact(this.doc, transaction => {
          typeMapSet(transaction, this, key, value);
        });
      } else {
        /** @type {Map<string, any>} */ (this._prelimContent).set(key, value);
      }
      return value
    }

    /**
     * Returns a specified element from this YMap.
     *
     * @param {string} key
     * @return {MapType|undefined}
     */
    get (key) {
      return /** @type {any} */ (typeMapGet(this, key))
    }

    /**
     * Returns a boolean indicating whether the specified key exists or not.
     *
     * @param {string} key The key to test.
     * @return {boolean}
     */
    has (key) {
      return typeMapHas(this, key)
    }

    /**
     * Removes all elements from this YMap.
     */
    clear () {
      if (this.doc !== null) {
        transact(this.doc, transaction => {
          this.forEach(function (value, key, map) {
            typeMapDelete(transaction, map, key);
          });
        });
      } else {
        /** @type {Map<string, any>} */ (this._prelimContent).clear();
      }
    }

    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     */
    _write (encoder) {
      encoder.writeTypeRef(YMapRefID);
    }
  }

  /**
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   *
   * @private
   * @function
   */
  const readYMap = decoder => new YMap();

  /**
   * @param {any} a
   * @param {any} b
   * @return {boolean}
   */
  const equalAttrs = (a, b) => a === b || (typeof a === 'object' && typeof b === 'object' && a && b && equalFlat(a, b));

  class ItemTextListPosition {
    /**
     * @param {Item|null} left
     * @param {Item|null} right
     * @param {number} index
     * @param {Map<string,any>} currentAttributes
     */
    constructor (left, right, index, currentAttributes) {
      this.left = left;
      this.right = right;
      this.index = index;
      this.currentAttributes = currentAttributes;
    }

    /**
     * Only call this if you know that this.right is defined
     */
    forward () {
      if (this.right === null) {
        unexpectedCase();
      }
      switch (this.right.content.constructor) {
        case ContentFormat:
          if (!this.right.deleted) {
            updateCurrentAttributes(this.currentAttributes, /** @type {ContentFormat} */ (this.right.content));
          }
          break
        default:
          if (!this.right.deleted) {
            this.index += this.right.length;
          }
          break
      }
      this.left = this.right;
      this.right = this.right.right;
    }
  }

  /**
   * @param {Transaction} transaction
   * @param {ItemTextListPosition} pos
   * @param {number} count steps to move forward
   * @return {ItemTextListPosition}
   *
   * @private
   * @function
   */
  const findNextPosition = (transaction, pos, count) => {
    while (pos.right !== null && count > 0) {
      switch (pos.right.content.constructor) {
        case ContentFormat:
          if (!pos.right.deleted) {
            updateCurrentAttributes(pos.currentAttributes, /** @type {ContentFormat} */ (pos.right.content));
          }
          break
        default:
          if (!pos.right.deleted) {
            if (count < pos.right.length) {
              // split right
              getItemCleanStart(transaction, createID(pos.right.id.client, pos.right.id.clock + count));
            }
            pos.index += pos.right.length;
            count -= pos.right.length;
          }
          break
      }
      pos.left = pos.right;
      pos.right = pos.right.right;
      // pos.forward() - we don't forward because that would halve the performance because we already do the checks above
    }
    return pos
  };

  /**
   * @param {Transaction} transaction
   * @param {AbstractType<any>} parent
   * @param {number} index
   * @return {ItemTextListPosition}
   *
   * @private
   * @function
   */
  const findPosition$1 = (transaction, parent, index) => {
    const currentAttributes = new Map();
    const marker = findMarker(parent, index);
    if (marker) {
      const pos = new ItemTextListPosition(marker.p.left, marker.p, marker.index, currentAttributes);
      return findNextPosition(transaction, pos, index - marker.index)
    } else {
      const pos = new ItemTextListPosition(null, parent._start, 0, currentAttributes);
      return findNextPosition(transaction, pos, index)
    }
  };

  /**
   * Negate applied formats
   *
   * @param {Transaction} transaction
   * @param {AbstractType<any>} parent
   * @param {ItemTextListPosition} currPos
   * @param {Map<string,any>} negatedAttributes
   *
   * @private
   * @function
   */
  const insertNegatedAttributes = (transaction, parent, currPos, negatedAttributes) => {
    // check if we really need to remove attributes
    while (
      currPos.right !== null && (
        currPos.right.deleted === true || (
          currPos.right.content.constructor === ContentFormat &&
          equalAttrs(negatedAttributes.get(/** @type {ContentFormat} */ (currPos.right.content).key), /** @type {ContentFormat} */ (currPos.right.content).value)
        )
      )
    ) {
      if (!currPos.right.deleted) {
        negatedAttributes.delete(/** @type {ContentFormat} */ (currPos.right.content).key);
      }
      currPos.forward();
    }
    const doc = transaction.doc;
    const ownClientId = doc.clientID;
    negatedAttributes.forEach((val, key) => {
      const left = currPos.left;
      const right = currPos.right;
      const nextFormat = new Item(createID(ownClientId, getState(doc.store, ownClientId)), left, left && left.lastId, right, right && right.id, parent, null, new ContentFormat(key, val));
      nextFormat.integrate(transaction, 0);
      currPos.right = nextFormat;
      currPos.forward();
    });
  };

  /**
   * @param {Map<string,any>} currentAttributes
   * @param {ContentFormat} format
   *
   * @private
   * @function
   */
  const updateCurrentAttributes = (currentAttributes, format) => {
    const { key, value } = format;
    if (value === null) {
      currentAttributes.delete(key);
    } else {
      currentAttributes.set(key, value);
    }
  };

  /**
   * @param {ItemTextListPosition} currPos
   * @param {Object<string,any>} attributes
   *
   * @private
   * @function
   */
  const minimizeAttributeChanges = (currPos, attributes) => {
    // go right while attributes[right.key] === right.value (or right is deleted)
    while (true) {
      if (currPos.right === null) {
        break
      } else if (currPos.right.deleted || (currPos.right.content.constructor === ContentFormat && equalAttrs(attributes[(/** @type {ContentFormat} */ (currPos.right.content)).key] || null, /** @type {ContentFormat} */ (currPos.right.content).value))) ; else {
        break
      }
      currPos.forward();
    }
  };

  /**
   * @param {Transaction} transaction
   * @param {AbstractType<any>} parent
   * @param {ItemTextListPosition} currPos
   * @param {Object<string,any>} attributes
   * @return {Map<string,any>}
   *
   * @private
   * @function
   **/
  const insertAttributes = (transaction, parent, currPos, attributes) => {
    const doc = transaction.doc;
    const ownClientId = doc.clientID;
    const negatedAttributes = new Map();
    // insert format-start items
    for (const key in attributes) {
      const val = attributes[key];
      const currentVal = currPos.currentAttributes.get(key) || null;
      if (!equalAttrs(currentVal, val)) {
        // save negated attribute (set null if currentVal undefined)
        negatedAttributes.set(key, currentVal);
        const { left, right } = currPos;
        currPos.right = new Item(createID(ownClientId, getState(doc.store, ownClientId)), left, left && left.lastId, right, right && right.id, parent, null, new ContentFormat(key, val));
        currPos.right.integrate(transaction, 0);
        currPos.forward();
      }
    }
    return negatedAttributes
  };

  /**
   * @param {Transaction} transaction
   * @param {AbstractType<any>} parent
   * @param {ItemTextListPosition} currPos
   * @param {string|object|AbstractType<any>} text
   * @param {Object<string,any>} attributes
   *
   * @private
   * @function
   **/
  const insertText = (transaction, parent, currPos, text, attributes) => {
    currPos.currentAttributes.forEach((val, key) => {
      if (attributes[key] === undefined) {
        attributes[key] = null;
      }
    });
    const doc = transaction.doc;
    const ownClientId = doc.clientID;
    minimizeAttributeChanges(currPos, attributes);
    const negatedAttributes = insertAttributes(transaction, parent, currPos, attributes);
    // insert content
    const content = text.constructor === String ? new ContentString(/** @type {string} */ (text)) : (text instanceof AbstractType ? new ContentType(text) : new ContentEmbed(text));
    let { left, right, index } = currPos;
    if (parent._searchMarker) {
      updateMarkerChanges(parent._searchMarker, currPos.index, content.getLength());
    }
    right = new Item(createID(ownClientId, getState(doc.store, ownClientId)), left, left && left.lastId, right, right && right.id, parent, null, content);
    right.integrate(transaction, 0);
    currPos.right = right;
    currPos.index = index;
    currPos.forward();
    insertNegatedAttributes(transaction, parent, currPos, negatedAttributes);
  };

  /**
   * @param {Transaction} transaction
   * @param {AbstractType<any>} parent
   * @param {ItemTextListPosition} currPos
   * @param {number} length
   * @param {Object<string,any>} attributes
   *
   * @private
   * @function
   */
  const formatText = (transaction, parent, currPos, length, attributes) => {
    const doc = transaction.doc;
    const ownClientId = doc.clientID;
    minimizeAttributeChanges(currPos, attributes);
    const negatedAttributes = insertAttributes(transaction, parent, currPos, attributes);
    // iterate until first non-format or null is found
    // delete all formats with attributes[format.key] != null
    // also check the attributes after the first non-format as we do not want to insert redundant negated attributes there
    // eslint-disable-next-line no-labels
    iterationLoop: while (
      currPos.right !== null &&
      (length > 0 ||
        (
          negatedAttributes.size > 0 &&
          (currPos.right.deleted || currPos.right.content.constructor === ContentFormat)
        )
      )
    ) {
      if (!currPos.right.deleted) {
        switch (currPos.right.content.constructor) {
          case ContentFormat: {
            const { key, value } = /** @type {ContentFormat} */ (currPos.right.content);
            const attr = attributes[key];
            if (attr !== undefined) {
              if (equalAttrs(attr, value)) {
                negatedAttributes.delete(key);
              } else {
                if (length === 0) {
                  // no need to further extend negatedAttributes
                  // eslint-disable-next-line no-labels
                  break iterationLoop
                }
                negatedAttributes.set(key, value);
              }
              currPos.right.delete(transaction);
            } else {
              currPos.currentAttributes.set(key, value);
            }
            break
          }
          default:
            if (length < currPos.right.length) {
              getItemCleanStart(transaction, createID(currPos.right.id.client, currPos.right.id.clock + length));
            }
            length -= currPos.right.length;
            break
        }
      }
      currPos.forward();
    }
    // Quill just assumes that the editor starts with a newline and that it always
    // ends with a newline. We only insert that newline when a new newline is
    // inserted - i.e when length is bigger than type.length
    if (length > 0) {
      let newlines = '';
      for (; length > 0; length--) {
        newlines += '\n';
      }
      currPos.right = new Item(createID(ownClientId, getState(doc.store, ownClientId)), currPos.left, currPos.left && currPos.left.lastId, currPos.right, currPos.right && currPos.right.id, parent, null, new ContentString(newlines));
      currPos.right.integrate(transaction, 0);
      currPos.forward();
    }
    insertNegatedAttributes(transaction, parent, currPos, negatedAttributes);
  };

  /**
   * Call this function after string content has been deleted in order to
   * clean up formatting Items.
   *
   * @param {Transaction} transaction
   * @param {Item} start
   * @param {Item|null} curr exclusive end, automatically iterates to the next Content Item
   * @param {Map<string,any>} startAttributes
   * @param {Map<string,any>} currAttributes
   * @return {number} The amount of formatting Items deleted.
   *
   * @function
   */
  const cleanupFormattingGap = (transaction, start, curr, startAttributes, currAttributes) => {
    let end = curr;
    const endAttributes = copy(currAttributes);
    while (end && (!end.countable || end.deleted)) {
      if (!end.deleted && end.content.constructor === ContentFormat) {
        updateCurrentAttributes(endAttributes, /** @type {ContentFormat} */ (end.content));
      }
      end = end.right;
    }
    let cleanups = 0;
    let reachedEndOfCurr = false;
    while (start !== end) {
      if (curr === start) {
        reachedEndOfCurr = true;
      }
      if (!start.deleted) {
        const content = start.content;
        switch (content.constructor) {
          case ContentFormat: {
            const { key, value } = /** @type {ContentFormat} */ (content);
            if ((endAttributes.get(key) || null) !== value || (startAttributes.get(key) || null) === value) {
              // Either this format is overwritten or it is not necessary because the attribute already existed.
              start.delete(transaction);
              cleanups++;
              if (!reachedEndOfCurr && (currAttributes.get(key) || null) === value && (startAttributes.get(key) || null) !== value) {
                currAttributes.delete(key);
              }
            }
            break
          }
        }
      }
      start = /** @type {Item} */ (start.right);
    }
    return cleanups
  };

  /**
   * @param {Transaction} transaction
   * @param {Item | null} item
   */
  const cleanupContextlessFormattingGap = (transaction, item) => {
    // iterate until item.right is null or content
    while (item && item.right && (item.right.deleted || !item.right.countable)) {
      item = item.right;
    }
    const attrs = new Set();
    // iterate back until a content item is found
    while (item && (item.deleted || !item.countable)) {
      if (!item.deleted && item.content.constructor === ContentFormat) {
        const key = /** @type {ContentFormat} */ (item.content).key;
        if (attrs.has(key)) {
          item.delete(transaction);
        } else {
          attrs.add(key);
        }
      }
      item = item.left;
    }
  };

  /**
   * This function is experimental and subject to change / be removed.
   *
   * Ideally, we don't need this function at all. Formatting attributes should be cleaned up
   * automatically after each change. This function iterates twice over the complete YText type
   * and removes unnecessary formatting attributes. This is also helpful for testing.
   *
   * This function won't be exported anymore as soon as there is confidence that the YText type works as intended.
   *
   * @param {YText} type
   * @return {number} How many formatting attributes have been cleaned up.
   */
  const cleanupYTextFormatting = type => {
    let res = 0;
    transact(/** @type {Doc} */ (type.doc), transaction => {
      let start = /** @type {Item} */ (type._start);
      let end = type._start;
      let startAttributes = create$5();
      const currentAttributes = copy(startAttributes);
      while (end) {
        if (end.deleted === false) {
          switch (end.content.constructor) {
            case ContentFormat:
              updateCurrentAttributes(currentAttributes, /** @type {ContentFormat} */ (end.content));
              break
            default:
              res += cleanupFormattingGap(transaction, start, end, startAttributes, currentAttributes);
              startAttributes = copy(currentAttributes);
              start = end;
              break
          }
        }
        end = end.right;
      }
    });
    return res
  };

  /**
   * @param {Transaction} transaction
   * @param {ItemTextListPosition} currPos
   * @param {number} length
   * @return {ItemTextListPosition}
   *
   * @private
   * @function
   */
  const deleteText = (transaction, currPos, length) => {
    const startLength = length;
    const startAttrs = copy(currPos.currentAttributes);
    const start = currPos.right;
    while (length > 0 && currPos.right !== null) {
      if (currPos.right.deleted === false) {
        switch (currPos.right.content.constructor) {
          case ContentType:
          case ContentEmbed:
          case ContentString:
            if (length < currPos.right.length) {
              getItemCleanStart(transaction, createID(currPos.right.id.client, currPos.right.id.clock + length));
            }
            length -= currPos.right.length;
            currPos.right.delete(transaction);
            break
        }
      }
      currPos.forward();
    }
    if (start) {
      cleanupFormattingGap(transaction, start, currPos.right, startAttrs, currPos.currentAttributes);
    }
    const parent = /** @type {AbstractType<any>} */ (/** @type {Item} */ (currPos.left || currPos.right).parent);
    if (parent._searchMarker) {
      updateMarkerChanges(parent._searchMarker, currPos.index, -startLength + length);
    }
    return currPos
  };

  /**
   * The Quill Delta format represents changes on a text document with
   * formatting information. For mor information visit {@link https://quilljs.com/docs/delta/|Quill Delta}
   *
   * @example
   *   {
   *     ops: [
   *       { insert: 'Gandalf', attributes: { bold: true } },
   *       { insert: ' the ' },
   *       { insert: 'Grey', attributes: { color: '#cccccc' } }
   *     ]
   *   }
   *
   */

  /**
    * Attributes that can be assigned to a selection of text.
    *
    * @example
    *   {
    *     bold: true,
    *     font-size: '40px'
    *   }
    *
    * @typedef {Object} TextAttributes
    */

  /**
   * @extends YEvent<YText>
   * Event that describes the changes on a YText type.
   */
  class YTextEvent extends YEvent {
    /**
     * @param {YText} ytext
     * @param {Transaction} transaction
     * @param {Set<any>} subs The keys that changed
     */
    constructor (ytext, transaction, subs) {
      super(ytext, transaction);
      /**
       * Whether the children changed.
       * @type {Boolean}
       * @private
       */
      this.childListChanged = false;
      /**
       * Set of all changed attributes.
       * @type {Set<string>}
       */
      this.keysChanged = new Set();
      subs.forEach((sub) => {
        if (sub === null) {
          this.childListChanged = true;
        } else {
          this.keysChanged.add(sub);
        }
      });
    }

    /**
     * @type {{added:Set<Item>,deleted:Set<Item>,keys:Map<string,{action:'add'|'update'|'delete',oldValue:any}>,delta:Array<{insert?:Array<any>|string, delete?:number, retain?:number}>}}
     */
    get changes () {
      if (this._changes === null) {
        /**
         * @type {{added:Set<Item>,deleted:Set<Item>,keys:Map<string,{action:'add'|'update'|'delete',oldValue:any}>,delta:Array<{insert?:Array<any>|string|AbstractType<any>|object, delete?:number, retain?:number}>}}
         */
        const changes = {
          keys: this.keys,
          delta: this.delta,
          added: new Set(),
          deleted: new Set()
        };
        this._changes = changes;
      }
      return /** @type {any} */ (this._changes)
    }

    /**
     * Compute the changes in the delta format.
     * A {@link https://quilljs.com/docs/delta/|Quill Delta}) that represents the changes on the document.
     *
     * @type {Array<{insert?:string|object|AbstractType<any>, delete?:number, retain?:number, attributes?: Object<string,any>}>}
     *
     * @public
     */
    get delta () {
      if (this._delta === null) {
        const y = /** @type {Doc} */ (this.target.doc);
        /**
         * @type {Array<{insert?:string|object|AbstractType<any>, delete?:number, retain?:number, attributes?: Object<string,any>}>}
         */
        const delta = [];
        transact(y, transaction => {
          const currentAttributes = new Map(); // saves all current attributes for insert
          const oldAttributes = new Map();
          let item = this.target._start;
          /**
           * @type {string?}
           */
          let action = null;
          /**
           * @type {Object<string,any>}
           */
          const attributes = {}; // counts added or removed new attributes for retain
          /**
           * @type {string|object}
           */
          let insert = '';
          let retain = 0;
          let deleteLen = 0;
          const addOp = () => {
            if (action !== null) {
              /**
               * @type {any}
               */
              let op;
              switch (action) {
                case 'delete':
                  op = { delete: deleteLen };
                  deleteLen = 0;
                  break
                case 'insert':
                  op = { insert };
                  if (currentAttributes.size > 0) {
                    op.attributes = {};
                    currentAttributes.forEach((value, key) => {
                      if (value !== null) {
                        op.attributes[key] = value;
                      }
                    });
                  }
                  insert = '';
                  break
                case 'retain':
                  op = { retain };
                  if (Object.keys(attributes).length > 0) {
                    op.attributes = {};
                    for (const key in attributes) {
                      op.attributes[key] = attributes[key];
                    }
                  }
                  retain = 0;
                  break
              }
              delta.push(op);
              action = null;
            }
          };
          while (item !== null) {
            switch (item.content.constructor) {
              case ContentType:
              case ContentEmbed:
                if (this.adds(item)) {
                  if (!this.deletes(item)) {
                    addOp();
                    action = 'insert';
                    insert = item.content.getContent()[0];
                    addOp();
                  }
                } else if (this.deletes(item)) {
                  if (action !== 'delete') {
                    addOp();
                    action = 'delete';
                  }
                  deleteLen += 1;
                } else if (!item.deleted) {
                  if (action !== 'retain') {
                    addOp();
                    action = 'retain';
                  }
                  retain += 1;
                }
                break
              case ContentString:
                if (this.adds(item)) {
                  if (!this.deletes(item)) {
                    if (action !== 'insert') {
                      addOp();
                      action = 'insert';
                    }
                    insert += /** @type {ContentString} */ (item.content).str;
                  }
                } else if (this.deletes(item)) {
                  if (action !== 'delete') {
                    addOp();
                    action = 'delete';
                  }
                  deleteLen += item.length;
                } else if (!item.deleted) {
                  if (action !== 'retain') {
                    addOp();
                    action = 'retain';
                  }
                  retain += item.length;
                }
                break
              case ContentFormat: {
                const { key, value } = /** @type {ContentFormat} */ (item.content);
                if (this.adds(item)) {
                  if (!this.deletes(item)) {
                    const curVal = currentAttributes.get(key) || null;
                    if (!equalAttrs(curVal, value)) {
                      if (action === 'retain') {
                        addOp();
                      }
                      if (equalAttrs(value, (oldAttributes.get(key) || null))) {
                        delete attributes[key];
                      } else {
                        attributes[key] = value;
                      }
                    } else if (value !== null) {
                      item.delete(transaction);
                    }
                  }
                } else if (this.deletes(item)) {
                  oldAttributes.set(key, value);
                  const curVal = currentAttributes.get(key) || null;
                  if (!equalAttrs(curVal, value)) {
                    if (action === 'retain') {
                      addOp();
                    }
                    attributes[key] = curVal;
                  }
                } else if (!item.deleted) {
                  oldAttributes.set(key, value);
                  const attr = attributes[key];
                  if (attr !== undefined) {
                    if (!equalAttrs(attr, value)) {
                      if (action === 'retain') {
                        addOp();
                      }
                      if (value === null) {
                        delete attributes[key];
                      } else {
                        attributes[key] = value;
                      }
                    } else if (attr !== null) { // this will be cleaned up automatically by the contextless cleanup function
                      item.delete(transaction);
                    }
                  }
                }
                if (!item.deleted) {
                  if (action === 'insert') {
                    addOp();
                  }
                  updateCurrentAttributes(currentAttributes, /** @type {ContentFormat} */ (item.content));
                }
                break
              }
            }
            item = item.right;
          }
          addOp();
          while (delta.length > 0) {
            const lastOp = delta[delta.length - 1];
            if (lastOp.retain !== undefined && lastOp.attributes === undefined) {
              // retain delta's if they don't assign attributes
              delta.pop();
            } else {
              break
            }
          }
        });
        this._delta = delta;
      }
      return /** @type {any} */ (this._delta)
    }
  }

  /**
   * Type that represents text with formatting information.
   *
   * This type replaces y-richtext as this implementation is able to handle
   * block formats (format information on a paragraph), embeds (complex elements
   * like pictures and videos), and text formats (**bold**, *italic*).
   *
   * @extends AbstractType<YTextEvent>
   */
  class YText extends AbstractType {
    /**
     * @param {String} [string] The initial value of the YText.
     */
    constructor (string) {
      super();
      /**
       * Array of pending operations on this type
       * @type {Array<function():void>?}
       */
      this._pending = string !== undefined ? [() => this.insert(0, string)] : [];
      /**
       * @type {Array<ArraySearchMarker>}
       */
      this._searchMarker = [];
    }

    /**
     * Number of characters of this text type.
     *
     * @type {number}
     */
    get length () {
      return this._length
    }

    /**
     * @param {Doc} y
     * @param {Item} item
     */
    _integrate (y, item) {
      super._integrate(y, item);
      try {
        /** @type {Array<function>} */ (this._pending).forEach(f => f());
      } catch (e) {
        console.error(e);
      }
      this._pending = null;
    }

    _copy () {
      return new YText()
    }

    /**
     * @return {YText}
     */
    clone () {
      const text = new YText();
      text.applyDelta(this.toDelta());
      return text
    }

    /**
     * Creates YTextEvent and calls observers.
     *
     * @param {Transaction} transaction
     * @param {Set<null|string>} parentSubs Keys changed on this type. `null` if list was modified.
     */
    _callObserver (transaction, parentSubs) {
      super._callObserver(transaction, parentSubs);
      const event = new YTextEvent(this, transaction, parentSubs);
      const doc = transaction.doc;
      callTypeObservers(this, transaction, event);
      // If a remote change happened, we try to cleanup potential formatting duplicates.
      if (!transaction.local) {
        // check if another formatting item was inserted
        let foundFormattingItem = false;
        for (const [client, afterClock] of transaction.afterState.entries()) {
          const clock = transaction.beforeState.get(client) || 0;
          if (afterClock === clock) {
            continue
          }
          iterateStructs(transaction, /** @type {Array<Item|GC>} */ (doc.store.clients.get(client)), clock, afterClock, item => {
            if (!item.deleted && /** @type {Item} */ (item).content.constructor === ContentFormat) {
              foundFormattingItem = true;
            }
          });
          if (foundFormattingItem) {
            break
          }
        }
        if (!foundFormattingItem) {
          iterateDeletedStructs(transaction, transaction.deleteSet, item => {
            if (item instanceof GC || foundFormattingItem) {
              return
            }
            if (item.parent === this && item.content.constructor === ContentFormat) {
              foundFormattingItem = true;
            }
          });
        }
        transact(doc, (t) => {
          if (foundFormattingItem) {
            // If a formatting item was inserted, we simply clean the whole type.
            // We need to compute currentAttributes for the current position anyway.
            cleanupYTextFormatting(this);
          } else {
            // If no formatting attribute was inserted, we can make due with contextless
            // formatting cleanups.
            // Contextless: it is not necessary to compute currentAttributes for the affected position.
            iterateDeletedStructs(t, t.deleteSet, item => {
              if (item instanceof GC) {
                return
              }
              if (item.parent === this) {
                cleanupContextlessFormattingGap(t, item);
              }
            });
          }
        });
      }
    }

    /**
     * Returns the unformatted string representation of this YText type.
     *
     * @public
     */
    toString () {
      let str = '';
      /**
       * @type {Item|null}
       */
      let n = this._start;
      while (n !== null) {
        if (!n.deleted && n.countable && n.content.constructor === ContentString) {
          str += /** @type {ContentString} */ (n.content).str;
        }
        n = n.right;
      }
      return str
    }

    /**
     * Returns the unformatted string representation of this YText type.
     *
     * @return {string}
     * @public
     */
    toJSON () {
      return this.toString()
    }

    /**
     * Apply a {@link Delta} on this shared YText type.
     *
     * @param {any} delta The changes to apply on this element.
     * @param {object}  [opts]
     * @param {boolean} [opts.sanitize] Sanitize input delta. Removes ending newlines if set to true.
     *
     *
     * @public
     */
    applyDelta (delta, { sanitize = true } = {}) {
      if (this.doc !== null) {
        transact(this.doc, transaction => {
          const currPos = new ItemTextListPosition(null, this._start, 0, new Map());
          for (let i = 0; i < delta.length; i++) {
            const op = delta[i];
            if (op.insert !== undefined) {
              // Quill assumes that the content starts with an empty paragraph.
              // Yjs/Y.Text assumes that it starts empty. We always hide that
              // there is a newline at the end of the content.
              // If we omit this step, clients will see a different number of
              // paragraphs, but nothing bad will happen.
              const ins = (!sanitize && typeof op.insert === 'string' && i === delta.length - 1 && currPos.right === null && op.insert.slice(-1) === '\n') ? op.insert.slice(0, -1) : op.insert;
              if (typeof ins !== 'string' || ins.length > 0) {
                insertText(transaction, this, currPos, ins, op.attributes || {});
              }
            } else if (op.retain !== undefined) {
              formatText(transaction, this, currPos, op.retain, op.attributes || {});
            } else if (op.delete !== undefined) {
              deleteText(transaction, currPos, op.delete);
            }
          }
        });
      } else {
        /** @type {Array<function>} */ (this._pending).push(() => this.applyDelta(delta));
      }
    }

    /**
     * Returns the Delta representation of this YText type.
     *
     * @param {Snapshot} [snapshot]
     * @param {Snapshot} [prevSnapshot]
     * @param {function('removed' | 'added', ID):any} [computeYChange]
     * @return {any} The Delta representation of this type.
     *
     * @public
     */
    toDelta (snapshot, prevSnapshot, computeYChange) {
      /**
       * @type{Array<any>}
       */
      const ops = [];
      const currentAttributes = new Map();
      const doc = /** @type {Doc} */ (this.doc);
      let str = '';
      let n = this._start;
      function packStr () {
        if (str.length > 0) {
          // pack str with attributes to ops
          /**
           * @type {Object<string,any>}
           */
          const attributes = {};
          let addAttributes = false;
          currentAttributes.forEach((value, key) => {
            addAttributes = true;
            attributes[key] = value;
          });
          /**
           * @type {Object<string,any>}
           */
          const op = { insert: str };
          if (addAttributes) {
            op.attributes = attributes;
          }
          ops.push(op);
          str = '';
        }
      }
      // snapshots are merged again after the transaction, so we need to keep the
      // transalive until we are done
      transact(doc, transaction => {
        if (snapshot) {
          splitSnapshotAffectedStructs(transaction, snapshot);
        }
        if (prevSnapshot) {
          splitSnapshotAffectedStructs(transaction, prevSnapshot);
        }
        while (n !== null) {
          if (isVisible(n, snapshot) || (prevSnapshot !== undefined && isVisible(n, prevSnapshot))) {
            switch (n.content.constructor) {
              case ContentString: {
                const cur = currentAttributes.get('ychange');
                if (snapshot !== undefined && !isVisible(n, snapshot)) {
                  if (cur === undefined || cur.user !== n.id.client || cur.state !== 'removed') {
                    packStr();
                    currentAttributes.set('ychange', computeYChange ? computeYChange('removed', n.id) : { type: 'removed' });
                  }
                } else if (prevSnapshot !== undefined && !isVisible(n, prevSnapshot)) {
                  if (cur === undefined || cur.user !== n.id.client || cur.state !== 'added') {
                    packStr();
                    currentAttributes.set('ychange', computeYChange ? computeYChange('added', n.id) : { type: 'added' });
                  }
                } else if (cur !== undefined) {
                  packStr();
                  currentAttributes.delete('ychange');
                }
                str += /** @type {ContentString} */ (n.content).str;
                break
              }
              case ContentType:
              case ContentEmbed: {
                packStr();
                /**
                 * @type {Object<string,any>}
                 */
                const op = {
                  insert: n.content.getContent()[0]
                };
                if (currentAttributes.size > 0) {
                  const attrs = /** @type {Object<string,any>} */ ({});
                  op.attributes = attrs;
                  currentAttributes.forEach((value, key) => {
                    attrs[key] = value;
                  });
                }
                ops.push(op);
                break
              }
              case ContentFormat:
                if (isVisible(n, snapshot)) {
                  packStr();
                  updateCurrentAttributes(currentAttributes, /** @type {ContentFormat} */ (n.content));
                }
                break
            }
          }
          n = n.right;
        }
        packStr();
      }, splitSnapshotAffectedStructs);
      return ops
    }

    /**
     * Insert text at a given index.
     *
     * @param {number} index The index at which to start inserting.
     * @param {String} text The text to insert at the specified position.
     * @param {TextAttributes} [attributes] Optionally define some formatting
     *                                    information to apply on the inserted
     *                                    Text.
     * @public
     */
    insert (index, text, attributes) {
      if (text.length <= 0) {
        return
      }
      const y = this.doc;
      if (y !== null) {
        transact(y, transaction => {
          const pos = findPosition$1(transaction, this, index);
          if (!attributes) {
            attributes = {};
            // @ts-ignore
            pos.currentAttributes.forEach((v, k) => { attributes[k] = v; });
          }
          insertText(transaction, this, pos, text, attributes);
        });
      } else {
        /** @type {Array<function>} */ (this._pending).push(() => this.insert(index, text, attributes));
      }
    }

    /**
     * Inserts an embed at a index.
     *
     * @param {number} index The index to insert the embed at.
     * @param {Object | AbstractType<any>} embed The Object that represents the embed.
     * @param {TextAttributes} attributes Attribute information to apply on the
     *                                    embed
     *
     * @public
     */
    insertEmbed (index, embed, attributes = {}) {
      const y = this.doc;
      if (y !== null) {
        transact(y, transaction => {
          const pos = findPosition$1(transaction, this, index);
          insertText(transaction, this, pos, embed, attributes);
        });
      } else {
        /** @type {Array<function>} */ (this._pending).push(() => this.insertEmbed(index, embed, attributes));
      }
    }

    /**
     * Deletes text starting from an index.
     *
     * @param {number} index Index at which to start deleting.
     * @param {number} length The number of characters to remove. Defaults to 1.
     *
     * @public
     */
    delete (index, length) {
      if (length === 0) {
        return
      }
      const y = this.doc;
      if (y !== null) {
        transact(y, transaction => {
          deleteText(transaction, findPosition$1(transaction, this, index), length);
        });
      } else {
        /** @type {Array<function>} */ (this._pending).push(() => this.delete(index, length));
      }
    }

    /**
     * Assigns properties to a range of text.
     *
     * @param {number} index The position where to start formatting.
     * @param {number} length The amount of characters to assign properties to.
     * @param {TextAttributes} attributes Attribute information to apply on the
     *                                    text.
     *
     * @public
     */
    format (index, length, attributes) {
      if (length === 0) {
        return
      }
      const y = this.doc;
      if (y !== null) {
        transact(y, transaction => {
          const pos = findPosition$1(transaction, this, index);
          if (pos.right === null) {
            return
          }
          formatText(transaction, this, pos, length, attributes);
        });
      } else {
        /** @type {Array<function>} */ (this._pending).push(() => this.format(index, length, attributes));
      }
    }

    /**
     * Removes an attribute.
     *
     * @note Xml-Text nodes don't have attributes. You can use this feature to assign properties to complete text-blocks.
     *
     * @param {String} attributeName The attribute name that is to be removed.
     *
     * @public
     */
    removeAttribute (attributeName) {
      if (this.doc !== null) {
        transact(this.doc, transaction => {
          typeMapDelete(transaction, this, attributeName);
        });
      } else {
        /** @type {Array<function>} */ (this._pending).push(() => this.removeAttribute(attributeName));
      }
    }

    /**
     * Sets or updates an attribute.
     *
     * @note Xml-Text nodes don't have attributes. You can use this feature to assign properties to complete text-blocks.
     *
     * @param {String} attributeName The attribute name that is to be set.
     * @param {any} attributeValue The attribute value that is to be set.
     *
     * @public
     */
    setAttribute (attributeName, attributeValue) {
      if (this.doc !== null) {
        transact(this.doc, transaction => {
          typeMapSet(transaction, this, attributeName, attributeValue);
        });
      } else {
        /** @type {Array<function>} */ (this._pending).push(() => this.setAttribute(attributeName, attributeValue));
      }
    }

    /**
     * Returns an attribute value that belongs to the attribute name.
     *
     * @note Xml-Text nodes don't have attributes. You can use this feature to assign properties to complete text-blocks.
     *
     * @param {String} attributeName The attribute name that identifies the
     *                               queried value.
     * @return {any} The queried attribute value.
     *
     * @public
     */
    getAttribute (attributeName) {
      return /** @type {any} */ (typeMapGet(this, attributeName))
    }

    /**
     * Returns all attribute name/value pairs in a JSON Object.
     *
     * @note Xml-Text nodes don't have attributes. You can use this feature to assign properties to complete text-blocks.
     *
     * @param {Snapshot} [snapshot]
     * @return {Object<string, any>} A JSON Object that describes the attributes.
     *
     * @public
     */
    getAttributes (snapshot) {
      return typeMapGetAll(this)
    }

    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     */
    _write (encoder) {
      encoder.writeTypeRef(YTextRefID);
    }
  }

  /**
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   * @return {YText}
   *
   * @private
   * @function
   */
  const readYText = decoder => new YText();

  /**
   * @module YXml
   */

  /**
   * Define the elements to which a set of CSS queries apply.
   * {@link https://developer.mozilla.org/en-US/docs/Web/CSS/CSS_Selectors|CSS_Selectors}
   *
   * @example
   *   query = '.classSelector'
   *   query = 'nodeSelector'
   *   query = '#idSelector'
   *
   * @typedef {string} CSS_Selector
   */

  /**
   * Dom filter function.
   *
   * @callback domFilter
   * @param {string} nodeName The nodeName of the element
   * @param {Map} attributes The map of attributes.
   * @return {boolean} Whether to include the Dom node in the YXmlElement.
   */

  /**
   * Represents a subset of the nodes of a YXmlElement / YXmlFragment and a
   * position within them.
   *
   * Can be created with {@link YXmlFragment#createTreeWalker}
   *
   * @public
   * @implements {Iterable<YXmlElement|YXmlText|YXmlElement|YXmlHook>}
   */
  class YXmlTreeWalker {
    /**
     * @param {YXmlFragment | YXmlElement} root
     * @param {function(AbstractType<any>):boolean} [f]
     */
    constructor (root, f = () => true) {
      this._filter = f;
      this._root = root;
      /**
       * @type {Item}
       */
      this._currentNode = /** @type {Item} */ (root._start);
      this._firstCall = true;
    }

    [Symbol.iterator] () {
      return this
    }

    /**
     * Get the next node.
     *
     * @return {IteratorResult<YXmlElement|YXmlText|YXmlHook>} The next node.
     *
     * @public
     */
    next () {
      /**
       * @type {Item|null}
       */
      let n = this._currentNode;
      let type = n && n.content && /** @type {any} */ (n.content).type;
      if (n !== null && (!this._firstCall || n.deleted || !this._filter(type))) { // if first call, we check if we can use the first item
        do {
          type = /** @type {any} */ (n.content).type;
          if (!n.deleted && (type.constructor === YXmlElement || type.constructor === YXmlFragment) && type._start !== null) {
            // walk down in the tree
            n = type._start;
          } else {
            // walk right or up in the tree
            while (n !== null) {
              if (n.right !== null) {
                n = n.right;
                break
              } else if (n.parent === this._root) {
                n = null;
              } else {
                n = /** @type {AbstractType<any>} */ (n.parent)._item;
              }
            }
          }
        } while (n !== null && (n.deleted || !this._filter(/** @type {ContentType} */ (n.content).type)))
      }
      this._firstCall = false;
      if (n === null) {
        // @ts-ignore
        return { value: undefined, done: true }
      }
      this._currentNode = n;
      return { value: /** @type {any} */ (n.content).type, done: false }
    }
  }

  /**
   * Represents a list of {@link YXmlElement}.and {@link YXmlText} types.
   * A YxmlFragment is similar to a {@link YXmlElement}, but it does not have a
   * nodeName and it does not have attributes. Though it can be bound to a DOM
   * element - in this case the attributes and the nodeName are not shared.
   *
   * @public
   * @extends AbstractType<YXmlEvent>
   */
  class YXmlFragment extends AbstractType {
    constructor () {
      super();
      /**
       * @type {Array<any>|null}
       */
      this._prelimContent = [];
    }

    /**
     * @type {YXmlElement|YXmlText|null}
     */
    get firstChild () {
      const first = this._first;
      return first ? first.content.getContent()[0] : null
    }

    /**
     * Integrate this type into the Yjs instance.
     *
     * * Save this struct in the os
     * * This type is sent to other client
     * * Observer functions are fired
     *
     * @param {Doc} y The Yjs instance
     * @param {Item} item
     */
    _integrate (y, item) {
      super._integrate(y, item);
      this.insert(0, /** @type {Array<any>} */ (this._prelimContent));
      this._prelimContent = null;
    }

    _copy () {
      return new YXmlFragment()
    }

    /**
     * @return {YXmlFragment}
     */
    clone () {
      const el = new YXmlFragment();
      // @ts-ignore
      el.insert(0, this.toArray().map(item => item instanceof AbstractType ? item.clone() : item));
      return el
    }

    get length () {
      return this._prelimContent === null ? this._length : this._prelimContent.length
    }

    /**
     * Create a subtree of childNodes.
     *
     * @example
     * const walker = elem.createTreeWalker(dom => dom.nodeName === 'div')
     * for (let node in walker) {
     *   // `node` is a div node
     *   nop(node)
     * }
     *
     * @param {function(AbstractType<any>):boolean} filter Function that is called on each child element and
     *                          returns a Boolean indicating whether the child
     *                          is to be included in the subtree.
     * @return {YXmlTreeWalker} A subtree and a position within it.
     *
     * @public
     */
    createTreeWalker (filter) {
      return new YXmlTreeWalker(this, filter)
    }

    /**
     * Returns the first YXmlElement that matches the query.
     * Similar to DOM's {@link querySelector}.
     *
     * Query support:
     *   - tagname
     * TODO:
     *   - id
     *   - attribute
     *
     * @param {CSS_Selector} query The query on the children.
     * @return {YXmlElement|YXmlText|YXmlHook|null} The first element that matches the query or null.
     *
     * @public
     */
    querySelector (query) {
      query = query.toUpperCase();
      // @ts-ignore
      const iterator = new YXmlTreeWalker(this, element => element.nodeName && element.nodeName.toUpperCase() === query);
      const next = iterator.next();
      if (next.done) {
        return null
      } else {
        return next.value
      }
    }

    /**
     * Returns all YXmlElements that match the query.
     * Similar to Dom's {@link querySelectorAll}.
     *
     * @todo Does not yet support all queries. Currently only query by tagName.
     *
     * @param {CSS_Selector} query The query on the children
     * @return {Array<YXmlElement|YXmlText|YXmlHook|null>} The elements that match this query.
     *
     * @public
     */
    querySelectorAll (query) {
      query = query.toUpperCase();
      // @ts-ignore
      return Array.from(new YXmlTreeWalker(this, element => element.nodeName && element.nodeName.toUpperCase() === query))
    }

    /**
     * Creates YXmlEvent and calls observers.
     *
     * @param {Transaction} transaction
     * @param {Set<null|string>} parentSubs Keys changed on this type. `null` if list was modified.
     */
    _callObserver (transaction, parentSubs) {
      callTypeObservers(this, transaction, new YXmlEvent(this, parentSubs, transaction));
    }

    /**
     * Get the string representation of all the children of this YXmlFragment.
     *
     * @return {string} The string representation of all children.
     */
    toString () {
      return typeListMap(this, xml => xml.toString()).join('')
    }

    /**
     * @return {string}
     */
    toJSON () {
      return this.toString()
    }

    /**
     * Creates a Dom Element that mirrors this YXmlElement.
     *
     * @param {Document} [_document=document] The document object (you must define
     *                                        this when calling this method in
     *                                        nodejs)
     * @param {Object<string, any>} [hooks={}] Optional property to customize how hooks
     *                                             are presented in the DOM
     * @param {any} [binding] You should not set this property. This is
     *                               used if DomBinding wants to create a
     *                               association to the created DOM type.
     * @return {Node} The {@link https://developer.mozilla.org/en-US/docs/Web/API/Element|Dom Element}
     *
     * @public
     */
    toDOM (_document = document, hooks = {}, binding) {
      const fragment = _document.createDocumentFragment();
      if (binding !== undefined) {
        binding._createAssociation(fragment, this);
      }
      typeListForEach(this, xmlType => {
        fragment.insertBefore(xmlType.toDOM(_document, hooks, binding), null);
      });
      return fragment
    }

    /**
     * Inserts new content at an index.
     *
     * @example
     *  // Insert character 'a' at position 0
     *  xml.insert(0, [new Y.XmlText('text')])
     *
     * @param {number} index The index to insert content at
     * @param {Array<YXmlElement|YXmlText>} content The array of content
     */
    insert (index, content) {
      if (this.doc !== null) {
        transact(this.doc, transaction => {
          typeListInsertGenerics(transaction, this, index, content);
        });
      } else {
        // @ts-ignore _prelimContent is defined because this is not yet integrated
        this._prelimContent.splice(index, 0, ...content);
      }
    }

    /**
     * Inserts new content at an index.
     *
     * @example
     *  // Insert character 'a' at position 0
     *  xml.insert(0, [new Y.XmlText('text')])
     *
     * @param {null|Item|YXmlElement|YXmlText} ref The index to insert content at
     * @param {Array<YXmlElement|YXmlText>} content The array of content
     */
    insertAfter (ref, content) {
      if (this.doc !== null) {
        transact(this.doc, transaction => {
          const refItem = (ref && ref instanceof AbstractType) ? ref._item : ref;
          typeListInsertGenericsAfter(transaction, this, refItem, content);
        });
      } else {
        const pc = /** @type {Array<any>} */ (this._prelimContent);
        const index = ref === null ? 0 : pc.findIndex(el => el === ref) + 1;
        if (index === 0 && ref !== null) {
          throw create$2('Reference item not found')
        }
        pc.splice(index, 0, ...content);
      }
    }

    /**
     * Deletes elements starting from an index.
     *
     * @param {number} index Index at which to start deleting elements
     * @param {number} [length=1] The number of elements to remove. Defaults to 1.
     */
    delete (index, length = 1) {
      if (this.doc !== null) {
        transact(this.doc, transaction => {
          typeListDelete(transaction, this, index, length);
        });
      } else {
        // @ts-ignore _prelimContent is defined because this is not yet integrated
        this._prelimContent.splice(index, length);
      }
    }

    /**
     * Transforms this YArray to a JavaScript Array.
     *
     * @return {Array<YXmlElement|YXmlText|YXmlHook>}
     */
    toArray () {
      return typeListToArray(this)
    }

    /**
     * Appends content to this YArray.
     *
     * @param {Array<YXmlElement|YXmlText>} content Array of content to append.
     */
    push (content) {
      this.insert(this.length, content);
    }

    /**
     * Preppends content to this YArray.
     *
     * @param {Array<YXmlElement|YXmlText>} content Array of content to preppend.
     */
    unshift (content) {
      this.insert(0, content);
    }

    /**
     * Returns the i-th element from a YArray.
     *
     * @param {number} index The index of the element to return from the YArray
     * @return {YXmlElement|YXmlText}
     */
    get (index) {
      return typeListGet(this, index)
    }

    /**
     * Transforms this YArray to a JavaScript Array.
     *
     * @param {number} [start]
     * @param {number} [end]
     * @return {Array<YXmlElement|YXmlText>}
     */
    slice (start = 0, end = this.length) {
      return typeListSlice(this, start, end)
    }

    /**
     * Executes a provided function on once on overy child element.
     *
     * @param {function(YXmlElement|YXmlText,number, typeof this):void} f A function to execute on every element of this YArray.
     */
    forEach (f) {
      typeListForEach(this, f);
    }

    /**
     * Transform the properties of this type to binary and write it to an
     * BinaryEncoder.
     *
     * This is called when this Item is sent to a remote peer.
     *
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder The encoder to write data to.
     */
    _write (encoder) {
      encoder.writeTypeRef(YXmlFragmentRefID);
    }
  }

  /**
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   * @return {YXmlFragment}
   *
   * @private
   * @function
   */
  const readYXmlFragment = decoder => new YXmlFragment();

  /**
   * An YXmlElement imitates the behavior of a
   * {@link https://developer.mozilla.org/en-US/docs/Web/API/Element|Dom Element}.
   *
   * * An YXmlElement has attributes (key value pairs)
   * * An YXmlElement has childElements that must inherit from YXmlElement
   */
  class YXmlElement extends YXmlFragment {
    constructor (nodeName = 'UNDEFINED') {
      super();
      this.nodeName = nodeName;
      /**
       * @type {Map<string, any>|null}
       */
      this._prelimAttrs = new Map();
    }

    /**
     * @type {YXmlElement|YXmlText|null}
     */
    get nextSibling () {
      const n = this._item ? this._item.next : null;
      return n ? /** @type {YXmlElement|YXmlText} */ (/** @type {ContentType} */ (n.content).type) : null
    }

    /**
     * @type {YXmlElement|YXmlText|null}
     */
    get prevSibling () {
      const n = this._item ? this._item.prev : null;
      return n ? /** @type {YXmlElement|YXmlText} */ (/** @type {ContentType} */ (n.content).type) : null
    }

    /**
     * Integrate this type into the Yjs instance.
     *
     * * Save this struct in the os
     * * This type is sent to other client
     * * Observer functions are fired
     *
     * @param {Doc} y The Yjs instance
     * @param {Item} item
     */
    _integrate (y, item) {
      super._integrate(y, item)
      ;(/** @type {Map<string, any>} */ (this._prelimAttrs)).forEach((value, key) => {
        this.setAttribute(key, value);
      });
      this._prelimAttrs = null;
    }

    /**
     * Creates an Item with the same effect as this Item (without position effect)
     *
     * @return {YXmlElement}
     */
    _copy () {
      return new YXmlElement(this.nodeName)
    }

    /**
     * @return {YXmlElement}
     */
    clone () {
      const el = new YXmlElement(this.nodeName);
      const attrs = this.getAttributes();
      for (const key in attrs) {
        el.setAttribute(key, attrs[key]);
      }
      // @ts-ignore
      el.insert(0, this.toArray().map(item => item instanceof AbstractType ? item.clone() : item));
      return el
    }

    /**
     * Returns the XML serialization of this YXmlElement.
     * The attributes are ordered by attribute-name, so you can easily use this
     * method to compare YXmlElements
     *
     * @return {string} The string representation of this type.
     *
     * @public
     */
    toString () {
      const attrs = this.getAttributes();
      const stringBuilder = [];
      const keys = [];
      for (const key in attrs) {
        keys.push(key);
      }
      keys.sort();
      const keysLen = keys.length;
      for (let i = 0; i < keysLen; i++) {
        const key = keys[i];
        stringBuilder.push(key + '="' + attrs[key] + '"');
      }
      const nodeName = this.nodeName.toLocaleLowerCase();
      const attrsString = stringBuilder.length > 0 ? ' ' + stringBuilder.join(' ') : '';
      return `<${nodeName}${attrsString}>${super.toString()}</${nodeName}>`
    }

    /**
     * Removes an attribute from this YXmlElement.
     *
     * @param {String} attributeName The attribute name that is to be removed.
     *
     * @public
     */
    removeAttribute (attributeName) {
      if (this.doc !== null) {
        transact(this.doc, transaction => {
          typeMapDelete(transaction, this, attributeName);
        });
      } else {
        /** @type {Map<string,any>} */ (this._prelimAttrs).delete(attributeName);
      }
    }

    /**
     * Sets or updates an attribute.
     *
     * @param {String} attributeName The attribute name that is to be set.
     * @param {String} attributeValue The attribute value that is to be set.
     *
     * @public
     */
    setAttribute (attributeName, attributeValue) {
      if (this.doc !== null) {
        transact(this.doc, transaction => {
          typeMapSet(transaction, this, attributeName, attributeValue);
        });
      } else {
        /** @type {Map<string, any>} */ (this._prelimAttrs).set(attributeName, attributeValue);
      }
    }

    /**
     * Returns an attribute value that belongs to the attribute name.
     *
     * @param {String} attributeName The attribute name that identifies the
     *                               queried value.
     * @return {String} The queried attribute value.
     *
     * @public
     */
    getAttribute (attributeName) {
      return /** @type {any} */ (typeMapGet(this, attributeName))
    }

    /**
     * Returns whether an attribute exists
     *
     * @param {String} attributeName The attribute name to check for existence.
     * @return {boolean} whether the attribute exists.
     *
     * @public
     */
    hasAttribute (attributeName) {
      return /** @type {any} */ (typeMapHas(this, attributeName))
    }

    /**
     * Returns all attribute name/value pairs in a JSON Object.
     *
     * @param {Snapshot} [snapshot]
     * @return {Object<string, any>} A JSON Object that describes the attributes.
     *
     * @public
     */
    getAttributes (snapshot) {
      return typeMapGetAll(this)
    }

    /**
     * Creates a Dom Element that mirrors this YXmlElement.
     *
     * @param {Document} [_document=document] The document object (you must define
     *                                        this when calling this method in
     *                                        nodejs)
     * @param {Object<string, any>} [hooks={}] Optional property to customize how hooks
     *                                             are presented in the DOM
     * @param {any} [binding] You should not set this property. This is
     *                               used if DomBinding wants to create a
     *                               association to the created DOM type.
     * @return {Node} The {@link https://developer.mozilla.org/en-US/docs/Web/API/Element|Dom Element}
     *
     * @public
     */
    toDOM (_document = document, hooks = {}, binding) {
      const dom = _document.createElement(this.nodeName);
      const attrs = this.getAttributes();
      for (const key in attrs) {
        dom.setAttribute(key, attrs[key]);
      }
      typeListForEach(this, yxml => {
        dom.appendChild(yxml.toDOM(_document, hooks, binding));
      });
      if (binding !== undefined) {
        binding._createAssociation(dom, this);
      }
      return dom
    }

    /**
     * Transform the properties of this type to binary and write it to an
     * BinaryEncoder.
     *
     * This is called when this Item is sent to a remote peer.
     *
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder The encoder to write data to.
     */
    _write (encoder) {
      encoder.writeTypeRef(YXmlElementRefID);
      encoder.writeKey(this.nodeName);
    }
  }

  /**
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   * @return {YXmlElement}
   *
   * @function
   */
  const readYXmlElement = decoder => new YXmlElement(decoder.readKey());

  /**
   * @extends YEvent<YXmlElement|YXmlText|YXmlFragment>
   * An Event that describes changes on a YXml Element or Yxml Fragment
   */
  class YXmlEvent extends YEvent {
    /**
     * @param {YXmlElement|YXmlText|YXmlFragment} target The target on which the event is created.
     * @param {Set<string|null>} subs The set of changed attributes. `null` is included if the
     *                   child list changed.
     * @param {Transaction} transaction The transaction instance with wich the
     *                                  change was created.
     */
    constructor (target, subs, transaction) {
      super(target, transaction);
      /**
       * Whether the children changed.
       * @type {Boolean}
       * @private
       */
      this.childListChanged = false;
      /**
       * Set of all changed attributes.
       * @type {Set<string>}
       */
      this.attributesChanged = new Set();
      subs.forEach((sub) => {
        if (sub === null) {
          this.childListChanged = true;
        } else {
          this.attributesChanged.add(sub);
        }
      });
    }
  }

  /**
   * You can manage binding to a custom type with YXmlHook.
   *
   * @extends {YMap<any>}
   */
  class YXmlHook extends YMap {
    /**
     * @param {string} hookName nodeName of the Dom Node.
     */
    constructor (hookName) {
      super();
      /**
       * @type {string}
       */
      this.hookName = hookName;
    }

    /**
     * Creates an Item with the same effect as this Item (without position effect)
     */
    _copy () {
      return new YXmlHook(this.hookName)
    }

    /**
     * @return {YXmlHook}
     */
    clone () {
      const el = new YXmlHook(this.hookName);
      this.forEach((value, key) => {
        el.set(key, value);
      });
      return el
    }

    /**
     * Creates a Dom Element that mirrors this YXmlElement.
     *
     * @param {Document} [_document=document] The document object (you must define
     *                                        this when calling this method in
     *                                        nodejs)
     * @param {Object.<string, any>} [hooks] Optional property to customize how hooks
     *                                             are presented in the DOM
     * @param {any} [binding] You should not set this property. This is
     *                               used if DomBinding wants to create a
     *                               association to the created DOM type
     * @return {Element} The {@link https://developer.mozilla.org/en-US/docs/Web/API/Element|Dom Element}
     *
     * @public
     */
    toDOM (_document = document, hooks = {}, binding) {
      const hook = hooks[this.hookName];
      let dom;
      if (hook !== undefined) {
        dom = hook.createDom(this);
      } else {
        dom = document.createElement(this.hookName);
      }
      dom.setAttribute('data-yjs-hook', this.hookName);
      if (binding !== undefined) {
        binding._createAssociation(dom, this);
      }
      return dom
    }

    /**
     * Transform the properties of this type to binary and write it to an
     * BinaryEncoder.
     *
     * This is called when this Item is sent to a remote peer.
     *
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder The encoder to write data to.
     */
    _write (encoder) {
      encoder.writeTypeRef(YXmlHookRefID);
      encoder.writeKey(this.hookName);
    }
  }

  /**
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   * @return {YXmlHook}
   *
   * @private
   * @function
   */
  const readYXmlHook = decoder =>
    new YXmlHook(decoder.readKey());

  /**
   * Represents text in a Dom Element. In the future this type will also handle
   * simple formatting information like bold and italic.
   */
  class YXmlText extends YText {
    /**
     * @type {YXmlElement|YXmlText|null}
     */
    get nextSibling () {
      const n = this._item ? this._item.next : null;
      return n ? /** @type {YXmlElement|YXmlText} */ (/** @type {ContentType} */ (n.content).type) : null
    }

    /**
     * @type {YXmlElement|YXmlText|null}
     */
    get prevSibling () {
      const n = this._item ? this._item.prev : null;
      return n ? /** @type {YXmlElement|YXmlText} */ (/** @type {ContentType} */ (n.content).type) : null
    }

    _copy () {
      return new YXmlText()
    }

    /**
     * @return {YXmlText}
     */
    clone () {
      const text = new YXmlText();
      text.applyDelta(this.toDelta());
      return text
    }

    /**
     * Creates a Dom Element that mirrors this YXmlText.
     *
     * @param {Document} [_document=document] The document object (you must define
     *                                        this when calling this method in
     *                                        nodejs)
     * @param {Object<string, any>} [hooks] Optional property to customize how hooks
     *                                             are presented in the DOM
     * @param {any} [binding] You should not set this property. This is
     *                               used if DomBinding wants to create a
     *                               association to the created DOM type.
     * @return {Text} The {@link https://developer.mozilla.org/en-US/docs/Web/API/Element|Dom Element}
     *
     * @public
     */
    toDOM (_document = document, hooks, binding) {
      const dom = _document.createTextNode(this.toString());
      if (binding !== undefined) {
        binding._createAssociation(dom, this);
      }
      return dom
    }

    toString () {
      // @ts-ignore
      return this.toDelta().map(delta => {
        const nestedNodes = [];
        for (const nodeName in delta.attributes) {
          const attrs = [];
          for (const key in delta.attributes[nodeName]) {
            attrs.push({ key, value: delta.attributes[nodeName][key] });
          }
          // sort attributes to get a unique order
          attrs.sort((a, b) => a.key < b.key ? -1 : 1);
          nestedNodes.push({ nodeName, attrs });
        }
        // sort node order to get a unique order
        nestedNodes.sort((a, b) => a.nodeName < b.nodeName ? -1 : 1);
        // now convert to dom string
        let str = '';
        for (let i = 0; i < nestedNodes.length; i++) {
          const node = nestedNodes[i];
          str += `<${node.nodeName}`;
          for (let j = 0; j < node.attrs.length; j++) {
            const attr = node.attrs[j];
            str += ` ${attr.key}="${attr.value}"`;
          }
          str += '>';
        }
        str += delta.insert;
        for (let i = nestedNodes.length - 1; i >= 0; i--) {
          str += `</${nestedNodes[i].nodeName}>`;
        }
        return str
      }).join('')
    }

    /**
     * @return {string}
     */
    toJSON () {
      return this.toString()
    }

    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     */
    _write (encoder) {
      encoder.writeTypeRef(YXmlTextRefID);
    }
  }

  /**
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   * @return {YXmlText}
   *
   * @private
   * @function
   */
  const readYXmlText = decoder => new YXmlText();

  class AbstractStruct {
    /**
     * @param {ID} id
     * @param {number} length
     */
    constructor (id, length) {
      this.id = id;
      this.length = length;
    }

    /**
     * @type {boolean}
     */
    get deleted () {
      throw methodUnimplemented()
    }

    /**
     * Merge this struct with the item to the right.
     * This method is already assuming that `this.id.clock + this.length === this.id.clock`.
     * Also this method does *not* remove right from StructStore!
     * @param {AbstractStruct} right
     * @return {boolean} wether this merged with right
     */
    mergeWith (right) {
      return false
    }

    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder The encoder to write data to.
     * @param {number} offset
     * @param {number} encodingRef
     */
    write (encoder, offset, encodingRef) {
      throw methodUnimplemented()
    }

    /**
     * @param {Transaction} transaction
     * @param {number} offset
     */
    integrate (transaction, offset) {
      throw methodUnimplemented()
    }
  }

  const structGCRefNumber = 0;

  /**
   * @private
   */
  class GC extends AbstractStruct {
    get deleted () {
      return true
    }

    delete () {}

    /**
     * @param {GC} right
     * @return {boolean}
     */
    mergeWith (right) {
      if (this.constructor !== right.constructor) {
        return false
      }
      this.length += right.length;
      return true
    }

    /**
     * @param {Transaction} transaction
     * @param {number} offset
     */
    integrate (transaction, offset) {
      if (offset > 0) {
        this.id.clock += offset;
        this.length -= offset;
      }
      addStruct(transaction.doc.store, this);
    }

    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     * @param {number} offset
     */
    write (encoder, offset) {
      encoder.writeInfo(structGCRefNumber);
      encoder.writeLen(this.length - offset);
    }

    /**
     * @param {Transaction} transaction
     * @param {StructStore} store
     * @return {null | number}
     */
    getMissing (transaction, store) {
      return null
    }
  }

  class ContentBinary {
    /**
     * @param {Uint8Array} content
     */
    constructor (content) {
      this.content = content;
    }

    /**
     * @return {number}
     */
    getLength () {
      return 1
    }

    /**
     * @return {Array<any>}
     */
    getContent () {
      return [this.content]
    }

    /**
     * @return {boolean}
     */
    isCountable () {
      return true
    }

    /**
     * @return {ContentBinary}
     */
    copy () {
      return new ContentBinary(this.content)
    }

    /**
     * @param {number} offset
     * @return {ContentBinary}
     */
    splice (offset) {
      throw methodUnimplemented()
    }

    /**
     * @param {ContentBinary} right
     * @return {boolean}
     */
    mergeWith (right) {
      return false
    }

    /**
     * @param {Transaction} transaction
     * @param {Item} item
     */
    integrate (transaction, item) {}
    /**
     * @param {Transaction} transaction
     */
    delete (transaction) {}
    /**
     * @param {StructStore} store
     */
    gc (store) {}
    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     * @param {number} offset
     */
    write (encoder, offset) {
      encoder.writeBuf(this.content);
    }

    /**
     * @return {number}
     */
    getRef () {
      return 3
    }
  }

  /**
   * @param {UpdateDecoderV1 | UpdateDecoderV2 } decoder
   * @return {ContentBinary}
   */
  const readContentBinary = decoder => new ContentBinary(decoder.readBuf());

  class ContentDeleted {
    /**
     * @param {number} len
     */
    constructor (len) {
      this.len = len;
    }

    /**
     * @return {number}
     */
    getLength () {
      return this.len
    }

    /**
     * @return {Array<any>}
     */
    getContent () {
      return []
    }

    /**
     * @return {boolean}
     */
    isCountable () {
      return false
    }

    /**
     * @return {ContentDeleted}
     */
    copy () {
      return new ContentDeleted(this.len)
    }

    /**
     * @param {number} offset
     * @return {ContentDeleted}
     */
    splice (offset) {
      const right = new ContentDeleted(this.len - offset);
      this.len = offset;
      return right
    }

    /**
     * @param {ContentDeleted} right
     * @return {boolean}
     */
    mergeWith (right) {
      this.len += right.len;
      return true
    }

    /**
     * @param {Transaction} transaction
     * @param {Item} item
     */
    integrate (transaction, item) {
      addToDeleteSet(transaction.deleteSet, item.id.client, item.id.clock, this.len);
      item.markDeleted();
    }

    /**
     * @param {Transaction} transaction
     */
    delete (transaction) {}
    /**
     * @param {StructStore} store
     */
    gc (store) {}
    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     * @param {number} offset
     */
    write (encoder, offset) {
      encoder.writeLen(this.len - offset);
    }

    /**
     * @return {number}
     */
    getRef () {
      return 1
    }
  }

  /**
   * @private
   *
   * @param {UpdateDecoderV1 | UpdateDecoderV2 } decoder
   * @return {ContentDeleted}
   */
  const readContentDeleted = decoder => new ContentDeleted(decoder.readLen());

  /**
   * @param {string} guid
   * @param {Object<string, any>} opts
   */
  const createDocFromOpts = (guid, opts) => new Doc({ guid, ...opts, shouldLoad: opts.shouldLoad || opts.autoLoad || false });

  /**
   * @private
   */
  class ContentDoc {
    /**
     * @param {Doc} doc
     */
    constructor (doc) {
      if (doc._item) {
        console.error('This document was already integrated as a sub-document. You should create a second instance instead with the same guid.');
      }
      /**
       * @type {Doc}
       */
      this.doc = doc;
      /**
       * @type {any}
       */
      const opts = {};
      this.opts = opts;
      if (!doc.gc) {
        opts.gc = false;
      }
      if (doc.autoLoad) {
        opts.autoLoad = true;
      }
      if (doc.meta !== null) {
        opts.meta = doc.meta;
      }
    }

    /**
     * @return {number}
     */
    getLength () {
      return 1
    }

    /**
     * @return {Array<any>}
     */
    getContent () {
      return [this.doc]
    }

    /**
     * @return {boolean}
     */
    isCountable () {
      return true
    }

    /**
     * @return {ContentDoc}
     */
    copy () {
      return new ContentDoc(createDocFromOpts(this.doc.guid, this.opts))
    }

    /**
     * @param {number} offset
     * @return {ContentDoc}
     */
    splice (offset) {
      throw methodUnimplemented()
    }

    /**
     * @param {ContentDoc} right
     * @return {boolean}
     */
    mergeWith (right) {
      return false
    }

    /**
     * @param {Transaction} transaction
     * @param {Item} item
     */
    integrate (transaction, item) {
      // this needs to be reflected in doc.destroy as well
      this.doc._item = item;
      transaction.subdocsAdded.add(this.doc);
      if (this.doc.shouldLoad) {
        transaction.subdocsLoaded.add(this.doc);
      }
    }

    /**
     * @param {Transaction} transaction
     */
    delete (transaction) {
      if (transaction.subdocsAdded.has(this.doc)) {
        transaction.subdocsAdded.delete(this.doc);
      } else {
        transaction.subdocsRemoved.add(this.doc);
      }
    }

    /**
     * @param {StructStore} store
     */
    gc (store) { }

    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     * @param {number} offset
     */
    write (encoder, offset) {
      encoder.writeString(this.doc.guid);
      encoder.writeAny(this.opts);
    }

    /**
     * @return {number}
     */
    getRef () {
      return 9
    }
  }

  /**
   * @private
   *
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   * @return {ContentDoc}
   */
  const readContentDoc = decoder => new ContentDoc(createDocFromOpts(decoder.readString(), decoder.readAny()));

  /**
   * @private
   */
  class ContentEmbed {
    /**
     * @param {Object} embed
     */
    constructor (embed) {
      this.embed = embed;
    }

    /**
     * @return {number}
     */
    getLength () {
      return 1
    }

    /**
     * @return {Array<any>}
     */
    getContent () {
      return [this.embed]
    }

    /**
     * @return {boolean}
     */
    isCountable () {
      return true
    }

    /**
     * @return {ContentEmbed}
     */
    copy () {
      return new ContentEmbed(this.embed)
    }

    /**
     * @param {number} offset
     * @return {ContentEmbed}
     */
    splice (offset) {
      throw methodUnimplemented()
    }

    /**
     * @param {ContentEmbed} right
     * @return {boolean}
     */
    mergeWith (right) {
      return false
    }

    /**
     * @param {Transaction} transaction
     * @param {Item} item
     */
    integrate (transaction, item) {}
    /**
     * @param {Transaction} transaction
     */
    delete (transaction) {}
    /**
     * @param {StructStore} store
     */
    gc (store) {}
    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     * @param {number} offset
     */
    write (encoder, offset) {
      encoder.writeJSON(this.embed);
    }

    /**
     * @return {number}
     */
    getRef () {
      return 5
    }
  }

  /**
   * @private
   *
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   * @return {ContentEmbed}
   */
  const readContentEmbed = decoder => new ContentEmbed(decoder.readJSON());

  /**
   * @private
   */
  class ContentFormat {
    /**
     * @param {string} key
     * @param {Object} value
     */
    constructor (key, value) {
      this.key = key;
      this.value = value;
    }

    /**
     * @return {number}
     */
    getLength () {
      return 1
    }

    /**
     * @return {Array<any>}
     */
    getContent () {
      return []
    }

    /**
     * @return {boolean}
     */
    isCountable () {
      return false
    }

    /**
     * @return {ContentFormat}
     */
    copy () {
      return new ContentFormat(this.key, this.value)
    }

    /**
     * @param {number} offset
     * @return {ContentFormat}
     */
    splice (offset) {
      throw methodUnimplemented()
    }

    /**
     * @param {ContentFormat} right
     * @return {boolean}
     */
    mergeWith (right) {
      return false
    }

    /**
     * @param {Transaction} transaction
     * @param {Item} item
     */
    integrate (transaction, item) {
      // @todo searchmarker are currently unsupported for rich text documents
      /** @type {AbstractType<any>} */ (item.parent)._searchMarker = null;
    }

    /**
     * @param {Transaction} transaction
     */
    delete (transaction) {}
    /**
     * @param {StructStore} store
     */
    gc (store) {}
    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     * @param {number} offset
     */
    write (encoder, offset) {
      encoder.writeKey(this.key);
      encoder.writeJSON(this.value);
    }

    /**
     * @return {number}
     */
    getRef () {
      return 6
    }
  }

  /**
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   * @return {ContentFormat}
   */
  const readContentFormat = decoder => new ContentFormat(decoder.readKey(), decoder.readJSON());

  /**
   * @private
   */
  class ContentJSON {
    /**
     * @param {Array<any>} arr
     */
    constructor (arr) {
      /**
       * @type {Array<any>}
       */
      this.arr = arr;
    }

    /**
     * @return {number}
     */
    getLength () {
      return this.arr.length
    }

    /**
     * @return {Array<any>}
     */
    getContent () {
      return this.arr
    }

    /**
     * @return {boolean}
     */
    isCountable () {
      return true
    }

    /**
     * @return {ContentJSON}
     */
    copy () {
      return new ContentJSON(this.arr)
    }

    /**
     * @param {number} offset
     * @return {ContentJSON}
     */
    splice (offset) {
      const right = new ContentJSON(this.arr.slice(offset));
      this.arr = this.arr.slice(0, offset);
      return right
    }

    /**
     * @param {ContentJSON} right
     * @return {boolean}
     */
    mergeWith (right) {
      this.arr = this.arr.concat(right.arr);
      return true
    }

    /**
     * @param {Transaction} transaction
     * @param {Item} item
     */
    integrate (transaction, item) {}
    /**
     * @param {Transaction} transaction
     */
    delete (transaction) {}
    /**
     * @param {StructStore} store
     */
    gc (store) {}
    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     * @param {number} offset
     */
    write (encoder, offset) {
      const len = this.arr.length;
      encoder.writeLen(len - offset);
      for (let i = offset; i < len; i++) {
        const c = this.arr[i];
        encoder.writeString(c === undefined ? 'undefined' : JSON.stringify(c));
      }
    }

    /**
     * @return {number}
     */
    getRef () {
      return 2
    }
  }

  /**
   * @private
   *
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   * @return {ContentJSON}
   */
  const readContentJSON = decoder => {
    const len = decoder.readLen();
    const cs = [];
    for (let i = 0; i < len; i++) {
      const c = decoder.readString();
      if (c === 'undefined') {
        cs.push(undefined);
      } else {
        cs.push(JSON.parse(c));
      }
    }
    return new ContentJSON(cs)
  };

  class ContentAny {
    /**
     * @param {Array<any>} arr
     */
    constructor (arr) {
      /**
       * @type {Array<any>}
       */
      this.arr = arr;
    }

    /**
     * @return {number}
     */
    getLength () {
      return this.arr.length
    }

    /**
     * @return {Array<any>}
     */
    getContent () {
      return this.arr
    }

    /**
     * @return {boolean}
     */
    isCountable () {
      return true
    }

    /**
     * @return {ContentAny}
     */
    copy () {
      return new ContentAny(this.arr)
    }

    /**
     * @param {number} offset
     * @return {ContentAny}
     */
    splice (offset) {
      const right = new ContentAny(this.arr.slice(offset));
      this.arr = this.arr.slice(0, offset);
      return right
    }

    /**
     * @param {ContentAny} right
     * @return {boolean}
     */
    mergeWith (right) {
      this.arr = this.arr.concat(right.arr);
      return true
    }

    /**
     * @param {Transaction} transaction
     * @param {Item} item
     */
    integrate (transaction, item) {}
    /**
     * @param {Transaction} transaction
     */
    delete (transaction) {}
    /**
     * @param {StructStore} store
     */
    gc (store) {}
    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     * @param {number} offset
     */
    write (encoder, offset) {
      const len = this.arr.length;
      encoder.writeLen(len - offset);
      for (let i = offset; i < len; i++) {
        const c = this.arr[i];
        encoder.writeAny(c);
      }
    }

    /**
     * @return {number}
     */
    getRef () {
      return 8
    }
  }

  /**
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   * @return {ContentAny}
   */
  const readContentAny = decoder => {
    const len = decoder.readLen();
    const cs = [];
    for (let i = 0; i < len; i++) {
      cs.push(decoder.readAny());
    }
    return new ContentAny(cs)
  };

  /**
   * @private
   */
  class ContentString {
    /**
     * @param {string} str
     */
    constructor (str) {
      /**
       * @type {string}
       */
      this.str = str;
    }

    /**
     * @return {number}
     */
    getLength () {
      return this.str.length
    }

    /**
     * @return {Array<any>}
     */
    getContent () {
      return this.str.split('')
    }

    /**
     * @return {boolean}
     */
    isCountable () {
      return true
    }

    /**
     * @return {ContentString}
     */
    copy () {
      return new ContentString(this.str)
    }

    /**
     * @param {number} offset
     * @return {ContentString}
     */
    splice (offset) {
      const right = new ContentString(this.str.slice(offset));
      this.str = this.str.slice(0, offset);

      // Prevent encoding invalid documents because of splitting of surrogate pairs: https://github.com/yjs/yjs/issues/248
      const firstCharCode = this.str.charCodeAt(offset - 1);
      if (firstCharCode >= 0xD800 && firstCharCode <= 0xDBFF) {
        // Last character of the left split is the start of a surrogate utf16/ucs2 pair.
        // We don't support splitting of surrogate pairs because this may lead to invalid documents.
        // Replace the invalid character with a unicode replacement character (� / U+FFFD)
        this.str = this.str.slice(0, offset - 1) + '�';
        // replace right as well
        right.str = '�' + right.str.slice(1);
      }
      return right
    }

    /**
     * @param {ContentString} right
     * @return {boolean}
     */
    mergeWith (right) {
      this.str += right.str;
      return true
    }

    /**
     * @param {Transaction} transaction
     * @param {Item} item
     */
    integrate (transaction, item) {}
    /**
     * @param {Transaction} transaction
     */
    delete (transaction) {}
    /**
     * @param {StructStore} store
     */
    gc (store) {}
    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     * @param {number} offset
     */
    write (encoder, offset) {
      encoder.writeString(offset === 0 ? this.str : this.str.slice(offset));
    }

    /**
     * @return {number}
     */
    getRef () {
      return 4
    }
  }

  /**
   * @private
   *
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   * @return {ContentString}
   */
  const readContentString = decoder => new ContentString(decoder.readString());

  /**
   * @type {Array<function(UpdateDecoderV1 | UpdateDecoderV2):AbstractType<any>>}
   * @private
   */
  const typeRefs = [
    readYArray,
    readYMap,
    readYText,
    readYXmlElement,
    readYXmlFragment,
    readYXmlHook,
    readYXmlText
  ];

  const YArrayRefID = 0;
  const YMapRefID = 1;
  const YTextRefID = 2;
  const YXmlElementRefID = 3;
  const YXmlFragmentRefID = 4;
  const YXmlHookRefID = 5;
  const YXmlTextRefID = 6;

  /**
   * @private
   */
  class ContentType {
    /**
     * @param {AbstractType<any>} type
     */
    constructor (type) {
      /**
       * @type {AbstractType<any>}
       */
      this.type = type;
    }

    /**
     * @return {number}
     */
    getLength () {
      return 1
    }

    /**
     * @return {Array<any>}
     */
    getContent () {
      return [this.type]
    }

    /**
     * @return {boolean}
     */
    isCountable () {
      return true
    }

    /**
     * @return {ContentType}
     */
    copy () {
      return new ContentType(this.type._copy())
    }

    /**
     * @param {number} offset
     * @return {ContentType}
     */
    splice (offset) {
      throw methodUnimplemented()
    }

    /**
     * @param {ContentType} right
     * @return {boolean}
     */
    mergeWith (right) {
      return false
    }

    /**
     * @param {Transaction} transaction
     * @param {Item} item
     */
    integrate (transaction, item) {
      this.type._integrate(transaction.doc, item);
    }

    /**
     * @param {Transaction} transaction
     */
    delete (transaction) {
      let item = this.type._start;
      while (item !== null) {
        if (!item.deleted) {
          item.delete(transaction);
        } else {
          // This will be gc'd later and we want to merge it if possible
          // We try to merge all deleted items after each transaction,
          // but we have no knowledge about that this needs to be merged
          // since it is not in transaction.ds. Hence we add it to transaction._mergeStructs
          transaction._mergeStructs.push(item);
        }
        item = item.right;
      }
      this.type._map.forEach(item => {
        if (!item.deleted) {
          item.delete(transaction);
        } else {
          // same as above
          transaction._mergeStructs.push(item);
        }
      });
      transaction.changed.delete(this.type);
    }

    /**
     * @param {StructStore} store
     */
    gc (store) {
      let item = this.type._start;
      while (item !== null) {
        item.gc(store, true);
        item = item.right;
      }
      this.type._start = null;
      this.type._map.forEach(/** @param {Item | null} item */ (item) => {
        while (item !== null) {
          item.gc(store, true);
          item = item.left;
        }
      });
      this.type._map = new Map();
    }

    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     * @param {number} offset
     */
    write (encoder, offset) {
      this.type._write(encoder);
    }

    /**
     * @return {number}
     */
    getRef () {
      return 7
    }
  }

  /**
   * @private
   *
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   * @return {ContentType}
   */
  const readContentType = decoder => new ContentType(typeRefs[decoder.readTypeRef()](decoder));

  /**
   * @todo This should return several items
   *
   * @param {StructStore} store
   * @param {ID} id
   * @return {{item:Item, diff:number}}
   */
  const followRedone = (store, id) => {
    /**
     * @type {ID|null}
     */
    let nextID = id;
    let diff = 0;
    let item;
    do {
      if (diff > 0) {
        nextID = createID(nextID.client, nextID.clock + diff);
      }
      item = getItem(store, nextID);
      diff = nextID.clock - item.id.clock;
      nextID = item.redone;
    } while (nextID !== null && item instanceof Item)
    return {
      item, diff
    }
  };

  /**
   * Make sure that neither item nor any of its parents is ever deleted.
   *
   * This property does not persist when storing it into a database or when
   * sending it to other peers
   *
   * @param {Item|null} item
   * @param {boolean} keep
   */
  const keepItem = (item, keep) => {
    while (item !== null && item.keep !== keep) {
      item.keep = keep;
      item = /** @type {AbstractType<any>} */ (item.parent)._item;
    }
  };

  /**
   * Split leftItem into two items
   * @param {Transaction} transaction
   * @param {Item} leftItem
   * @param {number} diff
   * @return {Item}
   *
   * @function
   * @private
   */
  const splitItem = (transaction, leftItem, diff) => {
    // create rightItem
    const { client, clock } = leftItem.id;
    const rightItem = new Item(
      createID(client, clock + diff),
      leftItem,
      createID(client, clock + diff - 1),
      leftItem.right,
      leftItem.rightOrigin,
      leftItem.parent,
      leftItem.parentSub,
      leftItem.content.splice(diff)
    );
    if (leftItem.deleted) {
      rightItem.markDeleted();
    }
    if (leftItem.keep) {
      rightItem.keep = true;
    }
    if (leftItem.redone !== null) {
      rightItem.redone = createID(leftItem.redone.client, leftItem.redone.clock + diff);
    }
    // update left (do not set leftItem.rightOrigin as it will lead to problems when syncing)
    leftItem.right = rightItem;
    // update right
    if (rightItem.right !== null) {
      rightItem.right.left = rightItem;
    }
    // right is more specific.
    transaction._mergeStructs.push(rightItem);
    // update parent._map
    if (rightItem.parentSub !== null && rightItem.right === null) {
      /** @type {AbstractType<any>} */ (rightItem.parent)._map.set(rightItem.parentSub, rightItem);
    }
    leftItem.length = diff;
    return rightItem
  };

  /**
   * Redoes the effect of this operation.
   *
   * @param {Transaction} transaction The Yjs instance.
   * @param {Item} item
   * @param {Set<Item>} redoitems
   * @param {DeleteSet} itemsToDelete
   * @param {boolean} ignoreRemoteMapChanges
   *
   * @return {Item|null}
   *
   * @private
   */
  const redoItem = (transaction, item, redoitems, itemsToDelete, ignoreRemoteMapChanges) => {
    const doc = transaction.doc;
    const store = doc.store;
    const ownClientID = doc.clientID;
    const redone = item.redone;
    if (redone !== null) {
      return getItemCleanStart(transaction, redone)
    }
    let parentItem = /** @type {AbstractType<any>} */ (item.parent)._item;
    /**
     * @type {Item|null}
     */
    let left = null;
    /**
     * @type {Item|null}
     */
    let right;
    // make sure that parent is redone
    if (parentItem !== null && parentItem.deleted === true) {
      // try to undo parent if it will be undone anyway
      if (parentItem.redone === null && (!redoitems.has(parentItem) || redoItem(transaction, parentItem, redoitems, itemsToDelete, ignoreRemoteMapChanges) === null)) {
        return null
      }
      while (parentItem.redone !== null) {
        parentItem = getItemCleanStart(transaction, parentItem.redone);
      }
    }
    const parentType = parentItem === null ? /** @type {AbstractType<any>} */ (item.parent) : /** @type {ContentType} */ (parentItem.content).type;

    if (item.parentSub === null) {
      // Is an array item. Insert at the old position
      left = item.left;
      right = item;
      // find next cloned_redo items
      while (left !== null) {
        /**
         * @type {Item|null}
         */
        let leftTrace = left;
        // trace redone until parent matches
        while (leftTrace !== null && /** @type {AbstractType<any>} */ (leftTrace.parent)._item !== parentItem) {
          leftTrace = leftTrace.redone === null ? null : getItemCleanStart(transaction, leftTrace.redone);
        }
        if (leftTrace !== null && /** @type {AbstractType<any>} */ (leftTrace.parent)._item === parentItem) {
          left = leftTrace;
          break
        }
        left = left.left;
      }
      while (right !== null) {
        /**
         * @type {Item|null}
         */
        let rightTrace = right;
        // trace redone until parent matches
        while (rightTrace !== null && /** @type {AbstractType<any>} */ (rightTrace.parent)._item !== parentItem) {
          rightTrace = rightTrace.redone === null ? null : getItemCleanStart(transaction, rightTrace.redone);
        }
        if (rightTrace !== null && /** @type {AbstractType<any>} */ (rightTrace.parent)._item === parentItem) {
          right = rightTrace;
          break
        }
        right = right.right;
      }
    } else {
      right = null;
      if (item.right && !ignoreRemoteMapChanges) {
        left = item;
        // Iterate right while right is in itemsToDelete
        // If it is intended to delete right while item is redone, we can expect that item should replace right.
        while (left !== null && left.right !== null && isDeleted(itemsToDelete, left.right.id)) {
          left = left.right;
        }
        // follow redone
        // trace redone until parent matches
        while (left !== null && left.redone !== null) {
          left = getItemCleanStart(transaction, left.redone);
        }
        if (left && left.right !== null) {
          // It is not possible to redo this item because it conflicts with a
          // change from another client
          return null
        }
      } else {
        left = parentType._map.get(item.parentSub) || null;
      }
    }
    const nextClock = getState(store, ownClientID);
    const nextId = createID(ownClientID, nextClock);
    const redoneItem = new Item(
      nextId,
      left, left && left.lastId,
      right, right && right.id,
      parentType,
      item.parentSub,
      item.content.copy()
    );
    item.redone = nextId;
    keepItem(redoneItem, true);
    redoneItem.integrate(transaction, 0);
    return redoneItem
  };

  /**
   * Abstract class that represents any content.
   */
  class Item extends AbstractStruct {
    /**
     * @param {ID} id
     * @param {Item | null} left
     * @param {ID | null} origin
     * @param {Item | null} right
     * @param {ID | null} rightOrigin
     * @param {AbstractType<any>|ID|null} parent Is a type if integrated, is null if it is possible to copy parent from left or right, is ID before integration to search for it.
     * @param {string | null} parentSub
     * @param {AbstractContent} content
     */
    constructor (id, left, origin, right, rightOrigin, parent, parentSub, content) {
      super(id, content.getLength());
      /**
       * The item that was originally to the left of this item.
       * @type {ID | null}
       */
      this.origin = origin;
      /**
       * The item that is currently to the left of this item.
       * @type {Item | null}
       */
      this.left = left;
      /**
       * The item that is currently to the right of this item.
       * @type {Item | null}
       */
      this.right = right;
      /**
       * The item that was originally to the right of this item.
       * @type {ID | null}
       */
      this.rightOrigin = rightOrigin;
      /**
       * @type {AbstractType<any>|ID|null}
       */
      this.parent = parent;
      /**
       * If the parent refers to this item with some kind of key (e.g. YMap, the
       * key is specified here. The key is then used to refer to the list in which
       * to insert this item. If `parentSub = null` type._start is the list in
       * which to insert to. Otherwise it is `parent._map`.
       * @type {String | null}
       */
      this.parentSub = parentSub;
      /**
       * If this type's effect is redone this type refers to the type that undid
       * this operation.
       * @type {ID | null}
       */
      this.redone = null;
      /**
       * @type {AbstractContent}
       */
      this.content = content;
      /**
       * bit1: keep
       * bit2: countable
       * bit3: deleted
       * bit4: mark - mark node as fast-search-marker
       * @type {number} byte
       */
      this.info = this.content.isCountable() ? BIT2 : 0;
    }

    /**
     * This is used to mark the item as an indexed fast-search marker
     *
     * @type {boolean}
     */
    set marker (isMarked) {
      if (((this.info & BIT4) > 0) !== isMarked) {
        this.info ^= BIT4;
      }
    }

    get marker () {
      return (this.info & BIT4) > 0
    }

    /**
     * If true, do not garbage collect this Item.
     */
    get keep () {
      return (this.info & BIT1) > 0
    }

    set keep (doKeep) {
      if (this.keep !== doKeep) {
        this.info ^= BIT1;
      }
    }

    get countable () {
      return (this.info & BIT2) > 0
    }

    /**
     * Whether this item was deleted or not.
     * @type {Boolean}
     */
    get deleted () {
      return (this.info & BIT3) > 0
    }

    set deleted (doDelete) {
      if (this.deleted !== doDelete) {
        this.info ^= BIT3;
      }
    }

    markDeleted () {
      this.info |= BIT3;
    }

    /**
     * Return the creator clientID of the missing op or define missing items and return null.
     *
     * @param {Transaction} transaction
     * @param {StructStore} store
     * @return {null | number}
     */
    getMissing (transaction, store) {
      if (this.origin && this.origin.client !== this.id.client && this.origin.clock >= getState(store, this.origin.client)) {
        return this.origin.client
      }
      if (this.rightOrigin && this.rightOrigin.client !== this.id.client && this.rightOrigin.clock >= getState(store, this.rightOrigin.client)) {
        return this.rightOrigin.client
      }
      if (this.parent && this.parent.constructor === ID && this.id.client !== this.parent.client && this.parent.clock >= getState(store, this.parent.client)) {
        return this.parent.client
      }

      // We have all missing ids, now find the items

      if (this.origin) {
        this.left = getItemCleanEnd(transaction, store, this.origin);
        this.origin = this.left.lastId;
      }
      if (this.rightOrigin) {
        this.right = getItemCleanStart(transaction, this.rightOrigin);
        this.rightOrigin = this.right.id;
      }
      if ((this.left && this.left.constructor === GC) || (this.right && this.right.constructor === GC)) {
        this.parent = null;
      }
      // only set parent if this shouldn't be garbage collected
      if (!this.parent) {
        if (this.left && this.left.constructor === Item) {
          this.parent = this.left.parent;
          this.parentSub = this.left.parentSub;
        }
        if (this.right && this.right.constructor === Item) {
          this.parent = this.right.parent;
          this.parentSub = this.right.parentSub;
        }
      } else if (this.parent.constructor === ID) {
        const parentItem = getItem(store, this.parent);
        if (parentItem.constructor === GC) {
          this.parent = null;
        } else {
          this.parent = /** @type {ContentType} */ (parentItem.content).type;
        }
      }
      return null
    }

    /**
     * @param {Transaction} transaction
     * @param {number} offset
     */
    integrate (transaction, offset) {
      if (offset > 0) {
        this.id.clock += offset;
        this.left = getItemCleanEnd(transaction, transaction.doc.store, createID(this.id.client, this.id.clock - 1));
        this.origin = this.left.lastId;
        this.content = this.content.splice(offset);
        this.length -= offset;
      }

      if (this.parent) {
        if ((!this.left && (!this.right || this.right.left !== null)) || (this.left && this.left.right !== this.right)) {
          /**
           * @type {Item|null}
           */
          let left = this.left;

          /**
           * @type {Item|null}
           */
          let o;
          // set o to the first conflicting item
          if (left !== null) {
            o = left.right;
          } else if (this.parentSub !== null) {
            o = /** @type {AbstractType<any>} */ (this.parent)._map.get(this.parentSub) || null;
            while (o !== null && o.left !== null) {
              o = o.left;
            }
          } else {
            o = /** @type {AbstractType<any>} */ (this.parent)._start;
          }
          // TODO: use something like DeleteSet here (a tree implementation would be best)
          // @todo use global set definitions
          /**
           * @type {Set<Item>}
           */
          const conflictingItems = new Set();
          /**
           * @type {Set<Item>}
           */
          const itemsBeforeOrigin = new Set();
          // Let c in conflictingItems, b in itemsBeforeOrigin
          // ***{origin}bbbb{this}{c,b}{c,b}{o}***
          // Note that conflictingItems is a subset of itemsBeforeOrigin
          while (o !== null && o !== this.right) {
            itemsBeforeOrigin.add(o);
            conflictingItems.add(o);
            if (compareIDs(this.origin, o.origin)) {
              // case 1
              if (o.id.client < this.id.client) {
                left = o;
                conflictingItems.clear();
              } else if (compareIDs(this.rightOrigin, o.rightOrigin)) {
                // this and o are conflicting and point to the same integration points. The id decides which item comes first.
                // Since this is to the left of o, we can break here
                break
              } // else, o might be integrated before an item that this conflicts with. If so, we will find it in the next iterations
            } else if (o.origin !== null && itemsBeforeOrigin.has(getItem(transaction.doc.store, o.origin))) { // use getItem instead of getItemCleanEnd because we don't want / need to split items.
              // case 2
              if (!conflictingItems.has(getItem(transaction.doc.store, o.origin))) {
                left = o;
                conflictingItems.clear();
              }
            } else {
              break
            }
            o = o.right;
          }
          this.left = left;
        }
        // reconnect left/right + update parent map/start if necessary
        if (this.left !== null) {
          const right = this.left.right;
          this.right = right;
          this.left.right = this;
        } else {
          let r;
          if (this.parentSub !== null) {
            r = /** @type {AbstractType<any>} */ (this.parent)._map.get(this.parentSub) || null;
            while (r !== null && r.left !== null) {
              r = r.left;
            }
          } else {
            r = /** @type {AbstractType<any>} */ (this.parent)._start
            ;/** @type {AbstractType<any>} */ (this.parent)._start = this;
          }
          this.right = r;
        }
        if (this.right !== null) {
          this.right.left = this;
        } else if (this.parentSub !== null) {
          // set as current parent value if right === null and this is parentSub
          /** @type {AbstractType<any>} */ (this.parent)._map.set(this.parentSub, this);
          if (this.left !== null) {
            // this is the current attribute value of parent. delete right
            this.left.delete(transaction);
          }
        }
        // adjust length of parent
        if (this.parentSub === null && this.countable && !this.deleted) {
          /** @type {AbstractType<any>} */ (this.parent)._length += this.length;
        }
        addStruct(transaction.doc.store, this);
        this.content.integrate(transaction, this);
        // add parent to transaction.changed
        addChangedTypeToTransaction(transaction, /** @type {AbstractType<any>} */ (this.parent), this.parentSub);
        if ((/** @type {AbstractType<any>} */ (this.parent)._item !== null && /** @type {AbstractType<any>} */ (this.parent)._item.deleted) || (this.parentSub !== null && this.right !== null)) {
          // delete if parent is deleted or if this is not the current attribute value of parent
          this.delete(transaction);
        }
      } else {
        // parent is not defined. Integrate GC struct instead
        new GC(this.id, this.length).integrate(transaction, 0);
      }
    }

    /**
     * Returns the next non-deleted item
     */
    get next () {
      let n = this.right;
      while (n !== null && n.deleted) {
        n = n.right;
      }
      return n
    }

    /**
     * Returns the previous non-deleted item
     */
    get prev () {
      let n = this.left;
      while (n !== null && n.deleted) {
        n = n.left;
      }
      return n
    }

    /**
     * Computes the last content address of this Item.
     */
    get lastId () {
      // allocating ids is pretty costly because of the amount of ids created, so we try to reuse whenever possible
      return this.length === 1 ? this.id : createID(this.id.client, this.id.clock + this.length - 1)
    }

    /**
     * Try to merge two items
     *
     * @param {Item} right
     * @return {boolean}
     */
    mergeWith (right) {
      if (
        this.constructor === right.constructor &&
        compareIDs(right.origin, this.lastId) &&
        this.right === right &&
        compareIDs(this.rightOrigin, right.rightOrigin) &&
        this.id.client === right.id.client &&
        this.id.clock + this.length === right.id.clock &&
        this.deleted === right.deleted &&
        this.redone === null &&
        right.redone === null &&
        this.content.constructor === right.content.constructor &&
        this.content.mergeWith(right.content)
      ) {
        const searchMarker = /** @type {AbstractType<any>} */ (this.parent)._searchMarker;
        if (searchMarker) {
          searchMarker.forEach(marker => {
            if (marker.p === right) {
              // right is going to be "forgotten" so we need to update the marker
              marker.p = this;
              // adjust marker index
              if (!this.deleted && this.countable) {
                marker.index -= this.length;
              }
            }
          });
        }
        if (right.keep) {
          this.keep = true;
        }
        this.right = right.right;
        if (this.right !== null) {
          this.right.left = this;
        }
        this.length += right.length;
        return true
      }
      return false
    }

    /**
     * Mark this Item as deleted.
     *
     * @param {Transaction} transaction
     */
    delete (transaction) {
      if (!this.deleted) {
        const parent = /** @type {AbstractType<any>} */ (this.parent);
        // adjust the length of parent
        if (this.countable && this.parentSub === null) {
          parent._length -= this.length;
        }
        this.markDeleted();
        addToDeleteSet(transaction.deleteSet, this.id.client, this.id.clock, this.length);
        addChangedTypeToTransaction(transaction, parent, this.parentSub);
        this.content.delete(transaction);
      }
    }

    /**
     * @param {StructStore} store
     * @param {boolean} parentGCd
     */
    gc (store, parentGCd) {
      if (!this.deleted) {
        throw unexpectedCase()
      }
      this.content.gc(store);
      if (parentGCd) {
        replaceStruct(store, this, new GC(this.id, this.length));
      } else {
        this.content = new ContentDeleted(this.length);
      }
    }

    /**
     * Transform the properties of this type to binary and write it to an
     * BinaryEncoder.
     *
     * This is called when this Item is sent to a remote peer.
     *
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder The encoder to write data to.
     * @param {number} offset
     */
    write (encoder, offset) {
      const origin = offset > 0 ? createID(this.id.client, this.id.clock + offset - 1) : this.origin;
      const rightOrigin = this.rightOrigin;
      const parentSub = this.parentSub;
      const info = (this.content.getRef() & BITS5) |
        (origin === null ? 0 : BIT8) | // origin is defined
        (rightOrigin === null ? 0 : BIT7) | // right origin is defined
        (parentSub === null ? 0 : BIT6); // parentSub is non-null
      encoder.writeInfo(info);
      if (origin !== null) {
        encoder.writeLeftID(origin);
      }
      if (rightOrigin !== null) {
        encoder.writeRightID(rightOrigin);
      }
      if (origin === null && rightOrigin === null) {
        const parent = /** @type {AbstractType<any>} */ (this.parent);
        if (parent._item !== undefined) {
          const parentItem = parent._item;
          if (parentItem === null) {
            // parent type on y._map
            // find the correct key
            const ykey = findRootTypeKey(parent);
            encoder.writeParentInfo(true); // write parentYKey
            encoder.writeString(ykey);
          } else {
            encoder.writeParentInfo(false); // write parent id
            encoder.writeLeftID(parentItem.id);
          }
        } else if (parent.constructor === String) { // this edge case was added by differential updates
          encoder.writeParentInfo(true); // write parentYKey
          encoder.writeString(parent);
        } else if (parent.constructor === ID) {
          encoder.writeParentInfo(false); // write parent id
          encoder.writeLeftID(parent);
        } else {
          unexpectedCase();
        }
        if (parentSub !== null) {
          encoder.writeString(parentSub);
        }
      }
      this.content.write(encoder, offset);
    }
  }

  /**
   * @param {UpdateDecoderV1 | UpdateDecoderV2} decoder
   * @param {number} info
   */
  const readItemContent = (decoder, info) => contentRefs[info & BITS5](decoder);

  /**
   * A lookup map for reading Item content.
   *
   * @type {Array<function(UpdateDecoderV1 | UpdateDecoderV2):AbstractContent>}
   */
  const contentRefs = [
    () => { unexpectedCase(); }, // GC is not ItemContent
    readContentDeleted, // 1
    readContentJSON, // 2
    readContentBinary, // 3
    readContentString, // 4
    readContentEmbed, // 5
    readContentFormat, // 6
    readContentType, // 7
    readContentAny, // 8
    readContentDoc, // 9
    () => { unexpectedCase(); } // 10 - Skip is not ItemContent
  ];

  const structSkipRefNumber = 10;

  /**
   * @private
   */
  class Skip extends AbstractStruct {
    get deleted () {
      return true
    }

    delete () {}

    /**
     * @param {Skip} right
     * @return {boolean}
     */
    mergeWith (right) {
      if (this.constructor !== right.constructor) {
        return false
      }
      this.length += right.length;
      return true
    }

    /**
     * @param {Transaction} transaction
     * @param {number} offset
     */
    integrate (transaction, offset) {
      // skip structs cannot be integrated
      unexpectedCase();
    }

    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     * @param {number} offset
     */
    write (encoder, offset) {
      encoder.writeInfo(structSkipRefNumber);
      // write as VarUint because Skips can't make use of predictable length-encoding
      writeVarUint(encoder.restEncoder, this.length - offset);
    }

    /**
     * @param {Transaction} transaction
     * @param {StructStore} store
     * @return {null | number}
     */
    getMissing (transaction, store) {
      return null
    }
  }

  /** eslint-env browser */

  const glo = /** @type {any} */ (typeof globalThis !== 'undefined'
    ? globalThis
    : typeof window !== 'undefined'
      ? window
      // @ts-ignore
      : typeof global !== 'undefined' ? global : {});

  const importIdentifier = '__ $YJS$ __';

  if (glo[importIdentifier] === true) {
    /**
     * Dear reader of this message. Please take this seriously.
     *
     * If you see this message, make sure that you only import one version of Yjs. In many cases,
     * your package manager installs two versions of Yjs that are used by different packages within your project.
     * Another reason for this message is that some parts of your project use the commonjs version of Yjs
     * and others use the EcmaScript version of Yjs.
     *
     * This often leads to issues that are hard to debug. We often need to perform constructor checks,
     * e.g. `struct instanceof GC`. If you imported different versions of Yjs, it is impossible for us to
     * do the constructor checks anymore - which might break the CRDT algorithm.
     *
     * https://github.com/yjs/yjs/issues/438
     */
    console.error('Yjs was already imported. This breaks constructor checks and will lead to issues! - https://github.com/yjs/yjs/issues/438');
  }
  glo[importIdentifier] = true;

  /**
  The data structure for documents. @nonabstract
  */
  class Text {
      /**
      @internal
      */
      constructor() { }
      /**
      Get the line description around the given position.
      */
      lineAt(pos) {
          if (pos < 0 || pos > this.length)
              throw new RangeError(`Invalid position ${pos} in document of length ${this.length}`);
          return this.lineInner(pos, false, 1, 0);
      }
      /**
      Get the description for the given (1-based) line number.
      */
      line(n) {
          if (n < 1 || n > this.lines)
              throw new RangeError(`Invalid line number ${n} in ${this.lines}-line document`);
          return this.lineInner(n, true, 1, 0);
      }
      /**
      Replace a range of the text with the given content.
      */
      replace(from, to, text) {
          let parts = [];
          this.decompose(0, from, parts, 2 /* To */);
          if (text.length)
              text.decompose(0, text.length, parts, 1 /* From */ | 2 /* To */);
          this.decompose(to, this.length, parts, 1 /* From */);
          return TextNode.from(parts, this.length - (to - from) + text.length);
      }
      /**
      Append another document to this one.
      */
      append(other) {
          return this.replace(this.length, this.length, other);
      }
      /**
      Retrieve the text between the given points.
      */
      slice(from, to = this.length) {
          let parts = [];
          this.decompose(from, to, parts, 0);
          return TextNode.from(parts, to - from);
      }
      /**
      Test whether this text is equal to another instance.
      */
      eq(other) {
          if (other == this)
              return true;
          if (other.length != this.length || other.lines != this.lines)
              return false;
          let start = this.scanIdentical(other, 1), end = this.length - this.scanIdentical(other, -1);
          let a = new RawTextCursor(this), b = new RawTextCursor(other);
          for (let skip = start, pos = start;;) {
              a.next(skip);
              b.next(skip);
              skip = 0;
              if (a.lineBreak != b.lineBreak || a.done != b.done || a.value != b.value)
                  return false;
              pos += a.value.length;
              if (a.done || pos >= end)
                  return true;
          }
      }
      /**
      Iterate over the text. When `dir` is `-1`, iteration happens
      from end to start. This will return lines and the breaks between
      them as separate strings.
      */
      iter(dir = 1) { return new RawTextCursor(this, dir); }
      /**
      Iterate over a range of the text. When `from` > `to`, the
      iterator will run in reverse.
      */
      iterRange(from, to = this.length) { return new PartialTextCursor(this, from, to); }
      /**
      Return a cursor that iterates over the given range of lines,
      _without_ returning the line breaks between, and yielding empty
      strings for empty lines.
      
      When `from` and `to` are given, they should be 1-based line numbers.
      */
      iterLines(from, to) {
          let inner;
          if (from == null) {
              inner = this.iter();
          }
          else {
              if (to == null)
                  to = this.lines + 1;
              let start = this.line(from).from;
              inner = this.iterRange(start, Math.max(start, to == this.lines + 1 ? this.length : to <= 1 ? 0 : this.line(to - 1).to));
          }
          return new LineCursor(inner);
      }
      /**
      @internal
      */
      toString() { return this.sliceString(0); }
      /**
      Convert the document to an array of lines (which can be
      deserialized again via [`Text.of`](https://codemirror.net/6/docs/ref/#state.Text^of)).
      */
      toJSON() {
          let lines = [];
          this.flatten(lines);
          return lines;
      }
      /**
      Create a `Text` instance for the given array of lines.
      */
      static of(text) {
          if (text.length == 0)
              throw new RangeError("A document must have at least one line");
          if (text.length == 1 && !text[0])
              return Text.empty;
          return text.length <= 32 /* Branch */ ? new TextLeaf(text) : TextNode.from(TextLeaf.split(text, []));
      }
  }
  // Leaves store an array of line strings. There are always line breaks
  // between these strings. Leaves are limited in size and have to be
  // contained in TextNode instances for bigger documents.
  class TextLeaf extends Text {
      constructor(text, length = textLength(text)) {
          super();
          this.text = text;
          this.length = length;
      }
      get lines() { return this.text.length; }
      get children() { return null; }
      lineInner(target, isLine, line, offset) {
          for (let i = 0;; i++) {
              let string = this.text[i], end = offset + string.length;
              if ((isLine ? line : end) >= target)
                  return new Line(offset, end, line, string);
              offset = end + 1;
              line++;
          }
      }
      decompose(from, to, target, open) {
          let text = from <= 0 && to >= this.length ? this
              : new TextLeaf(sliceText(this.text, from, to), Math.min(to, this.length) - Math.max(0, from));
          if (open & 1 /* From */) {
              let prev = target.pop();
              let joined = appendText(text.text, prev.text.slice(), 0, text.length);
              if (joined.length <= 32 /* Branch */) {
                  target.push(new TextLeaf(joined, prev.length + text.length));
              }
              else {
                  let mid = joined.length >> 1;
                  target.push(new TextLeaf(joined.slice(0, mid)), new TextLeaf(joined.slice(mid)));
              }
          }
          else {
              target.push(text);
          }
      }
      replace(from, to, text) {
          if (!(text instanceof TextLeaf))
              return super.replace(from, to, text);
          let lines = appendText(this.text, appendText(text.text, sliceText(this.text, 0, from)), to);
          let newLen = this.length + text.length - (to - from);
          if (lines.length <= 32 /* Branch */)
              return new TextLeaf(lines, newLen);
          return TextNode.from(TextLeaf.split(lines, []), newLen);
      }
      sliceString(from, to = this.length, lineSep = "\n") {
          let result = "";
          for (let pos = 0, i = 0; pos <= to && i < this.text.length; i++) {
              let line = this.text[i], end = pos + line.length;
              if (pos > from && i)
                  result += lineSep;
              if (from < end && to > pos)
                  result += line.slice(Math.max(0, from - pos), to - pos);
              pos = end + 1;
          }
          return result;
      }
      flatten(target) {
          for (let line of this.text)
              target.push(line);
      }
      scanIdentical() { return 0; }
      static split(text, target) {
          let part = [], len = -1;
          for (let line of text) {
              part.push(line);
              len += line.length + 1;
              if (part.length == 32 /* Branch */) {
                  target.push(new TextLeaf(part, len));
                  part = [];
                  len = -1;
              }
          }
          if (len > -1)
              target.push(new TextLeaf(part, len));
          return target;
      }
  }
  // Nodes provide the tree structure of the `Text` type. They store a
  // number of other nodes or leaves, taking care to balance themselves
  // on changes. There are implied line breaks _between_ the children of
  // a node (but not before the first or after the last child).
  class TextNode extends Text {
      constructor(children, length) {
          super();
          this.children = children;
          this.length = length;
          this.lines = 0;
          for (let child of children)
              this.lines += child.lines;
      }
      lineInner(target, isLine, line, offset) {
          for (let i = 0;; i++) {
              let child = this.children[i], end = offset + child.length, endLine = line + child.lines - 1;
              if ((isLine ? endLine : end) >= target)
                  return child.lineInner(target, isLine, line, offset);
              offset = end + 1;
              line = endLine + 1;
          }
      }
      decompose(from, to, target, open) {
          for (let i = 0, pos = 0; pos <= to && i < this.children.length; i++) {
              let child = this.children[i], end = pos + child.length;
              if (from <= end && to >= pos) {
                  let childOpen = open & ((pos <= from ? 1 /* From */ : 0) | (end >= to ? 2 /* To */ : 0));
                  if (pos >= from && end <= to && !childOpen)
                      target.push(child);
                  else
                      child.decompose(from - pos, to - pos, target, childOpen);
              }
              pos = end + 1;
          }
      }
      replace(from, to, text) {
          if (text.lines < this.lines)
              for (let i = 0, pos = 0; i < this.children.length; i++) {
                  let child = this.children[i], end = pos + child.length;
                  // Fast path: if the change only affects one child and the
                  // child's size remains in the acceptable range, only update
                  // that child
                  if (from >= pos && to <= end) {
                      let updated = child.replace(from - pos, to - pos, text);
                      let totalLines = this.lines - child.lines + updated.lines;
                      if (updated.lines < (totalLines >> (5 /* BranchShift */ - 1)) &&
                          updated.lines > (totalLines >> (5 /* BranchShift */ + 1))) {
                          let copy = this.children.slice();
                          copy[i] = updated;
                          return new TextNode(copy, this.length - (to - from) + text.length);
                      }
                      return super.replace(pos, end, updated);
                  }
                  pos = end + 1;
              }
          return super.replace(from, to, text);
      }
      sliceString(from, to = this.length, lineSep = "\n") {
          let result = "";
          for (let i = 0, pos = 0; i < this.children.length && pos <= to; i++) {
              let child = this.children[i], end = pos + child.length;
              if (pos > from && i)
                  result += lineSep;
              if (from < end && to > pos)
                  result += child.sliceString(from - pos, to - pos, lineSep);
              pos = end + 1;
          }
          return result;
      }
      flatten(target) {
          for (let child of this.children)
              child.flatten(target);
      }
      scanIdentical(other, dir) {
          if (!(other instanceof TextNode))
              return 0;
          let length = 0;
          let [iA, iB, eA, eB] = dir > 0 ? [0, 0, this.children.length, other.children.length]
              : [this.children.length - 1, other.children.length - 1, -1, -1];
          for (;; iA += dir, iB += dir) {
              if (iA == eA || iB == eB)
                  return length;
              let chA = this.children[iA], chB = other.children[iB];
              if (chA != chB)
                  return length + chA.scanIdentical(chB, dir);
              length += chA.length + 1;
          }
      }
      static from(children, length = children.reduce((l, ch) => l + ch.length + 1, -1)) {
          let lines = 0;
          for (let ch of children)
              lines += ch.lines;
          if (lines < 32 /* Branch */) {
              let flat = [];
              for (let ch of children)
                  ch.flatten(flat);
              return new TextLeaf(flat, length);
          }
          let chunk = Math.max(32 /* Branch */, lines >> 5 /* BranchShift */), maxChunk = chunk << 1, minChunk = chunk >> 1;
          let chunked = [], currentLines = 0, currentLen = -1, currentChunk = [];
          function add(child) {
              let last;
              if (child.lines > maxChunk && child instanceof TextNode) {
                  for (let node of child.children)
                      add(node);
              }
              else if (child.lines > minChunk && (currentLines > minChunk || !currentLines)) {
                  flush();
                  chunked.push(child);
              }
              else if (child instanceof TextLeaf && currentLines &&
                  (last = currentChunk[currentChunk.length - 1]) instanceof TextLeaf &&
                  child.lines + last.lines <= 32 /* Branch */) {
                  currentLines += child.lines;
                  currentLen += child.length + 1;
                  currentChunk[currentChunk.length - 1] = new TextLeaf(last.text.concat(child.text), last.length + 1 + child.length);
              }
              else {
                  if (currentLines + child.lines > chunk)
                      flush();
                  currentLines += child.lines;
                  currentLen += child.length + 1;
                  currentChunk.push(child);
              }
          }
          function flush() {
              if (currentLines == 0)
                  return;
              chunked.push(currentChunk.length == 1 ? currentChunk[0] : TextNode.from(currentChunk, currentLen));
              currentLen = -1;
              currentLines = currentChunk.length = 0;
          }
          for (let child of children)
              add(child);
          flush();
          return chunked.length == 1 ? chunked[0] : new TextNode(chunked, length);
      }
  }
  Text.empty = /*@__PURE__*/new TextLeaf([""], 0);
  function textLength(text) {
      let length = -1;
      for (let line of text)
          length += line.length + 1;
      return length;
  }
  function appendText(text, target, from = 0, to = 1e9) {
      for (let pos = 0, i = 0, first = true; i < text.length && pos <= to; i++) {
          let line = text[i], end = pos + line.length;
          if (end >= from) {
              if (end > to)
                  line = line.slice(0, to - pos);
              if (pos < from)
                  line = line.slice(from - pos);
              if (first) {
                  target[target.length - 1] += line;
                  first = false;
              }
              else
                  target.push(line);
          }
          pos = end + 1;
      }
      return target;
  }
  function sliceText(text, from, to) {
      return appendText(text, [""], from, to);
  }
  class RawTextCursor {
      constructor(text, dir = 1) {
          this.dir = dir;
          this.done = false;
          this.lineBreak = false;
          this.value = "";
          this.nodes = [text];
          this.offsets = [dir > 0 ? 1 : (text instanceof TextLeaf ? text.text.length : text.children.length) << 1];
      }
      nextInner(skip, dir) {
          this.done = this.lineBreak = false;
          for (;;) {
              let last = this.nodes.length - 1;
              let top = this.nodes[last], offsetValue = this.offsets[last], offset = offsetValue >> 1;
              let size = top instanceof TextLeaf ? top.text.length : top.children.length;
              if (offset == (dir > 0 ? size : 0)) {
                  if (last == 0) {
                      this.done = true;
                      this.value = "";
                      return this;
                  }
                  if (dir > 0)
                      this.offsets[last - 1]++;
                  this.nodes.pop();
                  this.offsets.pop();
              }
              else if ((offsetValue & 1) == (dir > 0 ? 0 : 1)) {
                  this.offsets[last] += dir;
                  if (skip == 0) {
                      this.lineBreak = true;
                      this.value = "\n";
                      return this;
                  }
                  skip--;
              }
              else if (top instanceof TextLeaf) {
                  // Move to the next string
                  let next = top.text[offset + (dir < 0 ? -1 : 0)];
                  this.offsets[last] += dir;
                  if (next.length > Math.max(0, skip)) {
                      this.value = skip == 0 ? next : dir > 0 ? next.slice(skip) : next.slice(0, next.length - skip);
                      return this;
                  }
                  skip -= next.length;
              }
              else {
                  let next = top.children[offset + (dir < 0 ? -1 : 0)];
                  if (skip > next.length) {
                      skip -= next.length;
                      this.offsets[last] += dir;
                  }
                  else {
                      if (dir < 0)
                          this.offsets[last]--;
                      this.nodes.push(next);
                      this.offsets.push(dir > 0 ? 1 : (next instanceof TextLeaf ? next.text.length : next.children.length) << 1);
                  }
              }
          }
      }
      next(skip = 0) {
          if (skip < 0) {
              this.nextInner(-skip, (-this.dir));
              skip = this.value.length;
          }
          return this.nextInner(skip, this.dir);
      }
  }
  class PartialTextCursor {
      constructor(text, start, end) {
          this.value = "";
          this.done = false;
          this.cursor = new RawTextCursor(text, start > end ? -1 : 1);
          this.pos = start > end ? text.length : 0;
          this.from = Math.min(start, end);
          this.to = Math.max(start, end);
      }
      nextInner(skip, dir) {
          if (dir < 0 ? this.pos <= this.from : this.pos >= this.to) {
              this.value = "";
              this.done = true;
              return this;
          }
          skip += Math.max(0, dir < 0 ? this.pos - this.to : this.from - this.pos);
          let limit = dir < 0 ? this.pos - this.from : this.to - this.pos;
          if (skip > limit)
              skip = limit;
          limit -= skip;
          let { value } = this.cursor.next(skip);
          this.pos += (value.length + skip) * dir;
          this.value = value.length <= limit ? value : dir < 0 ? value.slice(value.length - limit) : value.slice(0, limit);
          this.done = !this.value;
          return this;
      }
      next(skip = 0) {
          if (skip < 0)
              skip = Math.max(skip, this.from - this.pos);
          else if (skip > 0)
              skip = Math.min(skip, this.to - this.pos);
          return this.nextInner(skip, this.cursor.dir);
      }
      get lineBreak() { return this.cursor.lineBreak && this.value != ""; }
  }
  class LineCursor {
      constructor(inner) {
          this.inner = inner;
          this.afterBreak = true;
          this.value = "";
          this.done = false;
      }
      next(skip = 0) {
          let { done, lineBreak, value } = this.inner.next(skip);
          if (done) {
              this.done = true;
              this.value = "";
          }
          else if (lineBreak) {
              if (this.afterBreak) {
                  this.value = "";
              }
              else {
                  this.afterBreak = true;
                  this.next();
              }
          }
          else {
              this.value = value;
              this.afterBreak = false;
          }
          return this;
      }
      get lineBreak() { return false; }
  }
  if (typeof Symbol != "undefined") {
      Text.prototype[Symbol.iterator] = function () { return this.iter(); };
      RawTextCursor.prototype[Symbol.iterator] = PartialTextCursor.prototype[Symbol.iterator] =
          LineCursor.prototype[Symbol.iterator] = function () { return this; };
  }
  /**
  This type describes a line in the document. It is created
  on-demand when lines are [queried](https://codemirror.net/6/docs/ref/#state.Text.lineAt).
  */
  class Line {
      /**
      @internal
      */
      constructor(
      /**
      The position of the start of the line.
      */
      from, 
      /**
      The position at the end of the line (_before_ the line break,
      or at the end of document for the last line).
      */
      to, 
      /**
      This line's line number (1-based).
      */
      number, 
      /**
      The line's content.
      */
      text) {
          this.from = from;
          this.to = to;
          this.number = number;
          this.text = text;
      }
      /**
      The length of the line (not including any line break after it).
      */
      get length() { return this.to - this.from; }
  }

  // Compressed representation of the Grapheme_Cluster_Break=Extend
  // information from
  // http://www.unicode.org/Public/13.0.0/ucd/auxiliary/GraphemeBreakProperty.txt.
  // Each pair of elements represents a range, as an offet from the
  // previous range and a length. Numbers are in base-36, with the empty
  // string being a shorthand for 1.
  let extend = /*@__PURE__*/"lc,34,7n,7,7b,19,,,,2,,2,,,20,b,1c,l,g,,2t,7,2,6,2,2,,4,z,,u,r,2j,b,1m,9,9,,o,4,,9,,3,,5,17,3,3b,f,,w,1j,,,,4,8,4,,3,7,a,2,t,,1m,,,,2,4,8,,9,,a,2,q,,2,2,1l,,4,2,4,2,2,3,3,,u,2,3,,b,2,1l,,4,5,,2,4,,k,2,m,6,,,1m,,,2,,4,8,,7,3,a,2,u,,1n,,,,c,,9,,14,,3,,1l,3,5,3,,4,7,2,b,2,t,,1m,,2,,2,,3,,5,2,7,2,b,2,s,2,1l,2,,,2,4,8,,9,,a,2,t,,20,,4,,2,3,,,8,,29,,2,7,c,8,2q,,2,9,b,6,22,2,r,,,,,,1j,e,,5,,2,5,b,,10,9,,2u,4,,6,,2,2,2,p,2,4,3,g,4,d,,2,2,6,,f,,jj,3,qa,3,t,3,t,2,u,2,1s,2,,7,8,,2,b,9,,19,3,3b,2,y,,3a,3,4,2,9,,6,3,63,2,2,,1m,,,7,,,,,2,8,6,a,2,,1c,h,1r,4,1c,7,,,5,,14,9,c,2,w,4,2,2,,3,1k,,,2,3,,,3,1m,8,2,2,48,3,,d,,7,4,,6,,3,2,5i,1m,,5,ek,,5f,x,2da,3,3x,,2o,w,fe,6,2x,2,n9w,4,,a,w,2,28,2,7k,,3,,4,,p,2,5,,47,2,q,i,d,,12,8,p,b,1a,3,1c,,2,4,2,2,13,,1v,6,2,2,2,2,c,,8,,1b,,1f,,,3,2,2,5,2,,,16,2,8,,6m,,2,,4,,fn4,,kh,g,g,g,a6,2,gt,,6a,,45,5,1ae,3,,2,5,4,14,3,4,,4l,2,fx,4,ar,2,49,b,4w,,1i,f,1k,3,1d,4,2,2,1x,3,10,5,,8,1q,,c,2,1g,9,a,4,2,,2n,3,2,,,2,6,,4g,,3,8,l,2,1l,2,,,,,m,,e,7,3,5,5f,8,2,3,,,n,,29,,2,6,,,2,,,2,,2,6j,,2,4,6,2,,2,r,2,2d,8,2,,,2,2y,,,,2,6,,,2t,3,2,4,,5,77,9,,2,6t,,a,2,,,4,,40,4,2,2,4,,w,a,14,6,2,4,8,,9,6,2,3,1a,d,,2,ba,7,,6,,,2a,m,2,7,,2,,2,3e,6,3,,,2,,7,,,20,2,3,,,,9n,2,f0b,5,1n,7,t4,,1r,4,29,,f5k,2,43q,,,3,4,5,8,8,2,7,u,4,44,3,1iz,1j,4,1e,8,,e,,m,5,,f,11s,7,,h,2,7,,2,,5,79,7,c5,4,15s,7,31,7,240,5,gx7k,2o,3k,6o".split(",").map(s => s ? parseInt(s, 36) : 1);
  // Convert offsets into absolute values
  for (let i = 1; i < extend.length; i++)
      extend[i] += extend[i - 1];
  function isExtendingChar(code) {
      for (let i = 1; i < extend.length; i += 2)
          if (extend[i] > code)
              return extend[i - 1] <= code;
      return false;
  }
  function isRegionalIndicator(code) {
      return code >= 0x1F1E6 && code <= 0x1F1FF;
  }
  const ZWJ = 0x200d;
  /**
  Returns a next grapheme cluster break _after_ (not equal to)
  `pos`, if `forward` is true, or before otherwise. Returns `pos`
  itself if no further cluster break is available in the string.
  Moves across surrogate pairs, extending characters (when
  `includeExtending` is true), characters joined with zero-width
  joiners, and flag emoji.
  */
  function findClusterBreak(str, pos, forward = true, includeExtending = true) {
      return (forward ? nextClusterBreak : prevClusterBreak)(str, pos, includeExtending);
  }
  function nextClusterBreak(str, pos, includeExtending) {
      if (pos == str.length)
          return pos;
      // If pos is in the middle of a surrogate pair, move to its start
      if (pos && surrogateLow(str.charCodeAt(pos)) && surrogateHigh(str.charCodeAt(pos - 1)))
          pos--;
      let prev = codePointAt(str, pos);
      pos += codePointSize(prev);
      while (pos < str.length) {
          let next = codePointAt(str, pos);
          if (prev == ZWJ || next == ZWJ || includeExtending && isExtendingChar(next)) {
              pos += codePointSize(next);
              prev = next;
          }
          else if (isRegionalIndicator(next)) {
              let countBefore = 0, i = pos - 2;
              while (i >= 0 && isRegionalIndicator(codePointAt(str, i))) {
                  countBefore++;
                  i -= 2;
              }
              if (countBefore % 2 == 0)
                  break;
              else
                  pos += 2;
          }
          else {
              break;
          }
      }
      return pos;
  }
  function prevClusterBreak(str, pos, includeExtending) {
      while (pos > 0) {
          let found = nextClusterBreak(str, pos - 2, includeExtending);
          if (found < pos)
              return found;
          pos--;
      }
      return 0;
  }
  function surrogateLow(ch) { return ch >= 0xDC00 && ch < 0xE000; }
  function surrogateHigh(ch) { return ch >= 0xD800 && ch < 0xDC00; }
  /**
  Find the code point at the given position in a string (like the
  [`codePointAt`](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/codePointAt)
  string method).
  */
  function codePointAt(str, pos) {
      let code0 = str.charCodeAt(pos);
      if (!surrogateHigh(code0) || pos + 1 == str.length)
          return code0;
      let code1 = str.charCodeAt(pos + 1);
      if (!surrogateLow(code1))
          return code0;
      return ((code0 - 0xd800) << 10) + (code1 - 0xdc00) + 0x10000;
  }
  /**
  Given a Unicode codepoint, return the JavaScript string that
  respresents it (like
  [`String.fromCodePoint`](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/fromCodePoint)).
  */
  function fromCodePoint(code) {
      if (code <= 0xffff)
          return String.fromCharCode(code);
      code -= 0x10000;
      return String.fromCharCode((code >> 10) + 0xd800, (code & 1023) + 0xdc00);
  }
  /**
  The amount of positions a character takes up a JavaScript string.
  */
  function codePointSize(code) { return code < 0x10000 ? 1 : 2; }

  const DefaultSplit = /\r\n?|\n/;
  /**
  Distinguishes different ways in which positions can be mapped.
  */
  var MapMode = /*@__PURE__*/(function (MapMode) {
      /**
      Map a position to a valid new position, even when its context
      was deleted.
      */
      MapMode[MapMode["Simple"] = 0] = "Simple";
      /**
      Return null if deletion happens across the position.
      */
      MapMode[MapMode["TrackDel"] = 1] = "TrackDel";
      /**
      Return null if the character _before_ the position is deleted.
      */
      MapMode[MapMode["TrackBefore"] = 2] = "TrackBefore";
      /**
      Return null if the character _after_ the position is deleted.
      */
      MapMode[MapMode["TrackAfter"] = 3] = "TrackAfter";
  return MapMode})(MapMode || (MapMode = {}));
  /**
  A change description is a variant of [change set](https://codemirror.net/6/docs/ref/#state.ChangeSet)
  that doesn't store the inserted text. As such, it can't be
  applied, but is cheaper to store and manipulate.
  */
  class ChangeDesc {
      // Sections are encoded as pairs of integers. The first is the
      // length in the current document, and the second is -1 for
      // unaffected sections, and the length of the replacement content
      // otherwise. So an insertion would be (0, n>0), a deletion (n>0,
      // 0), and a replacement two positive numbers.
      /**
      @internal
      */
      constructor(
      /**
      @internal
      */
      sections) {
          this.sections = sections;
      }
      /**
      The length of the document before the change.
      */
      get length() {
          let result = 0;
          for (let i = 0; i < this.sections.length; i += 2)
              result += this.sections[i];
          return result;
      }
      /**
      The length of the document after the change.
      */
      get newLength() {
          let result = 0;
          for (let i = 0; i < this.sections.length; i += 2) {
              let ins = this.sections[i + 1];
              result += ins < 0 ? this.sections[i] : ins;
          }
          return result;
      }
      /**
      False when there are actual changes in this set.
      */
      get empty() { return this.sections.length == 0 || this.sections.length == 2 && this.sections[1] < 0; }
      /**
      Iterate over the unchanged parts left by these changes. `posA`
      provides the position of the range in the old document, `posB`
      the new position in the changed document.
      */
      iterGaps(f) {
          for (let i = 0, posA = 0, posB = 0; i < this.sections.length;) {
              let len = this.sections[i++], ins = this.sections[i++];
              if (ins < 0) {
                  f(posA, posB, len);
                  posB += len;
              }
              else {
                  posB += ins;
              }
              posA += len;
          }
      }
      /**
      Iterate over the ranges changed by these changes. (See
      [`ChangeSet.iterChanges`](https://codemirror.net/6/docs/ref/#state.ChangeSet.iterChanges) for a
      variant that also provides you with the inserted text.)
      `fromA`/`toA` provides the extent of the change in the starting
      document, `fromB`/`toB` the extent of the replacement in the
      changed document.
      
      When `individual` is true, adjacent changes (which are kept
      separate for [position mapping](https://codemirror.net/6/docs/ref/#state.ChangeDesc.mapPos)) are
      reported separately.
      */
      iterChangedRanges(f, individual = false) {
          iterChanges(this, f, individual);
      }
      /**
      Get a description of the inverted form of these changes.
      */
      get invertedDesc() {
          let sections = [];
          for (let i = 0; i < this.sections.length;) {
              let len = this.sections[i++], ins = this.sections[i++];
              if (ins < 0)
                  sections.push(len, ins);
              else
                  sections.push(ins, len);
          }
          return new ChangeDesc(sections);
      }
      /**
      Compute the combined effect of applying another set of changes
      after this one. The length of the document after this set should
      match the length before `other`.
      */
      composeDesc(other) { return this.empty ? other : other.empty ? this : composeSets(this, other); }
      /**
      Map this description, which should start with the same document
      as `other`, over another set of changes, so that it can be
      applied after it. When `before` is true, map as if the changes
      in `other` happened before the ones in `this`.
      */
      mapDesc(other, before = false) { return other.empty ? this : mapSet(this, other, before); }
      mapPos(pos, assoc = -1, mode = MapMode.Simple) {
          let posA = 0, posB = 0;
          for (let i = 0; i < this.sections.length;) {
              let len = this.sections[i++], ins = this.sections[i++], endA = posA + len;
              if (ins < 0) {
                  if (endA > pos)
                      return posB + (pos - posA);
                  posB += len;
              }
              else {
                  if (mode != MapMode.Simple && endA >= pos &&
                      (mode == MapMode.TrackDel && posA < pos && endA > pos ||
                          mode == MapMode.TrackBefore && posA < pos ||
                          mode == MapMode.TrackAfter && endA > pos))
                      return null;
                  if (endA > pos || endA == pos && assoc < 0 && !len)
                      return pos == posA || assoc < 0 ? posB : posB + ins;
                  posB += ins;
              }
              posA = endA;
          }
          if (pos > posA)
              throw new RangeError(`Position ${pos} is out of range for changeset of length ${posA}`);
          return posB;
      }
      /**
      Check whether these changes touch a given range. When one of the
      changes entirely covers the range, the string `"cover"` is
      returned.
      */
      touchesRange(from, to = from) {
          for (let i = 0, pos = 0; i < this.sections.length && pos <= to;) {
              let len = this.sections[i++], ins = this.sections[i++], end = pos + len;
              if (ins >= 0 && pos <= to && end >= from)
                  return pos < from && end > to ? "cover" : true;
              pos = end;
          }
          return false;
      }
      /**
      @internal
      */
      toString() {
          let result = "";
          for (let i = 0; i < this.sections.length;) {
              let len = this.sections[i++], ins = this.sections[i++];
              result += (result ? " " : "") + len + (ins >= 0 ? ":" + ins : "");
          }
          return result;
      }
      /**
      Serialize this change desc to a JSON-representable value.
      */
      toJSON() { return this.sections; }
      /**
      Create a change desc from its JSON representation (as produced
      by [`toJSON`](https://codemirror.net/6/docs/ref/#state.ChangeDesc.toJSON).
      */
      static fromJSON(json) {
          if (!Array.isArray(json) || json.length % 2 || json.some(a => typeof a != "number"))
              throw new RangeError("Invalid JSON representation of ChangeDesc");
          return new ChangeDesc(json);
      }
      /**
      @internal
      */
      static create(sections) { return new ChangeDesc(sections); }
  }
  /**
  A change set represents a group of modifications to a document. It
  stores the document length, and can only be applied to documents
  with exactly that length.
  */
  class ChangeSet extends ChangeDesc {
      constructor(sections, 
      /**
      @internal
      */
      inserted) {
          super(sections);
          this.inserted = inserted;
      }
      /**
      Apply the changes to a document, returning the modified
      document.
      */
      apply(doc) {
          if (this.length != doc.length)
              throw new RangeError("Applying change set to a document with the wrong length");
          iterChanges(this, (fromA, toA, fromB, _toB, text) => doc = doc.replace(fromB, fromB + (toA - fromA), text), false);
          return doc;
      }
      mapDesc(other, before = false) { return mapSet(this, other, before, true); }
      /**
      Given the document as it existed _before_ the changes, return a
      change set that represents the inverse of this set, which could
      be used to go from the document created by the changes back to
      the document as it existed before the changes.
      */
      invert(doc) {
          let sections = this.sections.slice(), inserted = [];
          for (let i = 0, pos = 0; i < sections.length; i += 2) {
              let len = sections[i], ins = sections[i + 1];
              if (ins >= 0) {
                  sections[i] = ins;
                  sections[i + 1] = len;
                  let index = i >> 1;
                  while (inserted.length < index)
                      inserted.push(Text.empty);
                  inserted.push(len ? doc.slice(pos, pos + len) : Text.empty);
              }
              pos += len;
          }
          return new ChangeSet(sections, inserted);
      }
      /**
      Combine two subsequent change sets into a single set. `other`
      must start in the document produced by `this`. If `this` goes
      `docA` → `docB` and `other` represents `docB` → `docC`, the
      returned value will represent the change `docA` → `docC`.
      */
      compose(other) { return this.empty ? other : other.empty ? this : composeSets(this, other, true); }
      /**
      Given another change set starting in the same document, maps this
      change set over the other, producing a new change set that can be
      applied to the document produced by applying `other`. When
      `before` is `true`, order changes as if `this` comes before
      `other`, otherwise (the default) treat `other` as coming first.
      
      Given two changes `A` and `B`, `A.compose(B.map(A))` and
      `B.compose(A.map(B, true))` will produce the same document. This
      provides a basic form of [operational
      transformation](https://en.wikipedia.org/wiki/Operational_transformation),
      and can be used for collaborative editing.
      */
      map(other, before = false) { return other.empty ? this : mapSet(this, other, before, true); }
      /**
      Iterate over the changed ranges in the document, calling `f` for
      each, with the range in the original document (`fromA`-`toA`)
      and the range that replaces it in the new document
      (`fromB`-`toB`).
      
      When `individual` is true, adjacent changes are reported
      separately.
      */
      iterChanges(f, individual = false) {
          iterChanges(this, f, individual);
      }
      /**
      Get a [change description](https://codemirror.net/6/docs/ref/#state.ChangeDesc) for this change
      set.
      */
      get desc() { return ChangeDesc.create(this.sections); }
      /**
      @internal
      */
      filter(ranges) {
          let resultSections = [], resultInserted = [], filteredSections = [];
          let iter = new SectionIter(this);
          done: for (let i = 0, pos = 0;;) {
              let next = i == ranges.length ? 1e9 : ranges[i++];
              while (pos < next || pos == next && iter.len == 0) {
                  if (iter.done)
                      break done;
                  let len = Math.min(iter.len, next - pos);
                  addSection(filteredSections, len, -1);
                  let ins = iter.ins == -1 ? -1 : iter.off == 0 ? iter.ins : 0;
                  addSection(resultSections, len, ins);
                  if (ins > 0)
                      addInsert(resultInserted, resultSections, iter.text);
                  iter.forward(len);
                  pos += len;
              }
              let end = ranges[i++];
              while (pos < end) {
                  if (iter.done)
                      break done;
                  let len = Math.min(iter.len, end - pos);
                  addSection(resultSections, len, -1);
                  addSection(filteredSections, len, iter.ins == -1 ? -1 : iter.off == 0 ? iter.ins : 0);
                  iter.forward(len);
                  pos += len;
              }
          }
          return { changes: new ChangeSet(resultSections, resultInserted),
              filtered: ChangeDesc.create(filteredSections) };
      }
      /**
      Serialize this change set to a JSON-representable value.
      */
      toJSON() {
          let parts = [];
          for (let i = 0; i < this.sections.length; i += 2) {
              let len = this.sections[i], ins = this.sections[i + 1];
              if (ins < 0)
                  parts.push(len);
              else if (ins == 0)
                  parts.push([len]);
              else
                  parts.push([len].concat(this.inserted[i >> 1].toJSON()));
          }
          return parts;
      }
      /**
      Create a change set for the given changes, for a document of the
      given length, using `lineSep` as line separator.
      */
      static of(changes, length, lineSep) {
          let sections = [], inserted = [], pos = 0;
          let total = null;
          function flush(force = false) {
              if (!force && !sections.length)
                  return;
              if (pos < length)
                  addSection(sections, length - pos, -1);
              let set = new ChangeSet(sections, inserted);
              total = total ? total.compose(set.map(total)) : set;
              sections = [];
              inserted = [];
              pos = 0;
          }
          function process(spec) {
              if (Array.isArray(spec)) {
                  for (let sub of spec)
                      process(sub);
              }
              else if (spec instanceof ChangeSet) {
                  if (spec.length != length)
                      throw new RangeError(`Mismatched change set length (got ${spec.length}, expected ${length})`);
                  flush();
                  total = total ? total.compose(spec.map(total)) : spec;
              }
              else {
                  let { from, to = from, insert } = spec;
                  if (from > to || from < 0 || to > length)
                      throw new RangeError(`Invalid change range ${from} to ${to} (in doc of length ${length})`);
                  let insText = !insert ? Text.empty : typeof insert == "string" ? Text.of(insert.split(lineSep || DefaultSplit)) : insert;
                  let insLen = insText.length;
                  if (from == to && insLen == 0)
                      return;
                  if (from < pos)
                      flush();
                  if (from > pos)
                      addSection(sections, from - pos, -1);
                  addSection(sections, to - from, insLen);
                  addInsert(inserted, sections, insText);
                  pos = to;
              }
          }
          process(changes);
          flush(!total);
          return total;
      }
      /**
      Create an empty changeset of the given length.
      */
      static empty(length) {
          return new ChangeSet(length ? [length, -1] : [], []);
      }
      /**
      Create a changeset from its JSON representation (as produced by
      [`toJSON`](https://codemirror.net/6/docs/ref/#state.ChangeSet.toJSON).
      */
      static fromJSON(json) {
          if (!Array.isArray(json))
              throw new RangeError("Invalid JSON representation of ChangeSet");
          let sections = [], inserted = [];
          for (let i = 0; i < json.length; i++) {
              let part = json[i];
              if (typeof part == "number") {
                  sections.push(part, -1);
              }
              else if (!Array.isArray(part) || typeof part[0] != "number" || part.some((e, i) => i && typeof e != "string")) {
                  throw new RangeError("Invalid JSON representation of ChangeSet");
              }
              else if (part.length == 1) {
                  sections.push(part[0], 0);
              }
              else {
                  while (inserted.length < i)
                      inserted.push(Text.empty);
                  inserted[i] = Text.of(part.slice(1));
                  sections.push(part[0], inserted[i].length);
              }
          }
          return new ChangeSet(sections, inserted);
      }
      /**
      @internal
      */
      static createSet(sections, inserted) {
          return new ChangeSet(sections, inserted);
      }
  }
  function addSection(sections, len, ins, forceJoin = false) {
      if (len == 0 && ins <= 0)
          return;
      let last = sections.length - 2;
      if (last >= 0 && ins <= 0 && ins == sections[last + 1])
          sections[last] += len;
      else if (len == 0 && sections[last] == 0)
          sections[last + 1] += ins;
      else if (forceJoin) {
          sections[last] += len;
          sections[last + 1] += ins;
      }
      else
          sections.push(len, ins);
  }
  function addInsert(values, sections, value) {
      if (value.length == 0)
          return;
      let index = (sections.length - 2) >> 1;
      if (index < values.length) {
          values[values.length - 1] = values[values.length - 1].append(value);
      }
      else {
          while (values.length < index)
              values.push(Text.empty);
          values.push(value);
      }
  }
  function iterChanges(desc, f, individual) {
      let inserted = desc.inserted;
      for (let posA = 0, posB = 0, i = 0; i < desc.sections.length;) {
          let len = desc.sections[i++], ins = desc.sections[i++];
          if (ins < 0) {
              posA += len;
              posB += len;
          }
          else {
              let endA = posA, endB = posB, text = Text.empty;
              for (;;) {
                  endA += len;
                  endB += ins;
                  if (ins && inserted)
                      text = text.append(inserted[(i - 2) >> 1]);
                  if (individual || i == desc.sections.length || desc.sections[i + 1] < 0)
                      break;
                  len = desc.sections[i++];
                  ins = desc.sections[i++];
              }
              f(posA, endA, posB, endB, text);
              posA = endA;
              posB = endB;
          }
      }
  }
  function mapSet(setA, setB, before, mkSet = false) {
      // Produce a copy of setA that applies to the document after setB
      // has been applied (assuming both start at the same document).
      let sections = [], insert = mkSet ? [] : null;
      let a = new SectionIter(setA), b = new SectionIter(setB);
      // Iterate over both sets in parallel. inserted tracks, for changes
      // in A that have to be processed piece-by-piece, whether their
      // content has been inserted already, and refers to the section
      // index.
      for (let inserted = -1;;) {
          if (a.ins == -1 && b.ins == -1) {
              // Move across ranges skipped by both sets.
              let len = Math.min(a.len, b.len);
              addSection(sections, len, -1);
              a.forward(len);
              b.forward(len);
          }
          else if (b.ins >= 0 && (a.ins < 0 || inserted == a.i || a.off == 0 && (b.len < a.len || b.len == a.len && !before))) {
              // If there's a change in B that comes before the next change in
              // A (ordered by start pos, then len, then before flag), skip
              // that (and process any changes in A it covers).
              let len = b.len;
              addSection(sections, b.ins, -1);
              while (len) {
                  let piece = Math.min(a.len, len);
                  if (a.ins >= 0 && inserted < a.i && a.len <= piece) {
                      addSection(sections, 0, a.ins);
                      if (insert)
                          addInsert(insert, sections, a.text);
                      inserted = a.i;
                  }
                  a.forward(piece);
                  len -= piece;
              }
              b.next();
          }
          else if (a.ins >= 0) {
              // Process the part of a change in A up to the start of the next
              // non-deletion change in B (if overlapping).
              let len = 0, left = a.len;
              while (left) {
                  if (b.ins == -1) {
                      let piece = Math.min(left, b.len);
                      len += piece;
                      left -= piece;
                      b.forward(piece);
                  }
                  else if (b.ins == 0 && b.len < left) {
                      left -= b.len;
                      b.next();
                  }
                  else {
                      break;
                  }
              }
              addSection(sections, len, inserted < a.i ? a.ins : 0);
              if (insert && inserted < a.i)
                  addInsert(insert, sections, a.text);
              inserted = a.i;
              a.forward(a.len - left);
          }
          else if (a.done && b.done) {
              return insert ? ChangeSet.createSet(sections, insert) : ChangeDesc.create(sections);
          }
          else {
              throw new Error("Mismatched change set lengths");
          }
      }
  }
  function composeSets(setA, setB, mkSet = false) {
      let sections = [];
      let insert = mkSet ? [] : null;
      let a = new SectionIter(setA), b = new SectionIter(setB);
      for (let open = false;;) {
          if (a.done && b.done) {
              return insert ? ChangeSet.createSet(sections, insert) : ChangeDesc.create(sections);
          }
          else if (a.ins == 0) { // Deletion in A
              addSection(sections, a.len, 0, open);
              a.next();
          }
          else if (b.len == 0 && !b.done) { // Insertion in B
              addSection(sections, 0, b.ins, open);
              if (insert)
                  addInsert(insert, sections, b.text);
              b.next();
          }
          else if (a.done || b.done) {
              throw new Error("Mismatched change set lengths");
          }
          else {
              let len = Math.min(a.len2, b.len), sectionLen = sections.length;
              if (a.ins == -1) {
                  let insB = b.ins == -1 ? -1 : b.off ? 0 : b.ins;
                  addSection(sections, len, insB, open);
                  if (insert && insB)
                      addInsert(insert, sections, b.text);
              }
              else if (b.ins == -1) {
                  addSection(sections, a.off ? 0 : a.len, len, open);
                  if (insert)
                      addInsert(insert, sections, a.textBit(len));
              }
              else {
                  addSection(sections, a.off ? 0 : a.len, b.off ? 0 : b.ins, open);
                  if (insert && !b.off)
                      addInsert(insert, sections, b.text);
              }
              open = (a.ins > len || b.ins >= 0 && b.len > len) && (open || sections.length > sectionLen);
              a.forward2(len);
              b.forward(len);
          }
      }
  }
  class SectionIter {
      constructor(set) {
          this.set = set;
          this.i = 0;
          this.next();
      }
      next() {
          let { sections } = this.set;
          if (this.i < sections.length) {
              this.len = sections[this.i++];
              this.ins = sections[this.i++];
          }
          else {
              this.len = 0;
              this.ins = -2;
          }
          this.off = 0;
      }
      get done() { return this.ins == -2; }
      get len2() { return this.ins < 0 ? this.len : this.ins; }
      get text() {
          let { inserted } = this.set, index = (this.i - 2) >> 1;
          return index >= inserted.length ? Text.empty : inserted[index];
      }
      textBit(len) {
          let { inserted } = this.set, index = (this.i - 2) >> 1;
          return index >= inserted.length && !len ? Text.empty
              : inserted[index].slice(this.off, len == null ? undefined : this.off + len);
      }
      forward(len) {
          if (len == this.len)
              this.next();
          else {
              this.len -= len;
              this.off += len;
          }
      }
      forward2(len) {
          if (this.ins == -1)
              this.forward(len);
          else if (len == this.ins)
              this.next();
          else {
              this.ins -= len;
              this.off += len;
          }
      }
  }

  /**
  A single selection range. When
  [`allowMultipleSelections`](https://codemirror.net/6/docs/ref/#state.EditorState^allowMultipleSelections)
  is enabled, a [selection](https://codemirror.net/6/docs/ref/#state.EditorSelection) may hold
  multiple ranges. By default, selections hold exactly one range.
  */
  class SelectionRange {
      constructor(
      /**
      The lower boundary of the range.
      */
      from, 
      /**
      The upper boundary of the range.
      */
      to, flags) {
          this.from = from;
          this.to = to;
          this.flags = flags;
      }
      /**
      The anchor of the range—the side that doesn't move when you
      extend it.
      */
      get anchor() { return this.flags & 16 /* Inverted */ ? this.to : this.from; }
      /**
      The head of the range, which is moved when the range is
      [extended](https://codemirror.net/6/docs/ref/#state.SelectionRange.extend).
      */
      get head() { return this.flags & 16 /* Inverted */ ? this.from : this.to; }
      /**
      True when `anchor` and `head` are at the same position.
      */
      get empty() { return this.from == this.to; }
      /**
      If this is a cursor that is explicitly associated with the
      character on one of its sides, this returns the side. -1 means
      the character before its position, 1 the character after, and 0
      means no association.
      */
      get assoc() { return this.flags & 4 /* AssocBefore */ ? -1 : this.flags & 8 /* AssocAfter */ ? 1 : 0; }
      /**
      The bidirectional text level associated with this cursor, if
      any.
      */
      get bidiLevel() {
          let level = this.flags & 3 /* BidiLevelMask */;
          return level == 3 ? null : level;
      }
      /**
      The goal column (stored vertical offset) associated with a
      cursor. This is used to preserve the vertical position when
      [moving](https://codemirror.net/6/docs/ref/#view.EditorView.moveVertically) across
      lines of different length.
      */
      get goalColumn() {
          let value = this.flags >> 5 /* GoalColumnOffset */;
          return value == 33554431 /* NoGoalColumn */ ? undefined : value;
      }
      /**
      Map this range through a change, producing a valid range in the
      updated document.
      */
      map(change, assoc = -1) {
          let from, to;
          if (this.empty) {
              from = to = change.mapPos(this.from, assoc);
          }
          else {
              from = change.mapPos(this.from, 1);
              to = change.mapPos(this.to, -1);
          }
          return from == this.from && to == this.to ? this : new SelectionRange(from, to, this.flags);
      }
      /**
      Extend this range to cover at least `from` to `to`.
      */
      extend(from, to = from) {
          if (from <= this.anchor && to >= this.anchor)
              return EditorSelection.range(from, to);
          let head = Math.abs(from - this.anchor) > Math.abs(to - this.anchor) ? from : to;
          return EditorSelection.range(this.anchor, head);
      }
      /**
      Compare this range to another range.
      */
      eq(other) {
          return this.anchor == other.anchor && this.head == other.head;
      }
      /**
      Return a JSON-serializable object representing the range.
      */
      toJSON() { return { anchor: this.anchor, head: this.head }; }
      /**
      Convert a JSON representation of a range to a `SelectionRange`
      instance.
      */
      static fromJSON(json) {
          if (!json || typeof json.anchor != "number" || typeof json.head != "number")
              throw new RangeError("Invalid JSON representation for SelectionRange");
          return EditorSelection.range(json.anchor, json.head);
      }
      /**
      @internal
      */
      static create(from, to, flags) {
          return new SelectionRange(from, to, flags);
      }
  }
  /**
  An editor selection holds one or more selection ranges.
  */
  class EditorSelection {
      constructor(
      /**
      The ranges in the selection, sorted by position. Ranges cannot
      overlap (but they may touch, if they aren't empty).
      */
      ranges, 
      /**
      The index of the _main_ range in the selection (which is
      usually the range that was added last).
      */
      mainIndex) {
          this.ranges = ranges;
          this.mainIndex = mainIndex;
      }
      /**
      Map a selection through a change. Used to adjust the selection
      position for changes.
      */
      map(change, assoc = -1) {
          if (change.empty)
              return this;
          return EditorSelection.create(this.ranges.map(r => r.map(change, assoc)), this.mainIndex);
      }
      /**
      Compare this selection to another selection.
      */
      eq(other) {
          if (this.ranges.length != other.ranges.length ||
              this.mainIndex != other.mainIndex)
              return false;
          for (let i = 0; i < this.ranges.length; i++)
              if (!this.ranges[i].eq(other.ranges[i]))
                  return false;
          return true;
      }
      /**
      Get the primary selection range. Usually, you should make sure
      your code applies to _all_ ranges, by using methods like
      [`changeByRange`](https://codemirror.net/6/docs/ref/#state.EditorState.changeByRange).
      */
      get main() { return this.ranges[this.mainIndex]; }
      /**
      Make sure the selection only has one range. Returns a selection
      holding only the main range from this selection.
      */
      asSingle() {
          return this.ranges.length == 1 ? this : new EditorSelection([this.main], 0);
      }
      /**
      Extend this selection with an extra range.
      */
      addRange(range, main = true) {
          return EditorSelection.create([range].concat(this.ranges), main ? 0 : this.mainIndex + 1);
      }
      /**
      Replace a given range with another range, and then normalize the
      selection to merge and sort ranges if necessary.
      */
      replaceRange(range, which = this.mainIndex) {
          let ranges = this.ranges.slice();
          ranges[which] = range;
          return EditorSelection.create(ranges, this.mainIndex);
      }
      /**
      Convert this selection to an object that can be serialized to
      JSON.
      */
      toJSON() {
          return { ranges: this.ranges.map(r => r.toJSON()), main: this.mainIndex };
      }
      /**
      Create a selection from a JSON representation.
      */
      static fromJSON(json) {
          if (!json || !Array.isArray(json.ranges) || typeof json.main != "number" || json.main >= json.ranges.length)
              throw new RangeError("Invalid JSON representation for EditorSelection");
          return new EditorSelection(json.ranges.map((r) => SelectionRange.fromJSON(r)), json.main);
      }
      /**
      Create a selection holding a single range.
      */
      static single(anchor, head = anchor) {
          return new EditorSelection([EditorSelection.range(anchor, head)], 0);
      }
      /**
      Sort and merge the given set of ranges, creating a valid
      selection.
      */
      static create(ranges, mainIndex = 0) {
          if (ranges.length == 0)
              throw new RangeError("A selection needs at least one range");
          for (let pos = 0, i = 0; i < ranges.length; i++) {
              let range = ranges[i];
              if (range.empty ? range.from <= pos : range.from < pos)
                  return EditorSelection.normalized(ranges.slice(), mainIndex);
              pos = range.to;
          }
          return new EditorSelection(ranges, mainIndex);
      }
      /**
      Create a cursor selection range at the given position. You can
      safely ignore the optional arguments in most situations.
      */
      static cursor(pos, assoc = 0, bidiLevel, goalColumn) {
          return SelectionRange.create(pos, pos, (assoc == 0 ? 0 : assoc < 0 ? 4 /* AssocBefore */ : 8 /* AssocAfter */) |
              (bidiLevel == null ? 3 : Math.min(2, bidiLevel)) |
              ((goalColumn !== null && goalColumn !== void 0 ? goalColumn : 33554431 /* NoGoalColumn */) << 5 /* GoalColumnOffset */));
      }
      /**
      Create a selection range.
      */
      static range(anchor, head, goalColumn) {
          let goal = (goalColumn !== null && goalColumn !== void 0 ? goalColumn : 33554431 /* NoGoalColumn */) << 5 /* GoalColumnOffset */;
          return head < anchor ? SelectionRange.create(head, anchor, 16 /* Inverted */ | goal | 8 /* AssocAfter */)
              : SelectionRange.create(anchor, head, goal | (head > anchor ? 4 /* AssocBefore */ : 0));
      }
      /**
      @internal
      */
      static normalized(ranges, mainIndex = 0) {
          let main = ranges[mainIndex];
          ranges.sort((a, b) => a.from - b.from);
          mainIndex = ranges.indexOf(main);
          for (let i = 1; i < ranges.length; i++) {
              let range = ranges[i], prev = ranges[i - 1];
              if (range.empty ? range.from <= prev.to : range.from < prev.to) {
                  let from = prev.from, to = Math.max(range.to, prev.to);
                  if (i <= mainIndex)
                      mainIndex--;
                  ranges.splice(--i, 2, range.anchor > range.head ? EditorSelection.range(to, from) : EditorSelection.range(from, to));
              }
          }
          return new EditorSelection(ranges, mainIndex);
      }
  }
  function checkSelection(selection, docLength) {
      for (let range of selection.ranges)
          if (range.to > docLength)
              throw new RangeError("Selection points outside of document");
  }

  let nextID = 0;
  /**
  A facet is a labeled value that is associated with an editor
  state. It takes inputs from any number of extensions, and combines
  those into a single output value.

  Examples of uses of facets are the [tab
  size](https://codemirror.net/6/docs/ref/#state.EditorState^tabSize), [editor
  attributes](https://codemirror.net/6/docs/ref/#view.EditorView^editorAttributes), and [update
  listeners](https://codemirror.net/6/docs/ref/#view.EditorView^updateListener).
  */
  class Facet {
      constructor(
      /**
      @internal
      */
      combine, 
      /**
      @internal
      */
      compareInput, 
      /**
      @internal
      */
      compare, isStatic, enables) {
          this.combine = combine;
          this.compareInput = compareInput;
          this.compare = compare;
          this.isStatic = isStatic;
          /**
          @internal
          */
          this.id = nextID++;
          this.default = combine([]);
          this.extensions = typeof enables == "function" ? enables(this) : enables;
      }
      /**
      Define a new facet.
      */
      static define(config = {}) {
          return new Facet(config.combine || ((a) => a), config.compareInput || ((a, b) => a === b), config.compare || (!config.combine ? sameArray$1 : (a, b) => a === b), !!config.static, config.enables);
      }
      /**
      Returns an extension that adds the given value to this facet.
      */
      of(value) {
          return new FacetProvider([], this, 0 /* Static */, value);
      }
      /**
      Create an extension that computes a value for the facet from a
      state. You must take care to declare the parts of the state that
      this value depends on, since your function is only called again
      for a new state when one of those parts changed.
      
      In cases where your value depends only on a single field, you'll
      want to use the [`from`](https://codemirror.net/6/docs/ref/#state.Facet.from) method instead.
      */
      compute(deps, get) {
          if (this.isStatic)
              throw new Error("Can't compute a static facet");
          return new FacetProvider(deps, this, 1 /* Single */, get);
      }
      /**
      Create an extension that computes zero or more values for this
      facet from a state.
      */
      computeN(deps, get) {
          if (this.isStatic)
              throw new Error("Can't compute a static facet");
          return new FacetProvider(deps, this, 2 /* Multi */, get);
      }
      from(field, get) {
          if (!get)
              get = x => x;
          return this.compute([field], state => get(state.field(field)));
      }
  }
  function sameArray$1(a, b) {
      return a == b || a.length == b.length && a.every((e, i) => e === b[i]);
  }
  class FacetProvider {
      constructor(dependencies, facet, type, value) {
          this.dependencies = dependencies;
          this.facet = facet;
          this.type = type;
          this.value = value;
          this.id = nextID++;
      }
      dynamicSlot(addresses) {
          var _a;
          let getter = this.value;
          let compare = this.facet.compareInput;
          let id = this.id, idx = addresses[id] >> 1, multi = this.type == 2 /* Multi */;
          let depDoc = false, depSel = false, depAddrs = [];
          for (let dep of this.dependencies) {
              if (dep == "doc")
                  depDoc = true;
              else if (dep == "selection")
                  depSel = true;
              else if ((((_a = addresses[dep.id]) !== null && _a !== void 0 ? _a : 1) & 1) == 0)
                  depAddrs.push(addresses[dep.id]);
          }
          return {
              create(state) {
                  state.values[idx] = getter(state);
                  return 1 /* Changed */;
              },
              update(state, tr) {
                  if ((depDoc && tr.docChanged) || (depSel && (tr.docChanged || tr.selection)) || ensureAll(state, depAddrs)) {
                      let newVal = getter(state);
                      if (multi ? !compareArray(newVal, state.values[idx], compare) : !compare(newVal, state.values[idx])) {
                          state.values[idx] = newVal;
                          return 1 /* Changed */;
                      }
                  }
                  return 0;
              },
              reconfigure: (state, oldState) => {
                  let newVal = getter(state);
                  let oldAddr = oldState.config.address[id];
                  if (oldAddr != null) {
                      let oldVal = getAddr(oldState, oldAddr);
                      if (this.dependencies.every(dep => {
                          return dep instanceof Facet ? oldState.facet(dep) === state.facet(dep) :
                              dep instanceof StateField ? oldState.field(dep, false) == state.field(dep, false) : true;
                      }) || (multi ? compareArray(newVal, oldVal, compare) : compare(newVal, oldVal))) {
                          state.values[idx] = oldVal;
                          return 0;
                      }
                  }
                  state.values[idx] = newVal;
                  return 1 /* Changed */;
              }
          };
      }
  }
  function compareArray(a, b, compare) {
      if (a.length != b.length)
          return false;
      for (let i = 0; i < a.length; i++)
          if (!compare(a[i], b[i]))
              return false;
      return true;
  }
  function ensureAll(state, addrs) {
      let changed = false;
      for (let addr of addrs)
          if (ensureAddr(state, addr) & 1 /* Changed */)
              changed = true;
      return changed;
  }
  function dynamicFacetSlot(addresses, facet, providers) {
      let providerAddrs = providers.map(p => addresses[p.id]);
      let providerTypes = providers.map(p => p.type);
      let dynamic = providerAddrs.filter(p => !(p & 1));
      let idx = addresses[facet.id] >> 1;
      function get(state) {
          let values = [];
          for (let i = 0; i < providerAddrs.length; i++) {
              let value = getAddr(state, providerAddrs[i]);
              if (providerTypes[i] == 2 /* Multi */)
                  for (let val of value)
                      values.push(val);
              else
                  values.push(value);
          }
          return facet.combine(values);
      }
      return {
          create(state) {
              for (let addr of providerAddrs)
                  ensureAddr(state, addr);
              state.values[idx] = get(state);
              return 1 /* Changed */;
          },
          update(state, tr) {
              if (!ensureAll(state, dynamic))
                  return 0;
              let value = get(state);
              if (facet.compare(value, state.values[idx]))
                  return 0;
              state.values[idx] = value;
              return 1 /* Changed */;
          },
          reconfigure(state, oldState) {
              let depChanged = ensureAll(state, providerAddrs);
              let oldProviders = oldState.config.facets[facet.id], oldValue = oldState.facet(facet);
              if (oldProviders && !depChanged && sameArray$1(providers, oldProviders)) {
                  state.values[idx] = oldValue;
                  return 0;
              }
              let value = get(state);
              if (facet.compare(value, oldValue)) {
                  state.values[idx] = oldValue;
                  return 0;
              }
              state.values[idx] = value;
              return 1 /* Changed */;
          }
      };
  }
  const initField = /*@__PURE__*/Facet.define({ static: true });
  /**
  Fields can store additional information in an editor state, and
  keep it in sync with the rest of the state.
  */
  class StateField {
      constructor(
      /**
      @internal
      */
      id, createF, updateF, compareF, 
      /**
      @internal
      */
      spec) {
          this.id = id;
          this.createF = createF;
          this.updateF = updateF;
          this.compareF = compareF;
          this.spec = spec;
          /**
          @internal
          */
          this.provides = undefined;
      }
      /**
      Define a state field.
      */
      static define(config) {
          let field = new StateField(nextID++, config.create, config.update, config.compare || ((a, b) => a === b), config);
          if (config.provide)
              field.provides = config.provide(field);
          return field;
      }
      create(state) {
          let init = state.facet(initField).find(i => i.field == this);
          return ((init === null || init === void 0 ? void 0 : init.create) || this.createF)(state);
      }
      /**
      @internal
      */
      slot(addresses) {
          let idx = addresses[this.id] >> 1;
          return {
              create: (state) => {
                  state.values[idx] = this.create(state);
                  return 1 /* Changed */;
              },
              update: (state, tr) => {
                  let oldVal = state.values[idx];
                  let value = this.updateF(oldVal, tr);
                  if (this.compareF(oldVal, value))
                      return 0;
                  state.values[idx] = value;
                  return 1 /* Changed */;
              },
              reconfigure: (state, oldState) => {
                  if (oldState.config.address[this.id] != null) {
                      state.values[idx] = oldState.field(this);
                      return 0;
                  }
                  state.values[idx] = this.create(state);
                  return 1 /* Changed */;
              }
          };
      }
      /**
      Returns an extension that enables this field and overrides the
      way it is initialized. Can be useful when you need to provide a
      non-default starting value for the field.
      */
      init(create) {
          return [this, initField.of({ field: this, create })];
      }
      /**
      State field instances can be used as
      [`Extension`](https://codemirror.net/6/docs/ref/#state.Extension) values to enable the field in a
      given state.
      */
      get extension() { return this; }
  }
  const Prec_ = { lowest: 4, low: 3, default: 2, high: 1, highest: 0 };
  function prec(value) {
      return (ext) => new PrecExtension(ext, value);
  }
  /**
  By default extensions are registered in the order they are found
  in the flattened form of nested array that was provided.
  Individual extension values can be assigned a precedence to
  override this. Extensions that do not have a precedence set get
  the precedence of the nearest parent with a precedence, or
  [`default`](https://codemirror.net/6/docs/ref/#state.Prec.default) if there is no such parent. The
  final ordering of extensions is determined by first sorting by
  precedence and then by order within each precedence.
  */
  const Prec = {
      /**
      The highest precedence level, for extensions that should end up
      near the start of the precedence ordering.
      */
      highest: /*@__PURE__*/prec(Prec_.highest),
      /**
      A higher-than-default precedence, for extensions that should
      come before those with default precedence.
      */
      high: /*@__PURE__*/prec(Prec_.high),
      /**
      The default precedence, which is also used for extensions
      without an explicit precedence.
      */
      default: /*@__PURE__*/prec(Prec_.default),
      /**
      A lower-than-default precedence.
      */
      low: /*@__PURE__*/prec(Prec_.low),
      /**
      The lowest precedence level. Meant for things that should end up
      near the end of the extension order.
      */
      lowest: /*@__PURE__*/prec(Prec_.lowest)
  };
  class PrecExtension {
      constructor(inner, prec) {
          this.inner = inner;
          this.prec = prec;
      }
  }
  /**
  Extension compartments can be used to make a configuration
  dynamic. By [wrapping](https://codemirror.net/6/docs/ref/#state.Compartment.of) part of your
  configuration in a compartment, you can later
  [replace](https://codemirror.net/6/docs/ref/#state.Compartment.reconfigure) that part through a
  transaction.
  */
  class Compartment {
      /**
      Create an instance of this compartment to add to your [state
      configuration](https://codemirror.net/6/docs/ref/#state.EditorStateConfig.extensions).
      */
      of(ext) { return new CompartmentInstance(this, ext); }
      /**
      Create an [effect](https://codemirror.net/6/docs/ref/#state.TransactionSpec.effects) that
      reconfigures this compartment.
      */
      reconfigure(content) {
          return Compartment.reconfigure.of({ compartment: this, extension: content });
      }
      /**
      Get the current content of the compartment in the state, or
      `undefined` if it isn't present.
      */
      get(state) {
          return state.config.compartments.get(this);
      }
  }
  class CompartmentInstance {
      constructor(compartment, inner) {
          this.compartment = compartment;
          this.inner = inner;
      }
  }
  class Configuration {
      constructor(base, compartments, dynamicSlots, address, staticValues, facets) {
          this.base = base;
          this.compartments = compartments;
          this.dynamicSlots = dynamicSlots;
          this.address = address;
          this.staticValues = staticValues;
          this.facets = facets;
          this.statusTemplate = [];
          while (this.statusTemplate.length < dynamicSlots.length)
              this.statusTemplate.push(0 /* Unresolved */);
      }
      staticFacet(facet) {
          let addr = this.address[facet.id];
          return addr == null ? facet.default : this.staticValues[addr >> 1];
      }
      static resolve(base, compartments, oldState) {
          let fields = [];
          let facets = Object.create(null);
          let newCompartments = new Map();
          for (let ext of flatten(base, compartments, newCompartments)) {
              if (ext instanceof StateField)
                  fields.push(ext);
              else
                  (facets[ext.facet.id] || (facets[ext.facet.id] = [])).push(ext);
          }
          let address = Object.create(null);
          let staticValues = [];
          let dynamicSlots = [];
          for (let field of fields) {
              address[field.id] = dynamicSlots.length << 1;
              dynamicSlots.push(a => field.slot(a));
          }
          let oldFacets = oldState === null || oldState === void 0 ? void 0 : oldState.config.facets;
          for (let id in facets) {
              let providers = facets[id], facet = providers[0].facet;
              let oldProviders = oldFacets && oldFacets[id] || [];
              if (providers.every(p => p.type == 0 /* Static */)) {
                  address[facet.id] = (staticValues.length << 1) | 1;
                  if (sameArray$1(oldProviders, providers)) {
                      staticValues.push(oldState.facet(facet));
                  }
                  else {
                      let value = facet.combine(providers.map(p => p.value));
                      staticValues.push(oldState && facet.compare(value, oldState.facet(facet)) ? oldState.facet(facet) : value);
                  }
              }
              else {
                  for (let p of providers) {
                      if (p.type == 0 /* Static */) {
                          address[p.id] = (staticValues.length << 1) | 1;
                          staticValues.push(p.value);
                      }
                      else {
                          address[p.id] = dynamicSlots.length << 1;
                          dynamicSlots.push(a => p.dynamicSlot(a));
                      }
                  }
                  address[facet.id] = dynamicSlots.length << 1;
                  dynamicSlots.push(a => dynamicFacetSlot(a, facet, providers));
              }
          }
          let dynamic = dynamicSlots.map(f => f(address));
          return new Configuration(base, newCompartments, dynamic, address, staticValues, facets);
      }
  }
  function flatten(extension, compartments, newCompartments) {
      let result = [[], [], [], [], []];
      let seen = new Map();
      function inner(ext, prec) {
          let known = seen.get(ext);
          if (known != null) {
              if (known <= prec)
                  return;
              let found = result[known].indexOf(ext);
              if (found > -1)
                  result[known].splice(found, 1);
              if (ext instanceof CompartmentInstance)
                  newCompartments.delete(ext.compartment);
          }
          seen.set(ext, prec);
          if (Array.isArray(ext)) {
              for (let e of ext)
                  inner(e, prec);
          }
          else if (ext instanceof CompartmentInstance) {
              if (newCompartments.has(ext.compartment))
                  throw new RangeError(`Duplicate use of compartment in extensions`);
              let content = compartments.get(ext.compartment) || ext.inner;
              newCompartments.set(ext.compartment, content);
              inner(content, prec);
          }
          else if (ext instanceof PrecExtension) {
              inner(ext.inner, ext.prec);
          }
          else if (ext instanceof StateField) {
              result[prec].push(ext);
              if (ext.provides)
                  inner(ext.provides, prec);
          }
          else if (ext instanceof FacetProvider) {
              result[prec].push(ext);
              if (ext.facet.extensions)
                  inner(ext.facet.extensions, prec);
          }
          else {
              let content = ext.extension;
              if (!content)
                  throw new Error(`Unrecognized extension value in extension set (${ext}). This sometimes happens because multiple instances of @codemirror/state are loaded, breaking instanceof checks.`);
              inner(content, prec);
          }
      }
      inner(extension, Prec_.default);
      return result.reduce((a, b) => a.concat(b));
  }
  function ensureAddr(state, addr) {
      if (addr & 1)
          return 2 /* Computed */;
      let idx = addr >> 1;
      let status = state.status[idx];
      if (status == 4 /* Computing */)
          throw new Error("Cyclic dependency between fields and/or facets");
      if (status & 2 /* Computed */)
          return status;
      state.status[idx] = 4 /* Computing */;
      let changed = state.computeSlot(state, state.config.dynamicSlots[idx]);
      return state.status[idx] = 2 /* Computed */ | changed;
  }
  function getAddr(state, addr) {
      return addr & 1 ? state.config.staticValues[addr >> 1] : state.values[addr >> 1];
  }

  const languageData = /*@__PURE__*/Facet.define();
  const allowMultipleSelections = /*@__PURE__*/Facet.define({
      combine: values => values.some(v => v),
      static: true
  });
  const lineSeparator = /*@__PURE__*/Facet.define({
      combine: values => values.length ? values[0] : undefined,
      static: true
  });
  const changeFilter = /*@__PURE__*/Facet.define();
  const transactionFilter = /*@__PURE__*/Facet.define();
  const transactionExtender = /*@__PURE__*/Facet.define();
  const readOnly = /*@__PURE__*/Facet.define({
      combine: values => values.length ? values[0] : false
  });

  /**
  Annotations are tagged values that are used to add metadata to
  transactions in an extensible way. They should be used to model
  things that effect the entire transaction (such as its [time
  stamp](https://codemirror.net/6/docs/ref/#state.Transaction^time) or information about its
  [origin](https://codemirror.net/6/docs/ref/#state.Transaction^userEvent)). For effects that happen
  _alongside_ the other changes made by the transaction, [state
  effects](https://codemirror.net/6/docs/ref/#state.StateEffect) are more appropriate.
  */
  class Annotation {
      /**
      @internal
      */
      constructor(
      /**
      The annotation type.
      */
      type, 
      /**
      The value of this annotation.
      */
      value) {
          this.type = type;
          this.value = value;
      }
      /**
      Define a new type of annotation.
      */
      static define() { return new AnnotationType(); }
  }
  /**
  Marker that identifies a type of [annotation](https://codemirror.net/6/docs/ref/#state.Annotation).
  */
  class AnnotationType {
      /**
      Create an instance of this annotation.
      */
      of(value) { return new Annotation(this, value); }
  }
  /**
  Representation of a type of state effect. Defined with
  [`StateEffect.define`](https://codemirror.net/6/docs/ref/#state.StateEffect^define).
  */
  class StateEffectType {
      /**
      @internal
      */
      constructor(
      // The `any` types in these function types are there to work
      // around TypeScript issue #37631, where the type guard on
      // `StateEffect.is` mysteriously stops working when these properly
      // have type `Value`.
      /**
      @internal
      */
      map) {
          this.map = map;
      }
      /**
      Create a [state effect](https://codemirror.net/6/docs/ref/#state.StateEffect) instance of this
      type.
      */
      of(value) { return new StateEffect(this, value); }
  }
  /**
  State effects can be used to represent additional effects
  associated with a [transaction](https://codemirror.net/6/docs/ref/#state.Transaction.effects). They
  are often useful to model changes to custom [state
  fields](https://codemirror.net/6/docs/ref/#state.StateField), when those changes aren't implicit in
  document or selection changes.
  */
  class StateEffect {
      /**
      @internal
      */
      constructor(
      /**
      @internal
      */
      type, 
      /**
      The value of this effect.
      */
      value) {
          this.type = type;
          this.value = value;
      }
      /**
      Map this effect through a position mapping. Will return
      `undefined` when that ends up deleting the effect.
      */
      map(mapping) {
          let mapped = this.type.map(this.value, mapping);
          return mapped === undefined ? undefined : mapped == this.value ? this : new StateEffect(this.type, mapped);
      }
      /**
      Tells you whether this effect object is of a given
      [type](https://codemirror.net/6/docs/ref/#state.StateEffectType).
      */
      is(type) { return this.type == type; }
      /**
      Define a new effect type. The type parameter indicates the type
      of values that his effect holds.
      */
      static define(spec = {}) {
          return new StateEffectType(spec.map || (v => v));
      }
      /**
      Map an array of effects through a change set.
      */
      static mapEffects(effects, mapping) {
          if (!effects.length)
              return effects;
          let result = [];
          for (let effect of effects) {
              let mapped = effect.map(mapping);
              if (mapped)
                  result.push(mapped);
          }
          return result;
      }
  }
  /**
  This effect can be used to reconfigure the root extensions of
  the editor. Doing this will discard any extensions
  [appended](https://codemirror.net/6/docs/ref/#state.StateEffect^appendConfig), but does not reset
  the content of [reconfigured](https://codemirror.net/6/docs/ref/#state.Compartment.reconfigure)
  compartments.
  */
  StateEffect.reconfigure = /*@__PURE__*/StateEffect.define();
  /**
  Append extensions to the top-level configuration of the editor.
  */
  StateEffect.appendConfig = /*@__PURE__*/StateEffect.define();
  /**
  Changes to the editor state are grouped into transactions.
  Typically, a user action creates a single transaction, which may
  contain any number of document changes, may change the selection,
  or have other effects. Create a transaction by calling
  [`EditorState.update`](https://codemirror.net/6/docs/ref/#state.EditorState.update), or immediately
  dispatch one by calling
  [`EditorView.dispatch`](https://codemirror.net/6/docs/ref/#view.EditorView.dispatch).
  */
  class Transaction {
      constructor(
      /**
      The state from which the transaction starts.
      */
      startState, 
      /**
      The document changes made by this transaction.
      */
      changes, 
      /**
      The selection set by this transaction, or undefined if it
      doesn't explicitly set a selection.
      */
      selection, 
      /**
      The effects added to the transaction.
      */
      effects, 
      /**
      @internal
      */
      annotations, 
      /**
      Whether the selection should be scrolled into view after this
      transaction is dispatched.
      */
      scrollIntoView) {
          this.startState = startState;
          this.changes = changes;
          this.selection = selection;
          this.effects = effects;
          this.annotations = annotations;
          this.scrollIntoView = scrollIntoView;
          /**
          @internal
          */
          this._doc = null;
          /**
          @internal
          */
          this._state = null;
          if (selection)
              checkSelection(selection, changes.newLength);
          if (!annotations.some((a) => a.type == Transaction.time))
              this.annotations = annotations.concat(Transaction.time.of(Date.now()));
      }
      /**
      @internal
      */
      static create(startState, changes, selection, effects, annotations, scrollIntoView) {
          return new Transaction(startState, changes, selection, effects, annotations, scrollIntoView);
      }
      /**
      The new document produced by the transaction. Contrary to
      [`.state`](https://codemirror.net/6/docs/ref/#state.Transaction.state)`.doc`, accessing this won't
      force the entire new state to be computed right away, so it is
      recommended that [transaction
      filters](https://codemirror.net/6/docs/ref/#state.EditorState^transactionFilter) use this getter
      when they need to look at the new document.
      */
      get newDoc() {
          return this._doc || (this._doc = this.changes.apply(this.startState.doc));
      }
      /**
      The new selection produced by the transaction. If
      [`this.selection`](https://codemirror.net/6/docs/ref/#state.Transaction.selection) is undefined,
      this will [map](https://codemirror.net/6/docs/ref/#state.EditorSelection.map) the start state's
      current selection through the changes made by the transaction.
      */
      get newSelection() {
          return this.selection || this.startState.selection.map(this.changes);
      }
      /**
      The new state created by the transaction. Computed on demand
      (but retained for subsequent access), so it is recommended not to
      access it in [transaction
      filters](https://codemirror.net/6/docs/ref/#state.EditorState^transactionFilter) when possible.
      */
      get state() {
          if (!this._state)
              this.startState.applyTransaction(this);
          return this._state;
      }
      /**
      Get the value of the given annotation type, if any.
      */
      annotation(type) {
          for (let ann of this.annotations)
              if (ann.type == type)
                  return ann.value;
          return undefined;
      }
      /**
      Indicates whether the transaction changed the document.
      */
      get docChanged() { return !this.changes.empty; }
      /**
      Indicates whether this transaction reconfigures the state
      (through a [configuration compartment](https://codemirror.net/6/docs/ref/#state.Compartment) or
      with a top-level configuration
      [effect](https://codemirror.net/6/docs/ref/#state.StateEffect^reconfigure).
      */
      get reconfigured() { return this.startState.config != this.state.config; }
      /**
      Returns true if the transaction has a [user
      event](https://codemirror.net/6/docs/ref/#state.Transaction^userEvent) annotation that is equal to
      or more specific than `event`. For example, if the transaction
      has `"select.pointer"` as user event, `"select"` and
      `"select.pointer"` will match it.
      */
      isUserEvent(event) {
          let e = this.annotation(Transaction.userEvent);
          return !!(e && (e == event || e.length > event.length && e.slice(0, event.length) == event && e[event.length] == "."));
      }
  }
  /**
  Annotation used to store transaction timestamps. Automatically
  added to every transaction, holding `Date.now()`.
  */
  Transaction.time = /*@__PURE__*/Annotation.define();
  /**
  Annotation used to associate a transaction with a user interface
  event. Holds a string identifying the event, using a
  dot-separated format to support attaching more specific
  information. The events used by the core libraries are:

   - `"input"` when content is entered
     - `"input.type"` for typed input
       - `"input.type.compose"` for composition
     - `"input.paste"` for pasted input
     - `"input.drop"` when adding content with drag-and-drop
     - `"input.complete"` when autocompleting
   - `"delete"` when the user deletes content
     - `"delete.selection"` when deleting the selection
     - `"delete.forward"` when deleting forward from the selection
     - `"delete.backward"` when deleting backward from the selection
     - `"delete.cut"` when cutting to the clipboard
   - `"move"` when content is moved
     - `"move.drop"` when content is moved within the editor through drag-and-drop
   - `"select"` when explicitly changing the selection
     - `"select.pointer"` when selecting with a mouse or other pointing device
   - `"undo"` and `"redo"` for history actions

  Use [`isUserEvent`](https://codemirror.net/6/docs/ref/#state.Transaction.isUserEvent) to check
  whether the annotation matches a given event.
  */
  Transaction.userEvent = /*@__PURE__*/Annotation.define();
  /**
  Annotation indicating whether a transaction should be added to
  the undo history or not.
  */
  Transaction.addToHistory = /*@__PURE__*/Annotation.define();
  /**
  Annotation indicating (when present and true) that a transaction
  represents a change made by some other actor, not the user. This
  is used, for example, to tag other people's changes in
  collaborative editing.
  */
  Transaction.remote = /*@__PURE__*/Annotation.define();
  function joinRanges(a, b) {
      let result = [];
      for (let iA = 0, iB = 0;;) {
          let from, to;
          if (iA < a.length && (iB == b.length || b[iB] >= a[iA])) {
              from = a[iA++];
              to = a[iA++];
          }
          else if (iB < b.length) {
              from = b[iB++];
              to = b[iB++];
          }
          else
              return result;
          if (!result.length || result[result.length - 1] < from)
              result.push(from, to);
          else if (result[result.length - 1] < to)
              result[result.length - 1] = to;
      }
  }
  function mergeTransaction(a, b, sequential) {
      var _a;
      let mapForA, mapForB, changes;
      if (sequential) {
          mapForA = b.changes;
          mapForB = ChangeSet.empty(b.changes.length);
          changes = a.changes.compose(b.changes);
      }
      else {
          mapForA = b.changes.map(a.changes);
          mapForB = a.changes.mapDesc(b.changes, true);
          changes = a.changes.compose(mapForA);
      }
      return {
          changes,
          selection: b.selection ? b.selection.map(mapForB) : (_a = a.selection) === null || _a === void 0 ? void 0 : _a.map(mapForA),
          effects: StateEffect.mapEffects(a.effects, mapForA).concat(StateEffect.mapEffects(b.effects, mapForB)),
          annotations: a.annotations.length ? a.annotations.concat(b.annotations) : b.annotations,
          scrollIntoView: a.scrollIntoView || b.scrollIntoView
      };
  }
  function resolveTransactionInner(state, spec, docSize) {
      let sel = spec.selection, annotations = asArray$1(spec.annotations);
      if (spec.userEvent)
          annotations = annotations.concat(Transaction.userEvent.of(spec.userEvent));
      return {
          changes: spec.changes instanceof ChangeSet ? spec.changes
              : ChangeSet.of(spec.changes || [], docSize, state.facet(lineSeparator)),
          selection: sel && (sel instanceof EditorSelection ? sel : EditorSelection.single(sel.anchor, sel.head)),
          effects: asArray$1(spec.effects),
          annotations,
          scrollIntoView: !!spec.scrollIntoView
      };
  }
  function resolveTransaction(state, specs, filter) {
      let s = resolveTransactionInner(state, specs.length ? specs[0] : {}, state.doc.length);
      if (specs.length && specs[0].filter === false)
          filter = false;
      for (let i = 1; i < specs.length; i++) {
          if (specs[i].filter === false)
              filter = false;
          let seq = !!specs[i].sequential;
          s = mergeTransaction(s, resolveTransactionInner(state, specs[i], seq ? s.changes.newLength : state.doc.length), seq);
      }
      let tr = Transaction.create(state, s.changes, s.selection, s.effects, s.annotations, s.scrollIntoView);
      return extendTransaction(filter ? filterTransaction(tr) : tr);
  }
  // Finish a transaction by applying filters if necessary.
  function filterTransaction(tr) {
      let state = tr.startState;
      // Change filters
      let result = true;
      for (let filter of state.facet(changeFilter)) {
          let value = filter(tr);
          if (value === false) {
              result = false;
              break;
          }
          if (Array.isArray(value))
              result = result === true ? value : joinRanges(result, value);
      }
      if (result !== true) {
          let changes, back;
          if (result === false) {
              back = tr.changes.invertedDesc;
              changes = ChangeSet.empty(state.doc.length);
          }
          else {
              let filtered = tr.changes.filter(result);
              changes = filtered.changes;
              back = filtered.filtered.mapDesc(filtered.changes).invertedDesc;
          }
          tr = Transaction.create(state, changes, tr.selection && tr.selection.map(back), StateEffect.mapEffects(tr.effects, back), tr.annotations, tr.scrollIntoView);
      }
      // Transaction filters
      let filters = state.facet(transactionFilter);
      for (let i = filters.length - 1; i >= 0; i--) {
          let filtered = filters[i](tr);
          if (filtered instanceof Transaction)
              tr = filtered;
          else if (Array.isArray(filtered) && filtered.length == 1 && filtered[0] instanceof Transaction)
              tr = filtered[0];
          else
              tr = resolveTransaction(state, asArray$1(filtered), false);
      }
      return tr;
  }
  function extendTransaction(tr) {
      let state = tr.startState, extenders = state.facet(transactionExtender), spec = tr;
      for (let i = extenders.length - 1; i >= 0; i--) {
          let extension = extenders[i](tr);
          if (extension && Object.keys(extension).length)
              spec = mergeTransaction(tr, resolveTransactionInner(state, extension, tr.changes.newLength), true);
      }
      return spec == tr ? tr : Transaction.create(state, tr.changes, tr.selection, spec.effects, spec.annotations, spec.scrollIntoView);
  }
  const none$2 = [];
  function asArray$1(value) {
      return value == null ? none$2 : Array.isArray(value) ? value : [value];
  }

  /**
  The categories produced by a [character
  categorizer](https://codemirror.net/6/docs/ref/#state.EditorState.charCategorizer). These are used
  do things like selecting by word.
  */
  var CharCategory = /*@__PURE__*/(function (CharCategory) {
      /**
      Word characters.
      */
      CharCategory[CharCategory["Word"] = 0] = "Word";
      /**
      Whitespace.
      */
      CharCategory[CharCategory["Space"] = 1] = "Space";
      /**
      Anything else.
      */
      CharCategory[CharCategory["Other"] = 2] = "Other";
  return CharCategory})(CharCategory || (CharCategory = {}));
  const nonASCIISingleCaseWordChar = /[\u00df\u0587\u0590-\u05f4\u0600-\u06ff\u3040-\u309f\u30a0-\u30ff\u3400-\u4db5\u4e00-\u9fcc\uac00-\ud7af]/;
  let wordChar;
  try {
      wordChar = /*@__PURE__*/new RegExp("[\\p{Alphabetic}\\p{Number}_]", "u");
  }
  catch (_) { }
  function hasWordChar(str) {
      if (wordChar)
          return wordChar.test(str);
      for (let i = 0; i < str.length; i++) {
          let ch = str[i];
          if (/\w/.test(ch) || ch > "\x80" && (ch.toUpperCase() != ch.toLowerCase() || nonASCIISingleCaseWordChar.test(ch)))
              return true;
      }
      return false;
  }
  function makeCategorizer(wordChars) {
      return (char) => {
          if (!/\S/.test(char))
              return CharCategory.Space;
          if (hasWordChar(char))
              return CharCategory.Word;
          for (let i = 0; i < wordChars.length; i++)
              if (char.indexOf(wordChars[i]) > -1)
                  return CharCategory.Word;
          return CharCategory.Other;
      };
  }

  /**
  The editor state class is a persistent (immutable) data structure.
  To update a state, you [create](https://codemirror.net/6/docs/ref/#state.EditorState.update) a
  [transaction](https://codemirror.net/6/docs/ref/#state.Transaction), which produces a _new_ state
  instance, without modifying the original object.

  As such, _never_ mutate properties of a state directly. That'll
  just break things.
  */
  class EditorState {
      constructor(
      /**
      @internal
      */
      config, 
      /**
      The current document.
      */
      doc, 
      /**
      The current selection.
      */
      selection, 
      /**
      @internal
      */
      values, computeSlot, tr) {
          this.config = config;
          this.doc = doc;
          this.selection = selection;
          this.values = values;
          this.status = config.statusTemplate.slice();
          this.computeSlot = computeSlot;
          // Fill in the computed state immediately, so that further queries
          // for it made during the update return this state
          if (tr)
              tr._state = this;
          for (let i = 0; i < this.config.dynamicSlots.length; i++)
              ensureAddr(this, i << 1);
          this.computeSlot = null;
      }
      field(field, require = true) {
          let addr = this.config.address[field.id];
          if (addr == null) {
              if (require)
                  throw new RangeError("Field is not present in this state");
              return undefined;
          }
          ensureAddr(this, addr);
          return getAddr(this, addr);
      }
      /**
      Create a [transaction](https://codemirror.net/6/docs/ref/#state.Transaction) that updates this
      state. Any number of [transaction specs](https://codemirror.net/6/docs/ref/#state.TransactionSpec)
      can be passed. Unless
      [`sequential`](https://codemirror.net/6/docs/ref/#state.TransactionSpec.sequential) is set, the
      [changes](https://codemirror.net/6/docs/ref/#state.TransactionSpec.changes) (if any) of each spec
      are assumed to start in the _current_ document (not the document
      produced by previous specs), and its
      [selection](https://codemirror.net/6/docs/ref/#state.TransactionSpec.selection) and
      [effects](https://codemirror.net/6/docs/ref/#state.TransactionSpec.effects) are assumed to refer
      to the document created by its _own_ changes. The resulting
      transaction contains the combined effect of all the different
      specs. For [selection](https://codemirror.net/6/docs/ref/#state.TransactionSpec.selection), later
      specs take precedence over earlier ones.
      */
      update(...specs) {
          return resolveTransaction(this, specs, true);
      }
      /**
      @internal
      */
      applyTransaction(tr) {
          let conf = this.config, { base, compartments } = conf;
          for (let effect of tr.effects) {
              if (effect.is(Compartment.reconfigure)) {
                  if (conf) {
                      compartments = new Map;
                      conf.compartments.forEach((val, key) => compartments.set(key, val));
                      conf = null;
                  }
                  compartments.set(effect.value.compartment, effect.value.extension);
              }
              else if (effect.is(StateEffect.reconfigure)) {
                  conf = null;
                  base = effect.value;
              }
              else if (effect.is(StateEffect.appendConfig)) {
                  conf = null;
                  base = asArray$1(base).concat(effect.value);
              }
          }
          let startValues;
          if (!conf) {
              conf = Configuration.resolve(base, compartments, this);
              let intermediateState = new EditorState(conf, this.doc, this.selection, conf.dynamicSlots.map(() => null), (state, slot) => slot.reconfigure(state, this), null);
              startValues = intermediateState.values;
          }
          else {
              startValues = tr.startState.values.slice();
          }
          new EditorState(conf, tr.newDoc, tr.newSelection, startValues, (state, slot) => slot.update(state, tr), tr);
      }
      /**
      Create a [transaction spec](https://codemirror.net/6/docs/ref/#state.TransactionSpec) that
      replaces every selection range with the given content.
      */
      replaceSelection(text) {
          if (typeof text == "string")
              text = this.toText(text);
          return this.changeByRange(range => ({ changes: { from: range.from, to: range.to, insert: text },
              range: EditorSelection.cursor(range.from + text.length) }));
      }
      /**
      Create a set of changes and a new selection by running the given
      function for each range in the active selection. The function
      can return an optional set of changes (in the coordinate space
      of the start document), plus an updated range (in the coordinate
      space of the document produced by the call's own changes). This
      method will merge all the changes and ranges into a single
      changeset and selection, and return it as a [transaction
      spec](https://codemirror.net/6/docs/ref/#state.TransactionSpec), which can be passed to
      [`update`](https://codemirror.net/6/docs/ref/#state.EditorState.update).
      */
      changeByRange(f) {
          let sel = this.selection;
          let result1 = f(sel.ranges[0]);
          let changes = this.changes(result1.changes), ranges = [result1.range];
          let effects = asArray$1(result1.effects);
          for (let i = 1; i < sel.ranges.length; i++) {
              let result = f(sel.ranges[i]);
              let newChanges = this.changes(result.changes), newMapped = newChanges.map(changes);
              for (let j = 0; j < i; j++)
                  ranges[j] = ranges[j].map(newMapped);
              let mapBy = changes.mapDesc(newChanges, true);
              ranges.push(result.range.map(mapBy));
              changes = changes.compose(newMapped);
              effects = StateEffect.mapEffects(effects, newMapped).concat(StateEffect.mapEffects(asArray$1(result.effects), mapBy));
          }
          return {
              changes,
              selection: EditorSelection.create(ranges, sel.mainIndex),
              effects
          };
      }
      /**
      Create a [change set](https://codemirror.net/6/docs/ref/#state.ChangeSet) from the given change
      description, taking the state's document length and line
      separator into account.
      */
      changes(spec = []) {
          if (spec instanceof ChangeSet)
              return spec;
          return ChangeSet.of(spec, this.doc.length, this.facet(EditorState.lineSeparator));
      }
      /**
      Using the state's [line
      separator](https://codemirror.net/6/docs/ref/#state.EditorState^lineSeparator), create a
      [`Text`](https://codemirror.net/6/docs/ref/#state.Text) instance from the given string.
      */
      toText(string) {
          return Text.of(string.split(this.facet(EditorState.lineSeparator) || DefaultSplit));
      }
      /**
      Return the given range of the document as a string.
      */
      sliceDoc(from = 0, to = this.doc.length) {
          return this.doc.sliceString(from, to, this.lineBreak);
      }
      /**
      Get the value of a state [facet](https://codemirror.net/6/docs/ref/#state.Facet).
      */
      facet(facet) {
          let addr = this.config.address[facet.id];
          if (addr == null)
              return facet.default;
          ensureAddr(this, addr);
          return getAddr(this, addr);
      }
      /**
      Convert this state to a JSON-serializable object. When custom
      fields should be serialized, you can pass them in as an object
      mapping property names (in the resulting object, which should
      not use `doc` or `selection`) to fields.
      */
      toJSON(fields) {
          let result = {
              doc: this.sliceDoc(),
              selection: this.selection.toJSON()
          };
          if (fields)
              for (let prop in fields) {
                  let value = fields[prop];
                  if (value instanceof StateField && this.config.address[value.id] != null)
                      result[prop] = value.spec.toJSON(this.field(fields[prop]), this);
              }
          return result;
      }
      /**
      Deserialize a state from its JSON representation. When custom
      fields should be deserialized, pass the same object you passed
      to [`toJSON`](https://codemirror.net/6/docs/ref/#state.EditorState.toJSON) when serializing as
      third argument.
      */
      static fromJSON(json, config = {}, fields) {
          if (!json || typeof json.doc != "string")
              throw new RangeError("Invalid JSON representation for EditorState");
          let fieldInit = [];
          if (fields)
              for (let prop in fields) {
                  if (Object.prototype.hasOwnProperty.call(json, prop)) {
                      let field = fields[prop], value = json[prop];
                      fieldInit.push(field.init(state => field.spec.fromJSON(value, state)));
                  }
              }
          return EditorState.create({
              doc: json.doc,
              selection: EditorSelection.fromJSON(json.selection),
              extensions: config.extensions ? fieldInit.concat([config.extensions]) : fieldInit
          });
      }
      /**
      Create a new state. You'll usually only need this when
      initializing an editor—updated states are created by applying
      transactions.
      */
      static create(config = {}) {
          let configuration = Configuration.resolve(config.extensions || [], new Map);
          let doc = config.doc instanceof Text ? config.doc
              : Text.of((config.doc || "").split(configuration.staticFacet(EditorState.lineSeparator) || DefaultSplit));
          let selection = !config.selection ? EditorSelection.single(0)
              : config.selection instanceof EditorSelection ? config.selection
                  : EditorSelection.single(config.selection.anchor, config.selection.head);
          checkSelection(selection, doc.length);
          if (!configuration.staticFacet(allowMultipleSelections))
              selection = selection.asSingle();
          return new EditorState(configuration, doc, selection, configuration.dynamicSlots.map(() => null), (state, slot) => slot.create(state), null);
      }
      /**
      The size (in columns) of a tab in the document, determined by
      the [`tabSize`](https://codemirror.net/6/docs/ref/#state.EditorState^tabSize) facet.
      */
      get tabSize() { return this.facet(EditorState.tabSize); }
      /**
      Get the proper [line-break](https://codemirror.net/6/docs/ref/#state.EditorState^lineSeparator)
      string for this state.
      */
      get lineBreak() { return this.facet(EditorState.lineSeparator) || "\n"; }
      /**
      Returns true when the editor is
      [configured](https://codemirror.net/6/docs/ref/#state.EditorState^readOnly) to be read-only.
      */
      get readOnly() { return this.facet(readOnly); }
      /**
      Look up a translation for the given phrase (via the
      [`phrases`](https://codemirror.net/6/docs/ref/#state.EditorState^phrases) facet), or return the
      original string if no translation is found.
      
      If additional arguments are passed, they will be inserted in
      place of markers like `$1` (for the first value) and `$2`, etc.
      A single `$` is equivalent to `$1`, and `$$` will produce a
      literal dollar sign.
      */
      phrase(phrase, ...insert) {
          for (let map of this.facet(EditorState.phrases))
              if (Object.prototype.hasOwnProperty.call(map, phrase)) {
                  phrase = map[phrase];
                  break;
              }
          if (insert.length)
              phrase = phrase.replace(/\$(\$|\d*)/g, (m, i) => {
                  if (i == "$")
                      return "$";
                  let n = +(i || 1);
                  return !n || n > insert.length ? m : insert[n - 1];
              });
          return phrase;
      }
      /**
      Find the values for a given language data field, provided by the
      the [`languageData`](https://codemirror.net/6/docs/ref/#state.EditorState^languageData) facet.
      */
      languageDataAt(name, pos, side = -1) {
          let values = [];
          for (let provider of this.facet(languageData)) {
              for (let result of provider(this, pos, side)) {
                  if (Object.prototype.hasOwnProperty.call(result, name))
                      values.push(result[name]);
              }
          }
          return values;
      }
      /**
      Return a function that can categorize strings (expected to
      represent a single [grapheme cluster](https://codemirror.net/6/docs/ref/#state.findClusterBreak))
      into one of:
      
       - Word (contains an alphanumeric character or a character
         explicitly listed in the local language's `"wordChars"`
         language data, which should be a string)
       - Space (contains only whitespace)
       - Other (anything else)
      */
      charCategorizer(at) {
          return makeCategorizer(this.languageDataAt("wordChars", at).join(""));
      }
      /**
      Find the word at the given position, meaning the range
      containing all [word](https://codemirror.net/6/docs/ref/#state.CharCategory.Word) characters
      around it. If no word characters are adjacent to the position,
      this returns null.
      */
      wordAt(pos) {
          let { text, from, length } = this.doc.lineAt(pos);
          let cat = this.charCategorizer(pos);
          let start = pos - from, end = pos - from;
          while (start > 0) {
              let prev = findClusterBreak(text, start, false);
              if (cat(text.slice(prev, start)) != CharCategory.Word)
                  break;
              start = prev;
          }
          while (end < length) {
              let next = findClusterBreak(text, end);
              if (cat(text.slice(end, next)) != CharCategory.Word)
                  break;
              end = next;
          }
          return start == end ? null : EditorSelection.range(start + from, end + from);
      }
  }
  /**
  A facet that, when enabled, causes the editor to allow multiple
  ranges to be selected. Be careful though, because by default the
  editor relies on the native DOM selection, which cannot handle
  multiple selections. An extension like
  [`drawSelection`](https://codemirror.net/6/docs/ref/#view.drawSelection) can be used to make
  secondary selections visible to the user.
  */
  EditorState.allowMultipleSelections = allowMultipleSelections;
  /**
  Configures the tab size to use in this state. The first
  (highest-precedence) value of the facet is used. If no value is
  given, this defaults to 4.
  */
  EditorState.tabSize = /*@__PURE__*/Facet.define({
      combine: values => values.length ? values[0] : 4
  });
  /**
  The line separator to use. By default, any of `"\n"`, `"\r\n"`
  and `"\r"` is treated as a separator when splitting lines, and
  lines are joined with `"\n"`.

  When you configure a value here, only that precise separator
  will be used, allowing you to round-trip documents through the
  editor without normalizing line separators.
  */
  EditorState.lineSeparator = lineSeparator;
  /**
  This facet controls the value of the
  [`readOnly`](https://codemirror.net/6/docs/ref/#state.EditorState.readOnly) getter, which is
  consulted by commands and extensions that implement editing
  functionality to determine whether they should apply. It
  defaults to false, but when its highest-precedence value is
  `true`, such functionality disables itself.

  Not to be confused with
  [`EditorView.editable`](https://codemirror.net/6/docs/ref/#view.EditorView^editable), which
  controls whether the editor's DOM is set to be editable (and
  thus focusable).
  */
  EditorState.readOnly = readOnly;
  /**
  Registers translation phrases. The
  [`phrase`](https://codemirror.net/6/docs/ref/#state.EditorState.phrase) method will look through
  all objects registered with this facet to find translations for
  its argument.
  */
  EditorState.phrases = /*@__PURE__*/Facet.define({
      compare(a, b) {
          let kA = Object.keys(a), kB = Object.keys(b);
          return kA.length == kB.length && kA.every(k => a[k] == b[k]);
      }
  });
  /**
  A facet used to register [language
  data](https://codemirror.net/6/docs/ref/#state.EditorState.languageDataAt) providers.
  */
  EditorState.languageData = languageData;
  /**
  Facet used to register change filters, which are called for each
  transaction (unless explicitly
  [disabled](https://codemirror.net/6/docs/ref/#state.TransactionSpec.filter)), and can suppress
  part of the transaction's changes.

  Such a function can return `true` to indicate that it doesn't
  want to do anything, `false` to completely stop the changes in
  the transaction, or a set of ranges in which changes should be
  suppressed. Such ranges are represented as an array of numbers,
  with each pair of two numbers indicating the start and end of a
  range. So for example `[10, 20, 100, 110]` suppresses changes
  between 10 and 20, and between 100 and 110.
  */
  EditorState.changeFilter = changeFilter;
  /**
  Facet used to register a hook that gets a chance to update or
  replace transaction specs before they are applied. This will
  only be applied for transactions that don't have
  [`filter`](https://codemirror.net/6/docs/ref/#state.TransactionSpec.filter) set to `false`. You
  can either return a single transaction spec (possibly the input
  transaction), or an array of specs (which will be combined in
  the same way as the arguments to
  [`EditorState.update`](https://codemirror.net/6/docs/ref/#state.EditorState.update)).

  When possible, it is recommended to avoid accessing
  [`Transaction.state`](https://codemirror.net/6/docs/ref/#state.Transaction.state) in a filter,
  since it will force creation of a state that will then be
  discarded again, if the transaction is actually filtered.

  (This functionality should be used with care. Indiscriminately
  modifying transaction is likely to break something or degrade
  the user experience.)
  */
  EditorState.transactionFilter = transactionFilter;
  /**
  This is a more limited form of
  [`transactionFilter`](https://codemirror.net/6/docs/ref/#state.EditorState^transactionFilter),
  which can only add
  [annotations](https://codemirror.net/6/docs/ref/#state.TransactionSpec.annotations) and
  [effects](https://codemirror.net/6/docs/ref/#state.TransactionSpec.effects). _But_, this type
  of filter runs even if the transaction has disabled regular
  [filtering](https://codemirror.net/6/docs/ref/#state.TransactionSpec.filter), making it suitable
  for effects that don't need to touch the changes or selection,
  but do want to process every transaction.

  Extenders run _after_ filters, when both are present.
  */
  EditorState.transactionExtender = transactionExtender;
  Compartment.reconfigure = /*@__PURE__*/StateEffect.define();

  /**
  Utility function for combining behaviors to fill in a config
  object from an array of provided configs. `defaults` should hold
  default values for all optional fields in `Config`.

  The function will, by default, error
  when a field gets two values that aren't `===`-equal, but you can
  provide combine functions per field to do something else.
  */
  function combineConfig(configs, defaults, // Should hold only the optional properties of Config, but I haven't managed to express that
  combine = {}) {
      let result = {};
      for (let config of configs)
          for (let key of Object.keys(config)) {
              let value = config[key], current = result[key];
              if (current === undefined)
                  result[key] = value;
              else if (current === value || value === undefined) ; // No conflict
              else if (Object.hasOwnProperty.call(combine, key))
                  result[key] = combine[key](current, value);
              else
                  throw new Error("Config merge conflict for field " + key);
          }
      for (let key in defaults)
          if (result[key] === undefined)
              result[key] = defaults[key];
      return result;
  }

  /**
  Each range is associated with a value, which must inherit from
  this class.
  */
  class RangeValue {
      /**
      Compare this value with another value. Used when comparing
      rangesets. The default implementation compares by identity.
      Unless you are only creating a fixed number of unique instances
      of your value type, it is a good idea to implement this
      properly.
      */
      eq(other) { return this == other; }
      /**
      Create a [range](https://codemirror.net/6/docs/ref/#state.Range) with this value.
      */
      range(from, to = from) { return Range$1.create(from, to, this); }
  }
  RangeValue.prototype.startSide = RangeValue.prototype.endSide = 0;
  RangeValue.prototype.point = false;
  RangeValue.prototype.mapMode = MapMode.TrackDel;
  /**
  A range associates a value with a range of positions.
  */
  class Range$1 {
      constructor(
      /**
      The range's start position.
      */
      from, 
      /**
      Its end position.
      */
      to, 
      /**
      The value associated with this range.
      */
      value) {
          this.from = from;
          this.to = to;
          this.value = value;
      }
      /**
      @internal
      */
      static create(from, to, value) {
          return new Range$1(from, to, value);
      }
  }
  function cmpRange(a, b) {
      return a.from - b.from || a.value.startSide - b.value.startSide;
  }
  class Chunk {
      constructor(from, to, value, 
      // Chunks are marked with the largest point that occurs
      // in them (or -1 for no points), so that scans that are
      // only interested in points (such as the
      // heightmap-related logic) can skip range-only chunks.
      maxPoint) {
          this.from = from;
          this.to = to;
          this.value = value;
          this.maxPoint = maxPoint;
      }
      get length() { return this.to[this.to.length - 1]; }
      // Find the index of the given position and side. Use the ranges'
      // `from` pos when `end == false`, `to` when `end == true`.
      findIndex(pos, side, end, startAt = 0) {
          let arr = end ? this.to : this.from;
          for (let lo = startAt, hi = arr.length;;) {
              if (lo == hi)
                  return lo;
              let mid = (lo + hi) >> 1;
              let diff = arr[mid] - pos || (end ? this.value[mid].endSide : this.value[mid].startSide) - side;
              if (mid == lo)
                  return diff >= 0 ? lo : hi;
              if (diff >= 0)
                  hi = mid;
              else
                  lo = mid + 1;
          }
      }
      between(offset, from, to, f) {
          for (let i = this.findIndex(from, -1000000000 /* Far */, true), e = this.findIndex(to, 1000000000 /* Far */, false, i); i < e; i++)
              if (f(this.from[i] + offset, this.to[i] + offset, this.value[i]) === false)
                  return false;
      }
      map(offset, changes) {
          let value = [], from = [], to = [], newPos = -1, maxPoint = -1;
          for (let i = 0; i < this.value.length; i++) {
              let val = this.value[i], curFrom = this.from[i] + offset, curTo = this.to[i] + offset, newFrom, newTo;
              if (curFrom == curTo) {
                  let mapped = changes.mapPos(curFrom, val.startSide, val.mapMode);
                  if (mapped == null)
                      continue;
                  newFrom = newTo = mapped;
                  if (val.startSide != val.endSide) {
                      newTo = changes.mapPos(curFrom, val.endSide);
                      if (newTo < newFrom)
                          continue;
                  }
              }
              else {
                  newFrom = changes.mapPos(curFrom, val.startSide);
                  newTo = changes.mapPos(curTo, val.endSide);
                  if (newFrom > newTo || newFrom == newTo && val.startSide > 0 && val.endSide <= 0)
                      continue;
              }
              if ((newTo - newFrom || val.endSide - val.startSide) < 0)
                  continue;
              if (newPos < 0)
                  newPos = newFrom;
              if (val.point)
                  maxPoint = Math.max(maxPoint, newTo - newFrom);
              value.push(val);
              from.push(newFrom - newPos);
              to.push(newTo - newPos);
          }
          return { mapped: value.length ? new Chunk(from, to, value, maxPoint) : null, pos: newPos };
      }
  }
  /**
  A range set stores a collection of [ranges](https://codemirror.net/6/docs/ref/#state.Range) in a
  way that makes them efficient to [map](https://codemirror.net/6/docs/ref/#state.RangeSet.map) and
  [update](https://codemirror.net/6/docs/ref/#state.RangeSet.update). This is an immutable data
  structure.
  */
  class RangeSet {
      constructor(
      /**
      @internal
      */
      chunkPos, 
      /**
      @internal
      */
      chunk, 
      /**
      @internal
      */
      nextLayer, 
      /**
      @internal
      */
      maxPoint) {
          this.chunkPos = chunkPos;
          this.chunk = chunk;
          this.nextLayer = nextLayer;
          this.maxPoint = maxPoint;
      }
      /**
      @internal
      */
      static create(chunkPos, chunk, nextLayer, maxPoint) {
          return new RangeSet(chunkPos, chunk, nextLayer, maxPoint);
      }
      /**
      @internal
      */
      get length() {
          let last = this.chunk.length - 1;
          return last < 0 ? 0 : Math.max(this.chunkEnd(last), this.nextLayer.length);
      }
      /**
      The number of ranges in the set.
      */
      get size() {
          if (this.isEmpty)
              return 0;
          let size = this.nextLayer.size;
          for (let chunk of this.chunk)
              size += chunk.value.length;
          return size;
      }
      /**
      @internal
      */
      chunkEnd(index) {
          return this.chunkPos[index] + this.chunk[index].length;
      }
      /**
      Update the range set, optionally adding new ranges or filtering
      out existing ones.
      
      (Note: The type parameter is just there as a kludge to work
      around TypeScript variance issues that prevented `RangeSet<X>`
      from being a subtype of `RangeSet<Y>` when `X` is a subtype of
      `Y`.)
      */
      update(updateSpec) {
          let { add = [], sort = false, filterFrom = 0, filterTo = this.length } = updateSpec;
          let filter = updateSpec.filter;
          if (add.length == 0 && !filter)
              return this;
          if (sort)
              add = add.slice().sort(cmpRange);
          if (this.isEmpty)
              return add.length ? RangeSet.of(add) : this;
          let cur = new LayerCursor(this, null, -1).goto(0), i = 0, spill = [];
          let builder = new RangeSetBuilder();
          while (cur.value || i < add.length) {
              if (i < add.length && (cur.from - add[i].from || cur.startSide - add[i].value.startSide) >= 0) {
                  let range = add[i++];
                  if (!builder.addInner(range.from, range.to, range.value))
                      spill.push(range);
              }
              else if (cur.rangeIndex == 1 && cur.chunkIndex < this.chunk.length &&
                  (i == add.length || this.chunkEnd(cur.chunkIndex) < add[i].from) &&
                  (!filter || filterFrom > this.chunkEnd(cur.chunkIndex) || filterTo < this.chunkPos[cur.chunkIndex]) &&
                  builder.addChunk(this.chunkPos[cur.chunkIndex], this.chunk[cur.chunkIndex])) {
                  cur.nextChunk();
              }
              else {
                  if (!filter || filterFrom > cur.to || filterTo < cur.from || filter(cur.from, cur.to, cur.value)) {
                      if (!builder.addInner(cur.from, cur.to, cur.value))
                          spill.push(Range$1.create(cur.from, cur.to, cur.value));
                  }
                  cur.next();
              }
          }
          return builder.finishInner(this.nextLayer.isEmpty && !spill.length ? RangeSet.empty
              : this.nextLayer.update({ add: spill, filter, filterFrom, filterTo }));
      }
      /**
      Map this range set through a set of changes, return the new set.
      */
      map(changes) {
          if (changes.empty || this.isEmpty)
              return this;
          let chunks = [], chunkPos = [], maxPoint = -1;
          for (let i = 0; i < this.chunk.length; i++) {
              let start = this.chunkPos[i], chunk = this.chunk[i];
              let touch = changes.touchesRange(start, start + chunk.length);
              if (touch === false) {
                  maxPoint = Math.max(maxPoint, chunk.maxPoint);
                  chunks.push(chunk);
                  chunkPos.push(changes.mapPos(start));
              }
              else if (touch === true) {
                  let { mapped, pos } = chunk.map(start, changes);
                  if (mapped) {
                      maxPoint = Math.max(maxPoint, mapped.maxPoint);
                      chunks.push(mapped);
                      chunkPos.push(pos);
                  }
              }
          }
          let next = this.nextLayer.map(changes);
          return chunks.length == 0 ? next : new RangeSet(chunkPos, chunks, next || RangeSet.empty, maxPoint);
      }
      /**
      Iterate over the ranges that touch the region `from` to `to`,
      calling `f` for each. There is no guarantee that the ranges will
      be reported in any specific order. When the callback returns
      `false`, iteration stops.
      */
      between(from, to, f) {
          if (this.isEmpty)
              return;
          for (let i = 0; i < this.chunk.length; i++) {
              let start = this.chunkPos[i], chunk = this.chunk[i];
              if (to >= start && from <= start + chunk.length &&
                  chunk.between(start, from - start, to - start, f) === false)
                  return;
          }
          this.nextLayer.between(from, to, f);
      }
      /**
      Iterate over the ranges in this set, in order, including all
      ranges that end at or after `from`.
      */
      iter(from = 0) {
          return HeapCursor.from([this]).goto(from);
      }
      /**
      @internal
      */
      get isEmpty() { return this.nextLayer == this; }
      /**
      Iterate over the ranges in a collection of sets, in order,
      starting from `from`.
      */
      static iter(sets, from = 0) {
          return HeapCursor.from(sets).goto(from);
      }
      /**
      Iterate over two groups of sets, calling methods on `comparator`
      to notify it of possible differences.
      */
      static compare(oldSets, newSets, 
      /**
      This indicates how the underlying data changed between these
      ranges, and is needed to synchronize the iteration. `from` and
      `to` are coordinates in the _new_ space, after these changes.
      */
      textDiff, comparator, 
      /**
      Can be used to ignore all non-point ranges, and points below
      the given size. When -1, all ranges are compared.
      */
      minPointSize = -1) {
          let a = oldSets.filter(set => set.maxPoint > 0 || !set.isEmpty && set.maxPoint >= minPointSize);
          let b = newSets.filter(set => set.maxPoint > 0 || !set.isEmpty && set.maxPoint >= minPointSize);
          let sharedChunks = findSharedChunks(a, b, textDiff);
          let sideA = new SpanCursor(a, sharedChunks, minPointSize);
          let sideB = new SpanCursor(b, sharedChunks, minPointSize);
          textDiff.iterGaps((fromA, fromB, length) => compare(sideA, fromA, sideB, fromB, length, comparator));
          if (textDiff.empty && textDiff.length == 0)
              compare(sideA, 0, sideB, 0, 0, comparator);
      }
      /**
      Compare the contents of two groups of range sets, returning true
      if they are equivalent in the given range.
      */
      static eq(oldSets, newSets, from = 0, to) {
          if (to == null)
              to = 1000000000 /* Far */;
          let a = oldSets.filter(set => !set.isEmpty && newSets.indexOf(set) < 0);
          let b = newSets.filter(set => !set.isEmpty && oldSets.indexOf(set) < 0);
          if (a.length != b.length)
              return false;
          if (!a.length)
              return true;
          let sharedChunks = findSharedChunks(a, b);
          let sideA = new SpanCursor(a, sharedChunks, 0).goto(from), sideB = new SpanCursor(b, sharedChunks, 0).goto(from);
          for (;;) {
              if (sideA.to != sideB.to ||
                  !sameValues(sideA.active, sideB.active) ||
                  sideA.point && (!sideB.point || !sideA.point.eq(sideB.point)))
                  return false;
              if (sideA.to > to)
                  return true;
              sideA.next();
              sideB.next();
          }
      }
      /**
      Iterate over a group of range sets at the same time, notifying
      the iterator about the ranges covering every given piece of
      content. Returns the open count (see
      [`SpanIterator.span`](https://codemirror.net/6/docs/ref/#state.SpanIterator.span)) at the end
      of the iteration.
      */
      static spans(sets, from, to, iterator, 
      /**
      When given and greater than -1, only points of at least this
      size are taken into account.
      */
      minPointSize = -1) {
          let cursor = new SpanCursor(sets, null, minPointSize).goto(from), pos = from;
          let open = cursor.openStart;
          for (;;) {
              let curTo = Math.min(cursor.to, to);
              if (cursor.point) {
                  iterator.point(pos, curTo, cursor.point, cursor.activeForPoint(cursor.to), open, cursor.pointRank);
                  open = cursor.openEnd(curTo) + (cursor.to > curTo ? 1 : 0);
              }
              else if (curTo > pos) {
                  iterator.span(pos, curTo, cursor.active, open);
                  open = cursor.openEnd(curTo);
              }
              if (cursor.to > to)
                  break;
              pos = cursor.to;
              cursor.next();
          }
          return open;
      }
      /**
      Create a range set for the given range or array of ranges. By
      default, this expects the ranges to be _sorted_ (by start
      position and, if two start at the same position,
      `value.startSide`). You can pass `true` as second argument to
      cause the method to sort them.
      */
      static of(ranges, sort = false) {
          let build = new RangeSetBuilder();
          for (let range of ranges instanceof Range$1 ? [ranges] : sort ? lazySort(ranges) : ranges)
              build.add(range.from, range.to, range.value);
          return build.finish();
      }
  }
  /**
  The empty set of ranges.
  */
  RangeSet.empty = /*@__PURE__*/new RangeSet([], [], null, -1);
  function lazySort(ranges) {
      if (ranges.length > 1)
          for (let prev = ranges[0], i = 1; i < ranges.length; i++) {
              let cur = ranges[i];
              if (cmpRange(prev, cur) > 0)
                  return ranges.slice().sort(cmpRange);
              prev = cur;
          }
      return ranges;
  }
  RangeSet.empty.nextLayer = RangeSet.empty;
  /**
  A range set builder is a data structure that helps build up a
  [range set](https://codemirror.net/6/docs/ref/#state.RangeSet) directly, without first allocating
  an array of [`Range`](https://codemirror.net/6/docs/ref/#state.Range) objects.
  */
  class RangeSetBuilder {
      /**
      Create an empty builder.
      */
      constructor() {
          this.chunks = [];
          this.chunkPos = [];
          this.chunkStart = -1;
          this.last = null;
          this.lastFrom = -1000000000 /* Far */;
          this.lastTo = -1000000000 /* Far */;
          this.from = [];
          this.to = [];
          this.value = [];
          this.maxPoint = -1;
          this.setMaxPoint = -1;
          this.nextLayer = null;
      }
      finishChunk(newArrays) {
          this.chunks.push(new Chunk(this.from, this.to, this.value, this.maxPoint));
          this.chunkPos.push(this.chunkStart);
          this.chunkStart = -1;
          this.setMaxPoint = Math.max(this.setMaxPoint, this.maxPoint);
          this.maxPoint = -1;
          if (newArrays) {
              this.from = [];
              this.to = [];
              this.value = [];
          }
      }
      /**
      Add a range. Ranges should be added in sorted (by `from` and
      `value.startSide`) order.
      */
      add(from, to, value) {
          if (!this.addInner(from, to, value))
              (this.nextLayer || (this.nextLayer = new RangeSetBuilder)).add(from, to, value);
      }
      /**
      @internal
      */
      addInner(from, to, value) {
          let diff = from - this.lastTo || value.startSide - this.last.endSide;
          if (diff <= 0 && (from - this.lastFrom || value.startSide - this.last.startSide) < 0)
              throw new Error("Ranges must be added sorted by `from` position and `startSide`");
          if (diff < 0)
              return false;
          if (this.from.length == 250 /* ChunkSize */)
              this.finishChunk(true);
          if (this.chunkStart < 0)
              this.chunkStart = from;
          this.from.push(from - this.chunkStart);
          this.to.push(to - this.chunkStart);
          this.last = value;
          this.lastFrom = from;
          this.lastTo = to;
          this.value.push(value);
          if (value.point)
              this.maxPoint = Math.max(this.maxPoint, to - from);
          return true;
      }
      /**
      @internal
      */
      addChunk(from, chunk) {
          if ((from - this.lastTo || chunk.value[0].startSide - this.last.endSide) < 0)
              return false;
          if (this.from.length)
              this.finishChunk(true);
          this.setMaxPoint = Math.max(this.setMaxPoint, chunk.maxPoint);
          this.chunks.push(chunk);
          this.chunkPos.push(from);
          let last = chunk.value.length - 1;
          this.last = chunk.value[last];
          this.lastFrom = chunk.from[last] + from;
          this.lastTo = chunk.to[last] + from;
          return true;
      }
      /**
      Finish the range set. Returns the new set. The builder can't be
      used anymore after this has been called.
      */
      finish() { return this.finishInner(RangeSet.empty); }
      /**
      @internal
      */
      finishInner(next) {
          if (this.from.length)
              this.finishChunk(false);
          if (this.chunks.length == 0)
              return next;
          let result = RangeSet.create(this.chunkPos, this.chunks, this.nextLayer ? this.nextLayer.finishInner(next) : next, this.setMaxPoint);
          this.from = null; // Make sure further `add` calls produce errors
          return result;
      }
  }
  function findSharedChunks(a, b, textDiff) {
      let inA = new Map();
      for (let set of a)
          for (let i = 0; i < set.chunk.length; i++)
              if (set.chunk[i].maxPoint <= 0)
                  inA.set(set.chunk[i], set.chunkPos[i]);
      let shared = new Set();
      for (let set of b)
          for (let i = 0; i < set.chunk.length; i++) {
              let known = inA.get(set.chunk[i]);
              if (known != null && (textDiff ? textDiff.mapPos(known) : known) == set.chunkPos[i] &&
                  !(textDiff === null || textDiff === void 0 ? void 0 : textDiff.touchesRange(known, known + set.chunk[i].length)))
                  shared.add(set.chunk[i]);
          }
      return shared;
  }
  class LayerCursor {
      constructor(layer, skip, minPoint, rank = 0) {
          this.layer = layer;
          this.skip = skip;
          this.minPoint = minPoint;
          this.rank = rank;
      }
      get startSide() { return this.value ? this.value.startSide : 0; }
      get endSide() { return this.value ? this.value.endSide : 0; }
      goto(pos, side = -1000000000 /* Far */) {
          this.chunkIndex = this.rangeIndex = 0;
          this.gotoInner(pos, side, false);
          return this;
      }
      gotoInner(pos, side, forward) {
          while (this.chunkIndex < this.layer.chunk.length) {
              let next = this.layer.chunk[this.chunkIndex];
              if (!(this.skip && this.skip.has(next) ||
                  this.layer.chunkEnd(this.chunkIndex) < pos ||
                  next.maxPoint < this.minPoint))
                  break;
              this.chunkIndex++;
              forward = false;
          }
          if (this.chunkIndex < this.layer.chunk.length) {
              let rangeIndex = this.layer.chunk[this.chunkIndex].findIndex(pos - this.layer.chunkPos[this.chunkIndex], side, true);
              if (!forward || this.rangeIndex < rangeIndex)
                  this.setRangeIndex(rangeIndex);
          }
          this.next();
      }
      forward(pos, side) {
          if ((this.to - pos || this.endSide - side) < 0)
              this.gotoInner(pos, side, true);
      }
      next() {
          for (;;) {
              if (this.chunkIndex == this.layer.chunk.length) {
                  this.from = this.to = 1000000000 /* Far */;
                  this.value = null;
                  break;
              }
              else {
                  let chunkPos = this.layer.chunkPos[this.chunkIndex], chunk = this.layer.chunk[this.chunkIndex];
                  let from = chunkPos + chunk.from[this.rangeIndex];
                  this.from = from;
                  this.to = chunkPos + chunk.to[this.rangeIndex];
                  this.value = chunk.value[this.rangeIndex];
                  this.setRangeIndex(this.rangeIndex + 1);
                  if (this.minPoint < 0 || this.value.point && this.to - this.from >= this.minPoint)
                      break;
              }
          }
      }
      setRangeIndex(index) {
          if (index == this.layer.chunk[this.chunkIndex].value.length) {
              this.chunkIndex++;
              if (this.skip) {
                  while (this.chunkIndex < this.layer.chunk.length && this.skip.has(this.layer.chunk[this.chunkIndex]))
                      this.chunkIndex++;
              }
              this.rangeIndex = 0;
          }
          else {
              this.rangeIndex = index;
          }
      }
      nextChunk() {
          this.chunkIndex++;
          this.rangeIndex = 0;
          this.next();
      }
      compare(other) {
          return this.from - other.from || this.startSide - other.startSide || this.rank - other.rank ||
              this.to - other.to || this.endSide - other.endSide;
      }
  }
  class HeapCursor {
      constructor(heap) {
          this.heap = heap;
      }
      static from(sets, skip = null, minPoint = -1) {
          let heap = [];
          for (let i = 0; i < sets.length; i++) {
              for (let cur = sets[i]; !cur.isEmpty; cur = cur.nextLayer) {
                  if (cur.maxPoint >= minPoint)
                      heap.push(new LayerCursor(cur, skip, minPoint, i));
              }
          }
          return heap.length == 1 ? heap[0] : new HeapCursor(heap);
      }
      get startSide() { return this.value ? this.value.startSide : 0; }
      goto(pos, side = -1000000000 /* Far */) {
          for (let cur of this.heap)
              cur.goto(pos, side);
          for (let i = this.heap.length >> 1; i >= 0; i--)
              heapBubble(this.heap, i);
          this.next();
          return this;
      }
      forward(pos, side) {
          for (let cur of this.heap)
              cur.forward(pos, side);
          for (let i = this.heap.length >> 1; i >= 0; i--)
              heapBubble(this.heap, i);
          if ((this.to - pos || this.value.endSide - side) < 0)
              this.next();
      }
      next() {
          if (this.heap.length == 0) {
              this.from = this.to = 1000000000 /* Far */;
              this.value = null;
              this.rank = -1;
          }
          else {
              let top = this.heap[0];
              this.from = top.from;
              this.to = top.to;
              this.value = top.value;
              this.rank = top.rank;
              if (top.value)
                  top.next();
              heapBubble(this.heap, 0);
          }
      }
  }
  function heapBubble(heap, index) {
      for (let cur = heap[index];;) {
          let childIndex = (index << 1) + 1;
          if (childIndex >= heap.length)
              break;
          let child = heap[childIndex];
          if (childIndex + 1 < heap.length && child.compare(heap[childIndex + 1]) >= 0) {
              child = heap[childIndex + 1];
              childIndex++;
          }
          if (cur.compare(child) < 0)
              break;
          heap[childIndex] = cur;
          heap[index] = child;
          index = childIndex;
      }
  }
  class SpanCursor {
      constructor(sets, skip, minPoint) {
          this.minPoint = minPoint;
          this.active = [];
          this.activeTo = [];
          this.activeRank = [];
          this.minActive = -1;
          // A currently active point range, if any
          this.point = null;
          this.pointFrom = 0;
          this.pointRank = 0;
          this.to = -1000000000 /* Far */;
          this.endSide = 0;
          this.openStart = -1;
          this.cursor = HeapCursor.from(sets, skip, minPoint);
      }
      goto(pos, side = -1000000000 /* Far */) {
          this.cursor.goto(pos, side);
          this.active.length = this.activeTo.length = this.activeRank.length = 0;
          this.minActive = -1;
          this.to = pos;
          this.endSide = side;
          this.openStart = -1;
          this.next();
          return this;
      }
      forward(pos, side) {
          while (this.minActive > -1 && (this.activeTo[this.minActive] - pos || this.active[this.minActive].endSide - side) < 0)
              this.removeActive(this.minActive);
          this.cursor.forward(pos, side);
      }
      removeActive(index) {
          remove(this.active, index);
          remove(this.activeTo, index);
          remove(this.activeRank, index);
          this.minActive = findMinIndex(this.active, this.activeTo);
      }
      addActive(trackOpen) {
          let i = 0, { value, to, rank } = this.cursor;
          while (i < this.activeRank.length && this.activeRank[i] <= rank)
              i++;
          insert(this.active, i, value);
          insert(this.activeTo, i, to);
          insert(this.activeRank, i, rank);
          if (trackOpen)
              insert(trackOpen, i, this.cursor.from);
          this.minActive = findMinIndex(this.active, this.activeTo);
      }
      // After calling this, if `this.point` != null, the next range is a
      // point. Otherwise, it's a regular range, covered by `this.active`.
      next() {
          let from = this.to, wasPoint = this.point;
          this.point = null;
          let trackOpen = this.openStart < 0 ? [] : null, trackExtra = 0;
          for (;;) {
              let a = this.minActive;
              if (a > -1 && (this.activeTo[a] - this.cursor.from || this.active[a].endSide - this.cursor.startSide) < 0) {
                  if (this.activeTo[a] > from) {
                      this.to = this.activeTo[a];
                      this.endSide = this.active[a].endSide;
                      break;
                  }
                  this.removeActive(a);
                  if (trackOpen)
                      remove(trackOpen, a);
              }
              else if (!this.cursor.value) {
                  this.to = this.endSide = 1000000000 /* Far */;
                  break;
              }
              else if (this.cursor.from > from) {
                  this.to = this.cursor.from;
                  this.endSide = this.cursor.startSide;
                  break;
              }
              else {
                  let nextVal = this.cursor.value;
                  if (!nextVal.point) { // Opening a range
                      this.addActive(trackOpen);
                      this.cursor.next();
                  }
                  else if (wasPoint && this.cursor.to == this.to && this.cursor.from < this.cursor.to) {
                      // Ignore any non-empty points that end precisely at the end of the prev point
                      this.cursor.next();
                  }
                  else { // New point
                      this.point = nextVal;
                      this.pointFrom = this.cursor.from;
                      this.pointRank = this.cursor.rank;
                      this.to = this.cursor.to;
                      this.endSide = nextVal.endSide;
                      if (this.cursor.from < from)
                          trackExtra = 1;
                      this.cursor.next();
                      this.forward(this.to, this.endSide);
                      break;
                  }
              }
          }
          if (trackOpen) {
              let openStart = 0;
              while (openStart < trackOpen.length && trackOpen[openStart] < from)
                  openStart++;
              this.openStart = openStart + trackExtra;
          }
      }
      activeForPoint(to) {
          if (!this.active.length)
              return this.active;
          let active = [];
          for (let i = this.active.length - 1; i >= 0; i--) {
              if (this.activeRank[i] < this.pointRank)
                  break;
              if (this.activeTo[i] > to || this.activeTo[i] == to && this.active[i].endSide >= this.point.endSide)
                  active.push(this.active[i]);
          }
          return active.reverse();
      }
      openEnd(to) {
          let open = 0;
          for (let i = this.activeTo.length - 1; i >= 0 && this.activeTo[i] > to; i--)
              open++;
          return open;
      }
  }
  function compare(a, startA, b, startB, length, comparator) {
      a.goto(startA);
      b.goto(startB);
      let endB = startB + length;
      let pos = startB, dPos = startB - startA;
      for (;;) {
          let diff = (a.to + dPos) - b.to || a.endSide - b.endSide;
          let end = diff < 0 ? a.to + dPos : b.to, clipEnd = Math.min(end, endB);
          if (a.point || b.point) {
              if (!(a.point && b.point && (a.point == b.point || a.point.eq(b.point)) &&
                  sameValues(a.activeForPoint(a.to + dPos), b.activeForPoint(b.to))))
                  comparator.comparePoint(pos, clipEnd, a.point, b.point);
          }
          else {
              if (clipEnd > pos && !sameValues(a.active, b.active))
                  comparator.compareRange(pos, clipEnd, a.active, b.active);
          }
          if (end > endB)
              break;
          pos = end;
          if (diff <= 0)
              a.next();
          if (diff >= 0)
              b.next();
      }
  }
  function sameValues(a, b) {
      if (a.length != b.length)
          return false;
      for (let i = 0; i < a.length; i++)
          if (a[i] != b[i] && !a[i].eq(b[i]))
              return false;
      return true;
  }
  function remove(array, index) {
      for (let i = index, e = array.length - 1; i < e; i++)
          array[i] = array[i + 1];
      array.pop();
  }
  function insert(array, index, value) {
      for (let i = array.length - 1; i >= index; i--)
          array[i + 1] = array[i];
      array[index] = value;
  }
  function findMinIndex(value, array) {
      let found = -1, foundPos = 1000000000 /* Far */;
      for (let i = 0; i < array.length; i++)
          if ((array[i] - foundPos || value[i].endSide - value[found].endSide) < 0) {
              found = i;
              foundPos = array[i];
          }
      return found;
  }

  /**
  Count the column position at the given offset into the string,
  taking extending characters and tab size into account.
  */
  function countColumn(string, tabSize, to = string.length) {
      let n = 0;
      for (let i = 0; i < to;) {
          if (string.charCodeAt(i) == 9) {
              n += tabSize - (n % tabSize);
              i++;
          }
          else {
              n++;
              i = findClusterBreak(string, i);
          }
      }
      return n;
  }
  /**
  Find the offset that corresponds to the given column position in a
  string, taking extending characters and tab size into account. By
  default, the string length is returned when it is too short to
  reach the column. Pass `strict` true to make it return -1 in that
  situation.
  */
  function findColumn(string, col, tabSize, strict) {
      for (let i = 0, n = 0;;) {
          if (n >= col)
              return i;
          if (i == string.length)
              break;
          n += string.charCodeAt(i) == 9 ? tabSize - (n % tabSize) : 1;
          i = findClusterBreak(string, i);
      }
      return strict === true ? -1 : string.length;
  }

  const C = "\u037c";
  const COUNT = typeof Symbol == "undefined" ? "__" + C : Symbol.for(C);
  const SET = typeof Symbol == "undefined" ? "__styleSet" + Math.floor(Math.random() * 1e8) : Symbol("styleSet");
  const top = typeof globalThis != "undefined" ? globalThis : typeof window != "undefined" ? window : {};

  // :: - Style modules encapsulate a set of CSS rules defined from
  // JavaScript. Their definitions are only available in a given DOM
  // root after it has been _mounted_ there with `StyleModule.mount`.
  //
  // Style modules should be created once and stored somewhere, as
  // opposed to re-creating them every time you need them. The amount of
  // CSS rules generated for a given DOM root is bounded by the amount
  // of style modules that were used. So to avoid leaking rules, don't
  // create these dynamically, but treat them as one-time allocations.
  class StyleModule {
    // :: (Object<Style>, ?{finish: ?(string) → string})
    // Create a style module from the given spec.
    //
    // When `finish` is given, it is called on regular (non-`@`)
    // selectors (after `&` expansion) to compute the final selector.
    constructor(spec, options) {
      this.rules = [];
      let {finish} = options || {};

      function splitSelector(selector) {
        return /^@/.test(selector) ? [selector] : selector.split(/,\s*/)
      }

      function render(selectors, spec, target, isKeyframes) {
        let local = [], isAt = /^@(\w+)\b/.exec(selectors[0]), keyframes = isAt && isAt[1] == "keyframes";
        if (isAt && spec == null) return target.push(selectors[0] + ";")
        for (let prop in spec) {
          let value = spec[prop];
          if (/&/.test(prop)) {
            render(prop.split(/,\s*/).map(part => selectors.map(sel => part.replace(/&/, sel))).reduce((a, b) => a.concat(b)),
                   value, target);
          } else if (value && typeof value == "object") {
            if (!isAt) throw new RangeError("The value of a property (" + prop + ") should be a primitive value.")
            render(splitSelector(prop), value, local, keyframes);
          } else if (value != null) {
            local.push(prop.replace(/_.*/, "").replace(/[A-Z]/g, l => "-" + l.toLowerCase()) + ": " + value + ";");
          }
        }
        if (local.length || keyframes) {
          target.push((finish && !isAt && !isKeyframes ? selectors.map(finish) : selectors).join(", ") +
                      " {" + local.join(" ") + "}");
        }
      }

      for (let prop in spec) render(splitSelector(prop), spec[prop], this.rules);
    }

    // :: () → string
    // Returns a string containing the module's CSS rules.
    getRules() { return this.rules.join("\n") }

    // :: () → string
    // Generate a new unique CSS class name.
    static newName() {
      let id = top[COUNT] || 1;
      top[COUNT] = id + 1;
      return C + id.toString(36)
    }

    // :: (union<Document, ShadowRoot>, union<[StyleModule], StyleModule>)
    //
    // Mount the given set of modules in the given DOM root, which ensures
    // that the CSS rules defined by the module are available in that
    // context.
    //
    // Rules are only added to the document once per root.
    //
    // Rule order will follow the order of the modules, so that rules from
    // modules later in the array take precedence of those from earlier
    // modules. If you call this function multiple times for the same root
    // in a way that changes the order of already mounted modules, the old
    // order will be changed.
    static mount(root, modules) {
      (root[SET] || new StyleSet(root)).mount(Array.isArray(modules) ? modules : [modules]);
    }
  }

  let adoptedSet = null;

  class StyleSet {
    constructor(root) {
      if (!root.head && root.adoptedStyleSheets && typeof CSSStyleSheet != "undefined") {
        if (adoptedSet) {
          root.adoptedStyleSheets = [adoptedSet.sheet].concat(root.adoptedStyleSheets);
          return root[SET] = adoptedSet
        }
        this.sheet = new CSSStyleSheet;
        root.adoptedStyleSheets = [this.sheet].concat(root.adoptedStyleSheets);
        adoptedSet = this;
      } else {
        this.styleTag = (root.ownerDocument || root).createElement("style");
        let target = root.head || root;
        target.insertBefore(this.styleTag, target.firstChild);
      }
      this.modules = [];
      root[SET] = this;
    }

    mount(modules) {
      let sheet = this.sheet;
      let pos = 0 /* Current rule offset */, j = 0; /* Index into this.modules */
      for (let i = 0; i < modules.length; i++) {
        let mod = modules[i], index = this.modules.indexOf(mod);
        if (index < j && index > -1) { // Ordering conflict
          this.modules.splice(index, 1);
          j--;
          index = -1;
        }
        if (index == -1) {
          this.modules.splice(j++, 0, mod);
          if (sheet) for (let k = 0; k < mod.rules.length; k++)
            sheet.insertRule(mod.rules[k], pos++);
        } else {
          while (j < index) pos += this.modules[j++].rules.length;
          pos += mod.rules.length;
          j++;
        }
      }

      if (!sheet) {
        let text = "";
        for (let i = 0; i < this.modules.length; i++)
          text += this.modules[i].getRules() + "\n";
        this.styleTag.textContent = text;
      }
    }
  }

  // Style::Object<union<Style,string>>
  //
  // A style is an object that, in the simple case, maps CSS property
  // names to strings holding their values, as in `{color: "red",
  // fontWeight: "bold"}`. The property names can be given in
  // camel-case—the library will insert a dash before capital letters
  // when converting them to CSS.
  //
  // If you include an underscore in a property name, it and everything
  // after it will be removed from the output, which can be useful when
  // providing a property multiple times, for browser compatibility
  // reasons.
  //
  // A property in a style object can also be a sub-selector, which
  // extends the current context to add a pseudo-selector or a child
  // selector. Such a property should contain a `&` character, which
  // will be replaced by the current selector. For example `{"&:before":
  // {content: '"hi"'}}`. Sub-selectors and regular properties can
  // freely be mixed in a given object. Any property containing a `&` is
  // assumed to be a sub-selector.
  //
  // Finally, a property can specify an @-block to be wrapped around the
  // styles defined inside the object that's the property's value. For
  // example to create a media query you can do `{"@media screen and
  // (min-width: 400px)": {...}}`.

  var base = {
    8: "Backspace",
    9: "Tab",
    10: "Enter",
    12: "NumLock",
    13: "Enter",
    16: "Shift",
    17: "Control",
    18: "Alt",
    20: "CapsLock",
    27: "Escape",
    32: " ",
    33: "PageUp",
    34: "PageDown",
    35: "End",
    36: "Home",
    37: "ArrowLeft",
    38: "ArrowUp",
    39: "ArrowRight",
    40: "ArrowDown",
    44: "PrintScreen",
    45: "Insert",
    46: "Delete",
    59: ";",
    61: "=",
    91: "Meta",
    92: "Meta",
    106: "*",
    107: "+",
    108: ",",
    109: "-",
    110: ".",
    111: "/",
    144: "NumLock",
    145: "ScrollLock",
    160: "Shift",
    161: "Shift",
    162: "Control",
    163: "Control",
    164: "Alt",
    165: "Alt",
    173: "-",
    186: ";",
    187: "=",
    188: ",",
    189: "-",
    190: ".",
    191: "/",
    192: "`",
    219: "[",
    220: "\\",
    221: "]",
    222: "'"
  };

  var shift = {
    48: ")",
    49: "!",
    50: "@",
    51: "#",
    52: "$",
    53: "%",
    54: "^",
    55: "&",
    56: "*",
    57: "(",
    59: ":",
    61: "+",
    173: "_",
    186: ":",
    187: "+",
    188: "<",
    189: "_",
    190: ">",
    191: "?",
    192: "~",
    219: "{",
    220: "|",
    221: "}",
    222: "\""
  };

  var chrome$1 = typeof navigator != "undefined" && /Chrome\/(\d+)/.exec(navigator.userAgent);
  var safari$1 = typeof navigator != "undefined" && /Apple Computer/.test(navigator.vendor);
  var gecko$1 = typeof navigator != "undefined" && /Gecko\/\d+/.test(navigator.userAgent);
  var mac = typeof navigator != "undefined" && /Mac/.test(navigator.platform);
  var ie$1 = typeof navigator != "undefined" && /MSIE \d|Trident\/(?:[7-9]|\d{2,})\..*rv:(\d+)/.exec(navigator.userAgent);
  var brokenModifierNames = chrome$1 && (mac || +chrome$1[1] < 57) || gecko$1 && mac;

  // Fill in the digit keys
  for (var i = 0; i < 10; i++) base[48 + i] = base[96 + i] = String(i);

  // The function keys
  for (var i = 1; i <= 24; i++) base[i + 111] = "F" + i;

  // And the alphabetic keys
  for (var i = 65; i <= 90; i++) {
    base[i] = String.fromCharCode(i + 32);
    shift[i] = String.fromCharCode(i);
  }

  // For each code that doesn't have a shift-equivalent, copy the base name
  for (var code in base) if (!shift.hasOwnProperty(code)) shift[code] = base[code];

  function keyName(event) {
    var ignoreKey = brokenModifierNames && (event.ctrlKey || event.altKey || event.metaKey) ||
      (safari$1 || ie$1) && event.shiftKey && event.key && event.key.length == 1 ||
      event.key == "Unidentified";
    var name = (!ignoreKey && event.key) ||
      (event.shiftKey ? shift : base)[event.keyCode] ||
      event.key || "Unidentified";
    // Edge sometimes produces wrong names (Issue #3)
    if (name == "Esc") name = "Escape";
    if (name == "Del") name = "Delete";
    // https://developer.microsoft.com/en-us/microsoft-edge/platform/issues/8860571/
    if (name == "Left") name = "ArrowLeft";
    if (name == "Up") name = "ArrowUp";
    if (name == "Right") name = "ArrowRight";
    if (name == "Down") name = "ArrowDown";
    return name
  }

  function getSelection(root) {
      let target;
      // Browsers differ on whether shadow roots have a getSelection
      // method. If it exists, use that, otherwise, call it on the
      // document.
      if (root.nodeType == 11) { // Shadow root
          target = root.getSelection ? root : root.ownerDocument;
      }
      else {
          target = root;
      }
      return target.getSelection();
  }
  function contains(dom, node) {
      return node ? dom == node || dom.contains(node.nodeType != 1 ? node.parentNode : node) : false;
  }
  function deepActiveElement() {
      let elt = document.activeElement;
      while (elt && elt.shadowRoot)
          elt = elt.shadowRoot.activeElement;
      return elt;
  }
  function hasSelection(dom, selection) {
      if (!selection.anchorNode)
          return false;
      try {
          // Firefox will raise 'permission denied' errors when accessing
          // properties of `sel.anchorNode` when it's in a generated CSS
          // element.
          return contains(dom, selection.anchorNode);
      }
      catch (_) {
          return false;
      }
  }
  function clientRectsFor(dom) {
      if (dom.nodeType == 3)
          return textRange(dom, 0, dom.nodeValue.length).getClientRects();
      else if (dom.nodeType == 1)
          return dom.getClientRects();
      else
          return [];
  }
  // Scans forward and backward through DOM positions equivalent to the
  // given one to see if the two are in the same place (i.e. after a
  // text node vs at the end of that text node)
  function isEquivalentPosition(node, off, targetNode, targetOff) {
      return targetNode ? (scanFor(node, off, targetNode, targetOff, -1) ||
          scanFor(node, off, targetNode, targetOff, 1)) : false;
  }
  function domIndex(node) {
      for (var index = 0;; index++) {
          node = node.previousSibling;
          if (!node)
              return index;
      }
  }
  function scanFor(node, off, targetNode, targetOff, dir) {
      for (;;) {
          if (node == targetNode && off == targetOff)
              return true;
          if (off == (dir < 0 ? 0 : maxOffset(node))) {
              if (node.nodeName == "DIV")
                  return false;
              let parent = node.parentNode;
              if (!parent || parent.nodeType != 1)
                  return false;
              off = domIndex(node) + (dir < 0 ? 0 : 1);
              node = parent;
          }
          else if (node.nodeType == 1) {
              node = node.childNodes[off + (dir < 0 ? -1 : 0)];
              if (node.nodeType == 1 && node.contentEditable == "false")
                  return false;
              off = dir < 0 ? maxOffset(node) : 0;
          }
          else {
              return false;
          }
      }
  }
  function maxOffset(node) {
      return node.nodeType == 3 ? node.nodeValue.length : node.childNodes.length;
  }
  const Rect0 = { left: 0, right: 0, top: 0, bottom: 0 };
  function flattenRect(rect, left) {
      let x = left ? rect.left : rect.right;
      return { left: x, right: x, top: rect.top, bottom: rect.bottom };
  }
  function windowRect(win) {
      return { left: 0, right: win.innerWidth,
          top: 0, bottom: win.innerHeight };
  }
  function scrollRectIntoView(dom, rect, side, x, y, xMargin, yMargin, ltr) {
      let doc = dom.ownerDocument, win = doc.defaultView;
      for (let cur = dom; cur;) {
          if (cur.nodeType == 1) { // Element
              let bounding, top = cur == doc.body;
              if (top) {
                  bounding = windowRect(win);
              }
              else {
                  if (cur.scrollHeight <= cur.clientHeight && cur.scrollWidth <= cur.clientWidth) {
                      cur = cur.parentNode;
                      continue;
                  }
                  let rect = cur.getBoundingClientRect();
                  // Make sure scrollbar width isn't included in the rectangle
                  bounding = { left: rect.left, right: rect.left + cur.clientWidth,
                      top: rect.top, bottom: rect.top + cur.clientHeight };
              }
              let moveX = 0, moveY = 0;
              if (y == "nearest") {
                  if (rect.top < bounding.top) {
                      moveY = -(bounding.top - rect.top + yMargin);
                      if (side > 0 && rect.bottom > bounding.bottom + moveY)
                          moveY = rect.bottom - bounding.bottom + moveY + yMargin;
                  }
                  else if (rect.bottom > bounding.bottom) {
                      moveY = rect.bottom - bounding.bottom + yMargin;
                      if (side < 0 && (rect.top - moveY) < bounding.top)
                          moveY = -(bounding.top + moveY - rect.top + yMargin);
                  }
              }
              else {
                  let rectHeight = rect.bottom - rect.top, boundingHeight = bounding.bottom - bounding.top;
                  let targetTop = y == "center" && rectHeight <= boundingHeight ? rect.top + rectHeight / 2 - boundingHeight / 2 :
                      y == "start" || y == "center" && side < 0 ? rect.top - yMargin :
                          rect.bottom - boundingHeight + yMargin;
                  moveY = targetTop - bounding.top;
              }
              if (x == "nearest") {
                  if (rect.left < bounding.left) {
                      moveX = -(bounding.left - rect.left + xMargin);
                      if (side > 0 && rect.right > bounding.right + moveX)
                          moveX = rect.right - bounding.right + moveX + xMargin;
                  }
                  else if (rect.right > bounding.right) {
                      moveX = rect.right - bounding.right + xMargin;
                      if (side < 0 && rect.left < bounding.left + moveX)
                          moveX = -(bounding.left + moveX - rect.left + xMargin);
                  }
              }
              else {
                  let targetLeft = x == "center" ? rect.left + (rect.right - rect.left) / 2 - (bounding.right - bounding.left) / 2 :
                      (x == "start") == ltr ? rect.left - xMargin :
                          rect.right - (bounding.right - bounding.left) + xMargin;
                  moveX = targetLeft - bounding.left;
              }
              if (moveX || moveY) {
                  if (top) {
                      win.scrollBy(moveX, moveY);
                  }
                  else {
                      if (moveY) {
                          let start = cur.scrollTop;
                          cur.scrollTop += moveY;
                          moveY = cur.scrollTop - start;
                      }
                      if (moveX) {
                          let start = cur.scrollLeft;
                          cur.scrollLeft += moveX;
                          moveX = cur.scrollLeft - start;
                      }
                      rect = { left: rect.left - moveX, top: rect.top - moveY,
                          right: rect.right - moveX, bottom: rect.bottom - moveY };
                  }
              }
              if (top)
                  break;
              cur = cur.assignedSlot || cur.parentNode;
              x = y = "nearest";
          }
          else if (cur.nodeType == 11) { // A shadow root
              cur = cur.host;
          }
          else {
              break;
          }
      }
  }
  class DOMSelectionState {
      constructor() {
          this.anchorNode = null;
          this.anchorOffset = 0;
          this.focusNode = null;
          this.focusOffset = 0;
      }
      eq(domSel) {
          return this.anchorNode == domSel.anchorNode && this.anchorOffset == domSel.anchorOffset &&
              this.focusNode == domSel.focusNode && this.focusOffset == domSel.focusOffset;
      }
      setRange(range) {
          this.set(range.anchorNode, range.anchorOffset, range.focusNode, range.focusOffset);
      }
      set(anchorNode, anchorOffset, focusNode, focusOffset) {
          this.anchorNode = anchorNode;
          this.anchorOffset = anchorOffset;
          this.focusNode = focusNode;
          this.focusOffset = focusOffset;
      }
  }
  let preventScrollSupported = null;
  // Feature-detects support for .focus({preventScroll: true}), and uses
  // a fallback kludge when not supported.
  function focusPreventScroll(dom) {
      if (dom.setActive)
          return dom.setActive(); // in IE
      if (preventScrollSupported)
          return dom.focus(preventScrollSupported);
      let stack = [];
      for (let cur = dom; cur; cur = cur.parentNode) {
          stack.push(cur, cur.scrollTop, cur.scrollLeft);
          if (cur == cur.ownerDocument)
              break;
      }
      dom.focus(preventScrollSupported == null ? {
          get preventScroll() {
              preventScrollSupported = { preventScroll: true };
              return true;
          }
      } : undefined);
      if (!preventScrollSupported) {
          preventScrollSupported = false;
          for (let i = 0; i < stack.length;) {
              let elt = stack[i++], top = stack[i++], left = stack[i++];
              if (elt.scrollTop != top)
                  elt.scrollTop = top;
              if (elt.scrollLeft != left)
                  elt.scrollLeft = left;
          }
      }
  }
  let scratchRange;
  function textRange(node, from, to = from) {
      let range = scratchRange || (scratchRange = document.createRange());
      range.setEnd(node, to);
      range.setStart(node, from);
      return range;
  }
  function dispatchKey(elt, name, code) {
      let options = { key: name, code: name, keyCode: code, which: code, cancelable: true };
      let down = new KeyboardEvent("keydown", options);
      down.synthetic = true;
      elt.dispatchEvent(down);
      let up = new KeyboardEvent("keyup", options);
      up.synthetic = true;
      elt.dispatchEvent(up);
      return down.defaultPrevented || up.defaultPrevented;
  }
  function getRoot(node) {
      while (node) {
          if (node && (node.nodeType == 9 || node.nodeType == 11 && node.host))
              return node;
          node = node.assignedSlot || node.parentNode;
      }
      return null;
  }
  function clearAttributes(node) {
      while (node.attributes.length)
          node.removeAttributeNode(node.attributes[0]);
  }
  function atElementStart(doc, selection) {
      let node = selection.focusNode, offset = selection.focusOffset;
      if (!node || selection.anchorNode != node || selection.anchorOffset != offset)
          return false;
      for (;;) {
          if (offset) {
              if (node.nodeType != 1)
                  return false;
              let prev = node.childNodes[offset - 1];
              if (prev.contentEditable == "false")
                  offset--;
              else {
                  node = prev;
                  offset = maxOffset(node);
              }
          }
          else if (node == doc) {
              return true;
          }
          else {
              offset = domIndex(node);
              node = node.parentNode;
          }
      }
  }

  class DOMPos {
      constructor(node, offset, precise = true) {
          this.node = node;
          this.offset = offset;
          this.precise = precise;
      }
      static before(dom, precise) { return new DOMPos(dom.parentNode, domIndex(dom), precise); }
      static after(dom, precise) { return new DOMPos(dom.parentNode, domIndex(dom) + 1, precise); }
  }
  const noChildren = [];
  class ContentView {
      constructor() {
          this.parent = null;
          this.dom = null;
          this.dirty = 2 /* Node */;
      }
      get editorView() {
          if (!this.parent)
              throw new Error("Accessing view in orphan content view");
          return this.parent.editorView;
      }
      get overrideDOMText() { return null; }
      get posAtStart() {
          return this.parent ? this.parent.posBefore(this) : 0;
      }
      get posAtEnd() {
          return this.posAtStart + this.length;
      }
      posBefore(view) {
          let pos = this.posAtStart;
          for (let child of this.children) {
              if (child == view)
                  return pos;
              pos += child.length + child.breakAfter;
          }
          throw new RangeError("Invalid child in posBefore");
      }
      posAfter(view) {
          return this.posBefore(view) + view.length;
      }
      // Will return a rectangle directly before (when side < 0), after
      // (side > 0) or directly on (when the browser supports it) the
      // given position.
      coordsAt(_pos, _side) { return null; }
      sync(track) {
          if (this.dirty & 2 /* Node */) {
              let parent = this.dom;
              let prev = null, next;
              for (let child of this.children) {
                  if (child.dirty) {
                      if (!child.dom && (next = prev ? prev.nextSibling : parent.firstChild)) {
                          let contentView = ContentView.get(next);
                          if (!contentView || !contentView.parent && contentView.constructor == child.constructor)
                              child.reuseDOM(next);
                      }
                      child.sync(track);
                      child.dirty = 0 /* Not */;
                  }
                  next = prev ? prev.nextSibling : parent.firstChild;
                  if (track && !track.written && track.node == parent && next != child.dom)
                      track.written = true;
                  if (child.dom.parentNode == parent) {
                      while (next && next != child.dom)
                          next = rm$1(next);
                  }
                  else {
                      parent.insertBefore(child.dom, next);
                  }
                  prev = child.dom;
              }
              next = prev ? prev.nextSibling : parent.firstChild;
              if (next && track && track.node == parent)
                  track.written = true;
              while (next)
                  next = rm$1(next);
          }
          else if (this.dirty & 1 /* Child */) {
              for (let child of this.children)
                  if (child.dirty) {
                      child.sync(track);
                      child.dirty = 0 /* Not */;
                  }
          }
      }
      reuseDOM(_dom) { }
      localPosFromDOM(node, offset) {
          let after;
          if (node == this.dom) {
              after = this.dom.childNodes[offset];
          }
          else {
              let bias = maxOffset(node) == 0 ? 0 : offset == 0 ? -1 : 1;
              for (;;) {
                  let parent = node.parentNode;
                  if (parent == this.dom)
                      break;
                  if (bias == 0 && parent.firstChild != parent.lastChild) {
                      if (node == parent.firstChild)
                          bias = -1;
                      else
                          bias = 1;
                  }
                  node = parent;
              }
              if (bias < 0)
                  after = node;
              else
                  after = node.nextSibling;
          }
          if (after == this.dom.firstChild)
              return 0;
          while (after && !ContentView.get(after))
              after = after.nextSibling;
          if (!after)
              return this.length;
          for (let i = 0, pos = 0;; i++) {
              let child = this.children[i];
              if (child.dom == after)
                  return pos;
              pos += child.length + child.breakAfter;
          }
      }
      domBoundsAround(from, to, offset = 0) {
          let fromI = -1, fromStart = -1, toI = -1, toEnd = -1;
          for (let i = 0, pos = offset, prevEnd = offset; i < this.children.length; i++) {
              let child = this.children[i], end = pos + child.length;
              if (pos < from && end > to)
                  return child.domBoundsAround(from, to, pos);
              if (end >= from && fromI == -1) {
                  fromI = i;
                  fromStart = pos;
              }
              if (pos > to && child.dom.parentNode == this.dom) {
                  toI = i;
                  toEnd = prevEnd;
                  break;
              }
              prevEnd = end;
              pos = end + child.breakAfter;
          }
          return { from: fromStart, to: toEnd < 0 ? offset + this.length : toEnd,
              startDOM: (fromI ? this.children[fromI - 1].dom.nextSibling : null) || this.dom.firstChild,
              endDOM: toI < this.children.length && toI >= 0 ? this.children[toI].dom : null };
      }
      markDirty(andParent = false) {
          this.dirty |= 2 /* Node */;
          this.markParentsDirty(andParent);
      }
      markParentsDirty(childList) {
          for (let parent = this.parent; parent; parent = parent.parent) {
              if (childList)
                  parent.dirty |= 2 /* Node */;
              if (parent.dirty & 1 /* Child */)
                  return;
              parent.dirty |= 1 /* Child */;
              childList = false;
          }
      }
      setParent(parent) {
          if (this.parent != parent) {
              this.parent = parent;
              if (this.dirty)
                  this.markParentsDirty(true);
          }
      }
      setDOM(dom) {
          if (this.dom)
              this.dom.cmView = null;
          this.dom = dom;
          dom.cmView = this;
      }
      get rootView() {
          for (let v = this;;) {
              let parent = v.parent;
              if (!parent)
                  return v;
              v = parent;
          }
      }
      replaceChildren(from, to, children = noChildren) {
          this.markDirty();
          for (let i = from; i < to; i++) {
              let child = this.children[i];
              if (child.parent == this)
                  child.destroy();
          }
          this.children.splice(from, to - from, ...children);
          for (let i = 0; i < children.length; i++)
              children[i].setParent(this);
      }
      ignoreMutation(_rec) { return false; }
      ignoreEvent(_event) { return false; }
      childCursor(pos = this.length) {
          return new ChildCursor(this.children, pos, this.children.length);
      }
      childPos(pos, bias = 1) {
          return this.childCursor().findPos(pos, bias);
      }
      toString() {
          let name = this.constructor.name.replace("View", "");
          return name + (this.children.length ? "(" + this.children.join() + ")" :
              this.length ? "[" + (name == "Text" ? this.text : this.length) + "]" : "") +
              (this.breakAfter ? "#" : "");
      }
      static get(node) { return node.cmView; }
      get isEditable() { return true; }
      merge(from, to, source, hasStart, openStart, openEnd) {
          return false;
      }
      become(other) { return false; }
      // When this is a zero-length view with a side, this should return a
      // number <= 0 to indicate it is before its position, or a
      // number > 0 when after its position.
      getSide() { return 0; }
      destroy() {
          this.parent = null;
      }
  }
  ContentView.prototype.breakAfter = 0;
  // Remove a DOM node and return its next sibling.
  function rm$1(dom) {
      let next = dom.nextSibling;
      dom.parentNode.removeChild(dom);
      return next;
  }
  class ChildCursor {
      constructor(children, pos, i) {
          this.children = children;
          this.pos = pos;
          this.i = i;
          this.off = 0;
      }
      findPos(pos, bias = 1) {
          for (;;) {
              if (pos > this.pos || pos == this.pos &&
                  (bias > 0 || this.i == 0 || this.children[this.i - 1].breakAfter)) {
                  this.off = pos - this.pos;
                  return this;
              }
              let next = this.children[--this.i];
              this.pos -= next.length + next.breakAfter;
          }
      }
  }
  function replaceRange(parent, fromI, fromOff, toI, toOff, insert, breakAtStart, openStart, openEnd) {
      let { children } = parent;
      let before = children.length ? children[fromI] : null;
      let last = insert.length ? insert[insert.length - 1] : null;
      let breakAtEnd = last ? last.breakAfter : breakAtStart;
      // Change within a single child
      if (fromI == toI && before && !breakAtStart && !breakAtEnd && insert.length < 2 &&
          before.merge(fromOff, toOff, insert.length ? last : null, fromOff == 0, openStart, openEnd))
          return;
      if (toI < children.length) {
          let after = children[toI];
          // Make sure the end of the child after the update is preserved in `after`
          if (after && toOff < after.length) {
              // If we're splitting a child, separate part of it to avoid that
              // being mangled when updating the child before the update.
              if (fromI == toI) {
                  after = after.split(toOff);
                  toOff = 0;
              }
              // If the element after the replacement should be merged with
              // the last replacing element, update `content`
              if (!breakAtEnd && last && after.merge(0, toOff, last, true, 0, openEnd)) {
                  insert[insert.length - 1] = after;
              }
              else {
                  // Remove the start of the after element, if necessary, and
                  // add it to `content`.
                  if (toOff)
                      after.merge(0, toOff, null, false, 0, openEnd);
                  insert.push(after);
              }
          }
          else if (after === null || after === void 0 ? void 0 : after.breakAfter) {
              // The element at `toI` is entirely covered by this range.
              // Preserve its line break, if any.
              if (last)
                  last.breakAfter = 1;
              else
                  breakAtStart = 1;
          }
          // Since we've handled the next element from the current elements
          // now, make sure `toI` points after that.
          toI++;
      }
      if (before) {
          before.breakAfter = breakAtStart;
          if (fromOff > 0) {
              if (!breakAtStart && insert.length && before.merge(fromOff, before.length, insert[0], false, openStart, 0)) {
                  before.breakAfter = insert.shift().breakAfter;
              }
              else if (fromOff < before.length || before.children.length && before.children[before.children.length - 1].length == 0) {
                  before.merge(fromOff, before.length, null, false, openStart, 0);
              }
              fromI++;
          }
      }
      // Try to merge widgets on the boundaries of the replacement
      while (fromI < toI && insert.length) {
          if (children[toI - 1].become(insert[insert.length - 1])) {
              toI--;
              insert.pop();
              openEnd = insert.length ? 0 : openStart;
          }
          else if (children[fromI].become(insert[0])) {
              fromI++;
              insert.shift();
              openStart = insert.length ? 0 : openEnd;
          }
          else {
              break;
          }
      }
      if (!insert.length && fromI && toI < children.length && !children[fromI - 1].breakAfter &&
          children[toI].merge(0, 0, children[fromI - 1], false, openStart, openEnd))
          fromI--;
      if (fromI < toI || insert.length)
          parent.replaceChildren(fromI, toI, insert);
  }
  function mergeChildrenInto(parent, from, to, insert, openStart, openEnd) {
      let cur = parent.childCursor();
      let { i: toI, off: toOff } = cur.findPos(to, 1);
      let { i: fromI, off: fromOff } = cur.findPos(from, -1);
      let dLen = from - to;
      for (let view of insert)
          dLen += view.length;
      parent.length += dLen;
      replaceRange(parent, fromI, fromOff, toI, toOff, insert, 0, openStart, openEnd);
  }

  let nav = typeof navigator != "undefined" ? navigator : { userAgent: "", vendor: "", platform: "" };
  let doc = typeof document != "undefined" ? document : { documentElement: { style: {} } };
  const ie_edge = /*@__PURE__*//Edge\/(\d+)/.exec(nav.userAgent);
  const ie_upto10 = /*@__PURE__*//MSIE \d/.test(nav.userAgent);
  const ie_11up = /*@__PURE__*//Trident\/(?:[7-9]|\d{2,})\..*rv:(\d+)/.exec(nav.userAgent);
  const ie = !!(ie_upto10 || ie_11up || ie_edge);
  const gecko = !ie && /*@__PURE__*//gecko\/(\d+)/i.test(nav.userAgent);
  const chrome = !ie && /*@__PURE__*//Chrome\/(\d+)/.exec(nav.userAgent);
  const webkit = "webkitFontSmoothing" in doc.documentElement.style;
  const safari = !ie && /*@__PURE__*//Apple Computer/.test(nav.vendor);
  const ios = safari && (/*@__PURE__*//Mobile\/\w+/.test(nav.userAgent) || nav.maxTouchPoints > 2);
  var browser = {
      mac: ios || /*@__PURE__*//Mac/.test(nav.platform),
      windows: /*@__PURE__*//Win/.test(nav.platform),
      linux: /*@__PURE__*//Linux|X11/.test(nav.platform),
      ie,
      ie_version: ie_upto10 ? doc.documentMode || 6 : ie_11up ? +ie_11up[1] : ie_edge ? +ie_edge[1] : 0,
      gecko,
      gecko_version: gecko ? +(/*@__PURE__*//Firefox\/(\d+)/.exec(nav.userAgent) || [0, 0])[1] : 0,
      chrome: !!chrome,
      chrome_version: chrome ? +chrome[1] : 0,
      ios,
      android: /*@__PURE__*//Android\b/.test(nav.userAgent),
      webkit,
      safari,
      webkit_version: webkit ? +(/*@__PURE__*//\bAppleWebKit\/(\d+)/.exec(navigator.userAgent) || [0, 0])[1] : 0,
      tabSize: doc.documentElement.style.tabSize != null ? "tab-size" : "-moz-tab-size"
  };

  const MaxJoinLen = 256;
  class TextView extends ContentView {
      constructor(text) {
          super();
          this.text = text;
      }
      get length() { return this.text.length; }
      createDOM(textDOM) {
          this.setDOM(textDOM || document.createTextNode(this.text));
      }
      sync(track) {
          if (!this.dom)
              this.createDOM();
          if (this.dom.nodeValue != this.text) {
              if (track && track.node == this.dom)
                  track.written = true;
              this.dom.nodeValue = this.text;
          }
      }
      reuseDOM(dom) {
          if (dom.nodeType == 3)
              this.createDOM(dom);
      }
      merge(from, to, source) {
          if (source && (!(source instanceof TextView) || this.length - (to - from) + source.length > MaxJoinLen))
              return false;
          this.text = this.text.slice(0, from) + (source ? source.text : "") + this.text.slice(to);
          this.markDirty();
          return true;
      }
      split(from) {
          let result = new TextView(this.text.slice(from));
          this.text = this.text.slice(0, from);
          this.markDirty();
          return result;
      }
      localPosFromDOM(node, offset) {
          return node == this.dom ? offset : offset ? this.text.length : 0;
      }
      domAtPos(pos) { return new DOMPos(this.dom, pos); }
      domBoundsAround(_from, _to, offset) {
          return { from: offset, to: offset + this.length, startDOM: this.dom, endDOM: this.dom.nextSibling };
      }
      coordsAt(pos, side) {
          return textCoords(this.dom, pos, side);
      }
  }
  class MarkView extends ContentView {
      constructor(mark, children = [], length = 0) {
          super();
          this.mark = mark;
          this.children = children;
          this.length = length;
          for (let ch of children)
              ch.setParent(this);
      }
      setAttrs(dom) {
          clearAttributes(dom);
          if (this.mark.class)
              dom.className = this.mark.class;
          if (this.mark.attrs)
              for (let name in this.mark.attrs)
                  dom.setAttribute(name, this.mark.attrs[name]);
          return dom;
      }
      reuseDOM(node) {
          if (node.nodeName == this.mark.tagName.toUpperCase()) {
              this.setDOM(node);
              this.dirty |= 4 /* Attrs */ | 2 /* Node */;
          }
      }
      sync(track) {
          if (!this.dom)
              this.setDOM(this.setAttrs(document.createElement(this.mark.tagName)));
          else if (this.dirty & 4 /* Attrs */)
              this.setAttrs(this.dom);
          super.sync(track);
      }
      merge(from, to, source, _hasStart, openStart, openEnd) {
          if (source && (!(source instanceof MarkView && source.mark.eq(this.mark)) ||
              (from && openStart <= 0) || (to < this.length && openEnd <= 0)))
              return false;
          mergeChildrenInto(this, from, to, source ? source.children : [], openStart - 1, openEnd - 1);
          this.markDirty();
          return true;
      }
      split(from) {
          let result = [], off = 0, detachFrom = -1, i = 0;
          for (let elt of this.children) {
              let end = off + elt.length;
              if (end > from)
                  result.push(off < from ? elt.split(from - off) : elt);
              if (detachFrom < 0 && off >= from)
                  detachFrom = i;
              off = end;
              i++;
          }
          let length = this.length - from;
          this.length = from;
          if (detachFrom > -1) {
              this.children.length = detachFrom;
              this.markDirty();
          }
          return new MarkView(this.mark, result, length);
      }
      domAtPos(pos) {
          return inlineDOMAtPos(this.dom, this.children, pos);
      }
      coordsAt(pos, side) {
          return coordsInChildren(this, pos, side);
      }
  }
  function textCoords(text, pos, side) {
      let length = text.nodeValue.length;
      if (pos > length)
          pos = length;
      let from = pos, to = pos, flatten = 0;
      if (pos == 0 && side < 0 || pos == length && side >= 0) {
          if (!(browser.chrome || browser.gecko)) { // These browsers reliably return valid rectangles for empty ranges
              if (pos) {
                  from--;
                  flatten = 1;
              } // FIXME this is wrong in RTL text
              else if (to < length) {
                  to++;
                  flatten = -1;
              }
          }
      }
      else {
          if (side < 0)
              from--;
          else if (to < length)
              to++;
      }
      let rects = textRange(text, from, to).getClientRects();
      if (!rects.length)
          return Rect0;
      let rect = rects[(flatten ? flatten < 0 : side >= 0) ? 0 : rects.length - 1];
      if (browser.safari && !flatten && rect.width == 0)
          rect = Array.prototype.find.call(rects, r => r.width) || rect;
      return flatten ? flattenRect(rect, flatten < 0) : rect || null;
  }
  // Also used for collapsed ranges that don't have a placeholder widget!
  class WidgetView extends ContentView {
      constructor(widget, length, side) {
          super();
          this.widget = widget;
          this.length = length;
          this.side = side;
          this.prevWidget = null;
      }
      static create(widget, length, side) {
          return new (widget.customView || WidgetView)(widget, length, side);
      }
      split(from) {
          let result = WidgetView.create(this.widget, this.length - from, this.side);
          this.length -= from;
          return result;
      }
      sync() {
          if (!this.dom || !this.widget.updateDOM(this.dom)) {
              if (this.dom && this.prevWidget)
                  this.prevWidget.destroy(this.dom);
              this.prevWidget = null;
              this.setDOM(this.widget.toDOM(this.editorView));
              this.dom.contentEditable = "false";
          }
      }
      getSide() { return this.side; }
      merge(from, to, source, hasStart, openStart, openEnd) {
          if (source && (!(source instanceof WidgetView) || !this.widget.compare(source.widget) ||
              from > 0 && openStart <= 0 || to < this.length && openEnd <= 0))
              return false;
          this.length = from + (source ? source.length : 0) + (this.length - to);
          return true;
      }
      become(other) {
          if (other.length == this.length && other instanceof WidgetView && other.side == this.side) {
              if (this.widget.constructor == other.widget.constructor) {
                  if (!this.widget.eq(other.widget))
                      this.markDirty(true);
                  if (this.dom && !this.prevWidget)
                      this.prevWidget = this.widget;
                  this.widget = other.widget;
                  return true;
              }
          }
          return false;
      }
      ignoreMutation() { return true; }
      ignoreEvent(event) { return this.widget.ignoreEvent(event); }
      get overrideDOMText() {
          if (this.length == 0)
              return Text.empty;
          let top = this;
          while (top.parent)
              top = top.parent;
          let view = top.editorView, text = view && view.state.doc, start = this.posAtStart;
          return text ? text.slice(start, start + this.length) : Text.empty;
      }
      domAtPos(pos) {
          return pos == 0 ? DOMPos.before(this.dom) : DOMPos.after(this.dom, pos == this.length);
      }
      domBoundsAround() { return null; }
      coordsAt(pos, side) {
          let rects = this.dom.getClientRects(), rect = null;
          if (!rects.length)
              return Rect0;
          for (let i = pos > 0 ? rects.length - 1 : 0;; i += (pos > 0 ? -1 : 1)) {
              rect = rects[i];
              if (pos > 0 ? i == 0 : i == rects.length - 1 || rect.top < rect.bottom)
                  break;
          }
          return (pos == 0 && side > 0 || pos == this.length && side <= 0) ? rect : flattenRect(rect, pos == 0);
      }
      get isEditable() { return false; }
      destroy() {
          super.destroy();
          if (this.dom)
              this.widget.destroy(this.dom);
      }
  }
  class CompositionView extends WidgetView {
      domAtPos(pos) {
          let { topView, text } = this.widget;
          if (!topView)
              return new DOMPos(text, Math.min(pos, text.nodeValue.length));
          return scanCompositionTree(pos, 0, topView, text, (v, p) => v.domAtPos(p), p => new DOMPos(text, Math.min(p, text.nodeValue.length)));
      }
      sync() { this.setDOM(this.widget.toDOM()); }
      localPosFromDOM(node, offset) {
          let { topView, text } = this.widget;
          if (!topView)
              return Math.min(offset, this.length);
          return posFromDOMInCompositionTree(node, offset, topView, text);
      }
      ignoreMutation() { return false; }
      get overrideDOMText() { return null; }
      coordsAt(pos, side) {
          let { topView, text } = this.widget;
          if (!topView)
              return textCoords(text, pos, side);
          return scanCompositionTree(pos, side, topView, text, (v, pos, side) => v.coordsAt(pos, side), (pos, side) => textCoords(text, pos, side));
      }
      destroy() {
          var _a;
          super.destroy();
          (_a = this.widget.topView) === null || _a === void 0 ? void 0 : _a.destroy();
      }
      get isEditable() { return true; }
  }
  // Uses the old structure of a chunk of content view frozen for
  // composition to try and find a reasonable DOM location for the given
  // offset.
  function scanCompositionTree(pos, side, view, text, enterView, fromText) {
      if (view instanceof MarkView) {
          for (let child of view.children) {
              let hasComp = contains(child.dom, text);
              let len = hasComp ? text.nodeValue.length : child.length;
              if (pos < len || pos == len && child.getSide() <= 0)
                  return hasComp ? scanCompositionTree(pos, side, child, text, enterView, fromText) : enterView(child, pos, side);
              pos -= len;
          }
          return enterView(view, view.length, -1);
      }
      else if (view.dom == text) {
          return fromText(pos, side);
      }
      else {
          return enterView(view, pos, side);
      }
  }
  function posFromDOMInCompositionTree(node, offset, view, text) {
      if (view instanceof MarkView) {
          for (let child of view.children) {
              let pos = 0, hasComp = contains(child.dom, text);
              if (contains(child.dom, node))
                  return pos + (hasComp ? posFromDOMInCompositionTree(node, offset, child, text) : child.localPosFromDOM(node, offset));
              pos += hasComp ? text.nodeValue.length : child.length;
          }
      }
      else if (view.dom == text) {
          return Math.min(offset, text.nodeValue.length);
      }
      return view.localPosFromDOM(node, offset);
  }
  // These are drawn around uneditable widgets to avoid a number of
  // browser bugs that show up when the cursor is directly next to
  // uneditable inline content.
  class WidgetBufferView extends ContentView {
      constructor(side) {
          super();
          this.side = side;
      }
      get length() { return 0; }
      merge() { return false; }
      become(other) {
          return other instanceof WidgetBufferView && other.side == this.side;
      }
      split() { return new WidgetBufferView(this.side); }
      sync() {
          if (!this.dom) {
              let dom = document.createElement("img");
              dom.className = "cm-widgetBuffer";
              dom.setAttribute("aria-hidden", "true");
              this.setDOM(dom);
          }
      }
      getSide() { return this.side; }
      domAtPos(pos) { return DOMPos.before(this.dom); }
      localPosFromDOM() { return 0; }
      domBoundsAround() { return null; }
      coordsAt(pos) {
          let imgRect = this.dom.getBoundingClientRect();
          // Since the <img> height doesn't correspond to text height, try
          // to borrow the height from some sibling node.
          let siblingRect = inlineSiblingRect(this, this.side > 0 ? -1 : 1);
          return siblingRect && siblingRect.top < imgRect.bottom && siblingRect.bottom > imgRect.top
              ? { left: imgRect.left, right: imgRect.right, top: siblingRect.top, bottom: siblingRect.bottom } : imgRect;
      }
      get overrideDOMText() {
          return Text.empty;
      }
  }
  TextView.prototype.children = WidgetView.prototype.children = WidgetBufferView.prototype.children = noChildren;
  function inlineSiblingRect(view, side) {
      let parent = view.parent, index = parent ? parent.children.indexOf(view) : -1;
      while (parent && index >= 0) {
          if (side < 0 ? index > 0 : index < parent.children.length) {
              let next = parent.children[index + side];
              if (next instanceof TextView) {
                  let nextRect = next.coordsAt(side < 0 ? next.length : 0, side);
                  if (nextRect)
                      return nextRect;
              }
              index += side;
          }
          else if (parent instanceof MarkView && parent.parent) {
              index = parent.parent.children.indexOf(parent) + (side < 0 ? 0 : 1);
              parent = parent.parent;
          }
          else {
              let last = parent.dom.lastChild;
              if (last && last.nodeName == "BR")
                  return last.getClientRects()[0];
              break;
          }
      }
      return undefined;
  }
  function inlineDOMAtPos(dom, children, pos) {
      let i = 0;
      for (let off = 0; i < children.length; i++) {
          let child = children[i], end = off + child.length;
          if (end == off && child.getSide() <= 0)
              continue;
          if (pos > off && pos < end && child.dom.parentNode == dom)
              return child.domAtPos(pos - off);
          if (pos <= off)
              break;
          off = end;
      }
      for (; i > 0; i--) {
          let before = children[i - 1].dom;
          if (before.parentNode == dom)
              return DOMPos.after(before);
      }
      return new DOMPos(dom, 0);
  }
  // Assumes `view`, if a mark view, has precisely 1 child.
  function joinInlineInto(parent, view, open) {
      let last, { children } = parent;
      if (open > 0 && view instanceof MarkView && children.length &&
          (last = children[children.length - 1]) instanceof MarkView && last.mark.eq(view.mark)) {
          joinInlineInto(last, view.children[0], open - 1);
      }
      else {
          children.push(view);
          view.setParent(parent);
      }
      parent.length += view.length;
  }
  function coordsInChildren(view, pos, side) {
      for (let off = 0, i = 0; i < view.children.length; i++) {
          let child = view.children[i], end = off + child.length, next;
          if ((side <= 0 || end == view.length || child.getSide() > 0 ? end >= pos : end > pos) &&
              (pos < end || i + 1 == view.children.length || (next = view.children[i + 1]).length || next.getSide() > 0)) {
              let flatten = 0;
              if (end == off) {
                  if (child.getSide() <= 0)
                      continue;
                  flatten = side = -child.getSide();
              }
              let rect = child.coordsAt(Math.max(0, pos - off), side);
              return flatten && rect ? flattenRect(rect, side < 0) : rect;
          }
          off = end;
      }
      let last = view.dom.lastChild;
      if (!last)
          return view.dom.getBoundingClientRect();
      let rects = clientRectsFor(last);
      return rects[rects.length - 1] || null;
  }

  function combineAttrs(source, target) {
      for (let name in source) {
          if (name == "class" && target.class)
              target.class += " " + source.class;
          else if (name == "style" && target.style)
              target.style += ";" + source.style;
          else
              target[name] = source[name];
      }
      return target;
  }
  function attrsEq(a, b) {
      if (a == b)
          return true;
      if (!a || !b)
          return false;
      let keysA = Object.keys(a), keysB = Object.keys(b);
      if (keysA.length != keysB.length)
          return false;
      for (let key of keysA) {
          if (keysB.indexOf(key) == -1 || a[key] !== b[key])
              return false;
      }
      return true;
  }
  function updateAttrs(dom, prev, attrs) {
      let changed = null;
      if (prev)
          for (let name in prev)
              if (!(attrs && name in attrs))
                  dom.removeAttribute(changed = name);
      if (attrs)
          for (let name in attrs)
              if (!(prev && prev[name] == attrs[name]))
                  dom.setAttribute(changed = name, attrs[name]);
      return !!changed;
  }

  /**
  Widgets added to the content are described by subclasses of this
  class. Using a description object like that makes it possible to
  delay creating of the DOM structure for a widget until it is
  needed, and to avoid redrawing widgets even if the decorations
  that define them are recreated.
  */
  class WidgetType {
      /**
      Compare this instance to another instance of the same type.
      (TypeScript can't express this, but only instances of the same
      specific class will be passed to this method.) This is used to
      avoid redrawing widgets when they are replaced by a new
      decoration of the same type. The default implementation just
      returns `false`, which will cause new instances of the widget to
      always be redrawn.
      */
      eq(widget) { return false; }
      /**
      Update a DOM element created by a widget of the same type (but
      different, non-`eq` content) to reflect this widget. May return
      true to indicate that it could update, false to indicate it
      couldn't (in which case the widget will be redrawn). The default
      implementation just returns false.
      */
      updateDOM(dom) { return false; }
      /**
      @internal
      */
      compare(other) {
          return this == other || this.constructor == other.constructor && this.eq(other);
      }
      /**
      The estimated height this widget will have, to be used when
      estimating the height of content that hasn't been drawn. May
      return -1 to indicate you don't know. The default implementation
      returns -1.
      */
      get estimatedHeight() { return -1; }
      /**
      Can be used to configure which kinds of events inside the widget
      should be ignored by the editor. The default is to ignore all
      events.
      */
      ignoreEvent(event) { return true; }
      /**
      @internal
      */
      get customView() { return null; }
      /**
      This is called when the an instance of the widget is removed
      from the editor view.
      */
      destroy(dom) { }
  }
  /**
  The different types of blocks that can occur in an editor view.
  */
  var BlockType = /*@__PURE__*/(function (BlockType) {
      /**
      A line of text.
      */
      BlockType[BlockType["Text"] = 0] = "Text";
      /**
      A block widget associated with the position after it.
      */
      BlockType[BlockType["WidgetBefore"] = 1] = "WidgetBefore";
      /**
      A block widget associated with the position before it.
      */
      BlockType[BlockType["WidgetAfter"] = 2] = "WidgetAfter";
      /**
      A block widget [replacing](https://codemirror.net/6/docs/ref/#view.Decoration^replace) a range of content.
      */
      BlockType[BlockType["WidgetRange"] = 3] = "WidgetRange";
  return BlockType})(BlockType || (BlockType = {}));
  /**
  A decoration provides information on how to draw or style a piece
  of content. You'll usually use it wrapped in a
  [`Range`](https://codemirror.net/6/docs/ref/#state.Range), which adds a start and end position.
  @nonabstract
  */
  class Decoration extends RangeValue {
      constructor(
      /**
      @internal
      */
      startSide, 
      /**
      @internal
      */
      endSide, 
      /**
      @internal
      */
      widget, 
      /**
      The config object used to create this decoration. You can
      include additional properties in there to store metadata about
      your decoration.
      */
      spec) {
          super();
          this.startSide = startSide;
          this.endSide = endSide;
          this.widget = widget;
          this.spec = spec;
      }
      /**
      @internal
      */
      get heightRelevant() { return false; }
      /**
      Create a mark decoration, which influences the styling of the
      content in its range. Nested mark decorations will cause nested
      DOM elements to be created. Nesting order is determined by
      precedence of the [facet](https://codemirror.net/6/docs/ref/#view.EditorView^decorations), with
      the higher-precedence decorations creating the inner DOM nodes.
      Such elements are split on line boundaries and on the boundaries
      of lower-precedence decorations.
      */
      static mark(spec) {
          return new MarkDecoration(spec);
      }
      /**
      Create a widget decoration, which displays a DOM element at the
      given position.
      */
      static widget(spec) {
          let side = spec.side || 0, block = !!spec.block;
          side += block ? (side > 0 ? 300000000 /* BlockAfter */ : -400000000 /* BlockBefore */) : (side > 0 ? 100000000 /* InlineAfter */ : -100000000 /* InlineBefore */);
          return new PointDecoration(spec, side, side, block, spec.widget || null, false);
      }
      /**
      Create a replace decoration which replaces the given range with
      a widget, or simply hides it.
      */
      static replace(spec) {
          let block = !!spec.block, startSide, endSide;
          if (spec.isBlockGap) {
              startSide = -500000000 /* GapStart */;
              endSide = 400000000 /* GapEnd */;
          }
          else {
              let { start, end } = getInclusive(spec, block);
              startSide = (start ? (block ? -300000000 /* BlockIncStart */ : -1 /* InlineIncStart */) : 500000000 /* NonIncStart */) - 1;
              endSide = (end ? (block ? 200000000 /* BlockIncEnd */ : 1 /* InlineIncEnd */) : -600000000 /* NonIncEnd */) + 1;
          }
          return new PointDecoration(spec, startSide, endSide, block, spec.widget || null, true);
      }
      /**
      Create a line decoration, which can add DOM attributes to the
      line starting at the given position.
      */
      static line(spec) {
          return new LineDecoration(spec);
      }
      /**
      Build a [`DecorationSet`](https://codemirror.net/6/docs/ref/#view.DecorationSet) from the given
      decorated range or ranges. If the ranges aren't already sorted,
      pass `true` for `sort` to make the library sort them for you.
      */
      static set(of, sort = false) {
          return RangeSet.of(of, sort);
      }
      /**
      @internal
      */
      hasHeight() { return this.widget ? this.widget.estimatedHeight > -1 : false; }
  }
  /**
  The empty set of decorations.
  */
  Decoration.none = RangeSet.empty;
  class MarkDecoration extends Decoration {
      constructor(spec) {
          let { start, end } = getInclusive(spec);
          super(start ? -1 /* InlineIncStart */ : 500000000 /* NonIncStart */, end ? 1 /* InlineIncEnd */ : -600000000 /* NonIncEnd */, null, spec);
          this.tagName = spec.tagName || "span";
          this.class = spec.class || "";
          this.attrs = spec.attributes || null;
      }
      eq(other) {
          return this == other ||
              other instanceof MarkDecoration &&
                  this.tagName == other.tagName &&
                  this.class == other.class &&
                  attrsEq(this.attrs, other.attrs);
      }
      range(from, to = from) {
          if (from >= to)
              throw new RangeError("Mark decorations may not be empty");
          return super.range(from, to);
      }
  }
  MarkDecoration.prototype.point = false;
  class LineDecoration extends Decoration {
      constructor(spec) {
          super(-200000000 /* Line */, -200000000 /* Line */, null, spec);
      }
      eq(other) {
          return other instanceof LineDecoration && attrsEq(this.spec.attributes, other.spec.attributes);
      }
      range(from, to = from) {
          if (to != from)
              throw new RangeError("Line decoration ranges must be zero-length");
          return super.range(from, to);
      }
  }
  LineDecoration.prototype.mapMode = MapMode.TrackBefore;
  LineDecoration.prototype.point = true;
  class PointDecoration extends Decoration {
      constructor(spec, startSide, endSide, block, widget, isReplace) {
          super(startSide, endSide, widget, spec);
          this.block = block;
          this.isReplace = isReplace;
          this.mapMode = !block ? MapMode.TrackDel : startSide <= 0 ? MapMode.TrackBefore : MapMode.TrackAfter;
      }
      // Only relevant when this.block == true
      get type() {
          return this.startSide < this.endSide ? BlockType.WidgetRange
              : this.startSide <= 0 ? BlockType.WidgetBefore : BlockType.WidgetAfter;
      }
      get heightRelevant() { return this.block || !!this.widget && this.widget.estimatedHeight >= 5; }
      eq(other) {
          return other instanceof PointDecoration &&
              widgetsEq(this.widget, other.widget) &&
              this.block == other.block &&
              this.startSide == other.startSide && this.endSide == other.endSide;
      }
      range(from, to = from) {
          if (this.isReplace && (from > to || (from == to && this.startSide > 0 && this.endSide <= 0)))
              throw new RangeError("Invalid range for replacement decoration");
          if (!this.isReplace && to != from)
              throw new RangeError("Widget decorations can only have zero-length ranges");
          return super.range(from, to);
      }
  }
  PointDecoration.prototype.point = true;
  function getInclusive(spec, block = false) {
      let { inclusiveStart: start, inclusiveEnd: end } = spec;
      if (start == null)
          start = spec.inclusive;
      if (end == null)
          end = spec.inclusive;
      return { start: start !== null && start !== void 0 ? start : block, end: end !== null && end !== void 0 ? end : block };
  }
  function widgetsEq(a, b) {
      return a == b || !!(a && b && a.compare(b));
  }
  function addRange(from, to, ranges, margin = 0) {
      let last = ranges.length - 1;
      if (last >= 0 && ranges[last] + margin >= from)
          ranges[last] = Math.max(ranges[last], to);
      else
          ranges.push(from, to);
  }

  class LineView extends ContentView {
      constructor() {
          super(...arguments);
          this.children = [];
          this.length = 0;
          this.prevAttrs = undefined;
          this.attrs = null;
          this.breakAfter = 0;
      }
      // Consumes source
      merge(from, to, source, hasStart, openStart, openEnd) {
          if (source) {
              if (!(source instanceof LineView))
                  return false;
              if (!this.dom)
                  source.transferDOM(this); // Reuse source.dom when appropriate
          }
          if (hasStart)
              this.setDeco(source ? source.attrs : null);
          mergeChildrenInto(this, from, to, source ? source.children : [], openStart, openEnd);
          return true;
      }
      split(at) {
          let end = new LineView;
          end.breakAfter = this.breakAfter;
          if (this.length == 0)
              return end;
          let { i, off } = this.childPos(at);
          if (off) {
              end.append(this.children[i].split(off), 0);
              this.children[i].merge(off, this.children[i].length, null, false, 0, 0);
              i++;
          }
          for (let j = i; j < this.children.length; j++)
              end.append(this.children[j], 0);
          while (i > 0 && this.children[i - 1].length == 0)
              this.children[--i].destroy();
          this.children.length = i;
          this.markDirty();
          this.length = at;
          return end;
      }
      transferDOM(other) {
          if (!this.dom)
              return;
          this.markDirty();
          other.setDOM(this.dom);
          other.prevAttrs = this.prevAttrs === undefined ? this.attrs : this.prevAttrs;
          this.prevAttrs = undefined;
          this.dom = null;
      }
      setDeco(attrs) {
          if (!attrsEq(this.attrs, attrs)) {
              if (this.dom) {
                  this.prevAttrs = this.attrs;
                  this.markDirty();
              }
              this.attrs = attrs;
          }
      }
      append(child, openStart) {
          joinInlineInto(this, child, openStart);
      }
      // Only called when building a line view in ContentBuilder
      addLineDeco(deco) {
          let attrs = deco.spec.attributes, cls = deco.spec.class;
          if (attrs)
              this.attrs = combineAttrs(attrs, this.attrs || {});
          if (cls)
              this.attrs = combineAttrs({ class: cls }, this.attrs || {});
      }
      domAtPos(pos) {
          return inlineDOMAtPos(this.dom, this.children, pos);
      }
      reuseDOM(node) {
          if (node.nodeName == "DIV") {
              this.setDOM(node);
              this.dirty |= 4 /* Attrs */ | 2 /* Node */;
          }
      }
      sync(track) {
          var _a;
          if (!this.dom) {
              this.setDOM(document.createElement("div"));
              this.dom.className = "cm-line";
              this.prevAttrs = this.attrs ? null : undefined;
          }
          else if (this.dirty & 4 /* Attrs */) {
              clearAttributes(this.dom);
              this.dom.className = "cm-line";
              this.prevAttrs = this.attrs ? null : undefined;
          }
          if (this.prevAttrs !== undefined) {
              updateAttrs(this.dom, this.prevAttrs, this.attrs);
              this.dom.classList.add("cm-line");
              this.prevAttrs = undefined;
          }
          super.sync(track);
          let last = this.dom.lastChild;
          while (last && ContentView.get(last) instanceof MarkView)
              last = last.lastChild;
          if (!last || !this.length ||
              last.nodeName != "BR" && ((_a = ContentView.get(last)) === null || _a === void 0 ? void 0 : _a.isEditable) == false &&
                  (!browser.ios || !this.children.some(ch => ch instanceof TextView))) {
              let hack = document.createElement("BR");
              hack.cmIgnore = true;
              this.dom.appendChild(hack);
          }
      }
      measureTextSize() {
          if (this.children.length == 0 || this.length > 20)
              return null;
          let totalWidth = 0;
          for (let child of this.children) {
              if (!(child instanceof TextView))
                  return null;
              let rects = clientRectsFor(child.dom);
              if (rects.length != 1)
                  return null;
              totalWidth += rects[0].width;
          }
          return { lineHeight: this.dom.getBoundingClientRect().height,
              charWidth: totalWidth / this.length };
      }
      coordsAt(pos, side) {
          return coordsInChildren(this, pos, side);
      }
      become(_other) { return false; }
      get type() { return BlockType.Text; }
      static find(docView, pos) {
          for (let i = 0, off = 0; i < docView.children.length; i++) {
              let block = docView.children[i], end = off + block.length;
              if (end >= pos) {
                  if (block instanceof LineView)
                      return block;
                  if (end > pos)
                      break;
              }
              off = end + block.breakAfter;
          }
          return null;
      }
  }
  class BlockWidgetView extends ContentView {
      constructor(widget, length, type) {
          super();
          this.widget = widget;
          this.length = length;
          this.type = type;
          this.breakAfter = 0;
          this.prevWidget = null;
      }
      merge(from, to, source, _takeDeco, openStart, openEnd) {
          if (source && (!(source instanceof BlockWidgetView) || !this.widget.compare(source.widget) ||
              from > 0 && openStart <= 0 || to < this.length && openEnd <= 0))
              return false;
          this.length = from + (source ? source.length : 0) + (this.length - to);
          return true;
      }
      domAtPos(pos) {
          return pos == 0 ? DOMPos.before(this.dom) : DOMPos.after(this.dom, pos == this.length);
      }
      split(at) {
          let len = this.length - at;
          this.length = at;
          let end = new BlockWidgetView(this.widget, len, this.type);
          end.breakAfter = this.breakAfter;
          return end;
      }
      get children() { return noChildren; }
      sync() {
          if (!this.dom || !this.widget.updateDOM(this.dom)) {
              if (this.dom && this.prevWidget)
                  this.prevWidget.destroy(this.dom);
              this.prevWidget = null;
              this.setDOM(this.widget.toDOM(this.editorView));
              this.dom.contentEditable = "false";
          }
      }
      get overrideDOMText() {
          return this.parent ? this.parent.view.state.doc.slice(this.posAtStart, this.posAtEnd) : Text.empty;
      }
      domBoundsAround() { return null; }
      become(other) {
          if (other instanceof BlockWidgetView && other.type == this.type &&
              other.widget.constructor == this.widget.constructor) {
              if (!other.widget.eq(this.widget))
                  this.markDirty(true);
              if (this.dom && !this.prevWidget)
                  this.prevWidget = this.widget;
              this.widget = other.widget;
              this.length = other.length;
              this.breakAfter = other.breakAfter;
              return true;
          }
          return false;
      }
      ignoreMutation() { return true; }
      ignoreEvent(event) { return this.widget.ignoreEvent(event); }
      destroy() {
          super.destroy();
          if (this.dom)
              this.widget.destroy(this.dom);
      }
  }

  class ContentBuilder {
      constructor(doc, pos, end, disallowBlockEffectsFor) {
          this.doc = doc;
          this.pos = pos;
          this.end = end;
          this.disallowBlockEffectsFor = disallowBlockEffectsFor;
          this.content = [];
          this.curLine = null;
          this.breakAtStart = 0;
          this.pendingBuffer = 0 /* No */;
          // Set to false directly after a widget that covers the position after it
          this.atCursorPos = true;
          this.openStart = -1;
          this.openEnd = -1;
          this.text = "";
          this.textOff = 0;
          this.cursor = doc.iter();
          this.skip = pos;
      }
      posCovered() {
          if (this.content.length == 0)
              return !this.breakAtStart && this.doc.lineAt(this.pos).from != this.pos;
          let last = this.content[this.content.length - 1];
          return !last.breakAfter && !(last instanceof BlockWidgetView && last.type == BlockType.WidgetBefore);
      }
      getLine() {
          if (!this.curLine) {
              this.content.push(this.curLine = new LineView);
              this.atCursorPos = true;
          }
          return this.curLine;
      }
      flushBuffer(active) {
          if (this.pendingBuffer) {
              this.curLine.append(wrapMarks(new WidgetBufferView(-1), active), active.length);
              this.pendingBuffer = 0 /* No */;
          }
      }
      addBlockWidget(view) {
          this.flushBuffer([]);
          this.curLine = null;
          this.content.push(view);
      }
      finish(openEnd) {
          if (!openEnd)
              this.flushBuffer([]);
          else
              this.pendingBuffer = 0 /* No */;
          if (!this.posCovered())
              this.getLine();
      }
      buildText(length, active, openStart) {
          while (length > 0) {
              if (this.textOff == this.text.length) {
                  let { value, lineBreak, done } = this.cursor.next(this.skip);
                  this.skip = 0;
                  if (done)
                      throw new Error("Ran out of text content when drawing inline views");
                  if (lineBreak) {
                      if (!this.posCovered())
                          this.getLine();
                      if (this.content.length)
                          this.content[this.content.length - 1].breakAfter = 1;
                      else
                          this.breakAtStart = 1;
                      this.flushBuffer([]);
                      this.curLine = null;
                      length--;
                      continue;
                  }
                  else {
                      this.text = value;
                      this.textOff = 0;
                  }
              }
              let take = Math.min(this.text.length - this.textOff, length, 512 /* Chunk */);
              this.flushBuffer(active.slice(0, openStart));
              this.getLine().append(wrapMarks(new TextView(this.text.slice(this.textOff, this.textOff + take)), active), openStart);
              this.atCursorPos = true;
              this.textOff += take;
              length -= take;
              openStart = 0;
          }
      }
      span(from, to, active, openStart) {
          this.buildText(to - from, active, openStart);
          this.pos = to;
          if (this.openStart < 0)
              this.openStart = openStart;
      }
      point(from, to, deco, active, openStart, index) {
          if (this.disallowBlockEffectsFor[index] && deco instanceof PointDecoration) {
              if (deco.block)
                  throw new RangeError("Block decorations may not be specified via plugins");
              if (to > this.doc.lineAt(this.pos).to)
                  throw new RangeError("Decorations that replace line breaks may not be specified via plugins");
          }
          let len = to - from;
          if (deco instanceof PointDecoration) {
              if (deco.block) {
                  let { type } = deco;
                  if (type == BlockType.WidgetAfter && !this.posCovered())
                      this.getLine();
                  this.addBlockWidget(new BlockWidgetView(deco.widget || new NullWidget("div"), len, type));
              }
              else {
                  let view = WidgetView.create(deco.widget || new NullWidget("span"), len, deco.startSide);
                  let cursorBefore = this.atCursorPos && !view.isEditable && openStart <= active.length && (from < to || deco.startSide > 0);
                  let cursorAfter = !view.isEditable && (from < to || deco.startSide <= 0);
                  let line = this.getLine();
                  if (this.pendingBuffer == 2 /* IfCursor */ && !cursorBefore)
                      this.pendingBuffer = 0 /* No */;
                  this.flushBuffer(active);
                  if (cursorBefore) {
                      line.append(wrapMarks(new WidgetBufferView(1), active), openStart);
                      openStart = active.length + Math.max(0, openStart - active.length);
                  }
                  line.append(wrapMarks(view, active), openStart);
                  this.atCursorPos = cursorAfter;
                  this.pendingBuffer = !cursorAfter ? 0 /* No */ : from < to ? 1 /* Yes */ : 2 /* IfCursor */;
              }
          }
          else if (this.doc.lineAt(this.pos).from == this.pos) { // Line decoration
              this.getLine().addLineDeco(deco);
          }
          if (len) {
              // Advance the iterator past the replaced content
              if (this.textOff + len <= this.text.length) {
                  this.textOff += len;
              }
              else {
                  this.skip += len - (this.text.length - this.textOff);
                  this.text = "";
                  this.textOff = 0;
              }
              this.pos = to;
          }
          if (this.openStart < 0)
              this.openStart = openStart;
      }
      static build(text, from, to, decorations, dynamicDecorationMap) {
          let builder = new ContentBuilder(text, from, to, dynamicDecorationMap);
          builder.openEnd = RangeSet.spans(decorations, from, to, builder);
          if (builder.openStart < 0)
              builder.openStart = builder.openEnd;
          builder.finish(builder.openEnd);
          return builder;
      }
  }
  function wrapMarks(view, active) {
      for (let mark of active)
          view = new MarkView(mark, [view], view.length);
      return view;
  }
  class NullWidget extends WidgetType {
      constructor(tag) {
          super();
          this.tag = tag;
      }
      eq(other) { return other.tag == this.tag; }
      toDOM() { return document.createElement(this.tag); }
      updateDOM(elt) { return elt.nodeName.toLowerCase() == this.tag; }
  }

  const clickAddsSelectionRange = /*@__PURE__*/Facet.define();
  const dragMovesSelection$1 = /*@__PURE__*/Facet.define();
  const mouseSelectionStyle = /*@__PURE__*/Facet.define();
  const exceptionSink = /*@__PURE__*/Facet.define();
  const updateListener = /*@__PURE__*/Facet.define();
  const inputHandler$1 = /*@__PURE__*/Facet.define();
  const perLineTextDirection = /*@__PURE__*/Facet.define({
      combine: values => values.some(x => x)
  });
  class ScrollTarget {
      constructor(range, y = "nearest", x = "nearest", yMargin = 5, xMargin = 5) {
          this.range = range;
          this.y = y;
          this.x = x;
          this.yMargin = yMargin;
          this.xMargin = xMargin;
      }
      map(changes) {
          return changes.empty ? this : new ScrollTarget(this.range.map(changes), this.y, this.x, this.yMargin, this.xMargin);
      }
  }
  const scrollIntoView$1 = /*@__PURE__*/StateEffect.define({ map: (t, ch) => t.map(ch) });
  /**
  Log or report an unhandled exception in client code. Should
  probably only be used by extension code that allows client code to
  provide functions, and calls those functions in a context where an
  exception can't be propagated to calling code in a reasonable way
  (for example when in an event handler).

  Either calls a handler registered with
  [`EditorView.exceptionSink`](https://codemirror.net/6/docs/ref/#view.EditorView^exceptionSink),
  `window.onerror`, if defined, or `console.error` (in which case
  it'll pass `context`, when given, as first argument).
  */
  function logException(state, exception, context) {
      let handler = state.facet(exceptionSink);
      if (handler.length)
          handler[0](exception);
      else if (window.onerror)
          window.onerror(String(exception), context, undefined, undefined, exception);
      else if (context)
          console.error(context + ":", exception);
      else
          console.error(exception);
  }
  const editable = /*@__PURE__*/Facet.define({ combine: values => values.length ? values[0] : true });
  let nextPluginID = 0;
  const viewPlugin = /*@__PURE__*/Facet.define();
  /**
  View plugins associate stateful values with a view. They can
  influence the way the content is drawn, and are notified of things
  that happen in the view.
  */
  class ViewPlugin {
      constructor(
      /**
      @internal
      */
      id, 
      /**
      @internal
      */
      create, 
      /**
      @internal
      */
      domEventHandlers, buildExtensions) {
          this.id = id;
          this.create = create;
          this.domEventHandlers = domEventHandlers;
          this.extension = buildExtensions(this);
      }
      /**
      Define a plugin from a constructor function that creates the
      plugin's value, given an editor view.
      */
      static define(create, spec) {
          const { eventHandlers, provide, decorations: deco } = spec || {};
          return new ViewPlugin(nextPluginID++, create, eventHandlers, plugin => {
              let ext = [viewPlugin.of(plugin)];
              if (deco)
                  ext.push(decorations.of(view => {
                      let pluginInst = view.plugin(plugin);
                      return pluginInst ? deco(pluginInst) : Decoration.none;
                  }));
              if (provide)
                  ext.push(provide(plugin));
              return ext;
          });
      }
      /**
      Create a plugin for a class whose constructor takes a single
      editor view as argument.
      */
      static fromClass(cls, spec) {
          return ViewPlugin.define(view => new cls(view), spec);
      }
  }
  class PluginInstance {
      constructor(spec) {
          this.spec = spec;
          // When starting an update, all plugins have this field set to the
          // update object, indicating they need to be updated. When finished
          // updating, it is set to `false`. Retrieving a plugin that needs to
          // be updated with `view.plugin` forces an eager update.
          this.mustUpdate = null;
          // This is null when the plugin is initially created, but
          // initialized on the first update.
          this.value = null;
      }
      update(view) {
          if (!this.value) {
              if (this.spec) {
                  try {
                      this.value = this.spec.create(view);
                  }
                  catch (e) {
                      logException(view.state, e, "CodeMirror plugin crashed");
                      this.deactivate();
                  }
              }
          }
          else if (this.mustUpdate) {
              let update = this.mustUpdate;
              this.mustUpdate = null;
              if (this.value.update) {
                  try {
                      this.value.update(update);
                  }
                  catch (e) {
                      logException(update.state, e, "CodeMirror plugin crashed");
                      if (this.value.destroy)
                          try {
                              this.value.destroy();
                          }
                          catch (_) { }
                      this.deactivate();
                  }
              }
          }
          return this;
      }
      destroy(view) {
          var _a;
          if ((_a = this.value) === null || _a === void 0 ? void 0 : _a.destroy) {
              try {
                  this.value.destroy();
              }
              catch (e) {
                  logException(view.state, e, "CodeMirror plugin crashed");
              }
          }
      }
      deactivate() {
          this.spec = this.value = null;
      }
  }
  const editorAttributes = /*@__PURE__*/Facet.define();
  const contentAttributes = /*@__PURE__*/Facet.define();
  // Provide decorations
  const decorations = /*@__PURE__*/Facet.define();
  const atomicRanges = /*@__PURE__*/Facet.define();
  const scrollMargins = /*@__PURE__*/Facet.define();
  const styleModule = /*@__PURE__*/Facet.define();
  class ChangedRange {
      constructor(fromA, toA, fromB, toB) {
          this.fromA = fromA;
          this.toA = toA;
          this.fromB = fromB;
          this.toB = toB;
      }
      join(other) {
          return new ChangedRange(Math.min(this.fromA, other.fromA), Math.max(this.toA, other.toA), Math.min(this.fromB, other.fromB), Math.max(this.toB, other.toB));
      }
      addToSet(set) {
          let i = set.length, me = this;
          for (; i > 0; i--) {
              let range = set[i - 1];
              if (range.fromA > me.toA)
                  continue;
              if (range.toA < me.fromA)
                  break;
              me = me.join(range);
              set.splice(i - 1, 1);
          }
          set.splice(i, 0, me);
          return set;
      }
      static extendWithRanges(diff, ranges) {
          if (ranges.length == 0)
              return diff;
          let result = [];
          for (let dI = 0, rI = 0, posA = 0, posB = 0;; dI++) {
              let next = dI == diff.length ? null : diff[dI], off = posA - posB;
              let end = next ? next.fromB : 1e9;
              while (rI < ranges.length && ranges[rI] < end) {
                  let from = ranges[rI], to = ranges[rI + 1];
                  let fromB = Math.max(posB, from), toB = Math.min(end, to);
                  if (fromB <= toB)
                      new ChangedRange(fromB + off, toB + off, fromB, toB).addToSet(result);
                  if (to > end)
                      break;
                  else
                      rI += 2;
              }
              if (!next)
                  return result;
              new ChangedRange(next.fromA, next.toA, next.fromB, next.toB).addToSet(result);
              posA = next.toA;
              posB = next.toB;
          }
      }
  }
  /**
  View [plugins](https://codemirror.net/6/docs/ref/#view.ViewPlugin) are given instances of this
  class, which describe what happened, whenever the view is updated.
  */
  class ViewUpdate {
      constructor(
      /**
      The editor view that the update is associated with.
      */
      view, 
      /**
      The new editor state.
      */
      state, 
      /**
      The transactions involved in the update. May be empty.
      */
      transactions) {
          this.view = view;
          this.state = state;
          this.transactions = transactions;
          /**
          @internal
          */
          this.flags = 0;
          this.startState = view.state;
          this.changes = ChangeSet.empty(this.startState.doc.length);
          for (let tr of transactions)
              this.changes = this.changes.compose(tr.changes);
          let changedRanges = [];
          this.changes.iterChangedRanges((fromA, toA, fromB, toB) => changedRanges.push(new ChangedRange(fromA, toA, fromB, toB)));
          this.changedRanges = changedRanges;
          let focus = view.hasFocus;
          if (focus != view.inputState.notifiedFocused) {
              view.inputState.notifiedFocused = focus;
              this.flags |= 1 /* Focus */;
          }
      }
      /**
      @internal
      */
      static create(view, state, transactions) {
          return new ViewUpdate(view, state, transactions);
      }
      /**
      Tells you whether the [viewport](https://codemirror.net/6/docs/ref/#view.EditorView.viewport) or
      [visible ranges](https://codemirror.net/6/docs/ref/#view.EditorView.visibleRanges) changed in this
      update.
      */
      get viewportChanged() {
          return (this.flags & 4 /* Viewport */) > 0;
      }
      /**
      Indicates whether the height of a block element in the editor
      changed in this update.
      */
      get heightChanged() {
          return (this.flags & 2 /* Height */) > 0;
      }
      /**
      Returns true when the document was modified or the size of the
      editor, or elements within the editor, changed.
      */
      get geometryChanged() {
          return this.docChanged || (this.flags & (8 /* Geometry */ | 2 /* Height */)) > 0;
      }
      /**
      True when this update indicates a focus change.
      */
      get focusChanged() {
          return (this.flags & 1 /* Focus */) > 0;
      }
      /**
      Whether the document changed in this update.
      */
      get docChanged() {
          return !this.changes.empty;
      }
      /**
      Whether the selection was explicitly set in this update.
      */
      get selectionSet() {
          return this.transactions.some(tr => tr.selection);
      }
      /**
      @internal
      */
      get empty() { return this.flags == 0 && this.transactions.length == 0; }
  }

  /**
  Used to indicate [text direction](https://codemirror.net/6/docs/ref/#view.EditorView.textDirection).
  */
  var Direction = /*@__PURE__*/(function (Direction) {
      // (These are chosen to match the base levels, in bidi algorithm
      // terms, of spans in that direction.)
      /**
      Left-to-right.
      */
      Direction[Direction["LTR"] = 0] = "LTR";
      /**
      Right-to-left.
      */
      Direction[Direction["RTL"] = 1] = "RTL";
  return Direction})(Direction || (Direction = {}));
  const LTR = Direction.LTR, RTL = Direction.RTL;
  // Decode a string with each type encoded as log2(type)
  function dec(str) {
      let result = [];
      for (let i = 0; i < str.length; i++)
          result.push(1 << +str[i]);
      return result;
  }
  // Character types for codepoints 0 to 0xf8
  const LowTypes = /*@__PURE__*/dec("88888888888888888888888888888888888666888888787833333333337888888000000000000000000000000008888880000000000000000000000000088888888888888888888888888888888888887866668888088888663380888308888800000000000000000000000800000000000000000000000000000008");
  // Character types for codepoints 0x600 to 0x6f9
  const ArabicTypes = /*@__PURE__*/dec("4444448826627288999999999992222222222222222222222222222222222222222222222229999999999999999999994444444444644222822222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222999999949999999229989999223333333333");
  const Brackets = /*@__PURE__*/Object.create(null), BracketStack = [];
  // There's a lot more in
  // https://www.unicode.org/Public/UCD/latest/ucd/BidiBrackets.txt,
  // which are left out to keep code size down.
  for (let p of ["()", "[]", "{}"]) {
      let l = /*@__PURE__*/p.charCodeAt(0), r = /*@__PURE__*/p.charCodeAt(1);
      Brackets[l] = r;
      Brackets[r] = -l;
  }
  function charType(ch) {
      return ch <= 0xf7 ? LowTypes[ch] :
          0x590 <= ch && ch <= 0x5f4 ? 2 /* R */ :
              0x600 <= ch && ch <= 0x6f9 ? ArabicTypes[ch - 0x600] :
                  0x6ee <= ch && ch <= 0x8ac ? 4 /* AL */ :
                      0x2000 <= ch && ch <= 0x200b ? 256 /* NI */ :
                          ch == 0x200c ? 256 /* NI */ : 1 /* L */;
  }
  const BidiRE = /[\u0590-\u05f4\u0600-\u06ff\u0700-\u08ac]/;
  /**
  Represents a contiguous range of text that has a single direction
  (as in left-to-right or right-to-left).
  */
  class BidiSpan {
      /**
      @internal
      */
      constructor(
      /**
      The start of the span (relative to the start of the line).
      */
      from, 
      /**
      The end of the span.
      */
      to, 
      /**
      The ["bidi
      level"](https://unicode.org/reports/tr9/#Basic_Display_Algorithm)
      of the span (in this context, 0 means
      left-to-right, 1 means right-to-left, 2 means left-to-right
      number inside right-to-left text).
      */
      level) {
          this.from = from;
          this.to = to;
          this.level = level;
      }
      /**
      The direction of this span.
      */
      get dir() { return this.level % 2 ? RTL : LTR; }
      /**
      @internal
      */
      side(end, dir) { return (this.dir == dir) == end ? this.to : this.from; }
      /**
      @internal
      */
      static find(order, index, level, assoc) {
          let maybe = -1;
          for (let i = 0; i < order.length; i++) {
              let span = order[i];
              if (span.from <= index && span.to >= index) {
                  if (span.level == level)
                      return i;
                  // When multiple spans match, if assoc != 0, take the one that
                  // covers that side, otherwise take the one with the minimum
                  // level.
                  if (maybe < 0 || (assoc != 0 ? (assoc < 0 ? span.from < index : span.to > index) : order[maybe].level > span.level))
                      maybe = i;
              }
          }
          if (maybe < 0)
              throw new RangeError("Index out of range");
          return maybe;
      }
  }
  // Reused array of character types
  const types = [];
  function computeOrder(line, direction) {
      let len = line.length, outerType = direction == LTR ? 1 /* L */ : 2 /* R */, oppositeType = direction == LTR ? 2 /* R */ : 1 /* L */;
      if (!line || outerType == 1 /* L */ && !BidiRE.test(line))
          return trivialOrder(len);
      // W1. Examine each non-spacing mark (NSM) in the level run, and
      // change the type of the NSM to the type of the previous
      // character. If the NSM is at the start of the level run, it will
      // get the type of sor.
      // W2. Search backwards from each instance of a European number
      // until the first strong type (R, L, AL, or sor) is found. If an
      // AL is found, change the type of the European number to Arabic
      // number.
      // W3. Change all ALs to R.
      // (Left after this: L, R, EN, AN, ET, CS, NI)
      for (let i = 0, prev = outerType, prevStrong = outerType; i < len; i++) {
          let type = charType(line.charCodeAt(i));
          if (type == 512 /* NSM */)
              type = prev;
          else if (type == 8 /* EN */ && prevStrong == 4 /* AL */)
              type = 16 /* AN */;
          types[i] = type == 4 /* AL */ ? 2 /* R */ : type;
          if (type & 7 /* Strong */)
              prevStrong = type;
          prev = type;
      }
      // W5. A sequence of European terminators adjacent to European
      // numbers changes to all European numbers.
      // W6. Otherwise, separators and terminators change to Other
      // Neutral.
      // W7. Search backwards from each instance of a European number
      // until the first strong type (R, L, or sor) is found. If an L is
      // found, then change the type of the European number to L.
      // (Left after this: L, R, EN+AN, NI)
      for (let i = 0, prev = outerType, prevStrong = outerType; i < len; i++) {
          let type = types[i];
          if (type == 128 /* CS */) {
              if (i < len - 1 && prev == types[i + 1] && (prev & 24 /* Num */))
                  type = types[i] = prev;
              else
                  types[i] = 256 /* NI */;
          }
          else if (type == 64 /* ET */) {
              let end = i + 1;
              while (end < len && types[end] == 64 /* ET */)
                  end++;
              let replace = (i && prev == 8 /* EN */) || (end < len && types[end] == 8 /* EN */) ? (prevStrong == 1 /* L */ ? 1 /* L */ : 8 /* EN */) : 256 /* NI */;
              for (let j = i; j < end; j++)
                  types[j] = replace;
              i = end - 1;
          }
          else if (type == 8 /* EN */ && prevStrong == 1 /* L */) {
              types[i] = 1 /* L */;
          }
          prev = type;
          if (type & 7 /* Strong */)
              prevStrong = type;
      }
      // N0. Process bracket pairs in an isolating run sequence
      // sequentially in the logical order of the text positions of the
      // opening paired brackets using the logic given below. Within this
      // scope, bidirectional types EN and AN are treated as R.
      for (let i = 0, sI = 0, context = 0, ch, br, type; i < len; i++) {
          // Keeps [startIndex, type, strongSeen] triples for each open
          // bracket on BracketStack.
          if (br = Brackets[ch = line.charCodeAt(i)]) {
              if (br < 0) { // Closing bracket
                  for (let sJ = sI - 3; sJ >= 0; sJ -= 3) {
                      if (BracketStack[sJ + 1] == -br) {
                          let flags = BracketStack[sJ + 2];
                          let type = (flags & 2 /* EmbedInside */) ? outerType :
                              !(flags & 4 /* OppositeInside */) ? 0 :
                                  (flags & 1 /* OppositeBefore */) ? oppositeType : outerType;
                          if (type)
                              types[i] = types[BracketStack[sJ]] = type;
                          sI = sJ;
                          break;
                      }
                  }
              }
              else if (BracketStack.length == 189 /* MaxDepth */) {
                  break;
              }
              else {
                  BracketStack[sI++] = i;
                  BracketStack[sI++] = ch;
                  BracketStack[sI++] = context;
              }
          }
          else if ((type = types[i]) == 2 /* R */ || type == 1 /* L */) {
              let embed = type == outerType;
              context = embed ? 0 : 1 /* OppositeBefore */;
              for (let sJ = sI - 3; sJ >= 0; sJ -= 3) {
                  let cur = BracketStack[sJ + 2];
                  if (cur & 2 /* EmbedInside */)
                      break;
                  if (embed) {
                      BracketStack[sJ + 2] |= 2 /* EmbedInside */;
                  }
                  else {
                      if (cur & 4 /* OppositeInside */)
                          break;
                      BracketStack[sJ + 2] |= 4 /* OppositeInside */;
                  }
              }
          }
      }
      // N1. A sequence of neutrals takes the direction of the
      // surrounding strong text if the text on both sides has the same
      // direction. European and Arabic numbers act as if they were R in
      // terms of their influence on neutrals. Start-of-level-run (sor)
      // and end-of-level-run (eor) are used at level run boundaries.
      // N2. Any remaining neutrals take the embedding direction.
      // (Left after this: L, R, EN+AN)
      for (let i = 0; i < len; i++) {
          if (types[i] == 256 /* NI */) {
              let end = i + 1;
              while (end < len && types[end] == 256 /* NI */)
                  end++;
              let beforeL = (i ? types[i - 1] : outerType) == 1 /* L */;
              let afterL = (end < len ? types[end] : outerType) == 1 /* L */;
              let replace = beforeL == afterL ? (beforeL ? 1 /* L */ : 2 /* R */) : outerType;
              for (let j = i; j < end; j++)
                  types[j] = replace;
              i = end - 1;
          }
      }
      // Here we depart from the documented algorithm, in order to avoid
      // building up an actual levels array. Since there are only three
      // levels (0, 1, 2) in an implementation that doesn't take
      // explicit embedding into account, we can build up the order on
      // the fly, without following the level-based algorithm.
      let order = [];
      if (outerType == 1 /* L */) {
          for (let i = 0; i < len;) {
              let start = i, rtl = types[i++] != 1 /* L */;
              while (i < len && rtl == (types[i] != 1 /* L */))
                  i++;
              if (rtl) {
                  for (let j = i; j > start;) {
                      let end = j, l = types[--j] != 2 /* R */;
                      while (j > start && l == (types[j - 1] != 2 /* R */))
                          j--;
                      order.push(new BidiSpan(j, end, l ? 2 : 1));
                  }
              }
              else {
                  order.push(new BidiSpan(start, i, 0));
              }
          }
      }
      else {
          for (let i = 0; i < len;) {
              let start = i, rtl = types[i++] == 2 /* R */;
              while (i < len && rtl == (types[i] == 2 /* R */))
                  i++;
              order.push(new BidiSpan(start, i, rtl ? 1 : 2));
          }
      }
      return order;
  }
  function trivialOrder(length) {
      return [new BidiSpan(0, length, 0)];
  }
  let movedOver = "";
  function moveVisually(line, order, dir, start, forward) {
      var _a;
      let startIndex = start.head - line.from, spanI = -1;
      if (startIndex == 0) {
          if (!forward || !line.length)
              return null;
          if (order[0].level != dir) {
              startIndex = order[0].side(false, dir);
              spanI = 0;
          }
      }
      else if (startIndex == line.length) {
          if (forward)
              return null;
          let last = order[order.length - 1];
          if (last.level != dir) {
              startIndex = last.side(true, dir);
              spanI = order.length - 1;
          }
      }
      if (spanI < 0)
          spanI = BidiSpan.find(order, startIndex, (_a = start.bidiLevel) !== null && _a !== void 0 ? _a : -1, start.assoc);
      let span = order[spanI];
      // End of span. (But not end of line--that was checked for above.)
      if (startIndex == span.side(forward, dir)) {
          span = order[spanI += forward ? 1 : -1];
          startIndex = span.side(!forward, dir);
      }
      let indexForward = forward == (span.dir == dir);
      let nextIndex = findClusterBreak(line.text, startIndex, indexForward);
      movedOver = line.text.slice(Math.min(startIndex, nextIndex), Math.max(startIndex, nextIndex));
      if (nextIndex != span.side(forward, dir))
          return EditorSelection.cursor(nextIndex + line.from, indexForward ? -1 : 1, span.level);
      let nextSpan = spanI == (forward ? order.length - 1 : 0) ? null : order[spanI + (forward ? 1 : -1)];
      if (!nextSpan && span.level != dir)
          return EditorSelection.cursor(forward ? line.to : line.from, forward ? -1 : 1, dir);
      if (nextSpan && nextSpan.level < span.level)
          return EditorSelection.cursor(nextSpan.side(!forward, dir) + line.from, forward ? 1 : -1, nextSpan.level);
      return EditorSelection.cursor(nextIndex + line.from, forward ? -1 : 1, span.level);
  }

  const LineBreakPlaceholder = "\uffff";
  class DOMReader {
      constructor(points, state) {
          this.points = points;
          this.text = "";
          this.lineSeparator = state.facet(EditorState.lineSeparator);
      }
      append(text) {
          this.text += text;
      }
      lineBreak() {
          this.text += LineBreakPlaceholder;
      }
      readRange(start, end) {
          if (!start)
              return this;
          let parent = start.parentNode;
          for (let cur = start;;) {
              this.findPointBefore(parent, cur);
              this.readNode(cur);
              let next = cur.nextSibling;
              if (next == end)
                  break;
              let view = ContentView.get(cur), nextView = ContentView.get(next);
              if (view && nextView ? view.breakAfter :
                  (view ? view.breakAfter : isBlockElement(cur)) ||
                      (isBlockElement(next) && (cur.nodeName != "BR" || cur.cmIgnore)))
                  this.lineBreak();
              cur = next;
          }
          this.findPointBefore(parent, end);
          return this;
      }
      readTextNode(node) {
          let text = node.nodeValue;
          for (let point of this.points)
              if (point.node == node)
                  point.pos = this.text.length + Math.min(point.offset, text.length);
          for (let off = 0, re = this.lineSeparator ? null : /\r\n?|\n/g;;) {
              let nextBreak = -1, breakSize = 1, m;
              if (this.lineSeparator) {
                  nextBreak = text.indexOf(this.lineSeparator, off);
                  breakSize = this.lineSeparator.length;
              }
              else if (m = re.exec(text)) {
                  nextBreak = m.index;
                  breakSize = m[0].length;
              }
              this.append(text.slice(off, nextBreak < 0 ? text.length : nextBreak));
              if (nextBreak < 0)
                  break;
              this.lineBreak();
              if (breakSize > 1)
                  for (let point of this.points)
                      if (point.node == node && point.pos > this.text.length)
                          point.pos -= breakSize - 1;
              off = nextBreak + breakSize;
          }
      }
      readNode(node) {
          if (node.cmIgnore)
              return;
          let view = ContentView.get(node);
          let fromView = view && view.overrideDOMText;
          if (fromView != null) {
              this.findPointInside(node, fromView.length);
              for (let i = fromView.iter(); !i.next().done;) {
                  if (i.lineBreak)
                      this.lineBreak();
                  else
                      this.append(i.value);
              }
          }
          else if (node.nodeType == 3) {
              this.readTextNode(node);
          }
          else if (node.nodeName == "BR") {
              if (node.nextSibling)
                  this.lineBreak();
          }
          else if (node.nodeType == 1) {
              this.readRange(node.firstChild, null);
          }
      }
      findPointBefore(node, next) {
          for (let point of this.points)
              if (point.node == node && node.childNodes[point.offset] == next)
                  point.pos = this.text.length;
      }
      findPointInside(node, maxLen) {
          for (let point of this.points)
              if (node.nodeType == 3 ? point.node == node : node.contains(point.node))
                  point.pos = this.text.length + Math.min(maxLen, point.offset);
      }
  }
  function isBlockElement(node) {
      return node.nodeType == 1 && /^(DIV|P|LI|UL|OL|BLOCKQUOTE|DD|DT|H\d|SECTION|PRE)$/.test(node.nodeName);
  }
  class DOMPoint {
      constructor(node, offset) {
          this.node = node;
          this.offset = offset;
          this.pos = -1;
      }
  }

  class DocView extends ContentView {
      constructor(view) {
          super();
          this.view = view;
          this.compositionDeco = Decoration.none;
          this.decorations = [];
          this.dynamicDecorationMap = [];
          // Track a minimum width for the editor. When measuring sizes in
          // measureVisibleLineHeights, this is updated to point at the width
          // of a given element and its extent in the document. When a change
          // happens in that range, these are reset. That way, once we've seen
          // a line/element of a given length, we keep the editor wide enough
          // to fit at least that element, until it is changed, at which point
          // we forget it again.
          this.minWidth = 0;
          this.minWidthFrom = 0;
          this.minWidthTo = 0;
          // Track whether the DOM selection was set in a lossy way, so that
          // we don't mess it up when reading it back it
          this.impreciseAnchor = null;
          this.impreciseHead = null;
          this.forceSelection = false;
          // Used by the resize observer to ignore resizes that we caused
          // ourselves
          this.lastUpdate = Date.now();
          this.setDOM(view.contentDOM);
          this.children = [new LineView];
          this.children[0].setParent(this);
          this.updateDeco();
          this.updateInner([new ChangedRange(0, 0, 0, view.state.doc.length)], 0);
      }
      get root() { return this.view.root; }
      get editorView() { return this.view; }
      get length() { return this.view.state.doc.length; }
      // Update the document view to a given state. scrollIntoView can be
      // used as a hint to compute a new viewport that includes that
      // position, if we know the editor is going to scroll that position
      // into view.
      update(update) {
          let changedRanges = update.changedRanges;
          if (this.minWidth > 0 && changedRanges.length) {
              if (!changedRanges.every(({ fromA, toA }) => toA < this.minWidthFrom || fromA > this.minWidthTo)) {
                  this.minWidth = this.minWidthFrom = this.minWidthTo = 0;
              }
              else {
                  this.minWidthFrom = update.changes.mapPos(this.minWidthFrom, 1);
                  this.minWidthTo = update.changes.mapPos(this.minWidthTo, 1);
              }
          }
          if (this.view.inputState.composing < 0)
              this.compositionDeco = Decoration.none;
          else if (update.transactions.length || this.dirty)
              this.compositionDeco = computeCompositionDeco(this.view, update.changes);
          // When the DOM nodes around the selection are moved to another
          // parent, Chrome sometimes reports a different selection through
          // getSelection than the one that it actually shows to the user.
          // This forces a selection update when lines are joined to work
          // around that. Issue #54
          if ((browser.ie || browser.chrome) && !this.compositionDeco.size && update &&
              update.state.doc.lines != update.startState.doc.lines)
              this.forceSelection = true;
          let prevDeco = this.decorations, deco = this.updateDeco();
          let decoDiff = findChangedDeco(prevDeco, deco, update.changes);
          changedRanges = ChangedRange.extendWithRanges(changedRanges, decoDiff);
          if (this.dirty == 0 /* Not */ && changedRanges.length == 0) {
              return false;
          }
          else {
              this.updateInner(changedRanges, update.startState.doc.length);
              if (update.transactions.length)
                  this.lastUpdate = Date.now();
              return true;
          }
      }
      // Used by update and the constructor do perform the actual DOM
      // update
      updateInner(changes, oldLength) {
          this.view.viewState.mustMeasureContent = true;
          this.updateChildren(changes, oldLength);
          let { observer } = this.view;
          observer.ignore(() => {
              // Lock the height during redrawing, since Chrome sometimes
              // messes with the scroll position during DOM mutation (though
              // no relayout is triggered and I cannot imagine how it can
              // recompute the scroll position without a layout)
              this.dom.style.height = this.view.viewState.contentHeight + "px";
              this.dom.style.flexBasis = this.minWidth ? this.minWidth + "px" : "";
              // Chrome will sometimes, when DOM mutations occur directly
              // around the selection, get confused and report a different
              // selection from the one it displays (issue #218). This tries
              // to detect that situation.
              let track = browser.chrome || browser.ios ? { node: observer.selectionRange.focusNode, written: false } : undefined;
              this.sync(track);
              this.dirty = 0 /* Not */;
              if (track && (track.written || observer.selectionRange.focusNode != track.node))
                  this.forceSelection = true;
              this.dom.style.height = "";
          });
          let gaps = [];
          if (this.view.viewport.from || this.view.viewport.to < this.view.state.doc.length)
              for (let child of this.children)
                  if (child instanceof BlockWidgetView && child.widget instanceof BlockGapWidget)
                      gaps.push(child.dom);
          observer.updateGaps(gaps);
      }
      updateChildren(changes, oldLength) {
          let cursor = this.childCursor(oldLength);
          for (let i = changes.length - 1;; i--) {
              let next = i >= 0 ? changes[i] : null;
              if (!next)
                  break;
              let { fromA, toA, fromB, toB } = next;
              let { content, breakAtStart, openStart, openEnd } = ContentBuilder.build(this.view.state.doc, fromB, toB, this.decorations, this.dynamicDecorationMap);
              let { i: toI, off: toOff } = cursor.findPos(toA, 1);
              let { i: fromI, off: fromOff } = cursor.findPos(fromA, -1);
              replaceRange(this, fromI, fromOff, toI, toOff, content, breakAtStart, openStart, openEnd);
          }
      }
      // Sync the DOM selection to this.state.selection
      updateSelection(mustRead = false, fromPointer = false) {
          if (mustRead || !this.view.observer.selectionRange.focusNode)
              this.view.observer.readSelectionRange();
          if (!(fromPointer || this.mayControlSelection()) ||
              browser.ios && this.view.inputState.rapidCompositionStart)
              return;
          let force = this.forceSelection;
          this.forceSelection = false;
          let main = this.view.state.selection.main;
          // FIXME need to handle the case where the selection falls inside a block range
          let anchor = this.domAtPos(main.anchor);
          let head = main.empty ? anchor : this.domAtPos(main.head);
          // Always reset on Firefox when next to an uneditable node to
          // avoid invisible cursor bugs (#111)
          if (browser.gecko && main.empty && betweenUneditable(anchor)) {
              let dummy = document.createTextNode("");
              this.view.observer.ignore(() => anchor.node.insertBefore(dummy, anchor.node.childNodes[anchor.offset] || null));
              anchor = head = new DOMPos(dummy, 0);
              force = true;
          }
          let domSel = this.view.observer.selectionRange;
          // If the selection is already here, or in an equivalent position, don't touch it
          if (force || !domSel.focusNode ||
              !isEquivalentPosition(anchor.node, anchor.offset, domSel.anchorNode, domSel.anchorOffset) ||
              !isEquivalentPosition(head.node, head.offset, domSel.focusNode, domSel.focusOffset)) {
              this.view.observer.ignore(() => {
                  // Chrome Android will hide the virtual keyboard when tapping
                  // inside an uneditable node, and not bring it back when we
                  // move the cursor to its proper position. This tries to
                  // restore the keyboard by cycling focus.
                  if (browser.android && browser.chrome && this.dom.contains(domSel.focusNode) &&
                      inUneditable(domSel.focusNode, this.dom)) {
                      this.dom.blur();
                      this.dom.focus({ preventScroll: true });
                  }
                  let rawSel = getSelection(this.root);
                  if (!rawSel) ;
                  else if (main.empty) {
                      // Work around https://bugzilla.mozilla.org/show_bug.cgi?id=1612076
                      if (browser.gecko) {
                          let nextTo = nextToUneditable(anchor.node, anchor.offset);
                          if (nextTo && nextTo != (1 /* Before */ | 2 /* After */)) {
                              let text = nearbyTextNode(anchor.node, anchor.offset, nextTo == 1 /* Before */ ? 1 : -1);
                              if (text)
                                  anchor = new DOMPos(text, nextTo == 1 /* Before */ ? 0 : text.nodeValue.length);
                          }
                      }
                      rawSel.collapse(anchor.node, anchor.offset);
                      if (main.bidiLevel != null && domSel.cursorBidiLevel != null)
                          domSel.cursorBidiLevel = main.bidiLevel;
                  }
                  else if (rawSel.extend) {
                      // Selection.extend can be used to create an 'inverted' selection
                      // (one where the focus is before the anchor), but not all
                      // browsers support it yet.
                      rawSel.collapse(anchor.node, anchor.offset);
                      rawSel.extend(head.node, head.offset);
                  }
                  else {
                      // Primitive (IE) way
                      let range = document.createRange();
                      if (main.anchor > main.head)
                          [anchor, head] = [head, anchor];
                      range.setEnd(head.node, head.offset);
                      range.setStart(anchor.node, anchor.offset);
                      rawSel.removeAllRanges();
                      rawSel.addRange(range);
                  }
              });
              this.view.observer.setSelectionRange(anchor, head);
          }
          this.impreciseAnchor = anchor.precise ? null : new DOMPos(domSel.anchorNode, domSel.anchorOffset);
          this.impreciseHead = head.precise ? null : new DOMPos(domSel.focusNode, domSel.focusOffset);
      }
      enforceCursorAssoc() {
          if (this.compositionDeco.size)
              return;
          let cursor = this.view.state.selection.main;
          let sel = getSelection(this.root);
          if (!sel || !cursor.empty || !cursor.assoc || !sel.modify)
              return;
          let line = LineView.find(this, cursor.head);
          if (!line)
              return;
          let lineStart = line.posAtStart;
          if (cursor.head == lineStart || cursor.head == lineStart + line.length)
              return;
          let before = this.coordsAt(cursor.head, -1), after = this.coordsAt(cursor.head, 1);
          if (!before || !after || before.bottom > after.top)
              return;
          let dom = this.domAtPos(cursor.head + cursor.assoc);
          sel.collapse(dom.node, dom.offset);
          sel.modify("move", cursor.assoc < 0 ? "forward" : "backward", "lineboundary");
      }
      mayControlSelection() {
          let active = this.root.activeElement;
          return active == this.dom ||
              hasSelection(this.dom, this.view.observer.selectionRange) && !(active && this.dom.contains(active));
      }
      nearest(dom) {
          for (let cur = dom; cur;) {
              let domView = ContentView.get(cur);
              if (domView && domView.rootView == this)
                  return domView;
              cur = cur.parentNode;
          }
          return null;
      }
      posFromDOM(node, offset) {
          let view = this.nearest(node);
          if (!view)
              throw new RangeError("Trying to find position for a DOM position outside of the document");
          return view.localPosFromDOM(node, offset) + view.posAtStart;
      }
      domAtPos(pos) {
          let { i, off } = this.childCursor().findPos(pos, -1);
          for (; i < this.children.length - 1;) {
              let child = this.children[i];
              if (off < child.length || child instanceof LineView)
                  break;
              i++;
              off = 0;
          }
          return this.children[i].domAtPos(off);
      }
      coordsAt(pos, side) {
          for (let off = this.length, i = this.children.length - 1;; i--) {
              let child = this.children[i], start = off - child.breakAfter - child.length;
              if (pos > start ||
                  (pos == start && child.type != BlockType.WidgetBefore && child.type != BlockType.WidgetAfter &&
                      (!i || side == 2 || this.children[i - 1].breakAfter ||
                          (this.children[i - 1].type == BlockType.WidgetBefore && side > -2))))
                  return child.coordsAt(pos - start, side);
              off = start;
          }
      }
      measureVisibleLineHeights(viewport) {
          let result = [], { from, to } = viewport;
          let contentWidth = this.view.contentDOM.clientWidth;
          let isWider = contentWidth > Math.max(this.view.scrollDOM.clientWidth, this.minWidth) + 1;
          let widest = -1, ltr = this.view.textDirection == Direction.LTR;
          for (let pos = 0, i = 0; i < this.children.length; i++) {
              let child = this.children[i], end = pos + child.length;
              if (end > to)
                  break;
              if (pos >= from) {
                  let childRect = child.dom.getBoundingClientRect();
                  result.push(childRect.height);
                  if (isWider) {
                      let last = child.dom.lastChild;
                      let rects = last ? clientRectsFor(last) : [];
                      if (rects.length) {
                          let rect = rects[rects.length - 1];
                          let width = ltr ? rect.right - childRect.left : childRect.right - rect.left;
                          if (width > widest) {
                              widest = width;
                              this.minWidth = contentWidth;
                              this.minWidthFrom = pos;
                              this.minWidthTo = end;
                          }
                      }
                  }
              }
              pos = end + child.breakAfter;
          }
          return result;
      }
      textDirectionAt(pos) {
          let { i } = this.childPos(pos, 1);
          return getComputedStyle(this.children[i].dom).direction == "rtl" ? Direction.RTL : Direction.LTR;
      }
      measureTextSize() {
          for (let child of this.children) {
              if (child instanceof LineView) {
                  let measure = child.measureTextSize();
                  if (measure)
                      return measure;
              }
          }
          // If no workable line exists, force a layout of a measurable element
          let dummy = document.createElement("div"), lineHeight, charWidth;
          dummy.className = "cm-line";
          dummy.style.width = "99999px";
          dummy.textContent = "abc def ghi jkl mno pqr stu";
          this.view.observer.ignore(() => {
              this.dom.appendChild(dummy);
              let rect = clientRectsFor(dummy.firstChild)[0];
              lineHeight = dummy.getBoundingClientRect().height;
              charWidth = rect ? rect.width / 27 : 7;
              dummy.remove();
          });
          return { lineHeight, charWidth };
      }
      childCursor(pos = this.length) {
          // Move back to start of last element when possible, so that
          // `ChildCursor.findPos` doesn't have to deal with the edge case
          // of being after the last element.
          let i = this.children.length;
          if (i)
              pos -= this.children[--i].length;
          return new ChildCursor(this.children, pos, i);
      }
      computeBlockGapDeco() {
          let deco = [], vs = this.view.viewState;
          for (let pos = 0, i = 0;; i++) {
              let next = i == vs.viewports.length ? null : vs.viewports[i];
              let end = next ? next.from - 1 : this.length;
              if (end > pos) {
                  let height = vs.lineBlockAt(end).bottom - vs.lineBlockAt(pos).top;
                  deco.push(Decoration.replace({
                      widget: new BlockGapWidget(height),
                      block: true,
                      inclusive: true,
                      isBlockGap: true,
                  }).range(pos, end));
              }
              if (!next)
                  break;
              pos = next.to + 1;
          }
          return Decoration.set(deco);
      }
      updateDeco() {
          let allDeco = this.view.state.facet(decorations).map((d, i) => {
              let dynamic = this.dynamicDecorationMap[i] = typeof d == "function";
              return dynamic ? d(this.view) : d;
          });
          for (let i = allDeco.length; i < allDeco.length + 3; i++)
              this.dynamicDecorationMap[i] = false;
          return this.decorations = [
              ...allDeco,
              this.compositionDeco,
              this.computeBlockGapDeco(),
              this.view.viewState.lineGapDeco
          ];
      }
      scrollIntoView(target) {
          let { range } = target;
          let rect = this.coordsAt(range.head, range.empty ? range.assoc : range.head > range.anchor ? -1 : 1), other;
          if (!rect)
              return;
          if (!range.empty && (other = this.coordsAt(range.anchor, range.anchor > range.head ? -1 : 1)))
              rect = { left: Math.min(rect.left, other.left), top: Math.min(rect.top, other.top),
                  right: Math.max(rect.right, other.right), bottom: Math.max(rect.bottom, other.bottom) };
          let mLeft = 0, mRight = 0, mTop = 0, mBottom = 0;
          for (let margins of this.view.state.facet(scrollMargins).map(f => f(this.view)))
              if (margins) {
                  let { left, right, top, bottom } = margins;
                  if (left != null)
                      mLeft = Math.max(mLeft, left);
                  if (right != null)
                      mRight = Math.max(mRight, right);
                  if (top != null)
                      mTop = Math.max(mTop, top);
                  if (bottom != null)
                      mBottom = Math.max(mBottom, bottom);
              }
          let targetRect = {
              left: rect.left - mLeft, top: rect.top - mTop,
              right: rect.right + mRight, bottom: rect.bottom + mBottom
          };
          scrollRectIntoView(this.view.scrollDOM, targetRect, range.head < range.anchor ? -1 : 1, target.x, target.y, target.xMargin, target.yMargin, this.view.textDirection == Direction.LTR);
      }
  }
  function betweenUneditable(pos) {
      return pos.node.nodeType == 1 && pos.node.firstChild &&
          (pos.offset == 0 || pos.node.childNodes[pos.offset - 1].contentEditable == "false") &&
          (pos.offset == pos.node.childNodes.length || pos.node.childNodes[pos.offset].contentEditable == "false");
  }
  class BlockGapWidget extends WidgetType {
      constructor(height) {
          super();
          this.height = height;
      }
      toDOM() {
          let elt = document.createElement("div");
          this.updateDOM(elt);
          return elt;
      }
      eq(other) { return other.height == this.height; }
      updateDOM(elt) {
          elt.style.height = this.height + "px";
          return true;
      }
      get estimatedHeight() { return this.height; }
  }
  function compositionSurroundingNode(view) {
      let sel = view.observer.selectionRange;
      let textNode = sel.focusNode && nearbyTextNode(sel.focusNode, sel.focusOffset, 0);
      if (!textNode)
          return null;
      let cView = view.docView.nearest(textNode);
      if (!cView)
          return null;
      if (cView instanceof LineView) {
          let topNode = textNode;
          while (topNode.parentNode != cView.dom)
              topNode = topNode.parentNode;
          let prev = topNode.previousSibling;
          while (prev && !ContentView.get(prev))
              prev = prev.previousSibling;
          let pos = prev ? ContentView.get(prev).posAtEnd : cView.posAtStart;
          return { from: pos, to: pos, node: topNode, text: textNode };
      }
      else {
          for (;;) {
              let { parent } = cView;
              if (!parent)
                  return null;
              if (parent instanceof LineView)
                  break;
              cView = parent;
          }
          let from = cView.posAtStart;
          return { from, to: from + cView.length, node: cView.dom, text: textNode };
      }
  }
  function computeCompositionDeco(view, changes) {
      let surrounding = compositionSurroundingNode(view);
      if (!surrounding)
          return Decoration.none;
      let { from, to, node, text: textNode } = surrounding;
      let newFrom = changes.mapPos(from, 1), newTo = Math.max(newFrom, changes.mapPos(to, -1));
      let { state } = view, text = node.nodeType == 3 ? node.nodeValue :
          new DOMReader([], state).readRange(node.firstChild, null).text;
      if (newTo - newFrom < text.length) {
          if (state.doc.sliceString(newFrom, Math.min(state.doc.length, newFrom + text.length), LineBreakPlaceholder) == text)
              newTo = newFrom + text.length;
          else if (state.doc.sliceString(Math.max(0, newTo - text.length), newTo, LineBreakPlaceholder) == text)
              newFrom = newTo - text.length;
          else
              return Decoration.none;
      }
      else if (state.doc.sliceString(newFrom, newTo, LineBreakPlaceholder) != text) {
          return Decoration.none;
      }
      let topView = ContentView.get(node);
      if (topView instanceof CompositionView)
          topView = topView.widget.topView;
      else if (topView)
          topView.parent = null;
      return Decoration.set(Decoration.replace({ widget: new CompositionWidget(node, textNode, topView), inclusive: true })
          .range(newFrom, newTo));
  }
  class CompositionWidget extends WidgetType {
      constructor(top, text, topView) {
          super();
          this.top = top;
          this.text = text;
          this.topView = topView;
      }
      eq(other) { return this.top == other.top && this.text == other.text; }
      toDOM() { return this.top; }
      ignoreEvent() { return false; }
      get customView() { return CompositionView; }
  }
  function nearbyTextNode(node, offset, side) {
      for (;;) {
          if (node.nodeType == 3)
              return node;
          if (node.nodeType == 1 && offset > 0 && side <= 0) {
              node = node.childNodes[offset - 1];
              offset = maxOffset(node);
          }
          else if (node.nodeType == 1 && offset < node.childNodes.length && side >= 0) {
              node = node.childNodes[offset];
              offset = 0;
          }
          else {
              return null;
          }
      }
  }
  function nextToUneditable(node, offset) {
      if (node.nodeType != 1)
          return 0;
      return (offset && node.childNodes[offset - 1].contentEditable == "false" ? 1 /* Before */ : 0) |
          (offset < node.childNodes.length && node.childNodes[offset].contentEditable == "false" ? 2 /* After */ : 0);
  }
  class DecorationComparator$1 {
      constructor() {
          this.changes = [];
      }
      compareRange(from, to) { addRange(from, to, this.changes); }
      comparePoint(from, to) { addRange(from, to, this.changes); }
  }
  function findChangedDeco(a, b, diff) {
      let comp = new DecorationComparator$1;
      RangeSet.compare(a, b, diff, comp);
      return comp.changes;
  }
  function inUneditable(node, inside) {
      for (let cur = node; cur && cur != inside; cur = cur.assignedSlot || cur.parentNode) {
          if (cur.nodeType == 1 && cur.contentEditable == 'false') {
              return true;
          }
      }
      return false;
  }

  function groupAt(state, pos, bias = 1) {
      let categorize = state.charCategorizer(pos);
      let line = state.doc.lineAt(pos), linePos = pos - line.from;
      if (line.length == 0)
          return EditorSelection.cursor(pos);
      if (linePos == 0)
          bias = 1;
      else if (linePos == line.length)
          bias = -1;
      let from = linePos, to = linePos;
      if (bias < 0)
          from = findClusterBreak(line.text, linePos, false);
      else
          to = findClusterBreak(line.text, linePos);
      let cat = categorize(line.text.slice(from, to));
      while (from > 0) {
          let prev = findClusterBreak(line.text, from, false);
          if (categorize(line.text.slice(prev, from)) != cat)
              break;
          from = prev;
      }
      while (to < line.length) {
          let next = findClusterBreak(line.text, to);
          if (categorize(line.text.slice(to, next)) != cat)
              break;
          to = next;
      }
      return EditorSelection.range(from + line.from, to + line.from);
  }
  // Search the DOM for the {node, offset} position closest to the given
  // coordinates. Very inefficient and crude, but can usually be avoided
  // by calling caret(Position|Range)FromPoint instead.
  function getdx(x, rect) {
      return rect.left > x ? rect.left - x : Math.max(0, x - rect.right);
  }
  function getdy(y, rect) {
      return rect.top > y ? rect.top - y : Math.max(0, y - rect.bottom);
  }
  function yOverlap(a, b) {
      return a.top < b.bottom - 1 && a.bottom > b.top + 1;
  }
  function upTop(rect, top) {
      return top < rect.top ? { top, left: rect.left, right: rect.right, bottom: rect.bottom } : rect;
  }
  function upBot(rect, bottom) {
      return bottom > rect.bottom ? { top: rect.top, left: rect.left, right: rect.right, bottom } : rect;
  }
  function domPosAtCoords(parent, x, y) {
      let closest, closestRect, closestX, closestY;
      let above, below, aboveRect, belowRect;
      for (let child = parent.firstChild; child; child = child.nextSibling) {
          let rects = clientRectsFor(child);
          for (let i = 0; i < rects.length; i++) {
              let rect = rects[i];
              if (closestRect && yOverlap(closestRect, rect))
                  rect = upTop(upBot(rect, closestRect.bottom), closestRect.top);
              let dx = getdx(x, rect), dy = getdy(y, rect);
              if (dx == 0 && dy == 0)
                  return child.nodeType == 3 ? domPosInText(child, x, y) : domPosAtCoords(child, x, y);
              if (!closest || closestY > dy || closestY == dy && closestX > dx) {
                  closest = child;
                  closestRect = rect;
                  closestX = dx;
                  closestY = dy;
              }
              if (dx == 0) {
                  if (y > rect.bottom && (!aboveRect || aboveRect.bottom < rect.bottom)) {
                      above = child;
                      aboveRect = rect;
                  }
                  else if (y < rect.top && (!belowRect || belowRect.top > rect.top)) {
                      below = child;
                      belowRect = rect;
                  }
              }
              else if (aboveRect && yOverlap(aboveRect, rect)) {
                  aboveRect = upBot(aboveRect, rect.bottom);
              }
              else if (belowRect && yOverlap(belowRect, rect)) {
                  belowRect = upTop(belowRect, rect.top);
              }
          }
      }
      if (aboveRect && aboveRect.bottom >= y) {
          closest = above;
          closestRect = aboveRect;
      }
      else if (belowRect && belowRect.top <= y) {
          closest = below;
          closestRect = belowRect;
      }
      if (!closest)
          return { node: parent, offset: 0 };
      let clipX = Math.max(closestRect.left, Math.min(closestRect.right, x));
      if (closest.nodeType == 3)
          return domPosInText(closest, clipX, y);
      if (!closestX && closest.contentEditable == "true")
          return domPosAtCoords(closest, clipX, y);
      let offset = Array.prototype.indexOf.call(parent.childNodes, closest) +
          (x >= (closestRect.left + closestRect.right) / 2 ? 1 : 0);
      return { node: parent, offset };
  }
  function domPosInText(node, x, y) {
      let len = node.nodeValue.length;
      let closestOffset = -1, closestDY = 1e9, generalSide = 0;
      for (let i = 0; i < len; i++) {
          let rects = textRange(node, i, i + 1).getClientRects();
          for (let j = 0; j < rects.length; j++) {
              let rect = rects[j];
              if (rect.top == rect.bottom)
                  continue;
              if (!generalSide)
                  generalSide = x - rect.left;
              let dy = (rect.top > y ? rect.top - y : y - rect.bottom) - 1;
              if (rect.left - 1 <= x && rect.right + 1 >= x && dy < closestDY) {
                  let right = x >= (rect.left + rect.right) / 2, after = right;
                  if (browser.chrome || browser.gecko) {
                      // Check for RTL on browsers that support getting client
                      // rects for empty ranges.
                      let rectBefore = textRange(node, i).getBoundingClientRect();
                      if (rectBefore.left == rect.right)
                          after = !right;
                  }
                  if (dy <= 0)
                      return { node, offset: i + (after ? 1 : 0) };
                  closestOffset = i + (after ? 1 : 0);
                  closestDY = dy;
              }
          }
      }
      return { node, offset: closestOffset > -1 ? closestOffset : generalSide > 0 ? node.nodeValue.length : 0 };
  }
  function posAtCoords(view, { x, y }, precise, bias = -1) {
      var _a;
      let content = view.contentDOM.getBoundingClientRect(), docTop = content.top + view.viewState.paddingTop;
      let block, { docHeight } = view.viewState;
      let yOffset = y - docTop;
      if (yOffset < 0)
          return 0;
      if (yOffset > docHeight)
          return view.state.doc.length;
      // Scan for a text block near the queried y position
      for (let halfLine = view.defaultLineHeight / 2, bounced = false;;) {
          block = view.elementAtHeight(yOffset);
          if (block.type == BlockType.Text)
              break;
          for (;;) {
              // Move the y position out of this block
              yOffset = bias > 0 ? block.bottom + halfLine : block.top - halfLine;
              if (yOffset >= 0 && yOffset <= docHeight)
                  break;
              // If the document consists entirely of replaced widgets, we
              // won't find a text block, so return 0
              if (bounced)
                  return precise ? null : 0;
              bounced = true;
              bias = -bias;
          }
      }
      y = docTop + yOffset;
      let lineStart = block.from;
      // If this is outside of the rendered viewport, we can't determine a position
      if (lineStart < view.viewport.from)
          return view.viewport.from == 0 ? 0 : precise ? null : posAtCoordsImprecise(view, content, block, x, y);
      if (lineStart > view.viewport.to)
          return view.viewport.to == view.state.doc.length ? view.state.doc.length :
              precise ? null : posAtCoordsImprecise(view, content, block, x, y);
      // Prefer ShadowRootOrDocument.elementFromPoint if present, fall back to document if not
      let doc = view.dom.ownerDocument;
      let root = view.root.elementFromPoint ? view.root : doc;
      let element = root.elementFromPoint(x, y);
      if (element && !view.contentDOM.contains(element))
          element = null;
      // If the element is unexpected, clip x at the sides of the content area and try again
      if (!element) {
          x = Math.max(content.left + 1, Math.min(content.right - 1, x));
          element = root.elementFromPoint(x, y);
          if (element && !view.contentDOM.contains(element))
              element = null;
      }
      // There's visible editor content under the point, so we can try
      // using caret(Position|Range)FromPoint as a shortcut
      let node, offset = -1;
      if (element && ((_a = view.docView.nearest(element)) === null || _a === void 0 ? void 0 : _a.isEditable) != false) {
          if (doc.caretPositionFromPoint) {
              let pos = doc.caretPositionFromPoint(x, y);
              if (pos)
                  ({ offsetNode: node, offset } = pos);
          }
          else if (doc.caretRangeFromPoint) {
              let range = doc.caretRangeFromPoint(x, y);
              if (range) {
                  ({ startContainer: node, startOffset: offset } = range);
                  if (browser.safari && isSuspiciousSafariCaretResult(node, offset, x) ||
                      browser.chrome && isSuspiciousChromeCaretResult(node, offset, x))
                      node = undefined;
              }
          }
      }
      // No luck, do our own (potentially expensive) search
      if (!node || !view.docView.dom.contains(node)) {
          let line = LineView.find(view.docView, lineStart);
          if (!line)
              return yOffset > block.top + block.height / 2 ? block.to : block.from;
          ({ node, offset } = domPosAtCoords(line.dom, x, y));
      }
      return view.docView.posFromDOM(node, offset);
  }
  function posAtCoordsImprecise(view, contentRect, block, x, y) {
      let into = Math.round((x - contentRect.left) * view.defaultCharacterWidth);
      if (view.lineWrapping && block.height > view.defaultLineHeight * 1.5) {
          let line = Math.floor((y - block.top) / view.defaultLineHeight);
          into += line * view.viewState.heightOracle.lineLength;
      }
      let content = view.state.sliceDoc(block.from, block.to);
      return block.from + findColumn(content, into, view.state.tabSize);
  }
  // In case of a high line height, Safari's caretRangeFromPoint treats
  // the space between lines as belonging to the last character of the
  // line before. This is used to detect such a result so that it can be
  // ignored (issue #401).
  function isSuspiciousSafariCaretResult(node, offset, x) {
      let len;
      if (node.nodeType != 3 || offset != (len = node.nodeValue.length))
          return false;
      for (let next = node.nextSibling; next; next = next.nextSibling)
          if (next.nodeType != 1 || next.nodeName != "BR")
              return false;
      return textRange(node, len - 1, len).getBoundingClientRect().left > x;
  }
  // Chrome will move positions between lines to the start of the next line
  function isSuspiciousChromeCaretResult(node, offset, x) {
      if (offset != 0)
          return false;
      for (let cur = node;;) {
          let parent = cur.parentNode;
          if (!parent || parent.nodeType != 1 || parent.firstChild != cur)
              return false;
          if (parent.classList.contains("cm-line"))
              break;
          cur = parent;
      }
      let rect = node.nodeType == 1 ? node.getBoundingClientRect()
          : textRange(node, 0, Math.max(node.nodeValue.length, 1)).getBoundingClientRect();
      return x - rect.left > 5;
  }
  function moveToLineBoundary(view, start, forward, includeWrap) {
      let line = view.state.doc.lineAt(start.head);
      let coords = !includeWrap || !view.lineWrapping ? null
          : view.coordsAtPos(start.assoc < 0 && start.head > line.from ? start.head - 1 : start.head);
      if (coords) {
          let editorRect = view.dom.getBoundingClientRect();
          let direction = view.textDirectionAt(line.from);
          let pos = view.posAtCoords({ x: forward == (direction == Direction.LTR) ? editorRect.right - 1 : editorRect.left + 1,
              y: (coords.top + coords.bottom) / 2 });
          if (pos != null)
              return EditorSelection.cursor(pos, forward ? -1 : 1);
      }
      let lineView = LineView.find(view.docView, start.head);
      let end = lineView ? (forward ? lineView.posAtEnd : lineView.posAtStart) : (forward ? line.to : line.from);
      return EditorSelection.cursor(end, forward ? -1 : 1);
  }
  function moveByChar(view, start, forward, by) {
      let line = view.state.doc.lineAt(start.head), spans = view.bidiSpans(line);
      let direction = view.textDirectionAt(line.from);
      for (let cur = start, check = null;;) {
          let next = moveVisually(line, spans, direction, cur, forward), char = movedOver;
          if (!next) {
              if (line.number == (forward ? view.state.doc.lines : 1))
                  return cur;
              char = "\n";
              line = view.state.doc.line(line.number + (forward ? 1 : -1));
              spans = view.bidiSpans(line);
              next = EditorSelection.cursor(forward ? line.from : line.to);
          }
          if (!check) {
              if (!by)
                  return next;
              check = by(char);
          }
          else if (!check(char)) {
              return cur;
          }
          cur = next;
      }
  }
  function byGroup(view, pos, start) {
      let categorize = view.state.charCategorizer(pos);
      let cat = categorize(start);
      return (next) => {
          let nextCat = categorize(next);
          if (cat == CharCategory.Space)
              cat = nextCat;
          return cat == nextCat;
      };
  }
  function moveVertically(view, start, forward, distance) {
      let startPos = start.head, dir = forward ? 1 : -1;
      if (startPos == (forward ? view.state.doc.length : 0))
          return EditorSelection.cursor(startPos, start.assoc);
      let goal = start.goalColumn, startY;
      let rect = view.contentDOM.getBoundingClientRect();
      let startCoords = view.coordsAtPos(startPos), docTop = view.documentTop;
      if (startCoords) {
          if (goal == null)
              goal = startCoords.left - rect.left;
          startY = dir < 0 ? startCoords.top : startCoords.bottom;
      }
      else {
          let line = view.viewState.lineBlockAt(startPos);
          if (goal == null)
              goal = Math.min(rect.right - rect.left, view.defaultCharacterWidth * (startPos - line.from));
          startY = (dir < 0 ? line.top : line.bottom) + docTop;
      }
      let resolvedGoal = rect.left + goal;
      let dist = distance !== null && distance !== void 0 ? distance : (view.defaultLineHeight >> 1);
      for (let extra = 0;; extra += 10) {
          let curY = startY + (dist + extra) * dir;
          let pos = posAtCoords(view, { x: resolvedGoal, y: curY }, false, dir);
          if (curY < rect.top || curY > rect.bottom || (dir < 0 ? pos < startPos : pos > startPos))
              return EditorSelection.cursor(pos, start.assoc, undefined, goal);
      }
  }
  function skipAtoms(view, oldPos, pos) {
      let atoms = view.state.facet(atomicRanges).map(f => f(view));
      for (;;) {
          let moved = false;
          for (let set of atoms) {
              set.between(pos.from - 1, pos.from + 1, (from, to, value) => {
                  if (pos.from > from && pos.from < to) {
                      pos = oldPos.from > pos.from ? EditorSelection.cursor(from, 1) : EditorSelection.cursor(to, -1);
                      moved = true;
                  }
              });
          }
          if (!moved)
              return pos;
      }
  }

  // This will also be where dragging info and such goes
  class InputState {
      constructor(view) {
          this.lastKeyCode = 0;
          this.lastKeyTime = 0;
          this.lastTouchTime = 0;
          this.lastFocusTime = 0;
          this.lastScrollTop = 0;
          this.lastScrollLeft = 0;
          this.chromeScrollHack = -1;
          // On iOS, some keys need to have their default behavior happen
          // (after which we retroactively handle them and reset the DOM) to
          // avoid messing up the virtual keyboard state.
          this.pendingIOSKey = undefined;
          this.lastSelectionOrigin = null;
          this.lastSelectionTime = 0;
          this.lastEscPress = 0;
          this.lastContextMenu = 0;
          this.scrollHandlers = [];
          this.registeredEvents = [];
          this.customHandlers = [];
          // -1 means not in a composition. Otherwise, this counts the number
          // of changes made during the composition. The count is used to
          // avoid treating the start state of the composition, before any
          // changes have been made, as part of the composition.
          this.composing = -1;
          // Tracks whether the next change should be marked as starting the
          // composition (null means no composition, true means next is the
          // first, false means first has already been marked for this
          // composition)
          this.compositionFirstChange = null;
          this.compositionEndedAt = 0;
          this.rapidCompositionStart = false;
          this.mouseSelection = null;
          for (let type in handlers) {
              let handler = handlers[type];
              view.contentDOM.addEventListener(type, (event) => {
                  if (!eventBelongsToEditor(view, event) || this.ignoreDuringComposition(event))
                      return;
                  if (type == "keydown" && this.keydown(view, event))
                      return;
                  if (this.mustFlushObserver(event))
                      view.observer.forceFlush();
                  if (this.runCustomHandlers(type, view, event))
                      event.preventDefault();
                  else
                      handler(view, event);
              }, handlerOptions[type]);
              this.registeredEvents.push(type);
          }
          if (browser.chrome && browser.chrome_version == 102) { // FIXME remove at some point
              // On Chrome 102, viewport updates somehow stop wheel-based
              // scrolling. Turning off pointer events during the scroll seems
              // to avoid the issue.
              view.scrollDOM.addEventListener("wheel", () => {
                  if (this.chromeScrollHack < 0)
                      view.contentDOM.style.pointerEvents = "none";
                  else
                      window.clearTimeout(this.chromeScrollHack);
                  this.chromeScrollHack = setTimeout(() => {
                      this.chromeScrollHack = -1;
                      view.contentDOM.style.pointerEvents = "";
                  }, 100);
              }, { passive: true });
          }
          this.notifiedFocused = view.hasFocus;
          // On Safari adding an input event handler somehow prevents an
          // issue where the composition vanishes when you press enter.
          if (browser.safari)
              view.contentDOM.addEventListener("input", () => null);
      }
      setSelectionOrigin(origin) {
          this.lastSelectionOrigin = origin;
          this.lastSelectionTime = Date.now();
      }
      ensureHandlers(view, plugins) {
          var _a;
          let handlers;
          this.customHandlers = [];
          for (let plugin of plugins)
              if (handlers = (_a = plugin.update(view).spec) === null || _a === void 0 ? void 0 : _a.domEventHandlers) {
                  this.customHandlers.push({ plugin: plugin.value, handlers });
                  for (let type in handlers)
                      if (this.registeredEvents.indexOf(type) < 0 && type != "scroll") {
                          this.registeredEvents.push(type);
                          view.contentDOM.addEventListener(type, (event) => {
                              if (!eventBelongsToEditor(view, event))
                                  return;
                              if (this.runCustomHandlers(type, view, event))
                                  event.preventDefault();
                          });
                      }
              }
      }
      runCustomHandlers(type, view, event) {
          for (let set of this.customHandlers) {
              let handler = set.handlers[type];
              if (handler) {
                  try {
                      if (handler.call(set.plugin, event, view) || event.defaultPrevented)
                          return true;
                  }
                  catch (e) {
                      logException(view.state, e);
                  }
              }
          }
          return false;
      }
      runScrollHandlers(view, event) {
          this.lastScrollTop = view.scrollDOM.scrollTop;
          this.lastScrollLeft = view.scrollDOM.scrollLeft;
          for (let set of this.customHandlers) {
              let handler = set.handlers.scroll;
              if (handler) {
                  try {
                      handler.call(set.plugin, event, view);
                  }
                  catch (e) {
                      logException(view.state, e);
                  }
              }
          }
      }
      keydown(view, event) {
          // Must always run, even if a custom handler handled the event
          this.lastKeyCode = event.keyCode;
          this.lastKeyTime = Date.now();
          if (event.keyCode == 9 && Date.now() < this.lastEscPress + 2000)
              return true;
          // Chrome for Android usually doesn't fire proper key events, but
          // occasionally does, usually surrounded by a bunch of complicated
          // composition changes. When an enter or backspace key event is
          // seen, hold off on handling DOM events for a bit, and then
          // dispatch it.
          if (browser.android && browser.chrome && !event.synthetic &&
              (event.keyCode == 13 || event.keyCode == 8)) {
              view.observer.delayAndroidKey(event.key, event.keyCode);
              return true;
          }
          // Prevent the default behavior of Enter on iOS makes the
          // virtual keyboard get stuck in the wrong (lowercase)
          // state. So we let it go through, and then, in
          // applyDOMChange, notify key handlers of it and reset to
          // the state they produce.
          let pending;
          if (browser.ios && (pending = PendingKeys.find(key => key.keyCode == event.keyCode)) &&
              !(event.ctrlKey || event.altKey || event.metaKey) && !event.synthetic) {
              this.pendingIOSKey = pending;
              setTimeout(() => this.flushIOSKey(view), 250);
              return true;
          }
          return false;
      }
      flushIOSKey(view) {
          let key = this.pendingIOSKey;
          if (!key)
              return false;
          this.pendingIOSKey = undefined;
          return dispatchKey(view.contentDOM, key.key, key.keyCode);
      }
      ignoreDuringComposition(event) {
          if (!/^key/.test(event.type))
              return false;
          if (this.composing > 0)
              return true;
          // See https://www.stum.de/2016/06/24/handling-ime-events-in-javascript/.
          // On some input method editors (IMEs), the Enter key is used to
          // confirm character selection. On Safari, when Enter is pressed,
          // compositionend and keydown events are sometimes emitted in the
          // wrong order. The key event should still be ignored, even when
          // it happens after the compositionend event.
          if (browser.safari && !browser.ios && Date.now() - this.compositionEndedAt < 100) {
              this.compositionEndedAt = 0;
              return true;
          }
          return false;
      }
      mustFlushObserver(event) {
          return (event.type == "keydown" && event.keyCode != 229) ||
              event.type == "compositionend" && !browser.ios;
      }
      startMouseSelection(mouseSelection) {
          if (this.mouseSelection)
              this.mouseSelection.destroy();
          this.mouseSelection = mouseSelection;
      }
      update(update) {
          if (this.mouseSelection)
              this.mouseSelection.update(update);
          if (update.transactions.length)
              this.lastKeyCode = this.lastSelectionTime = 0;
      }
      destroy() {
          if (this.mouseSelection)
              this.mouseSelection.destroy();
      }
  }
  const PendingKeys = [
      { key: "Backspace", keyCode: 8, inputType: "deleteContentBackward" },
      { key: "Enter", keyCode: 13, inputType: "insertParagraph" },
      { key: "Delete", keyCode: 46, inputType: "deleteContentForward" }
  ];
  // Key codes for modifier keys
  const modifierCodes = [16, 17, 18, 20, 91, 92, 224, 225];
  class MouseSelection {
      constructor(view, startEvent, style, mustSelect) {
          this.view = view;
          this.style = style;
          this.mustSelect = mustSelect;
          this.lastEvent = startEvent;
          let doc = view.contentDOM.ownerDocument;
          doc.addEventListener("mousemove", this.move = this.move.bind(this));
          doc.addEventListener("mouseup", this.up = this.up.bind(this));
          this.extend = startEvent.shiftKey;
          this.multiple = view.state.facet(EditorState.allowMultipleSelections) && addsSelectionRange(view, startEvent);
          this.dragMove = dragMovesSelection(view, startEvent);
          this.dragging = isInPrimarySelection(view, startEvent) && getClickType(startEvent) == 1 ? null : false;
          // When clicking outside of the selection, immediately apply the
          // effect of starting the selection
          if (this.dragging === false) {
              startEvent.preventDefault();
              this.select(startEvent);
          }
      }
      move(event) {
          if (event.buttons == 0)
              return this.destroy();
          if (this.dragging !== false)
              return;
          this.select(this.lastEvent = event);
      }
      up(event) {
          if (this.dragging == null)
              this.select(this.lastEvent);
          if (!this.dragging)
              event.preventDefault();
          this.destroy();
      }
      destroy() {
          let doc = this.view.contentDOM.ownerDocument;
          doc.removeEventListener("mousemove", this.move);
          doc.removeEventListener("mouseup", this.up);
          this.view.inputState.mouseSelection = null;
      }
      select(event) {
          let selection = this.style.get(event, this.extend, this.multiple);
          if (this.mustSelect || !selection.eq(this.view.state.selection) ||
              selection.main.assoc != this.view.state.selection.main.assoc)
              this.view.dispatch({
                  selection,
                  userEvent: "select.pointer",
                  scrollIntoView: true
              });
          this.mustSelect = false;
      }
      update(update) {
          if (update.docChanged && this.dragging)
              this.dragging = this.dragging.map(update.changes);
          if (this.style.update(update))
              setTimeout(() => this.select(this.lastEvent), 20);
      }
  }
  function addsSelectionRange(view, event) {
      let facet = view.state.facet(clickAddsSelectionRange);
      return facet.length ? facet[0](event) : browser.mac ? event.metaKey : event.ctrlKey;
  }
  function dragMovesSelection(view, event) {
      let facet = view.state.facet(dragMovesSelection$1);
      return facet.length ? facet[0](event) : browser.mac ? !event.altKey : !event.ctrlKey;
  }
  function isInPrimarySelection(view, event) {
      let { main } = view.state.selection;
      if (main.empty)
          return false;
      // On boundary clicks, check whether the coordinates are inside the
      // selection's client rectangles
      let sel = getSelection(view.root);
      if (!sel || sel.rangeCount == 0)
          return true;
      let rects = sel.getRangeAt(0).getClientRects();
      for (let i = 0; i < rects.length; i++) {
          let rect = rects[i];
          if (rect.left <= event.clientX && rect.right >= event.clientX &&
              rect.top <= event.clientY && rect.bottom >= event.clientY)
              return true;
      }
      return false;
  }
  function eventBelongsToEditor(view, event) {
      if (!event.bubbles)
          return true;
      if (event.defaultPrevented)
          return false;
      for (let node = event.target, cView; node != view.contentDOM; node = node.parentNode)
          if (!node || node.nodeType == 11 || ((cView = ContentView.get(node)) && cView.ignoreEvent(event)))
              return false;
      return true;
  }
  const handlers = /*@__PURE__*/Object.create(null);
  const handlerOptions = /*@__PURE__*/Object.create(null);
  // This is very crude, but unfortunately both these browsers _pretend_
  // that they have a clipboard API—all the objects and methods are
  // there, they just don't work, and they are hard to test.
  const brokenClipboardAPI = (browser.ie && browser.ie_version < 15) ||
      (browser.ios && browser.webkit_version < 604);
  function capturePaste(view) {
      let parent = view.dom.parentNode;
      if (!parent)
          return;
      let target = parent.appendChild(document.createElement("textarea"));
      target.style.cssText = "position: fixed; left: -10000px; top: 10px";
      target.focus();
      setTimeout(() => {
          view.focus();
          target.remove();
          doPaste(view, target.value);
      }, 50);
  }
  function doPaste(view, input) {
      let { state } = view, changes, i = 1, text = state.toText(input);
      let byLine = text.lines == state.selection.ranges.length;
      let linewise = lastLinewiseCopy != null && state.selection.ranges.every(r => r.empty) && lastLinewiseCopy == text.toString();
      if (linewise) {
          let lastLine = -1;
          changes = state.changeByRange(range => {
              let line = state.doc.lineAt(range.from);
              if (line.from == lastLine)
                  return { range };
              lastLine = line.from;
              let insert = state.toText((byLine ? text.line(i++).text : input) + state.lineBreak);
              return { changes: { from: line.from, insert },
                  range: EditorSelection.cursor(range.from + insert.length) };
          });
      }
      else if (byLine) {
          changes = state.changeByRange(range => {
              let line = text.line(i++);
              return { changes: { from: range.from, to: range.to, insert: line.text },
                  range: EditorSelection.cursor(range.from + line.length) };
          });
      }
      else {
          changes = state.replaceSelection(text);
      }
      view.dispatch(changes, {
          userEvent: "input.paste",
          scrollIntoView: true
      });
  }
  handlers.keydown = (view, event) => {
      view.inputState.setSelectionOrigin("select");
      if (event.keyCode == 27)
          view.inputState.lastEscPress = Date.now();
      else if (modifierCodes.indexOf(event.keyCode) < 0)
          view.inputState.lastEscPress = 0;
  };
  handlers.touchstart = (view, e) => {
      view.inputState.lastTouchTime = Date.now();
      view.inputState.setSelectionOrigin("select.pointer");
  };
  handlers.touchmove = view => {
      view.inputState.setSelectionOrigin("select.pointer");
  };
  handlerOptions.touchstart = handlerOptions.touchmove = { passive: true };
  handlers.mousedown = (view, event) => {
      view.observer.flush();
      if (view.inputState.lastTouchTime > Date.now() - 2000 && getClickType(event) == 1)
          return; // Ignore touch interaction
      let style = null;
      for (let makeStyle of view.state.facet(mouseSelectionStyle)) {
          style = makeStyle(view, event);
          if (style)
              break;
      }
      if (!style && event.button == 0)
          style = basicMouseSelection(view, event);
      if (style) {
          let mustFocus = view.root.activeElement != view.contentDOM;
          if (mustFocus)
              view.observer.ignore(() => focusPreventScroll(view.contentDOM));
          view.inputState.startMouseSelection(new MouseSelection(view, event, style, mustFocus));
      }
  };
  function rangeForClick(view, pos, bias, type) {
      if (type == 1) { // Single click
          return EditorSelection.cursor(pos, bias);
      }
      else if (type == 2) { // Double click
          return groupAt(view.state, pos, bias);
      }
      else { // Triple click
          let visual = LineView.find(view.docView, pos), line = view.state.doc.lineAt(visual ? visual.posAtEnd : pos);
          let from = visual ? visual.posAtStart : line.from, to = visual ? visual.posAtEnd : line.to;
          if (to < view.state.doc.length && to == line.to)
              to++;
          return EditorSelection.range(from, to);
      }
  }
  let insideY = (y, rect) => y >= rect.top && y <= rect.bottom;
  let inside = (x, y, rect) => insideY(y, rect) && x >= rect.left && x <= rect.right;
  // Try to determine, for the given coordinates, associated with the
  // given position, whether they are related to the element before or
  // the element after the position.
  function findPositionSide(view, pos, x, y) {
      let line = LineView.find(view.docView, pos);
      if (!line)
          return 1;
      let off = pos - line.posAtStart;
      // Line boundaries point into the line
      if (off == 0)
          return 1;
      if (off == line.length)
          return -1;
      // Positions on top of an element point at that element
      let before = line.coordsAt(off, -1);
      if (before && inside(x, y, before))
          return -1;
      let after = line.coordsAt(off, 1);
      if (after && inside(x, y, after))
          return 1;
      // This is probably a line wrap point. Pick before if the point is
      // beside it.
      return before && insideY(y, before) ? -1 : 1;
  }
  function queryPos(view, event) {
      let pos = view.posAtCoords({ x: event.clientX, y: event.clientY }, false);
      return { pos, bias: findPositionSide(view, pos, event.clientX, event.clientY) };
  }
  const BadMouseDetail = browser.ie && browser.ie_version <= 11;
  let lastMouseDown = null, lastMouseDownCount = 0, lastMouseDownTime = 0;
  function getClickType(event) {
      if (!BadMouseDetail)
          return event.detail;
      let last = lastMouseDown, lastTime = lastMouseDownTime;
      lastMouseDown = event;
      lastMouseDownTime = Date.now();
      return lastMouseDownCount = !last || (lastTime > Date.now() - 400 && Math.abs(last.clientX - event.clientX) < 2 &&
          Math.abs(last.clientY - event.clientY) < 2) ? (lastMouseDownCount + 1) % 3 : 1;
  }
  function basicMouseSelection(view, event) {
      let start = queryPos(view, event), type = getClickType(event);
      let startSel = view.state.selection;
      let last = start, lastEvent = event;
      return {
          update(update) {
              if (update.docChanged) {
                  if (start)
                      start.pos = update.changes.mapPos(start.pos);
                  startSel = startSel.map(update.changes);
                  lastEvent = null;
              }
          },
          get(event, extend, multiple) {
              let cur;
              if (lastEvent && event.clientX == lastEvent.clientX && event.clientY == lastEvent.clientY)
                  cur = last;
              else {
                  cur = last = queryPos(view, event);
                  lastEvent = event;
              }
              if (!cur || !start)
                  return startSel;
              let range = rangeForClick(view, cur.pos, cur.bias, type);
              if (start.pos != cur.pos && !extend) {
                  let startRange = rangeForClick(view, start.pos, start.bias, type);
                  let from = Math.min(startRange.from, range.from), to = Math.max(startRange.to, range.to);
                  range = from < range.from ? EditorSelection.range(from, to) : EditorSelection.range(to, from);
              }
              if (extend)
                  return startSel.replaceRange(startSel.main.extend(range.from, range.to));
              else if (multiple && startSel.ranges.length > 1 && startSel.ranges.some(r => r.eq(range)))
                  return removeRange(startSel, range);
              else if (multiple)
                  return startSel.addRange(range);
              else
                  return EditorSelection.create([range]);
          }
      };
  }
  function removeRange(sel, range) {
      for (let i = 0;; i++) {
          if (sel.ranges[i].eq(range))
              return EditorSelection.create(sel.ranges.slice(0, i).concat(sel.ranges.slice(i + 1)), sel.mainIndex == i ? 0 : sel.mainIndex - (sel.mainIndex > i ? 1 : 0));
      }
  }
  handlers.dragstart = (view, event) => {
      let { selection: { main } } = view.state;
      let { mouseSelection } = view.inputState;
      if (mouseSelection)
          mouseSelection.dragging = main;
      if (event.dataTransfer) {
          event.dataTransfer.setData("Text", view.state.sliceDoc(main.from, main.to));
          event.dataTransfer.effectAllowed = "copyMove";
      }
  };
  function dropText(view, event, text, direct) {
      if (!text)
          return;
      let dropPos = view.posAtCoords({ x: event.clientX, y: event.clientY }, false);
      event.preventDefault();
      let { mouseSelection } = view.inputState;
      let del = direct && mouseSelection && mouseSelection.dragging && mouseSelection.dragMove ?
          { from: mouseSelection.dragging.from, to: mouseSelection.dragging.to } : null;
      let ins = { from: dropPos, insert: text };
      let changes = view.state.changes(del ? [del, ins] : ins);
      view.focus();
      view.dispatch({
          changes,
          selection: { anchor: changes.mapPos(dropPos, -1), head: changes.mapPos(dropPos, 1) },
          userEvent: del ? "move.drop" : "input.drop"
      });
  }
  handlers.drop = (view, event) => {
      if (!event.dataTransfer)
          return;
      if (view.state.readOnly)
          return event.preventDefault();
      let files = event.dataTransfer.files;
      if (files && files.length) { // For a file drop, read the file's text.
          event.preventDefault();
          let text = Array(files.length), read = 0;
          let finishFile = () => {
              if (++read == files.length)
                  dropText(view, event, text.filter(s => s != null).join(view.state.lineBreak), false);
          };
          for (let i = 0; i < files.length; i++) {
              let reader = new FileReader;
              reader.onerror = finishFile;
              reader.onload = () => {
                  if (!/[\x00-\x08\x0e-\x1f]{2}/.test(reader.result))
                      text[i] = reader.result;
                  finishFile();
              };
              reader.readAsText(files[i]);
          }
      }
      else {
          dropText(view, event, event.dataTransfer.getData("Text"), true);
      }
  };
  handlers.paste = (view, event) => {
      if (view.state.readOnly)
          return event.preventDefault();
      view.observer.flush();
      let data = brokenClipboardAPI ? null : event.clipboardData;
      if (data) {
          doPaste(view, data.getData("text/plain"));
          event.preventDefault();
      }
      else {
          capturePaste(view);
      }
  };
  function captureCopy(view, text) {
      // The extra wrapper is somehow necessary on IE/Edge to prevent the
      // content from being mangled when it is put onto the clipboard
      let parent = view.dom.parentNode;
      if (!parent)
          return;
      let target = parent.appendChild(document.createElement("textarea"));
      target.style.cssText = "position: fixed; left: -10000px; top: 10px";
      target.value = text;
      target.focus();
      target.selectionEnd = text.length;
      target.selectionStart = 0;
      setTimeout(() => {
          target.remove();
          view.focus();
      }, 50);
  }
  function copiedRange(state) {
      let content = [], ranges = [], linewise = false;
      for (let range of state.selection.ranges)
          if (!range.empty) {
              content.push(state.sliceDoc(range.from, range.to));
              ranges.push(range);
          }
      if (!content.length) {
          // Nothing selected, do a line-wise copy
          let upto = -1;
          for (let { from } of state.selection.ranges) {
              let line = state.doc.lineAt(from);
              if (line.number > upto) {
                  content.push(line.text);
                  ranges.push({ from: line.from, to: Math.min(state.doc.length, line.to + 1) });
              }
              upto = line.number;
          }
          linewise = true;
      }
      return { text: content.join(state.lineBreak), ranges, linewise };
  }
  let lastLinewiseCopy = null;
  handlers.copy = handlers.cut = (view, event) => {
      let { text, ranges, linewise } = copiedRange(view.state);
      if (!text && !linewise)
          return;
      lastLinewiseCopy = linewise ? text : null;
      let data = brokenClipboardAPI ? null : event.clipboardData;
      if (data) {
          event.preventDefault();
          data.clearData();
          data.setData("text/plain", text);
      }
      else {
          captureCopy(view, text);
      }
      if (event.type == "cut" && !view.state.readOnly)
          view.dispatch({
              changes: ranges,
              scrollIntoView: true,
              userEvent: "delete.cut"
          });
  };
  function updateForFocusChange(view) {
      setTimeout(() => {
          if (view.hasFocus != view.inputState.notifiedFocused)
              view.update([]);
      }, 10);
  }
  handlers.focus = view => {
      view.inputState.lastFocusTime = Date.now();
      // When focusing reset the scroll position, move it back to where it was
      if (!view.scrollDOM.scrollTop && (view.inputState.lastScrollTop || view.inputState.lastScrollLeft)) {
          view.scrollDOM.scrollTop = view.inputState.lastScrollTop;
          view.scrollDOM.scrollLeft = view.inputState.lastScrollLeft;
      }
      updateForFocusChange(view);
  };
  handlers.blur = view => {
      view.observer.clearSelectionRange();
      updateForFocusChange(view);
  };
  function forceClearComposition(view, rapid) {
      if (view.docView.compositionDeco.size) {
          view.inputState.rapidCompositionStart = rapid;
          try {
              view.update([]);
          }
          finally {
              view.inputState.rapidCompositionStart = false;
          }
      }
  }
  handlers.compositionstart = handlers.compositionupdate = view => {
      if (view.inputState.compositionFirstChange == null)
          view.inputState.compositionFirstChange = true;
      if (view.inputState.composing < 0) {
          // FIXME possibly set a timeout to clear it again on Android
          view.inputState.composing = 0;
          if (view.docView.compositionDeco.size) {
              view.observer.flush();
              forceClearComposition(view, true);
          }
      }
  };
  handlers.compositionend = view => {
      view.inputState.composing = -1;
      view.inputState.compositionEndedAt = Date.now();
      view.inputState.compositionFirstChange = null;
      setTimeout(() => {
          if (view.inputState.composing < 0)
              forceClearComposition(view, false);
      }, 50);
  };
  handlers.contextmenu = view => {
      view.inputState.lastContextMenu = Date.now();
  };
  handlers.beforeinput = (view, event) => {
      var _a;
      // Because Chrome Android doesn't fire useful key events, use
      // beforeinput to detect backspace (and possibly enter and delete,
      // but those usually don't even seem to fire beforeinput events at
      // the moment) and fake a key event for it.
      //
      // (preventDefault on beforeinput, though supported in the spec,
      // seems to do nothing at all on Chrome).
      let pending;
      if (browser.chrome && browser.android && (pending = PendingKeys.find(key => key.inputType == event.inputType))) {
          view.observer.delayAndroidKey(pending.key, pending.keyCode);
          if (pending.key == "Backspace" || pending.key == "Delete") {
              let startViewHeight = ((_a = window.visualViewport) === null || _a === void 0 ? void 0 : _a.height) || 0;
              setTimeout(() => {
                  var _a;
                  // Backspacing near uneditable nodes on Chrome Android sometimes
                  // closes the virtual keyboard. This tries to crudely detect
                  // that and refocus to get it back.
                  if ((((_a = window.visualViewport) === null || _a === void 0 ? void 0 : _a.height) || 0) > startViewHeight + 10 && view.hasFocus) {
                      view.contentDOM.blur();
                      view.focus();
                  }
              }, 100);
          }
      }
  };

  const wrappingWhiteSpace = ["pre-wrap", "normal", "pre-line", "break-spaces"];
  class HeightOracle {
      constructor() {
          this.doc = Text.empty;
          this.lineWrapping = false;
          this.heightSamples = {};
          this.lineHeight = 14;
          this.charWidth = 7;
          this.lineLength = 30;
          // Used to track, during updateHeight, if any actual heights changed
          this.heightChanged = false;
      }
      heightForGap(from, to) {
          let lines = this.doc.lineAt(to).number - this.doc.lineAt(from).number + 1;
          if (this.lineWrapping)
              lines += Math.ceil(((to - from) - (lines * this.lineLength * 0.5)) / this.lineLength);
          return this.lineHeight * lines;
      }
      heightForLine(length) {
          if (!this.lineWrapping)
              return this.lineHeight;
          let lines = 1 + Math.max(0, Math.ceil((length - this.lineLength) / (this.lineLength - 5)));
          return lines * this.lineHeight;
      }
      setDoc(doc) { this.doc = doc; return this; }
      mustRefreshForWrapping(whiteSpace) {
          return (wrappingWhiteSpace.indexOf(whiteSpace) > -1) != this.lineWrapping;
      }
      mustRefreshForHeights(lineHeights) {
          let newHeight = false;
          for (let i = 0; i < lineHeights.length; i++) {
              let h = lineHeights[i];
              if (h < 0) {
                  i++;
              }
              else if (!this.heightSamples[Math.floor(h * 10)]) { // Round to .1 pixels
                  newHeight = true;
                  this.heightSamples[Math.floor(h * 10)] = true;
              }
          }
          return newHeight;
      }
      refresh(whiteSpace, lineHeight, charWidth, lineLength, knownHeights) {
          let lineWrapping = wrappingWhiteSpace.indexOf(whiteSpace) > -1;
          let changed = Math.round(lineHeight) != Math.round(this.lineHeight) || this.lineWrapping != lineWrapping;
          this.lineWrapping = lineWrapping;
          this.lineHeight = lineHeight;
          this.charWidth = charWidth;
          this.lineLength = lineLength;
          if (changed) {
              this.heightSamples = {};
              for (let i = 0; i < knownHeights.length; i++) {
                  let h = knownHeights[i];
                  if (h < 0)
                      i++;
                  else
                      this.heightSamples[Math.floor(h * 10)] = true;
              }
          }
          return changed;
      }
  }
  // This object is used by `updateHeight` to make DOM measurements
  // arrive at the right nides. The `heights` array is a sequence of
  // block heights, starting from position `from`.
  class MeasuredHeights {
      constructor(from, heights) {
          this.from = from;
          this.heights = heights;
          this.index = 0;
      }
      get more() { return this.index < this.heights.length; }
  }
  /**
  Record used to represent information about a block-level element
  in the editor view.
  */
  class BlockInfo {
      /**
      @internal
      */
      constructor(
      /**
      The start of the element in the document.
      */
      from, 
      /**
      The length of the element.
      */
      length, 
      /**
      The top position of the element (relative to the top of the
      document).
      */
      top, 
      /**
      Its height.
      */
      height, 
      /**
      The type of element this is. When querying lines, this may be
      an array of all the blocks that make up the line.
      */
      type) {
          this.from = from;
          this.length = length;
          this.top = top;
          this.height = height;
          this.type = type;
      }
      /**
      The end of the element as a document position.
      */
      get to() { return this.from + this.length; }
      /**
      The bottom position of the element.
      */
      get bottom() { return this.top + this.height; }
      /**
      @internal
      */
      join(other) {
          let detail = (Array.isArray(this.type) ? this.type : [this])
              .concat(Array.isArray(other.type) ? other.type : [other]);
          return new BlockInfo(this.from, this.length + other.length, this.top, this.height + other.height, detail);
      }
  }
  var QueryType$1 = /*@__PURE__*/(function (QueryType) {
      QueryType[QueryType["ByPos"] = 0] = "ByPos";
      QueryType[QueryType["ByHeight"] = 1] = "ByHeight";
      QueryType[QueryType["ByPosNoHeight"] = 2] = "ByPosNoHeight";
  return QueryType})(QueryType$1 || (QueryType$1 = {}));
  const Epsilon = 1e-3;
  class HeightMap {
      constructor(length, // The number of characters covered
      height, // Height of this part of the document
      flags = 2 /* Outdated */) {
          this.length = length;
          this.height = height;
          this.flags = flags;
      }
      get outdated() { return (this.flags & 2 /* Outdated */) > 0; }
      set outdated(value) { this.flags = (value ? 2 /* Outdated */ : 0) | (this.flags & ~2 /* Outdated */); }
      setHeight(oracle, height) {
          if (this.height != height) {
              if (Math.abs(this.height - height) > Epsilon)
                  oracle.heightChanged = true;
              this.height = height;
          }
      }
      // Base case is to replace a leaf node, which simply builds a tree
      // from the new nodes and returns that (HeightMapBranch and
      // HeightMapGap override this to actually use from/to)
      replace(_from, _to, nodes) {
          return HeightMap.of(nodes);
      }
      // Again, these are base cases, and are overridden for branch and gap nodes.
      decomposeLeft(_to, result) { result.push(this); }
      decomposeRight(_from, result) { result.push(this); }
      applyChanges(decorations, oldDoc, oracle, changes) {
          let me = this;
          for (let i = changes.length - 1; i >= 0; i--) {
              let { fromA, toA, fromB, toB } = changes[i];
              let start = me.lineAt(fromA, QueryType$1.ByPosNoHeight, oldDoc, 0, 0);
              let end = start.to >= toA ? start : me.lineAt(toA, QueryType$1.ByPosNoHeight, oldDoc, 0, 0);
              toB += end.to - toA;
              toA = end.to;
              while (i > 0 && start.from <= changes[i - 1].toA) {
                  fromA = changes[i - 1].fromA;
                  fromB = changes[i - 1].fromB;
                  i--;
                  if (fromA < start.from)
                      start = me.lineAt(fromA, QueryType$1.ByPosNoHeight, oldDoc, 0, 0);
              }
              fromB += start.from - fromA;
              fromA = start.from;
              let nodes = NodeBuilder.build(oracle, decorations, fromB, toB);
              me = me.replace(fromA, toA, nodes);
          }
          return me.updateHeight(oracle, 0);
      }
      static empty() { return new HeightMapText(0, 0); }
      // nodes uses null values to indicate the position of line breaks.
      // There are never line breaks at the start or end of the array, or
      // two line breaks next to each other, and the array isn't allowed
      // to be empty (same restrictions as return value from the builder).
      static of(nodes) {
          if (nodes.length == 1)
              return nodes[0];
          let i = 0, j = nodes.length, before = 0, after = 0;
          for (;;) {
              if (i == j) {
                  if (before > after * 2) {
                      let split = nodes[i - 1];
                      if (split.break)
                          nodes.splice(--i, 1, split.left, null, split.right);
                      else
                          nodes.splice(--i, 1, split.left, split.right);
                      j += 1 + split.break;
                      before -= split.size;
                  }
                  else if (after > before * 2) {
                      let split = nodes[j];
                      if (split.break)
                          nodes.splice(j, 1, split.left, null, split.right);
                      else
                          nodes.splice(j, 1, split.left, split.right);
                      j += 2 + split.break;
                      after -= split.size;
                  }
                  else {
                      break;
                  }
              }
              else if (before < after) {
                  let next = nodes[i++];
                  if (next)
                      before += next.size;
              }
              else {
                  let next = nodes[--j];
                  if (next)
                      after += next.size;
              }
          }
          let brk = 0;
          if (nodes[i - 1] == null) {
              brk = 1;
              i--;
          }
          else if (nodes[i] == null) {
              brk = 1;
              j++;
          }
          return new HeightMapBranch(HeightMap.of(nodes.slice(0, i)), brk, HeightMap.of(nodes.slice(j)));
      }
  }
  HeightMap.prototype.size = 1;
  class HeightMapBlock extends HeightMap {
      constructor(length, height, type) {
          super(length, height);
          this.type = type;
      }
      blockAt(_height, _doc, top, offset) {
          return new BlockInfo(offset, this.length, top, this.height, this.type);
      }
      lineAt(_value, _type, doc, top, offset) {
          return this.blockAt(0, doc, top, offset);
      }
      forEachLine(from, to, doc, top, offset, f) {
          if (from <= offset + this.length && to >= offset)
              f(this.blockAt(0, doc, top, offset));
      }
      updateHeight(oracle, offset = 0, _force = false, measured) {
          if (measured && measured.from <= offset && measured.more)
              this.setHeight(oracle, measured.heights[measured.index++]);
          this.outdated = false;
          return this;
      }
      toString() { return `block(${this.length})`; }
  }
  class HeightMapText extends HeightMapBlock {
      constructor(length, height) {
          super(length, height, BlockType.Text);
          this.collapsed = 0; // Amount of collapsed content in the line
          this.widgetHeight = 0; // Maximum inline widget height
      }
      replace(_from, _to, nodes) {
          let node = nodes[0];
          if (nodes.length == 1 && (node instanceof HeightMapText || node instanceof HeightMapGap && (node.flags & 4 /* SingleLine */)) &&
              Math.abs(this.length - node.length) < 10) {
              if (node instanceof HeightMapGap)
                  node = new HeightMapText(node.length, this.height);
              else
                  node.height = this.height;
              if (!this.outdated)
                  node.outdated = false;
              return node;
          }
          else {
              return HeightMap.of(nodes);
          }
      }
      updateHeight(oracle, offset = 0, force = false, measured) {
          if (measured && measured.from <= offset && measured.more)
              this.setHeight(oracle, measured.heights[measured.index++]);
          else if (force || this.outdated)
              this.setHeight(oracle, Math.max(this.widgetHeight, oracle.heightForLine(this.length - this.collapsed)));
          this.outdated = false;
          return this;
      }
      toString() {
          return `line(${this.length}${this.collapsed ? -this.collapsed : ""}${this.widgetHeight ? ":" + this.widgetHeight : ""})`;
      }
  }
  class HeightMapGap extends HeightMap {
      constructor(length) { super(length, 0); }
      lines(doc, offset) {
          let firstLine = doc.lineAt(offset).number, lastLine = doc.lineAt(offset + this.length).number;
          return { firstLine, lastLine, lineHeight: this.height / (lastLine - firstLine + 1) };
      }
      blockAt(height, doc, top, offset) {
          let { firstLine, lastLine, lineHeight } = this.lines(doc, offset);
          let line = Math.max(0, Math.min(lastLine - firstLine, Math.floor((height - top) / lineHeight)));
          let { from, length } = doc.line(firstLine + line);
          return new BlockInfo(from, length, top + lineHeight * line, lineHeight, BlockType.Text);
      }
      lineAt(value, type, doc, top, offset) {
          if (type == QueryType$1.ByHeight)
              return this.blockAt(value, doc, top, offset);
          if (type == QueryType$1.ByPosNoHeight) {
              let { from, to } = doc.lineAt(value);
              return new BlockInfo(from, to - from, 0, 0, BlockType.Text);
          }
          let { firstLine, lineHeight } = this.lines(doc, offset);
          let { from, length, number } = doc.lineAt(value);
          return new BlockInfo(from, length, top + lineHeight * (number - firstLine), lineHeight, BlockType.Text);
      }
      forEachLine(from, to, doc, top, offset, f) {
          let { firstLine, lineHeight } = this.lines(doc, offset);
          for (let pos = Math.max(from, offset), end = Math.min(offset + this.length, to); pos <= end;) {
              let line = doc.lineAt(pos);
              if (pos == from)
                  top += lineHeight * (line.number - firstLine);
              f(new BlockInfo(line.from, line.length, top, lineHeight, BlockType.Text));
              top += lineHeight;
              pos = line.to + 1;
          }
      }
      replace(from, to, nodes) {
          let after = this.length - to;
          if (after > 0) {
              let last = nodes[nodes.length - 1];
              if (last instanceof HeightMapGap)
                  nodes[nodes.length - 1] = new HeightMapGap(last.length + after);
              else
                  nodes.push(null, new HeightMapGap(after - 1));
          }
          if (from > 0) {
              let first = nodes[0];
              if (first instanceof HeightMapGap)
                  nodes[0] = new HeightMapGap(from + first.length);
              else
                  nodes.unshift(new HeightMapGap(from - 1), null);
          }
          return HeightMap.of(nodes);
      }
      decomposeLeft(to, result) {
          result.push(new HeightMapGap(to - 1), null);
      }
      decomposeRight(from, result) {
          result.push(null, new HeightMapGap(this.length - from - 1));
      }
      updateHeight(oracle, offset = 0, force = false, measured) {
          let end = offset + this.length;
          if (measured && measured.from <= offset + this.length && measured.more) {
              // Fill in part of this gap with measured lines. We know there
              // can't be widgets or collapsed ranges in those lines, because
              // they would already have been added to the heightmap (gaps
              // only contain plain text).
              let nodes = [], pos = Math.max(offset, measured.from), singleHeight = -1;
              let wasChanged = oracle.heightChanged;
              if (measured.from > offset)
                  nodes.push(new HeightMapGap(measured.from - offset - 1).updateHeight(oracle, offset));
              while (pos <= end && measured.more) {
                  let len = oracle.doc.lineAt(pos).length;
                  if (nodes.length)
                      nodes.push(null);
                  let height = measured.heights[measured.index++];
                  if (singleHeight == -1)
                      singleHeight = height;
                  else if (Math.abs(height - singleHeight) >= Epsilon)
                      singleHeight = -2;
                  let line = new HeightMapText(len, height);
                  line.outdated = false;
                  nodes.push(line);
                  pos += len + 1;
              }
              if (pos <= end)
                  nodes.push(null, new HeightMapGap(end - pos).updateHeight(oracle, pos));
              let result = HeightMap.of(nodes);
              oracle.heightChanged = wasChanged || singleHeight < 0 || Math.abs(result.height - this.height) >= Epsilon ||
                  Math.abs(singleHeight - this.lines(oracle.doc, offset).lineHeight) >= Epsilon;
              return result;
          }
          else if (force || this.outdated) {
              this.setHeight(oracle, oracle.heightForGap(offset, offset + this.length));
              this.outdated = false;
          }
          return this;
      }
      toString() { return `gap(${this.length})`; }
  }
  class HeightMapBranch extends HeightMap {
      constructor(left, brk, right) {
          super(left.length + brk + right.length, left.height + right.height, brk | (left.outdated || right.outdated ? 2 /* Outdated */ : 0));
          this.left = left;
          this.right = right;
          this.size = left.size + right.size;
      }
      get break() { return this.flags & 1 /* Break */; }
      blockAt(height, doc, top, offset) {
          let mid = top + this.left.height;
          return height < mid ? this.left.blockAt(height, doc, top, offset)
              : this.right.blockAt(height, doc, mid, offset + this.left.length + this.break);
      }
      lineAt(value, type, doc, top, offset) {
          let rightTop = top + this.left.height, rightOffset = offset + this.left.length + this.break;
          let left = type == QueryType$1.ByHeight ? value < rightTop : value < rightOffset;
          let base = left ? this.left.lineAt(value, type, doc, top, offset)
              : this.right.lineAt(value, type, doc, rightTop, rightOffset);
          if (this.break || (left ? base.to < rightOffset : base.from > rightOffset))
              return base;
          let subQuery = type == QueryType$1.ByPosNoHeight ? QueryType$1.ByPosNoHeight : QueryType$1.ByPos;
          if (left)
              return base.join(this.right.lineAt(rightOffset, subQuery, doc, rightTop, rightOffset));
          else
              return this.left.lineAt(rightOffset, subQuery, doc, top, offset).join(base);
      }
      forEachLine(from, to, doc, top, offset, f) {
          let rightTop = top + this.left.height, rightOffset = offset + this.left.length + this.break;
          if (this.break) {
              if (from < rightOffset)
                  this.left.forEachLine(from, to, doc, top, offset, f);
              if (to >= rightOffset)
                  this.right.forEachLine(from, to, doc, rightTop, rightOffset, f);
          }
          else {
              let mid = this.lineAt(rightOffset, QueryType$1.ByPos, doc, top, offset);
              if (from < mid.from)
                  this.left.forEachLine(from, mid.from - 1, doc, top, offset, f);
              if (mid.to >= from && mid.from <= to)
                  f(mid);
              if (to > mid.to)
                  this.right.forEachLine(mid.to + 1, to, doc, rightTop, rightOffset, f);
          }
      }
      replace(from, to, nodes) {
          let rightStart = this.left.length + this.break;
          if (to < rightStart)
              return this.balanced(this.left.replace(from, to, nodes), this.right);
          if (from > this.left.length)
              return this.balanced(this.left, this.right.replace(from - rightStart, to - rightStart, nodes));
          let result = [];
          if (from > 0)
              this.decomposeLeft(from, result);
          let left = result.length;
          for (let node of nodes)
              result.push(node);
          if (from > 0)
              mergeGaps(result, left - 1);
          if (to < this.length) {
              let right = result.length;
              this.decomposeRight(to, result);
              mergeGaps(result, right);
          }
          return HeightMap.of(result);
      }
      decomposeLeft(to, result) {
          let left = this.left.length;
          if (to <= left)
              return this.left.decomposeLeft(to, result);
          result.push(this.left);
          if (this.break) {
              left++;
              if (to >= left)
                  result.push(null);
          }
          if (to > left)
              this.right.decomposeLeft(to - left, result);
      }
      decomposeRight(from, result) {
          let left = this.left.length, right = left + this.break;
          if (from >= right)
              return this.right.decomposeRight(from - right, result);
          if (from < left)
              this.left.decomposeRight(from, result);
          if (this.break && from < right)
              result.push(null);
          result.push(this.right);
      }
      balanced(left, right) {
          if (left.size > 2 * right.size || right.size > 2 * left.size)
              return HeightMap.of(this.break ? [left, null, right] : [left, right]);
          this.left = left;
          this.right = right;
          this.height = left.height + right.height;
          this.outdated = left.outdated || right.outdated;
          this.size = left.size + right.size;
          this.length = left.length + this.break + right.length;
          return this;
      }
      updateHeight(oracle, offset = 0, force = false, measured) {
          let { left, right } = this, rightStart = offset + left.length + this.break, rebalance = null;
          if (measured && measured.from <= offset + left.length && measured.more)
              rebalance = left = left.updateHeight(oracle, offset, force, measured);
          else
              left.updateHeight(oracle, offset, force);
          if (measured && measured.from <= rightStart + right.length && measured.more)
              rebalance = right = right.updateHeight(oracle, rightStart, force, measured);
          else
              right.updateHeight(oracle, rightStart, force);
          if (rebalance)
              return this.balanced(left, right);
          this.height = this.left.height + this.right.height;
          this.outdated = false;
          return this;
      }
      toString() { return this.left + (this.break ? " " : "-") + this.right; }
  }
  function mergeGaps(nodes, around) {
      let before, after;
      if (nodes[around] == null &&
          (before = nodes[around - 1]) instanceof HeightMapGap &&
          (after = nodes[around + 1]) instanceof HeightMapGap)
          nodes.splice(around - 1, 3, new HeightMapGap(before.length + 1 + after.length));
  }
  const relevantWidgetHeight = 5;
  class NodeBuilder {
      constructor(pos, oracle) {
          this.pos = pos;
          this.oracle = oracle;
          this.nodes = [];
          this.lineStart = -1;
          this.lineEnd = -1;
          this.covering = null;
          this.writtenTo = pos;
      }
      get isCovered() {
          return this.covering && this.nodes[this.nodes.length - 1] == this.covering;
      }
      span(_from, to) {
          if (this.lineStart > -1) {
              let end = Math.min(to, this.lineEnd), last = this.nodes[this.nodes.length - 1];
              if (last instanceof HeightMapText)
                  last.length += end - this.pos;
              else if (end > this.pos || !this.isCovered)
                  this.nodes.push(new HeightMapText(end - this.pos, -1));
              this.writtenTo = end;
              if (to > end) {
                  this.nodes.push(null);
                  this.writtenTo++;
                  this.lineStart = -1;
              }
          }
          this.pos = to;
      }
      point(from, to, deco) {
          if (from < to || deco.heightRelevant) {
              let height = deco.widget ? deco.widget.estimatedHeight : 0;
              if (height < 0)
                  height = this.oracle.lineHeight;
              let len = to - from;
              if (deco.block) {
                  this.addBlock(new HeightMapBlock(len, height, deco.type));
              }
              else if (len || height >= relevantWidgetHeight) {
                  this.addLineDeco(height, len);
              }
          }
          else if (to > from) {
              this.span(from, to);
          }
          if (this.lineEnd > -1 && this.lineEnd < this.pos)
              this.lineEnd = this.oracle.doc.lineAt(this.pos).to;
      }
      enterLine() {
          if (this.lineStart > -1)
              return;
          let { from, to } = this.oracle.doc.lineAt(this.pos);
          this.lineStart = from;
          this.lineEnd = to;
          if (this.writtenTo < from) {
              if (this.writtenTo < from - 1 || this.nodes[this.nodes.length - 1] == null)
                  this.nodes.push(this.blankContent(this.writtenTo, from - 1));
              this.nodes.push(null);
          }
          if (this.pos > from)
              this.nodes.push(new HeightMapText(this.pos - from, -1));
          this.writtenTo = this.pos;
      }
      blankContent(from, to) {
          let gap = new HeightMapGap(to - from);
          if (this.oracle.doc.lineAt(from).to == to)
              gap.flags |= 4 /* SingleLine */;
          return gap;
      }
      ensureLine() {
          this.enterLine();
          let last = this.nodes.length ? this.nodes[this.nodes.length - 1] : null;
          if (last instanceof HeightMapText)
              return last;
          let line = new HeightMapText(0, -1);
          this.nodes.push(line);
          return line;
      }
      addBlock(block) {
          this.enterLine();
          if (block.type == BlockType.WidgetAfter && !this.isCovered)
              this.ensureLine();
          this.nodes.push(block);
          this.writtenTo = this.pos = this.pos + block.length;
          if (block.type != BlockType.WidgetBefore)
              this.covering = block;
      }
      addLineDeco(height, length) {
          let line = this.ensureLine();
          line.length += length;
          line.collapsed += length;
          line.widgetHeight = Math.max(line.widgetHeight, height);
          this.writtenTo = this.pos = this.pos + length;
      }
      finish(from) {
          let last = this.nodes.length == 0 ? null : this.nodes[this.nodes.length - 1];
          if (this.lineStart > -1 && !(last instanceof HeightMapText) && !this.isCovered)
              this.nodes.push(new HeightMapText(0, -1));
          else if (this.writtenTo < this.pos || last == null)
              this.nodes.push(this.blankContent(this.writtenTo, this.pos));
          let pos = from;
          for (let node of this.nodes) {
              if (node instanceof HeightMapText)
                  node.updateHeight(this.oracle, pos);
              pos += node ? node.length : 1;
          }
          return this.nodes;
      }
      // Always called with a region that on both sides either stretches
      // to a line break or the end of the document.
      // The returned array uses null to indicate line breaks, but never
      // starts or ends in a line break, or has multiple line breaks next
      // to each other.
      static build(oracle, decorations, from, to) {
          let builder = new NodeBuilder(from, oracle);
          RangeSet.spans(decorations, from, to, builder, 0);
          return builder.finish(from);
      }
  }
  function heightRelevantDecoChanges(a, b, diff) {
      let comp = new DecorationComparator;
      RangeSet.compare(a, b, diff, comp, 0);
      return comp.changes;
  }
  class DecorationComparator {
      constructor() {
          this.changes = [];
      }
      compareRange() { }
      comparePoint(from, to, a, b) {
          if (from < to || a && a.heightRelevant || b && b.heightRelevant)
              addRange(from, to, this.changes, 5);
      }
  }

  function visiblePixelRange(dom, paddingTop) {
      let rect = dom.getBoundingClientRect();
      let left = Math.max(0, rect.left), right = Math.min(innerWidth, rect.right);
      let top = Math.max(0, rect.top), bottom = Math.min(innerHeight, rect.bottom);
      let body = dom.ownerDocument.body;
      for (let parent = dom.parentNode; parent && parent != body;) {
          if (parent.nodeType == 1) {
              let elt = parent;
              let style = window.getComputedStyle(elt);
              if ((elt.scrollHeight > elt.clientHeight || elt.scrollWidth > elt.clientWidth) &&
                  style.overflow != "visible") {
                  let parentRect = elt.getBoundingClientRect();
                  left = Math.max(left, parentRect.left);
                  right = Math.min(right, parentRect.right);
                  top = Math.max(top, parentRect.top);
                  bottom = Math.min(bottom, parentRect.bottom);
              }
              parent = style.position == "absolute" || style.position == "fixed" ? elt.offsetParent : elt.parentNode;
          }
          else if (parent.nodeType == 11) { // Shadow root
              parent = parent.host;
          }
          else {
              break;
          }
      }
      return { left: left - rect.left, right: Math.max(left, right) - rect.left,
          top: top - (rect.top + paddingTop), bottom: Math.max(top, bottom) - (rect.top + paddingTop) };
  }
  function fullPixelRange(dom, paddingTop) {
      let rect = dom.getBoundingClientRect();
      return { left: 0, right: rect.right - rect.left,
          top: paddingTop, bottom: rect.bottom - (rect.top + paddingTop) };
  }
  // Line gaps are placeholder widgets used to hide pieces of overlong
  // lines within the viewport, as a kludge to keep the editor
  // responsive when a ridiculously long line is loaded into it.
  class LineGap {
      constructor(from, to, size) {
          this.from = from;
          this.to = to;
          this.size = size;
      }
      static same(a, b) {
          if (a.length != b.length)
              return false;
          for (let i = 0; i < a.length; i++) {
              let gA = a[i], gB = b[i];
              if (gA.from != gB.from || gA.to != gB.to || gA.size != gB.size)
                  return false;
          }
          return true;
      }
      draw(wrapping) {
          return Decoration.replace({ widget: new LineGapWidget(this.size, wrapping) }).range(this.from, this.to);
      }
  }
  class LineGapWidget extends WidgetType {
      constructor(size, vertical) {
          super();
          this.size = size;
          this.vertical = vertical;
      }
      eq(other) { return other.size == this.size && other.vertical == this.vertical; }
      toDOM() {
          let elt = document.createElement("div");
          if (this.vertical) {
              elt.style.height = this.size + "px";
          }
          else {
              elt.style.width = this.size + "px";
              elt.style.height = "2px";
              elt.style.display = "inline-block";
          }
          return elt;
      }
      get estimatedHeight() { return this.vertical ? this.size : -1; }
  }
  class ViewState {
      constructor(state) {
          this.state = state;
          // These are contentDOM-local coordinates
          this.pixelViewport = { left: 0, right: window.innerWidth, top: 0, bottom: 0 };
          this.inView = true;
          this.paddingTop = 0;
          this.paddingBottom = 0;
          this.contentDOMWidth = 0;
          this.contentDOMHeight = 0;
          this.editorHeight = 0;
          this.editorWidth = 0;
          this.heightOracle = new HeightOracle;
          // See VP.MaxDOMHeight
          this.scaler = IdScaler;
          this.scrollTarget = null;
          // Briefly set to true when printing, to disable viewport limiting
          this.printing = false;
          // Flag set when editor content was redrawn, so that the next
          // measure stage knows it must read DOM layout
          this.mustMeasureContent = true;
          this.defaultTextDirection = Direction.RTL;
          this.visibleRanges = [];
          // Cursor 'assoc' is only significant when the cursor is on a line
          // wrap point, where it must stick to the character that it is
          // associated with. Since browsers don't provide a reasonable
          // interface to set or query this, when a selection is set that
          // might cause this to be significant, this flag is set. The next
          // measure phase will check whether the cursor is on a line-wrapping
          // boundary and, if so, reset it to make sure it is positioned in
          // the right place.
          this.mustEnforceCursorAssoc = false;
          this.stateDeco = state.facet(decorations).filter(d => typeof d != "function");
          this.heightMap = HeightMap.empty().applyChanges(this.stateDeco, Text.empty, this.heightOracle.setDoc(state.doc), [new ChangedRange(0, 0, 0, state.doc.length)]);
          this.viewport = this.getViewport(0, null);
          this.updateViewportLines();
          this.updateForViewport();
          this.lineGaps = this.ensureLineGaps([]);
          this.lineGapDeco = Decoration.set(this.lineGaps.map(gap => gap.draw(false)));
          this.computeVisibleRanges();
      }
      updateForViewport() {
          let viewports = [this.viewport], { main } = this.state.selection;
          for (let i = 0; i <= 1; i++) {
              let pos = i ? main.head : main.anchor;
              if (!viewports.some(({ from, to }) => pos >= from && pos <= to)) {
                  let { from, to } = this.lineBlockAt(pos);
                  viewports.push(new Viewport(from, to));
              }
          }
          this.viewports = viewports.sort((a, b) => a.from - b.from);
          this.scaler = this.heightMap.height <= 7000000 /* MaxDOMHeight */ ? IdScaler :
              new BigScaler(this.heightOracle.doc, this.heightMap, this.viewports);
      }
      updateViewportLines() {
          this.viewportLines = [];
          this.heightMap.forEachLine(this.viewport.from, this.viewport.to, this.state.doc, 0, 0, block => {
              this.viewportLines.push(this.scaler.scale == 1 ? block : scaleBlock(block, this.scaler));
          });
      }
      update(update, scrollTarget = null) {
          this.state = update.state;
          let prevDeco = this.stateDeco;
          this.stateDeco = this.state.facet(decorations).filter(d => typeof d != "function");
          let contentChanges = update.changedRanges;
          let heightChanges = ChangedRange.extendWithRanges(contentChanges, heightRelevantDecoChanges(prevDeco, this.stateDeco, update ? update.changes : ChangeSet.empty(this.state.doc.length)));
          let prevHeight = this.heightMap.height;
          this.heightMap = this.heightMap.applyChanges(this.stateDeco, update.startState.doc, this.heightOracle.setDoc(this.state.doc), heightChanges);
          if (this.heightMap.height != prevHeight)
              update.flags |= 2 /* Height */;
          let viewport = heightChanges.length ? this.mapViewport(this.viewport, update.changes) : this.viewport;
          if (scrollTarget && (scrollTarget.range.head < viewport.from || scrollTarget.range.head > viewport.to) ||
              !this.viewportIsAppropriate(viewport))
              viewport = this.getViewport(0, scrollTarget);
          let updateLines = !update.changes.empty || (update.flags & 2 /* Height */) ||
              viewport.from != this.viewport.from || viewport.to != this.viewport.to;
          this.viewport = viewport;
          this.updateForViewport();
          if (updateLines)
              this.updateViewportLines();
          if (this.lineGaps.length || this.viewport.to - this.viewport.from > 4000 /* DoubleMargin */)
              this.updateLineGaps(this.ensureLineGaps(this.mapLineGaps(this.lineGaps, update.changes)));
          update.flags |= this.computeVisibleRanges();
          if (scrollTarget)
              this.scrollTarget = scrollTarget;
          if (!this.mustEnforceCursorAssoc && update.selectionSet && update.view.lineWrapping &&
              update.state.selection.main.empty && update.state.selection.main.assoc)
              this.mustEnforceCursorAssoc = true;
      }
      measure(view) {
          let dom = view.contentDOM, style = window.getComputedStyle(dom);
          let oracle = this.heightOracle;
          let whiteSpace = style.whiteSpace;
          this.defaultTextDirection = style.direction == "rtl" ? Direction.RTL : Direction.LTR;
          let refresh = this.heightOracle.mustRefreshForWrapping(whiteSpace);
          let measureContent = refresh || this.mustMeasureContent || this.contentDOMHeight != dom.clientHeight;
          this.contentDOMHeight = dom.clientHeight;
          this.mustMeasureContent = false;
          let result = 0, bias = 0;
          // Vertical padding
          let paddingTop = parseInt(style.paddingTop) || 0, paddingBottom = parseInt(style.paddingBottom) || 0;
          if (this.paddingTop != paddingTop || this.paddingBottom != paddingBottom) {
              this.paddingTop = paddingTop;
              this.paddingBottom = paddingBottom;
              result |= 8 /* Geometry */ | 2 /* Height */;
          }
          if (this.editorWidth != view.scrollDOM.clientWidth) {
              if (oracle.lineWrapping)
                  measureContent = true;
              this.editorWidth = view.scrollDOM.clientWidth;
              result |= 8 /* Geometry */;
          }
          // Pixel viewport
          let pixelViewport = (this.printing ? fullPixelRange : visiblePixelRange)(dom, this.paddingTop);
          let dTop = pixelViewport.top - this.pixelViewport.top, dBottom = pixelViewport.bottom - this.pixelViewport.bottom;
          this.pixelViewport = pixelViewport;
          let inView = this.pixelViewport.bottom > this.pixelViewport.top && this.pixelViewport.right > this.pixelViewport.left;
          if (inView != this.inView) {
              this.inView = inView;
              if (inView)
                  measureContent = true;
          }
          if (!this.inView)
              return 0;
          let contentWidth = dom.clientWidth;
          if (this.contentDOMWidth != contentWidth || this.editorHeight != view.scrollDOM.clientHeight) {
              this.contentDOMWidth = contentWidth;
              this.editorHeight = view.scrollDOM.clientHeight;
              result |= 8 /* Geometry */;
          }
          if (measureContent) {
              let lineHeights = view.docView.measureVisibleLineHeights(this.viewport);
              if (oracle.mustRefreshForHeights(lineHeights))
                  refresh = true;
              if (refresh || oracle.lineWrapping && Math.abs(contentWidth - this.contentDOMWidth) > oracle.charWidth) {
                  let { lineHeight, charWidth } = view.docView.measureTextSize();
                  refresh = oracle.refresh(whiteSpace, lineHeight, charWidth, contentWidth / charWidth, lineHeights);
                  if (refresh) {
                      view.docView.minWidth = 0;
                      result |= 8 /* Geometry */;
                  }
              }
              if (dTop > 0 && dBottom > 0)
                  bias = Math.max(dTop, dBottom);
              else if (dTop < 0 && dBottom < 0)
                  bias = Math.min(dTop, dBottom);
              oracle.heightChanged = false;
              for (let vp of this.viewports) {
                  let heights = vp.from == this.viewport.from ? lineHeights : view.docView.measureVisibleLineHeights(vp);
                  this.heightMap = this.heightMap.updateHeight(oracle, 0, refresh, new MeasuredHeights(vp.from, heights));
              }
              if (oracle.heightChanged)
                  result |= 2 /* Height */;
          }
          let viewportChange = !this.viewportIsAppropriate(this.viewport, bias) ||
              this.scrollTarget && (this.scrollTarget.range.head < this.viewport.from || this.scrollTarget.range.head > this.viewport.to);
          if (viewportChange)
              this.viewport = this.getViewport(bias, this.scrollTarget);
          this.updateForViewport();
          if ((result & 2 /* Height */) || viewportChange)
              this.updateViewportLines();
          if (this.lineGaps.length || this.viewport.to - this.viewport.from > 4000 /* DoubleMargin */)
              this.updateLineGaps(this.ensureLineGaps(refresh ? [] : this.lineGaps));
          result |= this.computeVisibleRanges();
          if (this.mustEnforceCursorAssoc) {
              this.mustEnforceCursorAssoc = false;
              // This is done in the read stage, because moving the selection
              // to a line end is going to trigger a layout anyway, so it
              // can't be a pure write. It should be rare that it does any
              // writing.
              view.docView.enforceCursorAssoc();
          }
          return result;
      }
      get visibleTop() { return this.scaler.fromDOM(this.pixelViewport.top); }
      get visibleBottom() { return this.scaler.fromDOM(this.pixelViewport.bottom); }
      getViewport(bias, scrollTarget) {
          // This will divide VP.Margin between the top and the
          // bottom, depending on the bias (the change in viewport position
          // since the last update). It'll hold a number between 0 and 1
          let marginTop = 0.5 - Math.max(-0.5, Math.min(0.5, bias / 1000 /* Margin */ / 2));
          let map = this.heightMap, doc = this.state.doc, { visibleTop, visibleBottom } = this;
          let viewport = new Viewport(map.lineAt(visibleTop - marginTop * 1000 /* Margin */, QueryType$1.ByHeight, doc, 0, 0).from, map.lineAt(visibleBottom + (1 - marginTop) * 1000 /* Margin */, QueryType$1.ByHeight, doc, 0, 0).to);
          // If scrollTarget is given, make sure the viewport includes that position
          if (scrollTarget) {
              let { head } = scrollTarget.range;
              if (head < viewport.from || head > viewport.to) {
                  let viewHeight = Math.min(this.editorHeight, this.pixelViewport.bottom - this.pixelViewport.top);
                  let block = map.lineAt(head, QueryType$1.ByPos, doc, 0, 0), topPos;
                  if (scrollTarget.y == "center")
                      topPos = (block.top + block.bottom) / 2 - viewHeight / 2;
                  else if (scrollTarget.y == "start" || scrollTarget.y == "nearest" && head < viewport.from)
                      topPos = block.top;
                  else
                      topPos = block.bottom - viewHeight;
                  viewport = new Viewport(map.lineAt(topPos - 1000 /* Margin */ / 2, QueryType$1.ByHeight, doc, 0, 0).from, map.lineAt(topPos + viewHeight + 1000 /* Margin */ / 2, QueryType$1.ByHeight, doc, 0, 0).to);
              }
          }
          return viewport;
      }
      mapViewport(viewport, changes) {
          let from = changes.mapPos(viewport.from, -1), to = changes.mapPos(viewport.to, 1);
          return new Viewport(this.heightMap.lineAt(from, QueryType$1.ByPos, this.state.doc, 0, 0).from, this.heightMap.lineAt(to, QueryType$1.ByPos, this.state.doc, 0, 0).to);
      }
      // Checks if a given viewport covers the visible part of the
      // document and not too much beyond that.
      viewportIsAppropriate({ from, to }, bias = 0) {
          if (!this.inView)
              return true;
          let { top } = this.heightMap.lineAt(from, QueryType$1.ByPos, this.state.doc, 0, 0);
          let { bottom } = this.heightMap.lineAt(to, QueryType$1.ByPos, this.state.doc, 0, 0);
          let { visibleTop, visibleBottom } = this;
          return (from == 0 || top <= visibleTop - Math.max(10 /* MinCoverMargin */, Math.min(-bias, 250 /* MaxCoverMargin */))) &&
              (to == this.state.doc.length ||
                  bottom >= visibleBottom + Math.max(10 /* MinCoverMargin */, Math.min(bias, 250 /* MaxCoverMargin */))) &&
              (top > visibleTop - 2 * 1000 /* Margin */ && bottom < visibleBottom + 2 * 1000 /* Margin */);
      }
      mapLineGaps(gaps, changes) {
          if (!gaps.length || changes.empty)
              return gaps;
          let mapped = [];
          for (let gap of gaps)
              if (!changes.touchesRange(gap.from, gap.to))
                  mapped.push(new LineGap(changes.mapPos(gap.from), changes.mapPos(gap.to), gap.size));
          return mapped;
      }
      // Computes positions in the viewport where the start or end of a
      // line should be hidden, trying to reuse existing line gaps when
      // appropriate to avoid unneccesary redraws.
      // Uses crude character-counting for the positioning and sizing,
      // since actual DOM coordinates aren't always available and
      // predictable. Relies on generous margins (see LG.Margin) to hide
      // the artifacts this might produce from the user.
      ensureLineGaps(current) {
          let gaps = [];
          // This won't work at all in predominantly right-to-left text.
          if (this.defaultTextDirection != Direction.LTR)
              return gaps;
          for (let line of this.viewportLines) {
              if (line.length < 4000 /* DoubleMargin */)
                  continue;
              let structure = lineStructure(line.from, line.to, this.stateDeco);
              if (structure.total < 4000 /* DoubleMargin */)
                  continue;
              let viewFrom, viewTo;
              if (this.heightOracle.lineWrapping) {
                  let marginHeight = (2000 /* Margin */ / this.heightOracle.lineLength) * this.heightOracle.lineHeight;
                  viewFrom = findPosition(structure, (this.visibleTop - line.top - marginHeight) / line.height);
                  viewTo = findPosition(structure, (this.visibleBottom - line.top + marginHeight) / line.height);
              }
              else {
                  let totalWidth = structure.total * this.heightOracle.charWidth;
                  let marginWidth = 2000 /* Margin */ * this.heightOracle.charWidth;
                  viewFrom = findPosition(structure, (this.pixelViewport.left - marginWidth) / totalWidth);
                  viewTo = findPosition(structure, (this.pixelViewport.right + marginWidth) / totalWidth);
              }
              let outside = [];
              if (viewFrom > line.from)
                  outside.push({ from: line.from, to: viewFrom });
              if (viewTo < line.to)
                  outside.push({ from: viewTo, to: line.to });
              let sel = this.state.selection.main;
              // Make sure the gaps don't cover a selection end
              if (sel.from >= line.from && sel.from <= line.to)
                  cutRange(outside, sel.from - 10 /* SelectionMargin */, sel.from + 10 /* SelectionMargin */);
              if (!sel.empty && sel.to >= line.from && sel.to <= line.to)
                  cutRange(outside, sel.to - 10 /* SelectionMargin */, sel.to + 10 /* SelectionMargin */);
              for (let { from, to } of outside)
                  if (to - from > 1000 /* HalfMargin */) {
                      gaps.push(find(current, gap => gap.from >= line.from && gap.to <= line.to &&
                          Math.abs(gap.from - from) < 1000 /* HalfMargin */ && Math.abs(gap.to - to) < 1000 /* HalfMargin */) ||
                          new LineGap(from, to, this.gapSize(line, from, to, structure)));
                  }
          }
          return gaps;
      }
      gapSize(line, from, to, structure) {
          let fraction = findFraction(structure, to) - findFraction(structure, from);
          if (this.heightOracle.lineWrapping) {
              return line.height * fraction;
          }
          else {
              return structure.total * this.heightOracle.charWidth * fraction;
          }
      }
      updateLineGaps(gaps) {
          if (!LineGap.same(gaps, this.lineGaps)) {
              this.lineGaps = gaps;
              this.lineGapDeco = Decoration.set(gaps.map(gap => gap.draw(this.heightOracle.lineWrapping)));
          }
      }
      computeVisibleRanges() {
          let deco = this.stateDeco;
          if (this.lineGaps.length)
              deco = deco.concat(this.lineGapDeco);
          let ranges = [];
          RangeSet.spans(deco, this.viewport.from, this.viewport.to, {
              span(from, to) { ranges.push({ from, to }); },
              point() { }
          }, 20);
          let changed = ranges.length != this.visibleRanges.length ||
              this.visibleRanges.some((r, i) => r.from != ranges[i].from || r.to != ranges[i].to);
          this.visibleRanges = ranges;
          return changed ? 4 /* Viewport */ : 0;
      }
      lineBlockAt(pos) {
          return (pos >= this.viewport.from && pos <= this.viewport.to && this.viewportLines.find(b => b.from <= pos && b.to >= pos)) ||
              scaleBlock(this.heightMap.lineAt(pos, QueryType$1.ByPos, this.state.doc, 0, 0), this.scaler);
      }
      lineBlockAtHeight(height) {
          return scaleBlock(this.heightMap.lineAt(this.scaler.fromDOM(height), QueryType$1.ByHeight, this.state.doc, 0, 0), this.scaler);
      }
      elementAtHeight(height) {
          return scaleBlock(this.heightMap.blockAt(this.scaler.fromDOM(height), this.state.doc, 0, 0), this.scaler);
      }
      get docHeight() {
          return this.scaler.toDOM(this.heightMap.height);
      }
      get contentHeight() {
          return this.docHeight + this.paddingTop + this.paddingBottom;
      }
  }
  class Viewport {
      constructor(from, to) {
          this.from = from;
          this.to = to;
      }
  }
  function lineStructure(from, to, stateDeco) {
      let ranges = [], pos = from, total = 0;
      RangeSet.spans(stateDeco, from, to, {
          span() { },
          point(from, to) {
              if (from > pos) {
                  ranges.push({ from: pos, to: from });
                  total += from - pos;
              }
              pos = to;
          }
      }, 20); // We're only interested in collapsed ranges of a significant size
      if (pos < to) {
          ranges.push({ from: pos, to });
          total += to - pos;
      }
      return { total, ranges };
  }
  function findPosition({ total, ranges }, ratio) {
      if (ratio <= 0)
          return ranges[0].from;
      if (ratio >= 1)
          return ranges[ranges.length - 1].to;
      let dist = Math.floor(total * ratio);
      for (let i = 0;; i++) {
          let { from, to } = ranges[i], size = to - from;
          if (dist <= size)
              return from + dist;
          dist -= size;
      }
  }
  function findFraction(structure, pos) {
      let counted = 0;
      for (let { from, to } of structure.ranges) {
          if (pos <= to) {
              counted += pos - from;
              break;
          }
          counted += to - from;
      }
      return counted / structure.total;
  }
  function cutRange(ranges, from, to) {
      for (let i = 0; i < ranges.length; i++) {
          let r = ranges[i];
          if (r.from < to && r.to > from) {
              let pieces = [];
              if (r.from < from)
                  pieces.push({ from: r.from, to: from });
              if (r.to > to)
                  pieces.push({ from: to, to: r.to });
              ranges.splice(i, 1, ...pieces);
              i += pieces.length - 1;
          }
      }
  }
  function find(array, f) {
      for (let val of array)
          if (f(val))
              return val;
      return undefined;
  }
  // Don't scale when the document height is within the range of what
  // the DOM can handle.
  const IdScaler = {
      toDOM(n) { return n; },
      fromDOM(n) { return n; },
      scale: 1
  };
  // When the height is too big (> VP.MaxDOMHeight), scale down the
  // regions outside the viewports so that the total height is
  // VP.MaxDOMHeight.
  class BigScaler {
      constructor(doc, heightMap, viewports) {
          let vpHeight = 0, base = 0, domBase = 0;
          this.viewports = viewports.map(({ from, to }) => {
              let top = heightMap.lineAt(from, QueryType$1.ByPos, doc, 0, 0).top;
              let bottom = heightMap.lineAt(to, QueryType$1.ByPos, doc, 0, 0).bottom;
              vpHeight += bottom - top;
              return { from, to, top, bottom, domTop: 0, domBottom: 0 };
          });
          this.scale = (7000000 /* MaxDOMHeight */ - vpHeight) / (heightMap.height - vpHeight);
          for (let obj of this.viewports) {
              obj.domTop = domBase + (obj.top - base) * this.scale;
              domBase = obj.domBottom = obj.domTop + (obj.bottom - obj.top);
              base = obj.bottom;
          }
      }
      toDOM(n) {
          for (let i = 0, base = 0, domBase = 0;; i++) {
              let vp = i < this.viewports.length ? this.viewports[i] : null;
              if (!vp || n < vp.top)
                  return domBase + (n - base) * this.scale;
              if (n <= vp.bottom)
                  return vp.domTop + (n - vp.top);
              base = vp.bottom;
              domBase = vp.domBottom;
          }
      }
      fromDOM(n) {
          for (let i = 0, base = 0, domBase = 0;; i++) {
              let vp = i < this.viewports.length ? this.viewports[i] : null;
              if (!vp || n < vp.domTop)
                  return base + (n - domBase) / this.scale;
              if (n <= vp.domBottom)
                  return vp.top + (n - vp.domTop);
              base = vp.bottom;
              domBase = vp.domBottom;
          }
      }
  }
  function scaleBlock(block, scaler) {
      if (scaler.scale == 1)
          return block;
      let bTop = scaler.toDOM(block.top), bBottom = scaler.toDOM(block.bottom);
      return new BlockInfo(block.from, block.length, bTop, bBottom - bTop, Array.isArray(block.type) ? block.type.map(b => scaleBlock(b, scaler)) : block.type);
  }

  const theme = /*@__PURE__*/Facet.define({ combine: strs => strs.join(" ") });
  const darkTheme = /*@__PURE__*/Facet.define({ combine: values => values.indexOf(true) > -1 });
  const baseThemeID = /*@__PURE__*/StyleModule.newName(), baseLightID = /*@__PURE__*/StyleModule.newName(), baseDarkID = /*@__PURE__*/StyleModule.newName();
  const lightDarkIDs = { "&light": "." + baseLightID, "&dark": "." + baseDarkID };
  function buildTheme(main, spec, scopes) {
      return new StyleModule(spec, {
          finish(sel) {
              return /&/.test(sel) ? sel.replace(/&\w*/, m => {
                  if (m == "&")
                      return main;
                  if (!scopes || !scopes[m])
                      throw new RangeError(`Unsupported selector: ${m}`);
                  return scopes[m];
              }) : main + " " + sel;
          }
      });
  }
  const baseTheme$1$3 = /*@__PURE__*/buildTheme("." + baseThemeID, {
      "&.cm-editor": {
          position: "relative !important",
          boxSizing: "border-box",
          "&.cm-focused": {
              // Provide a simple default outline to make sure a focused
              // editor is visually distinct. Can't leave the default behavior
              // because that will apply to the content element, which is
              // inside the scrollable container and doesn't include the
              // gutters. We also can't use an 'auto' outline, since those
              // are, for some reason, drawn behind the element content, which
              // will cause things like the active line background to cover
              // the outline (#297).
              outline: "1px dotted #212121"
          },
          display: "flex !important",
          flexDirection: "column"
      },
      ".cm-scroller": {
          display: "flex !important",
          alignItems: "flex-start !important",
          fontFamily: "monospace",
          lineHeight: 1.4,
          height: "100%",
          overflowX: "auto",
          position: "relative",
          zIndex: 0
      },
      ".cm-content": {
          margin: 0,
          flexGrow: 2,
          flexShrink: 0,
          minHeight: "100%",
          display: "block",
          whiteSpace: "pre",
          wordWrap: "normal",
          boxSizing: "border-box",
          padding: "4px 0",
          outline: "none",
          "&[contenteditable=true]": {
              WebkitUserModify: "read-write-plaintext-only",
          }
      },
      ".cm-lineWrapping": {
          whiteSpace_fallback: "pre-wrap",
          whiteSpace: "break-spaces",
          wordBreak: "break-word",
          overflowWrap: "anywhere",
          flexShrink: 1
      },
      "&light .cm-content": { caretColor: "black" },
      "&dark .cm-content": { caretColor: "white" },
      ".cm-line": {
          display: "block",
          padding: "0 2px 0 4px"
      },
      ".cm-selectionLayer": {
          zIndex: -1,
          contain: "size style"
      },
      ".cm-selectionBackground": {
          position: "absolute",
      },
      "&light .cm-selectionBackground": {
          background: "#d9d9d9"
      },
      "&dark .cm-selectionBackground": {
          background: "#222"
      },
      "&light.cm-focused .cm-selectionBackground": {
          background: "#d7d4f0"
      },
      "&dark.cm-focused .cm-selectionBackground": {
          background: "#233"
      },
      ".cm-cursorLayer": {
          zIndex: 100,
          contain: "size style",
          pointerEvents: "none"
      },
      "&.cm-focused .cm-cursorLayer": {
          animation: "steps(1) cm-blink 1.2s infinite"
      },
      // Two animations defined so that we can switch between them to
      // restart the animation without forcing another style
      // recomputation.
      "@keyframes cm-blink": { "0%": {}, "50%": { opacity: 0 }, "100%": {} },
      "@keyframes cm-blink2": { "0%": {}, "50%": { opacity: 0 }, "100%": {} },
      ".cm-cursor, .cm-dropCursor": {
          position: "absolute",
          borderLeft: "1.2px solid black",
          marginLeft: "-0.6px",
          pointerEvents: "none",
      },
      ".cm-cursor": {
          display: "none"
      },
      "&dark .cm-cursor": {
          borderLeftColor: "#444"
      },
      "&.cm-focused .cm-cursor": {
          display: "block"
      },
      "&light .cm-activeLine": { backgroundColor: "#f3f9ff" },
      "&dark .cm-activeLine": { backgroundColor: "#223039" },
      "&light .cm-specialChar": { color: "red" },
      "&dark .cm-specialChar": { color: "#f78" },
      ".cm-gutters": {
          flexShrink: 0,
          display: "flex",
          height: "100%",
          boxSizing: "border-box",
          left: 0,
          zIndex: 200
      },
      "&light .cm-gutters": {
          backgroundColor: "#f5f5f5",
          color: "#6c6c6c",
          borderRight: "1px solid #ddd"
      },
      "&dark .cm-gutters": {
          backgroundColor: "#333338",
          color: "#ccc"
      },
      ".cm-gutter": {
          display: "flex !important",
          flexDirection: "column",
          flexShrink: 0,
          boxSizing: "border-box",
          minHeight: "100%",
          overflow: "hidden"
      },
      ".cm-gutterElement": {
          boxSizing: "border-box"
      },
      ".cm-lineNumbers .cm-gutterElement": {
          padding: "0 3px 0 5px",
          minWidth: "20px",
          textAlign: "right",
          whiteSpace: "nowrap"
      },
      "&light .cm-activeLineGutter": {
          backgroundColor: "#e2f2ff"
      },
      "&dark .cm-activeLineGutter": {
          backgroundColor: "#222227"
      },
      ".cm-panels": {
          boxSizing: "border-box",
          position: "sticky",
          left: 0,
          right: 0
      },
      "&light .cm-panels": {
          backgroundColor: "#f5f5f5",
          color: "black"
      },
      "&light .cm-panels-top": {
          borderBottom: "1px solid #ddd"
      },
      "&light .cm-panels-bottom": {
          borderTop: "1px solid #ddd"
      },
      "&dark .cm-panels": {
          backgroundColor: "#333338",
          color: "white"
      },
      ".cm-tab": {
          display: "inline-block",
          overflow: "hidden",
          verticalAlign: "bottom"
      },
      ".cm-widgetBuffer": {
          verticalAlign: "text-top",
          height: "1em",
          display: "inline"
      },
      ".cm-placeholder": {
          color: "#888",
          display: "inline-block",
          verticalAlign: "top",
      },
      ".cm-button": {
          verticalAlign: "middle",
          color: "inherit",
          fontSize: "70%",
          padding: ".2em 1em",
          borderRadius: "1px"
      },
      "&light .cm-button": {
          backgroundImage: "linear-gradient(#eff1f5, #d9d9df)",
          border: "1px solid #888",
          "&:active": {
              backgroundImage: "linear-gradient(#b4b4b4, #d0d3d6)"
          }
      },
      "&dark .cm-button": {
          backgroundImage: "linear-gradient(#393939, #111)",
          border: "1px solid #888",
          "&:active": {
              backgroundImage: "linear-gradient(#111, #333)"
          }
      },
      ".cm-textfield": {
          verticalAlign: "middle",
          color: "inherit",
          fontSize: "70%",
          border: "1px solid silver",
          padding: ".2em .5em"
      },
      "&light .cm-textfield": {
          backgroundColor: "white"
      },
      "&dark .cm-textfield": {
          border: "1px solid #555",
          backgroundColor: "inherit"
      }
  }, lightDarkIDs);

  const observeOptions = {
      childList: true,
      characterData: true,
      subtree: true,
      attributes: true,
      characterDataOldValue: true
  };
  // IE11 has very broken mutation observers, so we also listen to
  // DOMCharacterDataModified there
  const useCharData = browser.ie && browser.ie_version <= 11;
  class DOMObserver {
      constructor(view, onChange, onScrollChanged) {
          this.view = view;
          this.onChange = onChange;
          this.onScrollChanged = onScrollChanged;
          this.active = false;
          // The known selection. Kept in our own object, as opposed to just
          // directly accessing the selection because:
          //  - Safari doesn't report the right selection in shadow DOM
          //  - Reading from the selection forces a DOM layout
          //  - This way, we can ignore selectionchange events if we have
          //    already seen the 'new' selection
          this.selectionRange = new DOMSelectionState;
          // Set when a selection change is detected, cleared on flush
          this.selectionChanged = false;
          this.delayedFlush = -1;
          this.resizeTimeout = -1;
          this.queue = [];
          this.delayedAndroidKey = null;
          this.scrollTargets = [];
          this.intersection = null;
          this.resize = null;
          this.intersecting = false;
          this.gapIntersection = null;
          this.gaps = [];
          // Timeout for scheduling check of the parents that need scroll handlers
          this.parentCheck = -1;
          this.dom = view.contentDOM;
          this.observer = new MutationObserver(mutations => {
              for (let mut of mutations)
                  this.queue.push(mut);
              // IE11 will sometimes (on typing over a selection or
              // backspacing out a single character text node) call the
              // observer callback before actually updating the DOM.
              //
              // Unrelatedly, iOS Safari will, when ending a composition,
              // sometimes first clear it, deliver the mutations, and then
              // reinsert the finished text. CodeMirror's handling of the
              // deletion will prevent the reinsertion from happening,
              // breaking composition.
              if ((browser.ie && browser.ie_version <= 11 || browser.ios && view.composing) &&
                  mutations.some(m => m.type == "childList" && m.removedNodes.length ||
                      m.type == "characterData" && m.oldValue.length > m.target.nodeValue.length))
                  this.flushSoon();
              else
                  this.flush();
          });
          if (useCharData)
              this.onCharData = (event) => {
                  this.queue.push({ target: event.target,
                      type: "characterData",
                      oldValue: event.prevValue });
                  this.flushSoon();
              };
          this.onSelectionChange = this.onSelectionChange.bind(this);
          window.addEventListener("resize", this.onResize = this.onResize.bind(this));
          if (typeof ResizeObserver == "function") {
              this.resize = new ResizeObserver(() => {
                  if (this.view.docView.lastUpdate < Date.now() - 75)
                      this.onResize();
              });
              this.resize.observe(view.scrollDOM);
          }
          window.addEventListener("beforeprint", this.onPrint = this.onPrint.bind(this));
          this.start();
          window.addEventListener("scroll", this.onScroll = this.onScroll.bind(this));
          if (typeof IntersectionObserver == "function") {
              this.intersection = new IntersectionObserver(entries => {
                  if (this.parentCheck < 0)
                      this.parentCheck = setTimeout(this.listenForScroll.bind(this), 1000);
                  if (entries.length > 0 && (entries[entries.length - 1].intersectionRatio > 0) != this.intersecting) {
                      this.intersecting = !this.intersecting;
                      if (this.intersecting != this.view.inView)
                          this.onScrollChanged(document.createEvent("Event"));
                  }
              }, {});
              this.intersection.observe(this.dom);
              this.gapIntersection = new IntersectionObserver(entries => {
                  if (entries.length > 0 && entries[entries.length - 1].intersectionRatio > 0)
                      this.onScrollChanged(document.createEvent("Event"));
              }, {});
          }
          this.listenForScroll();
          this.readSelectionRange();
          this.dom.ownerDocument.addEventListener("selectionchange", this.onSelectionChange);
      }
      onScroll(e) {
          if (this.intersecting)
              this.flush(false);
          this.onScrollChanged(e);
      }
      onResize() {
          if (this.resizeTimeout < 0)
              this.resizeTimeout = setTimeout(() => {
                  this.resizeTimeout = -1;
                  this.view.requestMeasure();
              }, 50);
      }
      onPrint() {
          this.view.viewState.printing = true;
          this.view.measure();
          setTimeout(() => {
              this.view.viewState.printing = false;
              this.view.requestMeasure();
          }, 500);
      }
      updateGaps(gaps) {
          if (this.gapIntersection && (gaps.length != this.gaps.length || this.gaps.some((g, i) => g != gaps[i]))) {
              this.gapIntersection.disconnect();
              for (let gap of gaps)
                  this.gapIntersection.observe(gap);
              this.gaps = gaps;
          }
      }
      onSelectionChange(event) {
          if (!this.readSelectionRange() || this.delayedAndroidKey)
              return;
          let { view } = this, sel = this.selectionRange;
          if (view.state.facet(editable) ? view.root.activeElement != this.dom : !hasSelection(view.dom, sel))
              return;
          let context = sel.anchorNode && view.docView.nearest(sel.anchorNode);
          if (context && context.ignoreEvent(event))
              return;
          // Deletions on IE11 fire their events in the wrong order, giving
          // us a selection change event before the DOM changes are
          // reported.
          // Chrome Android has a similar issue when backspacing out a
          // selection (#645).
          if ((browser.ie && browser.ie_version <= 11 || browser.android && browser.chrome) && !view.state.selection.main.empty &&
              // (Selection.isCollapsed isn't reliable on IE)
              sel.focusNode && isEquivalentPosition(sel.focusNode, sel.focusOffset, sel.anchorNode, sel.anchorOffset))
              this.flushSoon();
          else
              this.flush(false);
      }
      readSelectionRange() {
          let { view } = this;
          // The Selection object is broken in shadow roots in Safari. See
          // https://github.com/codemirror/dev/issues/414
          let range = browser.safari && view.root.nodeType == 11 && deepActiveElement() == this.dom &&
              safariSelectionRangeHack(this.view) || getSelection(view.root);
          if (!range || this.selectionRange.eq(range))
              return false;
          let local = hasSelection(this.dom, range);
          // Detect the situation where the browser has, on focus, moved the
          // selection to the start of the content element. Reset it to the
          // position from the editor state.
          if (local && !this.selectionChanged && this.selectionRange.focusNode &&
              view.inputState.lastFocusTime > Date.now() - 200 &&
              view.inputState.lastTouchTime < Date.now() - 300 &&
              atElementStart(this.dom, range)) {
              view.docView.updateSelection();
              return false;
          }
          this.selectionRange.setRange(range);
          if (local)
              this.selectionChanged = true;
          return true;
      }
      setSelectionRange(anchor, head) {
          this.selectionRange.set(anchor.node, anchor.offset, head.node, head.offset);
          this.selectionChanged = false;
      }
      clearSelectionRange() {
          this.selectionRange.set(null, 0, null, 0);
      }
      listenForScroll() {
          this.parentCheck = -1;
          let i = 0, changed = null;
          for (let dom = this.dom; dom;) {
              if (dom.nodeType == 1) {
                  if (!changed && i < this.scrollTargets.length && this.scrollTargets[i] == dom)
                      i++;
                  else if (!changed)
                      changed = this.scrollTargets.slice(0, i);
                  if (changed)
                      changed.push(dom);
                  dom = dom.assignedSlot || dom.parentNode;
              }
              else if (dom.nodeType == 11) { // Shadow root
                  dom = dom.host;
              }
              else {
                  break;
              }
          }
          if (i < this.scrollTargets.length && !changed)
              changed = this.scrollTargets.slice(0, i);
          if (changed) {
              for (let dom of this.scrollTargets)
                  dom.removeEventListener("scroll", this.onScroll);
              for (let dom of this.scrollTargets = changed)
                  dom.addEventListener("scroll", this.onScroll);
          }
      }
      ignore(f) {
          if (!this.active)
              return f();
          try {
              this.stop();
              return f();
          }
          finally {
              this.start();
              this.clear();
          }
      }
      start() {
          if (this.active)
              return;
          this.observer.observe(this.dom, observeOptions);
          if (useCharData)
              this.dom.addEventListener("DOMCharacterDataModified", this.onCharData);
          this.active = true;
      }
      stop() {
          if (!this.active)
              return;
          this.active = false;
          this.observer.disconnect();
          if (useCharData)
              this.dom.removeEventListener("DOMCharacterDataModified", this.onCharData);
      }
      // Throw away any pending changes
      clear() {
          this.processRecords();
          this.queue.length = 0;
          this.selectionChanged = false;
      }
      // Chrome Android, especially in combination with GBoard, not only
      // doesn't reliably fire regular key events, but also often
      // surrounds the effect of enter or backspace with a bunch of
      // composition events that, when interrupted, cause text duplication
      // or other kinds of corruption. This hack makes the editor back off
      // from handling DOM changes for a moment when such a key is
      // detected (via beforeinput or keydown), and then tries to flush
      // them or, if that has no effect, dispatches the given key.
      delayAndroidKey(key, keyCode) {
          if (!this.delayedAndroidKey)
              requestAnimationFrame(() => {
                  let key = this.delayedAndroidKey;
                  this.delayedAndroidKey = null;
                  this.delayedFlush = -1;
                  if (!this.flush())
                      dispatchKey(this.dom, key.key, key.keyCode);
              });
          // Since backspace beforeinput is sometimes signalled spuriously,
          // Enter always takes precedence.
          if (!this.delayedAndroidKey || key == "Enter")
              this.delayedAndroidKey = { key, keyCode };
      }
      flushSoon() {
          if (this.delayedFlush < 0)
              this.delayedFlush = window.setTimeout(() => { this.delayedFlush = -1; this.flush(); }, 20);
      }
      forceFlush() {
          if (this.delayedFlush >= 0) {
              window.clearTimeout(this.delayedFlush);
              this.delayedFlush = -1;
          }
          this.flush();
      }
      processRecords() {
          let records = this.queue;
          for (let mut of this.observer.takeRecords())
              records.push(mut);
          if (records.length)
              this.queue = [];
          let from = -1, to = -1, typeOver = false;
          for (let record of records) {
              let range = this.readMutation(record);
              if (!range)
                  continue;
              if (range.typeOver)
                  typeOver = true;
              if (from == -1) {
                  ({ from, to } = range);
              }
              else {
                  from = Math.min(range.from, from);
                  to = Math.max(range.to, to);
              }
          }
          return { from, to, typeOver };
      }
      // Apply pending changes, if any
      flush(readSelection = true) {
          // Completely hold off flushing when pending keys are set—the code
          // managing those will make sure processRecords is called and the
          // view is resynchronized after
          if (this.delayedFlush >= 0 || this.delayedAndroidKey)
              return;
          if (readSelection)
              this.readSelectionRange();
          let { from, to, typeOver } = this.processRecords();
          let newSel = this.selectionChanged && hasSelection(this.dom, this.selectionRange);
          if (from < 0 && !newSel)
              return;
          this.view.inputState.lastFocusTime = 0;
          this.selectionChanged = false;
          let startState = this.view.state;
          let handled = this.onChange(from, to, typeOver);
          // The view wasn't updated
          if (this.view.state == startState)
              this.view.update([]);
          return handled;
      }
      readMutation(rec) {
          let cView = this.view.docView.nearest(rec.target);
          if (!cView || cView.ignoreMutation(rec))
              return null;
          cView.markDirty(rec.type == "attributes");
          if (rec.type == "attributes")
              cView.dirty |= 4 /* Attrs */;
          if (rec.type == "childList") {
              let childBefore = findChild(cView, rec.previousSibling || rec.target.previousSibling, -1);
              let childAfter = findChild(cView, rec.nextSibling || rec.target.nextSibling, 1);
              return { from: childBefore ? cView.posAfter(childBefore) : cView.posAtStart,
                  to: childAfter ? cView.posBefore(childAfter) : cView.posAtEnd, typeOver: false };
          }
          else if (rec.type == "characterData") {
              return { from: cView.posAtStart, to: cView.posAtEnd, typeOver: rec.target.nodeValue == rec.oldValue };
          }
          else {
              return null;
          }
      }
      destroy() {
          var _a, _b, _c;
          this.stop();
          (_a = this.intersection) === null || _a === void 0 ? void 0 : _a.disconnect();
          (_b = this.gapIntersection) === null || _b === void 0 ? void 0 : _b.disconnect();
          (_c = this.resize) === null || _c === void 0 ? void 0 : _c.disconnect();
          for (let dom of this.scrollTargets)
              dom.removeEventListener("scroll", this.onScroll);
          window.removeEventListener("scroll", this.onScroll);
          window.removeEventListener("resize", this.onResize);
          window.removeEventListener("beforeprint", this.onPrint);
          this.dom.ownerDocument.removeEventListener("selectionchange", this.onSelectionChange);
          clearTimeout(this.parentCheck);
          clearTimeout(this.resizeTimeout);
      }
  }
  function findChild(cView, dom, dir) {
      while (dom) {
          let curView = ContentView.get(dom);
          if (curView && curView.parent == cView)
              return curView;
          let parent = dom.parentNode;
          dom = parent != cView.dom ? parent : dir > 0 ? dom.nextSibling : dom.previousSibling;
      }
      return null;
  }
  // Used to work around a Safari Selection/shadow DOM bug (#414)
  function safariSelectionRangeHack(view) {
      let found = null;
      // Because Safari (at least in 2018-2021) doesn't provide regular
      // access to the selection inside a shadowroot, we have to perform a
      // ridiculous hack to get at it—using `execCommand` to trigger a
      // `beforeInput` event so that we can read the target range from the
      // event.
      function read(event) {
          event.preventDefault();
          event.stopImmediatePropagation();
          found = event.getTargetRanges()[0];
      }
      view.contentDOM.addEventListener("beforeinput", read, true);
      document.execCommand("indent");
      view.contentDOM.removeEventListener("beforeinput", read, true);
      if (!found)
          return null;
      let anchorNode = found.startContainer, anchorOffset = found.startOffset;
      let focusNode = found.endContainer, focusOffset = found.endOffset;
      let curAnchor = view.docView.domAtPos(view.state.selection.main.anchor);
      // Since such a range doesn't distinguish between anchor and head,
      // use a heuristic that flips it around if its end matches the
      // current anchor.
      if (isEquivalentPosition(curAnchor.node, curAnchor.offset, focusNode, focusOffset))
          [anchorNode, anchorOffset, focusNode, focusOffset] = [focusNode, focusOffset, anchorNode, anchorOffset];
      return { anchorNode, anchorOffset, focusNode, focusOffset };
  }

  function applyDOMChange(view, start, end, typeOver) {
      let change, newSel;
      let sel = view.state.selection.main;
      if (start > -1) {
          let bounds = view.docView.domBoundsAround(start, end, 0);
          if (!bounds || view.state.readOnly)
              return false;
          let { from, to } = bounds;
          let selPoints = view.docView.impreciseHead || view.docView.impreciseAnchor ? [] : selectionPoints(view);
          let reader = new DOMReader(selPoints, view.state);
          reader.readRange(bounds.startDOM, bounds.endDOM);
          let preferredPos = sel.from, preferredSide = null;
          // Prefer anchoring to end when Backspace is pressed (or, on
          // Android, when something was deleted)
          if (view.inputState.lastKeyCode === 8 && view.inputState.lastKeyTime > Date.now() - 100 ||
              browser.android && reader.text.length < to - from) {
              preferredPos = sel.to;
              preferredSide = "end";
          }
          let diff = findDiff(view.state.doc.sliceString(from, to, LineBreakPlaceholder), reader.text, preferredPos - from, preferredSide);
          if (diff) {
              // Chrome inserts two newlines when pressing shift-enter at the
              // end of a line. This drops one of those.
              if (browser.chrome && view.inputState.lastKeyCode == 13 &&
                  diff.toB == diff.from + 2 && reader.text.slice(diff.from, diff.toB) == LineBreakPlaceholder + LineBreakPlaceholder)
                  diff.toB--;
              change = { from: from + diff.from, to: from + diff.toA,
                  insert: Text.of(reader.text.slice(diff.from, diff.toB).split(LineBreakPlaceholder)) };
          }
          newSel = selectionFromPoints(selPoints, from);
      }
      else if (view.hasFocus || !view.state.facet(editable)) {
          let domSel = view.observer.selectionRange;
          let { impreciseHead: iHead, impreciseAnchor: iAnchor } = view.docView;
          let head = iHead && iHead.node == domSel.focusNode && iHead.offset == domSel.focusOffset ||
              !contains(view.contentDOM, domSel.focusNode)
              ? view.state.selection.main.head
              : view.docView.posFromDOM(domSel.focusNode, domSel.focusOffset);
          let anchor = iAnchor && iAnchor.node == domSel.anchorNode && iAnchor.offset == domSel.anchorOffset ||
              !contains(view.contentDOM, domSel.anchorNode)
              ? view.state.selection.main.anchor
              : view.docView.posFromDOM(domSel.anchorNode, domSel.anchorOffset);
          if (head != sel.head || anchor != sel.anchor)
              newSel = EditorSelection.single(anchor, head);
      }
      if (!change && !newSel)
          return false;
      // Heuristic to notice typing over a selected character
      if (!change && typeOver && !sel.empty && newSel && newSel.main.empty)
          change = { from: sel.from, to: sel.to, insert: view.state.doc.slice(sel.from, sel.to) };
      // If the change is inside the selection and covers most of it,
      // assume it is a selection replace (with identical characters at
      // the start/end not included in the diff)
      else if (change && change.from >= sel.from && change.to <= sel.to &&
          (change.from != sel.from || change.to != sel.to) &&
          (sel.to - sel.from) - (change.to - change.from) <= 4)
          change = {
              from: sel.from, to: sel.to,
              insert: view.state.doc.slice(sel.from, change.from).append(change.insert).append(view.state.doc.slice(change.to, sel.to))
          };
      // Detect insert-period-on-double-space Mac behavior, and transform
      // it into a regular space insert.
      else if ((browser.mac || browser.android) && change && change.from == change.to && change.from == sel.head - 1 &&
          change.insert.toString() == ".")
          change = { from: sel.from, to: sel.to, insert: Text.of([" "]) };
      if (change) {
          let startState = view.state;
          if (browser.ios && view.inputState.flushIOSKey(view))
              return true;
          // Android browsers don't fire reasonable key events for enter,
          // backspace, or delete. So this detects changes that look like
          // they're caused by those keys, and reinterprets them as key
          // events. (Some of these keys are also handled by beforeinput
          // events and the pendingAndroidKey mechanism, but that's not
          // reliable in all situations.)
          if (browser.android &&
              ((change.from == sel.from && change.to == sel.to &&
                  change.insert.length == 1 && change.insert.lines == 2 &&
                  dispatchKey(view.contentDOM, "Enter", 13)) ||
                  (change.from == sel.from - 1 && change.to == sel.to && change.insert.length == 0 &&
                      dispatchKey(view.contentDOM, "Backspace", 8)) ||
                  (change.from == sel.from && change.to == sel.to + 1 && change.insert.length == 0 &&
                      dispatchKey(view.contentDOM, "Delete", 46))))
              return true;
          let text = change.insert.toString();
          if (view.state.facet(inputHandler$1).some(h => h(view, change.from, change.to, text)))
              return true;
          if (view.inputState.composing >= 0)
              view.inputState.composing++;
          let tr;
          if (change.from >= sel.from && change.to <= sel.to && change.to - change.from >= (sel.to - sel.from) / 3 &&
              (!newSel || newSel.main.empty && newSel.main.from == change.from + change.insert.length) &&
              view.inputState.composing < 0) {
              let before = sel.from < change.from ? startState.sliceDoc(sel.from, change.from) : "";
              let after = sel.to > change.to ? startState.sliceDoc(change.to, sel.to) : "";
              tr = startState.replaceSelection(view.state.toText(before + change.insert.sliceString(0, undefined, view.state.lineBreak) + after));
          }
          else {
              let changes = startState.changes(change);
              let mainSel = newSel && !startState.selection.main.eq(newSel.main) && newSel.main.to <= changes.newLength
                  ? newSel.main : undefined;
              // Try to apply a composition change to all cursors
              if (startState.selection.ranges.length > 1 && view.inputState.composing >= 0 &&
                  change.to <= sel.to && change.to >= sel.to - 10) {
                  let replaced = view.state.sliceDoc(change.from, change.to);
                  let compositionRange = compositionSurroundingNode(view) || view.state.doc.lineAt(sel.head);
                  let offset = sel.to - change.to, size = sel.to - sel.from;
                  tr = startState.changeByRange(range => {
                      if (range.from == sel.from && range.to == sel.to)
                          return { changes, range: mainSel || range.map(changes) };
                      let to = range.to - offset, from = to - replaced.length;
                      if (range.to - range.from != size || view.state.sliceDoc(from, to) != replaced ||
                          // Unfortunately, there's no way to make multiple
                          // changes in the same node work without aborting
                          // composition, so cursors in the composition range are
                          // ignored.
                          compositionRange && range.to >= compositionRange.from && range.from <= compositionRange.to)
                          return { range };
                      let rangeChanges = startState.changes({ from, to, insert: change.insert }), selOff = range.to - sel.to;
                      return {
                          changes: rangeChanges,
                          range: !mainSel ? range.map(rangeChanges) :
                              EditorSelection.range(Math.max(0, mainSel.anchor + selOff), Math.max(0, mainSel.head + selOff))
                      };
                  });
              }
              else {
                  tr = {
                      changes,
                      selection: mainSel && startState.selection.replaceRange(mainSel)
                  };
              }
          }
          let userEvent = "input.type";
          if (view.composing) {
              userEvent += ".compose";
              if (view.inputState.compositionFirstChange) {
                  userEvent += ".start";
                  view.inputState.compositionFirstChange = false;
              }
          }
          view.dispatch(tr, { scrollIntoView: true, userEvent });
          return true;
      }
      else if (newSel && !newSel.main.eq(sel)) {
          let scrollIntoView = false, userEvent = "select";
          if (view.inputState.lastSelectionTime > Date.now() - 50) {
              if (view.inputState.lastSelectionOrigin == "select")
                  scrollIntoView = true;
              userEvent = view.inputState.lastSelectionOrigin;
          }
          view.dispatch({ selection: newSel, scrollIntoView, userEvent });
          return true;
      }
      else {
          return false;
      }
  }
  function findDiff(a, b, preferredPos, preferredSide) {
      let minLen = Math.min(a.length, b.length);
      let from = 0;
      while (from < minLen && a.charCodeAt(from) == b.charCodeAt(from))
          from++;
      if (from == minLen && a.length == b.length)
          return null;
      let toA = a.length, toB = b.length;
      while (toA > 0 && toB > 0 && a.charCodeAt(toA - 1) == b.charCodeAt(toB - 1)) {
          toA--;
          toB--;
      }
      if (preferredSide == "end") {
          let adjust = Math.max(0, from - Math.min(toA, toB));
          preferredPos -= toA + adjust - from;
      }
      if (toA < from && a.length < b.length) {
          let move = preferredPos <= from && preferredPos >= toA ? from - preferredPos : 0;
          from -= move;
          toB = from + (toB - toA);
          toA = from;
      }
      else if (toB < from) {
          let move = preferredPos <= from && preferredPos >= toB ? from - preferredPos : 0;
          from -= move;
          toA = from + (toA - toB);
          toB = from;
      }
      return { from, toA, toB };
  }
  function selectionPoints(view) {
      let result = [];
      if (view.root.activeElement != view.contentDOM)
          return result;
      let { anchorNode, anchorOffset, focusNode, focusOffset } = view.observer.selectionRange;
      if (anchorNode) {
          result.push(new DOMPoint(anchorNode, anchorOffset));
          if (focusNode != anchorNode || focusOffset != anchorOffset)
              result.push(new DOMPoint(focusNode, focusOffset));
      }
      return result;
  }
  function selectionFromPoints(points, base) {
      if (points.length == 0)
          return null;
      let anchor = points[0].pos, head = points.length == 2 ? points[1].pos : anchor;
      return anchor > -1 && head > -1 ? EditorSelection.single(anchor + base, head + base) : null;
  }

  // The editor's update state machine looks something like this:
  //
  //     Idle → Updating ⇆ Idle (unchecked) → Measuring → Idle
  //                                         ↑      ↓
  //                                         Updating (measure)
  //
  // The difference between 'Idle' and 'Idle (unchecked)' lies in
  // whether a layout check has been scheduled. A regular update through
  // the `update` method updates the DOM in a write-only fashion, and
  // relies on a check (scheduled with `requestAnimationFrame`) to make
  // sure everything is where it should be and the viewport covers the
  // visible code. That check continues to measure and then optionally
  // update until it reaches a coherent state.
  /**
  An editor view represents the editor's user interface. It holds
  the editable DOM surface, and possibly other elements such as the
  line number gutter. It handles events and dispatches state
  transactions for editing actions.
  */
  class EditorView {
      /**
      Construct a new view. You'll want to either provide a `parent`
      option, or put `view.dom` into your document after creating a
      view, so that the user can see the editor.
      */
      constructor(config = {}) {
          this.plugins = [];
          this.pluginMap = new Map;
          this.editorAttrs = {};
          this.contentAttrs = {};
          this.bidiCache = [];
          this.destroyed = false;
          /**
          @internal
          */
          this.updateState = 2 /* Updating */;
          /**
          @internal
          */
          this.measureScheduled = -1;
          /**
          @internal
          */
          this.measureRequests = [];
          this.contentDOM = document.createElement("div");
          this.scrollDOM = document.createElement("div");
          this.scrollDOM.tabIndex = -1;
          this.scrollDOM.className = "cm-scroller";
          this.scrollDOM.appendChild(this.contentDOM);
          this.announceDOM = document.createElement("div");
          this.announceDOM.style.cssText = "position: absolute; top: -10000px";
          this.announceDOM.setAttribute("aria-live", "polite");
          this.dom = document.createElement("div");
          this.dom.appendChild(this.announceDOM);
          this.dom.appendChild(this.scrollDOM);
          this._dispatch = config.dispatch || ((tr) => this.update([tr]));
          this.dispatch = this.dispatch.bind(this);
          this.root = (config.root || getRoot(config.parent) || document);
          this.viewState = new ViewState(config.state || EditorState.create(config));
          this.plugins = this.state.facet(viewPlugin).map(spec => new PluginInstance(spec));
          for (let plugin of this.plugins)
              plugin.update(this);
          this.observer = new DOMObserver(this, (from, to, typeOver) => {
              return applyDOMChange(this, from, to, typeOver);
          }, event => {
              this.inputState.runScrollHandlers(this, event);
              if (this.observer.intersecting)
                  this.measure();
          });
          this.inputState = new InputState(this);
          this.inputState.ensureHandlers(this, this.plugins);
          this.docView = new DocView(this);
          this.mountStyles();
          this.updateAttrs();
          this.updateState = 0 /* Idle */;
          this.requestMeasure();
          if (config.parent)
              config.parent.appendChild(this.dom);
      }
      /**
      The current editor state.
      */
      get state() { return this.viewState.state; }
      /**
      To be able to display large documents without consuming too much
      memory or overloading the browser, CodeMirror only draws the
      code that is visible (plus a margin around it) to the DOM. This
      property tells you the extent of the current drawn viewport, in
      document positions.
      */
      get viewport() { return this.viewState.viewport; }
      /**
      When there are, for example, large collapsed ranges in the
      viewport, its size can be a lot bigger than the actual visible
      content. Thus, if you are doing something like styling the
      content in the viewport, it is preferable to only do so for
      these ranges, which are the subset of the viewport that is
      actually drawn.
      */
      get visibleRanges() { return this.viewState.visibleRanges; }
      /**
      Returns false when the editor is entirely scrolled out of view
      or otherwise hidden.
      */
      get inView() { return this.viewState.inView; }
      /**
      Indicates whether the user is currently composing text via
      [IME](https://en.wikipedia.org/wiki/Input_method), and at least
      one change has been made in the current composition.
      */
      get composing() { return this.inputState.composing > 0; }
      /**
      Indicates whether the user is currently in composing state. Note
      that on some platforms, like Android, this will be the case a
      lot, since just putting the cursor on a word starts a
      composition there.
      */
      get compositionStarted() { return this.inputState.composing >= 0; }
      dispatch(...input) {
          this._dispatch(input.length == 1 && input[0] instanceof Transaction ? input[0]
              : this.state.update(...input));
      }
      /**
      Update the view for the given array of transactions. This will
      update the visible document and selection to match the state
      produced by the transactions, and notify view plugins of the
      change. You should usually call
      [`dispatch`](https://codemirror.net/6/docs/ref/#view.EditorView.dispatch) instead, which uses this
      as a primitive.
      */
      update(transactions) {
          if (this.updateState != 0 /* Idle */)
              throw new Error("Calls to EditorView.update are not allowed while an update is in progress");
          let redrawn = false, attrsChanged = false, update;
          let state = this.state;
          for (let tr of transactions) {
              if (tr.startState != state)
                  throw new RangeError("Trying to update state with a transaction that doesn't start from the previous state.");
              state = tr.state;
          }
          if (this.destroyed) {
              this.viewState.state = state;
              return;
          }
          this.observer.clear();
          // When the phrases change, redraw the editor
          if (state.facet(EditorState.phrases) != this.state.facet(EditorState.phrases))
              return this.setState(state);
          update = ViewUpdate.create(this, state, transactions);
          let scrollTarget = this.viewState.scrollTarget;
          try {
              this.updateState = 2 /* Updating */;
              for (let tr of transactions) {
                  if (scrollTarget)
                      scrollTarget = scrollTarget.map(tr.changes);
                  if (tr.scrollIntoView) {
                      let { main } = tr.state.selection;
                      scrollTarget = new ScrollTarget(main.empty ? main : EditorSelection.cursor(main.head, main.head > main.anchor ? -1 : 1));
                  }
                  for (let e of tr.effects)
                      if (e.is(scrollIntoView$1))
                          scrollTarget = e.value;
              }
              this.viewState.update(update, scrollTarget);
              this.bidiCache = CachedOrder.update(this.bidiCache, update.changes);
              if (!update.empty) {
                  this.updatePlugins(update);
                  this.inputState.update(update);
              }
              redrawn = this.docView.update(update);
              if (this.state.facet(styleModule) != this.styleModules)
                  this.mountStyles();
              attrsChanged = this.updateAttrs();
              this.showAnnouncements(transactions);
              this.docView.updateSelection(redrawn, transactions.some(tr => tr.isUserEvent("select.pointer")));
          }
          finally {
              this.updateState = 0 /* Idle */;
          }
          if (update.startState.facet(theme) != update.state.facet(theme))
              this.viewState.mustMeasureContent = true;
          if (redrawn || attrsChanged || scrollTarget || this.viewState.mustEnforceCursorAssoc || this.viewState.mustMeasureContent)
              this.requestMeasure();
          if (!update.empty)
              for (let listener of this.state.facet(updateListener))
                  listener(update);
      }
      /**
      Reset the view to the given state. (This will cause the entire
      document to be redrawn and all view plugins to be reinitialized,
      so you should probably only use it when the new state isn't
      derived from the old state. Otherwise, use
      [`dispatch`](https://codemirror.net/6/docs/ref/#view.EditorView.dispatch) instead.)
      */
      setState(newState) {
          if (this.updateState != 0 /* Idle */)
              throw new Error("Calls to EditorView.setState are not allowed while an update is in progress");
          if (this.destroyed) {
              this.viewState.state = newState;
              return;
          }
          this.updateState = 2 /* Updating */;
          let hadFocus = this.hasFocus;
          try {
              for (let plugin of this.plugins)
                  plugin.destroy(this);
              this.viewState = new ViewState(newState);
              this.plugins = newState.facet(viewPlugin).map(spec => new PluginInstance(spec));
              this.pluginMap.clear();
              for (let plugin of this.plugins)
                  plugin.update(this);
              this.docView = new DocView(this);
              this.inputState.ensureHandlers(this, this.plugins);
              this.mountStyles();
              this.updateAttrs();
              this.bidiCache = [];
          }
          finally {
              this.updateState = 0 /* Idle */;
          }
          if (hadFocus)
              this.focus();
          this.requestMeasure();
      }
      updatePlugins(update) {
          let prevSpecs = update.startState.facet(viewPlugin), specs = update.state.facet(viewPlugin);
          if (prevSpecs != specs) {
              let newPlugins = [];
              for (let spec of specs) {
                  let found = prevSpecs.indexOf(spec);
                  if (found < 0) {
                      newPlugins.push(new PluginInstance(spec));
                  }
                  else {
                      let plugin = this.plugins[found];
                      plugin.mustUpdate = update;
                      newPlugins.push(plugin);
                  }
              }
              for (let plugin of this.plugins)
                  if (plugin.mustUpdate != update)
                      plugin.destroy(this);
              this.plugins = newPlugins;
              this.pluginMap.clear();
              this.inputState.ensureHandlers(this, this.plugins);
          }
          else {
              for (let p of this.plugins)
                  p.mustUpdate = update;
          }
          for (let i = 0; i < this.plugins.length; i++)
              this.plugins[i].update(this);
      }
      /**
      @internal
      */
      measure(flush = true) {
          if (this.destroyed)
              return;
          if (this.measureScheduled > -1)
              cancelAnimationFrame(this.measureScheduled);
          this.measureScheduled = 0; // Prevent requestMeasure calls from scheduling another animation frame
          if (flush)
              this.observer.forceFlush();
          let updated = null;
          try {
              for (let i = 0;; i++) {
                  this.updateState = 1 /* Measuring */;
                  let oldViewport = this.viewport;
                  let changed = this.viewState.measure(this);
                  if (!changed && !this.measureRequests.length && this.viewState.scrollTarget == null)
                      break;
                  if (i > 5) {
                      console.warn(this.measureRequests.length
                          ? "Measure loop restarted more than 5 times"
                          : "Viewport failed to stabilize");
                      break;
                  }
                  let measuring = [];
                  // Only run measure requests in this cycle when the viewport didn't change
                  if (!(changed & 4 /* Viewport */))
                      [this.measureRequests, measuring] = [measuring, this.measureRequests];
                  let measured = measuring.map(m => {
                      try {
                          return m.read(this);
                      }
                      catch (e) {
                          logException(this.state, e);
                          return BadMeasure;
                      }
                  });
                  let update = ViewUpdate.create(this, this.state, []), redrawn = false, scrolled = false;
                  update.flags |= changed;
                  if (!updated)
                      updated = update;
                  else
                      updated.flags |= changed;
                  this.updateState = 2 /* Updating */;
                  if (!update.empty) {
                      this.updatePlugins(update);
                      this.inputState.update(update);
                      this.updateAttrs();
                      redrawn = this.docView.update(update);
                  }
                  for (let i = 0; i < measuring.length; i++)
                      if (measured[i] != BadMeasure) {
                          try {
                              let m = measuring[i];
                              if (m.write)
                                  m.write(measured[i], this);
                          }
                          catch (e) {
                              logException(this.state, e);
                          }
                      }
                  if (this.viewState.scrollTarget) {
                      this.docView.scrollIntoView(this.viewState.scrollTarget);
                      this.viewState.scrollTarget = null;
                      scrolled = true;
                  }
                  if (redrawn)
                      this.docView.updateSelection(true);
                  if (this.viewport.from == oldViewport.from && this.viewport.to == oldViewport.to &&
                      !scrolled && this.measureRequests.length == 0)
                      break;
              }
          }
          finally {
              this.updateState = 0 /* Idle */;
              this.measureScheduled = -1;
          }
          if (updated && !updated.empty)
              for (let listener of this.state.facet(updateListener))
                  listener(updated);
      }
      /**
      Get the CSS classes for the currently active editor themes.
      */
      get themeClasses() {
          return baseThemeID + " " +
              (this.state.facet(darkTheme) ? baseDarkID : baseLightID) + " " +
              this.state.facet(theme);
      }
      updateAttrs() {
          let editorAttrs = attrsFromFacet(this, editorAttributes, {
              class: "cm-editor" + (this.hasFocus ? " cm-focused " : " ") + this.themeClasses
          });
          let contentAttrs = {
              spellcheck: "false",
              autocorrect: "off",
              autocapitalize: "off",
              translate: "no",
              contenteditable: !this.state.facet(editable) ? "false" : "true",
              class: "cm-content",
              style: `${browser.tabSize}: ${this.state.tabSize}`,
              role: "textbox",
              "aria-multiline": "true"
          };
          if (this.state.readOnly)
              contentAttrs["aria-readonly"] = "true";
          attrsFromFacet(this, contentAttributes, contentAttrs);
          let changed = this.observer.ignore(() => {
              let changedContent = updateAttrs(this.contentDOM, this.contentAttrs, contentAttrs);
              let changedEditor = updateAttrs(this.dom, this.editorAttrs, editorAttrs);
              return changedContent || changedEditor;
          });
          this.editorAttrs = editorAttrs;
          this.contentAttrs = contentAttrs;
          return changed;
      }
      showAnnouncements(trs) {
          let first = true;
          for (let tr of trs)
              for (let effect of tr.effects)
                  if (effect.is(EditorView.announce)) {
                      if (first)
                          this.announceDOM.textContent = "";
                      first = false;
                      let div = this.announceDOM.appendChild(document.createElement("div"));
                      div.textContent = effect.value;
                  }
      }
      mountStyles() {
          this.styleModules = this.state.facet(styleModule);
          StyleModule.mount(this.root, this.styleModules.concat(baseTheme$1$3).reverse());
      }
      readMeasured() {
          if (this.updateState == 2 /* Updating */)
              throw new Error("Reading the editor layout isn't allowed during an update");
          if (this.updateState == 0 /* Idle */ && this.measureScheduled > -1)
              this.measure(false);
      }
      /**
      Schedule a layout measurement, optionally providing callbacks to
      do custom DOM measuring followed by a DOM write phase. Using
      this is preferable reading DOM layout directly from, for
      example, an event handler, because it'll make sure measuring and
      drawing done by other components is synchronized, avoiding
      unnecessary DOM layout computations.
      */
      requestMeasure(request) {
          if (this.measureScheduled < 0)
              this.measureScheduled = requestAnimationFrame(() => this.measure());
          if (request) {
              if (request.key != null)
                  for (let i = 0; i < this.measureRequests.length; i++) {
                      if (this.measureRequests[i].key === request.key) {
                          this.measureRequests[i] = request;
                          return;
                      }
                  }
              this.measureRequests.push(request);
          }
      }
      /**
      Get the value of a specific plugin, if present. Note that
      plugins that crash can be dropped from a view, so even when you
      know you registered a given plugin, it is recommended to check
      the return value of this method.
      */
      plugin(plugin) {
          let known = this.pluginMap.get(plugin);
          if (known === undefined || known && known.spec != plugin)
              this.pluginMap.set(plugin, known = this.plugins.find(p => p.spec == plugin) || null);
          return known && known.update(this).value;
      }
      /**
      The top position of the document, in screen coordinates. This
      may be negative when the editor is scrolled down. Points
      directly to the top of the first line, not above the padding.
      */
      get documentTop() {
          return this.contentDOM.getBoundingClientRect().top + this.viewState.paddingTop;
      }
      /**
      Reports the padding above and below the document.
      */
      get documentPadding() {
          return { top: this.viewState.paddingTop, bottom: this.viewState.paddingBottom };
      }
      /**
      Find the text line or block widget at the given vertical
      position (which is interpreted as relative to the [top of the
      document](https://codemirror.net/6/docs/ref/#view.EditorView.documentTop)
      */
      elementAtHeight(height) {
          this.readMeasured();
          return this.viewState.elementAtHeight(height);
      }
      /**
      Find the line block (see
      [`lineBlockAt`](https://codemirror.net/6/docs/ref/#view.EditorView.lineBlockAt) at the given
      height.
      */
      lineBlockAtHeight(height) {
          this.readMeasured();
          return this.viewState.lineBlockAtHeight(height);
      }
      /**
      Get the extent and vertical position of all [line
      blocks](https://codemirror.net/6/docs/ref/#view.EditorView.lineBlockAt) in the viewport. Positions
      are relative to the [top of the
      document](https://codemirror.net/6/docs/ref/#view.EditorView.documentTop);
      */
      get viewportLineBlocks() {
          return this.viewState.viewportLines;
      }
      /**
      Find the line block around the given document position. A line
      block is a range delimited on both sides by either a
      non-[hidden](https://codemirror.net/6/docs/ref/#view.Decoration^replace) line breaks, or the
      start/end of the document. It will usually just hold a line of
      text, but may be broken into multiple textblocks by block
      widgets.
      */
      lineBlockAt(pos) {
          return this.viewState.lineBlockAt(pos);
      }
      /**
      The editor's total content height.
      */
      get contentHeight() {
          return this.viewState.contentHeight;
      }
      /**
      Move a cursor position by [grapheme
      cluster](https://codemirror.net/6/docs/ref/#state.findClusterBreak). `forward` determines whether
      the motion is away from the line start, or towards it. In
      bidirectional text, the line is traversed in visual order, using
      the editor's [text direction](https://codemirror.net/6/docs/ref/#view.EditorView.textDirection).
      When the start position was the last one on the line, the
      returned position will be across the line break. If there is no
      further line, the original position is returned.
      
      By default, this method moves over a single cluster. The
      optional `by` argument can be used to move across more. It will
      be called with the first cluster as argument, and should return
      a predicate that determines, for each subsequent cluster,
      whether it should also be moved over.
      */
      moveByChar(start, forward, by) {
          return skipAtoms(this, start, moveByChar(this, start, forward, by));
      }
      /**
      Move a cursor position across the next group of either
      [letters](https://codemirror.net/6/docs/ref/#state.EditorState.charCategorizer) or non-letter
      non-whitespace characters.
      */
      moveByGroup(start, forward) {
          return skipAtoms(this, start, moveByChar(this, start, forward, initial => byGroup(this, start.head, initial)));
      }
      /**
      Move to the next line boundary in the given direction. If
      `includeWrap` is true, line wrapping is on, and there is a
      further wrap point on the current line, the wrap point will be
      returned. Otherwise this function will return the start or end
      of the line.
      */
      moveToLineBoundary(start, forward, includeWrap = true) {
          return moveToLineBoundary(this, start, forward, includeWrap);
      }
      /**
      Move a cursor position vertically. When `distance` isn't given,
      it defaults to moving to the next line (including wrapped
      lines). Otherwise, `distance` should provide a positive distance
      in pixels.
      
      When `start` has a
      [`goalColumn`](https://codemirror.net/6/docs/ref/#state.SelectionRange.goalColumn), the vertical
      motion will use that as a target horizontal position. Otherwise,
      the cursor's own horizontal position is used. The returned
      cursor will have its goal column set to whichever column was
      used.
      */
      moveVertically(start, forward, distance) {
          return skipAtoms(this, start, moveVertically(this, start, forward, distance));
      }
      /**
      Find the DOM parent node and offset (child offset if `node` is
      an element, character offset when it is a text node) at the
      given document position.
      
      Note that for positions that aren't currently in
      `visibleRanges`, the resulting DOM position isn't necessarily
      meaningful (it may just point before or after a placeholder
      element).
      */
      domAtPos(pos) {
          return this.docView.domAtPos(pos);
      }
      /**
      Find the document position at the given DOM node. Can be useful
      for associating positions with DOM events. Will raise an error
      when `node` isn't part of the editor content.
      */
      posAtDOM(node, offset = 0) {
          return this.docView.posFromDOM(node, offset);
      }
      posAtCoords(coords, precise = true) {
          this.readMeasured();
          return posAtCoords(this, coords, precise);
      }
      /**
      Get the screen coordinates at the given document position.
      `side` determines whether the coordinates are based on the
      element before (-1) or after (1) the position (if no element is
      available on the given side, the method will transparently use
      another strategy to get reasonable coordinates).
      */
      coordsAtPos(pos, side = 1) {
          this.readMeasured();
          let rect = this.docView.coordsAt(pos, side);
          if (!rect || rect.left == rect.right)
              return rect;
          let line = this.state.doc.lineAt(pos), order = this.bidiSpans(line);
          let span = order[BidiSpan.find(order, pos - line.from, -1, side)];
          return flattenRect(rect, (span.dir == Direction.LTR) == (side > 0));
      }
      /**
      The default width of a character in the editor. May not
      accurately reflect the width of all characters (given variable
      width fonts or styling of invididual ranges).
      */
      get defaultCharacterWidth() { return this.viewState.heightOracle.charWidth; }
      /**
      The default height of a line in the editor. May not be accurate
      for all lines.
      */
      get defaultLineHeight() { return this.viewState.heightOracle.lineHeight; }
      /**
      The text direction
      ([`direction`](https://developer.mozilla.org/en-US/docs/Web/CSS/direction)
      CSS property) of the editor's content element.
      */
      get textDirection() { return this.viewState.defaultTextDirection; }
      /**
      Find the text direction of the block at the given position, as
      assigned by CSS. If
      [`perLineTextDirection`](https://codemirror.net/6/docs/ref/#view.EditorView^perLineTextDirection)
      isn't enabled, or the given position is outside of the viewport,
      this will always return the same as
      [`textDirection`](https://codemirror.net/6/docs/ref/#view.EditorView.textDirection). Note that
      this may trigger a DOM layout.
      */
      textDirectionAt(pos) {
          let perLine = this.state.facet(perLineTextDirection);
          if (!perLine || pos < this.viewport.from || pos > this.viewport.to)
              return this.textDirection;
          this.readMeasured();
          return this.docView.textDirectionAt(pos);
      }
      /**
      Whether this editor [wraps lines](https://codemirror.net/6/docs/ref/#view.EditorView.lineWrapping)
      (as determined by the
      [`white-space`](https://developer.mozilla.org/en-US/docs/Web/CSS/white-space)
      CSS property of its content element).
      */
      get lineWrapping() { return this.viewState.heightOracle.lineWrapping; }
      /**
      Returns the bidirectional text structure of the given line
      (which should be in the current document) as an array of span
      objects. The order of these spans matches the [text
      direction](https://codemirror.net/6/docs/ref/#view.EditorView.textDirection)—if that is
      left-to-right, the leftmost spans come first, otherwise the
      rightmost spans come first.
      */
      bidiSpans(line) {
          if (line.length > MaxBidiLine)
              return trivialOrder(line.length);
          let dir = this.textDirectionAt(line.from);
          for (let entry of this.bidiCache)
              if (entry.from == line.from && entry.dir == dir)
                  return entry.order;
          let order = computeOrder(line.text, dir);
          this.bidiCache.push(new CachedOrder(line.from, line.to, dir, order));
          return order;
      }
      /**
      Check whether the editor has focus.
      */
      get hasFocus() {
          var _a;
          // Safari return false for hasFocus when the context menu is open
          // or closing, which leads us to ignore selection changes from the
          // context menu because it looks like the editor isn't focused.
          // This kludges around that.
          return (document.hasFocus() || browser.safari && ((_a = this.inputState) === null || _a === void 0 ? void 0 : _a.lastContextMenu) > Date.now() - 3e4) &&
              this.root.activeElement == this.contentDOM;
      }
      /**
      Put focus on the editor.
      */
      focus() {
          this.observer.ignore(() => {
              focusPreventScroll(this.contentDOM);
              this.docView.updateSelection();
          });
      }
      /**
      Clean up this editor view, removing its element from the
      document, unregistering event handlers, and notifying
      plugins. The view instance can no longer be used after
      calling this.
      */
      destroy() {
          for (let plugin of this.plugins)
              plugin.destroy(this);
          this.plugins = [];
          this.inputState.destroy();
          this.dom.remove();
          this.observer.destroy();
          if (this.measureScheduled > -1)
              cancelAnimationFrame(this.measureScheduled);
          this.destroyed = true;
      }
      /**
      Returns an effect that can be
      [added](https://codemirror.net/6/docs/ref/#state.TransactionSpec.effects) to a transaction to
      cause it to scroll the given position or range into view.
      */
      static scrollIntoView(pos, options = {}) {
          return scrollIntoView$1.of(new ScrollTarget(typeof pos == "number" ? EditorSelection.cursor(pos) : pos, options.y, options.x, options.yMargin, options.xMargin));
      }
      /**
      Returns an extension that can be used to add DOM event handlers.
      The value should be an object mapping event names to handler
      functions. For any given event, such functions are ordered by
      extension precedence, and the first handler to return true will
      be assumed to have handled that event, and no other handlers or
      built-in behavior will be activated for it. These are registered
      on the [content element](https://codemirror.net/6/docs/ref/#view.EditorView.contentDOM), except
      for `scroll` handlers, which will be called any time the
      editor's [scroll element](https://codemirror.net/6/docs/ref/#view.EditorView.scrollDOM) or one of
      its parent nodes is scrolled.
      */
      static domEventHandlers(handlers) {
          return ViewPlugin.define(() => ({}), { eventHandlers: handlers });
      }
      /**
      Create a theme extension. The first argument can be a
      [`style-mod`](https://github.com/marijnh/style-mod#documentation)
      style spec providing the styles for the theme. These will be
      prefixed with a generated class for the style.
      
      Because the selectors will be prefixed with a scope class, rule
      that directly match the editor's [wrapper
      element](https://codemirror.net/6/docs/ref/#view.EditorView.dom)—to which the scope class will be
      added—need to be explicitly differentiated by adding an `&` to
      the selector for that element—for example
      `&.cm-focused`.
      
      When `dark` is set to true, the theme will be marked as dark,
      which will cause the `&dark` rules from [base
      themes](https://codemirror.net/6/docs/ref/#view.EditorView^baseTheme) to be used (as opposed to
      `&light` when a light theme is active).
      */
      static theme(spec, options) {
          let prefix = StyleModule.newName();
          let result = [theme.of(prefix), styleModule.of(buildTheme(`.${prefix}`, spec))];
          if (options && options.dark)
              result.push(darkTheme.of(true));
          return result;
      }
      /**
      Create an extension that adds styles to the base theme. Like
      with [`theme`](https://codemirror.net/6/docs/ref/#view.EditorView^theme), use `&` to indicate the
      place of the editor wrapper element when directly targeting
      that. You can also use `&dark` or `&light` instead to only
      target editors with a dark or light theme.
      */
      static baseTheme(spec) {
          return Prec.lowest(styleModule.of(buildTheme("." + baseThemeID, spec, lightDarkIDs)));
      }
      /**
      Retrieve an editor view instance from the view's DOM
      representation.
      */
      static findFromDOM(dom) {
          var _a;
          let content = dom.querySelector(".cm-content");
          let cView = content && ContentView.get(content) || ContentView.get(dom);
          return ((_a = cView === null || cView === void 0 ? void 0 : cView.rootView) === null || _a === void 0 ? void 0 : _a.view) || null;
      }
  }
  /**
  Facet to add a [style
  module](https://github.com/marijnh/style-mod#documentation) to
  an editor view. The view will ensure that the module is
  mounted in its [document
  root](https://codemirror.net/6/docs/ref/#view.EditorView.constructor^config.root).
  */
  EditorView.styleModule = styleModule;
  /**
  An input handler can override the way changes to the editable
  DOM content are handled. Handlers are passed the document
  positions between which the change was found, and the new
  content. When one returns true, no further input handlers are
  called and the default behavior is prevented.
  */
  EditorView.inputHandler = inputHandler$1;
  /**
  By default, the editor assumes all its content has the same
  [text direction](https://codemirror.net/6/docs/ref/#view.Direction). Configure this with a `true`
  value to make it read the text direction of every (rendered)
  line separately.
  */
  EditorView.perLineTextDirection = perLineTextDirection;
  /**
  Allows you to provide a function that should be called when the
  library catches an exception from an extension (mostly from view
  plugins, but may be used by other extensions to route exceptions
  from user-code-provided callbacks). This is mostly useful for
  debugging and logging. See [`logException`](https://codemirror.net/6/docs/ref/#view.logException).
  */
  EditorView.exceptionSink = exceptionSink;
  /**
  A facet that can be used to register a function to be called
  every time the view updates.
  */
  EditorView.updateListener = updateListener;
  /**
  Facet that controls whether the editor content DOM is editable.
  When its highest-precedence value is `false`, the element will
  not have its `contenteditable` attribute set. (Note that this
  doesn't affect API calls that change the editor content, even
  when those are bound to keys or buttons. See the
  [`readOnly`](https://codemirror.net/6/docs/ref/#state.EditorState.readOnly) facet for that.)
  */
  EditorView.editable = editable;
  /**
  Allows you to influence the way mouse selection happens. The
  functions in this facet will be called for a `mousedown` event
  on the editor, and can return an object that overrides the way a
  selection is computed from that mouse click or drag.
  */
  EditorView.mouseSelectionStyle = mouseSelectionStyle;
  /**
  Facet used to configure whether a given selection drag event
  should move or copy the selection. The given predicate will be
  called with the `mousedown` event, and can return `true` when
  the drag should move the content.
  */
  EditorView.dragMovesSelection = dragMovesSelection$1;
  /**
  Facet used to configure whether a given selecting click adds a
  new range to the existing selection or replaces it entirely. The
  default behavior is to check `event.metaKey` on macOS, and
  `event.ctrlKey` elsewhere.
  */
  EditorView.clickAddsSelectionRange = clickAddsSelectionRange;
  /**
  A facet that determines which [decorations](https://codemirror.net/6/docs/ref/#view.Decoration)
  are shown in the view. Decorations can be provided in two
  ways—directly, or via a function that takes an editor view.

  Only decoration sets provided directly are allowed to influence
  the editor's vertical layout structure. The ones provided as
  functions are called _after_ the new viewport has been computed,
  and thus **must not** introduce block widgets or replacing
  decorations that cover line breaks.
  */
  EditorView.decorations = decorations;
  /**
  Used to provide ranges that should be treated as atoms as far as
  cursor motion is concerned. This causes methods like
  [`moveByChar`](https://codemirror.net/6/docs/ref/#view.EditorView.moveByChar) and
  [`moveVertically`](https://codemirror.net/6/docs/ref/#view.EditorView.moveVertically) (and the
  commands built on top of them) to skip across such regions when
  a selection endpoint would enter them. This does _not_ prevent
  direct programmatic [selection
  updates](https://codemirror.net/6/docs/ref/#state.TransactionSpec.selection) from moving into such
  regions.
  */
  EditorView.atomicRanges = atomicRanges;
  /**
  Facet that allows extensions to provide additional scroll
  margins (space around the sides of the scrolling element that
  should be considered invisible). This can be useful when the
  plugin introduces elements that cover part of that element (for
  example a horizontally fixed gutter).
  */
  EditorView.scrollMargins = scrollMargins;
  /**
  This facet records whether a dark theme is active. The extension
  returned by [`theme`](https://codemirror.net/6/docs/ref/#view.EditorView^theme) automatically
  includes an instance of this when the `dark` option is set to
  true.
  */
  EditorView.darkTheme = darkTheme;
  /**
  Facet that provides additional DOM attributes for the editor's
  editable DOM element.
  */
  EditorView.contentAttributes = contentAttributes;
  /**
  Facet that provides DOM attributes for the editor's outer
  element.
  */
  EditorView.editorAttributes = editorAttributes;
  /**
  An extension that enables line wrapping in the editor (by
  setting CSS `white-space` to `pre-wrap` in the content).
  */
  EditorView.lineWrapping = /*@__PURE__*/EditorView.contentAttributes.of({ "class": "cm-lineWrapping" });
  /**
  State effect used to include screen reader announcements in a
  transaction. These will be added to the DOM in a visually hidden
  element with `aria-live="polite"` set, and should be used to
  describe effects that are visually obvious but may not be
  noticed by screen reader users (such as moving to the next
  search match).
  */
  EditorView.announce = /*@__PURE__*/StateEffect.define();
  // Maximum line length for which we compute accurate bidi info
  const MaxBidiLine = 4096;
  const BadMeasure = {};
  class CachedOrder {
      constructor(from, to, dir, order) {
          this.from = from;
          this.to = to;
          this.dir = dir;
          this.order = order;
      }
      static update(cache, changes) {
          if (changes.empty)
              return cache;
          let result = [], lastDir = cache.length ? cache[cache.length - 1].dir : Direction.LTR;
          for (let i = Math.max(0, cache.length - 10); i < cache.length; i++) {
              let entry = cache[i];
              if (entry.dir == lastDir && !changes.touchesRange(entry.from, entry.to))
                  result.push(new CachedOrder(changes.mapPos(entry.from, 1), changes.mapPos(entry.to, -1), entry.dir, entry.order));
          }
          return result;
      }
  }
  function attrsFromFacet(view, facet, base) {
      for (let sources = view.state.facet(facet), i = sources.length - 1; i >= 0; i--) {
          let source = sources[i], value = typeof source == "function" ? source(view) : source;
          if (value)
              combineAttrs(value, base);
      }
      return base;
  }

  const currentPlatform = browser.mac ? "mac" : browser.windows ? "win" : browser.linux ? "linux" : "key";
  function normalizeKeyName(name, platform) {
      const parts = name.split(/-(?!$)/);
      let result = parts[parts.length - 1];
      if (result == "Space")
          result = " ";
      let alt, ctrl, shift, meta;
      for (let i = 0; i < parts.length - 1; ++i) {
          const mod = parts[i];
          if (/^(cmd|meta|m)$/i.test(mod))
              meta = true;
          else if (/^a(lt)?$/i.test(mod))
              alt = true;
          else if (/^(c|ctrl|control)$/i.test(mod))
              ctrl = true;
          else if (/^s(hift)?$/i.test(mod))
              shift = true;
          else if (/^mod$/i.test(mod)) {
              if (platform == "mac")
                  meta = true;
              else
                  ctrl = true;
          }
          else
              throw new Error("Unrecognized modifier name: " + mod);
      }
      if (alt)
          result = "Alt-" + result;
      if (ctrl)
          result = "Ctrl-" + result;
      if (meta)
          result = "Meta-" + result;
      if (shift)
          result = "Shift-" + result;
      return result;
  }
  function modifiers(name, event, shift) {
      if (event.altKey)
          name = "Alt-" + name;
      if (event.ctrlKey)
          name = "Ctrl-" + name;
      if (event.metaKey)
          name = "Meta-" + name;
      if (shift !== false && event.shiftKey)
          name = "Shift-" + name;
      return name;
  }
  const handleKeyEvents = /*@__PURE__*/Prec.default(/*@__PURE__*/EditorView.domEventHandlers({
      keydown(event, view) {
          return runHandlers(getKeymap(view.state), event, view, "editor");
      }
  }));
  /**
  Facet used for registering keymaps.

  You can add multiple keymaps to an editor. Their priorities
  determine their precedence (the ones specified early or with high
  priority get checked first). When a handler has returned `true`
  for a given key, no further handlers are called.
  */
  const keymap = /*@__PURE__*/Facet.define({ enables: handleKeyEvents });
  const Keymaps = /*@__PURE__*/new WeakMap();
  // This is hidden behind an indirection, rather than directly computed
  // by the facet, to keep internal types out of the facet's type.
  function getKeymap(state) {
      let bindings = state.facet(keymap);
      let map = Keymaps.get(bindings);
      if (!map)
          Keymaps.set(bindings, map = buildKeymap(bindings.reduce((a, b) => a.concat(b), [])));
      return map;
  }
  /**
  Run the key handlers registered for a given scope. The event
  object should be a `"keydown"` event. Returns true if any of the
  handlers handled it.
  */
  function runScopeHandlers(view, event, scope) {
      return runHandlers(getKeymap(view.state), event, view, scope);
  }
  let storedPrefix = null;
  const PrefixTimeout = 4000;
  function buildKeymap(bindings, platform = currentPlatform) {
      let bound = Object.create(null);
      let isPrefix = Object.create(null);
      let checkPrefix = (name, is) => {
          let current = isPrefix[name];
          if (current == null)
              isPrefix[name] = is;
          else if (current != is)
              throw new Error("Key binding " + name + " is used both as a regular binding and as a multi-stroke prefix");
      };
      let add = (scope, key, command, preventDefault) => {
          let scopeObj = bound[scope] || (bound[scope] = Object.create(null));
          let parts = key.split(/ (?!$)/).map(k => normalizeKeyName(k, platform));
          for (let i = 1; i < parts.length; i++) {
              let prefix = parts.slice(0, i).join(" ");
              checkPrefix(prefix, true);
              if (!scopeObj[prefix])
                  scopeObj[prefix] = {
                      preventDefault: true,
                      commands: [(view) => {
                              let ourObj = storedPrefix = { view, prefix, scope };
                              setTimeout(() => { if (storedPrefix == ourObj)
                                  storedPrefix = null; }, PrefixTimeout);
                              return true;
                          }]
                  };
          }
          let full = parts.join(" ");
          checkPrefix(full, false);
          let binding = scopeObj[full] || (scopeObj[full] = { preventDefault: false, commands: [] });
          binding.commands.push(command);
          if (preventDefault)
              binding.preventDefault = true;
      };
      for (let b of bindings) {
          let name = b[platform] || b.key;
          if (!name)
              continue;
          for (let scope of b.scope ? b.scope.split(" ") : ["editor"]) {
              add(scope, name, b.run, b.preventDefault);
              if (b.shift)
                  add(scope, "Shift-" + name, b.shift, b.preventDefault);
          }
      }
      return bound;
  }
  function runHandlers(map, event, view, scope) {
      let name = keyName(event);
      let charCode = codePointAt(name, 0), isChar = codePointSize(charCode) == name.length && name != " ";
      let prefix = "", fallthrough = false;
      if (storedPrefix && storedPrefix.view == view && storedPrefix.scope == scope) {
          prefix = storedPrefix.prefix + " ";
          if (fallthrough = modifierCodes.indexOf(event.keyCode) < 0)
              storedPrefix = null;
      }
      let runFor = (binding) => {
          if (binding) {
              for (let cmd of binding.commands)
                  if (cmd(view))
                      return true;
              if (binding.preventDefault)
                  fallthrough = true;
          }
          return false;
      };
      let scopeObj = map[scope], baseName;
      if (scopeObj) {
          if (runFor(scopeObj[prefix + modifiers(name, event, !isChar)]))
              return true;
          if (isChar && (event.shiftKey || event.altKey || event.metaKey || charCode > 127) &&
              (baseName = base[event.keyCode]) && baseName != name) {
              if (runFor(scopeObj[prefix + modifiers(baseName, event, true)]))
                  return true;
              else if (event.shiftKey && shift[event.keyCode] != baseName &&
                  runFor(scopeObj[prefix + modifiers(shift[event.keyCode], event, false)]))
                  return true;
          }
          else if (isChar && event.shiftKey) {
              if (runFor(scopeObj[prefix + modifiers(name, event, true)]))
                  return true;
          }
      }
      return fallthrough;
  }

  const CanHidePrimary = !browser.ios; // FIXME test IE
  const selectionConfig = /*@__PURE__*/Facet.define({
      combine(configs) {
          return combineConfig(configs, {
              cursorBlinkRate: 1200,
              drawRangeCursor: true
          }, {
              cursorBlinkRate: (a, b) => Math.min(a, b),
              drawRangeCursor: (a, b) => a || b
          });
      }
  });
  /**
  Returns an extension that hides the browser's native selection and
  cursor, replacing the selection with a background behind the text
  (with the `cm-selectionBackground` class), and the
  cursors with elements overlaid over the code (using
  `cm-cursor-primary` and `cm-cursor-secondary`).

  This allows the editor to display secondary selection ranges, and
  tends to produce a type of selection more in line with that users
  expect in a text editor (the native selection styling will often
  leave gaps between lines and won't fill the horizontal space after
  a line when the selection continues past it).

  It does have a performance cost, in that it requires an extra DOM
  layout cycle for many updates (the selection is drawn based on DOM
  layout information that's only available after laying out the
  content).
  */
  function drawSelection(config = {}) {
      return [
          selectionConfig.of(config),
          drawSelectionPlugin,
          hideNativeSelection
      ];
  }
  class Piece {
      constructor(left, top, width, height, className) {
          this.left = left;
          this.top = top;
          this.width = width;
          this.height = height;
          this.className = className;
      }
      draw() {
          let elt = document.createElement("div");
          elt.className = this.className;
          this.adjust(elt);
          return elt;
      }
      adjust(elt) {
          elt.style.left = this.left + "px";
          elt.style.top = this.top + "px";
          if (this.width >= 0)
              elt.style.width = this.width + "px";
          elt.style.height = this.height + "px";
      }
      eq(p) {
          return this.left == p.left && this.top == p.top && this.width == p.width && this.height == p.height &&
              this.className == p.className;
      }
  }
  const drawSelectionPlugin = /*@__PURE__*/ViewPlugin.fromClass(class {
      constructor(view) {
          this.view = view;
          this.rangePieces = [];
          this.cursors = [];
          this.measureReq = { read: this.readPos.bind(this), write: this.drawSel.bind(this) };
          this.selectionLayer = view.scrollDOM.appendChild(document.createElement("div"));
          this.selectionLayer.className = "cm-selectionLayer";
          this.selectionLayer.setAttribute("aria-hidden", "true");
          this.cursorLayer = view.scrollDOM.appendChild(document.createElement("div"));
          this.cursorLayer.className = "cm-cursorLayer";
          this.cursorLayer.setAttribute("aria-hidden", "true");
          view.requestMeasure(this.measureReq);
          this.setBlinkRate();
      }
      setBlinkRate() {
          this.cursorLayer.style.animationDuration = this.view.state.facet(selectionConfig).cursorBlinkRate + "ms";
      }
      update(update) {
          let confChanged = update.startState.facet(selectionConfig) != update.state.facet(selectionConfig);
          if (confChanged || update.selectionSet || update.geometryChanged || update.viewportChanged)
              this.view.requestMeasure(this.measureReq);
          if (update.transactions.some(tr => tr.scrollIntoView))
              this.cursorLayer.style.animationName = this.cursorLayer.style.animationName == "cm-blink" ? "cm-blink2" : "cm-blink";
          if (confChanged)
              this.setBlinkRate();
      }
      readPos() {
          let { state } = this.view, conf = state.facet(selectionConfig);
          let rangePieces = state.selection.ranges.map(r => r.empty ? [] : measureRange(this.view, r)).reduce((a, b) => a.concat(b));
          let cursors = [];
          for (let r of state.selection.ranges) {
              let prim = r == state.selection.main;
              if (r.empty ? !prim || CanHidePrimary : conf.drawRangeCursor) {
                  let piece = measureCursor(this.view, r, prim);
                  if (piece)
                      cursors.push(piece);
              }
          }
          return { rangePieces, cursors };
      }
      drawSel({ rangePieces, cursors }) {
          if (rangePieces.length != this.rangePieces.length || rangePieces.some((p, i) => !p.eq(this.rangePieces[i]))) {
              this.selectionLayer.textContent = "";
              for (let p of rangePieces)
                  this.selectionLayer.appendChild(p.draw());
              this.rangePieces = rangePieces;
          }
          if (cursors.length != this.cursors.length || cursors.some((c, i) => !c.eq(this.cursors[i]))) {
              let oldCursors = this.cursorLayer.children;
              if (oldCursors.length !== cursors.length) {
                  this.cursorLayer.textContent = "";
                  for (const c of cursors)
                      this.cursorLayer.appendChild(c.draw());
              }
              else {
                  cursors.forEach((c, idx) => c.adjust(oldCursors[idx]));
              }
              this.cursors = cursors;
          }
      }
      destroy() {
          this.selectionLayer.remove();
          this.cursorLayer.remove();
      }
  });
  const themeSpec = {
      ".cm-line": {
          "& ::selection": { backgroundColor: "transparent !important" },
          "&::selection": { backgroundColor: "transparent !important" }
      }
  };
  if (CanHidePrimary)
      themeSpec[".cm-line"].caretColor = "transparent !important";
  const hideNativeSelection = /*@__PURE__*/Prec.highest(/*@__PURE__*/EditorView.theme(themeSpec));
  function getBase(view) {
      let rect = view.scrollDOM.getBoundingClientRect();
      let left = view.textDirection == Direction.LTR ? rect.left : rect.right - view.scrollDOM.clientWidth;
      return { left: left - view.scrollDOM.scrollLeft, top: rect.top - view.scrollDOM.scrollTop };
  }
  function wrappedLine(view, pos, inside) {
      let range = EditorSelection.cursor(pos);
      return { from: Math.max(inside.from, view.moveToLineBoundary(range, false, true).from),
          to: Math.min(inside.to, view.moveToLineBoundary(range, true, true).from),
          type: BlockType.Text };
  }
  function blockAt(view, pos) {
      let line = view.lineBlockAt(pos);
      if (Array.isArray(line.type))
          for (let l of line.type) {
              if (l.to > pos || l.to == pos && (l.to == line.to || l.type == BlockType.Text))
                  return l;
          }
      return line;
  }
  function measureRange(view, range) {
      if (range.to <= view.viewport.from || range.from >= view.viewport.to)
          return [];
      let from = Math.max(range.from, view.viewport.from), to = Math.min(range.to, view.viewport.to);
      let ltr = view.textDirection == Direction.LTR;
      let content = view.contentDOM, contentRect = content.getBoundingClientRect(), base = getBase(view);
      let lineStyle = window.getComputedStyle(content.firstChild);
      let leftSide = contentRect.left + parseInt(lineStyle.paddingLeft) + Math.min(0, parseInt(lineStyle.textIndent));
      let rightSide = contentRect.right - parseInt(lineStyle.paddingRight);
      let startBlock = blockAt(view, from), endBlock = blockAt(view, to);
      let visualStart = startBlock.type == BlockType.Text ? startBlock : null;
      let visualEnd = endBlock.type == BlockType.Text ? endBlock : null;
      if (view.lineWrapping) {
          if (visualStart)
              visualStart = wrappedLine(view, from, visualStart);
          if (visualEnd)
              visualEnd = wrappedLine(view, to, visualEnd);
      }
      if (visualStart && visualEnd && visualStart.from == visualEnd.from) {
          return pieces(drawForLine(range.from, range.to, visualStart));
      }
      else {
          let top = visualStart ? drawForLine(range.from, null, visualStart) : drawForWidget(startBlock, false);
          let bottom = visualEnd ? drawForLine(null, range.to, visualEnd) : drawForWidget(endBlock, true);
          let between = [];
          if ((visualStart || startBlock).to < (visualEnd || endBlock).from - 1)
              between.push(piece(leftSide, top.bottom, rightSide, bottom.top));
          else if (top.bottom < bottom.top && view.elementAtHeight((top.bottom + bottom.top) / 2).type == BlockType.Text)
              top.bottom = bottom.top = (top.bottom + bottom.top) / 2;
          return pieces(top).concat(between).concat(pieces(bottom));
      }
      function piece(left, top, right, bottom) {
          return new Piece(left - base.left, top - base.top - 0.01 /* Epsilon */, right - left, bottom - top + 0.01 /* Epsilon */, "cm-selectionBackground");
      }
      function pieces({ top, bottom, horizontal }) {
          let pieces = [];
          for (let i = 0; i < horizontal.length; i += 2)
              pieces.push(piece(horizontal[i], top, horizontal[i + 1], bottom));
          return pieces;
      }
      // Gets passed from/to in line-local positions
      function drawForLine(from, to, line) {
          let top = 1e9, bottom = -1e9, horizontal = [];
          function addSpan(from, fromOpen, to, toOpen, dir) {
              // Passing 2/-2 is a kludge to force the view to return
              // coordinates on the proper side of block widgets, since
              // normalizing the side there, though appropriate for most
              // coordsAtPos queries, would break selection drawing.
              let fromCoords = view.coordsAtPos(from, (from == line.to ? -2 : 2));
              let toCoords = view.coordsAtPos(to, (to == line.from ? 2 : -2));
              top = Math.min(fromCoords.top, toCoords.top, top);
              bottom = Math.max(fromCoords.bottom, toCoords.bottom, bottom);
              if (dir == Direction.LTR)
                  horizontal.push(ltr && fromOpen ? leftSide : fromCoords.left, ltr && toOpen ? rightSide : toCoords.right);
              else
                  horizontal.push(!ltr && toOpen ? leftSide : toCoords.left, !ltr && fromOpen ? rightSide : fromCoords.right);
          }
          let start = from !== null && from !== void 0 ? from : line.from, end = to !== null && to !== void 0 ? to : line.to;
          // Split the range by visible range and document line
          for (let r of view.visibleRanges)
              if (r.to > start && r.from < end) {
                  for (let pos = Math.max(r.from, start), endPos = Math.min(r.to, end);;) {
                      let docLine = view.state.doc.lineAt(pos);
                      for (let span of view.bidiSpans(docLine)) {
                          let spanFrom = span.from + docLine.from, spanTo = span.to + docLine.from;
                          if (spanFrom >= endPos)
                              break;
                          if (spanTo > pos)
                              addSpan(Math.max(spanFrom, pos), from == null && spanFrom <= start, Math.min(spanTo, endPos), to == null && spanTo >= end, span.dir);
                      }
                      pos = docLine.to + 1;
                      if (pos >= endPos)
                          break;
                  }
              }
          if (horizontal.length == 0)
              addSpan(start, from == null, end, to == null, view.textDirection);
          return { top, bottom, horizontal };
      }
      function drawForWidget(block, top) {
          let y = contentRect.top + (top ? block.top : block.bottom);
          return { top: y, bottom: y, horizontal: [] };
      }
  }
  function measureCursor(view, cursor, primary) {
      let pos = view.coordsAtPos(cursor.head, cursor.assoc || 1);
      if (!pos)
          return null;
      let base = getBase(view);
      return new Piece(pos.left - base.left, pos.top - base.top, -1, pos.bottom - pos.top, primary ? "cm-cursor cm-cursor-primary" : "cm-cursor cm-cursor-secondary");
  }

  const setDropCursorPos = /*@__PURE__*/StateEffect.define({
      map(pos, mapping) { return pos == null ? null : mapping.mapPos(pos); }
  });
  const dropCursorPos = /*@__PURE__*/StateField.define({
      create() { return null; },
      update(pos, tr) {
          if (pos != null)
              pos = tr.changes.mapPos(pos);
          return tr.effects.reduce((pos, e) => e.is(setDropCursorPos) ? e.value : pos, pos);
      }
  });
  const drawDropCursor = /*@__PURE__*/ViewPlugin.fromClass(class {
      constructor(view) {
          this.view = view;
          this.cursor = null;
          this.measureReq = { read: this.readPos.bind(this), write: this.drawCursor.bind(this) };
      }
      update(update) {
          var _a;
          let cursorPos = update.state.field(dropCursorPos);
          if (cursorPos == null) {
              if (this.cursor != null) {
                  (_a = this.cursor) === null || _a === void 0 ? void 0 : _a.remove();
                  this.cursor = null;
              }
          }
          else {
              if (!this.cursor) {
                  this.cursor = this.view.scrollDOM.appendChild(document.createElement("div"));
                  this.cursor.className = "cm-dropCursor";
              }
              if (update.startState.field(dropCursorPos) != cursorPos || update.docChanged || update.geometryChanged)
                  this.view.requestMeasure(this.measureReq);
          }
      }
      readPos() {
          let pos = this.view.state.field(dropCursorPos);
          let rect = pos != null && this.view.coordsAtPos(pos);
          if (!rect)
              return null;
          let outer = this.view.scrollDOM.getBoundingClientRect();
          return {
              left: rect.left - outer.left + this.view.scrollDOM.scrollLeft,
              top: rect.top - outer.top + this.view.scrollDOM.scrollTop,
              height: rect.bottom - rect.top
          };
      }
      drawCursor(pos) {
          if (this.cursor) {
              if (pos) {
                  this.cursor.style.left = pos.left + "px";
                  this.cursor.style.top = pos.top + "px";
                  this.cursor.style.height = pos.height + "px";
              }
              else {
                  this.cursor.style.left = "-100000px";
              }
          }
      }
      destroy() {
          if (this.cursor)
              this.cursor.remove();
      }
      setDropPos(pos) {
          if (this.view.state.field(dropCursorPos) != pos)
              this.view.dispatch({ effects: setDropCursorPos.of(pos) });
      }
  }, {
      eventHandlers: {
          dragover(event) {
              this.setDropPos(this.view.posAtCoords({ x: event.clientX, y: event.clientY }));
          },
          dragleave(event) {
              if (event.target == this.view.contentDOM || !this.view.contentDOM.contains(event.relatedTarget))
                  this.setDropPos(null);
          },
          dragend() {
              this.setDropPos(null);
          },
          drop() {
              this.setDropPos(null);
          }
      }
  });
  /**
  Draws a cursor at the current drop position when something is
  dragged over the editor.
  */
  function dropCursor() {
      return [dropCursorPos, drawDropCursor];
  }

  function iterMatches(doc, re, from, to, f) {
      re.lastIndex = 0;
      for (let cursor = doc.iterRange(from, to), pos = from, m; !cursor.next().done; pos += cursor.value.length) {
          if (!cursor.lineBreak)
              while (m = re.exec(cursor.value))
                  f(pos + m.index, m);
      }
  }
  function matchRanges(view, maxLength) {
      let visible = view.visibleRanges;
      if (visible.length == 1 && visible[0].from == view.viewport.from &&
          visible[0].to == view.viewport.to)
          return visible;
      let result = [];
      for (let { from, to } of visible) {
          from = Math.max(view.state.doc.lineAt(from).from, from - maxLength);
          to = Math.min(view.state.doc.lineAt(to).to, to + maxLength);
          if (result.length && result[result.length - 1].to >= from)
              result[result.length - 1].to = to;
          else
              result.push({ from, to });
      }
      return result;
  }
  /**
  Helper class used to make it easier to maintain decorations on
  visible code that matches a given regular expression. To be used
  in a [view plugin](https://codemirror.net/6/docs/ref/#view.ViewPlugin). Instances of this object
  represent a matching configuration.
  */
  class MatchDecorator {
      /**
      Create a decorator.
      */
      constructor(config) {
          const { regexp, decoration, decorate, boundary, maxLength = 1000 } = config;
          if (!regexp.global)
              throw new RangeError("The regular expression given to MatchDecorator should have its 'g' flag set");
          this.regexp = regexp;
          if (decorate) {
              this.addMatch = (match, view, from, add) => decorate(add, from, from + match[0].length, match, view);
          }
          else if (decoration) {
              let getDeco = typeof decoration == "function" ? decoration : () => decoration;
              this.addMatch = (match, view, from, add) => add(from, from + match[0].length, getDeco(match, view, from));
          }
          else {
              throw new RangeError("Either 'decorate' or 'decoration' should be provided to MatchDecorator");
          }
          this.boundary = boundary;
          this.maxLength = maxLength;
      }
      /**
      Compute the full set of decorations for matches in the given
      view's viewport. You'll want to call this when initializing your
      plugin.
      */
      createDeco(view) {
          let build = new RangeSetBuilder(), add = build.add.bind(build);
          for (let { from, to } of matchRanges(view, this.maxLength))
              iterMatches(view.state.doc, this.regexp, from, to, (from, m) => this.addMatch(m, view, from, add));
          return build.finish();
      }
      /**
      Update a set of decorations for a view update. `deco` _must_ be
      the set of decorations produced by _this_ `MatchDecorator` for
      the view state before the update.
      */
      updateDeco(update, deco) {
          let changeFrom = 1e9, changeTo = -1;
          if (update.docChanged)
              update.changes.iterChanges((_f, _t, from, to) => {
                  if (to > update.view.viewport.from && from < update.view.viewport.to) {
                      changeFrom = Math.min(from, changeFrom);
                      changeTo = Math.max(to, changeTo);
                  }
              });
          if (update.viewportChanged || changeTo - changeFrom > 1000)
              return this.createDeco(update.view);
          if (changeTo > -1)
              return this.updateRange(update.view, deco.map(update.changes), changeFrom, changeTo);
          return deco;
      }
      updateRange(view, deco, updateFrom, updateTo) {
          for (let r of view.visibleRanges) {
              let from = Math.max(r.from, updateFrom), to = Math.min(r.to, updateTo);
              if (to > from) {
                  let fromLine = view.state.doc.lineAt(from), toLine = fromLine.to < to ? view.state.doc.lineAt(to) : fromLine;
                  let start = Math.max(r.from, fromLine.from), end = Math.min(r.to, toLine.to);
                  if (this.boundary) {
                      for (; from > fromLine.from; from--)
                          if (this.boundary.test(fromLine.text[from - 1 - fromLine.from])) {
                              start = from;
                              break;
                          }
                      for (; to < toLine.to; to++)
                          if (this.boundary.test(toLine.text[to - toLine.from])) {
                              end = to;
                              break;
                          }
                  }
                  let ranges = [], m;
                  let add = (from, to, deco) => ranges.push(deco.range(from, to));
                  if (fromLine == toLine) {
                      this.regexp.lastIndex = start - fromLine.from;
                      while ((m = this.regexp.exec(fromLine.text)) && m.index < end - fromLine.from)
                          this.addMatch(m, view, m.index + fromLine.from, add);
                  }
                  else {
                      iterMatches(view.state.doc, this.regexp, start, end, (from, m) => this.addMatch(m, view, from, add));
                  }
                  deco = deco.update({ filterFrom: start, filterTo: end, filter: (from, to) => from < start || to > end, add: ranges });
              }
          }
          return deco;
      }
  }

  const UnicodeRegexpSupport = /x/.unicode != null ? "gu" : "g";
  const Specials = /*@__PURE__*/new RegExp("[\u0000-\u0008\u000a-\u001f\u007f-\u009f\u00ad\u061c\u200b\u200e\u200f\u2028\u2029\u202d\u202e\u2066\u2067\u2069\ufeff\ufff9-\ufffc]", UnicodeRegexpSupport);
  const Names = {
      0: "null",
      7: "bell",
      8: "backspace",
      10: "newline",
      11: "vertical tab",
      13: "carriage return",
      27: "escape",
      8203: "zero width space",
      8204: "zero width non-joiner",
      8205: "zero width joiner",
      8206: "left-to-right mark",
      8207: "right-to-left mark",
      8232: "line separator",
      8237: "left-to-right override",
      8238: "right-to-left override",
      8294: "left-to-right isolate",
      8295: "right-to-left isolate",
      8297: "pop directional isolate",
      8233: "paragraph separator",
      65279: "zero width no-break space",
      65532: "object replacement"
  };
  let _supportsTabSize = null;
  function supportsTabSize() {
      var _a;
      if (_supportsTabSize == null && typeof document != "undefined" && document.body) {
          let styles = document.body.style;
          _supportsTabSize = ((_a = styles.tabSize) !== null && _a !== void 0 ? _a : styles.MozTabSize) != null;
      }
      return _supportsTabSize || false;
  }
  const specialCharConfig = /*@__PURE__*/Facet.define({
      combine(configs) {
          let config = combineConfig(configs, {
              render: null,
              specialChars: Specials,
              addSpecialChars: null
          });
          if (config.replaceTabs = !supportsTabSize())
              config.specialChars = new RegExp("\t|" + config.specialChars.source, UnicodeRegexpSupport);
          if (config.addSpecialChars)
              config.specialChars = new RegExp(config.specialChars.source + "|" + config.addSpecialChars.source, UnicodeRegexpSupport);
          return config;
      }
  });
  /**
  Returns an extension that installs highlighting of special
  characters.
  */
  function highlightSpecialChars(
  /**
  Configuration options.
  */
  config = {}) {
      return [specialCharConfig.of(config), specialCharPlugin()];
  }
  let _plugin = null;
  function specialCharPlugin() {
      return _plugin || (_plugin = ViewPlugin.fromClass(class {
          constructor(view) {
              this.view = view;
              this.decorations = Decoration.none;
              this.decorationCache = Object.create(null);
              this.decorator = this.makeDecorator(view.state.facet(specialCharConfig));
              this.decorations = this.decorator.createDeco(view);
          }
          makeDecorator(conf) {
              return new MatchDecorator({
                  regexp: conf.specialChars,
                  decoration: (m, view, pos) => {
                      let { doc } = view.state;
                      let code = codePointAt(m[0], 0);
                      if (code == 9) {
                          let line = doc.lineAt(pos);
                          let size = view.state.tabSize, col = countColumn(line.text, size, pos - line.from);
                          return Decoration.replace({ widget: new TabWidget((size - (col % size)) * this.view.defaultCharacterWidth) });
                      }
                      return this.decorationCache[code] ||
                          (this.decorationCache[code] = Decoration.replace({ widget: new SpecialCharWidget(conf, code) }));
                  },
                  boundary: conf.replaceTabs ? undefined : /[^]/
              });
          }
          update(update) {
              let conf = update.state.facet(specialCharConfig);
              if (update.startState.facet(specialCharConfig) != conf) {
                  this.decorator = this.makeDecorator(conf);
                  this.decorations = this.decorator.createDeco(update.view);
              }
              else {
                  this.decorations = this.decorator.updateDeco(update, this.decorations);
              }
          }
      }, {
          decorations: v => v.decorations
      }));
  }
  const DefaultPlaceholder = "\u2022";
  // Assigns placeholder characters from the Control Pictures block to
  // ASCII control characters
  function placeholder$1(code) {
      if (code >= 32)
          return DefaultPlaceholder;
      if (code == 10)
          return "\u2424";
      return String.fromCharCode(9216 + code);
  }
  class SpecialCharWidget extends WidgetType {
      constructor(options, code) {
          super();
          this.options = options;
          this.code = code;
      }
      eq(other) { return other.code == this.code; }
      toDOM(view) {
          let ph = placeholder$1(this.code);
          let desc = view.state.phrase("Control character") + " " + (Names[this.code] || "0x" + this.code.toString(16));
          let custom = this.options.render && this.options.render(this.code, desc, ph);
          if (custom)
              return custom;
          let span = document.createElement("span");
          span.textContent = ph;
          span.title = desc;
          span.setAttribute("aria-label", desc);
          span.className = "cm-specialChar";
          return span;
      }
      ignoreEvent() { return false; }
  }
  class TabWidget extends WidgetType {
      constructor(width) {
          super();
          this.width = width;
      }
      eq(other) { return other.width == this.width; }
      toDOM() {
          let span = document.createElement("span");
          span.textContent = "\t";
          span.className = "cm-tab";
          span.style.width = this.width + "px";
          return span;
      }
      ignoreEvent() { return false; }
  }

  /**
  Mark lines that have a cursor on them with the `"cm-activeLine"`
  DOM class.
  */
  function highlightActiveLine() {
      return activeLineHighlighter;
  }
  const lineDeco = /*@__PURE__*/Decoration.line({ class: "cm-activeLine" });
  const activeLineHighlighter = /*@__PURE__*/ViewPlugin.fromClass(class {
      constructor(view) {
          this.decorations = this.getDeco(view);
      }
      update(update) {
          if (update.docChanged || update.selectionSet)
              this.decorations = this.getDeco(update.view);
      }
      getDeco(view) {
          let lastLineStart = -1, deco = [];
          for (let r of view.state.selection.ranges) {
              if (!r.empty)
                  return Decoration.none;
              let line = view.lineBlockAt(r.head);
              if (line.from > lastLineStart) {
                  deco.push(lineDeco.range(line.from));
                  lastLineStart = line.from;
              }
          }
          return Decoration.set(deco);
      }
  }, {
      decorations: v => v.decorations
  });

  // Don't compute precise column positions for line offsets above this
  // (since it could get expensive). Assume offset==column for them.
  const MaxOff = 2000;
  function rectangleFor(state, a, b) {
      let startLine = Math.min(a.line, b.line), endLine = Math.max(a.line, b.line);
      let ranges = [];
      if (a.off > MaxOff || b.off > MaxOff || a.col < 0 || b.col < 0) {
          let startOff = Math.min(a.off, b.off), endOff = Math.max(a.off, b.off);
          for (let i = startLine; i <= endLine; i++) {
              let line = state.doc.line(i);
              if (line.length <= endOff)
                  ranges.push(EditorSelection.range(line.from + startOff, line.to + endOff));
          }
      }
      else {
          let startCol = Math.min(a.col, b.col), endCol = Math.max(a.col, b.col);
          for (let i = startLine; i <= endLine; i++) {
              let line = state.doc.line(i);
              let start = findColumn(line.text, startCol, state.tabSize, true);
              if (start > -1) {
                  let end = findColumn(line.text, endCol, state.tabSize);
                  ranges.push(EditorSelection.range(line.from + start, line.from + end));
              }
          }
      }
      return ranges;
  }
  function absoluteColumn(view, x) {
      let ref = view.coordsAtPos(view.viewport.from);
      return ref ? Math.round(Math.abs((ref.left - x) / view.defaultCharacterWidth)) : -1;
  }
  function getPos(view, event) {
      let offset = view.posAtCoords({ x: event.clientX, y: event.clientY }, false);
      let line = view.state.doc.lineAt(offset), off = offset - line.from;
      let col = off > MaxOff ? -1
          : off == line.length ? absoluteColumn(view, event.clientX)
              : countColumn(line.text, view.state.tabSize, offset - line.from);
      return { line: line.number, col, off };
  }
  function rectangleSelectionStyle(view, event) {
      let start = getPos(view, event), startSel = view.state.selection;
      if (!start)
          return null;
      return {
          update(update) {
              if (update.docChanged) {
                  let newStart = update.changes.mapPos(update.startState.doc.line(start.line).from);
                  let newLine = update.state.doc.lineAt(newStart);
                  start = { line: newLine.number, col: start.col, off: Math.min(start.off, newLine.length) };
                  startSel = startSel.map(update.changes);
              }
          },
          get(event, _extend, multiple) {
              let cur = getPos(view, event);
              if (!cur)
                  return startSel;
              let ranges = rectangleFor(view.state, start, cur);
              if (!ranges.length)
                  return startSel;
              if (multiple)
                  return EditorSelection.create(ranges.concat(startSel.ranges));
              else
                  return EditorSelection.create(ranges);
          }
      };
  }
  /**
  Create an extension that enables rectangular selections. By
  default, it will react to left mouse drag with the Alt key held
  down. When such a selection occurs, the text within the rectangle
  that was dragged over will be selected, as one selection
  [range](https://codemirror.net/6/docs/ref/#state.SelectionRange) per line.
  */
  function rectangularSelection(options) {
      let filter = (options === null || options === void 0 ? void 0 : options.eventFilter) || (e => e.altKey && e.button == 0);
      return EditorView.mouseSelectionStyle.of((view, event) => filter(event) ? rectangleSelectionStyle(view, event) : null);
  }
  const keys = {
      Alt: [18, e => e.altKey],
      Control: [17, e => e.ctrlKey],
      Shift: [16, e => e.shiftKey],
      Meta: [91, e => e.metaKey]
  };
  const showCrosshair = { style: "cursor: crosshair" };
  /**
  Returns an extension that turns the pointer cursor into a
  crosshair when a given modifier key, defaulting to Alt, is held
  down. Can serve as a visual hint that rectangular selection is
  going to happen when paired with
  [`rectangularSelection`](https://codemirror.net/6/docs/ref/#view.rectangularSelection).
  */
  function crosshairCursor(options = {}) {
      let [code, getter] = keys[options.key || "Alt"];
      let plugin = ViewPlugin.fromClass(class {
          constructor(view) {
              this.view = view;
              this.isDown = false;
          }
          set(isDown) {
              if (this.isDown != isDown) {
                  this.isDown = isDown;
                  this.view.update([]);
              }
          }
      }, {
          eventHandlers: {
              keydown(e) {
                  this.set(e.keyCode == code || getter(e));
              },
              keyup(e) {
                  if (e.keyCode == code || !getter(e))
                      this.set(false);
              }
          }
      });
      return [
          plugin,
          EditorView.contentAttributes.of(view => { var _a; return ((_a = view.plugin(plugin)) === null || _a === void 0 ? void 0 : _a.isDown) ? showCrosshair : null; })
      ];
  }

  const Outside = "-10000px";
  class TooltipViewManager {
      constructor(view, facet, createTooltipView) {
          this.facet = facet;
          this.createTooltipView = createTooltipView;
          this.input = view.state.facet(facet);
          this.tooltips = this.input.filter(t => t);
          this.tooltipViews = this.tooltips.map(createTooltipView);
      }
      update(update) {
          let input = update.state.facet(this.facet);
          let tooltips = input.filter(x => x);
          if (input === this.input) {
              for (let t of this.tooltipViews)
                  if (t.update)
                      t.update(update);
              return false;
          }
          let tooltipViews = [];
          for (let i = 0; i < tooltips.length; i++) {
              let tip = tooltips[i], known = -1;
              if (!tip)
                  continue;
              for (let i = 0; i < this.tooltips.length; i++) {
                  let other = this.tooltips[i];
                  if (other && other.create == tip.create)
                      known = i;
              }
              if (known < 0) {
                  tooltipViews[i] = this.createTooltipView(tip);
              }
              else {
                  let tooltipView = tooltipViews[i] = this.tooltipViews[known];
                  if (tooltipView.update)
                      tooltipView.update(update);
              }
          }
          for (let t of this.tooltipViews)
              if (tooltipViews.indexOf(t) < 0)
                  t.dom.remove();
          this.input = input;
          this.tooltips = tooltips;
          this.tooltipViews = tooltipViews;
          return true;
      }
  }
  function windowSpace() {
      return { top: 0, left: 0, bottom: innerHeight, right: innerWidth };
  }
  const tooltipConfig = /*@__PURE__*/Facet.define({
      combine: values => {
          var _a, _b, _c;
          return ({
              position: browser.ios ? "absolute" : ((_a = values.find(conf => conf.position)) === null || _a === void 0 ? void 0 : _a.position) || "fixed",
              parent: ((_b = values.find(conf => conf.parent)) === null || _b === void 0 ? void 0 : _b.parent) || null,
              tooltipSpace: ((_c = values.find(conf => conf.tooltipSpace)) === null || _c === void 0 ? void 0 : _c.tooltipSpace) || windowSpace,
          });
      }
  });
  const tooltipPlugin = /*@__PURE__*/ViewPlugin.fromClass(class {
      constructor(view) {
          var _a;
          this.view = view;
          this.inView = true;
          this.lastTransaction = 0;
          this.measureTimeout = -1;
          let config = view.state.facet(tooltipConfig);
          this.position = config.position;
          this.parent = config.parent;
          this.classes = view.themeClasses;
          this.createContainer();
          this.measureReq = { read: this.readMeasure.bind(this), write: this.writeMeasure.bind(this), key: this };
          this.manager = new TooltipViewManager(view, showTooltip, t => this.createTooltip(t));
          this.intersectionObserver = typeof IntersectionObserver == "function" ? new IntersectionObserver(entries => {
              if (Date.now() > this.lastTransaction - 50 &&
                  entries.length > 0 && entries[entries.length - 1].intersectionRatio < 1)
                  this.measureSoon();
          }, { threshold: [1] }) : null;
          this.observeIntersection();
          (_a = view.dom.ownerDocument.defaultView) === null || _a === void 0 ? void 0 : _a.addEventListener("resize", this.measureSoon = this.measureSoon.bind(this));
          this.maybeMeasure();
      }
      createContainer() {
          if (this.parent) {
              this.container = document.createElement("div");
              this.container.style.position = "relative";
              this.container.className = this.view.themeClasses;
              this.parent.appendChild(this.container);
          }
          else {
              this.container = this.view.dom;
          }
      }
      observeIntersection() {
          if (this.intersectionObserver) {
              this.intersectionObserver.disconnect();
              for (let tooltip of this.manager.tooltipViews)
                  this.intersectionObserver.observe(tooltip.dom);
          }
      }
      measureSoon() {
          if (this.measureTimeout < 0)
              this.measureTimeout = setTimeout(() => {
                  this.measureTimeout = -1;
                  this.maybeMeasure();
              }, 50);
      }
      update(update) {
          if (update.transactions.length)
              this.lastTransaction = Date.now();
          let updated = this.manager.update(update);
          if (updated)
              this.observeIntersection();
          let shouldMeasure = updated || update.geometryChanged;
          let newConfig = update.state.facet(tooltipConfig);
          if (newConfig.position != this.position) {
              this.position = newConfig.position;
              for (let t of this.manager.tooltipViews)
                  t.dom.style.position = this.position;
              shouldMeasure = true;
          }
          if (newConfig.parent != this.parent) {
              if (this.parent)
                  this.container.remove();
              this.parent = newConfig.parent;
              this.createContainer();
              for (let t of this.manager.tooltipViews)
                  this.container.appendChild(t.dom);
              shouldMeasure = true;
          }
          else if (this.parent && this.view.themeClasses != this.classes) {
              this.classes = this.container.className = this.view.themeClasses;
          }
          if (shouldMeasure)
              this.maybeMeasure();
      }
      createTooltip(tooltip) {
          let tooltipView = tooltip.create(this.view);
          tooltipView.dom.classList.add("cm-tooltip");
          if (tooltip.arrow && !tooltipView.dom.querySelector(".cm-tooltip > .cm-tooltip-arrow")) {
              let arrow = document.createElement("div");
              arrow.className = "cm-tooltip-arrow";
              tooltipView.dom.appendChild(arrow);
          }
          tooltipView.dom.style.position = this.position;
          tooltipView.dom.style.top = Outside;
          this.container.appendChild(tooltipView.dom);
          if (tooltipView.mount)
              tooltipView.mount(this.view);
          return tooltipView;
      }
      destroy() {
          var _a, _b;
          (_a = this.view.dom.ownerDocument.defaultView) === null || _a === void 0 ? void 0 : _a.removeEventListener("resize", this.measureSoon);
          for (let { dom } of this.manager.tooltipViews)
              dom.remove();
          (_b = this.intersectionObserver) === null || _b === void 0 ? void 0 : _b.disconnect();
          clearTimeout(this.measureTimeout);
      }
      readMeasure() {
          let editor = this.view.dom.getBoundingClientRect();
          return {
              editor,
              parent: this.parent ? this.container.getBoundingClientRect() : editor,
              pos: this.manager.tooltips.map((t, i) => {
                  let tv = this.manager.tooltipViews[i];
                  return tv.getCoords ? tv.getCoords(t.pos) : this.view.coordsAtPos(t.pos);
              }),
              size: this.manager.tooltipViews.map(({ dom }) => dom.getBoundingClientRect()),
              space: this.view.state.facet(tooltipConfig).tooltipSpace(this.view),
          };
      }
      writeMeasure(measured) {
          let { editor, space } = measured;
          let others = [];
          for (let i = 0; i < this.manager.tooltips.length; i++) {
              let tooltip = this.manager.tooltips[i], tView = this.manager.tooltipViews[i], { dom } = tView;
              let pos = measured.pos[i], size = measured.size[i];
              // Hide tooltips that are outside of the editor.
              if (!pos || pos.bottom <= Math.max(editor.top, space.top) ||
                  pos.top >= Math.min(editor.bottom, space.bottom) ||
                  pos.right < Math.max(editor.left, space.left) - .1 ||
                  pos.left > Math.min(editor.right, space.right) + .1) {
                  dom.style.top = Outside;
                  continue;
              }
              let arrow = tooltip.arrow ? tView.dom.querySelector(".cm-tooltip-arrow") : null;
              let arrowHeight = arrow ? 7 /* Size */ : 0;
              let width = size.right - size.left, height = size.bottom - size.top;
              let offset = tView.offset || noOffset, ltr = this.view.textDirection == Direction.LTR;
              let left = size.width > space.right - space.left ? (ltr ? space.left : space.right - size.width)
                  : ltr ? Math.min(pos.left - (arrow ? 14 /* Offset */ : 0) + offset.x, space.right - width)
                      : Math.max(space.left, pos.left - width + (arrow ? 14 /* Offset */ : 0) - offset.x);
              let above = !!tooltip.above;
              if (!tooltip.strictSide && (above
                  ? pos.top - (size.bottom - size.top) - offset.y < space.top
                  : pos.bottom + (size.bottom - size.top) + offset.y > space.bottom) &&
                  above == (space.bottom - pos.bottom > pos.top - space.top))
                  above = !above;
              let top = above ? pos.top - height - arrowHeight - offset.y : pos.bottom + arrowHeight + offset.y;
              let right = left + width;
              if (tView.overlap !== true)
                  for (let r of others)
                      if (r.left < right && r.right > left && r.top < top + height && r.bottom > top)
                          top = above ? r.top - height - 2 - arrowHeight : r.bottom + arrowHeight + 2;
              if (this.position == "absolute") {
                  dom.style.top = (top - measured.parent.top) + "px";
                  dom.style.left = (left - measured.parent.left) + "px";
              }
              else {
                  dom.style.top = top + "px";
                  dom.style.left = left + "px";
              }
              if (arrow)
                  arrow.style.left = `${pos.left + (ltr ? offset.x : -offset.x) - (left + 14 /* Offset */ - 7 /* Size */)}px`;
              if (tView.overlap !== true)
                  others.push({ left, top, right, bottom: top + height });
              dom.classList.toggle("cm-tooltip-above", above);
              dom.classList.toggle("cm-tooltip-below", !above);
              if (tView.positioned)
                  tView.positioned();
          }
      }
      maybeMeasure() {
          if (this.manager.tooltips.length) {
              if (this.view.inView)
                  this.view.requestMeasure(this.measureReq);
              if (this.inView != this.view.inView) {
                  this.inView = this.view.inView;
                  if (!this.inView)
                      for (let tv of this.manager.tooltipViews)
                          tv.dom.style.top = Outside;
              }
          }
      }
  }, {
      eventHandlers: {
          scroll() { this.maybeMeasure(); }
      }
  });
  const baseTheme$4 = /*@__PURE__*/EditorView.baseTheme({
      ".cm-tooltip": {
          zIndex: 100
      },
      "&light .cm-tooltip": {
          border: "1px solid #bbb",
          backgroundColor: "#f5f5f5"
      },
      "&light .cm-tooltip-section:not(:first-child)": {
          borderTop: "1px solid #bbb",
      },
      "&dark .cm-tooltip": {
          backgroundColor: "#333338",
          color: "white"
      },
      ".cm-tooltip-arrow": {
          height: `${7 /* Size */}px`,
          width: `${7 /* Size */ * 2}px`,
          position: "absolute",
          zIndex: -1,
          overflow: "hidden",
          "&:before, &:after": {
              content: "''",
              position: "absolute",
              width: 0,
              height: 0,
              borderLeft: `${7 /* Size */}px solid transparent`,
              borderRight: `${7 /* Size */}px solid transparent`,
          },
          ".cm-tooltip-above &": {
              bottom: `-${7 /* Size */}px`,
              "&:before": {
                  borderTop: `${7 /* Size */}px solid #bbb`,
              },
              "&:after": {
                  borderTop: `${7 /* Size */}px solid #f5f5f5`,
                  bottom: "1px"
              }
          },
          ".cm-tooltip-below &": {
              top: `-${7 /* Size */}px`,
              "&:before": {
                  borderBottom: `${7 /* Size */}px solid #bbb`,
              },
              "&:after": {
                  borderBottom: `${7 /* Size */}px solid #f5f5f5`,
                  top: "1px"
              }
          },
      },
      "&dark .cm-tooltip .cm-tooltip-arrow": {
          "&:before": {
              borderTopColor: "#333338",
              borderBottomColor: "#333338"
          },
          "&:after": {
              borderTopColor: "transparent",
              borderBottomColor: "transparent"
          }
      }
  });
  const noOffset = { x: 0, y: 0 };
  /**
  Facet to which an extension can add a value to show a tooltip.
  */
  const showTooltip = /*@__PURE__*/Facet.define({
      enables: [tooltipPlugin, baseTheme$4]
  });
  const showHoverTooltip = /*@__PURE__*/Facet.define();
  class HoverTooltipHost {
      constructor(view) {
          this.view = view;
          this.mounted = false;
          this.dom = document.createElement("div");
          this.dom.classList.add("cm-tooltip-hover");
          this.manager = new TooltipViewManager(view, showHoverTooltip, t => this.createHostedView(t));
      }
      // Needs to be static so that host tooltip instances always match
      static create(view) {
          return new HoverTooltipHost(view);
      }
      createHostedView(tooltip) {
          let hostedView = tooltip.create(this.view);
          hostedView.dom.classList.add("cm-tooltip-section");
          this.dom.appendChild(hostedView.dom);
          if (this.mounted && hostedView.mount)
              hostedView.mount(this.view);
          return hostedView;
      }
      mount(view) {
          for (let hostedView of this.manager.tooltipViews) {
              if (hostedView.mount)
                  hostedView.mount(view);
          }
          this.mounted = true;
      }
      positioned() {
          for (let hostedView of this.manager.tooltipViews) {
              if (hostedView.positioned)
                  hostedView.positioned();
          }
      }
      update(update) {
          this.manager.update(update);
      }
  }
  const showHoverTooltipHost = /*@__PURE__*/showTooltip.compute([showHoverTooltip], state => {
      let tooltips = state.facet(showHoverTooltip).filter(t => t);
      if (tooltips.length === 0)
          return null;
      return {
          pos: Math.min(...tooltips.map(t => t.pos)),
          end: Math.max(...tooltips.filter(t => t.end != null).map(t => t.end)),
          create: HoverTooltipHost.create,
          above: tooltips[0].above,
          arrow: tooltips.some(t => t.arrow),
      };
  });
  class HoverPlugin {
      constructor(view, source, field, setHover, hoverTime) {
          this.view = view;
          this.source = source;
          this.field = field;
          this.setHover = setHover;
          this.hoverTime = hoverTime;
          this.hoverTimeout = -1;
          this.restartTimeout = -1;
          this.pending = null;
          this.lastMove = { x: 0, y: 0, target: view.dom, time: 0 };
          this.checkHover = this.checkHover.bind(this);
          view.dom.addEventListener("mouseleave", this.mouseleave = this.mouseleave.bind(this));
          view.dom.addEventListener("mousemove", this.mousemove = this.mousemove.bind(this));
      }
      update() {
          if (this.pending) {
              this.pending = null;
              clearTimeout(this.restartTimeout);
              this.restartTimeout = setTimeout(() => this.startHover(), 20);
          }
      }
      get active() {
          return this.view.state.field(this.field);
      }
      checkHover() {
          this.hoverTimeout = -1;
          if (this.active)
              return;
          let hovered = Date.now() - this.lastMove.time;
          if (hovered < this.hoverTime)
              this.hoverTimeout = setTimeout(this.checkHover, this.hoverTime - hovered);
          else
              this.startHover();
      }
      startHover() {
          clearTimeout(this.restartTimeout);
          let { lastMove } = this;
          let pos = this.view.contentDOM.contains(lastMove.target) ? this.view.posAtCoords(lastMove) : null;
          if (pos == null)
              return;
          let posCoords = this.view.coordsAtPos(pos);
          if (posCoords == null || lastMove.y < posCoords.top || lastMove.y > posCoords.bottom ||
              lastMove.x < posCoords.left - this.view.defaultCharacterWidth ||
              lastMove.x > posCoords.right + this.view.defaultCharacterWidth)
              return;
          let bidi = this.view.bidiSpans(this.view.state.doc.lineAt(pos)).find(s => s.from <= pos && s.to >= pos);
          let rtl = bidi && bidi.dir == Direction.RTL ? -1 : 1;
          let open = this.source(this.view, pos, (lastMove.x < posCoords.left ? -rtl : rtl));
          if (open === null || open === void 0 ? void 0 : open.then) {
              let pending = this.pending = { pos };
              open.then(result => {
                  if (this.pending == pending) {
                      this.pending = null;
                      if (result)
                          this.view.dispatch({ effects: this.setHover.of(result) });
                  }
              }, e => logException(this.view.state, e, "hover tooltip"));
          }
          else if (open) {
              this.view.dispatch({ effects: this.setHover.of(open) });
          }
      }
      mousemove(event) {
          var _a;
          this.lastMove = { x: event.clientX, y: event.clientY, target: event.target, time: Date.now() };
          if (this.hoverTimeout < 0)
              this.hoverTimeout = setTimeout(this.checkHover, this.hoverTime);
          let tooltip = this.active;
          if (tooltip && !isInTooltip(this.lastMove.target) || this.pending) {
              let { pos } = tooltip || this.pending, end = (_a = tooltip === null || tooltip === void 0 ? void 0 : tooltip.end) !== null && _a !== void 0 ? _a : pos;
              if ((pos == end ? this.view.posAtCoords(this.lastMove) != pos
                  : !isOverRange(this.view, pos, end, event.clientX, event.clientY, 6 /* MaxDist */))) {
                  this.view.dispatch({ effects: this.setHover.of(null) });
                  this.pending = null;
              }
          }
      }
      mouseleave() {
          clearTimeout(this.hoverTimeout);
          this.hoverTimeout = -1;
          if (this.active)
              this.view.dispatch({ effects: this.setHover.of(null) });
      }
      destroy() {
          clearTimeout(this.hoverTimeout);
          this.view.dom.removeEventListener("mouseleave", this.mouseleave);
          this.view.dom.removeEventListener("mousemove", this.mousemove);
      }
  }
  function isInTooltip(elt) {
      for (let cur = elt; cur; cur = cur.parentNode)
          if (cur.nodeType == 1 && cur.classList.contains("cm-tooltip"))
              return true;
      return false;
  }
  function isOverRange(view, from, to, x, y, margin) {
      let range = document.createRange();
      let fromDOM = view.domAtPos(from), toDOM = view.domAtPos(to);
      range.setEnd(toDOM.node, toDOM.offset);
      range.setStart(fromDOM.node, fromDOM.offset);
      let rects = range.getClientRects();
      range.detach();
      for (let i = 0; i < rects.length; i++) {
          let rect = rects[i];
          let dist = Math.max(rect.top - y, y - rect.bottom, rect.left - x, x - rect.right);
          if (dist <= margin)
              return true;
      }
      return false;
  }
  /**
  Set up a hover tooltip, which shows up when the pointer hovers
  over ranges of text. The callback is called when the mouse hovers
  over the document text. It should, if there is a tooltip
  associated with position `pos`, return the tooltip description
  (either directly or in a promise). The `side` argument indicates
  on which side of the position the pointer is—it will be -1 if the
  pointer is before the position, 1 if after the position.

  Note that all hover tooltips are hosted within a single tooltip
  container element. This allows multiple tooltips over the same
  range to be "merged" together without overlapping.
  */
  function hoverTooltip(source, options = {}) {
      let setHover = StateEffect.define();
      let hoverState = StateField.define({
          create() { return null; },
          update(value, tr) {
              if (value && (options.hideOnChange && (tr.docChanged || tr.selection) ||
                  options.hideOn && options.hideOn(tr, value)))
                  return null;
              if (value && tr.docChanged) {
                  let newPos = tr.changes.mapPos(value.pos, -1, MapMode.TrackDel);
                  if (newPos == null)
                      return null;
                  let copy = Object.assign(Object.create(null), value);
                  copy.pos = newPos;
                  if (value.end != null)
                      copy.end = tr.changes.mapPos(value.end);
                  value = copy;
              }
              for (let effect of tr.effects) {
                  if (effect.is(setHover))
                      value = effect.value;
                  if (effect.is(closeHoverTooltipEffect))
                      value = null;
              }
              return value;
          },
          provide: f => showHoverTooltip.from(f)
      });
      return [
          hoverState,
          ViewPlugin.define(view => new HoverPlugin(view, source, hoverState, setHover, options.hoverTime || 300 /* Time */)),
          showHoverTooltipHost
      ];
  }
  /**
  Get the active tooltip view for a given tooltip, if available.
  */
  function getTooltip(view, tooltip) {
      let plugin = view.plugin(tooltipPlugin);
      if (!plugin)
          return null;
      let found = plugin.manager.tooltips.indexOf(tooltip);
      return found < 0 ? null : plugin.manager.tooltipViews[found];
  }
  const closeHoverTooltipEffect = /*@__PURE__*/StateEffect.define();

  const panelConfig = /*@__PURE__*/Facet.define({
      combine(configs) {
          let topContainer, bottomContainer;
          for (let c of configs) {
              topContainer = topContainer || c.topContainer;
              bottomContainer = bottomContainer || c.bottomContainer;
          }
          return { topContainer, bottomContainer };
      }
  });
  /**
  Get the active panel created by the given constructor, if any.
  This can be useful when you need access to your panels' DOM
  structure.
  */
  function getPanel(view, panel) {
      let plugin = view.plugin(panelPlugin);
      let index = plugin ? plugin.specs.indexOf(panel) : -1;
      return index > -1 ? plugin.panels[index] : null;
  }
  const panelPlugin = /*@__PURE__*/ViewPlugin.fromClass(class {
      constructor(view) {
          this.input = view.state.facet(showPanel);
          this.specs = this.input.filter(s => s);
          this.panels = this.specs.map(spec => spec(view));
          let conf = view.state.facet(panelConfig);
          this.top = new PanelGroup(view, true, conf.topContainer);
          this.bottom = new PanelGroup(view, false, conf.bottomContainer);
          this.top.sync(this.panels.filter(p => p.top));
          this.bottom.sync(this.panels.filter(p => !p.top));
          for (let p of this.panels) {
              p.dom.classList.add("cm-panel");
              if (p.mount)
                  p.mount();
          }
      }
      update(update) {
          let conf = update.state.facet(panelConfig);
          if (this.top.container != conf.topContainer) {
              this.top.sync([]);
              this.top = new PanelGroup(update.view, true, conf.topContainer);
          }
          if (this.bottom.container != conf.bottomContainer) {
              this.bottom.sync([]);
              this.bottom = new PanelGroup(update.view, false, conf.bottomContainer);
          }
          this.top.syncClasses();
          this.bottom.syncClasses();
          let input = update.state.facet(showPanel);
          if (input != this.input) {
              let specs = input.filter(x => x);
              let panels = [], top = [], bottom = [], mount = [];
              for (let spec of specs) {
                  let known = this.specs.indexOf(spec), panel;
                  if (known < 0) {
                      panel = spec(update.view);
                      mount.push(panel);
                  }
                  else {
                      panel = this.panels[known];
                      if (panel.update)
                          panel.update(update);
                  }
                  panels.push(panel);
                  (panel.top ? top : bottom).push(panel);
              }
              this.specs = specs;
              this.panels = panels;
              this.top.sync(top);
              this.bottom.sync(bottom);
              for (let p of mount) {
                  p.dom.classList.add("cm-panel");
                  if (p.mount)
                      p.mount();
              }
          }
          else {
              for (let p of this.panels)
                  if (p.update)
                      p.update(update);
          }
      }
      destroy() {
          this.top.sync([]);
          this.bottom.sync([]);
      }
  }, {
      provide: plugin => EditorView.scrollMargins.of(view => {
          let value = view.plugin(plugin);
          return value && { top: value.top.scrollMargin(), bottom: value.bottom.scrollMargin() };
      })
  });
  class PanelGroup {
      constructor(view, top, container) {
          this.view = view;
          this.top = top;
          this.container = container;
          this.dom = undefined;
          this.classes = "";
          this.panels = [];
          this.syncClasses();
      }
      sync(panels) {
          for (let p of this.panels)
              if (p.destroy && panels.indexOf(p) < 0)
                  p.destroy();
          this.panels = panels;
          this.syncDOM();
      }
      syncDOM() {
          if (this.panels.length == 0) {
              if (this.dom) {
                  this.dom.remove();
                  this.dom = undefined;
              }
              return;
          }
          if (!this.dom) {
              this.dom = document.createElement("div");
              this.dom.className = this.top ? "cm-panels cm-panels-top" : "cm-panels cm-panels-bottom";
              this.dom.style[this.top ? "top" : "bottom"] = "0";
              let parent = this.container || this.view.dom;
              parent.insertBefore(this.dom, this.top ? parent.firstChild : null);
          }
          let curDOM = this.dom.firstChild;
          for (let panel of this.panels) {
              if (panel.dom.parentNode == this.dom) {
                  while (curDOM != panel.dom)
                      curDOM = rm(curDOM);
                  curDOM = curDOM.nextSibling;
              }
              else {
                  this.dom.insertBefore(panel.dom, curDOM);
              }
          }
          while (curDOM)
              curDOM = rm(curDOM);
      }
      scrollMargin() {
          return !this.dom || this.container ? 0
              : Math.max(0, this.top ?
                  this.dom.getBoundingClientRect().bottom - Math.max(0, this.view.scrollDOM.getBoundingClientRect().top) :
                  Math.min(innerHeight, this.view.scrollDOM.getBoundingClientRect().bottom) - this.dom.getBoundingClientRect().top);
      }
      syncClasses() {
          if (!this.container || this.classes == this.view.themeClasses)
              return;
          for (let cls of this.classes.split(" "))
              if (cls)
                  this.container.classList.remove(cls);
          for (let cls of (this.classes = this.view.themeClasses).split(" "))
              if (cls)
                  this.container.classList.add(cls);
      }
  }
  function rm(node) {
      let next = node.nextSibling;
      node.remove();
      return next;
  }
  /**
  Opening a panel is done by providing a constructor function for
  the panel through this facet. (The panel is closed again when its
  constructor is no longer provided.) Values of `null` are ignored.
  */
  const showPanel = /*@__PURE__*/Facet.define({
      enables: panelPlugin
  });

  /**
  A gutter marker represents a bit of information attached to a line
  in a specific gutter. Your own custom markers have to extend this
  class.
  */
  class GutterMarker extends RangeValue {
      /**
      @internal
      */
      compare(other) {
          return this == other || this.constructor == other.constructor && this.eq(other);
      }
      /**
      Compare this marker to another marker of the same type.
      */
      eq(other) { return false; }
      /**
      Called if the marker has a `toDOM` method and its representation
      was removed from a gutter.
      */
      destroy(dom) { }
  }
  GutterMarker.prototype.elementClass = "";
  GutterMarker.prototype.toDOM = undefined;
  GutterMarker.prototype.mapMode = MapMode.TrackBefore;
  GutterMarker.prototype.startSide = GutterMarker.prototype.endSide = -1;
  GutterMarker.prototype.point = true;
  /**
  Facet used to add a class to all gutter elements for a given line.
  Markers given to this facet should _only_ define an
  [`elementclass`](https://codemirror.net/6/docs/ref/#view.GutterMarker.elementClass), not a
  [`toDOM`](https://codemirror.net/6/docs/ref/#view.GutterMarker.toDOM) (or the marker will appear
  in all gutters for the line).
  */
  const gutterLineClass = /*@__PURE__*/Facet.define();
  const defaults$1 = {
      class: "",
      renderEmptyElements: false,
      elementStyle: "",
      markers: () => RangeSet.empty,
      lineMarker: () => null,
      lineMarkerChange: null,
      initialSpacer: null,
      updateSpacer: null,
      domEventHandlers: {}
  };
  const activeGutters = /*@__PURE__*/Facet.define();
  /**
  Define an editor gutter. The order in which the gutters appear is
  determined by their extension priority.
  */
  function gutter(config) {
      return [gutters(), activeGutters.of(Object.assign(Object.assign({}, defaults$1), config))];
  }
  const unfixGutters = /*@__PURE__*/Facet.define({
      combine: values => values.some(x => x)
  });
  /**
  The gutter-drawing plugin is automatically enabled when you add a
  gutter, but you can use this function to explicitly configure it.

  Unless `fixed` is explicitly set to `false`, the gutters are
  fixed, meaning they don't scroll along with the content
  horizontally (except on Internet Explorer, which doesn't support
  CSS [`position:
  sticky`](https://developer.mozilla.org/en-US/docs/Web/CSS/position#sticky)).
  */
  function gutters(config) {
      let result = [
          gutterView,
      ];
      if (config && config.fixed === false)
          result.push(unfixGutters.of(true));
      return result;
  }
  const gutterView = /*@__PURE__*/ViewPlugin.fromClass(class {
      constructor(view) {
          this.view = view;
          this.prevViewport = view.viewport;
          this.dom = document.createElement("div");
          this.dom.className = "cm-gutters";
          this.dom.setAttribute("aria-hidden", "true");
          this.dom.style.minHeight = this.view.contentHeight + "px";
          this.gutters = view.state.facet(activeGutters).map(conf => new SingleGutterView(view, conf));
          for (let gutter of this.gutters)
              this.dom.appendChild(gutter.dom);
          this.fixed = !view.state.facet(unfixGutters);
          if (this.fixed) {
              // FIXME IE11 fallback, which doesn't support position: sticky,
              // by using position: relative + event handlers that realign the
              // gutter (or just force fixed=false on IE11?)
              this.dom.style.position = "sticky";
          }
          this.syncGutters(false);
          view.scrollDOM.insertBefore(this.dom, view.contentDOM);
      }
      update(update) {
          if (this.updateGutters(update)) {
              // Detach during sync when the viewport changed significantly
              // (such as during scrolling), since for large updates that is
              // faster.
              let vpA = this.prevViewport, vpB = update.view.viewport;
              let vpOverlap = Math.min(vpA.to, vpB.to) - Math.max(vpA.from, vpB.from);
              this.syncGutters(vpOverlap < (vpB.to - vpB.from) * 0.8);
          }
          if (update.geometryChanged)
              this.dom.style.minHeight = this.view.contentHeight + "px";
          if (this.view.state.facet(unfixGutters) != !this.fixed) {
              this.fixed = !this.fixed;
              this.dom.style.position = this.fixed ? "sticky" : "";
          }
          this.prevViewport = update.view.viewport;
      }
      syncGutters(detach) {
          let after = this.dom.nextSibling;
          if (detach)
              this.dom.remove();
          let lineClasses = RangeSet.iter(this.view.state.facet(gutterLineClass), this.view.viewport.from);
          let classSet = [];
          let contexts = this.gutters.map(gutter => new UpdateContext(gutter, this.view.viewport, -this.view.documentPadding.top));
          for (let line of this.view.viewportLineBlocks) {
              let text;
              if (Array.isArray(line.type)) {
                  for (let b of line.type)
                      if (b.type == BlockType.Text) {
                          text = b;
                          break;
                      }
              }
              else {
                  text = line.type == BlockType.Text ? line : undefined;
              }
              if (!text)
                  continue;
              if (classSet.length)
                  classSet = [];
              advanceCursor(lineClasses, classSet, line.from);
              for (let cx of contexts)
                  cx.line(this.view, text, classSet);
          }
          for (let cx of contexts)
              cx.finish();
          if (detach)
              this.view.scrollDOM.insertBefore(this.dom, after);
      }
      updateGutters(update) {
          let prev = update.startState.facet(activeGutters), cur = update.state.facet(activeGutters);
          let change = update.docChanged || update.heightChanged || update.viewportChanged ||
              !RangeSet.eq(update.startState.facet(gutterLineClass), update.state.facet(gutterLineClass), update.view.viewport.from, update.view.viewport.to);
          if (prev == cur) {
              for (let gutter of this.gutters)
                  if (gutter.update(update))
                      change = true;
          }
          else {
              change = true;
              let gutters = [];
              for (let conf of cur) {
                  let known = prev.indexOf(conf);
                  if (known < 0) {
                      gutters.push(new SingleGutterView(this.view, conf));
                  }
                  else {
                      this.gutters[known].update(update);
                      gutters.push(this.gutters[known]);
                  }
              }
              for (let g of this.gutters) {
                  g.dom.remove();
                  if (gutters.indexOf(g) < 0)
                      g.destroy();
              }
              for (let g of gutters)
                  this.dom.appendChild(g.dom);
              this.gutters = gutters;
          }
          return change;
      }
      destroy() {
          for (let view of this.gutters)
              view.destroy();
          this.dom.remove();
      }
  }, {
      provide: plugin => EditorView.scrollMargins.of(view => {
          let value = view.plugin(plugin);
          if (!value || value.gutters.length == 0 || !value.fixed)
              return null;
          return view.textDirection == Direction.LTR ? { left: value.dom.offsetWidth } : { right: value.dom.offsetWidth };
      })
  });
  function asArray(val) { return (Array.isArray(val) ? val : [val]); }
  function advanceCursor(cursor, collect, pos) {
      while (cursor.value && cursor.from <= pos) {
          if (cursor.from == pos)
              collect.push(cursor.value);
          cursor.next();
      }
  }
  class UpdateContext {
      constructor(gutter, viewport, height) {
          this.gutter = gutter;
          this.height = height;
          this.localMarkers = [];
          this.i = 0;
          this.cursor = RangeSet.iter(gutter.markers, viewport.from);
      }
      line(view, line, extraMarkers) {
          if (this.localMarkers.length)
              this.localMarkers = [];
          advanceCursor(this.cursor, this.localMarkers, line.from);
          let localMarkers = extraMarkers.length ? this.localMarkers.concat(extraMarkers) : this.localMarkers;
          let forLine = this.gutter.config.lineMarker(view, line, localMarkers);
          if (forLine)
              localMarkers.unshift(forLine);
          let gutter = this.gutter;
          if (localMarkers.length == 0 && !gutter.config.renderEmptyElements)
              return;
          let above = line.top - this.height;
          if (this.i == gutter.elements.length) {
              let newElt = new GutterElement(view, line.height, above, localMarkers);
              gutter.elements.push(newElt);
              gutter.dom.appendChild(newElt.dom);
          }
          else {
              gutter.elements[this.i].update(view, line.height, above, localMarkers);
          }
          this.height = line.bottom;
          this.i++;
      }
      finish() {
          let gutter = this.gutter;
          while (gutter.elements.length > this.i) {
              let last = gutter.elements.pop();
              gutter.dom.removeChild(last.dom);
              last.destroy();
          }
      }
  }
  class SingleGutterView {
      constructor(view, config) {
          this.view = view;
          this.config = config;
          this.elements = [];
          this.spacer = null;
          this.dom = document.createElement("div");
          this.dom.className = "cm-gutter" + (this.config.class ? " " + this.config.class : "");
          for (let prop in config.domEventHandlers) {
              this.dom.addEventListener(prop, (event) => {
                  let line = view.lineBlockAtHeight(event.clientY - view.documentTop);
                  if (config.domEventHandlers[prop](view, line, event))
                      event.preventDefault();
              });
          }
          this.markers = asArray(config.markers(view));
          if (config.initialSpacer) {
              this.spacer = new GutterElement(view, 0, 0, [config.initialSpacer(view)]);
              this.dom.appendChild(this.spacer.dom);
              this.spacer.dom.style.cssText += "visibility: hidden; pointer-events: none";
          }
      }
      update(update) {
          let prevMarkers = this.markers;
          this.markers = asArray(this.config.markers(update.view));
          if (this.spacer && this.config.updateSpacer) {
              let updated = this.config.updateSpacer(this.spacer.markers[0], update);
              if (updated != this.spacer.markers[0])
                  this.spacer.update(update.view, 0, 0, [updated]);
          }
          let vp = update.view.viewport;
          return !RangeSet.eq(this.markers, prevMarkers, vp.from, vp.to) ||
              (this.config.lineMarkerChange ? this.config.lineMarkerChange(update) : false);
      }
      destroy() {
          for (let elt of this.elements)
              elt.destroy();
      }
  }
  class GutterElement {
      constructor(view, height, above, markers) {
          this.height = -1;
          this.above = 0;
          this.markers = [];
          this.dom = document.createElement("div");
          this.dom.className = "cm-gutterElement";
          this.update(view, height, above, markers);
      }
      update(view, height, above, markers) {
          if (this.height != height)
              this.dom.style.height = (this.height = height) + "px";
          if (this.above != above)
              this.dom.style.marginTop = (this.above = above) ? above + "px" : "";
          if (!sameMarkers(this.markers, markers))
              this.setMarkers(view, markers);
      }
      setMarkers(view, markers) {
          let cls = "cm-gutterElement", domPos = this.dom.firstChild;
          for (let iNew = 0, iOld = 0;;) {
              let skipTo = iOld, marker = iNew < markers.length ? markers[iNew++] : null, matched = false;
              if (marker) {
                  let c = marker.elementClass;
                  if (c)
                      cls += " " + c;
                  for (let i = iOld; i < this.markers.length; i++)
                      if (this.markers[i].compare(marker)) {
                          skipTo = i;
                          matched = true;
                          break;
                      }
              }
              else {
                  skipTo = this.markers.length;
              }
              while (iOld < skipTo) {
                  let next = this.markers[iOld++];
                  if (next.toDOM) {
                      next.destroy(domPos);
                      let after = domPos.nextSibling;
                      domPos.remove();
                      domPos = after;
                  }
              }
              if (!marker)
                  break;
              if (marker.toDOM) {
                  if (matched)
                      domPos = domPos.nextSibling;
                  else
                      this.dom.insertBefore(marker.toDOM(view), domPos);
              }
              if (matched)
                  iOld++;
          }
          this.dom.className = cls;
          this.markers = markers;
      }
      destroy() {
          this.setMarkers(null, []); // First argument not used unless creating markers
      }
  }
  function sameMarkers(a, b) {
      if (a.length != b.length)
          return false;
      for (let i = 0; i < a.length; i++)
          if (!a[i].compare(b[i]))
              return false;
      return true;
  }
  /**
  Facet used to provide markers to the line number gutter.
  */
  const lineNumberMarkers = /*@__PURE__*/Facet.define();
  const lineNumberConfig = /*@__PURE__*/Facet.define({
      combine(values) {
          return combineConfig(values, { formatNumber: String, domEventHandlers: {} }, {
              domEventHandlers(a, b) {
                  let result = Object.assign({}, a);
                  for (let event in b) {
                      let exists = result[event], add = b[event];
                      result[event] = exists ? (view, line, event) => exists(view, line, event) || add(view, line, event) : add;
                  }
                  return result;
              }
          });
      }
  });
  class NumberMarker extends GutterMarker {
      constructor(number) {
          super();
          this.number = number;
      }
      eq(other) { return this.number == other.number; }
      toDOM() { return document.createTextNode(this.number); }
  }
  function formatNumber(view, number) {
      return view.state.facet(lineNumberConfig).formatNumber(number, view.state);
  }
  const lineNumberGutter = /*@__PURE__*/activeGutters.compute([lineNumberConfig], state => ({
      class: "cm-lineNumbers",
      renderEmptyElements: false,
      markers(view) { return view.state.facet(lineNumberMarkers); },
      lineMarker(view, line, others) {
          if (others.some(m => m.toDOM))
              return null;
          return new NumberMarker(formatNumber(view, view.state.doc.lineAt(line.from).number));
      },
      lineMarkerChange: update => update.startState.facet(lineNumberConfig) != update.state.facet(lineNumberConfig),
      initialSpacer(view) {
          return new NumberMarker(formatNumber(view, maxLineNumber(view.state.doc.lines)));
      },
      updateSpacer(spacer, update) {
          let max = formatNumber(update.view, maxLineNumber(update.view.state.doc.lines));
          return max == spacer.number ? spacer : new NumberMarker(max);
      },
      domEventHandlers: state.facet(lineNumberConfig).domEventHandlers
  }));
  /**
  Create a line number gutter extension.
  */
  function lineNumbers(config = {}) {
      return [
          lineNumberConfig.of(config),
          gutters(),
          lineNumberGutter
      ];
  }
  function maxLineNumber(lines) {
      let last = 9;
      while (last < lines)
          last = last * 10 + 9;
      return last;
  }
  const activeLineGutterMarker = /*@__PURE__*/new class extends GutterMarker {
      constructor() {
          super(...arguments);
          this.elementClass = "cm-activeLineGutter";
      }
  };
  const activeLineGutterHighlighter = /*@__PURE__*/gutterLineClass.compute(["selection"], state => {
      let marks = [], last = -1;
      for (let range of state.selection.ranges)
          if (range.empty) {
              let linePos = state.doc.lineAt(range.head).from;
              if (linePos > last) {
                  last = linePos;
                  marks.push(activeLineGutterMarker.range(linePos));
              }
          }
      return RangeSet.of(marks);
  });
  /**
  Returns an extension that adds a `cm-activeLineGutter` class to
  all gutter elements on the [active
  line](https://codemirror.net/6/docs/ref/#view.highlightActiveLine).
  */
  function highlightActiveLineGutter() {
      return activeLineGutterHighlighter;
  }

  /**
   * Defines a range on text using relative positions that can be transformed back to
   * absolute positions. (https://docs.yjs.dev/api/relative-positions)
   */
  class YRange {
    /**
     * @param {Y.RelativePosition} yanchor
     * @param {Y.RelativePosition} yhead
     */
    constructor (yanchor, yhead) {
      this.yanchor = yanchor;
      this.yhead = yhead;
    }

    /**
     * @returns {any}
     */
    toJSON () {
      return {
        yanchor: relativePositionToJSON(this.yanchor),
        yhead: relativePositionToJSON(this.yhead)
      }
    }

    /**
     * @param {any} json
     * @return {YRange}
     */
    static fromJSON (json) {
      return new YRange(createRelativePositionFromJSON(json.yanchor), createRelativePositionFromJSON(json.yhead))
    }
  }

  class YSyncConfig {
    constructor (ytext, awareness) {
      this.ytext = ytext;
      this.awareness = awareness;
      this.undoManager = new UndoManager(ytext);
    }

    /**
     * Helper function to transform an absolute index position to a Yjs-based relative position
     * (https://docs.yjs.dev/api/relative-positions).
     *
     * A relative position can be transformed back to an absolute position even after the document has changed. The position is
     * automatically adapted. This does not require any position transformations. Relative positions are computed based on
     * the internal Yjs document model. Peers that share content through Yjs are guaranteed that their positions will always
     * synced up when using relatve positions.
     *
     * ```js
     * import { ySyncFacet } from 'y-codemirror'
     *
     * ..
     * const ysync = view.state.facet(ySyncFacet)
     * // transform an absolute index position to a ypos
     * const ypos = ysync.getYPos(3)
     * // transform the ypos back to an absolute position
     * ysync.fromYPos(ypos) // => 3
     * ```
     *
     * It cannot be guaranteed that absolute index positions can be synced up between peers.
     * This might lead to undesired behavior when implementing features that require that all peers see the
     * same marked range (e.g. a comment plugin).
     *
     * @param {number} pos
     * @param {number} [assoc]
     */
    toYPos (pos, assoc = 0) {
      return createRelativePositionFromTypeIndex(this.ytext, pos, assoc)
    }

    /**
     * @param {Y.RelativePosition | Object} rpos
     */
    fromYPos (rpos) {
      const pos = createAbsolutePositionFromRelativePosition(createRelativePositionFromJSON(rpos), this.ytext.doc);
      if (pos == null || pos.type !== this.ytext) {
        throw new Error('[y-codemirror] The position you want to retrieve was created by a different document')
      }
      return {
        pos: pos.index,
        assoc: pos.assoc
      }
    }

    /**
     * @param {cmState.SelectionRange} range
     * @return {YRange}
     */
    toYRange (range) {
      const assoc = range.assoc;
      const yanchor = this.toYPos(range.anchor, assoc);
      const yhead = this.toYPos(range.head, assoc);
      return new YRange(yanchor, yhead)
    }

    /**
     * @param {YRange} yrange
     */
    fromYRange (yrange) {
      const anchor = this.fromYPos(yrange.yanchor);
      const head = this.fromYPos(yrange.yhead);
      if (anchor.pos === head.pos) {
        return EditorSelection.cursor(head.pos, head.assoc)
      }
      return EditorSelection.range(anchor.pos, head.pos)
    }
  }

  /**
   * @type {cmState.Facet<YSyncConfig, YSyncConfig>}
   */
  const ySyncFacet = Facet.define({
    combine (inputs) {
      return inputs[inputs.length - 1]
    }
  });

  /**
   * @type {cmState.AnnotationType<YSyncConfig>}
   */
  const ySyncAnnotation = Annotation.define();

  /**
   * @extends {PluginValue}
   */
  class YSyncPluginValue {
    /**
     * @param {cmView.EditorView} view
     */
    constructor (view) {
      this.view = view;
      this.conf = view.state.facet(ySyncFacet);
      this._observer = (event, tr) => {
        if (tr.origin !== this.conf) {
          const delta = event.delta;
          const changes = [];
          let pos = 0;
          for (let i = 0; i < delta.length; i++) {
            const d = delta[i];
            if (d.insert != null) {
              changes.push({ from: pos, to: pos, insert: d.insert });
            } else if (d.delete != null) {
              changes.push({ from: pos, to: pos + d.delete, insert: '' });
              pos += d.delete;
            } else {
              pos += d.retain;
            }
          }
          view.dispatch({ changes, annotations: [ySyncAnnotation.of(this.conf)] });
        }
      };
      this._ytext = this.conf.ytext;
      this._ytext.observe(this._observer);
    }

    /**
     * @param {cmView.ViewUpdate} update
     */
    update (update) {
      if (!update.docChanged || (update.transactions.length > 0 && update.transactions[0].annotation(ySyncAnnotation) === this.conf)) {
        return
      }
      const ytext = this.conf.ytext;
      ytext.doc.transact(() => {
        /**
         * This variable adjusts the fromA position to the current position in the Y.Text type.
         */
        let adj = 0;
        update.changes.iterChanges((fromA, toA, fromB, toB, insert) => {
          const insertText = insert.sliceString(0, insert.length, '\n');
          if (fromA !== toA) {
            ytext.delete(fromA + adj, toA - fromA);
          }
          if (insertText.length > 0) {
            ytext.insert(fromA + adj, insertText);
          }
          adj += insertText.length - (toA - fromA);
        });
      }, this.conf);
    }

    destroy () {
      this._ytext.unobserve(this._observer);
    }
  }

  const ySync = ViewPlugin.fromClass(YSyncPluginValue);

  const yRemoteSelectionsTheme = EditorView.baseTheme({
    '.cm-ySelection': {
    },
    '.cm-yLineSelection': {
      padding: 0,
      margin: '0px 2px 0px 4px'
    },
    '.cm-ySelectionCaret': {
      position: 'relative',
      borderLeft: '1px solid black',
      borderRight: '1px solid black',
      marginLeft: '-1px',
      marginRight: '-1px',
      boxSizing: 'border-box',
      display: 'inline'
    },
    '.cm-ySelectionCaretDot': {
      borderRadius: '50%',
      position: 'absolute',
      width: '.4em',
      height: '.4em',
      top: '-.2em',
      left: '-.2em',
      backgroundColor: 'inherit',
      transition: 'transform .3s ease-in-out',
      boxSizing: 'border-box'
    },
    '.cm-ySelectionCaret:hover > .cm-ySelectionCaretDot': {
      transformOrigin: 'bottom center',
      transform: 'scale(0)'
    },
    '.cm-ySelectionInfo': {
      position: 'absolute',
      top: '-1.05em',
      left: '-1px',
      fontSize: '.75em',
      fontFamily: 'serif',
      fontStyle: 'normal',
      fontWeight: 'normal',
      lineHeight: 'normal',
      userSelect: 'none',
      color: 'white',
      paddingLeft: '2px',
      paddingRight: '2px',
      zIndex: 101,
      transition: 'opacity .3s ease-in-out',
      backgroundColor: 'inherit',
      // these should be separate
      opacity: 0,
      transitionDelay: '0s',
      whiteSpace: 'nowrap'
    },
    '.cm-ySelectionCaret:hover > .cm-ySelectionInfo': {
      opacity: 1,
      transitionDelay: '0s'
    }
  });

  /**
   * @todo specify the users that actually changed. Currently, we recalculate positions for every user.
   * @type {cmState.AnnotationType<Array<number>>}
   */
  const yRemoteSelectionsAnnotation = Annotation.define();

  class YRemoteCaretWidget extends WidgetType {
    /**
     * @param {string} color
     * @param {string} name
     */
    constructor (color, name) {
      super();
      this.color = color;
      this.name = name;
    }

    toDOM () {
      return /** @type {HTMLElement} */ (element('span', [create('class', 'cm-ySelectionCaret'), create('style', `background-color: ${this.color}; border-color: ${this.color}`)], [
        text('\u2060'),
        element('div', [
          create('class', 'cm-ySelectionCaretDot')
        ]),
        text('\u2060'),
        element('div', [
          create('class', 'cm-ySelectionInfo')
        ], [
          text(this.name)
        ]),
        text('\u2060')
      ]))
    }

    eq (widget) {
      return widget.color === this.color
    }

    compare (widget) {
      return widget.color === this.color
    }

    updateDOM () {
      return false
    }

    get estimatedHeight () { return -1 }

    ignoreEvent () {
      return true
    }
  }

  class YRemoteSelectionsPluginValue {
    /**
     * @param {cmView.EditorView} view
     */
    constructor (view) {
      this.conf = view.state.facet(ySyncFacet);
      this._listener = ({ added, updated, removed }, s, t) => {
        const clients = added.concat(updated).concat(removed);
        if (clients.findIndex(id => id !== this.conf.awareness.doc.clientID) >= 0) {
          view.dispatch({ annotations: [yRemoteSelectionsAnnotation.of([])] });
        }
      };
      this._awareness = this.conf.awareness;
      this._awareness.on('change', this._listener);
      /**
       * @type {cmView.DecorationSet}
       */
      this.decorations = RangeSet.of([]);
    }

    destroy () {
      this._awareness.off('change', this._listener);
    }

    /**
     * @param {cmView.ViewUpdate} update
     */
    update (update) {
      const ytext = this.conf.ytext;
      const ydoc = /** @type {Y.Doc} */ (ytext.doc);
      const awareness = this.conf.awareness;
      /**
       * @type {Array<cmState.Range<cmView.Decoration>>}
       */
      const decorations = [];
      const localAwarenessState = this.conf.awareness.getLocalState();

      // set local awareness state (update cursors)
      if (localAwarenessState != null) {
        const hasFocus = update.view.hasFocus && update.view.dom.ownerDocument.hasFocus();
        const sel = hasFocus ? update.state.selection.main : null;
        const currentAnchor = localAwarenessState.cursor == null ? null : createRelativePositionFromJSON(localAwarenessState.cursor.anchor);
        const currentHead = localAwarenessState.cursor == null ? null : createRelativePositionFromJSON(localAwarenessState.cursor.head);

        if (sel != null) {
          const anchor = createRelativePositionFromTypeIndex(ytext, sel.anchor);
          const head = createRelativePositionFromTypeIndex(ytext, sel.head);
          if (localAwarenessState.cursor == null || !compareRelativePositions(currentAnchor, anchor) || !compareRelativePositions(currentHead, head)) {
            awareness.setLocalStateField('cursor', {
              anchor,
              head
            });
          }
        } else if (localAwarenessState.cursor != null && hasFocus) {
          awareness.setLocalStateField('cursor', null);
        }
      }

      // update decorations (remote selections)
      awareness.getStates().forEach((state, clientid) => {
        if (clientid === awareness.doc.clientID) {
          return
        }
        const cursor = state.cursor;
        if (cursor == null || cursor.anchor == null || cursor.head == null) {
          return
        }
        const anchor = createAbsolutePositionFromRelativePosition(cursor.anchor, ydoc);
        const head = createAbsolutePositionFromRelativePosition(cursor.head, ydoc);
        if (anchor == null || head == null || anchor.type !== ytext || head.type !== ytext) {
          return
        }
        const { color = '#30bced', name = 'Anonymous' } = state.user || {};
        const colorLight = (state.user && state.user.colorLight) || color + '33';
        const start = min(anchor.index, head.index);
        const end = max(anchor.index, head.index);
        const startLine = update.view.state.doc.lineAt(start);
        const endLine = update.view.state.doc.lineAt(end);
        if (startLine.number === endLine.number) {
          // selected content in a single line.
          decorations.push({
            from: start,
            to: end,
            value: Decoration.mark({
              attributes: { style: `background-color: ${colorLight}` },
              class: 'cm-ySelection'
            })
          });
        } else {
          // selected content in multiple lines
          // first, render text-selection in the first line
          decorations.push({
            from: start,
            to: startLine.from + startLine.length,
            value: Decoration.mark({
              attributes: { style: `background-color: ${colorLight}` },
              class: 'cm-ySelection'
            })
          });
          // render text-selection in the last line
          decorations.push({
            from: endLine.from,
            to: end,
            value: Decoration.mark({
              attributes: { style: `background-color: ${colorLight}` },
              class: 'cm-ySelection'
            })
          });
          for (let i = startLine.number + 1; i < endLine.number; i++) {
            const linePos = update.view.state.doc.line(i).from;
            decorations.push({
              from: linePos,
              to: linePos,
              value: Decoration.line({
                attributes: { style: `background-color: ${colorLight}`, class: 'cm-yLineSelection' }
              })
            });
          }
        }
        decorations.push({
          from: head.index,
          to: head.index,
          value: Decoration.widget({
            side: head.index - anchor.index > 0 ? -1 : 1, // the local cursor should be rendered outside the remote selection
            block: false,
            widget: new YRemoteCaretWidget(color, name)
          })
        });
      });
      this.decorations = Decoration.set(decorations, true);
    }
  }

  const yRemoteSelections = ViewPlugin.fromClass(YRemoteSelectionsPluginValue, {
    decorations: v => v.decorations
  });

  /**
   * Mutual exclude for JavaScript.
   *
   * @module mutex
   */

  /**
   * @callback mutex
   * @param {function():void} cb Only executed when this mutex is not in the current stack
   * @param {function():void} [elseCb] Executed when this mutex is in the current stack
   */

  /**
   * Creates a mutual exclude function with the following property:
   *
   * ```js
   * const mutex = createMutex()
   * mutex(() => {
   *   // This function is immediately executed
   *   mutex(() => {
   *     // This function is not executed, as the mutex is already active.
   *   })
   * })
   * ```
   *
   * @return {mutex} A mutual exclude function
   * @public
   */
  const createMutex = () => {
    let token = true;
    return (f, g) => {
      if (token) {
        token = false;
        try {
          f();
        } finally {
          token = true;
        }
      } else if (g !== undefined) {
        g();
      }
    }
  };

  class YUndoManagerConfig {
    /**
     * @param {Y.UndoManager} undoManager
     */
    constructor (undoManager) {
      this.undoManager = undoManager;
    }

    /**
     * @param {any} origin
     */
    addTrackedOrigin (origin) {
      this.undoManager.addTrackedOrigin(origin);
    }

    /**
     * @param {any} origin
     */
    removeTrackedOrigin (origin) {
      this.undoManager.removeTrackedOrigin(origin);
    }

    /**
     * @return {boolean} Whether a change was undone.
     */
    undo () {
      return this.undoManager.undo() != null
    }

    /**
     * @return {boolean} Whether a change was redone.
     */
    redo () {
      return this.undoManager.redo() != null
    }
  }

  /**
   * @type {cmState.Facet<YUndoManagerConfig, YUndoManagerConfig>}
   */
  const yUndoManagerFacet = Facet.define({
    combine (inputs) {
      return inputs[inputs.length - 1]
    }
  });

  /**
   * @extends {PluginValue}
   */
  class YUndoManagerPluginValue {
    /**
     * @param {cmView.EditorView} view
     */
    constructor (view) {
      this.view = view;
      this.conf = view.state.facet(yUndoManagerFacet);
      this._undoManager = this.conf.undoManager;
      this.syncConf = view.state.facet(ySyncFacet);
      /**
       * @type {null | YRange}
       */
      this._beforeChangeSelection = null;
      this._mux = createMutex();

      this._onStackItemAdded = ({ stackItem, changedParentTypes }) => {
        // only store metadata if this type was affected
        if (changedParentTypes.has(this.syncConf.ytext) && this._beforeChangeSelection && !stackItem.meta.has(this)) { // do not overwrite previous stored selection
          stackItem.meta.set(this, this._beforeChangeSelection);
        }
      };
      this._onStackItemPopped = ({ stackItem }) => {
        const sel = stackItem.meta.get(this);
        if (sel) {
          const selection = this.syncConf.fromYRange(sel);
          view.dispatch(view.state.update({ selection }));
          this._storeSelection();
        }
      };
      /**
       * Do this without mutex, simply use the sync annotation
       */
      this._storeSelection = () => {
        // store the selection before the change is applied so we can restore it with the undo manager.
        this._beforeChangeSelection = this.syncConf.toYRange(this.view.state.selection.main);
      };
      this._undoManager.on('stack-item-added', this._onStackItemAdded);
      this._undoManager.on('stack-item-popped', this._onStackItemPopped);
      this._undoManager.addTrackedOrigin(this.syncConf);
    }

    /**
     * @param {cmView.ViewUpdate} update
     */
    update (update) {
      if (update.selectionSet && (update.transactions.length === 0 || update.transactions[0].annotation(ySyncAnnotation) !== this.syncConf)) {
        // This only works when YUndoManagerPlugin is included before the sync plugin
        this._storeSelection();
      }
    }

    destroy () {
      this._undoManager.off('stack-item-added', this._onStackItemAdded);
      this._undoManager.off('stack-item-popped', this._onStackItemPopped);
      this._undoManager.removeTrackedOrigin(this.syncConf);
    }
  }
  const yUndoManager = ViewPlugin.fromClass(YUndoManagerPluginValue);

  /**
   * @type {cmState.StateCommand}
   */
  const undo$1 = ({ state, dispatch }) =>
    state.facet(yUndoManagerFacet).undo() || true;

  /**
   * @type {cmState.StateCommand}
   */
  const redo$1 = ({ state, dispatch }) =>
    state.facet(yUndoManagerFacet).redo() || true;

  /**
   * Default key bindigs for the undo manager.
   * @type {Array<cmView.KeyBinding>}
   */
  const yUndoManagerKeymap = [
    { key: 'Mod-z', run: undo$1, preventDefault: true },
    { key: 'Mod-y', mac: 'Mod-Shift-z', run: redo$1, preventDefault: true },
    { key: 'Mod-Shift-z', run: redo$1, preventDefault: true }
  ];

  /**
   * @param {Y.Text} ytext
   * @param {any} awareness
   * @param {Object} [opts]
   * @param {Y.UndoManager | false} [opts.undoManager] Set undoManager to false to disable the undo-redo plugin
   * @return {cmState.Extension}
   */
  const yCollab = (ytext, awareness, { undoManager = new UndoManager(ytext) } = {}) => {
    const ySyncConfig = new YSyncConfig(ytext, awareness);
    const plugins = [
      ySyncFacet.of(ySyncConfig),
      ySync
    ];
    if (awareness) {
      plugins.push(
        yRemoteSelectionsTheme,
        yRemoteSelections
      );
    }
    if (undoManager !== false) {
      // By default, only track changes that are produced by the sync plugin (local edits)
      plugins.push(
        yUndoManagerFacet.of(new YUndoManagerConfig(undoManager)),
        yUndoManager,
        EditorView.domEventHandlers({
          beforeinput (e, view) {
            if (e.inputType === 'historyUndo') return undo$1(view)
            if (e.inputType === 'historyRedo') return redo$1(view)
            return false
          }
        })
      );
    }
    return plugins
  };

  /* eslint-env browser */

  const reconnectTimeoutBase = 1200;
  const maxReconnectTimeout = 2500;
  // @todo - this should depend on awareness.outdatedTime
  const messageReconnectTimeout = 30000;

  /**
   * @param {WebsocketClient} wsclient
   */
  const setupWS = (wsclient) => {
    if (wsclient.shouldConnect && wsclient.ws === null) {
      const websocket = new WebSocket(wsclient.url);
      const binaryType = wsclient.binaryType;
      /**
       * @type {any}
       */
      let pingTimeout = null;
      if (binaryType) {
        websocket.binaryType = binaryType;
      }
      wsclient.ws = websocket;
      wsclient.connecting = true;
      wsclient.connected = false;
      websocket.onmessage = event => {
        wsclient.lastMessageReceived = getUnixTime();
        const data = event.data;
        const message = typeof data === 'string' ? JSON.parse(data) : data;
        if (message && message.type === 'pong') {
          clearTimeout(pingTimeout);
          pingTimeout = setTimeout(sendPing, messageReconnectTimeout / 2);
        }
        wsclient.emit('message', [message, wsclient]);
      };
      /**
       * @param {any} error
       */
      const onclose = error => {
        if (wsclient.ws !== null) {
          wsclient.ws = null;
          wsclient.connecting = false;
          if (wsclient.connected) {
            wsclient.connected = false;
            wsclient.emit('disconnect', [{ type: 'disconnect', error }, wsclient]);
          } else {
            wsclient.unsuccessfulReconnects++;
          }
          // Start with no reconnect timeout and increase timeout by
          // log10(wsUnsuccessfulReconnects).
          // The idea is to increase reconnect timeout slowly and have no reconnect
          // timeout at the beginning (log(1) = 0)
          setTimeout(setupWS, min(log10(wsclient.unsuccessfulReconnects + 1) * reconnectTimeoutBase, maxReconnectTimeout), wsclient);
        }
        clearTimeout(pingTimeout);
      };
      const sendPing = () => {
        if (wsclient.ws === websocket) {
          wsclient.send({
            type: 'ping'
          });
        }
      };
      websocket.onclose = () => onclose(null);
      websocket.onerror = error => onclose(error);
      websocket.onopen = () => {
        wsclient.lastMessageReceived = getUnixTime();
        wsclient.connecting = false;
        wsclient.connected = true;
        wsclient.unsuccessfulReconnects = 0;
        wsclient.emit('connect', [{ type: 'connect' }, wsclient]);
        // set ping
        pingTimeout = setTimeout(sendPing, messageReconnectTimeout / 2);
      };
    }
  };

  /**
   * @extends Observable<string>
   */
  class WebsocketClient extends Observable {
    /**
     * @param {string} url
     * @param {object} [opts]
     * @param {'arraybuffer' | 'blob' | null} [opts.binaryType] Set `ws.binaryType`
     */
    constructor (url, { binaryType } = {}) {
      super();
      this.url = url;
      /**
       * @type {WebSocket?}
       */
      this.ws = null;
      this.binaryType = binaryType || null;
      this.connected = false;
      this.connecting = false;
      this.unsuccessfulReconnects = 0;
      this.lastMessageReceived = 0;
      /**
       * Whether to connect to other peers or not
       * @type {boolean}
       */
      this.shouldConnect = true;
      this._checkInterval = setInterval(() => {
        if (this.connected && messageReconnectTimeout < getUnixTime() - this.lastMessageReceived) {
          // no message received in a long time - not even your own awareness
          // updates (which are updated every 15 seconds)
          /** @type {WebSocket} */ (this.ws).close();
        }
      }, messageReconnectTimeout / 2);
      setupWS(this);
    }

    /**
     * @param {any} message
     */
    send (message) {
      if (this.ws) {
        this.ws.send(JSON.stringify(message));
      }
    }

    destroy () {
      clearInterval(this._checkInterval);
      this.disconnect();
      super.destroy();
    }

    disconnect () {
      this.shouldConnect = false;
      if (this.ws !== null) {
        this.ws.close();
      }
    }

    connect () {
      this.shouldConnect = true;
      if (!this.connected && this.ws === null) {
        setupWS(this);
      }
    }
  }

  /* eslint-env browser */

  /**
   * @typedef {Object} Channel
   * @property {Set<function(any, any):any>} Channel.subs
   * @property {any} Channel.bc
   */

  /**
   * @type {Map<string, Channel>}
   */
  const channels = new Map();

  class LocalStoragePolyfill {
    /**
     * @param {string} room
     */
    constructor (room) {
      this.room = room;
      /**
       * @type {null|function({data:ArrayBuffer}):void}
       */
      this.onmessage = null;
      onChange(e => e.key === room && this.onmessage !== null && this.onmessage({ data: fromBase64(e.newValue || '') }));
    }

    /**
     * @param {ArrayBuffer} buf
     */
    postMessage (buf) {
      varStorage.setItem(this.room, toBase64(createUint8ArrayFromArrayBuffer(buf)));
    }
  }

  // Use BroadcastChannel or Polyfill
  const BC = typeof BroadcastChannel === 'undefined' ? LocalStoragePolyfill : BroadcastChannel;

  /**
   * @param {string} room
   * @return {Channel}
   */
  const getChannel = room =>
    setIfUndefined(channels, room, () => {
      const subs = new Set();
      const bc = new BC(room);
      /**
       * @param {{data:ArrayBuffer}} e
       */
      bc.onmessage = e => subs.forEach(sub => sub(e.data, 'broadcastchannel'));
      return {
        bc, subs
      }
    });

  /**
   * Subscribe to global `publish` events.
   *
   * @function
   * @param {string} room
   * @param {function(any, any):any} f
   */
  const subscribe = (room, f) => getChannel(room).subs.add(f);

  /**
   * Unsubscribe from `publish` global events.
   *
   * @function
   * @param {string} room
   * @param {function(any, any):any} f
   */
  const unsubscribe = (room, f) => getChannel(room).subs.delete(f);

  /**
   * Publish data to all subscribers (including subscribers on this tab)
   *
   * @function
   * @param {string} room
   * @param {any} data
   * @param {any} [origin]
   */
  const publish = (room, data, origin = null) => {
    const c = getChannel(room);
    c.bc.postMessage(data);
    c.subs.forEach(sub => sub(data, origin));
  };

  var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

  function createCommonjsModule(fn) {
    var module = { exports: {} };
  	return fn(module, module.exports), module.exports;
  }

  function commonjsRequire (target) {
  	throw new Error('Could not dynamically require "' + target + '". Please configure the dynamicRequireTargets option of @rollup/plugin-commonjs appropriately for this require call to behave properly.');
  }

  var simplepeer_min = createCommonjsModule(function (module, exports) {
  (function(e){module.exports=e();})(function(){var t=Math.floor,n=Math.abs,r=Math.pow;return function(){function d(s,e,n){function t(o,i){if(!e[o]){if(!s[o]){var l="function"==typeof commonjsRequire&&commonjsRequire;if(!i&&l)return l(o,!0);if(r)return r(o,!0);var c=new Error("Cannot find module '"+o+"'");throw c.code="MODULE_NOT_FOUND",c}var a=e[o]={exports:{}};s[o][0].call(a.exports,function(e){var r=s[o][1][e];return t(r||e)},a,a.exports,d,s,e,n);}return e[o].exports}for(var r="function"==typeof commonjsRequire&&commonjsRequire,a=0;a<n.length;a++)t(n[a]);return t}return d}()({1:[function(e,t,n){function r(e){var t=e.length;if(0<t%4)throw new Error("Invalid string. Length must be a multiple of 4");var n=e.indexOf("=");-1===n&&(n=t);var r=n===t?0:4-n%4;return [n,r]}function a(e,t,n){return 3*(t+n)/4-n}function o(e){var t,n,o=r(e),d=o[0],s=o[1],l=new p(a(e,d,s)),c=0,f=0<s?d-4:d;for(n=0;n<f;n+=4)t=u[e.charCodeAt(n)]<<18|u[e.charCodeAt(n+1)]<<12|u[e.charCodeAt(n+2)]<<6|u[e.charCodeAt(n+3)],l[c++]=255&t>>16,l[c++]=255&t>>8,l[c++]=255&t;return 2===s&&(t=u[e.charCodeAt(n)]<<2|u[e.charCodeAt(n+1)]>>4,l[c++]=255&t),1===s&&(t=u[e.charCodeAt(n)]<<10|u[e.charCodeAt(n+1)]<<4|u[e.charCodeAt(n+2)]>>2,l[c++]=255&t>>8,l[c++]=255&t),l}function d(e){return c[63&e>>18]+c[63&e>>12]+c[63&e>>6]+c[63&e]}function s(e,t,n){for(var r,a=[],o=t;o<n;o+=3)r=(16711680&e[o]<<16)+(65280&e[o+1]<<8)+(255&e[o+2]),a.push(d(r));return a.join("")}function l(e){for(var t,n=e.length,r=n%3,a=[],o=16383,d=0,l=n-r;d<l;d+=o)a.push(s(e,d,d+o>l?l:d+o));return 1===r?(t=e[n-1],a.push(c[t>>2]+c[63&t<<4]+"==")):2===r&&(t=(e[n-2]<<8)+e[n-1],a.push(c[t>>10]+c[63&t>>4]+c[63&t<<2]+"=")),a.join("")}n.byteLength=function(e){var t=r(e),n=t[0],a=t[1];return 3*(n+a)/4-a},n.toByteArray=o,n.fromByteArray=l;for(var c=[],u=[],p="undefined"==typeof Uint8Array?Array:Uint8Array,f="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",g=0,_=f.length;g<_;++g)c[g]=f[g],u[f.charCodeAt(g)]=g;u[45]=62,u[95]=63;},{}],2:[function(){},{}],3:[function(e,t,n){(function(){(function(){var t=String.fromCharCode,o=Math.min;function d(e){if(2147483647<e)throw new RangeError("The value \""+e+"\" is invalid for option \"size\"");var t=new Uint8Array(e);return t.__proto__=s.prototype,t}function s(e,t,n){if("number"==typeof e){if("string"==typeof t)throw new TypeError("The \"string\" argument must be of type string. Received type number");return p(e)}return l(e,t,n)}function l(e,t,n){if("string"==typeof e)return f(e,t);if(ArrayBuffer.isView(e))return g(e);if(null==e)throw TypeError("The first argument must be one of type string, Buffer, ArrayBuffer, Array, or Array-like Object. Received type "+typeof e);if(K(e,ArrayBuffer)||e&&K(e.buffer,ArrayBuffer))return _(e,t,n);if("number"==typeof e)throw new TypeError("The \"value\" argument must not be of type number. Received type number");var r=e.valueOf&&e.valueOf();if(null!=r&&r!==e)return s.from(r,t,n);var a=h(e);if(a)return a;if("undefined"!=typeof Symbol&&null!=Symbol.toPrimitive&&"function"==typeof e[Symbol.toPrimitive])return s.from(e[Symbol.toPrimitive]("string"),t,n);throw new TypeError("The first argument must be one of type string, Buffer, ArrayBuffer, Array, or Array-like Object. Received type "+typeof e)}function c(e){if("number"!=typeof e)throw new TypeError("\"size\" argument must be of type number");else if(0>e)throw new RangeError("The value \""+e+"\" is invalid for option \"size\"")}function u(e,t,n){return c(e),0>=e?d(e):void 0===t?d(e):"string"==typeof n?d(e).fill(t,n):d(e).fill(t)}function p(e){return c(e),d(0>e?0:0|m(e))}function f(e,t){if(("string"!=typeof t||""===t)&&(t="utf8"),!s.isEncoding(t))throw new TypeError("Unknown encoding: "+t);var n=0|b(e,t),r=d(n),a=r.write(e,t);return a!==n&&(r=r.slice(0,a)),r}function g(e){for(var t=0>e.length?0:0|m(e.length),n=d(t),r=0;r<t;r+=1)n[r]=255&e[r];return n}function _(e,t,n){if(0>t||e.byteLength<t)throw new RangeError("\"offset\" is outside of buffer bounds");if(e.byteLength<t+(n||0))throw new RangeError("\"length\" is outside of buffer bounds");var r;return r=void 0===t&&void 0===n?new Uint8Array(e):void 0===n?new Uint8Array(e,t):new Uint8Array(e,t,n),r.__proto__=s.prototype,r}function h(e){if(s.isBuffer(e)){var t=0|m(e.length),n=d(t);return 0===n.length?n:(e.copy(n,0,0,t),n)}return void 0===e.length?"Buffer"===e.type&&Array.isArray(e.data)?g(e.data):void 0:"number"!=typeof e.length||X(e.length)?d(0):g(e)}function m(e){if(e>=2147483647)throw new RangeError("Attempt to allocate Buffer larger than maximum size: 0x"+2147483647 .toString(16)+" bytes");return 0|e}function b(e,t){if(s.isBuffer(e))return e.length;if(ArrayBuffer.isView(e)||K(e,ArrayBuffer))return e.byteLength;if("string"!=typeof e)throw new TypeError("The \"string\" argument must be one of type string, Buffer, or ArrayBuffer. Received type "+typeof e);var n=e.length,r=2<arguments.length&&!0===arguments[2];if(!r&&0===n)return 0;for(var a=!1;;)switch(t){case"ascii":case"latin1":case"binary":return n;case"utf8":case"utf-8":return H(e).length;case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return 2*n;case"hex":return n>>>1;case"base64":return z(e).length;default:if(a)return r?-1:H(e).length;t=(""+t).toLowerCase(),a=!0;}}function y(e,t,n){var r=!1;if((void 0===t||0>t)&&(t=0),t>this.length)return "";if((void 0===n||n>this.length)&&(n=this.length),0>=n)return "";if(n>>>=0,t>>>=0,n<=t)return "";for(e||(e="utf8");;)switch(e){case"hex":return P(this,t,n);case"utf8":case"utf-8":return x(this,t,n);case"ascii":return D(this,t,n);case"latin1":case"binary":return I(this,t,n);case"base64":return A(this,t,n);case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return M(this,t,n);default:if(r)throw new TypeError("Unknown encoding: "+e);e=(e+"").toLowerCase(),r=!0;}}function C(e,t,n){var r=e[t];e[t]=e[n],e[n]=r;}function R(e,t,n,r,a){if(0===e.length)return -1;if("string"==typeof n?(r=n,n=0):2147483647<n?n=2147483647:-2147483648>n&&(n=-2147483648),n=+n,X(n)&&(n=a?0:e.length-1),0>n&&(n=e.length+n),n>=e.length){if(a)return -1;n=e.length-1;}else if(0>n)if(a)n=0;else return -1;if("string"==typeof t&&(t=s.from(t,r)),s.isBuffer(t))return 0===t.length?-1:E(e,t,n,r,a);if("number"==typeof t)return t&=255,"function"==typeof Uint8Array.prototype.indexOf?a?Uint8Array.prototype.indexOf.call(e,t,n):Uint8Array.prototype.lastIndexOf.call(e,t,n):E(e,[t],n,r,a);throw new TypeError("val must be string, number or Buffer")}function E(e,t,n,r,a){function o(e,t){return 1===d?e[t]:e.readUInt16BE(t*d)}var d=1,s=e.length,l=t.length;if(void 0!==r&&(r=(r+"").toLowerCase(),"ucs2"===r||"ucs-2"===r||"utf16le"===r||"utf-16le"===r)){if(2>e.length||2>t.length)return -1;d=2,s/=2,l/=2,n/=2;}var c;if(a){var u=-1;for(c=n;c<s;c++)if(o(e,c)!==o(t,-1===u?0:c-u))-1!==u&&(c-=c-u),u=-1;else if(-1===u&&(u=c),c-u+1===l)return u*d}else for(n+l>s&&(n=s-l),c=n;0<=c;c--){for(var p=!0,f=0;f<l;f++)if(o(e,c+f)!==o(t,f)){p=!1;break}if(p)return c}return -1}function w(e,t,n,r){n=+n||0;var a=e.length-n;r?(r=+r,r>a&&(r=a)):r=a;var o=t.length;r>o/2&&(r=o/2);for(var d,s=0;s<r;++s){if(d=parseInt(t.substr(2*s,2),16),X(d))return s;e[n+s]=d;}return s}function S(e,t,n,r){return G(H(t,e.length-n),e,n,r)}function T(e,t,n,r){return G(Y(t),e,n,r)}function v(e,t,n,r){return T(e,t,n,r)}function k(e,t,n,r){return G(z(t),e,n,r)}function L(e,t,n,r){return G(V(t,e.length-n),e,n,r)}function A(e,t,n){return 0===t&&n===e.length?$.fromByteArray(e):$.fromByteArray(e.slice(t,n))}function x(e,t,n){n=o(e.length,n);for(var r=[],a=t;a<n;){var d=e[a],s=null,l=239<d?4:223<d?3:191<d?2:1;if(a+l<=n){var c,u,p,f;1===l?128>d&&(s=d):2===l?(c=e[a+1],128==(192&c)&&(f=(31&d)<<6|63&c,127<f&&(s=f))):3===l?(c=e[a+1],u=e[a+2],128==(192&c)&&128==(192&u)&&(f=(15&d)<<12|(63&c)<<6|63&u,2047<f&&(55296>f||57343<f)&&(s=f))):4===l?(c=e[a+1],u=e[a+2],p=e[a+3],128==(192&c)&&128==(192&u)&&128==(192&p)&&(f=(15&d)<<18|(63&c)<<12|(63&u)<<6|63&p,65535<f&&1114112>f&&(s=f))):void 0;}null===s?(s=65533,l=1):65535<s&&(s-=65536,r.push(55296|1023&s>>>10),s=56320|1023&s),r.push(s),a+=l;}return N(r)}function N(e){var n=e.length;if(n<=4096)return t.apply(String,e);for(var r="",a=0;a<n;)r+=t.apply(String,e.slice(a,a+=4096));return r}function D(e,n,r){var a="";r=o(e.length,r);for(var d=n;d<r;++d)a+=t(127&e[d]);return a}function I(e,n,r){var a="";r=o(e.length,r);for(var d=n;d<r;++d)a+=t(e[d]);return a}function P(e,t,n){var r=e.length;(!t||0>t)&&(t=0),(!n||0>n||n>r)&&(n=r);for(var a="",o=t;o<n;++o)a+=W(e[o]);return a}function M(e,n,r){for(var a=e.slice(n,r),o="",d=0;d<a.length;d+=2)o+=t(a[d]+256*a[d+1]);return o}function O(e,t,n){if(0!=e%1||0>e)throw new RangeError("offset is not uint");if(e+t>n)throw new RangeError("Trying to access beyond buffer length")}function F(e,t,n,r,a,o){if(!s.isBuffer(e))throw new TypeError("\"buffer\" argument must be a Buffer instance");if(t>a||t<o)throw new RangeError("\"value\" argument is out of bounds");if(n+r>e.length)throw new RangeError("Index out of range")}function B(e,t,n,r){if(n+r>e.length)throw new RangeError("Index out of range");if(0>n)throw new RangeError("Index out of range")}function U(e,t,n,r,a){return t=+t,n>>>=0,a||B(e,t,n,4),J.write(e,t,n,r,23,4),n+4}function j(e,t,n,r,a){return t=+t,n>>>=0,a||B(e,t,n,8),J.write(e,t,n,r,52,8),n+8}function q(e){if(e=e.split("=")[0],e=e.trim().replace(Q,""),2>e.length)return "";for(;0!=e.length%4;)e+="=";return e}function W(e){return 16>e?"0"+e.toString(16):e.toString(16)}function H(e,t){t=t||1/0;for(var n,r=e.length,a=null,o=[],d=0;d<r;++d){if(n=e.charCodeAt(d),55295<n&&57344>n){if(!a){if(56319<n){-1<(t-=3)&&o.push(239,191,189);continue}else if(d+1===r){-1<(t-=3)&&o.push(239,191,189);continue}a=n;continue}if(56320>n){-1<(t-=3)&&o.push(239,191,189),a=n;continue}n=(a-55296<<10|n-56320)+65536;}else a&&-1<(t-=3)&&o.push(239,191,189);if(a=null,128>n){if(0>(t-=1))break;o.push(n);}else if(2048>n){if(0>(t-=2))break;o.push(192|n>>6,128|63&n);}else if(65536>n){if(0>(t-=3))break;o.push(224|n>>12,128|63&n>>6,128|63&n);}else if(1114112>n){if(0>(t-=4))break;o.push(240|n>>18,128|63&n>>12,128|63&n>>6,128|63&n);}else throw new Error("Invalid code point")}return o}function Y(e){for(var t=[],n=0;n<e.length;++n)t.push(255&e.charCodeAt(n));return t}function V(e,t){for(var n,r,a,o=[],d=0;d<e.length&&!(0>(t-=2));++d)n=e.charCodeAt(d),r=n>>8,a=n%256,o.push(a),o.push(r);return o}function z(e){return $.toByteArray(q(e))}function G(e,t,n,r){for(var a=0;a<r&&!(a+n>=t.length||a>=e.length);++a)t[a+n]=e[a];return a}function K(e,t){return e instanceof t||null!=e&&null!=e.constructor&&null!=e.constructor.name&&e.constructor.name===t.name}function X(e){return e!==e}var $=e("base64-js"),J=e("ieee754");n.Buffer=s,n.SlowBuffer=function(e){return +e!=e&&(e=0),s.alloc(+e)},n.INSPECT_MAX_BYTES=50;n.kMaxLength=2147483647,s.TYPED_ARRAY_SUPPORT=function(){try{var e=new Uint8Array(1);return e.__proto__={__proto__:Uint8Array.prototype,foo:function(){return 42}},42===e.foo()}catch(t){return !1}}(),s.TYPED_ARRAY_SUPPORT||"undefined"==typeof console||"function"!=typeof console.error||console.error("This browser lacks typed array (Uint8Array) support which is required by `buffer` v5.x. Use `buffer` v4.x if you require old browser support."),Object.defineProperty(s.prototype,"parent",{enumerable:!0,get:function(){return s.isBuffer(this)?this.buffer:void 0}}),Object.defineProperty(s.prototype,"offset",{enumerable:!0,get:function(){return s.isBuffer(this)?this.byteOffset:void 0}}),"undefined"!=typeof Symbol&&null!=Symbol.species&&s[Symbol.species]===s&&Object.defineProperty(s,Symbol.species,{value:null,configurable:!0,enumerable:!1,writable:!1}),s.poolSize=8192,s.from=function(e,t,n){return l(e,t,n)},s.prototype.__proto__=Uint8Array.prototype,s.__proto__=Uint8Array,s.alloc=function(e,t,n){return u(e,t,n)},s.allocUnsafe=function(e){return p(e)},s.allocUnsafeSlow=function(e){return p(e)},s.isBuffer=function(e){return null!=e&&!0===e._isBuffer&&e!==s.prototype},s.compare=function(e,t){if(K(e,Uint8Array)&&(e=s.from(e,e.offset,e.byteLength)),K(t,Uint8Array)&&(t=s.from(t,t.offset,t.byteLength)),!s.isBuffer(e)||!s.isBuffer(t))throw new TypeError("The \"buf1\", \"buf2\" arguments must be one of type Buffer or Uint8Array");if(e===t)return 0;for(var n=e.length,r=t.length,d=0,l=o(n,r);d<l;++d)if(e[d]!==t[d]){n=e[d],r=t[d];break}return n<r?-1:r<n?1:0},s.isEncoding=function(e){switch((e+"").toLowerCase()){case"hex":case"utf8":case"utf-8":case"ascii":case"latin1":case"binary":case"base64":case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return !0;default:return !1;}},s.concat=function(e,t){if(!Array.isArray(e))throw new TypeError("\"list\" argument must be an Array of Buffers");if(0===e.length)return s.alloc(0);var n;if(t===void 0)for(t=0,n=0;n<e.length;++n)t+=e[n].length;var r=s.allocUnsafe(t),a=0;for(n=0;n<e.length;++n){var o=e[n];if(K(o,Uint8Array)&&(o=s.from(o)),!s.isBuffer(o))throw new TypeError("\"list\" argument must be an Array of Buffers");o.copy(r,a),a+=o.length;}return r},s.byteLength=b,s.prototype._isBuffer=!0,s.prototype.swap16=function(){var e=this.length;if(0!=e%2)throw new RangeError("Buffer size must be a multiple of 16-bits");for(var t=0;t<e;t+=2)C(this,t,t+1);return this},s.prototype.swap32=function(){var e=this.length;if(0!=e%4)throw new RangeError("Buffer size must be a multiple of 32-bits");for(var t=0;t<e;t+=4)C(this,t,t+3),C(this,t+1,t+2);return this},s.prototype.swap64=function(){var e=this.length;if(0!=e%8)throw new RangeError("Buffer size must be a multiple of 64-bits");for(var t=0;t<e;t+=8)C(this,t,t+7),C(this,t+1,t+6),C(this,t+2,t+5),C(this,t+3,t+4);return this},s.prototype.toString=function(){var e=this.length;return 0===e?"":0===arguments.length?x(this,0,e):y.apply(this,arguments)},s.prototype.toLocaleString=s.prototype.toString,s.prototype.equals=function(e){if(!s.isBuffer(e))throw new TypeError("Argument must be a Buffer");return this===e||0===s.compare(this,e)},s.prototype.inspect=function(){var e="",t=n.INSPECT_MAX_BYTES;return e=this.toString("hex",0,t).replace(/(.{2})/g,"$1 ").trim(),this.length>t&&(e+=" ... "),"<Buffer "+e+">"},s.prototype.compare=function(e,t,n,r,a){if(K(e,Uint8Array)&&(e=s.from(e,e.offset,e.byteLength)),!s.isBuffer(e))throw new TypeError("The \"target\" argument must be one of type Buffer or Uint8Array. Received type "+typeof e);if(void 0===t&&(t=0),void 0===n&&(n=e?e.length:0),void 0===r&&(r=0),void 0===a&&(a=this.length),0>t||n>e.length||0>r||a>this.length)throw new RangeError("out of range index");if(r>=a&&t>=n)return 0;if(r>=a)return -1;if(t>=n)return 1;if(t>>>=0,n>>>=0,r>>>=0,a>>>=0,this===e)return 0;for(var d=a-r,l=n-t,c=o(d,l),u=this.slice(r,a),p=e.slice(t,n),f=0;f<c;++f)if(u[f]!==p[f]){d=u[f],l=p[f];break}return d<l?-1:l<d?1:0},s.prototype.includes=function(e,t,n){return -1!==this.indexOf(e,t,n)},s.prototype.indexOf=function(e,t,n){return R(this,e,t,n,!0)},s.prototype.lastIndexOf=function(e,t,n){return R(this,e,t,n,!1)},s.prototype.write=function(e,t,n,r){if(void 0===t)r="utf8",n=this.length,t=0;else if(void 0===n&&"string"==typeof t)r=t,n=this.length,t=0;else if(isFinite(t))t>>>=0,isFinite(n)?(n>>>=0,void 0===r&&(r="utf8")):(r=n,n=void 0);else throw new Error("Buffer.write(string, encoding, offset[, length]) is no longer supported");var a=this.length-t;if((void 0===n||n>a)&&(n=a),0<e.length&&(0>n||0>t)||t>this.length)throw new RangeError("Attempt to write outside buffer bounds");r||(r="utf8");for(var o=!1;;)switch(r){case"hex":return w(this,e,t,n);case"utf8":case"utf-8":return S(this,e,t,n);case"ascii":return T(this,e,t,n);case"latin1":case"binary":return v(this,e,t,n);case"base64":return k(this,e,t,n);case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return L(this,e,t,n);default:if(o)throw new TypeError("Unknown encoding: "+r);r=(""+r).toLowerCase(),o=!0;}},s.prototype.toJSON=function(){return {type:"Buffer",data:Array.prototype.slice.call(this._arr||this,0)}};s.prototype.slice=function(e,t){var n=this.length;e=~~e,t=t===void 0?n:~~t,0>e?(e+=n,0>e&&(e=0)):e>n&&(e=n),0>t?(t+=n,0>t&&(t=0)):t>n&&(t=n),t<e&&(t=e);var r=this.subarray(e,t);return r.__proto__=s.prototype,r},s.prototype.readUIntLE=function(e,t,n){e>>>=0,t>>>=0,n||O(e,t,this.length);for(var r=this[e],a=1,o=0;++o<t&&(a*=256);)r+=this[e+o]*a;return r},s.prototype.readUIntBE=function(e,t,n){e>>>=0,t>>>=0,n||O(e,t,this.length);for(var r=this[e+--t],a=1;0<t&&(a*=256);)r+=this[e+--t]*a;return r},s.prototype.readUInt8=function(e,t){return e>>>=0,t||O(e,1,this.length),this[e]},s.prototype.readUInt16LE=function(e,t){return e>>>=0,t||O(e,2,this.length),this[e]|this[e+1]<<8},s.prototype.readUInt16BE=function(e,t){return e>>>=0,t||O(e,2,this.length),this[e]<<8|this[e+1]},s.prototype.readUInt32LE=function(e,t){return e>>>=0,t||O(e,4,this.length),(this[e]|this[e+1]<<8|this[e+2]<<16)+16777216*this[e+3]},s.prototype.readUInt32BE=function(e,t){return e>>>=0,t||O(e,4,this.length),16777216*this[e]+(this[e+1]<<16|this[e+2]<<8|this[e+3])},s.prototype.readIntLE=function(e,t,n){e>>>=0,t>>>=0,n||O(e,t,this.length);for(var a=this[e],o=1,d=0;++d<t&&(o*=256);)a+=this[e+d]*o;return o*=128,a>=o&&(a-=r(2,8*t)),a},s.prototype.readIntBE=function(e,t,n){e>>>=0,t>>>=0,n||O(e,t,this.length);for(var a=t,o=1,d=this[e+--a];0<a&&(o*=256);)d+=this[e+--a]*o;return o*=128,d>=o&&(d-=r(2,8*t)),d},s.prototype.readInt8=function(e,t){return e>>>=0,t||O(e,1,this.length),128&this[e]?-1*(255-this[e]+1):this[e]},s.prototype.readInt16LE=function(e,t){e>>>=0,t||O(e,2,this.length);var n=this[e]|this[e+1]<<8;return 32768&n?4294901760|n:n},s.prototype.readInt16BE=function(e,t){e>>>=0,t||O(e,2,this.length);var n=this[e+1]|this[e]<<8;return 32768&n?4294901760|n:n},s.prototype.readInt32LE=function(e,t){return e>>>=0,t||O(e,4,this.length),this[e]|this[e+1]<<8|this[e+2]<<16|this[e+3]<<24},s.prototype.readInt32BE=function(e,t){return e>>>=0,t||O(e,4,this.length),this[e]<<24|this[e+1]<<16|this[e+2]<<8|this[e+3]},s.prototype.readFloatLE=function(e,t){return e>>>=0,t||O(e,4,this.length),J.read(this,e,!0,23,4)},s.prototype.readFloatBE=function(e,t){return e>>>=0,t||O(e,4,this.length),J.read(this,e,!1,23,4)},s.prototype.readDoubleLE=function(e,t){return e>>>=0,t||O(e,8,this.length),J.read(this,e,!0,52,8)},s.prototype.readDoubleBE=function(e,t){return e>>>=0,t||O(e,8,this.length),J.read(this,e,!1,52,8)},s.prototype.writeUIntLE=function(e,t,n,a){if(e=+e,t>>>=0,n>>>=0,!a){var o=r(2,8*n)-1;F(this,e,t,n,o,0);}var d=1,s=0;for(this[t]=255&e;++s<n&&(d*=256);)this[t+s]=255&e/d;return t+n},s.prototype.writeUIntBE=function(e,t,n,a){if(e=+e,t>>>=0,n>>>=0,!a){var o=r(2,8*n)-1;F(this,e,t,n,o,0);}var d=n-1,s=1;for(this[t+d]=255&e;0<=--d&&(s*=256);)this[t+d]=255&e/s;return t+n},s.prototype.writeUInt8=function(e,t,n){return e=+e,t>>>=0,n||F(this,e,t,1,255,0),this[t]=255&e,t+1},s.prototype.writeUInt16LE=function(e,t,n){return e=+e,t>>>=0,n||F(this,e,t,2,65535,0),this[t]=255&e,this[t+1]=e>>>8,t+2},s.prototype.writeUInt16BE=function(e,t,n){return e=+e,t>>>=0,n||F(this,e,t,2,65535,0),this[t]=e>>>8,this[t+1]=255&e,t+2},s.prototype.writeUInt32LE=function(e,t,n){return e=+e,t>>>=0,n||F(this,e,t,4,4294967295,0),this[t+3]=e>>>24,this[t+2]=e>>>16,this[t+1]=e>>>8,this[t]=255&e,t+4},s.prototype.writeUInt32BE=function(e,t,n){return e=+e,t>>>=0,n||F(this,e,t,4,4294967295,0),this[t]=e>>>24,this[t+1]=e>>>16,this[t+2]=e>>>8,this[t+3]=255&e,t+4},s.prototype.writeIntLE=function(e,t,n,a){if(e=+e,t>>>=0,!a){var o=r(2,8*n-1);F(this,e,t,n,o-1,-o);}var d=0,s=1,l=0;for(this[t]=255&e;++d<n&&(s*=256);)0>e&&0===l&&0!==this[t+d-1]&&(l=1),this[t+d]=255&(e/s>>0)-l;return t+n},s.prototype.writeIntBE=function(e,t,n,a){if(e=+e,t>>>=0,!a){var o=r(2,8*n-1);F(this,e,t,n,o-1,-o);}var d=n-1,s=1,l=0;for(this[t+d]=255&e;0<=--d&&(s*=256);)0>e&&0===l&&0!==this[t+d+1]&&(l=1),this[t+d]=255&(e/s>>0)-l;return t+n},s.prototype.writeInt8=function(e,t,n){return e=+e,t>>>=0,n||F(this,e,t,1,127,-128),0>e&&(e=255+e+1),this[t]=255&e,t+1},s.prototype.writeInt16LE=function(e,t,n){return e=+e,t>>>=0,n||F(this,e,t,2,32767,-32768),this[t]=255&e,this[t+1]=e>>>8,t+2},s.prototype.writeInt16BE=function(e,t,n){return e=+e,t>>>=0,n||F(this,e,t,2,32767,-32768),this[t]=e>>>8,this[t+1]=255&e,t+2},s.prototype.writeInt32LE=function(e,t,n){return e=+e,t>>>=0,n||F(this,e,t,4,2147483647,-2147483648),this[t]=255&e,this[t+1]=e>>>8,this[t+2]=e>>>16,this[t+3]=e>>>24,t+4},s.prototype.writeInt32BE=function(e,t,n){return e=+e,t>>>=0,n||F(this,e,t,4,2147483647,-2147483648),0>e&&(e=4294967295+e+1),this[t]=e>>>24,this[t+1]=e>>>16,this[t+2]=e>>>8,this[t+3]=255&e,t+4},s.prototype.writeFloatLE=function(e,t,n){return U(this,e,t,!0,n)},s.prototype.writeFloatBE=function(e,t,n){return U(this,e,t,!1,n)},s.prototype.writeDoubleLE=function(e,t,n){return j(this,e,t,!0,n)},s.prototype.writeDoubleBE=function(e,t,n){return j(this,e,t,!1,n)},s.prototype.copy=function(e,t,n,r){if(!s.isBuffer(e))throw new TypeError("argument should be a Buffer");if(n||(n=0),r||0===r||(r=this.length),t>=e.length&&(t=e.length),t||(t=0),0<r&&r<n&&(r=n),r===n)return 0;if(0===e.length||0===this.length)return 0;if(0>t)throw new RangeError("targetStart out of bounds");if(0>n||n>=this.length)throw new RangeError("Index out of range");if(0>r)throw new RangeError("sourceEnd out of bounds");r>this.length&&(r=this.length),e.length-t<r-n&&(r=e.length-t+n);var a=r-n;if(this===e&&"function"==typeof Uint8Array.prototype.copyWithin)this.copyWithin(t,n,r);else if(this===e&&n<t&&t<r)for(var o=a-1;0<=o;--o)e[o+t]=this[o+n];else Uint8Array.prototype.set.call(e,this.subarray(n,r),t);return a},s.prototype.fill=function(e,t,n,r){if("string"==typeof e){if("string"==typeof t?(r=t,t=0,n=this.length):"string"==typeof n&&(r=n,n=this.length),void 0!==r&&"string"!=typeof r)throw new TypeError("encoding must be a string");if("string"==typeof r&&!s.isEncoding(r))throw new TypeError("Unknown encoding: "+r);if(1===e.length){var a=e.charCodeAt(0);("utf8"===r&&128>a||"latin1"===r)&&(e=a);}}else "number"==typeof e&&(e&=255);if(0>t||this.length<t||this.length<n)throw new RangeError("Out of range index");if(n<=t)return this;t>>>=0,n=n===void 0?this.length:n>>>0,e||(e=0);var o;if("number"==typeof e)for(o=t;o<n;++o)this[o]=e;else {var d=s.isBuffer(e)?e:s.from(e,r),l=d.length;if(0===l)throw new TypeError("The value \""+e+"\" is invalid for argument \"value\"");for(o=0;o<n-t;++o)this[o+t]=d[o%l];}return this};var Q=/[^+/0-9A-Za-z-_]/g;}).call(this);}).call(this,e("buffer").Buffer);},{"base64-js":1,buffer:3,ieee754:9}],4:[function(e,t,n){(function(a){(function(){function r(){let e;try{e=n.storage.getItem("debug");}catch(e){}return !e&&"undefined"!=typeof a&&"env"in a&&(e=a.env.DEBUG),e}n.formatArgs=function(e){if(e[0]=(this.useColors?"%c":"")+this.namespace+(this.useColors?" %c":" ")+e[0]+(this.useColors?"%c ":" ")+"+"+t.exports.humanize(this.diff),!this.useColors)return;const n="color: "+this.color;e.splice(1,0,n,"color: inherit");let r=0,a=0;e[0].replace(/%[a-zA-Z%]/g,e=>{"%%"===e||(r++,"%c"===e&&(a=r));}),e.splice(a,0,n);},n.save=function(e){try{e?n.storage.setItem("debug",e):n.storage.removeItem("debug");}catch(e){}},n.load=r,n.useColors=function(){return !!("undefined"!=typeof window&&window.process&&("renderer"===window.process.type||window.process.__nwjs))||!("undefined"!=typeof navigator&&navigator.userAgent&&navigator.userAgent.toLowerCase().match(/(edge|trident)\/(\d+)/))&&("undefined"!=typeof document&&document.documentElement&&document.documentElement.style&&document.documentElement.style.WebkitAppearance||"undefined"!=typeof window&&window.console&&(window.console.firebug||window.console.exception&&window.console.table)||"undefined"!=typeof navigator&&navigator.userAgent&&navigator.userAgent.toLowerCase().match(/firefox\/(\d+)/)&&31<=parseInt(RegExp.$1,10)||"undefined"!=typeof navigator&&navigator.userAgent&&navigator.userAgent.toLowerCase().match(/applewebkit\/(\d+)/))},n.storage=function(){try{return localStorage}catch(e){}}(),n.destroy=(()=>{let e=!1;return ()=>{e||(e=!0,console.warn("Instance method `debug.destroy()` is deprecated and no longer does anything. It will be removed in the next major version of `debug`."));}})(),n.colors=["#0000CC","#0000FF","#0033CC","#0033FF","#0066CC","#0066FF","#0099CC","#0099FF","#00CC00","#00CC33","#00CC66","#00CC99","#00CCCC","#00CCFF","#3300CC","#3300FF","#3333CC","#3333FF","#3366CC","#3366FF","#3399CC","#3399FF","#33CC00","#33CC33","#33CC66","#33CC99","#33CCCC","#33CCFF","#6600CC","#6600FF","#6633CC","#6633FF","#66CC00","#66CC33","#9900CC","#9900FF","#9933CC","#9933FF","#99CC00","#99CC33","#CC0000","#CC0033","#CC0066","#CC0099","#CC00CC","#CC00FF","#CC3300","#CC3333","#CC3366","#CC3399","#CC33CC","#CC33FF","#CC6600","#CC6633","#CC9900","#CC9933","#CCCC00","#CCCC33","#FF0000","#FF0033","#FF0066","#FF0099","#FF00CC","#FF00FF","#FF3300","#FF3333","#FF3366","#FF3399","#FF33CC","#FF33FF","#FF6600","#FF6633","#FF9900","#FF9933","#FFCC00","#FFCC33"],n.log=console.debug||console.log||(()=>{}),t.exports=e("./common")(n);const{formatters:o}=t.exports;o.j=function(e){try{return JSON.stringify(e)}catch(e){return "[UnexpectedJSONParseError]: "+e.message}};}).call(this);}).call(this,e("_process"));},{"./common":5,_process:12}],5:[function(e,t){t.exports=function(t){function r(e){function t(...e){if(!t.enabled)return;const a=t,o=+new Date,i=o-(n||o);a.diff=i,a.prev=n,a.curr=o,n=o,e[0]=r.coerce(e[0]),"string"!=typeof e[0]&&e.unshift("%O");let d=0;e[0]=e[0].replace(/%([a-zA-Z%])/g,(t,n)=>{if("%%"===t)return "%";d++;const o=r.formatters[n];if("function"==typeof o){const n=e[d];t=o.call(a,n),e.splice(d,1),d--;}return t}),r.formatArgs.call(a,e);const s=a.log||r.log;s.apply(a,e);}let n,o=null;return t.namespace=e,t.useColors=r.useColors(),t.color=r.selectColor(e),t.extend=a,t.destroy=r.destroy,Object.defineProperty(t,"enabled",{enumerable:!0,configurable:!1,get:()=>null===o?r.enabled(e):o,set:e=>{o=e;}}),"function"==typeof r.init&&r.init(t),t}function a(e,t){const n=r(this.namespace+("undefined"==typeof t?":":t)+e);return n.log=this.log,n}function o(e){return e.toString().substring(2,e.toString().length-2).replace(/\.\*\?$/,"*")}return r.debug=r,r.default=r,r.coerce=function(e){return e instanceof Error?e.stack||e.message:e},r.disable=function(){const e=[...r.names.map(o),...r.skips.map(o).map(e=>"-"+e)].join(",");return r.enable(""),e},r.enable=function(e){r.save(e),r.names=[],r.skips=[];let t;const n=("string"==typeof e?e:"").split(/[\s,]+/),a=n.length;for(t=0;t<a;t++)n[t]&&(e=n[t].replace(/\*/g,".*?"),"-"===e[0]?r.skips.push(new RegExp("^"+e.substr(1)+"$")):r.names.push(new RegExp("^"+e+"$")));},r.enabled=function(e){if("*"===e[e.length-1])return !0;let t,n;for(t=0,n=r.skips.length;t<n;t++)if(r.skips[t].test(e))return !1;for(t=0,n=r.names.length;t<n;t++)if(r.names[t].test(e))return !0;return !1},r.humanize=e("ms"),r.destroy=function(){console.warn("Instance method `debug.destroy()` is deprecated and no longer does anything. It will be removed in the next major version of `debug`.");},Object.keys(t).forEach(e=>{r[e]=t[e];}),r.names=[],r.skips=[],r.formatters={},r.selectColor=function(e){let t=0;for(let n=0;n<e.length;n++)t=(t<<5)-t+e.charCodeAt(n),t|=0;return r.colors[n(t)%r.colors.length]},r.enable(r.load()),r};},{ms:11}],6:[function(e,t){function n(e,t){for(const n in t)Object.defineProperty(e,n,{value:t[n],enumerable:!0,configurable:!0});return e}t.exports=function(e,t,r){if(!e||"string"==typeof e)throw new TypeError("Please pass an Error to err-code");r||(r={}),"object"==typeof t&&(r=t,t=""),t&&(r.code=t);try{return n(e,r)}catch(t){r.message=e.message,r.stack=e.stack;const a=function(){};a.prototype=Object.create(Object.getPrototypeOf(e));const o=n(new a,r);return o}};},{}],7:[function(e,t){function n(e){console&&console.warn&&console.warn(e);}function r(){r.init.call(this);}function a(e){if("function"!=typeof e)throw new TypeError("The \"listener\" argument must be of type Function. Received type "+typeof e)}function o(e){return void 0===e._maxListeners?r.defaultMaxListeners:e._maxListeners}function i(e,t,r,i){var d,s,l;if(a(r),s=e._events,void 0===s?(s=e._events=Object.create(null),e._eventsCount=0):(void 0!==s.newListener&&(e.emit("newListener",t,r.listener?r.listener:r),s=e._events),l=s[t]),void 0===l)l=s[t]=r,++e._eventsCount;else if("function"==typeof l?l=s[t]=i?[r,l]:[l,r]:i?l.unshift(r):l.push(r),d=o(e),0<d&&l.length>d&&!l.warned){l.warned=!0;var c=new Error("Possible EventEmitter memory leak detected. "+l.length+" "+(t+" listeners added. Use emitter.setMaxListeners() to increase limit"));c.name="MaxListenersExceededWarning",c.emitter=e,c.type=t,c.count=l.length,n(c);}return e}function d(){if(!this.fired)return this.target.removeListener(this.type,this.wrapFn),this.fired=!0,0===arguments.length?this.listener.call(this.target):this.listener.apply(this.target,arguments)}function s(e,t,n){var r={fired:!1,wrapFn:void 0,target:e,type:t,listener:n},a=d.bind(r);return a.listener=n,r.wrapFn=a,a}function l(e,t,n){var r=e._events;if(r===void 0)return [];var a=r[t];return void 0===a?[]:"function"==typeof a?n?[a.listener||a]:[a]:n?f(a):u(a,a.length)}function c(e){var t=this._events;if(t!==void 0){var n=t[e];if("function"==typeof n)return 1;if(void 0!==n)return n.length}return 0}function u(e,t){for(var n=Array(t),r=0;r<t;++r)n[r]=e[r];return n}function p(e,t){for(;t+1<e.length;t++)e[t]=e[t+1];e.pop();}function f(e){for(var t=Array(e.length),n=0;n<t.length;++n)t[n]=e[n].listener||e[n];return t}function g(e,t,n){"function"==typeof e.on&&_(e,"error",t,n);}function _(e,t,n,r){if("function"==typeof e.on)r.once?e.once(t,n):e.on(t,n);else if("function"==typeof e.addEventListener)e.addEventListener(t,function a(o){r.once&&e.removeEventListener(t,a),n(o);});else throw new TypeError("The \"emitter\" argument must be of type EventEmitter. Received type "+typeof e)}var h,m="object"==typeof Reflect?Reflect:null,b=m&&"function"==typeof m.apply?m.apply:function(e,t,n){return Function.prototype.apply.call(e,t,n)};h=m&&"function"==typeof m.ownKeys?m.ownKeys:Object.getOwnPropertySymbols?function(e){return Object.getOwnPropertyNames(e).concat(Object.getOwnPropertySymbols(e))}:function(e){return Object.getOwnPropertyNames(e)};var y=Number.isNaN||function(e){return e!==e};t.exports=r,t.exports.once=function(e,t){return new Promise(function(n,r){function a(n){e.removeListener(t,o),r(n);}function o(){"function"==typeof e.removeListener&&e.removeListener("error",a),n([].slice.call(arguments));}_(e,t,o,{once:!0}),"error"!==t&&g(e,a,{once:!0});})},r.EventEmitter=r,r.prototype._events=void 0,r.prototype._eventsCount=0,r.prototype._maxListeners=void 0;var C=10;Object.defineProperty(r,"defaultMaxListeners",{enumerable:!0,get:function(){return C},set:function(e){if("number"!=typeof e||0>e||y(e))throw new RangeError("The value of \"defaultMaxListeners\" is out of range. It must be a non-negative number. Received "+e+".");C=e;}}),r.init=function(){(this._events===void 0||this._events===Object.getPrototypeOf(this)._events)&&(this._events=Object.create(null),this._eventsCount=0),this._maxListeners=this._maxListeners||void 0;},r.prototype.setMaxListeners=function(e){if("number"!=typeof e||0>e||y(e))throw new RangeError("The value of \"n\" is out of range. It must be a non-negative number. Received "+e+".");return this._maxListeners=e,this},r.prototype.getMaxListeners=function(){return o(this)},r.prototype.emit=function(e){for(var t=[],n=1;n<arguments.length;n++)t.push(arguments[n]);var r="error"===e,a=this._events;if(a!==void 0)r=r&&a.error===void 0;else if(!r)return !1;if(r){var o;if(0<t.length&&(o=t[0]),o instanceof Error)throw o;var d=new Error("Unhandled error."+(o?" ("+o.message+")":""));throw d.context=o,d}var s=a[e];if(s===void 0)return !1;if("function"==typeof s)b(s,this,t);else for(var l=s.length,c=u(s,l),n=0;n<l;++n)b(c[n],this,t);return !0},r.prototype.addListener=function(e,t){return i(this,e,t,!1)},r.prototype.on=r.prototype.addListener,r.prototype.prependListener=function(e,t){return i(this,e,t,!0)},r.prototype.once=function(e,t){return a(t),this.on(e,s(this,e,t)),this},r.prototype.prependOnceListener=function(e,t){return a(t),this.prependListener(e,s(this,e,t)),this},r.prototype.removeListener=function(e,t){var n,r,o,d,s;if(a(t),r=this._events,void 0===r)return this;if(n=r[e],void 0===n)return this;if(n===t||n.listener===t)0==--this._eventsCount?this._events=Object.create(null):(delete r[e],r.removeListener&&this.emit("removeListener",e,n.listener||t));else if("function"!=typeof n){for(o=-1,d=n.length-1;0<=d;d--)if(n[d]===t||n[d].listener===t){s=n[d].listener,o=d;break}if(0>o)return this;0===o?n.shift():p(n,o),1===n.length&&(r[e]=n[0]),void 0!==r.removeListener&&this.emit("removeListener",e,s||t);}return this},r.prototype.off=r.prototype.removeListener,r.prototype.removeAllListeners=function(e){var t,n,r;if(n=this._events,void 0===n)return this;if(void 0===n.removeListener)return 0===arguments.length?(this._events=Object.create(null),this._eventsCount=0):void 0!==n[e]&&(0==--this._eventsCount?this._events=Object.create(null):delete n[e]),this;if(0===arguments.length){var a,o=Object.keys(n);for(r=0;r<o.length;++r)a=o[r],"removeListener"!==a&&this.removeAllListeners(a);return this.removeAllListeners("removeListener"),this._events=Object.create(null),this._eventsCount=0,this}if(t=n[e],"function"==typeof t)this.removeListener(e,t);else if(void 0!==t)for(r=t.length-1;0<=r;r--)this.removeListener(e,t[r]);return this},r.prototype.listeners=function(e){return l(this,e,!0)},r.prototype.rawListeners=function(e){return l(this,e,!1)},r.listenerCount=function(e,t){return "function"==typeof e.listenerCount?e.listenerCount(t):c.call(e,t)},r.prototype.listenerCount=c,r.prototype.eventNames=function(){return 0<this._eventsCount?h(this._events):[]};},{}],8:[function(e,t){t.exports=function(){if("undefined"==typeof globalThis)return null;var e={RTCPeerConnection:globalThis.RTCPeerConnection||globalThis.mozRTCPeerConnection||globalThis.webkitRTCPeerConnection,RTCSessionDescription:globalThis.RTCSessionDescription||globalThis.mozRTCSessionDescription||globalThis.webkitRTCSessionDescription,RTCIceCandidate:globalThis.RTCIceCandidate||globalThis.mozRTCIceCandidate||globalThis.webkitRTCIceCandidate};return e.RTCPeerConnection?e:null};},{}],9:[function(e,a,o){/*! ieee754. BSD-3-Clause License. Feross Aboukhadijeh <https://feross.org/opensource> */o.read=function(t,n,a,o,l){var c,u,p=8*l-o-1,f=(1<<p)-1,g=f>>1,_=-7,h=a?l-1:0,b=a?-1:1,d=t[n+h];for(h+=b,c=d&(1<<-_)-1,d>>=-_,_+=p;0<_;c=256*c+t[n+h],h+=b,_-=8);for(u=c&(1<<-_)-1,c>>=-_,_+=o;0<_;u=256*u+t[n+h],h+=b,_-=8);if(0===c)c=1-g;else {if(c===f)return u?NaN:(d?-1:1)*(1/0);u+=r(2,o),c-=g;}return (d?-1:1)*u*r(2,c-o)},o.write=function(a,o,l,u,p,f){var h,b,y,g=Math.LN2,_=Math.log,C=8*f-p-1,R=(1<<C)-1,E=R>>1,w=23===p?r(2,-24)-r(2,-77):0,S=u?0:f-1,T=u?1:-1,d=0>o||0===o&&0>1/o?1:0;for(o=n(o),isNaN(o)||o===1/0?(b=isNaN(o)?1:0,h=R):(h=t(_(o)/g),1>o*(y=r(2,-h))&&(h--,y*=2),o+=1<=h+E?w/y:w*r(2,1-E),2<=o*y&&(h++,y/=2),h+E>=R?(b=0,h=R):1<=h+E?(b=(o*y-1)*r(2,p),h+=E):(b=o*r(2,E-1)*r(2,p),h=0));8<=p;a[l+S]=255&b,S+=T,b/=256,p-=8);for(h=h<<p|b,C+=p;0<C;a[l+S]=255&h,S+=T,h/=256,C-=8);a[l+S-T]|=128*d;};},{}],10:[function(e,t){t.exports="function"==typeof Object.create?function(e,t){t&&(e.super_=t,e.prototype=Object.create(t.prototype,{constructor:{value:e,enumerable:!1,writable:!0,configurable:!0}}));}:function(e,t){if(t){e.super_=t;var n=function(){};n.prototype=t.prototype,e.prototype=new n,e.prototype.constructor=e;}};},{}],11:[function(e,t){var r=Math.round;function a(e){if(e+="",!(100<e.length)){var t=/^(-?(?:\d+)?\.?\d+) *(milliseconds?|msecs?|ms|seconds?|secs?|s|minutes?|mins?|m|hours?|hrs?|h|days?|d|weeks?|w|years?|yrs?|y)?$/i.exec(e);if(t){var r=parseFloat(t[1]),n=(t[2]||"ms").toLowerCase();return "years"===n||"year"===n||"yrs"===n||"yr"===n||"y"===n?31557600000*r:"weeks"===n||"week"===n||"w"===n?604800000*r:"days"===n||"day"===n||"d"===n?86400000*r:"hours"===n||"hour"===n||"hrs"===n||"hr"===n||"h"===n?3600000*r:"minutes"===n||"minute"===n||"mins"===n||"min"===n||"m"===n?60000*r:"seconds"===n||"second"===n||"secs"===n||"sec"===n||"s"===n?1000*r:"milliseconds"===n||"millisecond"===n||"msecs"===n||"msec"===n||"ms"===n?r:void 0}}}function o(e){var t=n(e);return 86400000<=t?r(e/86400000)+"d":3600000<=t?r(e/3600000)+"h":60000<=t?r(e/60000)+"m":1000<=t?r(e/1000)+"s":e+"ms"}function i(e){var t=n(e);return 86400000<=t?s(e,t,86400000,"day"):3600000<=t?s(e,t,3600000,"hour"):60000<=t?s(e,t,60000,"minute"):1000<=t?s(e,t,1000,"second"):e+" ms"}function s(e,t,a,n){return r(e/a)+" "+n+(t>=1.5*a?"s":"")}t.exports=function(e,t){t=t||{};var n=typeof e;if("string"==n&&0<e.length)return a(e);if("number"===n&&isFinite(e))return t.long?i(e):o(e);throw new Error("val is not a non-empty string or a valid number. val="+JSON.stringify(e))};},{}],12:[function(e,t){function n(){throw new Error("setTimeout has not been defined")}function r(){throw new Error("clearTimeout has not been defined")}function a(t){if(c===setTimeout)return setTimeout(t,0);if((c===n||!c)&&setTimeout)return c=setTimeout,setTimeout(t,0);try{return c(t,0)}catch(n){try{return c.call(null,t,0)}catch(n){return c.call(this,t,0)}}}function o(t){if(u===clearTimeout)return clearTimeout(t);if((u===r||!u)&&clearTimeout)return u=clearTimeout,clearTimeout(t);try{return u(t)}catch(n){try{return u.call(null,t)}catch(n){return u.call(this,t)}}}function i(){_&&f&&(_=!1,f.length?g=f.concat(g):h=-1,g.length&&d());}function d(){if(!_){var e=a(i);_=!0;for(var t=g.length;t;){for(f=g,g=[];++h<t;)f&&f[h].run();h=-1,t=g.length;}f=null,_=!1,o(e);}}function s(e,t){this.fun=e,this.array=t;}function l(){}var c,u,p=t.exports={};(function(){try{c="function"==typeof setTimeout?setTimeout:n;}catch(t){c=n;}try{u="function"==typeof clearTimeout?clearTimeout:r;}catch(t){u=r;}})();var f,g=[],_=!1,h=-1;p.nextTick=function(e){var t=Array(arguments.length-1);if(1<arguments.length)for(var n=1;n<arguments.length;n++)t[n-1]=arguments[n];g.push(new s(e,t)),1!==g.length||_||a(d);},s.prototype.run=function(){this.fun.apply(null,this.array);},p.title="browser",p.browser=!0,p.env={},p.argv=[],p.version="",p.versions={},p.on=l,p.addListener=l,p.once=l,p.off=l,p.removeListener=l,p.removeAllListeners=l,p.emit=l,p.prependListener=l,p.prependOnceListener=l,p.listeners=function(){return []},p.binding=function(){throw new Error("process.binding is not supported")},p.cwd=function(){return "/"},p.chdir=function(){throw new Error("process.chdir is not supported")},p.umask=function(){return 0};},{}],13:[function(e,t){(function(e){(function(){/*! queue-microtask. MIT License. Feross Aboukhadijeh <https://feross.org/opensource> */let n;t.exports="function"==typeof queueMicrotask?queueMicrotask.bind("undefined"==typeof window?e:window):e=>(n||(n=Promise.resolve())).then(e).catch(e=>setTimeout(()=>{throw e},0));}).call(this);}).call(this,"undefined"==typeof commonjsGlobal?"undefined"==typeof self?"undefined"==typeof window?{}:window:self:commonjsGlobal);},{}],14:[function(e,t){(function(n,r){(function(){var a=e("safe-buffer").Buffer,o=r.crypto||r.msCrypto;t.exports=o&&o.getRandomValues?function(e,t){if(e>4294967295)throw new RangeError("requested too many random bytes");var r=a.allocUnsafe(e);if(0<e)if(65536<e)for(var i=0;i<e;i+=65536)o.getRandomValues(r.slice(i,i+65536));else o.getRandomValues(r);return "function"==typeof t?n.nextTick(function(){t(null,r);}):r}:function(){throw new Error("Secure random number generation is not supported by this browser.\nUse Chrome, Firefox or Internet Explorer 11")};}).call(this);}).call(this,e("_process"),"undefined"==typeof commonjsGlobal?"undefined"==typeof self?"undefined"==typeof window?{}:window:self:commonjsGlobal);},{_process:12,"safe-buffer":30}],15:[function(e,t){function n(e,t){e.prototype=Object.create(t.prototype),e.prototype.constructor=e,e.__proto__=t;}function r(e,t,r){function a(e,n,r){return "string"==typeof t?t:t(e,n,r)}r||(r=Error);var o=function(e){function t(t,n,r){return e.call(this,a(t,n,r))||this}return n(t,e),t}(r);o.prototype.name=r.name,o.prototype.code=e,s[e]=o;}function a(e,t){if(Array.isArray(e)){var n=e.length;return e=e.map(function(e){return e+""}),2<n?"one of ".concat(t," ").concat(e.slice(0,n-1).join(", "),", or ")+e[n-1]:2===n?"one of ".concat(t," ").concat(e[0]," or ").concat(e[1]):"of ".concat(t," ").concat(e[0])}return "of ".concat(t," ").concat(e+"")}function o(e,t,n){return e.substr(!n||0>n?0:+n,t.length)===t}function i(e,t,n){return (void 0===n||n>e.length)&&(n=e.length),e.substring(n-t.length,n)===t}function d(e,t,n){return "number"!=typeof n&&(n=0),!(n+t.length>e.length)&&-1!==e.indexOf(t,n)}var s={};r("ERR_INVALID_OPT_VALUE",function(e,t){return "The value \""+t+"\" is invalid for option \""+e+"\""},TypeError),r("ERR_INVALID_ARG_TYPE",function(e,t,n){var r;"string"==typeof t&&o(t,"not ")?(r="must not be",t=t.replace(/^not /,"")):r="must be";var s;if(i(e," argument"))s="The ".concat(e," ").concat(r," ").concat(a(t,"type"));else {var l=d(e,".")?"property":"argument";s="The \"".concat(e,"\" ").concat(l," ").concat(r," ").concat(a(t,"type"));}return s+=". Received type ".concat(typeof n),s},TypeError),r("ERR_STREAM_PUSH_AFTER_EOF","stream.push() after EOF"),r("ERR_METHOD_NOT_IMPLEMENTED",function(e){return "The "+e+" method is not implemented"}),r("ERR_STREAM_PREMATURE_CLOSE","Premature close"),r("ERR_STREAM_DESTROYED",function(e){return "Cannot call "+e+" after a stream was destroyed"}),r("ERR_MULTIPLE_CALLBACK","Callback called multiple times"),r("ERR_STREAM_CANNOT_PIPE","Cannot pipe, not readable"),r("ERR_STREAM_WRITE_AFTER_END","write after end"),r("ERR_STREAM_NULL_VALUES","May not write null values to stream",TypeError),r("ERR_UNKNOWN_ENCODING",function(e){return "Unknown encoding: "+e},TypeError),r("ERR_STREAM_UNSHIFT_AFTER_END_EVENT","stream.unshift() after end event"),t.exports.codes=s;},{}],16:[function(e,t){(function(n){(function(){function r(e){return this instanceof r?void(d.call(this,e),s.call(this,e),this.allowHalfOpen=!0,e&&(!1===e.readable&&(this.readable=!1),!1===e.writable&&(this.writable=!1),!1===e.allowHalfOpen&&(this.allowHalfOpen=!1,this.once("end",a)))):new r(e)}function a(){this._writableState.ended||n.nextTick(o,this);}function o(e){e.end();}var i=Object.keys||function(e){var t=[];for(var n in e)t.push(n);return t};t.exports=r;var d=e("./_stream_readable"),s=e("./_stream_writable");e("inherits")(r,d);for(var l,c=i(s.prototype),u=0;u<c.length;u++)l=c[u],r.prototype[l]||(r.prototype[l]=s.prototype[l]);Object.defineProperty(r.prototype,"writableHighWaterMark",{enumerable:!1,get:function(){return this._writableState.highWaterMark}}),Object.defineProperty(r.prototype,"writableBuffer",{enumerable:!1,get:function(){return this._writableState&&this._writableState.getBuffer()}}),Object.defineProperty(r.prototype,"writableLength",{enumerable:!1,get:function(){return this._writableState.length}}),Object.defineProperty(r.prototype,"destroyed",{enumerable:!1,get:function(){return void 0!==this._readableState&&void 0!==this._writableState&&this._readableState.destroyed&&this._writableState.destroyed},set:function(e){void 0===this._readableState||void 0===this._writableState||(this._readableState.destroyed=e,this._writableState.destroyed=e);}});}).call(this);}).call(this,e("_process"));},{"./_stream_readable":18,"./_stream_writable":20,_process:12,inherits:10}],17:[function(e,t){function n(e){return this instanceof n?void r.call(this,e):new n(e)}t.exports=n;var r=e("./_stream_transform");e("inherits")(n,r),n.prototype._transform=function(e,t,n){n(null,e);};},{"./_stream_transform":19,inherits:10}],18:[function(e,t){(function(n,r){(function(){function a(e){return P.from(e)}function o(e){return P.isBuffer(e)||e instanceof M}function i(e,t,n){return "function"==typeof e.prependListener?e.prependListener(t,n):void(e._events&&e._events[t]?Array.isArray(e._events[t])?e._events[t].unshift(n):e._events[t]=[n,e._events[t]]:e.on(t,n))}function d(t,n,r){A=A||e("./_stream_duplex"),t=t||{},"boolean"!=typeof r&&(r=n instanceof A),this.objectMode=!!t.objectMode,r&&(this.objectMode=this.objectMode||!!t.readableObjectMode),this.highWaterMark=H(this,t,"readableHighWaterMark",r),this.buffer=new j,this.length=0,this.pipes=null,this.pipesCount=0,this.flowing=null,this.ended=!1,this.endEmitted=!1,this.reading=!1,this.sync=!0,this.needReadable=!1,this.emittedReadable=!1,this.readableListening=!1,this.resumeScheduled=!1,this.paused=!0,this.emitClose=!1!==t.emitClose,this.autoDestroy=!!t.autoDestroy,this.destroyed=!1,this.defaultEncoding=t.defaultEncoding||"utf8",this.awaitDrain=0,this.readingMore=!1,this.decoder=null,this.encoding=null,t.encoding&&(!F&&(F=e("string_decoder/").StringDecoder),this.decoder=new F(t.encoding),this.encoding=t.encoding);}function s(t){if(A=A||e("./_stream_duplex"),!(this instanceof s))return new s(t);var n=this instanceof A;this._readableState=new d(t,this,n),this.readable=!0,t&&("function"==typeof t.read&&(this._read=t.read),"function"==typeof t.destroy&&(this._destroy=t.destroy)),I.call(this);}function l(e,t,n,r,o){x("readableAddChunk",t);var i=e._readableState;if(null===t)i.reading=!1,g(e,i);else {var d;if(o||(d=u(i,t)),d)X(e,d);else if(!(i.objectMode||t&&0<t.length))r||(i.reading=!1,m(e,i));else if("string"==typeof t||i.objectMode||Object.getPrototypeOf(t)===P.prototype||(t=a(t)),r)i.endEmitted?X(e,new K):c(e,i,t,!0);else if(i.ended)X(e,new z);else {if(i.destroyed)return !1;i.reading=!1,i.decoder&&!n?(t=i.decoder.write(t),i.objectMode||0!==t.length?c(e,i,t,!1):m(e,i)):c(e,i,t,!1);}}return !i.ended&&(i.length<i.highWaterMark||0===i.length)}function c(e,t,n,r){t.flowing&&0===t.length&&!t.sync?(t.awaitDrain=0,e.emit("data",n)):(t.length+=t.objectMode?1:n.length,r?t.buffer.unshift(n):t.buffer.push(n),t.needReadable&&_(e)),m(e,t);}function u(e,t){var n;return o(t)||"string"==typeof t||void 0===t||e.objectMode||(n=new V("chunk",["string","Buffer","Uint8Array"],t)),n}function p(e){return 1073741824<=e?e=1073741824:(e--,e|=e>>>1,e|=e>>>2,e|=e>>>4,e|=e>>>8,e|=e>>>16,e++),e}function f(e,t){return 0>=e||0===t.length&&t.ended?0:t.objectMode?1:e===e?(e>t.highWaterMark&&(t.highWaterMark=p(e)),e<=t.length?e:t.ended?t.length:(t.needReadable=!0,0)):t.flowing&&t.length?t.buffer.head.data.length:t.length}function g(e,t){if(x("onEofChunk"),!t.ended){if(t.decoder){var n=t.decoder.end();n&&n.length&&(t.buffer.push(n),t.length+=t.objectMode?1:n.length);}t.ended=!0,t.sync?_(e):(t.needReadable=!1,!t.emittedReadable&&(t.emittedReadable=!0,h(e)));}}function _(e){var t=e._readableState;x("emitReadable",t.needReadable,t.emittedReadable),t.needReadable=!1,t.emittedReadable||(x("emitReadable",t.flowing),t.emittedReadable=!0,n.nextTick(h,e));}function h(e){var t=e._readableState;x("emitReadable_",t.destroyed,t.length,t.ended),!t.destroyed&&(t.length||t.ended)&&(e.emit("readable"),t.emittedReadable=!1),t.needReadable=!t.flowing&&!t.ended&&t.length<=t.highWaterMark,S(e);}function m(e,t){t.readingMore||(t.readingMore=!0,n.nextTick(b,e,t));}function b(e,t){for(;!t.reading&&!t.ended&&(t.length<t.highWaterMark||t.flowing&&0===t.length);){var n=t.length;if(x("maybeReadMore read 0"),e.read(0),n===t.length)break}t.readingMore=!1;}function y(e){return function(){var t=e._readableState;x("pipeOnDrain",t.awaitDrain),t.awaitDrain&&t.awaitDrain--,0===t.awaitDrain&&D(e,"data")&&(t.flowing=!0,S(e));}}function C(e){var t=e._readableState;t.readableListening=0<e.listenerCount("readable"),t.resumeScheduled&&!t.paused?t.flowing=!0:0<e.listenerCount("data")&&e.resume();}function R(e){x("readable nexttick read 0"),e.read(0);}function E(e,t){t.resumeScheduled||(t.resumeScheduled=!0,n.nextTick(w,e,t));}function w(e,t){x("resume",t.reading),t.reading||e.read(0),t.resumeScheduled=!1,e.emit("resume"),S(e),t.flowing&&!t.reading&&e.read(0);}function S(e){var t=e._readableState;for(x("flow",t.flowing);t.flowing&&null!==e.read(););}function T(e,t){if(0===t.length)return null;var n;return t.objectMode?n=t.buffer.shift():!e||e>=t.length?(n=t.decoder?t.buffer.join(""):1===t.buffer.length?t.buffer.first():t.buffer.concat(t.length),t.buffer.clear()):n=t.buffer.consume(e,t.decoder),n}function v(e){var t=e._readableState;x("endReadable",t.endEmitted),t.endEmitted||(t.ended=!0,n.nextTick(k,t,e));}function k(e,t){if(x("endReadableNT",e.endEmitted,e.length),!e.endEmitted&&0===e.length&&(e.endEmitted=!0,t.readable=!1,t.emit("end"),e.autoDestroy)){var n=t._writableState;(!n||n.autoDestroy&&n.finished)&&t.destroy();}}function L(e,t){for(var n=0,r=e.length;n<r;n++)if(e[n]===t)return n;return -1}t.exports=s;var A;s.ReadableState=d;var x;e("events").EventEmitter;var D=function(e,t){return e.listeners(t).length},I=e("./internal/streams/stream"),P=e("buffer").Buffer,M=r.Uint8Array||function(){},O=e("util");x=O&&O.debuglog?O.debuglog("stream"):function(){};var F,B,U,j=e("./internal/streams/buffer_list"),q=e("./internal/streams/destroy"),W=e("./internal/streams/state"),H=W.getHighWaterMark,Y=e("../errors").codes,V=Y.ERR_INVALID_ARG_TYPE,z=Y.ERR_STREAM_PUSH_AFTER_EOF,G=Y.ERR_METHOD_NOT_IMPLEMENTED,K=Y.ERR_STREAM_UNSHIFT_AFTER_END_EVENT;e("inherits")(s,I);var X=q.errorOrDestroy,$=["error","close","destroy","pause","resume"];Object.defineProperty(s.prototype,"destroyed",{enumerable:!1,get:function(){return void 0!==this._readableState&&this._readableState.destroyed},set:function(e){this._readableState&&(this._readableState.destroyed=e);}}),s.prototype.destroy=q.destroy,s.prototype._undestroy=q.undestroy,s.prototype._destroy=function(e,t){t(e);},s.prototype.push=function(e,t){var n,r=this._readableState;return r.objectMode?n=!0:"string"==typeof e&&(t=t||r.defaultEncoding,t!==r.encoding&&(e=P.from(e,t),t=""),n=!0),l(this,e,t,!1,n)},s.prototype.unshift=function(e){return l(this,e,null,!0,!1)},s.prototype.isPaused=function(){return !1===this._readableState.flowing},s.prototype.setEncoding=function(t){F||(F=e("string_decoder/").StringDecoder);var n=new F(t);this._readableState.decoder=n,this._readableState.encoding=this._readableState.decoder.encoding;for(var r=this._readableState.buffer.head,a="";null!==r;)a+=n.write(r.data),r=r.next;return this._readableState.buffer.clear(),""!==a&&this._readableState.buffer.push(a),this._readableState.length=a.length,this};s.prototype.read=function(e){x("read",e),e=parseInt(e,10);var t=this._readableState,r=e;if(0!==e&&(t.emittedReadable=!1),0===e&&t.needReadable&&((0===t.highWaterMark?0<t.length:t.length>=t.highWaterMark)||t.ended))return x("read: emitReadable",t.length,t.ended),0===t.length&&t.ended?v(this):_(this),null;if(e=f(e,t),0===e&&t.ended)return 0===t.length&&v(this),null;var a=t.needReadable;x("need readable",a),(0===t.length||t.length-e<t.highWaterMark)&&(a=!0,x("length less than watermark",a)),t.ended||t.reading?(a=!1,x("reading or ended",a)):a&&(x("do read"),t.reading=!0,t.sync=!0,0===t.length&&(t.needReadable=!0),this._read(t.highWaterMark),t.sync=!1,!t.reading&&(e=f(r,t)));var o;return o=0<e?T(e,t):null,null===o?(t.needReadable=t.length<=t.highWaterMark,e=0):(t.length-=e,t.awaitDrain=0),0===t.length&&(!t.ended&&(t.needReadable=!0),r!==e&&t.ended&&v(this)),null!==o&&this.emit("data",o),o},s.prototype._read=function(){X(this,new G("_read()"));},s.prototype.pipe=function(e,t){function r(e,t){x("onunpipe"),e===p&&t&&!1===t.hasUnpiped&&(t.hasUnpiped=!0,o());}function a(){x("onend"),e.end();}function o(){x("cleanup"),e.removeListener("close",l),e.removeListener("finish",c),e.removeListener("drain",h),e.removeListener("error",s),e.removeListener("unpipe",r),p.removeListener("end",a),p.removeListener("end",u),p.removeListener("data",d),m=!0,f.awaitDrain&&(!e._writableState||e._writableState.needDrain)&&h();}function d(t){x("ondata");var n=e.write(t);x("dest.write",n),!1===n&&((1===f.pipesCount&&f.pipes===e||1<f.pipesCount&&-1!==L(f.pipes,e))&&!m&&(x("false write response, pause",f.awaitDrain),f.awaitDrain++),p.pause());}function s(t){x("onerror",t),u(),e.removeListener("error",s),0===D(e,"error")&&X(e,t);}function l(){e.removeListener("finish",c),u();}function c(){x("onfinish"),e.removeListener("close",l),u();}function u(){x("unpipe"),p.unpipe(e);}var p=this,f=this._readableState;switch(f.pipesCount){case 0:f.pipes=e;break;case 1:f.pipes=[f.pipes,e];break;default:f.pipes.push(e);}f.pipesCount+=1,x("pipe count=%d opts=%j",f.pipesCount,t);var g=(!t||!1!==t.end)&&e!==n.stdout&&e!==n.stderr,_=g?a:u;f.endEmitted?n.nextTick(_):p.once("end",_),e.on("unpipe",r);var h=y(p);e.on("drain",h);var m=!1;return p.on("data",d),i(e,"error",s),e.once("close",l),e.once("finish",c),e.emit("pipe",p),f.flowing||(x("pipe resume"),p.resume()),e},s.prototype.unpipe=function(e){var t=this._readableState,n={hasUnpiped:!1};if(0===t.pipesCount)return this;if(1===t.pipesCount)return e&&e!==t.pipes?this:(e||(e=t.pipes),t.pipes=null,t.pipesCount=0,t.flowing=!1,e&&e.emit("unpipe",this,n),this);if(!e){var r=t.pipes,a=t.pipesCount;t.pipes=null,t.pipesCount=0,t.flowing=!1;for(var o=0;o<a;o++)r[o].emit("unpipe",this,{hasUnpiped:!1});return this}var d=L(t.pipes,e);return -1===d?this:(t.pipes.splice(d,1),t.pipesCount-=1,1===t.pipesCount&&(t.pipes=t.pipes[0]),e.emit("unpipe",this,n),this)},s.prototype.on=function(e,t){var r=I.prototype.on.call(this,e,t),a=this._readableState;return "data"===e?(a.readableListening=0<this.listenerCount("readable"),!1!==a.flowing&&this.resume()):"readable"==e&&!a.endEmitted&&!a.readableListening&&(a.readableListening=a.needReadable=!0,a.flowing=!1,a.emittedReadable=!1,x("on readable",a.length,a.reading),a.length?_(this):!a.reading&&n.nextTick(R,this)),r},s.prototype.addListener=s.prototype.on,s.prototype.removeListener=function(e,t){var r=I.prototype.removeListener.call(this,e,t);return "readable"===e&&n.nextTick(C,this),r},s.prototype.removeAllListeners=function(e){var t=I.prototype.removeAllListeners.apply(this,arguments);return ("readable"===e||void 0===e)&&n.nextTick(C,this),t},s.prototype.resume=function(){var e=this._readableState;return e.flowing||(x("resume"),e.flowing=!e.readableListening,E(this,e)),e.paused=!1,this},s.prototype.pause=function(){return x("call pause flowing=%j",this._readableState.flowing),!1!==this._readableState.flowing&&(x("pause"),this._readableState.flowing=!1,this.emit("pause")),this._readableState.paused=!0,this},s.prototype.wrap=function(e){var t=this,r=this._readableState,a=!1;for(var o in e.on("end",function(){if(x("wrapped end"),r.decoder&&!r.ended){var e=r.decoder.end();e&&e.length&&t.push(e);}t.push(null);}),e.on("data",function(n){if((x("wrapped data"),r.decoder&&(n=r.decoder.write(n)),!(r.objectMode&&(null===n||void 0===n)))&&(r.objectMode||n&&n.length)){var o=t.push(n);o||(a=!0,e.pause());}}),e)void 0===this[o]&&"function"==typeof e[o]&&(this[o]=function(t){return function(){return e[t].apply(e,arguments)}}(o));for(var i=0;i<$.length;i++)e.on($[i],this.emit.bind(this,$[i]));return this._read=function(t){x("wrapped _read",t),a&&(a=!1,e.resume());},this},"function"==typeof Symbol&&(s.prototype[Symbol.asyncIterator]=function(){return void 0===B&&(B=e("./internal/streams/async_iterator")),B(this)}),Object.defineProperty(s.prototype,"readableHighWaterMark",{enumerable:!1,get:function(){return this._readableState.highWaterMark}}),Object.defineProperty(s.prototype,"readableBuffer",{enumerable:!1,get:function(){return this._readableState&&this._readableState.buffer}}),Object.defineProperty(s.prototype,"readableFlowing",{enumerable:!1,get:function(){return this._readableState.flowing},set:function(e){this._readableState&&(this._readableState.flowing=e);}}),s._fromList=T,Object.defineProperty(s.prototype,"readableLength",{enumerable:!1,get:function(){return this._readableState.length}}),"function"==typeof Symbol&&(s.from=function(t,n){return void 0===U&&(U=e("./internal/streams/from")),U(s,t,n)});}).call(this);}).call(this,e("_process"),"undefined"==typeof commonjsGlobal?"undefined"==typeof self?"undefined"==typeof window?{}:window:self:commonjsGlobal);},{"../errors":15,"./_stream_duplex":16,"./internal/streams/async_iterator":21,"./internal/streams/buffer_list":22,"./internal/streams/destroy":23,"./internal/streams/from":25,"./internal/streams/state":27,"./internal/streams/stream":28,_process:12,buffer:3,events:7,inherits:10,"string_decoder/":31,util:2}],19:[function(e,t){function n(e,t){var n=this._transformState;n.transforming=!1;var r=n.writecb;if(null===r)return this.emit("error",new s);n.writechunk=null,n.writecb=null,null!=t&&this.push(t),r(e);var a=this._readableState;a.reading=!1,(a.needReadable||a.length<a.highWaterMark)&&this._read(a.highWaterMark);}function r(e){return this instanceof r?void(u.call(this,e),this._transformState={afterTransform:n.bind(this),needTransform:!1,transforming:!1,writecb:null,writechunk:null,writeencoding:null},this._readableState.needReadable=!0,this._readableState.sync=!1,e&&("function"==typeof e.transform&&(this._transform=e.transform),"function"==typeof e.flush&&(this._flush=e.flush)),this.on("prefinish",a)):new r(e)}function a(){var e=this;"function"!=typeof this._flush||this._readableState.destroyed?o(this,null,null):this._flush(function(t,n){o(e,t,n);});}function o(e,t,n){if(t)return e.emit("error",t);if(null!=n&&e.push(n),e._writableState.length)throw new c;if(e._transformState.transforming)throw new l;return e.push(null)}t.exports=r;var i=e("../errors").codes,d=i.ERR_METHOD_NOT_IMPLEMENTED,s=i.ERR_MULTIPLE_CALLBACK,l=i.ERR_TRANSFORM_ALREADY_TRANSFORMING,c=i.ERR_TRANSFORM_WITH_LENGTH_0,u=e("./_stream_duplex");e("inherits")(r,u),r.prototype.push=function(e,t){return this._transformState.needTransform=!1,u.prototype.push.call(this,e,t)},r.prototype._transform=function(e,t,n){n(new d("_transform()"));},r.prototype._write=function(e,t,n){var r=this._transformState;if(r.writecb=n,r.writechunk=e,r.writeencoding=t,!r.transforming){var a=this._readableState;(r.needTransform||a.needReadable||a.length<a.highWaterMark)&&this._read(a.highWaterMark);}},r.prototype._read=function(){var e=this._transformState;null===e.writechunk||e.transforming?e.needTransform=!0:(e.transforming=!0,this._transform(e.writechunk,e.writeencoding,e.afterTransform));},r.prototype._destroy=function(e,t){u.prototype._destroy.call(this,e,function(e){t(e);});};},{"../errors":15,"./_stream_duplex":16,inherits:10}],20:[function(e,t){(function(n,r){(function(){function a(e){var t=this;this.next=null,this.entry=null,this.finish=function(){v(t,e);};}function o(e){return x.from(e)}function i(e){return x.isBuffer(e)||e instanceof N}function d(){}function s(t,n,r){k=k||e("./_stream_duplex"),t=t||{},"boolean"!=typeof r&&(r=n instanceof k),this.objectMode=!!t.objectMode,r&&(this.objectMode=this.objectMode||!!t.writableObjectMode),this.highWaterMark=P(this,t,"writableHighWaterMark",r),this.finalCalled=!1,this.needDrain=!1,this.ending=!1,this.ended=!1,this.finished=!1,this.destroyed=!1;var o=!1===t.decodeStrings;this.decodeStrings=!o,this.defaultEncoding=t.defaultEncoding||"utf8",this.length=0,this.writing=!1,this.corked=0,this.sync=!0,this.bufferProcessing=!1,this.onwrite=function(e){m(n,e);},this.writecb=null,this.writelen=0,this.bufferedRequest=null,this.lastBufferedRequest=null,this.pendingcb=0,this.prefinished=!1,this.errorEmitted=!1,this.emitClose=!1!==t.emitClose,this.autoDestroy=!!t.autoDestroy,this.bufferedRequestCount=0,this.corkedRequestsFree=new a(this);}function l(t){k=k||e("./_stream_duplex");var n=this instanceof k;return n||V.call(l,this)?void(this._writableState=new s(t,this,n),this.writable=!0,t&&("function"==typeof t.write&&(this._write=t.write),"function"==typeof t.writev&&(this._writev=t.writev),"function"==typeof t.destroy&&(this._destroy=t.destroy),"function"==typeof t.final&&(this._final=t.final)),A.call(this)):new l(t)}function c(e,t){var r=new W;Y(e,r),n.nextTick(t,r);}function u(e,t,r,a){var o;return null===r?o=new q:"string"!=typeof r&&!t.objectMode&&(o=new O("chunk",["string","Buffer"],r)),!o||(Y(e,o),n.nextTick(a,o),!1)}function p(e,t,n){return e.objectMode||!1===e.decodeStrings||"string"!=typeof t||(t=x.from(t,n)),t}function f(e,t,n,r,a,o){if(!n){var i=p(t,r,a);r!==i&&(n=!0,a="buffer",r=i);}var d=t.objectMode?1:r.length;t.length+=d;var s=t.length<t.highWaterMark;if(s||(t.needDrain=!0),t.writing||t.corked){var l=t.lastBufferedRequest;t.lastBufferedRequest={chunk:r,encoding:a,isBuf:n,callback:o,next:null},l?l.next=t.lastBufferedRequest:t.bufferedRequest=t.lastBufferedRequest,t.bufferedRequestCount+=1;}else g(e,t,!1,d,r,a,o);return s}function g(e,t,n,r,a,o,i){t.writelen=r,t.writecb=i,t.writing=!0,t.sync=!0,t.destroyed?t.onwrite(new j("write")):n?e._writev(a,t.onwrite):e._write(a,o,t.onwrite),t.sync=!1;}function _(e,t,r,a,o){--t.pendingcb,r?(n.nextTick(o,a),n.nextTick(S,e,t),e._writableState.errorEmitted=!0,Y(e,a)):(o(a),e._writableState.errorEmitted=!0,Y(e,a),S(e,t));}function h(e){e.writing=!1,e.writecb=null,e.length-=e.writelen,e.writelen=0;}function m(e,t){var r=e._writableState,a=r.sync,o=r.writecb;if("function"!=typeof o)throw new B;if(h(r),t)_(e,r,a,t,o);else {var i=R(r)||e.destroyed;i||r.corked||r.bufferProcessing||!r.bufferedRequest||C(e,r),a?n.nextTick(b,e,r,i,o):b(e,r,i,o);}}function b(e,t,n,r){n||y(e,t),t.pendingcb--,r(),S(e,t);}function y(e,t){0===t.length&&t.needDrain&&(t.needDrain=!1,e.emit("drain"));}function C(e,t){t.bufferProcessing=!0;var n=t.bufferedRequest;if(e._writev&&n&&n.next){var r=t.bufferedRequestCount,o=Array(r),i=t.corkedRequestsFree;i.entry=n;for(var d=0,s=!0;n;)o[d]=n,n.isBuf||(s=!1),n=n.next,d+=1;o.allBuffers=s,g(e,t,!0,t.length,o,"",i.finish),t.pendingcb++,t.lastBufferedRequest=null,i.next?(t.corkedRequestsFree=i.next,i.next=null):t.corkedRequestsFree=new a(t),t.bufferedRequestCount=0;}else {for(;n;){var l=n.chunk,c=n.encoding,u=n.callback,p=t.objectMode?1:l.length;if(g(e,t,!1,p,l,c,u),n=n.next,t.bufferedRequestCount--,t.writing)break}null===n&&(t.lastBufferedRequest=null);}t.bufferedRequest=n,t.bufferProcessing=!1;}function R(e){return e.ending&&0===e.length&&null===e.bufferedRequest&&!e.finished&&!e.writing}function E(e,t){e._final(function(n){t.pendingcb--,n&&Y(e,n),t.prefinished=!0,e.emit("prefinish"),S(e,t);});}function w(e,t){t.prefinished||t.finalCalled||("function"!=typeof e._final||t.destroyed?(t.prefinished=!0,e.emit("prefinish")):(t.pendingcb++,t.finalCalled=!0,n.nextTick(E,e,t)));}function S(e,t){var n=R(t);if(n&&(w(e,t),0===t.pendingcb&&(t.finished=!0,e.emit("finish"),t.autoDestroy))){var r=e._readableState;(!r||r.autoDestroy&&r.endEmitted)&&e.destroy();}return n}function T(e,t,r){t.ending=!0,S(e,t),r&&(t.finished?n.nextTick(r):e.once("finish",r)),t.ended=!0,e.writable=!1;}function v(e,t,n){var r=e.entry;for(e.entry=null;r;){var a=r.callback;t.pendingcb--,a(n),r=r.next;}t.corkedRequestsFree.next=e;}t.exports=l;var k;l.WritableState=s;var L={deprecate:e("util-deprecate")},A=e("./internal/streams/stream"),x=e("buffer").Buffer,N=r.Uint8Array||function(){},D=e("./internal/streams/destroy"),I=e("./internal/streams/state"),P=I.getHighWaterMark,M=e("../errors").codes,O=M.ERR_INVALID_ARG_TYPE,F=M.ERR_METHOD_NOT_IMPLEMENTED,B=M.ERR_MULTIPLE_CALLBACK,U=M.ERR_STREAM_CANNOT_PIPE,j=M.ERR_STREAM_DESTROYED,q=M.ERR_STREAM_NULL_VALUES,W=M.ERR_STREAM_WRITE_AFTER_END,H=M.ERR_UNKNOWN_ENCODING,Y=D.errorOrDestroy;e("inherits")(l,A),s.prototype.getBuffer=function(){for(var e=this.bufferedRequest,t=[];e;)t.push(e),e=e.next;return t},function(){try{Object.defineProperty(s.prototype,"buffer",{get:L.deprecate(function(){return this.getBuffer()},"_writableState.buffer is deprecated. Use _writableState.getBuffer instead.","DEP0003")});}catch(e){}}();var V;"function"==typeof Symbol&&Symbol.hasInstance&&"function"==typeof Function.prototype[Symbol.hasInstance]?(V=Function.prototype[Symbol.hasInstance],Object.defineProperty(l,Symbol.hasInstance,{value:function(e){return !!V.call(this,e)||!(this!==l)&&e&&e._writableState instanceof s}})):V=function(e){return e instanceof this},l.prototype.pipe=function(){Y(this,new U);},l.prototype.write=function(e,t,n){var r=this._writableState,a=!1,s=!r.objectMode&&i(e);return s&&!x.isBuffer(e)&&(e=o(e)),"function"==typeof t&&(n=t,t=null),s?t="buffer":!t&&(t=r.defaultEncoding),"function"!=typeof n&&(n=d),r.ending?c(this,n):(s||u(this,r,e,n))&&(r.pendingcb++,a=f(this,r,s,e,t,n)),a},l.prototype.cork=function(){this._writableState.corked++;},l.prototype.uncork=function(){var e=this._writableState;e.corked&&(e.corked--,!e.writing&&!e.corked&&!e.bufferProcessing&&e.bufferedRequest&&C(this,e));},l.prototype.setDefaultEncoding=function(e){if("string"==typeof e&&(e=e.toLowerCase()),!(-1<["hex","utf8","utf-8","ascii","binary","base64","ucs2","ucs-2","utf16le","utf-16le","raw"].indexOf((e+"").toLowerCase())))throw new H(e);return this._writableState.defaultEncoding=e,this},Object.defineProperty(l.prototype,"writableBuffer",{enumerable:!1,get:function(){return this._writableState&&this._writableState.getBuffer()}}),Object.defineProperty(l.prototype,"writableHighWaterMark",{enumerable:!1,get:function(){return this._writableState.highWaterMark}}),l.prototype._write=function(e,t,n){n(new F("_write()"));},l.prototype._writev=null,l.prototype.end=function(e,t,n){var r=this._writableState;return "function"==typeof e?(n=e,e=null,t=null):"function"==typeof t&&(n=t,t=null),null!==e&&void 0!==e&&this.write(e,t),r.corked&&(r.corked=1,this.uncork()),r.ending||T(this,r,n),this},Object.defineProperty(l.prototype,"writableLength",{enumerable:!1,get:function(){return this._writableState.length}}),Object.defineProperty(l.prototype,"destroyed",{enumerable:!1,get:function(){return void 0!==this._writableState&&this._writableState.destroyed},set:function(e){this._writableState&&(this._writableState.destroyed=e);}}),l.prototype.destroy=D.destroy,l.prototype._undestroy=D.undestroy,l.prototype._destroy=function(e,t){t(e);};}).call(this);}).call(this,e("_process"),"undefined"==typeof commonjsGlobal?"undefined"==typeof self?"undefined"==typeof window?{}:window:self:commonjsGlobal);},{"../errors":15,"./_stream_duplex":16,"./internal/streams/destroy":23,"./internal/streams/state":27,"./internal/streams/stream":28,_process:12,buffer:3,inherits:10,"util-deprecate":32}],21:[function(e,t){(function(n){(function(){function r(e,t,n){return t in e?Object.defineProperty(e,t,{value:n,enumerable:!0,configurable:!0,writable:!0}):e[t]=n,e}function a(e,t){return {value:e,done:t}}function o(e){var t=e[c];if(null!==t){var n=e[h].read();null!==n&&(e[g]=null,e[c]=null,e[u]=null,t(a(n,!1)));}}function i(e){n.nextTick(o,e);}function d(e,t){return function(n,r){e.then(function(){return t[f]?void n(a(void 0,!0)):void t[_](n,r)},r);}}var s,l=e("./end-of-stream"),c=Symbol("lastResolve"),u=Symbol("lastReject"),p=Symbol("error"),f=Symbol("ended"),g=Symbol("lastPromise"),_=Symbol("handlePromise"),h=Symbol("stream"),m=Object.getPrototypeOf(function(){}),b=Object.setPrototypeOf((s={get stream(){return this[h]},next:function(){var e=this,t=this[p];if(null!==t)return Promise.reject(t);if(this[f])return Promise.resolve(a(void 0,!0));if(this[h].destroyed)return new Promise(function(t,r){n.nextTick(function(){e[p]?r(e[p]):t(a(void 0,!0));});});var r,o=this[g];if(o)r=new Promise(d(o,this));else {var i=this[h].read();if(null!==i)return Promise.resolve(a(i,!1));r=new Promise(this[_]);}return this[g]=r,r}},r(s,Symbol.asyncIterator,function(){return this}),r(s,"return",function(){var e=this;return new Promise(function(t,n){e[h].destroy(null,function(e){return e?void n(e):void t(a(void 0,!0))});})}),s),m);t.exports=function(e){var t,n=Object.create(b,(t={},r(t,h,{value:e,writable:!0}),r(t,c,{value:null,writable:!0}),r(t,u,{value:null,writable:!0}),r(t,p,{value:null,writable:!0}),r(t,f,{value:e._readableState.endEmitted,writable:!0}),r(t,_,{value:function(e,t){var r=n[h].read();r?(n[g]=null,n[c]=null,n[u]=null,e(a(r,!1))):(n[c]=e,n[u]=t);},writable:!0}),t));return n[g]=null,l(e,function(e){if(e&&"ERR_STREAM_PREMATURE_CLOSE"!==e.code){var t=n[u];return null!==t&&(n[g]=null,n[c]=null,n[u]=null,t(e)),void(n[p]=e)}var r=n[c];null!==r&&(n[g]=null,n[c]=null,n[u]=null,r(a(void 0,!0))),n[f]=!0;}),e.on("readable",i.bind(null,n)),n};}).call(this);}).call(this,e("_process"));},{"./end-of-stream":24,_process:12}],22:[function(e,t){function n(e,t){var n=Object.keys(e);if(Object.getOwnPropertySymbols){var r=Object.getOwnPropertySymbols(e);t&&(r=r.filter(function(t){return Object.getOwnPropertyDescriptor(e,t).enumerable})),n.push.apply(n,r);}return n}function r(e){for(var t,r=1;r<arguments.length;r++)t=null==arguments[r]?{}:arguments[r],r%2?n(Object(t),!0).forEach(function(n){a(e,n,t[n]);}):Object.getOwnPropertyDescriptors?Object.defineProperties(e,Object.getOwnPropertyDescriptors(t)):n(Object(t)).forEach(function(n){Object.defineProperty(e,n,Object.getOwnPropertyDescriptor(t,n));});return e}function a(e,t,n){return t in e?Object.defineProperty(e,t,{value:n,enumerable:!0,configurable:!0,writable:!0}):e[t]=n,e}function o(e,t){if(!(e instanceof t))throw new TypeError("Cannot call a class as a function")}function i(e,t){for(var n,r=0;r<t.length;r++)n=t[r],n.enumerable=n.enumerable||!1,n.configurable=!0,"value"in n&&(n.writable=!0),Object.defineProperty(e,n.key,n);}function d(e,t,n){return t&&i(e.prototype,t),n&&i(e,n),e}function s(e,t,n){u.prototype.copy.call(e,t,n);}var l=e("buffer"),u=l.Buffer,p=e("util"),f=p.inspect,g=f&&f.custom||"inspect";t.exports=function(){function e(){o(this,e),this.head=null,this.tail=null,this.length=0;}return d(e,[{key:"push",value:function(e){var t={data:e,next:null};0<this.length?this.tail.next=t:this.head=t,this.tail=t,++this.length;}},{key:"unshift",value:function(e){var t={data:e,next:this.head};0===this.length&&(this.tail=t),this.head=t,++this.length;}},{key:"shift",value:function(){if(0!==this.length){var e=this.head.data;return this.head=1===this.length?this.tail=null:this.head.next,--this.length,e}}},{key:"clear",value:function(){this.head=this.tail=null,this.length=0;}},{key:"join",value:function(e){if(0===this.length)return "";for(var t=this.head,n=""+t.data;t=t.next;)n+=e+t.data;return n}},{key:"concat",value:function(e){if(0===this.length)return u.alloc(0);for(var t=u.allocUnsafe(e>>>0),n=this.head,r=0;n;)s(n.data,t,r),r+=n.data.length,n=n.next;return t}},{key:"consume",value:function(e,t){var n;return e<this.head.data.length?(n=this.head.data.slice(0,e),this.head.data=this.head.data.slice(e)):e===this.head.data.length?n=this.shift():n=t?this._getString(e):this._getBuffer(e),n}},{key:"first",value:function(){return this.head.data}},{key:"_getString",value:function(e){var t=this.head,r=1,a=t.data;for(e-=a.length;t=t.next;){var o=t.data,i=e>o.length?o.length:e;if(a+=i===o.length?o:o.slice(0,e),e-=i,0===e){i===o.length?(++r,this.head=t.next?t.next:this.tail=null):(this.head=t,t.data=o.slice(i));break}++r;}return this.length-=r,a}},{key:"_getBuffer",value:function(e){var t=u.allocUnsafe(e),r=this.head,a=1;for(r.data.copy(t),e-=r.data.length;r=r.next;){var o=r.data,i=e>o.length?o.length:e;if(o.copy(t,t.length-e,0,i),e-=i,0===e){i===o.length?(++a,this.head=r.next?r.next:this.tail=null):(this.head=r,r.data=o.slice(i));break}++a;}return this.length-=a,t}},{key:g,value:function(e,t){return f(this,r({},t,{depth:0,customInspect:!1}))}}]),e}();},{buffer:3,util:2}],23:[function(e,t){(function(e){(function(){function n(e,t){a(e,t),r(e);}function r(e){e._writableState&&!e._writableState.emitClose||e._readableState&&!e._readableState.emitClose||e.emit("close");}function a(e,t){e.emit("error",t);}t.exports={destroy:function(t,o){var i=this,d=this._readableState&&this._readableState.destroyed,s=this._writableState&&this._writableState.destroyed;return d||s?(o?o(t):t&&(this._writableState?!this._writableState.errorEmitted&&(this._writableState.errorEmitted=!0,e.nextTick(a,this,t)):e.nextTick(a,this,t)),this):(this._readableState&&(this._readableState.destroyed=!0),this._writableState&&(this._writableState.destroyed=!0),this._destroy(t||null,function(t){!o&&t?i._writableState?i._writableState.errorEmitted?e.nextTick(r,i):(i._writableState.errorEmitted=!0,e.nextTick(n,i,t)):e.nextTick(n,i,t):o?(e.nextTick(r,i),o(t)):e.nextTick(r,i);}),this)},undestroy:function(){this._readableState&&(this._readableState.destroyed=!1,this._readableState.reading=!1,this._readableState.ended=!1,this._readableState.endEmitted=!1),this._writableState&&(this._writableState.destroyed=!1,this._writableState.ended=!1,this._writableState.ending=!1,this._writableState.finalCalled=!1,this._writableState.prefinished=!1,this._writableState.finished=!1,this._writableState.errorEmitted=!1);},errorOrDestroy:function(e,t){var n=e._readableState,r=e._writableState;n&&n.autoDestroy||r&&r.autoDestroy?e.destroy(t):e.emit("error",t);}};}).call(this);}).call(this,e("_process"));},{_process:12}],24:[function(e,t){function n(e){var t=!1;return function(){if(!t){t=!0;for(var n=arguments.length,r=Array(n),a=0;a<n;a++)r[a]=arguments[a];e.apply(this,r);}}}function r(){}function a(e){return e.setHeader&&"function"==typeof e.abort}function o(e,t,d){if("function"==typeof t)return o(e,null,t);t||(t={}),d=n(d||r);var s=t.readable||!1!==t.readable&&e.readable,l=t.writable||!1!==t.writable&&e.writable,c=function(){e.writable||p();},u=e._writableState&&e._writableState.finished,p=function(){l=!1,u=!0,s||d.call(e);},f=e._readableState&&e._readableState.endEmitted,g=function(){s=!1,f=!0,l||d.call(e);},_=function(t){d.call(e,t);},h=function(){var t;return s&&!f?(e._readableState&&e._readableState.ended||(t=new i),d.call(e,t)):l&&!u?(e._writableState&&e._writableState.ended||(t=new i),d.call(e,t)):void 0},m=function(){e.req.on("finish",p);};return a(e)?(e.on("complete",p),e.on("abort",h),e.req?m():e.on("request",m)):l&&!e._writableState&&(e.on("end",c),e.on("close",c)),e.on("end",g),e.on("finish",p),!1!==t.error&&e.on("error",_),e.on("close",h),function(){e.removeListener("complete",p),e.removeListener("abort",h),e.removeListener("request",m),e.req&&e.req.removeListener("finish",p),e.removeListener("end",c),e.removeListener("close",c),e.removeListener("finish",p),e.removeListener("end",g),e.removeListener("error",_),e.removeListener("close",h);}}var i=e("../../../errors").codes.ERR_STREAM_PREMATURE_CLOSE;t.exports=o;},{"../../../errors":15}],25:[function(e,t){t.exports=function(){throw new Error("Readable.from is not available in the browser")};},{}],26:[function(e,t){function n(e){var t=!1;return function(){t||(t=!0,e.apply(void 0,arguments));}}function r(e){if(e)throw e}function a(e){return e.setHeader&&"function"==typeof e.abort}function o(t,r,o,i){i=n(i);var d=!1;t.on("close",function(){d=!0;}),l===void 0&&(l=e("./end-of-stream")),l(t,{readable:r,writable:o},function(e){return e?i(e):void(d=!0,i())});var s=!1;return function(e){if(!d)return s?void 0:(s=!0,a(t)?t.abort():"function"==typeof t.destroy?t.destroy():void i(e||new p("pipe")))}}function i(e){e();}function d(e,t){return e.pipe(t)}function s(e){return e.length?"function"==typeof e[e.length-1]?e.pop():r:r}var l,c=e("../../../errors").codes,u=c.ERR_MISSING_ARGS,p=c.ERR_STREAM_DESTROYED;t.exports=function(){for(var e=arguments.length,t=Array(e),n=0;n<e;n++)t[n]=arguments[n];var r=s(t);if(Array.isArray(t[0])&&(t=t[0]),2>t.length)throw new u("streams");var a,l=t.map(function(e,n){var d=n<t.length-1;return o(e,d,0<n,function(e){a||(a=e),e&&l.forEach(i),d||(l.forEach(i),r(a));})});return t.reduce(d)};},{"../../../errors":15,"./end-of-stream":24}],27:[function(e,n){function r(e,t,n){return null==e.highWaterMark?t?e[n]:null:e.highWaterMark}var a=e("../../../errors").codes.ERR_INVALID_OPT_VALUE;n.exports={getHighWaterMark:function(e,n,o,i){var d=r(n,i,o);if(null!=d){if(!(isFinite(d)&&t(d)===d)||0>d){var s=i?o:"highWaterMark";throw new a(s,d)}return t(d)}return e.objectMode?16:16384}};},{"../../../errors":15}],28:[function(e,t){t.exports=e("events").EventEmitter;},{events:7}],29:[function(e,t,n){n=t.exports=e("./lib/_stream_readable.js"),n.Stream=n,n.Readable=n,n.Writable=e("./lib/_stream_writable.js"),n.Duplex=e("./lib/_stream_duplex.js"),n.Transform=e("./lib/_stream_transform.js"),n.PassThrough=e("./lib/_stream_passthrough.js"),n.finished=e("./lib/internal/streams/end-of-stream.js"),n.pipeline=e("./lib/internal/streams/pipeline.js");},{"./lib/_stream_duplex.js":16,"./lib/_stream_passthrough.js":17,"./lib/_stream_readable.js":18,"./lib/_stream_transform.js":19,"./lib/_stream_writable.js":20,"./lib/internal/streams/end-of-stream.js":24,"./lib/internal/streams/pipeline.js":26}],30:[function(e,t,n){function r(e,t){for(var n in e)t[n]=e[n];}function a(e,t,n){return i(e,t,n)}/*! safe-buffer. MIT License. Feross Aboukhadijeh <https://feross.org/opensource> */var o=e("buffer"),i=o.Buffer;i.from&&i.alloc&&i.allocUnsafe&&i.allocUnsafeSlow?t.exports=o:(r(o,n),n.Buffer=a),a.prototype=Object.create(i.prototype),r(i,a),a.from=function(e,t,n){if("number"==typeof e)throw new TypeError("Argument must not be a number");return i(e,t,n)},a.alloc=function(e,t,n){if("number"!=typeof e)throw new TypeError("Argument must be a number");var r=i(e);return void 0===t?r.fill(0):"string"==typeof n?r.fill(t,n):r.fill(t),r},a.allocUnsafe=function(e){if("number"!=typeof e)throw new TypeError("Argument must be a number");return i(e)},a.allocUnsafeSlow=function(e){if("number"!=typeof e)throw new TypeError("Argument must be a number");return o.SlowBuffer(e)};},{buffer:3}],31:[function(e,t,n){function r(e){if(!e)return "utf8";for(var t;;)switch(e){case"utf8":case"utf-8":return "utf8";case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return "utf16le";case"latin1":case"binary":return "latin1";case"base64":case"ascii":case"hex":return e;default:if(t)return;e=(""+e).toLowerCase(),t=!0;}}function a(e){var t=r(e);if("string"!=typeof t&&(m.isEncoding===b||!b(e)))throw new Error("Unknown encoding: "+e);return t||e}function o(e){this.encoding=a(e);var t;switch(this.encoding){case"utf16le":this.text=u,this.end=p,t=4;break;case"utf8":this.fillLast=c,t=4;break;case"base64":this.text=f,this.end=g,t=3;break;default:return this.write=_,void(this.end=h);}this.lastNeed=0,this.lastTotal=0,this.lastChar=m.allocUnsafe(t);}function d(e){if(127>=e)return 0;return 6==e>>5?2:14==e>>4?3:30==e>>3?4:2==e>>6?-1:-2}function s(e,t,n){var r=t.length-1;if(r<n)return 0;var a=d(t[r]);return 0<=a?(0<a&&(e.lastNeed=a-1),a):--r<n||-2===a?0:(a=d(t[r]),0<=a)?(0<a&&(e.lastNeed=a-2),a):--r<n||-2===a?0:(a=d(t[r]),0<=a?(0<a&&(2===a?a=0:e.lastNeed=a-3),a):0)}function l(e,t){if(128!=(192&t[0]))return e.lastNeed=0,"\uFFFD";if(1<e.lastNeed&&1<t.length){if(128!=(192&t[1]))return e.lastNeed=1,"\uFFFD";if(2<e.lastNeed&&2<t.length&&128!=(192&t[2]))return e.lastNeed=2,"\uFFFD"}}function c(e){var t=this.lastTotal-this.lastNeed,n=l(this,e);return void 0===n?this.lastNeed<=e.length?(e.copy(this.lastChar,t,0,this.lastNeed),this.lastChar.toString(this.encoding,0,this.lastTotal)):void(e.copy(this.lastChar,t,0,e.length),this.lastNeed-=e.length):n}function u(e,t){if(0==(e.length-t)%2){var n=e.toString("utf16le",t);if(n){var r=n.charCodeAt(n.length-1);if(55296<=r&&56319>=r)return this.lastNeed=2,this.lastTotal=4,this.lastChar[0]=e[e.length-2],this.lastChar[1]=e[e.length-1],n.slice(0,-1)}return n}return this.lastNeed=1,this.lastTotal=2,this.lastChar[0]=e[e.length-1],e.toString("utf16le",t,e.length-1)}function p(e){var t=e&&e.length?this.write(e):"";if(this.lastNeed){var n=this.lastTotal-this.lastNeed;return t+this.lastChar.toString("utf16le",0,n)}return t}function f(e,t){var r=(e.length-t)%3;return 0==r?e.toString("base64",t):(this.lastNeed=3-r,this.lastTotal=3,1==r?this.lastChar[0]=e[e.length-1]:(this.lastChar[0]=e[e.length-2],this.lastChar[1]=e[e.length-1]),e.toString("base64",t,e.length-r))}function g(e){var t=e&&e.length?this.write(e):"";return this.lastNeed?t+this.lastChar.toString("base64",0,3-this.lastNeed):t}function _(e){return e.toString(this.encoding)}function h(e){return e&&e.length?this.write(e):""}var m=e("safe-buffer").Buffer,b=m.isEncoding||function(e){switch(e=""+e,e&&e.toLowerCase()){case"hex":case"utf8":case"utf-8":case"ascii":case"binary":case"base64":case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":case"raw":return !0;default:return !1;}};n.StringDecoder=o,o.prototype.write=function(e){if(0===e.length)return "";var t,n;if(this.lastNeed){if(t=this.fillLast(e),void 0===t)return "";n=this.lastNeed,this.lastNeed=0;}else n=0;return n<e.length?t?t+this.text(e,n):this.text(e,n):t||""},o.prototype.end=function(e){var t=e&&e.length?this.write(e):"";return this.lastNeed?t+"\uFFFD":t},o.prototype.text=function(e,t){var n=s(this,e,t);if(!this.lastNeed)return e.toString("utf8",t);this.lastTotal=n;var r=e.length-(n-this.lastNeed);return e.copy(this.lastChar,0,r),e.toString("utf8",t,r)},o.prototype.fillLast=function(e){return this.lastNeed<=e.length?(e.copy(this.lastChar,this.lastTotal-this.lastNeed,0,this.lastNeed),this.lastChar.toString(this.encoding,0,this.lastTotal)):void(e.copy(this.lastChar,this.lastTotal-this.lastNeed,0,e.length),this.lastNeed-=e.length)};},{"safe-buffer":30}],32:[function(e,t){(function(e){(function(){function n(t){try{if(!e.localStorage)return !1}catch(e){return !1}var n=e.localStorage[t];return null!=n&&"true"===(n+"").toLowerCase()}t.exports=function(e,t){function r(){if(!a){if(n("throwDeprecation"))throw new Error(t);else n("traceDeprecation")?console.trace(t):console.warn(t);a=!0;}return e.apply(this,arguments)}if(n("noDeprecation"))return e;var a=!1;return r};}).call(this);}).call(this,"undefined"==typeof commonjsGlobal?"undefined"==typeof self?"undefined"==typeof window?{}:window:self:commonjsGlobal);},{}],"/":[function(e,t){function n(e){return e.replace(/a=ice-options:trickle\s\n/g,"")}function r(e){console.warn(e);}/*! simple-peer. MIT License. Feross Aboukhadijeh <https://feross.org/opensource> */const a=e("debug")("simple-peer"),o=e("get-browser-rtc"),i=e("randombytes"),d=e("readable-stream"),s=e("queue-microtask"),l=e("err-code"),{Buffer:c}=e("buffer"),u=65536;class p extends d.Duplex{constructor(e){if(e=Object.assign({allowHalfOpen:!1},e),super(e),this._id=i(4).toString("hex").slice(0,7),this._debug("new peer %o",e),this.channelName=e.initiator?e.channelName||i(20).toString("hex"):null,this.initiator=e.initiator||!1,this.channelConfig=e.channelConfig||p.channelConfig,this.channelNegotiated=this.channelConfig.negotiated,this.config=Object.assign({},p.config,e.config),this.offerOptions=e.offerOptions||{},this.answerOptions=e.answerOptions||{},this.sdpTransform=e.sdpTransform||(e=>e),this.streams=e.streams||(e.stream?[e.stream]:[]),this.trickle=void 0===e.trickle||e.trickle,this.allowHalfTrickle=void 0!==e.allowHalfTrickle&&e.allowHalfTrickle,this.iceCompleteTimeout=e.iceCompleteTimeout||5000,this.destroyed=!1,this.destroying=!1,this._connected=!1,this.remoteAddress=void 0,this.remoteFamily=void 0,this.remotePort=void 0,this.localAddress=void 0,this.localFamily=void 0,this.localPort=void 0,this._wrtc=e.wrtc&&"object"==typeof e.wrtc?e.wrtc:o(),!this._wrtc)if("undefined"==typeof window)throw l(new Error("No WebRTC support: Specify `opts.wrtc` option in this environment"),"ERR_WEBRTC_SUPPORT");else throw l(new Error("No WebRTC support: Not a supported browser"),"ERR_WEBRTC_SUPPORT");this._pcReady=!1,this._channelReady=!1,this._iceComplete=!1,this._iceCompleteTimer=null,this._channel=null,this._pendingCandidates=[],this._isNegotiating=!1,this._firstNegotiation=!0,this._batchedNegotiation=!1,this._queuedNegotiation=!1,this._sendersAwaitingStable=[],this._senderMap=new Map,this._closingInterval=null,this._remoteTracks=[],this._remoteStreams=[],this._chunk=null,this._cb=null,this._interval=null;try{this._pc=new this._wrtc.RTCPeerConnection(this.config);}catch(e){return void this.destroy(l(e,"ERR_PC_CONSTRUCTOR"))}this._isReactNativeWebrtc="number"==typeof this._pc._peerConnectionId,this._pc.oniceconnectionstatechange=()=>{this._onIceStateChange();},this._pc.onicegatheringstatechange=()=>{this._onIceStateChange();},this._pc.onconnectionstatechange=()=>{this._onConnectionStateChange();},this._pc.onsignalingstatechange=()=>{this._onSignalingStateChange();},this._pc.onicecandidate=e=>{this._onIceCandidate(e);},"object"==typeof this._pc.peerIdentity&&this._pc.peerIdentity.catch(e=>{this.destroy(l(e,"ERR_PC_PEER_IDENTITY"));}),this.initiator||this.channelNegotiated?this._setupData({channel:this._pc.createDataChannel(this.channelName,this.channelConfig)}):this._pc.ondatachannel=e=>{this._setupData(e);},this.streams&&this.streams.forEach(e=>{this.addStream(e);}),this._pc.ontrack=e=>{this._onTrack(e);},this._debug("initial negotiation"),this._needsNegotiation(),this._onFinishBound=()=>{this._onFinish();},this.once("finish",this._onFinishBound);}get bufferSize(){return this._channel&&this._channel.bufferedAmount||0}get connected(){return this._connected&&"open"===this._channel.readyState}address(){return {port:this.localPort,family:this.localFamily,address:this.localAddress}}signal(e){if(!this.destroying){if(this.destroyed)throw l(new Error("cannot signal after peer is destroyed"),"ERR_DESTROYED");if("string"==typeof e)try{e=JSON.parse(e);}catch(t){e={};}this._debug("signal()"),e.renegotiate&&this.initiator&&(this._debug("got request to renegotiate"),this._needsNegotiation()),e.transceiverRequest&&this.initiator&&(this._debug("got request for transceiver"),this.addTransceiver(e.transceiverRequest.kind,e.transceiverRequest.init)),e.candidate&&(this._pc.remoteDescription&&this._pc.remoteDescription.type?this._addIceCandidate(e.candidate):this._pendingCandidates.push(e.candidate)),e.sdp&&this._pc.setRemoteDescription(new this._wrtc.RTCSessionDescription(e)).then(()=>{this.destroyed||(this._pendingCandidates.forEach(e=>{this._addIceCandidate(e);}),this._pendingCandidates=[],"offer"===this._pc.remoteDescription.type&&this._createAnswer());}).catch(e=>{this.destroy(l(e,"ERR_SET_REMOTE_DESCRIPTION"));}),e.sdp||e.candidate||e.renegotiate||e.transceiverRequest||this.destroy(l(new Error("signal() called with invalid signal data"),"ERR_SIGNALING"));}}_addIceCandidate(e){const t=new this._wrtc.RTCIceCandidate(e);this._pc.addIceCandidate(t).catch(e=>{!t.address||t.address.endsWith(".local")?r("Ignoring unsupported ICE candidate."):this.destroy(l(e,"ERR_ADD_ICE_CANDIDATE"));});}send(e){if(!this.destroying){if(this.destroyed)throw l(new Error("cannot send after peer is destroyed"),"ERR_DESTROYED");this._channel.send(e);}}addTransceiver(e,t){if(!this.destroying){if(this.destroyed)throw l(new Error("cannot addTransceiver after peer is destroyed"),"ERR_DESTROYED");if(this._debug("addTransceiver()"),this.initiator)try{this._pc.addTransceiver(e,t),this._needsNegotiation();}catch(e){this.destroy(l(e,"ERR_ADD_TRANSCEIVER"));}else this.emit("signal",{type:"transceiverRequest",transceiverRequest:{kind:e,init:t}});}}addStream(e){if(!this.destroying){if(this.destroyed)throw l(new Error("cannot addStream after peer is destroyed"),"ERR_DESTROYED");this._debug("addStream()"),e.getTracks().forEach(t=>{this.addTrack(t,e);});}}addTrack(e,t){if(this.destroying)return;if(this.destroyed)throw l(new Error("cannot addTrack after peer is destroyed"),"ERR_DESTROYED");this._debug("addTrack()");const n=this._senderMap.get(e)||new Map;let r=n.get(t);if(!r)r=this._pc.addTrack(e,t),n.set(t,r),this._senderMap.set(e,n),this._needsNegotiation();else if(r.removed)throw l(new Error("Track has been removed. You should enable/disable tracks that you want to re-add."),"ERR_SENDER_REMOVED");else throw l(new Error("Track has already been added to that stream."),"ERR_SENDER_ALREADY_ADDED")}replaceTrack(e,t,n){if(this.destroying)return;if(this.destroyed)throw l(new Error("cannot replaceTrack after peer is destroyed"),"ERR_DESTROYED");this._debug("replaceTrack()");const r=this._senderMap.get(e),a=r?r.get(n):null;if(!a)throw l(new Error("Cannot replace track that was never added."),"ERR_TRACK_NOT_ADDED");t&&this._senderMap.set(t,r),null==a.replaceTrack?this.destroy(l(new Error("replaceTrack is not supported in this browser"),"ERR_UNSUPPORTED_REPLACETRACK")):a.replaceTrack(t);}removeTrack(e,t){if(this.destroying)return;if(this.destroyed)throw l(new Error("cannot removeTrack after peer is destroyed"),"ERR_DESTROYED");this._debug("removeSender()");const n=this._senderMap.get(e),r=n?n.get(t):null;if(!r)throw l(new Error("Cannot remove track that was never added."),"ERR_TRACK_NOT_ADDED");try{r.removed=!0,this._pc.removeTrack(r);}catch(e){"NS_ERROR_UNEXPECTED"===e.name?this._sendersAwaitingStable.push(r):this.destroy(l(e,"ERR_REMOVE_TRACK"));}this._needsNegotiation();}removeStream(e){if(!this.destroying){if(this.destroyed)throw l(new Error("cannot removeStream after peer is destroyed"),"ERR_DESTROYED");this._debug("removeSenders()"),e.getTracks().forEach(t=>{this.removeTrack(t,e);});}}_needsNegotiation(){this._debug("_needsNegotiation"),this._batchedNegotiation||(this._batchedNegotiation=!0,s(()=>{this._batchedNegotiation=!1,this.initiator||!this._firstNegotiation?(this._debug("starting batched negotiation"),this.negotiate()):this._debug("non-initiator initial negotiation request discarded"),this._firstNegotiation=!1;}));}negotiate(){if(!this.destroying){if(this.destroyed)throw l(new Error("cannot negotiate after peer is destroyed"),"ERR_DESTROYED");this.initiator?this._isNegotiating?(this._queuedNegotiation=!0,this._debug("already negotiating, queueing")):(this._debug("start negotiation"),setTimeout(()=>{this._createOffer();},0)):this._isNegotiating?(this._queuedNegotiation=!0,this._debug("already negotiating, queueing")):(this._debug("requesting negotiation from initiator"),this.emit("signal",{type:"renegotiate",renegotiate:!0})),this._isNegotiating=!0;}}destroy(e){this._destroy(e,()=>{});}_destroy(e,t){this.destroyed||this.destroying||(this.destroying=!0,this._debug("destroying (error: %s)",e&&(e.message||e)),s(()=>{if(this.destroyed=!0,this.destroying=!1,this._debug("destroy (error: %s)",e&&(e.message||e)),this.readable=this.writable=!1,this._readableState.ended||this.push(null),this._writableState.finished||this.end(),this._connected=!1,this._pcReady=!1,this._channelReady=!1,this._remoteTracks=null,this._remoteStreams=null,this._senderMap=null,clearInterval(this._closingInterval),this._closingInterval=null,clearInterval(this._interval),this._interval=null,this._chunk=null,this._cb=null,this._onFinishBound&&this.removeListener("finish",this._onFinishBound),this._onFinishBound=null,this._channel){try{this._channel.close();}catch(e){}this._channel.onmessage=null,this._channel.onopen=null,this._channel.onclose=null,this._channel.onerror=null;}if(this._pc){try{this._pc.close();}catch(e){}this._pc.oniceconnectionstatechange=null,this._pc.onicegatheringstatechange=null,this._pc.onsignalingstatechange=null,this._pc.onicecandidate=null,this._pc.ontrack=null,this._pc.ondatachannel=null;}this._pc=null,this._channel=null,e&&this.emit("error",e),this.emit("close"),t();}));}_setupData(e){if(!e.channel)return this.destroy(l(new Error("Data channel event is missing `channel` property"),"ERR_DATA_CHANNEL"));this._channel=e.channel,this._channel.binaryType="arraybuffer","number"==typeof this._channel.bufferedAmountLowThreshold&&(this._channel.bufferedAmountLowThreshold=u),this.channelName=this._channel.label,this._channel.onmessage=e=>{this._onChannelMessage(e);},this._channel.onbufferedamountlow=()=>{this._onChannelBufferedAmountLow();},this._channel.onopen=()=>{this._onChannelOpen();},this._channel.onclose=()=>{this._onChannelClose();},this._channel.onerror=e=>{const t=e.error instanceof Error?e.error:new Error(`Datachannel error: ${e.message} ${e.filename}:${e.lineno}:${e.colno}`);this.destroy(l(t,"ERR_DATA_CHANNEL"));};let t=!1;this._closingInterval=setInterval(()=>{this._channel&&"closing"===this._channel.readyState?(t&&this._onChannelClose(),t=!0):t=!1;},5000);}_read(){}_write(e,t,n){if(this.destroyed)return n(l(new Error("cannot write after peer is destroyed"),"ERR_DATA_CHANNEL"));if(this._connected){try{this.send(e);}catch(e){return this.destroy(l(e,"ERR_DATA_CHANNEL"))}this._channel.bufferedAmount>u?(this._debug("start backpressure: bufferedAmount %d",this._channel.bufferedAmount),this._cb=n):n(null);}else this._debug("write before connect"),this._chunk=e,this._cb=n;}_onFinish(){if(!this.destroyed){const e=()=>{setTimeout(()=>this.destroy(),1e3);};this._connected?e():this.once("connect",e);}}_startIceCompleteTimeout(){this.destroyed||this._iceCompleteTimer||(this._debug("started iceComplete timeout"),this._iceCompleteTimer=setTimeout(()=>{this._iceComplete||(this._iceComplete=!0,this._debug("iceComplete timeout completed"),this.emit("iceTimeout"),this.emit("_iceComplete"));},this.iceCompleteTimeout));}_createOffer(){this.destroyed||this._pc.createOffer(this.offerOptions).then(e=>{if(this.destroyed)return;this.trickle||this.allowHalfTrickle||(e.sdp=n(e.sdp)),e.sdp=this.sdpTransform(e.sdp);const t=()=>{if(!this.destroyed){const t=this._pc.localDescription||e;this._debug("signal"),this.emit("signal",{type:t.type,sdp:t.sdp});}};this._pc.setLocalDescription(e).then(()=>{this._debug("createOffer success"),this.destroyed||(this.trickle||this._iceComplete?t():this.once("_iceComplete",t));}).catch(e=>{this.destroy(l(e,"ERR_SET_LOCAL_DESCRIPTION"));});}).catch(e=>{this.destroy(l(e,"ERR_CREATE_OFFER"));});}_requestMissingTransceivers(){this._pc.getTransceivers&&this._pc.getTransceivers().forEach(e=>{e.mid||!e.sender.track||e.requested||(e.requested=!0,this.addTransceiver(e.sender.track.kind));});}_createAnswer(){this.destroyed||this._pc.createAnswer(this.answerOptions).then(e=>{if(this.destroyed)return;this.trickle||this.allowHalfTrickle||(e.sdp=n(e.sdp)),e.sdp=this.sdpTransform(e.sdp);const t=()=>{if(!this.destroyed){const t=this._pc.localDescription||e;this._debug("signal"),this.emit("signal",{type:t.type,sdp:t.sdp}),this.initiator||this._requestMissingTransceivers();}};this._pc.setLocalDescription(e).then(()=>{this.destroyed||(this.trickle||this._iceComplete?t():this.once("_iceComplete",t));}).catch(e=>{this.destroy(l(e,"ERR_SET_LOCAL_DESCRIPTION"));});}).catch(e=>{this.destroy(l(e,"ERR_CREATE_ANSWER"));});}_onConnectionStateChange(){this.destroyed||"failed"===this._pc.connectionState&&this.destroy(l(new Error("Connection failed."),"ERR_CONNECTION_FAILURE"));}_onIceStateChange(){if(this.destroyed)return;const e=this._pc.iceConnectionState,t=this._pc.iceGatheringState;this._debug("iceStateChange (connection: %s) (gathering: %s)",e,t),this.emit("iceStateChange",e,t),("connected"===e||"completed"===e)&&(this._pcReady=!0,this._maybeReady()),"failed"===e&&this.destroy(l(new Error("Ice connection failed."),"ERR_ICE_CONNECTION_FAILURE")),"closed"===e&&this.destroy(l(new Error("Ice connection closed."),"ERR_ICE_CONNECTION_CLOSED"));}getStats(e){const t=e=>("[object Array]"===Object.prototype.toString.call(e.values)&&e.values.forEach(t=>{Object.assign(e,t);}),e);0===this._pc.getStats.length||this._isReactNativeWebrtc?this._pc.getStats().then(n=>{const r=[];n.forEach(e=>{r.push(t(e));}),e(null,r);},t=>e(t)):0<this._pc.getStats.length?this._pc.getStats(n=>{if(this.destroyed)return;const r=[];n.result().forEach(e=>{const n={};e.names().forEach(t=>{n[t]=e.stat(t);}),n.id=e.id,n.type=e.type,n.timestamp=e.timestamp,r.push(t(n));}),e(null,r);},t=>e(t)):e(null,[]);}_maybeReady(){if(this._debug("maybeReady pc %s channel %s",this._pcReady,this._channelReady),this._connected||this._connecting||!this._pcReady||!this._channelReady)return;this._connecting=!0;const e=()=>{this.destroyed||this.getStats((t,n)=>{if(this.destroyed)return;t&&(n=[]);const r={},a={},o={};let i=!1;n.forEach(e=>{("remotecandidate"===e.type||"remote-candidate"===e.type)&&(r[e.id]=e),("localcandidate"===e.type||"local-candidate"===e.type)&&(a[e.id]=e),("candidatepair"===e.type||"candidate-pair"===e.type)&&(o[e.id]=e);});const d=e=>{i=!0;let t=a[e.localCandidateId];t&&(t.ip||t.address)?(this.localAddress=t.ip||t.address,this.localPort=+t.port):t&&t.ipAddress?(this.localAddress=t.ipAddress,this.localPort=+t.portNumber):"string"==typeof e.googLocalAddress&&(t=e.googLocalAddress.split(":"),this.localAddress=t[0],this.localPort=+t[1]),this.localAddress&&(this.localFamily=this.localAddress.includes(":")?"IPv6":"IPv4");let n=r[e.remoteCandidateId];n&&(n.ip||n.address)?(this.remoteAddress=n.ip||n.address,this.remotePort=+n.port):n&&n.ipAddress?(this.remoteAddress=n.ipAddress,this.remotePort=+n.portNumber):"string"==typeof e.googRemoteAddress&&(n=e.googRemoteAddress.split(":"),this.remoteAddress=n[0],this.remotePort=+n[1]),this.remoteAddress&&(this.remoteFamily=this.remoteAddress.includes(":")?"IPv6":"IPv4"),this._debug("connect local: %s:%s remote: %s:%s",this.localAddress,this.localPort,this.remoteAddress,this.remotePort);};if(n.forEach(e=>{"transport"===e.type&&e.selectedCandidatePairId&&d(o[e.selectedCandidatePairId]),("googCandidatePair"===e.type&&"true"===e.googActiveConnection||("candidatepair"===e.type||"candidate-pair"===e.type)&&e.selected)&&d(e);}),!i&&(!Object.keys(o).length||Object.keys(a).length))return void setTimeout(e,100);if(this._connecting=!1,this._connected=!0,this._chunk){try{this.send(this._chunk);}catch(e){return this.destroy(l(e,"ERR_DATA_CHANNEL"))}this._chunk=null,this._debug("sent chunk from \"write before connect\"");const e=this._cb;this._cb=null,e(null);}"number"!=typeof this._channel.bufferedAmountLowThreshold&&(this._interval=setInterval(()=>this._onInterval(),150),this._interval.unref&&this._interval.unref()),this._debug("connect"),this.emit("connect");});};e();}_onInterval(){this._cb&&this._channel&&!(this._channel.bufferedAmount>u)&&this._onChannelBufferedAmountLow();}_onSignalingStateChange(){this.destroyed||("stable"===this._pc.signalingState&&(this._isNegotiating=!1,this._debug("flushing sender queue",this._sendersAwaitingStable),this._sendersAwaitingStable.forEach(e=>{this._pc.removeTrack(e),this._queuedNegotiation=!0;}),this._sendersAwaitingStable=[],this._queuedNegotiation?(this._debug("flushing negotiation queue"),this._queuedNegotiation=!1,this._needsNegotiation()):(this._debug("negotiated"),this.emit("negotiated"))),this._debug("signalingStateChange %s",this._pc.signalingState),this.emit("signalingStateChange",this._pc.signalingState));}_onIceCandidate(e){this.destroyed||(e.candidate&&this.trickle?this.emit("signal",{type:"candidate",candidate:{candidate:e.candidate.candidate,sdpMLineIndex:e.candidate.sdpMLineIndex,sdpMid:e.candidate.sdpMid}}):!e.candidate&&!this._iceComplete&&(this._iceComplete=!0,this.emit("_iceComplete")),e.candidate&&this._startIceCompleteTimeout());}_onChannelMessage(e){if(this.destroyed)return;let t=e.data;t instanceof ArrayBuffer&&(t=c.from(t)),this.push(t);}_onChannelBufferedAmountLow(){if(!this.destroyed&&this._cb){this._debug("ending backpressure: bufferedAmount %d",this._channel.bufferedAmount);const e=this._cb;this._cb=null,e(null);}}_onChannelOpen(){this._connected||this.destroyed||(this._debug("on channel open"),this._channelReady=!0,this._maybeReady());}_onChannelClose(){this.destroyed||(this._debug("on channel close"),this.destroy());}_onTrack(e){this.destroyed||e.streams.forEach(t=>{this._debug("on track"),this.emit("track",e.track,t),this._remoteTracks.push({track:e.track,stream:t}),this._remoteStreams.some(e=>e.id===t.id)||(this._remoteStreams.push(t),s(()=>{this._debug("on stream"),this.emit("stream",t);}));});}_debug(){const e=[].slice.call(arguments);e[0]="["+this._id+"] "+e[0],a.apply(null,e);}}p.WEBRTC_SUPPORT=!!o(),p.config={iceServers:[{urls:["stun:stun.l.google.com:19302","stun:global.stun.twilio.com:3478"]}],sdpSemantics:"unified-plan"},p.channelConfig={},t.exports=p;},{buffer:3,debug:4,"err-code":6,"get-browser-rtc":8,"queue-microtask":13,randombytes:14,"readable-stream":29}]},{},[])("/")});
  });

  /**
   * @module sync-protocol
   */

  /**
   * @typedef {Map<number, number>} StateMap
   */

  /**
   * Core Yjs defines two message types:
   * • YjsSyncStep1: Includes the State Set of the sending client. When received, the client should reply with YjsSyncStep2.
   * • YjsSyncStep2: Includes all missing structs and the complete delete set. When received, the client is assured that it
   *   received all information from the remote client.
   *
   * In a peer-to-peer network, you may want to introduce a SyncDone message type. Both parties should initiate the connection
   * with SyncStep1. When a client received SyncStep2, it should reply with SyncDone. When the local client received both
   * SyncStep2 and SyncDone, it is assured that it is synced to the remote client.
   *
   * In a client-server model, you want to handle this differently: The client should initiate the connection with SyncStep1.
   * When the server receives SyncStep1, it should reply with SyncStep2 immediately followed by SyncStep1. The client replies
   * with SyncStep2 when it receives SyncStep1. Optionally the server may send a SyncDone after it received SyncStep2, so the
   * client knows that the sync is finished.  There are two reasons for this more elaborated sync model: 1. This protocol can
   * easily be implemented on top of http and websockets. 2. The server shoul only reply to requests, and not initiate them.
   * Therefore it is necesarry that the client initiates the sync.
   *
   * Construction of a message:
   * [messageType : varUint, message definition..]
   *
   * Note: A message does not include information about the room name. This must to be handled by the upper layer protocol!
   *
   * stringify[messageType] stringifies a message definition (messageType is already read from the bufffer)
   */

  const messageYjsSyncStep1 = 0;
  const messageYjsSyncStep2 = 1;
  const messageYjsUpdate = 2;

  /**
   * Create a sync step 1 message based on the state of the current shared document.
   *
   * @param {encoding.Encoder} encoder
   * @param {Y.Doc} doc
   */
  const writeSyncStep1 = (encoder, doc) => {
    writeVarUint(encoder, messageYjsSyncStep1);
    const sv = encodeStateVector(doc);
    writeVarUint8Array(encoder, sv);
  };

  /**
   * @param 