import type { Gen, GenYield } from '../fn/types';
export type Skip = {
    <T extends Gen<any>>(num: number): {
        (gen: T): Gen<GenYield<T>>;
    };
    <T extends Gen<any>>(num: number, gen: T): Gen<GenYield<T>>;
};
/**
 * Do not `yield` values until after first ignoring `count` values.
 *
 * A higher-order function that takes a number `num` as its argument and returns a new function that can be used to skip
 * the first `num` values generated by a generator.
 *
 * The returned function takes a generator as its argument and returns a new generator that yields all the values
 * generated by the input generator except for the first `num` values.
 *
 * ## Example
 *
 * Here's an example of how to use the `skip` function to skip the first two values generated by a `range` generator:
 *
 * ```ts
 * import { pipe } from 'tightrope/fn/pipe';
 * import { range } from 'tightrope/gen/range';
 * import { skip } from 'tightrope/gen/skip';
 *
 * const gen = pipe(range(1, 10), skip(2));
 *
 * for (const value of gen) {
 *   console.log(value); // 3, 4, 5, 6, 7, 8, 9, 10
 * }
 * ```
 *
 * ## Use Cases
 *
 * Common use cases for the `skip` function:
 *
 * 1. Skipping the first `n` items in a sequence: You may need to skip the first `n` items in a sequence of data, either
 *    because they are not relevant to your use case or because you have already processed them.
 * 2. Paginating data: If you are working with large datasets, you may need to paginate the data to improve performance.
 *    The `skip` function can be used to skip the first `n` items in each page of data.
 * 3. Splitting data into batches: If you are processing data in batches, you may need to skip the first `n` items in each
 *    batch to avoid processing duplicate data.
 * 4. Skipping headers or footers in a file: If you are processing a file with a header or footer, you may need to skip
 *    those lines before processing the rest of the file.
 *
 * Overall, the skip function is a useful tool for skipping over irrelevant or already-processed data in a sequence, and
 * can be used in a variety of contexts to improve the efficiency and readability of your code.
 *
 * @tags generator
 */
export declare const skip: Skip;
