import path from 'node:path';
import fs from 'node:fs/promises';
import { createRequire, isBuiltin } from 'node:module';
// Get our own version
const { version } = createRequire(import.meta.url)('../package.json');
// Prepare node built-in modules lists.
const nodePrefix = 'node:';
const nodePrefixRx = /^node:/;
// Files that mark the root of a workspace.
const workspaceRootFiles = new Set([
    'pnpm-workspace.yaml', // pnpm
    'lerna.json', // Lerna
    // Note: is there any interest in the following?
    // 'workspace.jsonc',      // Bit
    // 'nx.json',              // Nx
    // 'rush.json',            // Rush
]);
const defaults = {
    builtins: true,
    builtinsPrefix: 'add',
    packagePath: [],
    deps: true,
    devDeps: false,
    peerDeps: true,
    optDeps: true,
    include: [],
    exclude: []
};
const isString = (str) => typeof str === 'string' && str.length > 0;
/**
 * A Rollup plugin that automatically declares NodeJS built-in modules,
 * and optionally npm dependencies, as 'external'.
 */
function nodeExternals(options = {}) {
    const config = { ...defaults, ...options };
    let include, exclude;
    const isIncluded = (id) => include.some(rx => rx.test(id)), isExcluded = (id) => exclude.some(rx => rx.test(id));
    return {
        name: 'node-externals',
        version,
        async buildStart() {
            // Map the include and exclude options to arrays of regexes.
            [include, exclude] = ['include', 'exclude'].map(option => []
                .concat(config[option])
                .reduce((result, entry, index) => {
                if (entry instanceof RegExp)
                    result.push(entry);
                else if (isString(entry))
                    result.push(new RegExp('^' + entry.replace(/[.*+?^${}()|[\]\\]/g, '\\$&') + '$'));
                else if (entry)
                    this.warn(`Ignoring wrong entry type #${index} in '${option}' option: ${JSON.stringify(entry)}`);
                return result;
            }, []));
            // Populate the packagePath option if not given by getting all package.json files
            // from cwd up to the root of the git repo, the root of the monorepo,
            // or the root of the volume, whichever comes first.
            const packagePaths = []
                .concat(config.packagePath)
                .filter(isString)
                .map(packagePath => path.resolve(packagePath));
            if (packagePaths.length === 0) {
                for (let current = process.cwd(), previous = undefined; previous !== current; previous = current, current = path.dirname(current)) {
                    const entries = await fs.readdir(current, { withFileTypes: true }).catch(() => null);
                    if (entries === null)
                        this.error(`Could not read contents of directory ${JSON.stringify(current)}.`);
                    // Gather package.json files.
                    if (entries.some(entry => entry.name === 'package.json' && entry.isFile()))
                        packagePaths.push(path.join(current, 'package.json'));
                    // Break early if this is a git repo root or there is a known workspace root file.
                    if (entries.some(entry => (entry.name === '.git' && entry.isDirectory())
                        || (workspaceRootFiles.has(entry.name) && entry.isFile()))) {
                        break;
                    }
                }
            }
            // Gather dependencies names.
            const dependencies = {};
            for (const packagePath of packagePaths) {
                this.addWatchFile(packagePath);
                try {
                    const json = (await fs.readFile(packagePath)).toString();
                    try {
                        const pkg = JSON.parse(json);
                        Object.assign(dependencies, config.deps ? pkg.dependencies : undefined, config.devDeps ? pkg.devDependencies : undefined, config.peerDeps ? pkg.peerDependencies : undefined, config.optDeps ? pkg.optionalDependencies : undefined);
                        // Break early if this is a npm/yarn workspace root.
                        if ('workspaces' in pkg)
                            break;
                    }
                    catch {
                        this.error({
                            message: `File ${JSON.stringify(packagePath)} does not look like a valid package.json file.`,
                            stack: undefined
                        });
                    }
                }
                catch {
                    this.error({
                        message: `Cannot read file ${JSON.stringify(packagePath)}`,
                        stack: undefined
                    });
                }
            }
            // Add all dependencies as an include RegEx.
            const names = Object.keys(dependencies);
            if (names.length > 0)
                include.push(new RegExp('^(?:' + names.join('|') + ')(?:/.+)?$'));
        },
        async resolveId(specifier) {
            // Ignore absolute (already resolved) ids, relative imports and virtual modules.
            if (/^(?:\0|\.{0,2}\/)/.test(specifier) || path.isAbsolute(specifier))
                return null;
            // Handle node builtins.
            if (isBuiltin(specifier)) {
                const stripped = specifier.replace(nodePrefixRx, '');
                return {
                    id: config.builtinsPrefix === 'ignore'
                        ? specifier
                        : config.builtinsPrefix === 'add' || (specifier.startsWith(nodePrefix) && !isBuiltin(stripped))
                            ? nodePrefix + stripped
                            : stripped,
                    external: (config.builtins || isIncluded(specifier)) && !isExcluded(specifier),
                    moduleSideEffects: false
                };
            }
            // Handle npm dependencies.
            return isIncluded(specifier) && !isExcluded(specifier)
                ? false // external
                : null; // normal handling
        }
    };
}
export default nodeExternals;
export { nodeExternals };
//# sourceMappingURL=index.js.map