"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardVersionGroup = void 0;
const effect_1 = require("effect");
const uniq_1 = require("tightrope/array/uniq");
const report_1 = require("../report");
const specifier_1 = require("../specifier");
const get_preferred_version_1 = require("./lib/get-preferred-version");
const group_by_1 = require("./lib/group-by");
class StandardVersionGroup extends effect_1.Data.TaggedClass('Standard') {
    groupType = 'versionGroup';
    constructor(isCatchAll, config) {
        super({
            config,
            instances: [],
            isCatchAll,
        });
    }
    canAdd(_) {
        return true;
    }
    inspectAll() {
        return effect_1.Effect.all(Object.entries((0, group_by_1.groupBy)('name', this.instances)).flatMap(([name, instances]) => {
            const localInstance = getLocalInstance(instances);
            if (localInstance) {
                const localVersion = localInstance?.rawSpecifier.raw;
                return (0, effect_1.pipe)(effect_1.Effect.succeed(specifier_1.Specifier.create(localInstance, localVersion)), effect_1.Effect.flatMap((local) => effect_1.Effect.all(local._tag !== 'Exact' && instances.length > 1
                    ? instances.map((instance) => 
                    // ! dependency is a package developed in this repo
                    // ✘ local package has an invalid .version property
                    // ✘ is a mismatch we can't auto-fix
                    effect_1.Effect.succeed(new report_1.Report.MissingLocalVersion(instance, localInstance)))
                    : instances.flatMap((instance) => 
                    // instances.flatMap((instance) =>
                    (0, effect_1.pipe)(effect_1.Effect.succeed(specifier_1.Specifier.create(instance, instance.rawSpecifier.raw)), effect_1.Effect.flatMap((specifier) => specifier.instance === localInstance
                        ? // ✓ this is the local package which the others should match
                            // ! its version must always remain as exact semver
                            // ! other instances need to be adjusted for their semver groups
                            effect_1.Effect.succeed(new report_1.Report.Valid(specifier))
                        : (0, effect_1.pipe)(specifier.replaceWith(local), specifier.instance.semverGroup.getFixed, effect_1.Effect.match({
                            onFailure: /* istanbul ignore next */ () => 
                            // ! is not the local package instance
                            // ✘ local version is not fixable by this semver group
                            // ✘ is a mismatch we can't auto-fix
                            // ✘ this should be impossible - we already proved the local version is exact semver
                            new report_1.Report.UnsupportedMismatch(specifier.instance),
                            onSuccess: (valid) => specifier.instance.rawSpecifier.raw === valid.raw
                                ? // ! is not the local package instance
                                    // ✓ local version matches this semver group
                                    // ✓ current version matches local
                                    new report_1.Report.Valid(specifier)
                                : localVersion === 'PACKAGE_JSON_HAS_NO_VERSION'
                                    ? // ! is not the local package instance
                                        // ✘ local package has a version defined
                                        // ✓ local version matches this semver group
                                        // ✘ current version mismatches local
                                        new report_1.Report.MissingLocalVersion(specifier.instance, localInstance)
                                    : // ! is not the local package instance
                                        // ✓ local package has a version defined
                                        // ✓ local version matches this semver group
                                        // ✘ current version mismatches local
                                        new report_1.Report.LocalPackageMismatch(valid, localInstance),
                        }))))))), effect_1.Effect.map((reports) => ({ name, reports })));
            }
            const PreferredMismatch = this.config.preferVersion === 'lowestSemver'
                ? report_1.Report.LowestSemverMismatch
                : report_1.Report.HighestSemverMismatch;
            return (0, effect_1.pipe)(effect_1.Effect.succeed(instances.map((instance) => specifier_1.Specifier.create(instance, instance.rawSpecifier.raw))), effect_1.Effect.flatMap((specifiers) => (0, effect_1.pipe)((0, get_preferred_version_1.getPreferredVersion)(this.config.preferVersion, specifiers), effect_1.Effect.matchEffect({
                onFailure: () => effect_1.Effect.succeed((0, uniq_1.uniq)(specifiers.map((specifier) => specifier.instance.rawSpecifier.raw))
                    .length === 1
                    ? specifiers.map((specifier) => 
                    // ✘ not every version is semver
                    // ✓ every version is identical
                    // ✓ is a match
                    new report_1.Report.Valid(specifier))
                    : instances.map((instance) => 
                    // ✘ not every version is semver
                    // ✘ some versions are not identical
                    // ✘ is a mismatch we can't auto-fix
                    new report_1.Report.UnsupportedMismatch(instance))),
                onSuccess: (expectedVersion) => (0, effect_1.pipe)(specifiers, effect_1.Effect.forEach((current) => (0, effect_1.pipe)(current.replaceWith(expectedVersion), current.instance.semverGroup.getFixed, effect_1.Effect.match({
                    onFailure: /* istanbul ignore next */ () => 
                    // ✓ every version is semver
                    // ✘ expected version is not fixable by its semver group
                    // ✘ is a mismatch we can't auto-fix
                    // ✘ this should be impossible - any valid semver is fixable by a semver group
                    new report_1.Report.UnsupportedMismatch(current.instance),
                    onSuccess: (expectedRange) => current.instance.rawSpecifier.raw === expectedRange.raw
                        ? // ✓ every version is semver
                            // ✓ current version matches expected semver
                            // ✓ current version matches expected version
                            new report_1.Report.Valid(current)
                        : current.instance.rawSpecifier.raw === expectedVersion.raw
                            ? // ✓ every version is semver
                                // ✓ current version matches expected version
                                // ✘ current version does not match expected semver
                                // ✓ is a mismatch we can auto-fix
                                new report_1.Report.SemverRangeMismatch(expectedRange)
                            : // ✓ every version is semver
                                // ✘ current version does not match expected version
                                // ✘ expected version does not match expected semver
                                // ✓ is a mismatch we can auto-fix
                                new PreferredMismatch(expectedRange),
                })))),
            }))), effect_1.Effect.map((reports) => ({ name, reports })));
        }));
    }
}
exports.StandardVersionGroup = StandardVersionGroup;
/**
 * If this dependency is developed in this monorepo, get the instance which
 * represents the canonical .version property of its package.json file.
 */
function getLocalInstance(instances) {
    return instances.find(isLocalInstance);
}
/** Is this dependency developed in this monorepo */
function isLocalInstance(instance) {
    return instance.strategy.name === 'local';
}
