"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createVersionGroups = void 0;
const effect_1 = require("effect");
const is_array_of_strings_1 = require("tightrope/guard/is-array-of-strings");
const is_non_empty_array_1 = require("tightrope/guard/is-non-empty-array");
const is_non_empty_string_1 = require("tightrope/guard/is-non-empty-string");
const is_object_1 = require("tightrope/guard/is-object");
const _1 = require(".");
const banned_1 = require("./banned");
const filtered_out_1 = require("./filtered-out");
const ignored_1 = require("./ignored");
const pinned_1 = require("./pinned");
const same_range_1 = require("./same-range");
const snapped_to_1 = require("./snapped-to");
const standard_1 = require("./standard");
function createVersionGroups(ctx) {
    const { rcFile } = ctx.config;
    const versionGroups = [
        effect_1.Effect.succeed(new filtered_out_1.FilteredOutVersionGroup(ctx)),
    ];
    if ((0, is_non_empty_array_1.isNonEmptyArray)(rcFile.versionGroups)) {
        rcFile.versionGroups.forEach((config) => {
            if (!(0, is_object_1.isObject)(config)) {
                return versionGroups.push(effect_1.Effect.fail(new _1.VersionGroup.ConfigError({
                    config,
                    error: 'config is not an object',
                })));
            }
            const mutuallyExclusiveProps = ['isBanned', 'isIgnored', 'pinVersion', 'snapTo', 'policy'].filter((prop) => Boolean(config[prop]));
            if (mutuallyExclusiveProps.length > 1) {
                return versionGroups.push(effect_1.Effect.fail(new _1.VersionGroup.ConfigError({
                    config,
                    error: `it's unclear what kind of version group you want, as it contains both ${mutuallyExclusiveProps.join(' and ')}`,
                })));
            }
            const label = (0, is_non_empty_string_1.isNonEmptyString)(config.label) ? config.label : '';
            const dependencyTypes = (0, is_array_of_strings_1.isArrayOfStrings)(config.dependencyTypes)
                ? config.dependencyTypes
                : ['**'];
            const dependencies = (0, is_array_of_strings_1.isArrayOfStrings)(config.dependencies) ? config.dependencies : ['**'];
            const packages = (0, is_array_of_strings_1.isArrayOfStrings)(config.packages) ? config.packages : ['**'];
            const specifierTypes = (0, is_array_of_strings_1.isArrayOfStrings)(config.specifierTypes)
                ? config.specifierTypes
                : ['**'];
            if (config.isBanned === true) {
                versionGroups.push(effect_1.Effect.succeed(new banned_1.BannedVersionGroup({
                    dependencies,
                    dependencyTypes,
                    specifierTypes,
                    isBanned: true,
                    label,
                    packages,
                })));
            }
            else if (config.isIgnored === true) {
                versionGroups.push(effect_1.Effect.succeed(new ignored_1.IgnoredVersionGroup({
                    dependencies,
                    dependencyTypes,
                    specifierTypes,
                    isIgnored: true,
                    label,
                    packages,
                })));
            }
            else if ((0, is_non_empty_string_1.isNonEmptyString)(config.pinVersion)) {
                versionGroups.push(effect_1.Effect.succeed(new pinned_1.PinnedVersionGroup({
                    dependencies,
                    dependencyTypes,
                    specifierTypes,
                    label,
                    packages,
                    pinVersion: config.pinVersion,
                })));
            }
            else if ((0, is_array_of_strings_1.isArrayOfStrings)(config.snapTo)) {
                versionGroups.push(effect_1.Effect.succeed(new snapped_to_1.SnappedToVersionGroup({
                    dependencies,
                    dependencyTypes,
                    specifierTypes,
                    label,
                    packages,
                    snapTo: config.snapTo,
                })));
            }
            else if (config.policy === 'sameRange') {
                versionGroups.push(effect_1.Effect.succeed(new same_range_1.SameRangeVersionGroup(ctx, {
                    dependencies,
                    dependencyTypes,
                    specifierTypes,
                    label,
                    packages,
                    policy: config.policy,
                })));
            }
            else {
                versionGroups.push(effect_1.Effect.succeed(new standard_1.StandardVersionGroup(false, {
                    dependencies,
                    dependencyTypes,
                    specifierTypes,
                    label,
                    packages,
                    preferVersion: config.preferVersion === 'lowestSemver' ? 'lowestSemver' : 'highestSemver',
                })));
            }
        });
    }
    versionGroups.push(effect_1.Effect.succeed(new standard_1.StandardVersionGroup(true, {
        dependencies: ['**'],
        dependencyTypes: ['**'],
        specifierTypes: ['**'],
        label: 'Default Version Group',
        packages: ['**'],
        preferVersion: 'highestSemver',
    })));
    return effect_1.Effect.all(versionGroups);
}
exports.createVersionGroups = createVersionGroups;
