"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseSpecifier = void 0;
const effect_1 = require("effect");
const _1 = require(".");
const non_semver_error_1 = require("./lib/non-semver-error");
const parse_specifier_1 = require("./lib/parse-specifier");
class BaseSpecifier {
    /** should be overridden by sub classes */
    _tag = 'Base';
    /** should be overridden by sub classes */
    name = 'base';
    /** The raw semver/workspace:/git etc version value */
    raw;
    /**
     * A `Specifier` describes the version specifier (eg "^1.4.4") of a given
     * instance.
     *
     * Initially there will be a `Specifier` which describes the version as it is
     * on disk, but other `Specifier`s will need to be created which reflect what
     * the version should be once fixed – first for the instance's version group
     * and second for its semver group.
     *
     * The intermediate and final variants of `Specifier` could differ along the
     * way and whether we're linting the current state of the monorepo or
     * determining whether possible fixes will ultimately still have a valid
     * version and range, each one has a reference back to the original `Instance`
     * for writing back to it when we do finally commit our fixes once verified.
     */
    instance;
    constructor(data) {
        this.raw = data.raw;
        this.instance = data.instance;
    }
    /**
     * Parse the raw version specifier using
     * https://github.com/npm/npm-package-arg
     */
    parse() {
        const name = this.instance.name;
        const raw = this.raw;
        const packageJsonFile = this.instance.packageJsonFile;
        return (0, effect_1.pipe)(effect_1.Effect.try(() => (0, parse_specifier_1.parseSpecifier)(name, raw, packageJsonFile)), effect_1.Effect.tapError(() => effect_1.Effect.logError(`parseSpecifier threw on ${name}@${raw} at ${packageJsonFile.jsonFile.shortPath}`)));
    }
    /** Return the version portion if it is valid semver */
    getSemver() {
        return non_semver_error_1.NonSemverError.asEffect(this);
    }
    /** Get a new `Specifier` from the given semver version applied to this one */
    setSemver(_version) {
        return non_semver_error_1.NonSemverError.asEffect(this);
    }
    /** Apply the given specifier to a new one with this instance bound to it */
    replaceWith(specifier) {
        return _1.Specifier.create(this.instance, specifier.raw);
    }
}
exports.BaseSpecifier = BaseSpecifier;
