"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSemverGroups = void 0;
const effect_1 = require("effect");
const is_array_of_strings_1 = require("tightrope/guard/is-array-of-strings");
const is_non_empty_array_1 = require("tightrope/guard/is-non-empty-array");
const is_non_empty_string_1 = require("tightrope/guard/is-non-empty-string");
const is_object_1 = require("tightrope/guard/is-object");
const _1 = require(".");
const is_valid_semver_range_1 = require("../guards/is-valid-semver-range");
const disabled_1 = require("./disabled");
const filtered_out_1 = require("./filtered-out");
const ignored_1 = require("./ignored");
const with_range_1 = require("./with-range");
function createSemverGroups(ctx) {
    const { rcFile } = ctx.config;
    const semverGroups = [
        effect_1.Effect.succeed(new filtered_out_1.FilteredOutSemverGroup(ctx)),
        effect_1.Effect.succeed(new with_range_1.WithRangeSemverGroup(false, {
            dependencies: ['**'],
            dependencyTypes: ['local'],
            label: 'the version property of package.json files must always be exact',
            packages: ['**'],
            range: '',
        })),
    ];
    if ((0, is_non_empty_array_1.isNonEmptyArray)(rcFile.semverGroups)) {
        rcFile.semverGroups.forEach((config) => {
            if (!(0, is_object_1.isObject)(config)) {
                return semverGroups.push(effect_1.Effect.fail(new _1.SemverGroup.ConfigError({
                    config,
                    error: 'config is not an object',
                })));
            }
            const mutuallyExclusiveProps = ['isIgnored', 'range'].filter((prop) => Boolean(config[prop]));
            if (mutuallyExclusiveProps.length > 1) {
                return semverGroups.push(effect_1.Effect.fail(new _1.SemverGroup.ConfigError({
                    config,
                    error: `it's unclear what kind of semver group you want, as it contains both ${mutuallyExclusiveProps.join(' and ')}`,
                })));
            }
            const label = (0, is_non_empty_string_1.isNonEmptyString)(config.label) ? config.label : '';
            const dependencyTypes = (0, is_array_of_strings_1.isArrayOfStrings)(config.dependencyTypes)
                ? config.dependencyTypes
                : ['**'];
            const dependencies = (0, is_array_of_strings_1.isArrayOfStrings)(config.dependencies) ? config.dependencies : ['**'];
            const packages = (0, is_array_of_strings_1.isArrayOfStrings)(config.packages) ? config.packages : ['**'];
            const specifierTypes = (0, is_array_of_strings_1.isArrayOfStrings)(config.specifierTypes)
                ? config.specifierTypes
                : ['**'];
            if (config.isIgnored === true) {
                semverGroups.push(effect_1.Effect.succeed(new ignored_1.IgnoredSemverGroup({
                    dependencies,
                    dependencyTypes,
                    specifierTypes,
                    isIgnored: true,
                    label,
                    packages,
                })));
            }
            else if ((0, is_valid_semver_range_1.isValidSemverRange)(config.range)) {
                semverGroups.push(effect_1.Effect.succeed(new with_range_1.WithRangeSemverGroup(false, {
                    dependencies,
                    dependencyTypes,
                    specifierTypes,
                    label,
                    packages,
                    range: config.range,
                })));
            }
        });
    }
    semverGroups.push(effect_1.Effect.succeed(new disabled_1.DisabledSemverGroup(true, {
        dependencies: ['**'],
        dependencyTypes: ['**'],
        specifierTypes: ['**'],
        label: 'Default Semver Group',
        packages: ['**'],
        isDisabled: true,
    })));
    return effect_1.Effect.all(semverGroups);
}
exports.createSemverGroups = createSemverGroups;
