"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.get = void 0;
const effect_1 = require("effect");
const is_function_1 = require("tightrope/guard/is-function");
const is_primitive_1 = require("tightrope/guard/is-primitive");
const is_readable_1 = require("tightrope/guard/is-readable");
function get(origin, ...props) {
    return (0, is_readable_1.isReadable)(origin) ? props.reduce(getChild, origin) : ERR_UNREADABLE_ORIGIN(props, origin);
}
exports.get = get;
/** Used internally by get to retrieve a single child property from a parent object. */
function getChild(parent, prop) {
    // quit if any ancestor was already not found
    if (effect_1.Either.isEither(parent) && effect_1.Either.isLeft(parent))
        return parent;
    // ensure we have a plain value and not an Ok
    const value = effect_1.Either.isEither(parent) && effect_1.Either.isRight(parent) ? effect_1.Either.getOrThrow(parent) : parent;
    // quit if we can't read properties of value (eg value.likeThis)
    if (!(0, is_readable_1.isReadable)(value))
        return ERR_UNREADABLE_CHILD(prop, value);
    // quit if value is object/array/function etc but the child is not found
    if (!(0, is_primitive_1.isPrimitive)(value) && prop in value === false)
        return ERR_NOT_FOUND(prop, value);
    // quit if eg true.toFixed, 12.toUpperCase
    if ((0, is_primitive_1.isPrimitive)(value) && value[prop] === undefined)
        return ERR_NOT_FOUND(prop, value);
    // the value is present, return it
    const child = value[prop];
    return effect_1.Either.right((0, is_function_1.isFunction)(child) ? child.bind(value) : child);
}
function ERR_UNREADABLE_CHILD(child, value) {
    return effect_1.Either.left(new Error(`Cannot read "${child}" from unreadable value: ${value}`));
}
function ERR_NOT_FOUND(child, value) {
    return effect_1.Either.left(new Error(`Property "${child}" not found on value: ${value}`));
}
function ERR_UNREADABLE_ORIGIN(props, origin) {
    return effect_1.Either.left(new Error(`Cannot read "${props.join('.')}" from unreadable value: ${origin}`));
}
