"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getInstances = void 0;
const effect_1 = require("effect");
const get_enabled_types_1 = require("../config/get-enabled-types");
const can_add_to_group_1 = require("../guards/can-add-to-group");
const sort_by_name_1 = require("../lib/sort-by-name");
const create_semver_groups_1 = require("../semver-group/create-semver-groups");
const create_version_groups_1 = require("../version-group/create-version-groups");
function getInstances(ctx, io, errorHandlers) {
    const exitOnError = effect_1.Effect.flatMap(() => effect_1.Effect.failSync(() => io.process.exit(1)));
    return (0, effect_1.pipe)(effect_1.Effect.Do, effect_1.Effect.bind('enabledTypes', () => (0, get_enabled_types_1.getEnabledTypes)(ctx.config)), effect_1.Effect.bind('semverGroups', () => (0, create_semver_groups_1.createSemverGroups)(ctx)), effect_1.Effect.bind('versionGroups', () => (0, create_version_groups_1.createVersionGroups)(ctx)), effect_1.Effect.bind('instances', (acc) => (0, effect_1.pipe)(ctx.packageJsonFiles, effect_1.Effect.forEach((file) => file.getInstances(acc.enabledTypes)), effect_1.Effect.map((instancesByFile) => instancesByFile.flat()))), effect_1.Effect.tap(({ instances, semverGroups, versionGroups }) => effect_1.Effect.sync(() => {
        for (const instance of instances) {
            // assign each instance to its semver group, first match wins
            assignToSemverGroup: for (const group of semverGroups) {
                if ((0, can_add_to_group_1.canAddToGroup)(ctx.packageJsonFilesByName, group, instance)) {
                    instance.semverGroup = group;
                    group.instances.push(instance);
                    break assignToSemverGroup;
                }
            }
            // assign each instance to its version group, first match wins
            assignToVersionGroup: for (const group of versionGroups) {
                if ((0, can_add_to_group_1.canAddToGroup)(ctx.packageJsonFilesByName, group, instance)) {
                    instance.versionGroup = group;
                    group.instances.push(instance);
                    break assignToVersionGroup;
                }
            }
        }
    })), effect_1.Effect.map(({ instances, semverGroups, versionGroups }) => ({
        all: instances,
        semverGroups: getSortedAndFiltered(semverGroups),
        versionGroups: getSortedAndFiltered(versionGroups),
    })), effect_1.Effect.catchTags({
        DeprecatedTypesError: (0, effect_1.flow)(errorHandlers.DeprecatedTypesError, exitOnError),
        InvalidCustomTypeError: (0, effect_1.flow)(errorHandlers.InvalidCustomTypeError, exitOnError),
        RenamedWorkspaceTypeError: (0, effect_1.flow)(errorHandlers.RenamedWorkspaceTypeError, exitOnError),
        SemverGroupConfigError: (0, effect_1.flow)(errorHandlers.SemverGroupConfigError, exitOnError),
        VersionGroupConfigError: (0, effect_1.flow)(errorHandlers.VersionGroupConfigError, exitOnError),
    }));
    function getSortedAndFiltered(groups) {
        return groups.filter((group) => group.instances.sort(sort_by_name_1.sortByName).length > 0);
    }
}
exports.getInstances = getInstances;
