"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getEnabledTypes = exports.RenamedWorkspaceTypeError = exports.DeprecatedTypesError = void 0;
const effect_1 = require("effect");
const is_array_of_strings_1 = require("tightrope/guard/is-array-of-strings");
const is_boolean_1 = require("tightrope/guard/is-boolean");
const is_empty_array_1 = require("tightrope/guard/is-empty-array");
const is_non_empty_array_1 = require("tightrope/guard/is-non-empty-array");
const is_non_empty_string_1 = require("tightrope/guard/is-non-empty-string");
const constants_1 = require("../constants");
const name_and_version_props_1 = require("../strategy/name-and-version-props");
const versions_by_name_1 = require("../strategy/versions-by-name");
const get_custom_types_1 = require("./get-custom-types");
class DeprecatedTypesError extends effect_1.Data.TaggedClass('DeprecatedTypesError') {
}
exports.DeprecatedTypesError = DeprecatedTypesError;
class RenamedWorkspaceTypeError extends effect_1.Data.TaggedClass('RenamedWorkspaceTypeError') {
}
exports.RenamedWorkspaceTypeError = RenamedWorkspaceTypeError;
function getEnabledTypes({ cli, rcFile, }) {
    return (0, effect_1.pipe)(
    // Look for dependency types defined using the old `{ prod: true }` syntax
    // deprecated in syncpack@9.0.0
    effect_1.Effect.succeed(constants_1.INTERNAL_TYPES.filter((key) => (0, is_boolean_1.isBoolean)(rcFile[key]))), 
    // Short-circuit and quit if deprecated config is used
    effect_1.Effect.flatMap((deprecatedTypeProps) => deprecatedTypeProps.length > 0
        ? effect_1.Effect.fail(new DeprecatedTypesError({ types: deprecatedTypeProps }))
        : effect_1.Effect.unit), effect_1.Effect.flatMap(() => (0, effect_1.pipe)(effect_1.Effect.Do, 
    // Get index of every available strategy, keyed by their names as
    // they're referred to in config
    effect_1.Effect.bind('allStrategiesByName', () => (0, effect_1.pipe)(
    // Get custom types if any are defined, short-circuit and quit if
    // any are invalid
    (0, get_custom_types_1.getCustomTypes)({ cli, rcFile }), 
    // Combine them with the default/internal dependency types
    effect_1.Effect.map((customTypes) => Object.fromEntries([
        ['dev', new versions_by_name_1.VersionsByNameStrategy('dev', 'devDependencies')],
        ['local', new name_and_version_props_1.NameAndVersionPropsStrategy('local', 'version', 'name')],
        ['overrides', new versions_by_name_1.VersionsByNameStrategy('overrides', 'overrides')],
        ['peer', new versions_by_name_1.VersionsByNameStrategy('peer', 'peerDependencies')],
        ['pnpmOverrides', new versions_by_name_1.VersionsByNameStrategy('pnpmOverrides', 'pnpm.overrides')],
        ['prod', new versions_by_name_1.VersionsByNameStrategy('prod', 'dependencies')],
        ['resolutions', new versions_by_name_1.VersionsByNameStrategy('resolutions', 'resolutions')],
        ...customTypes.map((type) => [type.name, type]),
    ])))), 
    // The names of every available strategy
    effect_1.Effect.bind('allStrategyNames', ({ allStrategiesByName }) => effect_1.Effect.succeed(Object.keys(allStrategiesByName))), 
    // Create groupings to assign each provided dependencyType to
    effect_1.Effect.bind('strategyNamesByStatus', () => effect_1.Effect.succeed({
        provided: ((0, is_non_empty_string_1.isNonEmptyString)(cli.types)
            ? cli.types.split(',')
            : (0, is_array_of_strings_1.isArrayOfStrings)(rcFile.dependencyTypes)
                ? rcFile.dependencyTypes
                : []).filter(is_non_empty_string_1.isNonEmptyString),
        enabled: [],
        positive: [],
        negative: [],
    })))), effect_1.Effect.tap(({ strategyNamesByStatus }) => effect_1.Effect.logDebug(`dependency types provided by user: ${JSON.stringify(strategyNamesByStatus.provided)}`)), 
    // Determine which dependencyTypes should be enabled based on:
    // * which are defined
    // * which were listed to be enabled
    // * which were listed but !negated
    // * etc.
    effect_1.Effect.flatMap(({ allStrategiesByName, allStrategyNames, strategyNamesByStatus }) => {
        if ((0, is_empty_array_1.isEmptyArray)(strategyNamesByStatus.provided) ||
            strategyNamesByStatus.provided.join('') === '**') {
            return effect_1.Effect.succeed(allStrategyNames.map(getStrategyByName));
        }
        strategyNamesByStatus.provided.forEach((name) => {
            if (name.startsWith('!')) {
                strategyNamesByStatus.negative.push(name.replace('!', ''));
            }
            else {
                strategyNamesByStatus.positive.push(name);
            }
        });
        if ((0, is_non_empty_array_1.isNonEmptyArray)(strategyNamesByStatus.negative)) {
            allStrategyNames.forEach((name) => {
                if (!strategyNamesByStatus.negative.includes(name)) {
                    strategyNamesByStatus.enabled.push(name);
                }
            });
        }
        if ((0, is_non_empty_array_1.isNonEmptyArray)(strategyNamesByStatus.positive)) {
            strategyNamesByStatus.positive.forEach((name) => {
                if (!strategyNamesByStatus.enabled.includes(name)) {
                    strategyNamesByStatus.enabled.push(name);
                }
            });
        }
        if (strategyNamesByStatus.enabled.includes('workspace')) {
            return effect_1.Effect.fail(new RenamedWorkspaceTypeError({}));
        }
        return effect_1.Effect.succeed(strategyNamesByStatus.enabled.map(getStrategyByName));
        function getStrategyByName(type) {
            return allStrategiesByName[type];
        }
    }), effect_1.Effect.tap((enabledTypes) => effect_1.Effect.logDebug(`enabled dependency types determined to be: ${JSON.stringify(enabledTypes)}`)));
}
exports.getEnabledTypes = getEnabledTypes;
