"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCustomTypes = exports.InvalidCustomTypeError = void 0;
const effect_1 = require("effect");
const is_non_empty_object_1 = require("tightrope/guard/is-non-empty-object");
const is_non_empty_string_1 = require("tightrope/guard/is-non-empty-string");
const is_object_1 = require("tightrope/guard/is-object");
const constants_1 = require("../constants");
const name_and_version_props_1 = require("../strategy/name-and-version-props");
const named_version_string_1 = require("../strategy/named-version-string");
const unnamed_version_string_1 = require("../strategy/unnamed-version-string");
const versions_by_name_1 = require("../strategy/versions-by-name");
class InvalidCustomTypeError extends effect_1.Data.TaggedClass('InvalidCustomTypeError') {
}
exports.InvalidCustomTypeError = InvalidCustomTypeError;
function getCustomTypes({ rcFile, }) {
    if (!(0, is_non_empty_object_1.isNonEmptyObject)(rcFile.customTypes))
        return effect_1.Effect.succeed([]);
    return effect_1.Effect.all([...Object.entries(rcFile.customTypes), ...Object.entries(constants_1.DEFAULT_CONFIG.customTypes)].map(([name, config]) => {
        const ERR_OBJ = 'Invalid customType';
        const ERR_PATH = 'Invalid customType.path';
        const ERR_NAME_PATH = 'Invalid customType.namePath';
        const ERR_STRATEGY = 'Invalid customType.strategy';
        if (!(0, is_object_1.isObject)(config))
            return createError(config, ERR_OBJ);
        if (!(0, is_non_empty_string_1.isNonEmptyString)(config.path))
            return createError(config, ERR_PATH);
        const path = config.path;
        const strategy = config.strategy;
        switch (strategy) {
            case 'name~version': {
                const namePath = config.namePath;
                if (!(0, is_non_empty_string_1.isNonEmptyString)(namePath))
                    return createError(config, ERR_NAME_PATH);
                return effect_1.Effect.succeed(new name_and_version_props_1.NameAndVersionPropsStrategy(name, path, namePath));
            }
            case 'name@version': {
                return effect_1.Effect.succeed(new named_version_string_1.NamedVersionStringStrategy(name, path));
            }
            case 'version': {
                return effect_1.Effect.succeed(new unnamed_version_string_1.UnnamedVersionStringStrategy(name, path));
            }
            case 'versionsByName': {
                return effect_1.Effect.succeed(new versions_by_name_1.VersionsByNameStrategy(name, path));
            }
            default: {
                return createError(config, ERR_STRATEGY);
            }
        }
    }));
}
exports.getCustomTypes = getCustomTypes;
function createError(config, reason) {
    return effect_1.Effect.fail(new InvalidCustomTypeError({ config, reason }));
}
