"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setSemverRanges = void 0;
const effect_1 = require("effect");
const is_non_empty_array_1 = require("tightrope/guard/is-non-empty-array");
const fix_mismatches_1 = require("../bin-fix-mismatches/fix-mismatches");
const lint_semver_ranges_1 = require("../bin-lint-semver-ranges/lint-semver-ranges");
const list_mismatches_1 = require("../bin-list-mismatches/list-mismatches");
const tag_1 = require("../config/tag");
const default_error_handlers_1 = require("../error-handlers/default-error-handlers");
const get_context_1 = require("../get-context");
const get_instances_1 = require("../get-instances");
const io_1 = require("../io");
const exit_if_invalid_1 = require("../io/exit-if-invalid");
const write_if_changed_1 = require("../io/write-if-changed");
const with_logger_1 = require("../lib/with-logger");
function setSemverRanges({ io, cli, errorHandlers = default_error_handlers_1.defaultErrorHandlers }) {
    return (0, effect_1.pipe)((0, get_context_1.getContext)({ io, cli, errorHandlers }), effect_1.Effect.flatMap((ctx) => (0, effect_1.pipe)(effect_1.Effect.gen(function* ($) {
        // no semver groups have been configured, they are disabled by default
        if (!(0, is_non_empty_array_1.isNonEmptyArray)(ctx.config.rcFile.semverGroups)) {
            ctx.isInvalid = true;
            yield* $((0, lint_semver_ranges_1.logSemverGroupsDisabledWarning)());
            return ctx;
        }
        const { semverGroups } = yield* $((0, get_instances_1.getInstances)(ctx, io, errorHandlers));
        let fixedCount = 0;
        let unfixableCount = 0;
        let validCount = 0;
        for (const group of semverGroups) {
            if (group._tag === 'WithRange') {
                for (const instance of group.instances) {
                    const report = yield* $(group.inspect(instance));
                    const _tag = report._tag;
                    if (_tag === 'SemverRangeMismatch') {
                        yield* $((0, fix_mismatches_1.fixMismatch)(report));
                        fixedCount++;
                    }
                    else if (_tag === 'UnsupportedMismatch') {
                        yield* $((0, list_mismatches_1.logUnsupportedMismatch)(report));
                        unfixableCount++;
                    }
                    else {
                        validCount++;
                    }
                }
            }
        }
        if (validCount)
            yield* $((0, fix_mismatches_1.logAlreadyValidSize)(validCount));
        if (fixedCount)
            yield* $((0, fix_mismatches_1.logFixedSize)(fixedCount));
        if (unfixableCount)
            yield* $((0, fix_mismatches_1.logUnfixableSize)(unfixableCount));
        return ctx;
    }), effect_1.Effect.flatMap(write_if_changed_1.writeIfChanged), effect_1.Effect.catchTags({
        WriteFileError: (0, effect_1.flow)(errorHandlers.WriteFileError, effect_1.Effect.map(() => {
            ctx.isInvalid = true;
            return ctx;
        })),
    }), effect_1.Effect.flatMap(exit_if_invalid_1.exitIfInvalid))), effect_1.Effect.provide((0, effect_1.pipe)(effect_1.Context.empty(), effect_1.Context.add(tag_1.CliConfigTag, cli), effect_1.Context.add(io_1.IoTag, io))), with_logger_1.withLogger);
}
exports.setSemverRanges = setSemverRanges;
