"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.prompt = void 0;
const chalk_1 = __importDefault(require("chalk"));
const effect_1 = require("effect");
const uniq_1 = require("tightrope/array/uniq");
const is_string_1 = require("tightrope/guard/is-string");
const list_1 = require("../bin-list/list");
const tag_1 = require("../config/tag");
const constants_1 = require("../constants");
const default_error_handlers_1 = require("../error-handlers/default-error-handlers");
const get_context_1 = require("../get-context");
const get_instances_1 = require("../get-instances");
const io_1 = require("../io");
const ask_for_choice_1 = require("../io/ask-for-choice");
const ask_for_input_1 = require("../io/ask-for-input");
const exit_if_invalid_1 = require("../io/exit-if-invalid");
const write_if_changed_1 = require("../io/write-if-changed");
const get_group_header_1 = require("../lib/get-group-header");
const with_logger_1 = require("../lib/with-logger");
function prompt({ io, cli, errorHandlers = default_error_handlers_1.defaultErrorHandlers }) {
    return (0, effect_1.pipe)((0, get_context_1.getContext)({ io, cli, errorHandlers }), effect_1.Effect.flatMap((ctx) => (0, effect_1.pipe)(effect_1.Effect.gen(function* ($) {
        const { versionGroups } = yield* $((0, get_instances_1.getInstances)(ctx, io, errorHandlers));
        let unfixableCount = 0;
        let index = 0;
        for (const group of versionGroups) {
            const unfixable = [];
            for (const groupReport of yield* $(group.inspectAll())) {
                for (const report of groupReport.reports) {
                    if (isUnfixable(report)) {
                        unfixable.push(report);
                    }
                }
                if (unfixable.length) {
                    unfixableCount += unfixable.length;
                    effect_1.Effect.logInfo((0, get_group_header_1.getVersionGroupHeader)({ group, index }));
                    yield* $(askForNextVersion(groupReport, unfixable));
                }
            }
            index++;
        }
        if (unfixableCount) {
            yield* $((0, write_if_changed_1.writeIfChanged)(ctx));
        }
        else {
            const msg = (0, chalk_1.default) `{green ${constants_1.ICON.tick}} no issues which syncpack cannot fix automatically`;
            yield* $(effect_1.Effect.logInfo(msg));
            yield* $((0, list_1.logOtherCommands)());
        }
        return ctx;
    }), effect_1.Effect.catchTags({
        WriteFileError: (0, effect_1.flow)(errorHandlers.WriteFileError, effect_1.Effect.map(() => {
            ctx.isInvalid = true;
            return ctx;
        })),
    }), effect_1.Effect.flatMap(exit_if_invalid_1.exitIfInvalid))), effect_1.Effect.provide((0, effect_1.pipe)(effect_1.Context.empty(), effect_1.Context.add(tag_1.CliConfigTag, cli), effect_1.Context.add(io_1.IoTag, io))), with_logger_1.withLogger);
}
exports.prompt = prompt;
function isUnfixable(report) {
    return (report._tag === 'MissingLocalVersion' ||
        report._tag === 'MissingSnappedToMismatch' ||
        report._tag === 'SameRangeMismatch' ||
        report._tag === 'UnsupportedMismatch');
}
function askForNextVersion(groupReport, unfixable) {
    return (0, effect_1.pipe)(effect_1.Effect.gen(function* ($) {
        const choices = (0, uniq_1.uniq)(groupReport.reports.map((report) => report._tagGroup === 'Fixable'
            ? report.fixable.raw
            : report._tagGroup === 'Unfixable'
                ? report.unfixable.rawSpecifier
                : report._tagGroup === 'Valid'
                    ? report.specifier.raw
                    : null)).filter(is_string_1.isString);
        const OTHER = chalk_1.default.dim('Other');
        const SKIP = chalk_1.default.dim('Skip');
        const QUIT = chalk_1.default.dim('Quit');
        // Ask user to choose a version to align on
        const choice = yield* $((0, ask_for_choice_1.askForChoice)({
            message: groupReport.name,
            choices: [...choices, OTHER, SKIP, QUIT],
        }));
        if (choice === SKIP)
            return;
        // @TODO: Learn https://www.effect.website/docs/data-types/exit
        if (choice === QUIT)
            return process.exit(0);
        const nextVersion = choice === OTHER
            ? yield* $((0, ask_for_input_1.askForInput)({
                message: (0, chalk_1.default) `${groupReport.name} {dim Enter a replacement version}`,
            }))
            : choice;
        yield* $((0, effect_1.pipe)(unfixable, effect_1.Effect.forEach((report) => report.unfixable.write(nextVersion))));
    }), effect_1.Effect.catchTags({
        AskForChoiceError: effect_1.Effect.logDebug,
        AskForInputError: effect_1.Effect.logDebug,
    }));
}
