"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.logOtherCommands = exports.pipeline = exports.list = void 0;
const chalk_1 = __importDefault(require("chalk"));
const effect_1 = require("effect");
const os_1 = require("os");
const tag_1 = require("../config/tag");
const constants_1 = require("../constants");
const default_error_handlers_1 = require("../error-handlers/default-error-handlers");
const get_context_1 = require("../get-context");
const get_instances_1 = require("../get-instances");
const io_1 = require("../io");
const exit_if_invalid_1 = require("../io/exit-if-invalid");
const get_group_header_1 = require("../lib/get-group-header");
const pad_start_1 = require("../lib/pad-start");
const with_logger_1 = require("../lib/with-logger");
function list({ io, cli, errorHandlers = default_error_handlers_1.defaultErrorHandlers }) {
    return (0, effect_1.pipe)((0, get_context_1.getContext)({ io, cli, errorHandlers }), effect_1.Effect.flatMap((ctx) => pipeline(ctx, io, errorHandlers)), effect_1.Effect.flatMap(exit_if_invalid_1.exitIfInvalid), effect_1.Effect.provide((0, effect_1.pipe)(effect_1.Context.empty(), effect_1.Context.add(tag_1.CliConfigTag, cli), effect_1.Context.add(io_1.IoTag, io))), with_logger_1.withLogger);
}
exports.list = list;
function pipeline(ctx, io, errorHandlers) {
    return effect_1.Effect.gen(function* ($) {
        const { versionGroups } = yield* $((0, get_instances_1.getInstances)(ctx, io, errorHandlers));
        let index = 0;
        for (const group of versionGroups) {
            yield* $(effect_1.Effect.logInfo((0, get_group_header_1.getVersionGroupHeader)({ group, index })));
            yield* $(onGroupTag[group._tag](group));
            if (group._tag === 'Banned' || group._tag === 'FilteredOut' || group._tag === 'Ignored') {
                if (group._tag === 'Banned')
                    ctx.isInvalid = true;
                index++;
                continue;
            }
            for (const groupReport of yield* $(group.inspectAll())) {
                const matches = new Set();
                const mismatches = new Set();
                for (const report of groupReport.reports) {
                    if (report.isInvalid)
                        ctx.isInvalid = true;
                    switch (report._tagGroup) {
                        case 'Valid': {
                            const actual = report.specifier.raw;
                            matches.add((0, chalk_1.default) `{gray ${actual}}`);
                            break;
                        }
                        case 'Fixable': {
                            mismatches.add(getLogForFixable(report));
                            break;
                        }
                        case 'Unfixable': {
                            mismatches.add(getLogForUnfixable(report));
                            break;
                        }
                    }
                }
                if (mismatches.size === 0) {
                    yield* $(logMatchingReport(groupReport, matches));
                }
                else {
                    yield* $(logMismatchingReport(groupReport, mismatches));
                }
            }
            index++;
        }
        yield* $(logOtherCommands());
        return ctx;
    });
}
exports.pipeline = pipeline;
const onGroupTag = {
    Banned(group) {
        return effect_1.Effect.gen(function* ($) {
            for (const groupReport of yield* $(group.inspectAll())) {
                const name = groupReport.name;
                const usages = `${(0, pad_start_1.padStart)(groupReport.reports.length)}x`;
                const invalidLabel = (0, chalk_1.default) `{gray ${usages}} {red ${name}}`;
                const msg = (0, chalk_1.default) `${invalidLabel} {gray [Banned]}`;
                yield* $(effect_1.Effect.logInfo(msg));
            }
        });
    },
    FilteredOut(group) {
        return effect_1.Effect.gen(function* ($) {
            for (const groupReport of yield* $(group.inspectAll())) {
                const name = groupReport.name;
                const usages = `${(0, pad_start_1.padStart)(groupReport.reports.length)}x`;
                const invalidLabel = (0, chalk_1.default) `{gray ${usages}} {gray ${name}}`;
                const msg = (0, chalk_1.default) `${invalidLabel} {gray [FilteredOut]}`;
                yield* $(effect_1.Effect.logInfo(msg));
            }
        });
    },
    Ignored(group) {
        return effect_1.Effect.gen(function* ($) {
            for (const groupReport of yield* $(group.inspectAll())) {
                const name = groupReport.name;
                const usages = `${(0, pad_start_1.padStart)(groupReport.reports.length)}x`;
                const invalidLabel = (0, chalk_1.default) `{gray ${usages}} {gray ${name}}`;
                const msg = (0, chalk_1.default) `${invalidLabel} {gray [Ignored]}`;
                yield* $(effect_1.Effect.logInfo(msg));
            }
        });
    },
    Pinned(_group) {
        return effect_1.Effect.unit;
    },
    SameRange(_group) {
        return effect_1.Effect.unit;
    },
    SnappedTo(_group) {
        return effect_1.Effect.unit;
    },
    Standard(_group) {
        return effect_1.Effect.unit;
    },
};
function logMatchingReport(groupReport, messages) {
    const name = groupReport.name;
    const usages = `${(0, pad_start_1.padStart)(groupReport.reports.length)}x`;
    const label = (0, chalk_1.default) `{gray ${usages}} ${name}{gray :}`;
    return effect_1.Effect.logInfo((0, chalk_1.default) `${label} ${[...messages].join((0, chalk_1.default) `{gray , }`)}`);
}
function logMismatchingReport(groupReport, messages) {
    const name = groupReport.name;
    const usages = `${(0, pad_start_1.padStart)(groupReport.reports.length)}x`;
    const label = (0, chalk_1.default) `{gray ${usages}} {red ${name}}{gray :}`;
    const indent = usages.replace(/./g, ' ');
    return effect_1.Effect.logInfo((0, chalk_1.default) `${label}${['', ...messages].join((0, chalk_1.default) `${os_1.EOL}${indent} {red ${constants_1.ICON.cross}} `)}`);
}
function getLogForFixable(report) {
    const _tag = report._tag;
    const actual = report.fixable.instance.rawSpecifier.raw;
    const expected = report.fixable.raw;
    return (0, chalk_1.default) `{red ${actual}} {gray ${constants_1.ICON.rightArrow}} {green ${expected}} {gray [${_tag}]}`;
}
function getLogForUnfixable(report) {
    const _tag = report._tag;
    const actual = report.unfixable.rawSpecifier.raw;
    return (0, chalk_1.default) `{red ${actual}} {gray ${constants_1.ICON.rightArrow}} {gray [${_tag}]}`;
}
function logOtherCommands() {
    return effect_1.Effect.logInfo([
        '',
        '  What next?',
        (0, chalk_1.default) `{dim -} {yellow syncpack list-mismatches} to see more detail about mismatching versions`,
        (0, chalk_1.default) `{dim -} {yellow syncpack fix-mismatches} to fix version mismatches automatically`,
        (0, chalk_1.default) `{dim -} {yellow syncpack format} to sort and prettify your package.json files`,
        (0, chalk_1.default) `{dim -} {yellow syncpack update} to choose updates from the npm registry`,
        (0, chalk_1.default) `{dim -} {yellow syncpack --help} for everything else`,
        '',
    ].join(os_1.EOL));
}
exports.logOtherCommands = logOtherCommands;
