"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.logSameRangeMismatch = exports.logUnsupportedMismatch = exports.logMissingSnappedToMismatch = exports.logMissingLocalVersion = exports.pipeline = exports.listMismatches = void 0;
const chalk_1 = __importDefault(require("chalk"));
const effect_1 = require("effect");
const os_1 = require("os");
const tag_1 = require("../config/tag");
const constants_1 = require("../constants");
const default_error_handlers_1 = require("../error-handlers/default-error-handlers");
const get_context_1 = require("../get-context");
const get_instances_1 = require("../get-instances");
const io_1 = require("../io");
const exit_if_invalid_1 = require("../io/exit-if-invalid");
const get_group_header_1 = require("../lib/get-group-header");
const pad_start_1 = require("../lib/pad-start");
const with_logger_1 = require("../lib/with-logger");
function listMismatches({ io, cli, errorHandlers = default_error_handlers_1.defaultErrorHandlers }) {
    return (0, effect_1.pipe)((0, get_context_1.getContext)({ io, cli, errorHandlers }), effect_1.Effect.flatMap((ctx) => pipeline(ctx, io, errorHandlers)), effect_1.Effect.flatMap(exit_if_invalid_1.exitIfInvalid), effect_1.Effect.provide((0, effect_1.pipe)(effect_1.Context.empty(), effect_1.Context.add(tag_1.CliConfigTag, cli), effect_1.Context.add(io_1.IoTag, io))), with_logger_1.withLogger);
}
exports.listMismatches = listMismatches;
/** Exported to be reused by `syncpack lint` */
function pipeline(ctx, io, errorHandlers) {
    return effect_1.Effect.gen(function* ($) {
        const { versionGroups } = yield* $((0, get_instances_1.getInstances)(ctx, io, errorHandlers));
        let index = 0;
        for (const group of versionGroups) {
            const countByReportGroup = {
                Excluded: 0,
                Fixable: 0,
                Unfixable: 0,
                Valid: 0,
            };
            yield* $(effect_1.Effect.logInfo((0, get_group_header_1.getVersionGroupHeader)({ group, index })));
            for (const groupReport of yield* $(group.inspectAll())) {
                for (const report of groupReport.reports) {
                    countByReportGroup[report._tagGroup]++;
                    if (report.isInvalid)
                        ctx.isInvalid = true;
                    const logReport = onReportTag[report._tag];
                    if (logReport)
                        yield* $(logReport(report));
                }
            }
            yield* $(onReportGroup.Valid(countByReportGroup.Valid));
            yield* $(onReportGroup.Fixable(countByReportGroup.Fixable));
            yield* $(onReportGroup.Unfixable(countByReportGroup.Unfixable));
            yield* $(onReportGroup.Excluded(countByReportGroup.Excluded));
            index++;
        }
        return ctx;
    });
}
exports.pipeline = pipeline;
const onReportGroup = {
    Excluded(amount) {
        if (amount === 0)
            return effect_1.Effect.unit;
        const msg = (0, chalk_1.default) `{gray ${(0, pad_start_1.padStart)(amount)} ${constants_1.ICON.rightArrow} ignored}`;
        return effect_1.Effect.logInfo(msg);
    },
    Fixable(amount) {
        if (amount === 0)
            return effect_1.Effect.unit;
        const msg = (0, chalk_1.default) `${(0, pad_start_1.padStart)(amount)} {green ${constants_1.ICON.tick}} can be auto-fixed`;
        return effect_1.Effect.logInfo(msg);
    },
    Unfixable(amount) {
        if (amount === 0)
            return effect_1.Effect.unit;
        const msg = (0, chalk_1.default) `{red ${(0, pad_start_1.padStart)(amount)} ${constants_1.ICON.panic} can be fixed manually using} {blue syncpack prompt}`;
        return effect_1.Effect.logInfo(msg);
    },
    Valid(amount) {
        if (amount === 0)
            return effect_1.Effect.unit;
        const msg = (0, chalk_1.default) `${(0, pad_start_1.padStart)(amount)} {green ${constants_1.ICON.tick}} already valid`;
        return effect_1.Effect.logInfo(msg);
    },
};
const onReportTag = {
    Banned(report) {
        const _tag = report._tag;
        const instance = report.fixable.instance;
        const name = instance.name;
        const jsonFile = instance.packageJsonFile.jsonFile;
        const path = instance.strategy.path;
        const shortPath = jsonFile.shortPath;
        return effect_1.Effect.logInfo((0, chalk_1.default) `{red ${constants_1.ICON.cross}} ${name} {red banned} {gray ${shortPath} > ${path}} {blue [${_tag}]}`);
    },
    Disabled(_report) {
        return effect_1.Effect.unit;
    },
    FilteredOut(_report) {
        return effect_1.Effect.unit;
    },
    HighestSemverMismatch(report) {
        const _tag = report._tag;
        const fixable = report.fixable;
        const instance = fixable.instance;
        const jsonFile = instance.packageJsonFile.jsonFile;
        const actual = instance.rawSpecifier.raw;
        const expected = fixable.raw;
        const name = instance.name;
        const path = instance.strategy.path;
        const shortPath = jsonFile.shortPath;
        return effect_1.Effect.logInfo((0, chalk_1.default) `{red ${constants_1.ICON.cross}} ${name} {red ${actual}} {dim ${constants_1.ICON.rightArrow}} {green ${expected}} {gray ${shortPath} > ${path}} {blue [${_tag}]}`);
    },
    Ignored(_report) {
        return effect_1.Effect.unit;
    },
    LocalPackageMismatch(report) {
        const _tag = report._tag;
        const fixable = report.fixable;
        const instance = fixable.instance;
        const actual = instance.rawSpecifier.raw;
        const expected = fixable.raw;
        const name = instance.name;
        const path = instance.strategy.path;
        const shortPath = instance.packageJsonFile.jsonFile.shortPath;
        return effect_1.Effect.logInfo((0, chalk_1.default) `{red ${constants_1.ICON.cross}} ${name} {red ${actual}} {dim ${constants_1.ICON.rightArrow}} {green ${expected}} {gray ${shortPath} > ${path}} {blue [${_tag}]}`);
    },
    LowestSemverMismatch(report) {
        const _tag = report._tag;
        const fixable = report.fixable;
        const instance = fixable.instance;
        const actual = instance.rawSpecifier.raw;
        const expected = fixable.raw;
        const name = instance.name;
        const path = instance.strategy.path;
        const shortPath = instance.packageJsonFile.jsonFile.shortPath;
        return effect_1.Effect.logInfo((0, chalk_1.default) `{red ${constants_1.ICON.cross}} ${name} {red ${actual}} {dim ${constants_1.ICON.rightArrow}} {green ${expected}} {gray ${shortPath} > ${path}} {blue [${_tag}]}`);
    },
    MissingLocalVersion(report) {
        const instance = report.unfixable;
        const localPath = report.localInstance.packageJsonFile.jsonFile.shortPath;
        const jsonFile = instance.packageJsonFile.jsonFile;
        const actual = instance.rawSpecifier.raw;
        const name = instance.name;
        const path = instance.strategy.path;
        const shortPath = jsonFile.shortPath;
        return effect_1.Effect.logInfo([
            (0, chalk_1.default) `{red ${constants_1.ICON.cross}} ${name} {red ${actual}} {dim ${constants_1.ICON.rightArrow}} {red ???} {gray ${shortPath} > ${path}} {blue [MissingLocalVersion]}`,
            (0, chalk_1.default) `  {red ${localPath} does not have a .version property which is exact semver}`,
        ].join(os_1.EOL));
    },
    MissingSnappedToMismatch(report) {
        const instance = report.unfixable;
        const jsonFile = instance.packageJsonFile.jsonFile;
        const actual = instance.rawSpecifier.raw;
        const name = instance.name;
        const path = instance.strategy.path;
        const shortPath = jsonFile.shortPath;
        return effect_1.Effect.logInfo([
            (0, chalk_1.default) `{red ${constants_1.ICON.cross}} ${name} {red ${actual}} {dim ${constants_1.ICON.rightArrow}} {red ???} {gray ${shortPath} > ${path}} {blue [MissingSnappedToMismatch]}`,
            (0, chalk_1.default) `  {red no package in this groups .snapTo array depend on ${name}}`,
        ].join(os_1.EOL));
    },
    PinnedMismatch(report) {
        const _tag = report._tag;
        const fixable = report.fixable;
        const instance = fixable.instance;
        const actual = instance.rawSpecifier.raw;
        const expected = fixable.raw;
        const name = instance.name;
        const path = instance.strategy.path;
        const shortPath = instance.packageJsonFile.jsonFile.shortPath;
        return effect_1.Effect.logInfo((0, chalk_1.default) `{red ${constants_1.ICON.cross}} ${name} {red ${actual}} {dim ${constants_1.ICON.rightArrow}} {green ${expected}} {gray ${shortPath} > ${path}} {blue [${_tag}]}`);
    },
    SameRangeMismatch(report) {
        const instance = report.unfixable;
        const jsonFile = instance.packageJsonFile.jsonFile;
        const actual = instance.rawSpecifier.raw;
        const name = instance.name;
        const path = instance.strategy.path;
        const shortPath = jsonFile.shortPath;
        const mismatches = report.mismatches;
        return effect_1.Effect.logInfo([
            (0, chalk_1.default) `{red ${constants_1.ICON.cross}} ${name} {red range ${actual} does not include ${mismatches.join(', ')}} {gray ${shortPath} > ${path}} {blue [SameRangeMismatch]}`,
            (0, chalk_1.default) `  {gray use {blue syncpack prompt} to fix this manually}`,
        ].join(os_1.EOL));
    },
    SemverRangeMismatch(report) {
        const _tag = report._tag;
        const fixable = report.fixable;
        const instance = fixable.instance;
        const actual = instance.rawSpecifier.raw;
        const expected = fixable.raw;
        const name = instance.name;
        const path = instance.strategy.path;
        const shortPath = instance.packageJsonFile.jsonFile.shortPath;
        return effect_1.Effect.logInfo((0, chalk_1.default) `{red ${constants_1.ICON.cross}} ${name} {red ${actual}} {dim ${constants_1.ICON.rightArrow}} {green ${expected}} {gray ${shortPath} > ${path}} {blue [${_tag}]}`);
    },
    SnappedToMismatch(report) {
        const _tag = report._tag;
        const fixable = report.fixable;
        const instance = fixable.instance;
        const actual = instance.rawSpecifier.raw;
        const expected = fixable.raw;
        const name = instance.name;
        const path = instance.strategy.path;
        const shortPath = instance.packageJsonFile.jsonFile.shortPath;
        return effect_1.Effect.logInfo((0, chalk_1.default) `{red ${constants_1.ICON.cross}} ${name} {red ${actual}} {dim ${constants_1.ICON.rightArrow}} {green ${expected}} {gray ${shortPath} > ${path}} {blue [${_tag}]}`);
    },
    UnsupportedMismatch(report) {
        const instance = report.unfixable;
        const jsonFile = instance.packageJsonFile.jsonFile;
        const actual = instance.rawSpecifier.raw;
        const name = instance.name;
        const path = instance.strategy.path;
        const shortPath = jsonFile.shortPath;
        return effect_1.Effect.logInfo([
            (0, chalk_1.default) `{red ${constants_1.ICON.cross}} ${name} {red ${actual}} {dim ${constants_1.ICON.rightArrow}} {red ???} {gray ${shortPath} > ${path}} {blue [UnsupportedMismatch]}`,
            (0, chalk_1.default) `  {red use {blue syncpack prompt} to fix this manually}`,
        ].join(os_1.EOL));
    },
    Valid(_report) {
        return effect_1.Effect.unit;
    },
};
exports.logMissingLocalVersion = onReportTag.MissingLocalVersion;
exports.logMissingSnappedToMismatch = onReportTag.MissingSnappedToMismatch;
exports.logUnsupportedMismatch = onReportTag.UnsupportedMismatch;
exports.logSameRangeMismatch = onReportTag.SameRangeMismatch;
