"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.logIgnoredSize = exports.logSemverGroupsDisabledWarning = exports.pipeline = exports.lintSemverRanges = void 0;
const chalk_1 = __importDefault(require("chalk"));
const effect_1 = require("effect");
const os_1 = require("os");
const is_non_empty_array_1 = require("tightrope/guard/is-non-empty-array");
const tag_1 = require("../config/tag");
const constants_1 = require("../constants");
const default_error_handlers_1 = require("../error-handlers/default-error-handlers");
const get_context_1 = require("../get-context");
const get_instances_1 = require("../get-instances");
const io_1 = require("../io");
const exit_if_invalid_1 = require("../io/exit-if-invalid");
const get_group_header_1 = require("../lib/get-group-header");
const pad_start_1 = require("../lib/pad-start");
const with_logger_1 = require("../lib/with-logger");
function lintSemverRanges({ io, cli, errorHandlers = default_error_handlers_1.defaultErrorHandlers }) {
    return (0, effect_1.pipe)((0, get_context_1.getContext)({ io, cli, errorHandlers }), effect_1.Effect.flatMap((ctx) => pipeline(ctx, io, errorHandlers)), effect_1.Effect.flatMap(exit_if_invalid_1.exitIfInvalid), effect_1.Effect.provide((0, effect_1.pipe)(effect_1.Context.empty(), effect_1.Context.add(tag_1.CliConfigTag, cli), effect_1.Context.add(io_1.IoTag, io))), with_logger_1.withLogger);
}
exports.lintSemverRanges = lintSemverRanges;
/** Exported to be reused by `syncpack lint` */
function pipeline(ctx, io, errorHandlers) {
    return effect_1.Effect.gen(function* ($) {
        // no semver groups have been configured, they are disabled by default
        if (!(0, is_non_empty_array_1.isNonEmptyArray)(ctx.config.rcFile.semverGroups)) {
            yield* $(logSemverGroupsDisabledWarning());
            return ctx;
        }
        const { semverGroups } = yield* $((0, get_instances_1.getInstances)(ctx, io, errorHandlers));
        let index = 0;
        for (const group of semverGroups) {
            const groupSize = group.instances.length;
            let validCount = 0;
            if (group._tag === 'Ignored') {
                yield* $(effect_1.Effect.logInfo((0, get_group_header_1.getSemverGroupHeader)({ group, index })));
                yield* $(logIgnoredSize(groupSize));
            }
            else if (group._tag === 'WithRange') {
                yield* $(effect_1.Effect.logInfo((0, get_group_header_1.getSemverGroupHeader)({ group, index })));
                for (const instance of group.instances) {
                    const report = yield* $(group.inspect(instance));
                    if (report._tag === 'Valid') {
                        validCount++;
                        continue;
                    }
                    ctx.isInvalid = true;
                    if (report._tag === 'SemverRangeMismatch') {
                        yield* $(logSemverRangeMismatch(report));
                    }
                    else if (report._tag === 'UnsupportedMismatch') {
                        yield* $(logUnsupportedMismatch(report));
                    }
                }
            }
            if (validCount > 0)
                yield* $(logValidSize(validCount));
            index++;
        }
        return ctx;
    });
}
exports.pipeline = pipeline;
function logSemverGroupsDisabledWarning() {
    return effect_1.Effect.logInfo([
        (0, chalk_1.default) `{red ${constants_1.ICON.panic} it looks like semver ranges have not yet been configured for this project}`,
        (0, chalk_1.default) `  {red see the guide at} {yellow https://jamiemason.github.io/syncpack/guide/semver-groups}`,
    ].join(os_1.EOL));
}
exports.logSemverGroupsDisabledWarning = logSemverGroupsDisabledWarning;
function logIgnoredSize(amount) {
    const msg = (0, chalk_1.default) `${(0, pad_start_1.padStart)(amount)} {dim ${constants_1.ICON.skip}} ${amount} ignored`;
    return effect_1.Effect.logInfo(msg);
}
exports.logIgnoredSize = logIgnoredSize;
function logValidSize(amount) {
    const msg = (0, chalk_1.default) `{green ${constants_1.ICON.tick}} {dim ${amount} valid}`;
    return effect_1.Effect.logInfo(msg);
}
function logSemverRangeMismatch(report) {
    const _tag = report._tag;
    const instance = report.fixable.instance;
    const name = instance.name;
    const actual = instance.rawSpecifier.raw;
    const expected = report.fixable.raw;
    const propPath = instance.strategy.path;
    const filePath = instance.packageJsonFile.jsonFile.shortPath;
    return effect_1.Effect.logInfo((0, chalk_1.default) `{red ${constants_1.ICON.cross}} ${name} {red ${actual}} {dim ${constants_1.ICON.rightArrow}} {green ${expected}} {gray ${filePath} > ${propPath}} {blue [${_tag}]}`);
}
function logUnsupportedMismatch(report) {
    const _tag = report._tag;
    const instance = report.unfixable;
    const name = instance.name;
    const actual = instance.rawSpecifier.raw;
    const propPath = instance.strategy.path;
    const filePath = instance.packageJsonFile.jsonFile.shortPath;
    return effect_1.Effect.logInfo((0, chalk_1.default) `{red ${constants_1.ICON.cross} name {white ${name}} or version {white ${actual}} are not supported} {gray ${filePath} > ${propPath}} {blue [${_tag}]}`);
}
