#!/usr/bin/env node
"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const chalk_1 = __importDefault(require("chalk"));
const commander_1 = require("commander");
const effect_1 = require("effect");
const io_1 = require("../io");
const show_help_on_error_1 = require("../lib/show-help-on-error");
const option_1 = require("../option");
const format_1 = require("./format");
commander_1.program.description((0, chalk_1.default) `
  Organise package.json files according to a conventional format, where fields
  appear in a predictable order and nested fields are ordered alphabetically.
  Shorthand properties are used where available, such as the {yellow repository} and
  {yellow bugs} fields.`.replace(/^\n/, ''));
commander_1.program.on('--help', () => {
    console.log((0, chalk_1.default) `
Examples:
  {dim # uses defaults for resolving packages}
  syncpack format
  {dim # uses packages defined by --source when provided}
  syncpack format --source {yellow "apps/*/package.json"}
  {dim # multiple globs can be provided like this}
  syncpack format --source {yellow "apps/*/package.json"} --source {yellow "core/*/package.json"}
  {dim # indent package.json with 4 spaces instead of 2}
  syncpack format --indent {yellow "    "}

Resolving Packages:
  1. If {yellow --source} globs are provided, use those.
  2. If using Pnpm Workspaces, read {yellow packages} from {yellow pnpm-workspace.yaml} in the root of the project.
  3. If using Yarn Workspaces, read {yellow workspaces} from {yellow package.json}.
  4. If using Lerna, read {yellow packages} from {yellow lerna.json}.
  5. Default to {yellow "package.json"} and {yellow "packages/*/package.json"}.

Reference:
  globs            {blue.underline https://github.com/isaacs/node-glob#glob-primer}
  lerna.json       {blue.underline https://github.com/lerna/lerna#lernajson}
  Yarn Workspaces  {blue.underline https://yarnpkg.com/lang/en/docs/workspaces}
  Pnpm Workspaces  {blue.underline https://pnpm.js.org/en/workspaces}
`);
});
(0, show_help_on_error_1.showHelpOnError)(commander_1.program);
commander_1.program
    .option(...option_1.option.source)
    .option(...option_1.option.config)
    .option(...option_1.option.indent)
    .parse(process.argv);
effect_1.Effect.runPromise((0, format_1.format)({
    io: io_1.io,
    cli: {
        configPath: commander_1.program.opts().config,
        indent: commander_1.program.opts().indent,
        source: commander_1.program.opts().source,
    },
}));
