"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.pipeline = exports.format = void 0;
const effect_1 = require("effect");
const is_array_1 = require("tightrope/guard/is-array");
const is_non_empty_string_1 = require("tightrope/guard/is-non-empty-string");
const is_object_1 = require("tightrope/guard/is-object");
const get_sort_az_1 = require("../config/get-sort-az");
const get_sort_exports_1 = require("../config/get-sort-exports");
const get_sort_first_1 = require("../config/get-sort-first");
const tag_1 = require("../config/tag");
const default_error_handlers_1 = require("../error-handlers/default-error-handlers");
const get_context_1 = require("../get-context");
const io_1 = require("../io");
const exit_if_invalid_1 = require("../io/exit-if-invalid");
const write_if_changed_1 = require("../io/write-if-changed");
const with_logger_1 = require("../lib/with-logger");
function format({ io, cli, errorHandlers = default_error_handlers_1.defaultErrorHandlers }) {
    return (0, effect_1.pipe)((0, get_context_1.getContext)({ io, cli, errorHandlers }), effect_1.Effect.flatMap(pipeline), effect_1.Effect.flatMap((ctx) => (0, effect_1.pipe)((0, write_if_changed_1.writeIfChanged)(ctx), effect_1.Effect.catchTags({
        WriteFileError: (0, effect_1.flow)(errorHandlers.WriteFileError, effect_1.Effect.map(() => {
            ctx.isInvalid = true;
            return ctx;
        })),
    }))), effect_1.Effect.flatMap(exit_if_invalid_1.exitIfInvalid), effect_1.Effect.provide((0, effect_1.pipe)(effect_1.Context.empty(), effect_1.Context.add(tag_1.CliConfigTag, cli), effect_1.Context.add(io_1.IoTag, io))), with_logger_1.withLogger);
}
exports.format = format;
function pipeline(ctx) {
    const { config, packageJsonFiles } = ctx;
    const sortAz = (0, get_sort_az_1.getSortAz)(config);
    const sortExports = (0, get_sort_exports_1.getSortExports)(config);
    const sortFirst = (0, get_sort_first_1.getSortFirst)(config);
    const sortPackages = config.rcFile.sortPackages !== false;
    const formatBugs = config.rcFile.formatBugs !== false;
    const formatRepository = config.rcFile.formatRepository !== false;
    packageJsonFiles.forEach((file) => {
        const { contents } = file.jsonFile;
        const chain = contents;
        if (formatBugs) {
            const bugsUrl = chain?.bugs?.url;
            if (bugsUrl) {
                contents.bugs = bugsUrl;
            }
        }
        if (formatRepository) {
            const repoUrl = chain?.repository?.url;
            const repoDir = chain?.repository?.directory;
            if ((0, is_non_empty_string_1.isNonEmptyString)(repoUrl) && !(0, is_non_empty_string_1.isNonEmptyString)(repoDir)) {
                contents.repository = repoUrl.includes('github.com')
                    ? repoUrl.replace(/^.+github\.com\//, '')
                    : repoUrl;
            }
        }
        if (sortExports.length > 0) {
            visitExports(sortExports, contents.exports);
        }
        if (sortAz.length > 0) {
            sortAz.forEach((key) => sortAlphabetically(contents[key]));
        }
        if (sortPackages) {
            const sortedKeys = Object.keys(contents).sort();
            sortObject(sortedKeys, contents);
        }
        if (sortFirst.length > 0) {
            const otherKeys = Object.keys(contents);
            const sortedKeys = new Set([...sortFirst, ...otherKeys]);
            sortObject(sortedKeys, contents);
        }
    });
    return effect_1.Effect.succeed(ctx);
}
exports.pipeline = pipeline;
function visitExports(sortExports, value) {
    if ((0, is_object_1.isObject)(value)) {
        const otherKeys = Object.keys(value);
        const sortedKeys = new Set([...sortExports, ...otherKeys]);
        sortObject(sortedKeys, value);
        Object.values(value).forEach((nextValue) => visitExports(sortExports, nextValue));
    }
}
function sortObject(sortedKeys, obj) {
    sortedKeys.forEach((key) => {
        const value = obj[key];
        delete obj[key];
        obj[key] = value;
    });
}
function sortAlphabetically(value) {
    if ((0, is_array_1.isArray)(value)) {
        value.sort();
    }
    else if ((0, is_object_1.isObject)(value)) {
        sortObject(Object.keys(value).sort(), value);
    }
}
