"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.logUnfixableSize = exports.logFixedSize = exports.logAlreadyValidSize = exports.fixMismatch = exports.fixMismatches = void 0;
const chalk_1 = __importDefault(require("chalk"));
const effect_1 = require("effect");
const is_object_1 = require("tightrope/guard/is-object");
const is_undefined_1 = require("tightrope/guard/is-undefined");
const lint_semver_ranges_1 = require("../bin-lint-semver-ranges/lint-semver-ranges");
const list_mismatches_1 = require("../bin-list-mismatches/list-mismatches");
const tag_1 = require("../config/tag");
const constants_1 = require("../constants");
const default_error_handlers_1 = require("../error-handlers/default-error-handlers");
const get_context_1 = require("../get-context");
const get_instances_1 = require("../get-instances");
const io_1 = require("../io");
const exit_if_invalid_1 = require("../io/exit-if-invalid");
const write_if_changed_1 = require("../io/write-if-changed");
const get_group_header_1 = require("../lib/get-group-header");
const pad_start_1 = require("../lib/pad-start");
const with_logger_1 = require("../lib/with-logger");
const delete_1 = require("../version-group/lib/delete");
function fixMismatches({ io, cli, errorHandlers = default_error_handlers_1.defaultErrorHandlers }) {
    return (0, effect_1.pipe)((0, get_context_1.getContext)({ io, cli, errorHandlers }), effect_1.Effect.flatMap((ctx) => (0, effect_1.pipe)(effect_1.Effect.gen(function* ($) {
        const { versionGroups } = yield* $((0, get_instances_1.getInstances)(ctx, io, errorHandlers));
        let index = 0;
        for (const group of versionGroups) {
            const groupSize = group.instances.length;
            let fixedCount = 0;
            let unfixableCount = 0;
            let validCount = 0;
            if (group._tag === 'FilteredOut') {
                index++;
                continue;
            }
            yield* $(effect_1.Effect.logInfo((0, get_group_header_1.getVersionGroupHeader)({ group, index })));
            if (group._tag === 'Ignored') {
                yield* $((0, lint_semver_ranges_1.logIgnoredSize)(groupSize));
                index++;
                continue;
            }
            for (const groupReport of yield* $(group.inspectAll())) {
                for (const report of groupReport.reports) {
                    if (report._tagGroup === 'Valid') {
                        validCount++;
                    }
                    else if (report._tagGroup === 'Fixable') {
                        fixedCount++;
                        yield* $(fixMismatch(report));
                    }
                    else if (report._tagGroup === 'Unfixable') {
                        ctx.isInvalid = true;
                        unfixableCount++;
                    }
                    if (report._tag === 'MissingLocalVersion') {
                        yield* $((0, list_mismatches_1.logMissingLocalVersion)(report));
                    }
                    else if (report._tag === 'MissingSnappedToMismatch') {
                        yield* $((0, list_mismatches_1.logMissingSnappedToMismatch)(report));
                    }
                    else if (report._tag === 'UnsupportedMismatch') {
                        yield* $((0, list_mismatches_1.logUnsupportedMismatch)(report));
                    }
                    else if (report._tag === 'SameRangeMismatch') {
                        yield* $((0, list_mismatches_1.logSameRangeMismatch)(report));
                    }
                }
            }
            if (validCount)
                yield* $(logAlreadyValidSize(validCount));
            if (fixedCount)
                yield* $(logFixedSize(fixedCount));
            if (unfixableCount)
                yield* $(logUnfixableSize(unfixableCount));
            index++;
        }
        yield* $(removeEmptyObjects(ctx));
        return ctx;
    }), effect_1.Effect.flatMap(write_if_changed_1.writeIfChanged), effect_1.Effect.catchTags({
        WriteFileError: (0, effect_1.flow)(errorHandlers.WriteFileError, effect_1.Effect.map(() => {
            ctx.isInvalid = true;
            return ctx;
        })),
    }), effect_1.Effect.flatMap(exit_if_invalid_1.exitIfInvalid))), effect_1.Effect.provide((0, effect_1.pipe)(effect_1.Context.empty(), effect_1.Context.add(tag_1.CliConfigTag, cli), effect_1.Context.add(io_1.IoTag, io))), with_logger_1.withLogger);
}
exports.fixMismatches = fixMismatches;
function fixMismatch(report) {
    return report.fixable.instance.write(report._tag === 'Banned' ? delete_1.DELETE : report.fixable.raw);
}
exports.fixMismatch = fixMismatch;
/** Remove empty objects such as `{"dependencies": {}}` left after deleting */
function removeEmptyObjects(ctx) {
    return effect_1.Effect.sync(() => {
        ctx.packageJsonFiles.forEach((file) => {
            const contents = file.jsonFile.contents;
            Object.keys(contents).forEach((key) => {
                const value = contents[key];
                if ((0, is_object_1.isObject)(value) && Object.values(value).every(is_undefined_1.isUndefined)) {
                    delete contents[key];
                }
            });
        });
    });
}
function logAlreadyValidSize(amount) {
    const msg = (0, chalk_1.default) `${(0, pad_start_1.padStart)(amount)} {green ${constants_1.ICON.tick}} already valid`;
    return effect_1.Effect.logInfo(msg);
}
exports.logAlreadyValidSize = logAlreadyValidSize;
function logFixedSize(amount) {
    const msg = (0, chalk_1.default) `${(0, pad_start_1.padStart)(amount)} {green ${constants_1.ICON.tick}} fixed`;
    return effect_1.Effect.logInfo(msg);
}
exports.logFixedSize = logFixedSize;
function logUnfixableSize(amount) {
    const msg = (0, chalk_1.default) `{red ${(0, pad_start_1.padStart)(amount)} ${constants_1.ICON.panic} can be fixed manually using} {blue syncpack prompt}`;
    return effect_1.Effect.logInfo(msg);
}
exports.logUnfixableSize = logUnfixableSize;
